"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CleanroomsMl = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [cleanrooms-ml](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscleanroomsml.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class CleanroomsMl extends shared_1.PolicyStatement {
    /**
     * Grants permission to cancel a trained model
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_CancelTrainedModel.html
     */
    toCancelTrainedModel() {
        return this.to('CancelTrainedModel');
    }
    /**
     * Grants permission to cancel a trained model inference job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_CancelTrainedModelInferenceJob.html
     */
    toCancelTrainedModelInferenceJob() {
        return this.to('CancelTrainedModelInferenceJob');
    }
    /**
     * Grants permission to create an audience model
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_CreateAudienceModel.html
     */
    toCreateAudienceModel() {
        return this.to('CreateAudienceModel');
    }
    /**
     * Grants permission to create a configured audience model
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_CreateConfiguredAudienceModel.html
     */
    toCreateConfiguredAudienceModel() {
        return this.to('CreateConfiguredAudienceModel');
    }
    /**
     * Grants permission to create a configured model algorithm
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_CreateConfiguredModelAlgorithm.html
     */
    toCreateConfiguredModelAlgorithm() {
        return this.to('CreateConfiguredModelAlgorithm');
    }
    /**
     * Grants permission to create a configured model algorithm association
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_CreateConfiguredModelAlgorithmAssociation.html
     */
    toCreateConfiguredModelAlgorithmAssociation() {
        return this.to('CreateConfiguredModelAlgorithmAssociation');
    }
    /**
     * Grants permission to create an ML input channel
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_CreateMLInputChannel.html
     */
    toCreateMLInputChannel() {
        return this.to('CreateMLInputChannel');
    }
    /**
     * Grants permission to create a trained model
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_CreateTrainedModel.html
     */
    toCreateTrainedModel() {
        return this.to('CreateTrainedModel');
    }
    /**
     * Grants permission to create a training dataset, or seed audience. In Clean Rooms ML, the TrainingDataset is metadata that points to a Glue table, which is read only during AudienceModel creation
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_CreateTrainingDataset.html
     */
    toCreateTrainingDataset() {
        return this.to('CreateTrainingDataset');
    }
    /**
     * Grants permission to delete the specified audience generation job, and removes all data associated with the job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_DeleteAudienceGenerationJob.html
     */
    toDeleteAudienceGenerationJob() {
        return this.to('DeleteAudienceGenerationJob');
    }
    /**
     * Grants permission to delete the specified audience generation job, and removes all data associated with the job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_DeleteAudienceModel.html
     */
    toDeleteAudienceModel() {
        return this.to('DeleteAudienceModel');
    }
    /**
     * Grants permission to delete the specified configured audience model
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_DeleteConfiguredAudienceModel.html
     */
    toDeleteConfiguredAudienceModel() {
        return this.to('DeleteConfiguredAudienceModel');
    }
    /**
     * Grants permission to delete the specified configured audience model policy
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_DeleteConfiguredAudienceModelPolicy.html
     */
    toDeleteConfiguredAudienceModelPolicy() {
        return this.to('DeleteConfiguredAudienceModelPolicy');
    }
    /**
     * Grants permission to delete a configured model algorithm
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_DeleteConfiguredModelAlgorithm.html
     */
    toDeleteConfiguredModelAlgorithm() {
        return this.to('DeleteConfiguredModelAlgorithm');
    }
    /**
     * Grants permission to delete a configured model algorithm association
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_DeleteConfiguredModelAlgorithmAssociation.html
     */
    toDeleteConfiguredModelAlgorithmAssociation() {
        return this.to('DeleteConfiguredModelAlgorithmAssociation');
    }
    /**
     * Grants permission to delete an ML configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_DeleteMLConfiguration.html
     */
    toDeleteMLConfiguration() {
        return this.to('DeleteMLConfiguration');
    }
    /**
     * Grants permission to delete all data associated with the ML input channel
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_DeleteMLInputChannelData.html
     */
    toDeleteMLInputChannelData() {
        return this.to('DeleteMLInputChannelData');
    }
    /**
     * Grants permission to delete all output associated with the trained model
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_DeleteTrainedModelOutput.html
     */
    toDeleteTrainedModelOutput() {
        return this.to('DeleteTrainedModelOutput');
    }
    /**
     * Grants permission to delete a training dataset
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_DeleteTrainingDataset.html
     */
    toDeleteTrainingDataset() {
        return this.to('DeleteTrainingDataset');
    }
    /**
     * Grants permission to return information about an audience generation job
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_GetAudienceGenerationJob.html
     */
    toGetAudienceGenerationJob() {
        return this.to('GetAudienceGenerationJob');
    }
    /**
     * Grants permission to return information about an audience model
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_GetAudienceModel.html
     */
    toGetAudienceModel() {
        return this.to('GetAudienceModel');
    }
    /**
     * Grants permission to return information about a configured model algorithm association created by any member in the collaboration
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifCollaborationId()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_GetCollaborationConfiguredModelAlgorithmAssociation.html
     */
    toGetCollaborationConfiguredModelAlgorithmAssociation() {
        return this.to('GetCollaborationConfiguredModelAlgorithmAssociation');
    }
    /**
     * Grants permission to return information about an ML input channel created by any member in the collaboration
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifCollaborationId()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_GetCollaborationMLInputChannel.html
     */
    toGetCollaborationMLInputChannel() {
        return this.to('GetCollaborationMLInputChannel');
    }
    /**
     * Grants permission to return information about a trained model created by any member in the collaboration
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifCollaborationId()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_GetCollaborationTrainedModel.html
     */
    toGetCollaborationTrainedModel() {
        return this.to('GetCollaborationTrainedModel');
    }
    /**
     * Grants permission to return information about a configured audience model
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_GetConfiguredAudienceModel.html
     */
    toGetConfiguredAudienceModel() {
        return this.to('GetConfiguredAudienceModel');
    }
    /**
     * Grants permission to return information about a configured audience model policy
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_GetConfiguredAudienceModelPolicy.html
     */
    toGetConfiguredAudienceModelPolicy() {
        return this.to('GetConfiguredAudienceModelPolicy');
    }
    /**
     * Grants permission to return information about a configured model algorithm
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_GetConfiguredModelAlgorithm.html
     */
    toGetConfiguredModelAlgorithm() {
        return this.to('GetConfiguredModelAlgorithm');
    }
    /**
     * Grants permission to return information about a configured model algorithm association
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_GetConfiguredModelAlgorithmAssociation.html
     */
    toGetConfiguredModelAlgorithmAssociation() {
        return this.to('GetConfiguredModelAlgorithmAssociation');
    }
    /**
     * Grants permission to return information about an ML configuration
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_GetMLConfiguration.html
     */
    toGetMLConfiguration() {
        return this.to('GetMLConfiguration');
    }
    /**
     * Grants permission to return information about an ML input channel
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_GetMLInputChannel.html
     */
    toGetMLInputChannel() {
        return this.to('GetMLInputChannel');
    }
    /**
     * Grants permission to return information about a trained model
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_GetTrainedModel.html
     */
    toGetTrainedModel() {
        return this.to('GetTrainedModel');
    }
    /**
     * Grants permission to return information about a trained model inference job
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_GetTrainedModelInferenceJob.html
     */
    toGetTrainedModelInferenceJob() {
        return this.to('GetTrainedModelInferenceJob');
    }
    /**
     * Grants permission to return information about a training dataset
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_GetTrainingDataset.html
     */
    toGetTrainingDataset() {
        return this.to('GetTrainingDataset');
    }
    /**
     * Grants permission to return a list of the audience export jobs
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_ListAudienceExportJobs.html
     */
    toListAudienceExportJobs() {
        return this.to('ListAudienceExportJobs');
    }
    /**
     * Grants permission to return a list of audience generation jobs
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_ListAudienceGenerationJobs.html
     */
    toListAudienceGenerationJobs() {
        return this.to('ListAudienceGenerationJobs');
    }
    /**
     * Grants permission to return a list of audience models
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_ListAudienceModels.html
     */
    toListAudienceModels() {
        return this.to('ListAudienceModels');
    }
    /**
     * Grants permission to return a list of configured model algorithms created by any member in the collaboration
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifCollaborationId()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_ListCollaborationConfiguredModelAlgorithmAssociations.html
     */
    toListCollaborationConfiguredModelAlgorithmAssociations() {
        return this.to('ListCollaborationConfiguredModelAlgorithmAssociations');
    }
    /**
     * Grants permission to return a list of ML input channels created by any member in the collaboration
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifCollaborationId()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_ListCollaborationMLInputChannels.html
     */
    toListCollaborationMLInputChannels() {
        return this.to('ListCollaborationMLInputChannels');
    }
    /**
     * Grants permission to return a list of trained model export jobs started by any member in the collaboration
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifCollaborationId()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_ListCollaborationTrainedModelExportJobs.html
     */
    toListCollaborationTrainedModelExportJobs() {
        return this.to('ListCollaborationTrainedModelExportJobs');
    }
    /**
     * Grants permission to return a list of trained model inference jobs started by any member in the collaboration
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifCollaborationId()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_ListCollaborationTrainedModelInferenceJobs.html
     */
    toListCollaborationTrainedModelInferenceJobs() {
        return this.to('ListCollaborationTrainedModelInferenceJobs');
    }
    /**
     * Grants permission to return a list of trained models created by any member in the collaboration
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifCollaborationId()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_ListCollaborationTrainedModels.html
     */
    toListCollaborationTrainedModels() {
        return this.to('ListCollaborationTrainedModels');
    }
    /**
     * Grants permission to return a list of configured audience models
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_ListConfiguredAudienceModels.html
     */
    toListConfiguredAudienceModels() {
        return this.to('ListConfiguredAudienceModels');
    }
    /**
     * Grants permission to return a list of configured model algorithm associations
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_ListConfiguredModelAlgorithmAssociations.html
     */
    toListConfiguredModelAlgorithmAssociations() {
        return this.to('ListConfiguredModelAlgorithmAssociations');
    }
    /**
     * Grants permission to return a list of configured model algorithms
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_ListConfiguredModelAlgorithms.html
     */
    toListConfiguredModelAlgorithms() {
        return this.to('ListConfiguredModelAlgorithms');
    }
    /**
     * Grants permission to return a list of ML input channels
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_ListMLInputChannels.html
     */
    toListMLInputChannels() {
        return this.to('ListMLInputChannels');
    }
    /**
     * Grants permission to return a list of tags for a provided resource
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to return a list of trained model inference jobs
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_ListTrainedModelInferenceJobs.html
     */
    toListTrainedModelInferenceJobs() {
        return this.to('ListTrainedModelInferenceJobs');
    }
    /**
     * Grants permission to return a list of trained models
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_ListTrainedModels.html
     */
    toListTrainedModels() {
        return this.to('ListTrainedModels');
    }
    /**
     * Grants permission to return a list of training datasets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_ListTrainingDatasets.html
     */
    toListTrainingDatasets() {
        return this.to('ListTrainingDatasets');
    }
    /**
     * Grants permission to create or update the resource policy for a configured audience model
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_PutConfiguredAudienceModelPolicy.html
     */
    toPutConfiguredAudienceModelPolicy() {
        return this.to('PutConfiguredAudienceModelPolicy');
    }
    /**
     * Grants permission to put an ML configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_PutMLConfiguration.html
     */
    toPutMLConfiguration() {
        return this.to('PutMLConfiguration');
    }
    /**
     * Grants permission to export an audience of a specified size after you have generated an audience
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_StartAudienceExportJob.html
     */
    toStartAudienceExportJob() {
        return this.to('StartAudienceExportJob');
    }
    /**
     * Grants permission to start the audience generation job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifCollaborationId()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_StartAudienceGenerationJob.html
     */
    toStartAudienceGenerationJob() {
        return this.to('StartAudienceGenerationJob');
    }
    /**
     * Grants permission to start a trained model export job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_StartTrainedModelExportJob.html
     */
    toStartTrainedModelExportJob() {
        return this.to('StartTrainedModelExportJob');
    }
    /**
     * Grants permission to start a trained model inference job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_StartTrainedModelInferenceJob.html
     */
    toStartTrainedModelInferenceJob() {
        return this.to('StartTrainedModelInferenceJob');
    }
    /**
     * Grants permission to tag a specific resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a specific resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_UntagResource.html
     */
    toUnTagResource() {
        return this.to('UnTagResource');
    }
    /**
     * Grants permission to update a configured audience model.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cleanrooms-ml/latest/APIReference/API_UpdateConfiguredAudienceModel.html
     */
    toUpdateConfiguredAudienceModel() {
        return this.to('UpdateConfiguredAudienceModel');
    }
    /**
     * Adds a resource of type trainingdataset to the statement
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTrainingdataset(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cleanrooms-ml:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:training-dataset/${resourceId}`);
    }
    /**
     * Adds a resource of type audiencemodel to the statement
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAudiencemodel(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cleanrooms-ml:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:audience-model/${resourceId}`);
    }
    /**
     * Adds a resource of type configuredaudiencemodel to the statement
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfiguredaudiencemodel(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cleanrooms-ml:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:configured-audience-model/${resourceId}`);
    }
    /**
     * Adds a resource of type audiencegenerationjob to the statement
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAudiencegenerationjob(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cleanrooms-ml:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:audience-generation-job/${resourceId}`);
    }
    /**
     * Adds a resource of type ConfiguredModelAlgorithm to the statement
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfiguredModelAlgorithm(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cleanrooms-ml:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:configured-model-algorithm/${resourceId}`);
    }
    /**
     * Adds a resource of type ConfiguredModelAlgorithmAssociation to the statement
     *
     * @param membershipId - Identifier for the membershipId.
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfiguredModelAlgorithmAssociation(membershipId, resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cleanrooms-ml:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:membership/${membershipId}/configured-model-algorithm-association/${resourceId}`);
    }
    /**
     * Adds a resource of type MLInputChannel to the statement
     *
     * @param membershipId - Identifier for the membershipId.
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMLInputChannel(membershipId, resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cleanrooms-ml:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:membership/${membershipId}/ml-input-channel/${resourceId}`);
    }
    /**
     * Adds a resource of type TrainedModel to the statement
     *
     * @param membershipId - Identifier for the membershipId.
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTrainedModel(membershipId, resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cleanrooms-ml:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:membership/${membershipId}/trained-model/${resourceId}`);
    }
    /**
     * Adds a resource of type TrainedModelInferenceJob to the statement
     *
     * @param membershipId - Identifier for the membershipId.
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTrainedModelInferenceJob(membershipId, resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cleanrooms-ml:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:membership/${membershipId}/trained-model-inference-job/${resourceId}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCancelTrainedModel()
     * - .toCancelTrainedModelInferenceJob()
     * - .toCreateAudienceModel()
     * - .toCreateConfiguredAudienceModel()
     * - .toCreateConfiguredModelAlgorithm()
     * - .toCreateConfiguredModelAlgorithmAssociation()
     * - .toCreateMLInputChannel()
     * - .toCreateTrainedModel()
     * - .toCreateTrainingDataset()
     * - .toDeleteAudienceGenerationJob()
     * - .toDeleteAudienceModel()
     * - .toDeleteConfiguredAudienceModel()
     * - .toDeleteConfiguredAudienceModelPolicy()
     * - .toDeleteConfiguredModelAlgorithm()
     * - .toDeleteConfiguredModelAlgorithmAssociation()
     * - .toDeleteMLConfiguration()
     * - .toDeleteMLInputChannelData()
     * - .toDeleteTrainedModelOutput()
     * - .toDeleteTrainingDataset()
     * - .toGetAudienceGenerationJob()
     * - .toGetAudienceModel()
     * - .toGetCollaborationConfiguredModelAlgorithmAssociation()
     * - .toGetCollaborationMLInputChannel()
     * - .toGetCollaborationTrainedModel()
     * - .toGetConfiguredAudienceModel()
     * - .toGetConfiguredAudienceModelPolicy()
     * - .toGetConfiguredModelAlgorithm()
     * - .toGetConfiguredModelAlgorithmAssociation()
     * - .toGetMLConfiguration()
     * - .toGetMLInputChannel()
     * - .toGetTrainedModel()
     * - .toGetTrainedModelInferenceJob()
     * - .toGetTrainingDataset()
     * - .toListAudienceExportJobs()
     * - .toListAudienceGenerationJobs()
     * - .toListCollaborationConfiguredModelAlgorithmAssociations()
     * - .toListCollaborationMLInputChannels()
     * - .toListCollaborationTrainedModelExportJobs()
     * - .toListCollaborationTrainedModelInferenceJobs()
     * - .toListCollaborationTrainedModels()
     * - .toListConfiguredModelAlgorithmAssociations()
     * - .toListConfiguredModelAlgorithms()
     * - .toListMLInputChannels()
     * - .toListTrainedModelInferenceJobs()
     * - .toListTrainedModels()
     * - .toPutMLConfiguration()
     * - .toStartAudienceExportJob()
     * - .toStartAudienceGenerationJob()
     * - .toStartTrainedModelExportJob()
     * - .toStartTrainedModelInferenceJob()
     * - .toTagResource()
     * - .toUpdateConfiguredAudienceModel()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toListTagsForResource()
     * - .toTagResource()
     * - .toUnTagResource()
     *
     * Applies to resource types:
     * - trainingdataset
     * - audiencemodel
     * - configuredaudiencemodel
     * - audiencegenerationjob
     * - ConfiguredModelAlgorithm
     * - ConfiguredModelAlgorithmAssociation
     * - MLInputChannel
     * - TrainedModel
     * - TrainedModelInferenceJob
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCancelTrainedModel()
     * - .toCancelTrainedModelInferenceJob()
     * - .toCreateAudienceModel()
     * - .toCreateConfiguredAudienceModel()
     * - .toCreateConfiguredModelAlgorithm()
     * - .toCreateConfiguredModelAlgorithmAssociation()
     * - .toCreateMLInputChannel()
     * - .toCreateTrainedModel()
     * - .toCreateTrainingDataset()
     * - .toDeleteAudienceGenerationJob()
     * - .toDeleteAudienceModel()
     * - .toDeleteConfiguredAudienceModel()
     * - .toDeleteConfiguredAudienceModelPolicy()
     * - .toDeleteConfiguredModelAlgorithm()
     * - .toDeleteConfiguredModelAlgorithmAssociation()
     * - .toDeleteMLConfiguration()
     * - .toDeleteMLInputChannelData()
     * - .toDeleteTrainedModelOutput()
     * - .toDeleteTrainingDataset()
     * - .toGetAudienceGenerationJob()
     * - .toGetAudienceModel()
     * - .toGetCollaborationConfiguredModelAlgorithmAssociation()
     * - .toGetCollaborationMLInputChannel()
     * - .toGetCollaborationTrainedModel()
     * - .toGetConfiguredAudienceModel()
     * - .toGetConfiguredAudienceModelPolicy()
     * - .toGetConfiguredModelAlgorithm()
     * - .toGetConfiguredModelAlgorithmAssociation()
     * - .toGetMLConfiguration()
     * - .toGetMLInputChannel()
     * - .toGetTrainedModel()
     * - .toGetTrainedModelInferenceJob()
     * - .toGetTrainingDataset()
     * - .toListAudienceExportJobs()
     * - .toListAudienceGenerationJobs()
     * - .toListCollaborationConfiguredModelAlgorithmAssociations()
     * - .toListCollaborationMLInputChannels()
     * - .toListCollaborationTrainedModelExportJobs()
     * - .toListCollaborationTrainedModelInferenceJobs()
     * - .toListCollaborationTrainedModels()
     * - .toListConfiguredModelAlgorithmAssociations()
     * - .toListConfiguredModelAlgorithms()
     * - .toListMLInputChannels()
     * - .toListTagsForResource()
     * - .toListTrainedModelInferenceJobs()
     * - .toListTrainedModels()
     * - .toPutMLConfiguration()
     * - .toStartAudienceExportJob()
     * - .toStartAudienceGenerationJob()
     * - .toStartTrainedModelExportJob()
     * - .toStartTrainedModelInferenceJob()
     * - .toTagResource()
     * - .toUnTagResource()
     * - .toUpdateConfiguredAudienceModel()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by Clean rooms collaboration id
     *
     * https://docs.aws.amazon.com/TBD
     *
     * Applies to actions:
     * - .toGetCollaborationConfiguredModelAlgorithmAssociation()
     * - .toGetCollaborationMLInputChannel()
     * - .toGetCollaborationTrainedModel()
     * - .toListCollaborationConfiguredModelAlgorithmAssociations()
     * - .toListCollaborationMLInputChannels()
     * - .toListCollaborationTrainedModelExportJobs()
     * - .toListCollaborationTrainedModelInferenceJobs()
     * - .toListCollaborationTrainedModels()
     * - .toStartAudienceGenerationJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCollaborationId(value, operator) {
        return this.if(`CollaborationId`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [cleanrooms-ml](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscleanroomsml.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'cleanrooms-ml';
        this.accessLevelList = {
            Write: [
                'CancelTrainedModel',
                'CancelTrainedModelInferenceJob',
                'CreateAudienceModel',
                'CreateConfiguredAudienceModel',
                'CreateConfiguredModelAlgorithm',
                'CreateConfiguredModelAlgorithmAssociation',
                'CreateMLInputChannel',
                'CreateTrainedModel',
                'CreateTrainingDataset',
                'DeleteAudienceGenerationJob',
                'DeleteAudienceModel',
                'DeleteConfiguredAudienceModel',
                'DeleteConfiguredAudienceModelPolicy',
                'DeleteConfiguredModelAlgorithm',
                'DeleteConfiguredModelAlgorithmAssociation',
                'DeleteMLConfiguration',
                'DeleteMLInputChannelData',
                'DeleteTrainedModelOutput',
                'DeleteTrainingDataset',
                'PutMLConfiguration',
                'StartAudienceExportJob',
                'StartAudienceGenerationJob',
                'StartTrainedModelExportJob',
                'StartTrainedModelInferenceJob',
                'UpdateConfiguredAudienceModel'
            ],
            Read: [
                'GetAudienceGenerationJob',
                'GetAudienceModel',
                'GetCollaborationConfiguredModelAlgorithmAssociation',
                'GetCollaborationMLInputChannel',
                'GetCollaborationTrainedModel',
                'GetConfiguredAudienceModel',
                'GetConfiguredAudienceModelPolicy',
                'GetConfiguredModelAlgorithm',
                'GetConfiguredModelAlgorithmAssociation',
                'GetMLConfiguration',
                'GetMLInputChannel',
                'GetTrainedModel',
                'GetTrainedModelInferenceJob',
                'GetTrainingDataset'
            ],
            List: [
                'ListAudienceExportJobs',
                'ListAudienceGenerationJobs',
                'ListAudienceModels',
                'ListCollaborationConfiguredModelAlgorithmAssociations',
                'ListCollaborationMLInputChannels',
                'ListCollaborationTrainedModelExportJobs',
                'ListCollaborationTrainedModelInferenceJobs',
                'ListCollaborationTrainedModels',
                'ListConfiguredAudienceModels',
                'ListConfiguredModelAlgorithmAssociations',
                'ListConfiguredModelAlgorithms',
                'ListMLInputChannels',
                'ListTagsForResource',
                'ListTrainedModelInferenceJobs',
                'ListTrainedModels',
                'ListTrainingDatasets'
            ],
            'Permissions management': [
                'PutConfiguredAudienceModelPolicy'
            ],
            Tagging: [
                'TagResource',
                'UnTagResource'
            ]
        };
    }
}
exports.CleanroomsMl = CleanroomsMl;
//# sourceMappingURL=data:application/json;base64,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