"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudfrontKeyvaluestore = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [cloudfront-keyvaluestore](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncloudfrontkeyvaluestore.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class CloudfrontKeyvaluestore extends shared_1.PolicyStatement {
    /**
     * Grants permission to delete the key value pair specified by the key
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_kvs_DeleteKey.html
     */
    toDeleteKey() {
        return this.to('DeleteKey');
    }
    /**
     * Grants permission to return metadata information about Key Value Store
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_kvs_DescribeKeyValueStore.html
     */
    toDescribeKeyValueStore() {
        return this.to('DescribeKeyValueStore');
    }
    /**
     * Grants permission to return a key value pair
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_kvs_GetKey.html
     */
    toGetKey() {
        return this.to('GetKey');
    }
    /**
     * Grants permission to returns a list of key value pairs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_kvs_ListKeys.html
     */
    toListKeys() {
        return this.to('ListKeys');
    }
    /**
     * Grants permission to create a new key value pair or replace the value of an existing key
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_kvs_PutKey.html
     */
    toPutKey() {
        return this.to('PutKey');
    }
    /**
     * Grants permission to put or delete multiple key value pairs in a single, all-or-nothing operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_kvs_UpdateKeys.html
     */
    toUpdateKeys() {
        return this.to('UpdateKeys');
    }
    /**
     * Adds a resource of type key-value-store to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/kvs-with-functions.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onKeyValueStore(resourceId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudfront::${account ?? this.defaultAccount}:key-value-store/${resourceId}`);
    }
    /**
     * Statement provider for service [cloudfront-keyvaluestore](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncloudfrontkeyvaluestore.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'cloudfront-keyvaluestore';
        this.accessLevelList = {
            Write: [
                'DeleteKey',
                'PutKey',
                'UpdateKeys'
            ],
            Read: [
                'DescribeKeyValueStore',
                'GetKey'
            ],
            List: [
                'ListKeys'
            ]
        };
    }
}
exports.CloudfrontKeyvaluestore = CloudfrontKeyvaluestore;
//# sourceMappingURL=data:application/json;base64,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