import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [synthetics](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncloudwatchsynthetics.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class Synthetics extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to associate a resource with a group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_AssociateResource.html
     */
    toAssociateResource(): this;
    /**
     * Grants permission to create a canary
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_CreateCanary.html
     */
    toCreateCanary(): this;
    /**
     * Grants permission to create a group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_CreateGroup.html
     */
    toCreateGroup(): this;
    /**
     * Grants permission to delete a canary. Amazon Synthetics deletes all the resources except for the Lambda function and the CloudWatch Alarms if you created one
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_DeleteCanary.html
     */
    toDeleteCanary(): this;
    /**
     * Grants permission to delete a group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_DeleteGroup.html
     */
    toDeleteGroup(): this;
    /**
     * Grants permission to list information of all canaries
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifNames()
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_DescribeCanaries.html
     */
    toDescribeCanaries(): this;
    /**
     * Grants permission to list information about the last test run associated with all canaries
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifNames()
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_DescribeCanariesLastRun.html
     */
    toDescribeCanariesLastRun(): this;
    /**
     * Grants permission to list information about Synthetics canary runtime versions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_DescribeRuntimeVersions.html
     */
    toDescribeRuntimeVersions(): this;
    /**
     * Grants permission to disassociate a resource from a group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_DisassociateResource.html
     */
    toDisassociateResource(): this;
    /**
     * Grants permission to view the details of a canary
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_GetCanary.html
     */
    toGetCanary(): this;
    /**
     * Grants permission to list information about all the test runs associated with a canary
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_GetCanaryRuns.html
     */
    toGetCanaryRuns(): this;
    /**
     * Grants permission to view the details of a group
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_GetGroup.html
     */
    toGetGroup(): this;
    /**
     * Grants permission to list information about the associated groups of a canary
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_ListAssociatedGroups.html
     */
    toListAssociatedGroups(): this;
    /**
     * Grants permission to list information about canaries in a group
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_ListGroupResources.html
     */
    toListGroupResources(): this;
    /**
     * Grants permission to list information of all groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_ListGroups.html
     */
    toListGroups(): this;
    /**
     * Grants permission to list all tags and values associated with a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to start a canary, so that Amazon CloudWatch Synthetics starts monitoring a website
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_StartCanary.html
     */
    toStartCanary(): this;
    /**
     * Grants permission to start a canary dry run, so that Amazon CloudWatch Synthetics can execute a test execution of a canary with provided parameters
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_StartCanaryDryRun.html
     */
    toStartCanaryDryRun(): this;
    /**
     * Grants permission to stop a canary
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_StopCanary.html
     */
    toStopCanary(): this;
    /**
     * Grants permission to add one or more tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_TagResource.html
     */
    toTagResource(): this;
    /**
     * Grants permission to remove one or more tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_UntagResource.html
     */
    toUntagResource(): this;
    /**
     * Grants permission to update a canary
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonSynthetics/latest/APIReference/API_UpdateCanary.html
     */
    toUpdateCanary(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type canary to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries.html
     *
     * @param canaryName - Identifier for the canaryName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCanary(canaryName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type group to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Groups.html
     *
     * @param groupId - Identifier for the groupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGroup(groupId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access based on the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateCanary()
     * - .toCreateGroup()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access based on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toAssociateResource()
     * - .toDeleteCanary()
     * - .toDeleteGroup()
     * - .toDisassociateResource()
     * - .toGetCanary()
     * - .toGetCanaryRuns()
     * - .toGetGroup()
     * - .toListAssociatedGroups()
     * - .toListGroupResources()
     * - .toStartCanary()
     * - .toStartCanaryDryRun()
     * - .toStopCanary()
     * - .toUntagResource()
     * - .toUpdateCanary()
     *
     * Applies to resource types:
     * - canary
     * - group
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access based on the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toAssociateResource()
     * - .toCreateCanary()
     * - .toCreateGroup()
     * - .toDeleteCanary()
     * - .toDeleteGroup()
     * - .toDisassociateResource()
     * - .toGetCanary()
     * - .toGetCanaryRuns()
     * - .toGetGroup()
     * - .toListAssociatedGroups()
     * - .toListGroupResources()
     * - .toStartCanary()
     * - .toStartCanaryDryRun()
     * - .toStopCanary()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateCanary()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access based on the name of the canary
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_Restricted.html
     *
     * Applies to actions:
     * - .toDescribeCanaries()
     * - .toDescribeCanariesLastRun()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifNames(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [synthetics](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncloudwatchsynthetics.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
