"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Codecommit = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [codecommit](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscodecommit.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Codecommit extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate an approval rule template with a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_AssociateApprovalRuleTemplateWithRepository.html
     */
    toAssociateApprovalRuleTemplateWithRepository() {
        return this.to('AssociateApprovalRuleTemplateWithRepository');
    }
    /**
     * Grants permission to associate an approval rule template with multiple repositories in a single operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchAssociateApprovalRuleTemplateWithRepositories.html
     */
    toBatchAssociateApprovalRuleTemplateWithRepositories() {
        return this.to('BatchAssociateApprovalRuleTemplateWithRepositories');
    }
    /**
     * Grants permission to get information about multiple merge conflicts when attempting to merge two commits using either the three-way merge or the squash merge option
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchDescribeMergeConflicts.html
     */
    toBatchDescribeMergeConflicts() {
        return this.to('BatchDescribeMergeConflicts');
    }
    /**
     * Grants permission to remove the association between an approval rule template and multiple repositories in a single operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchDisassociateApprovalRuleTemplateFromRepositories.html
     */
    toBatchDisassociateApprovalRuleTemplateFromRepositories() {
        return this.to('BatchDisassociateApprovalRuleTemplateFromRepositories');
    }
    /**
     * Grants permission to return information about one or more commits in an AWS CodeCommit repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchGetCommits.html
     */
    toBatchGetCommits() {
        return this.to('BatchGetCommits');
    }
    /**
     * Grants permission to return information about one or more pull requests in an AWS CodeCommit repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-pr
     */
    toBatchGetPullRequests() {
        return this.to('BatchGetPullRequests');
    }
    /**
     * Grants permission to get information about multiple repositories
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchGetRepositories.html
     */
    toBatchGetRepositories() {
        return this.to('BatchGetRepositories');
    }
    /**
     * Grants permission to cancel the uploading of an archive to a pipeline in AWS CodePipeline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-acp
     */
    toCancelUploadArchive() {
        return this.to('CancelUploadArchive');
    }
    /**
     * Grants permission to create an approval rule template that will automatically create approval rules in pull requests that match the conditions defined in the template; does not grant permission to create approval rules for individual pull requests
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateApprovalRuleTemplate.html
     */
    toCreateApprovalRuleTemplate() {
        return this.to('CreateApprovalRuleTemplate');
    }
    /**
     * Grants permission to create a branch in an AWS CodeCommit repository with this API; does not control Git create branch actions
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateBranch.html
     */
    toCreateBranch() {
        return this.to('CreateBranch');
    }
    /**
     * Grants permission to add, copy, move or update single or multiple files in a branch in an AWS CodeCommit repository, and generate a commit for the changes in the specified branch
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateCommit.html
     */
    toCreateCommit() {
        return this.to('CreateCommit');
    }
    /**
     * Grants permission to create a pull request in the specified repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreatePullRequest.html
     */
    toCreatePullRequest() {
        return this.to('CreatePullRequest');
    }
    /**
     * Grants permission to create an approval rule specific to an individual pull request; does not grant permission to create approval rule templates
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreatePullRequestApprovalRule.html
     */
    toCreatePullRequestApprovalRule() {
        return this.to('CreatePullRequestApprovalRule');
    }
    /**
     * Grants permission to create an AWS CodeCommit repository
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateRepository.html
     */
    toCreateRepository() {
        return this.to('CreateRepository');
    }
    /**
     * Grants permission to create an unreferenced commit that contains the result of merging two commits using either the three-way or the squash merge option; does not control Git merge actions
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateUnreferencedMergeCommit.html
     */
    toCreateUnreferencedMergeCommit() {
        return this.to('CreateUnreferencedMergeCommit');
    }
    /**
     * Grants permission to delete an approval rule template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteApprovalRuleTemplate.html
     */
    toDeleteApprovalRuleTemplate() {
        return this.to('DeleteApprovalRuleTemplate');
    }
    /**
     * Grants permission to delete a branch in an AWS CodeCommit repository with this API; does not control Git delete branch actions
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteBranch.html
     */
    toDeleteBranch() {
        return this.to('DeleteBranch');
    }
    /**
     * Grants permission to delete the content of a comment made on a change, file, or commit in a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteCommentContent.html
     */
    toDeleteCommentContent() {
        return this.to('DeleteCommentContent');
    }
    /**
     * Grants permission to delete a specified file from a specified branch
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteFile.html
     */
    toDeleteFile() {
        return this.to('DeleteFile');
    }
    /**
     * Grants permission to delete approval rule created for a pull request if the rule was not created by an approval rule template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeletePullRequestApprovalRule.html
     */
    toDeletePullRequestApprovalRule() {
        return this.to('DeletePullRequestApprovalRule');
    }
    /**
     * Grants permission to delete an AWS CodeCommit repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteRepository.html
     */
    toDeleteRepository() {
        return this.to('DeleteRepository');
    }
    /**
     * Grants permission to get information about specific merge conflicts when attempting to merge two commits using either the three-way or the squash merge option
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DescribeMergeConflicts.html
     */
    toDescribeMergeConflicts() {
        return this.to('DescribeMergeConflicts');
    }
    /**
     * Grants permission to return information about one or more pull request events
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DescribePullRequestEvents.html
     */
    toDescribePullRequestEvents() {
        return this.to('DescribePullRequestEvents');
    }
    /**
     * Grants permission to remove the association between an approval rule template and a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DisassociateApprovalRuleTemplateFromRepository.html
     */
    toDisassociateApprovalRuleTemplateFromRepository() {
        return this.to('DisassociateApprovalRuleTemplateFromRepository');
    }
    /**
     * Grants permission to evaluate whether a pull request is mergable based on its current approval state and approval rule requirements
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_EvaluatePullRequestApprovalRules.html
     */
    toEvaluatePullRequestApprovalRules() {
        return this.to('EvaluatePullRequestApprovalRules');
    }
    /**
     * Grants permission to return information about an approval rule template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetApprovalRuleTemplate.html
     */
    toGetApprovalRuleTemplate() {
        return this.to('GetApprovalRuleTemplate');
    }
    /**
     * Grants permission to view the encoded content of an individual file in an AWS CodeCommit repository from the AWS CodeCommit console
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetBlob.html
     */
    toGetBlob() {
        return this.to('GetBlob');
    }
    /**
     * Grants permission to get details about a branch in an AWS CodeCommit repository with this API; does not control Git branch actions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetBranch.html
     */
    toGetBranch() {
        return this.to('GetBranch');
    }
    /**
     * Grants permission to get the content of a comment made on a change, file, or commit in a repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetComment.html
     */
    toGetComment() {
        return this.to('GetComment');
    }
    /**
     * Grants permission to get the reactions on a comment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetCommentReactions.html
     */
    toGetCommentReactions() {
        return this.to('GetCommentReactions');
    }
    /**
     * Grants permission to get information about comments made on the comparison between two commits
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetCommentsForComparedCommit.html
     */
    toGetCommentsForComparedCommit() {
        return this.to('GetCommentsForComparedCommit');
    }
    /**
     * Grants permission to get comments made on a pull request
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetCommentsForPullRequest.html
     */
    toGetCommentsForPullRequest() {
        return this.to('GetCommentsForPullRequest');
    }
    /**
     * Grants permission to return information about a commit, including commit message and committer information, with this API; does not control Git log actions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetCommit.html
     */
    toGetCommit() {
        return this.to('GetCommit');
    }
    /**
     * Grants permission to get information about the history of commits in a repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-code
     */
    toGetCommitHistory() {
        return this.to('GetCommitHistory');
    }
    /**
     * Grants permission to get information about the difference between commits in the context of a potential merge
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-pr
     */
    toGetCommitsFromMergeBase() {
        return this.to('GetCommitsFromMergeBase');
    }
    /**
     * Grants permission to view information about the differences between valid commit specifiers such as a branch, tag, HEAD, commit ID, or other fully qualified reference
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetDifferences.html
     */
    toGetDifferences() {
        return this.to('GetDifferences');
    }
    /**
     * Grants permission to return the base-64 encoded contents of a specified file and its metadata
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetFile.html
     */
    toGetFile() {
        return this.to('GetFile');
    }
    /**
     * Grants permission to return the contents of a specified folder in a repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetFolder.html
     */
    toGetFolder() {
        return this.to('GetFolder');
    }
    /**
     * Grants permission to get information about a merge commit created by one of the merge options for pull requests that creates merge commits. Not all merge options create merge commits. This permission does not control Git merge actions
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetMergeCommit.html
     */
    toGetMergeCommit() {
        return this.to('GetMergeCommit');
    }
    /**
     * Grants permission to get information about merge conflicts between the before and after commit IDs for a pull request in a repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetMergeConflicts.html
     */
    toGetMergeConflicts() {
        return this.to('GetMergeConflicts');
    }
    /**
     * Grants permission to get information about merge options for pull requests that can be used to merge two commits; does not control Git merge actions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetMergeOptions.html
     */
    toGetMergeOptions() {
        return this.to('GetMergeOptions');
    }
    /**
     * Grants permission to resolve blobs, trees, and commits to their identifier
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-code
     */
    toGetObjectIdentifier() {
        return this.to('GetObjectIdentifier');
    }
    /**
     * Grants permission to get information about a pull request in a specified repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetPullRequest.html
     */
    toGetPullRequest() {
        return this.to('GetPullRequest');
    }
    /**
     * Grants permission to retrieve the current approvals on an inputted pull request
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetPullRequestApprovalStates.html
     */
    toGetPullRequestApprovalStates() {
        return this.to('GetPullRequestApprovalStates');
    }
    /**
     * Grants permission to retrieve the current override state of a given pull request
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetPullRequestOverrideState.html
     */
    toGetPullRequestOverrideState() {
        return this.to('GetPullRequestOverrideState');
    }
    /**
     * Grants permission to get details about references in an AWS CodeCommit repository; does not control Git reference actions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-code
     */
    toGetReferences() {
        return this.to('GetReferences');
    }
    /**
     * Grants permission to get information about an AWS CodeCommit repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetRepository.html
     */
    toGetRepository() {
        return this.to('GetRepository');
    }
    /**
     * Grants permission to get information about triggers configured for a repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetRepositoryTriggers.html
     */
    toGetRepositoryTriggers() {
        return this.to('GetRepositoryTriggers');
    }
    /**
     * Grants permission to view the contents of a specified tree in an AWS CodeCommit repository from the AWS CodeCommit console
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-code
     */
    toGetTree() {
        return this.to('GetTree');
    }
    /**
     * Grants permission to get status information about an archive upload to a pipeline in AWS CodePipeline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-acp
     */
    toGetUploadArchiveStatus() {
        return this.to('GetUploadArchiveStatus');
    }
    /**
     * Grants permission to pull information from an AWS CodeCommit repository to a local repo
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-git
     */
    toGitPull() {
        return this.to('GitPull');
    }
    /**
     * Grants permission to push information from a local repo to an AWS CodeCommit repository
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-git
     */
    toGitPush() {
        return this.to('GitPush');
    }
    /**
     * Grants permission to list all approval rule templates in an AWS Region for the AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListApprovalRuleTemplates.html
     */
    toListApprovalRuleTemplates() {
        return this.to('ListApprovalRuleTemplates');
    }
    /**
     * Grants permission to list approval rule templates that are associated with a repository
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListAssociatedApprovalRuleTemplatesForRepository.html
     */
    toListAssociatedApprovalRuleTemplatesForRepository() {
        return this.to('ListAssociatedApprovalRuleTemplatesForRepository');
    }
    /**
     * Grants permission to list branches for an AWS CodeCommit repository with this API; does not control Git branch actions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListBranches.html
     */
    toListBranches() {
        return this.to('ListBranches');
    }
    /**
     * Grants permission to list commits and changes to a specified file
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListFileCommitHistory.html
     */
    toListFileCommitHistory() {
        return this.to('ListFileCommitHistory');
    }
    /**
     * Grants permission to list pull requests for a specified repository
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListPullRequests.html
     */
    toListPullRequests() {
        return this.to('ListPullRequests');
    }
    /**
     * Grants permission to list information about AWS CodeCommit repositories in the current Region for your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListRepositories.html
     */
    toListRepositories() {
        return this.to('ListRepositories');
    }
    /**
     * Grants permission to list repositories that are associated with an approval rule template
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListRepositoriesForApprovalRuleTemplate.html
     */
    toListRepositoriesForApprovalRuleTemplate() {
        return this.to('ListRepositoriesForApprovalRuleTemplate');
    }
    /**
     * Grants permission to list the resource attached to a CodeCommit resource ARN
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to merge two commits into the specified destination branch using the fast-forward merge option
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergeBranchesByFastForward.html
     */
    toMergeBranchesByFastForward() {
        return this.to('MergeBranchesByFastForward');
    }
    /**
     * Grants permission to merge two commits into the specified destination branch using the squash merge option
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergeBranchesBySquash.html
     */
    toMergeBranchesBySquash() {
        return this.to('MergeBranchesBySquash');
    }
    /**
     * Grants permission to merge two commits into the specified destination branch using the three-way merge option
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergeBranchesByThreeWay.html
     */
    toMergeBranchesByThreeWay() {
        return this.to('MergeBranchesByThreeWay');
    }
    /**
     * Grants permission to close a pull request and attempt to merge it into the specified destination branch for that pull request at the specified commit using the fast-forward merge option
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergePullRequestByFastForward.html
     */
    toMergePullRequestByFastForward() {
        return this.to('MergePullRequestByFastForward');
    }
    /**
     * Grants permission to close a pull request and attempt to merge it into the specified destination branch for that pull request at the specified commit using the squash merge option
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergePullRequestBySquash.html
     */
    toMergePullRequestBySquash() {
        return this.to('MergePullRequestBySquash');
    }
    /**
     * Grants permission to close a pull request and attempt to merge it into the specified destination branch for that pull request at the specified commit using the three-way merge option
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergePullRequestByThreeWay.html
     */
    toMergePullRequestByThreeWay() {
        return this.to('MergePullRequestByThreeWay');
    }
    /**
     * Grants permission to override all approval rules for a pull request, including approval rules created by a template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_OverridePullRequestApprovalRules.html
     */
    toOverridePullRequestApprovalRules() {
        return this.to('OverridePullRequestApprovalRules');
    }
    /**
     * Grants permission to post a comment on the comparison between two commits
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PostCommentForComparedCommit.html
     */
    toPostCommentForComparedCommit() {
        return this.to('PostCommentForComparedCommit');
    }
    /**
     * Grants permission to post a comment on a pull request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PostCommentForPullRequest.html
     */
    toPostCommentForPullRequest() {
        return this.to('PostCommentForPullRequest');
    }
    /**
     * Grants permission to post a comment in reply to a comment on a comparison between commits or a pull request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PostCommentReply.html
     */
    toPostCommentReply() {
        return this.to('PostCommentReply');
    }
    /**
     * Grants permission to post a reaction on a comment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PutCommentReaction.html
     */
    toPutCommentReaction() {
        return this.to('PutCommentReaction');
    }
    /**
     * Grants permission to add or update a file in a branch in an AWS CodeCommit repository, and generate a commit for the addition in the specified branch
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PutFile.html
     */
    toPutFile() {
        return this.to('PutFile');
    }
    /**
     * Grants permission to create, update, or delete triggers for a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PutRepositoryTriggers.html
     */
    toPutRepositoryTriggers() {
        return this.to('PutRepositoryTriggers');
    }
    /**
     * Grants permission to attach resource tags to a CodeCommit resource ARN
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to test the functionality of repository triggers by sending information to the trigger target
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_TestRepositoryTriggers.html
     */
    toTestRepositoryTriggers() {
        return this.to('TestRepositoryTriggers');
    }
    /**
     * Grants permission to disassociate resource tags from a CodeCommit resource ARN
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the content of approval rule templates; does not grant permission to update content of approval rules created specifically for pull requests
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateApprovalRuleTemplateContent.html
     */
    toUpdateApprovalRuleTemplateContent() {
        return this.to('UpdateApprovalRuleTemplateContent');
    }
    /**
     * Grants permission to update the description of approval rule templates
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateApprovalRuleTemplateDescription.html
     */
    toUpdateApprovalRuleTemplateDescription() {
        return this.to('UpdateApprovalRuleTemplateDescription');
    }
    /**
     * Grants permission to update the name of approval rule templates
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateApprovalRuleTemplateName.html
     */
    toUpdateApprovalRuleTemplateName() {
        return this.to('UpdateApprovalRuleTemplateName');
    }
    /**
     * Grants permission to update the contents of a comment if the identity matches the identity used to create the comment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateComment.html
     */
    toUpdateComment() {
        return this.to('UpdateComment');
    }
    /**
     * Grants permission to change the default branch in an AWS CodeCommit repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateDefaultBranch.html
     */
    toUpdateDefaultBranch() {
        return this.to('UpdateDefaultBranch');
    }
    /**
     * Grants permission to update the content for approval rules created for a specific pull requests; does not grant permission to update approval rule content for rules created with an approval rule template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestApprovalRuleContent.html
     */
    toUpdatePullRequestApprovalRuleContent() {
        return this.to('UpdatePullRequestApprovalRuleContent');
    }
    /**
     * Grants permission to update the approval state for pull requests
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestApprovalState.html
     */
    toUpdatePullRequestApprovalState() {
        return this.to('UpdatePullRequestApprovalState');
    }
    /**
     * Grants permission to update the description of a pull request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestDescription.html
     */
    toUpdatePullRequestDescription() {
        return this.to('UpdatePullRequestDescription');
    }
    /**
     * Grants permission to update the status of a pull request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestStatus.html
     */
    toUpdatePullRequestStatus() {
        return this.to('UpdatePullRequestStatus');
    }
    /**
     * Grants permission to update the title of a pull request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestTitle.html
     */
    toUpdatePullRequestTitle() {
        return this.to('UpdatePullRequestTitle');
    }
    /**
     * Grants permission to change the description of an AWS CodeCommit repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateRepositoryDescription.html
     */
    toUpdateRepositoryDescription() {
        return this.to('UpdateRepositoryDescription');
    }
    /**
     * Grants permission to change the AWS KMS encryption key used to encrypt and decrypt an AWS CodeCommit repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateRepositoryEncryptionKey.html
     */
    toUpdateRepositoryEncryptionKey() {
        return this.to('UpdateRepositoryEncryptionKey');
    }
    /**
     * Grants permission to change the name of an AWS CodeCommit repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateRepositoryName.html
     */
    toUpdateRepositoryName() {
        return this.to('UpdateRepositoryName');
    }
    /**
     * Grants permission to the service role for AWS CodePipeline to upload repository changes into a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-acp
     */
    toUploadArchive() {
        return this.to('UploadArchive');
    }
    /**
     * Adds a resource of type repository to the statement
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control.html#arn-formats
     *
     * @param repositoryName - Identifier for the repositoryName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRepository(repositoryName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codecommit:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:${repositoryName}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateRepository()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - repository
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateRepository()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by Git reference to specified AWS CodeCommit actions
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/how-to-conditional-branch.html
     *
     * Applies to actions:
     * - .toCreateBranch()
     * - .toCreateCommit()
     * - .toCreateUnreferencedMergeCommit()
     * - .toDeleteBranch()
     * - .toDeleteFile()
     * - .toGetMergeCommit()
     * - .toGitPush()
     * - .toMergeBranchesByFastForward()
     * - .toMergeBranchesBySquash()
     * - .toMergeBranchesByThreeWay()
     * - .toMergePullRequestByFastForward()
     * - .toMergePullRequestBySquash()
     * - .toMergePullRequestByThreeWay()
     * - .toPutFile()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReferences(value, operator) {
        return this.if(`References`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [codecommit](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscodecommit.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'codecommit';
        this.accessLevelList = {
            Write: [
                'AssociateApprovalRuleTemplateWithRepository',
                'BatchAssociateApprovalRuleTemplateWithRepositories',
                'BatchDisassociateApprovalRuleTemplateFromRepositories',
                'CreateApprovalRuleTemplate',
                'CreateBranch',
                'CreateCommit',
                'CreatePullRequest',
                'CreatePullRequestApprovalRule',
                'CreateRepository',
                'CreateUnreferencedMergeCommit',
                'DeleteApprovalRuleTemplate',
                'DeleteBranch',
                'DeleteCommentContent',
                'DeleteFile',
                'DeletePullRequestApprovalRule',
                'DeleteRepository',
                'DisassociateApprovalRuleTemplateFromRepository',
                'GitPush',
                'MergeBranchesByFastForward',
                'MergeBranchesBySquash',
                'MergeBranchesByThreeWay',
                'MergePullRequestByFastForward',
                'MergePullRequestBySquash',
                'MergePullRequestByThreeWay',
                'OverridePullRequestApprovalRules',
                'PostCommentForComparedCommit',
                'PostCommentForPullRequest',
                'PostCommentReply',
                'PutCommentReaction',
                'PutFile',
                'PutRepositoryTriggers',
                'TestRepositoryTriggers',
                'UpdateApprovalRuleTemplateContent',
                'UpdateApprovalRuleTemplateDescription',
                'UpdateApprovalRuleTemplateName',
                'UpdateComment',
                'UpdateDefaultBranch',
                'UpdatePullRequestApprovalRuleContent',
                'UpdatePullRequestApprovalState',
                'UpdatePullRequestDescription',
                'UpdatePullRequestStatus',
                'UpdatePullRequestTitle',
                'UpdateRepositoryDescription',
                'UpdateRepositoryEncryptionKey',
                'UpdateRepositoryName',
                'UploadArchive'
            ],
            Read: [
                'BatchDescribeMergeConflicts',
                'BatchGetCommits',
                'BatchGetPullRequests',
                'BatchGetRepositories',
                'CancelUploadArchive',
                'DescribeMergeConflicts',
                'DescribePullRequestEvents',
                'EvaluatePullRequestApprovalRules',
                'GetApprovalRuleTemplate',
                'GetBlob',
                'GetBranch',
                'GetComment',
                'GetCommentReactions',
                'GetCommentsForComparedCommit',
                'GetCommentsForPullRequest',
                'GetCommit',
                'GetCommitHistory',
                'GetCommitsFromMergeBase',
                'GetDifferences',
                'GetFile',
                'GetFolder',
                'GetMergeCommit',
                'GetMergeConflicts',
                'GetMergeOptions',
                'GetObjectIdentifier',
                'GetPullRequest',
                'GetPullRequestApprovalStates',
                'GetPullRequestOverrideState',
                'GetReferences',
                'GetRepository',
                'GetRepositoryTriggers',
                'GetTree',
                'GetUploadArchiveStatus',
                'GitPull'
            ],
            List: [
                'ListApprovalRuleTemplates',
                'ListAssociatedApprovalRuleTemplatesForRepository',
                'ListBranches',
                'ListFileCommitHistory',
                'ListPullRequests',
                'ListRepositories',
                'ListRepositoriesForApprovalRuleTemplate',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Codecommit = Codecommit;
//# sourceMappingURL=data:application/json;base64,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