"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeguruProfiler = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [codeguru-profiler](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncodeguruprofiler.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class CodeguruProfiler extends shared_1.PolicyStatement {
    /**
     * Grants permission to add up to 2 topic ARNs of existing AWS SNS topics to publish notifications
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_AddNotificationChannels.html
     */
    toAddNotificationChannels() {
        return this.to('AddNotificationChannels');
    }
    /**
     * Grants permission to get the frame metric data for a Profiling Group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_BatchGetFrameMetricData.html
     */
    toBatchGetFrameMetricData() {
        return this.to('BatchGetFrameMetricData');
    }
    /**
     * Grants permission to register with the orchestration service and retrieve profiling configuration information, used by agents
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ConfigureAgent.html
     */
    toConfigureAgent() {
        return this.to('ConfigureAgent');
    }
    /**
     * Grants permission to create a profiling group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_CreateProfilingGroup.html
     */
    toCreateProfilingGroup() {
        return this.to('CreateProfilingGroup');
    }
    /**
     * Grants permission to delete a profiling group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_DeleteProfilingGroup.html
     */
    toDeleteProfilingGroup() {
        return this.to('DeleteProfilingGroup');
    }
    /**
     * Grants permission to describe a profiling group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_DescribeProfilingGroup.html
     */
    toDescribeProfilingGroup() {
        return this.to('DescribeProfilingGroup');
    }
    /**
     * Grants permission to get a summary of recent recommendations for each profiling group in the account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetFindingsReportAccountSummary.html
     */
    toGetFindingsReportAccountSummary() {
        return this.to('GetFindingsReportAccountSummary');
    }
    /**
     * Grants permission to get the notification configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetNotificationConfiguration.html
     */
    toGetNotificationConfiguration() {
        return this.to('GetNotificationConfiguration');
    }
    /**
     * Grants permission to get the resource policy associated with the specified Profiling Group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetPolicy.html
     */
    toGetPolicy() {
        return this.to('GetPolicy');
    }
    /**
     * Grants permission to get aggregated profiles for a specific profiling group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetProfile.html
     */
    toGetProfile() {
        return this.to('GetProfile');
    }
    /**
     * Grants permission to get recommendations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetRecommendations.html
     */
    toGetRecommendations() {
        return this.to('GetRecommendations');
    }
    /**
     * Grants permission to list the available recommendations reports for a specific profiling group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ListFindingsReports.html
     */
    toListFindingsReports() {
        return this.to('ListFindingsReports');
    }
    /**
     * Grants permission to list the start times of the available aggregated profiles for a specific profiling group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ListProfileTimes.html
     */
    toListProfileTimes() {
        return this.to('ListProfileTimes');
    }
    /**
     * Grants permission to list profiling groups in the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ListProfilingGroups.html
     */
    toListProfilingGroups() {
        return this.to('ListProfilingGroups');
    }
    /**
     * Grants permission to list tags for a Profiling Group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to submit a profile collected by an agent belonging to a specific profiling group for aggregation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_PostAgentProfile.html
     */
    toPostAgentProfile() {
        return this.to('PostAgentProfile');
    }
    /**
     * Grants permission to update the list of principals allowed for an action group in the resource policy associated with the specified Profiling Group
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_PutPermission.html
     */
    toPutPermission() {
        return this.to('PutPermission');
    }
    /**
     * Grants permission to delete an already configured SNStopic arn from the notification configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_RemoveNotificationChannel.html
     */
    toRemoveNotificationChannel() {
        return this.to('RemoveNotificationChannel');
    }
    /**
     * Grants permission to remove the permission of specified Action Group from the resource policy associated with the specified Profiling Group
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_RemovePermission.html
     */
    toRemovePermission() {
        return this.to('RemovePermission');
    }
    /**
     * Grants permission to submit user feedback for useful or non useful anomaly
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_SubmitFeedback.html
     */
    toSubmitFeedback() {
        return this.to('SubmitFeedback');
    }
    /**
     * Grants permission to add or overwrite tags to a Profiling Group
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a Profiling Group
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a specific profiling group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_UpdateProfilingGroup.html
     */
    toUpdateProfilingGroup() {
        return this.to('UpdateProfilingGroup');
    }
    /**
     * Adds a resource of type ProfilingGroup to the statement
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-ug/working-with-profiling-groups.html
     *
     * @param profilingGroupName - Identifier for the profilingGroupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProfilingGroup(profilingGroupName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codeguru-profiler:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:profilingGroup/${profilingGroupName}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateProfilingGroup()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - ProfilingGroup
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateProfilingGroup()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [codeguru-profiler](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncodeguruprofiler.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'codeguru-profiler';
        this.accessLevelList = {
            Write: [
                'AddNotificationChannels',
                'ConfigureAgent',
                'CreateProfilingGroup',
                'DeleteProfilingGroup',
                'PostAgentProfile',
                'RemoveNotificationChannel',
                'SubmitFeedback',
                'UpdateProfilingGroup'
            ],
            List: [
                'BatchGetFrameMetricData',
                'ListFindingsReports',
                'ListProfileTimes',
                'ListProfilingGroups',
                'ListTagsForResource'
            ],
            Read: [
                'DescribeProfilingGroup',
                'GetFindingsReportAccountSummary',
                'GetNotificationConfiguration',
                'GetPolicy',
                'GetProfile',
                'GetRecommendations'
            ],
            'Permissions management': [
                'PutPermission',
                'RemovePermission'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.CodeguruProfiler = CodeguruProfiler;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29kZWd1cnVwcm9maWxlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImNvZGVndXJ1cHJvZmlsZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0EseUNBQXlEO0FBR3pEOzs7O0dBSUc7QUFDSCxNQUFhLGdCQUFpQixTQUFRLHdCQUFlO0lBR25EOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLENBQUMsQ0FBQztJQUM1QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUJBQXlCO1FBQzlCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxzQkFBc0I7UUFDM0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQ0FBaUM7UUFDdEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlDQUFpQyxDQUFDLENBQUM7SUFDcEQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDhCQUE4QjtRQUNuQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsOEJBQThCLENBQUMsQ0FBQztJQUNqRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksV0FBVztRQUNoQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsV0FBVyxDQUFDLENBQUM7SUFDOUIsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVk7UUFDakIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFlBQVksQ0FBQyxDQUFDO0lBQy9CLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMkJBQTJCO1FBQ2hDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO0lBQzlDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGFBQWE7UUFDbEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFzQ0Q7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksZ0JBQWdCLENBQUMsa0JBQTBCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDdkcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsc0JBQXVCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxtQkFBb0Isa0JBQW1CLEVBQUUsQ0FBQyxDQUFDO0lBQy9MLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxlQUFlLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDM0YsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFtQixNQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2hGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLGdCQUFnQixDQUFDLE1BQWMsRUFBRSxLQUF3QixFQUFFLFFBQTRCO1FBQzVGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBb0IsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksWUFBWSxDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDeEUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2pFLENBQUM7SUFFRDs7O09BR0c7SUFDSCxZQUFZLEtBQWdDO1FBQzFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztRQXRYUixrQkFBYSxHQUFHLG1CQUFtQixDQUFDO1FBMFFqQyxvQkFBZSxHQUFvQjtZQUMzQyxLQUFLLEVBQUU7Z0JBQ0wseUJBQXlCO2dCQUN6QixnQkFBZ0I7Z0JBQ2hCLHNCQUFzQjtnQkFDdEIsc0JBQXNCO2dCQUN0QixrQkFBa0I7Z0JBQ2xCLDJCQUEyQjtnQkFDM0IsZ0JBQWdCO2dCQUNoQixzQkFBc0I7YUFDdkI7WUFDRCxJQUFJLEVBQUU7Z0JBQ0oseUJBQXlCO2dCQUN6QixxQkFBcUI7Z0JBQ3JCLGtCQUFrQjtnQkFDbEIscUJBQXFCO2dCQUNyQixxQkFBcUI7YUFDdEI7WUFDRCxJQUFJLEVBQUU7Z0JBQ0osd0JBQXdCO2dCQUN4QixpQ0FBaUM7Z0JBQ2pDLDhCQUE4QjtnQkFDOUIsV0FBVztnQkFDWCxZQUFZO2dCQUNaLG9CQUFvQjthQUNyQjtZQUNELHdCQUF3QixFQUFFO2dCQUN4QixlQUFlO2dCQUNmLGtCQUFrQjthQUNuQjtZQUNELE9BQU8sRUFBRTtnQkFDUCxhQUFhO2dCQUNiLGVBQWU7YUFDaEI7U0FDRixDQUFDO0lBMkVGLENBQUM7Q0FDRjtBQXpYRCw0Q0F5WEMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBY2Nlc3NMZXZlbExpc3QgfSBmcm9tICcuLi8uLi9zaGFyZWQvYWNjZXNzLWxldmVsJztcbmltcG9ydCB7IFBvbGljeVN0YXRlbWVudCwgT3BlcmF0b3IgfSBmcm9tICcuLi8uLi9zaGFyZWQnO1xuaW1wb3J0IHsgYXdzX2lhbSBhcyBpYW0gfSBmcm9tIFwiYXdzLWNkay1saWJcIjtcblxuLyoqXG4gKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW2NvZGVndXJ1LXByb2ZpbGVyXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hbWF6b25jb2RlZ3VydXByb2ZpbGVyLmh0bWwpLlxuICpcbiAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICovXG5leHBvcnQgY2xhc3MgQ29kZWd1cnVQcm9maWxlciBleHRlbmRzIFBvbGljeVN0YXRlbWVudCB7XG4gIHB1YmxpYyBzZXJ2aWNlUHJlZml4ID0gJ2NvZGVndXJ1LXByb2ZpbGVyJztcblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYWRkIHVwIHRvIDIgdG9waWMgQVJOcyBvZiBleGlzdGluZyBBV1MgU05TIHRvcGljcyB0byBwdWJsaXNoIG5vdGlmaWNhdGlvbnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29kZWd1cnUvbGF0ZXN0L3Byb2ZpbGVyLWFwaS9BUElfQWRkTm90aWZpY2F0aW9uQ2hhbm5lbHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQWRkTm90aWZpY2F0aW9uQ2hhbm5lbHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0FkZE5vdGlmaWNhdGlvbkNoYW5uZWxzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IHRoZSBmcmFtZSBtZXRyaWMgZGF0YSBmb3IgYSBQcm9maWxpbmcgR3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlZ3VydS9sYXRlc3QvcHJvZmlsZXItYXBpL0FQSV9CYXRjaEdldEZyYW1lTWV0cmljRGF0YS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9CYXRjaEdldEZyYW1lTWV0cmljRGF0YSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQmF0Y2hHZXRGcmFtZU1ldHJpY0RhdGEnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZWdpc3RlciB3aXRoIHRoZSBvcmNoZXN0cmF0aW9uIHNlcnZpY2UgYW5kIHJldHJpZXZlIHByb2ZpbGluZyBjb25maWd1cmF0aW9uIGluZm9ybWF0aW9uLCB1c2VkIGJ5IGFnZW50c1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlZ3VydS9sYXRlc3QvcHJvZmlsZXItYXBpL0FQSV9Db25maWd1cmVBZ2VudC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Db25maWd1cmVBZ2VudCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ29uZmlndXJlQWdlbnQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBwcm9maWxpbmcgZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29kZWd1cnUvbGF0ZXN0L3Byb2ZpbGVyLWFwaS9BUElfQ3JlYXRlUHJvZmlsaW5nR3JvdXAuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlUHJvZmlsaW5nR3JvdXAoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZVByb2ZpbGluZ0dyb3VwJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgcHJvZmlsaW5nIGdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZGVndXJ1L2xhdGVzdC9wcm9maWxlci1hcGkvQVBJX0RlbGV0ZVByb2ZpbGluZ0dyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVByb2ZpbGluZ0dyb3VwKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVQcm9maWxpbmdHcm91cCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlc2NyaWJlIGEgcHJvZmlsaW5nIGdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29kZWd1cnUvbGF0ZXN0L3Byb2ZpbGVyLWFwaS9BUElfRGVzY3JpYmVQcm9maWxpbmdHcm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZVByb2ZpbGluZ0dyb3VwKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZVByb2ZpbGluZ0dyb3VwJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGEgc3VtbWFyeSBvZiByZWNlbnQgcmVjb21tZW5kYXRpb25zIGZvciBlYWNoIHByb2ZpbGluZyBncm91cCBpbiB0aGUgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZGVndXJ1L2xhdGVzdC9wcm9maWxlci1hcGkvQVBJX0dldEZpbmRpbmdzUmVwb3J0QWNjb3VudFN1bW1hcnkuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0RmluZGluZ3NSZXBvcnRBY2NvdW50U3VtbWFyeSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0RmluZGluZ3NSZXBvcnRBY2NvdW50U3VtbWFyeScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCB0aGUgbm90aWZpY2F0aW9uIGNvbmZpZ3VyYXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlZ3VydS9sYXRlc3QvcHJvZmlsZXItYXBpL0FQSV9HZXROb3RpZmljYXRpb25Db25maWd1cmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldE5vdGlmaWNhdGlvbkNvbmZpZ3VyYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldE5vdGlmaWNhdGlvbkNvbmZpZ3VyYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgdGhlIHJlc291cmNlIHBvbGljeSBhc3NvY2lhdGVkIHdpdGggdGhlIHNwZWNpZmllZCBQcm9maWxpbmcgR3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlZ3VydS9sYXRlc3QvcHJvZmlsZXItYXBpL0FQSV9HZXRQb2xpY3kuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0UG9saWN5KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRQb2xpY3knKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYWdncmVnYXRlZCBwcm9maWxlcyBmb3IgYSBzcGVjaWZpYyBwcm9maWxpbmcgZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlZ3VydS9sYXRlc3QvcHJvZmlsZXItYXBpL0FQSV9HZXRQcm9maWxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFByb2ZpbGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldFByb2ZpbGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgcmVjb21tZW5kYXRpb25zXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29kZWd1cnUvbGF0ZXN0L3Byb2ZpbGVyLWFwaS9BUElfR2V0UmVjb21tZW5kYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFJlY29tbWVuZGF0aW9ucygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0UmVjb21tZW5kYXRpb25zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCB0aGUgYXZhaWxhYmxlIHJlY29tbWVuZGF0aW9ucyByZXBvcnRzIGZvciBhIHNwZWNpZmljIHByb2ZpbGluZyBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZGVndXJ1L2xhdGVzdC9wcm9maWxlci1hcGkvQVBJX0xpc3RGaW5kaW5nc1JlcG9ydHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEZpbmRpbmdzUmVwb3J0cygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdEZpbmRpbmdzUmVwb3J0cycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGhlIHN0YXJ0IHRpbWVzIG9mIHRoZSBhdmFpbGFibGUgYWdncmVnYXRlZCBwcm9maWxlcyBmb3IgYSBzcGVjaWZpYyBwcm9maWxpbmcgZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlZ3VydS9sYXRlc3QvcHJvZmlsZXItYXBpL0FQSV9MaXN0UHJvZmlsZVRpbWVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RQcm9maWxlVGltZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RQcm9maWxlVGltZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHByb2ZpbGluZyBncm91cHMgaW4gdGhlIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlZ3VydS9sYXRlc3QvcHJvZmlsZXItYXBpL0FQSV9MaXN0UHJvZmlsaW5nR3JvdXBzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RQcm9maWxpbmdHcm91cHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RQcm9maWxpbmdHcm91cHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHRhZ3MgZm9yIGEgUHJvZmlsaW5nIEdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29kZWd1cnUvbGF0ZXN0L3Byb2ZpbGVyLWFwaS9BUElfTGlzdFRhZ3NGb3JSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0VGFnc0ZvclJlc291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0VGFnc0ZvclJlc291cmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3VibWl0IGEgcHJvZmlsZSBjb2xsZWN0ZWQgYnkgYW4gYWdlbnQgYmVsb25naW5nIHRvIGEgc3BlY2lmaWMgcHJvZmlsaW5nIGdyb3VwIGZvciBhZ2dyZWdhdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlZ3VydS9sYXRlc3QvcHJvZmlsZXItYXBpL0FQSV9Qb3N0QWdlbnRQcm9maWxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1Bvc3RBZ2VudFByb2ZpbGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1Bvc3RBZ2VudFByb2ZpbGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgdGhlIGxpc3Qgb2YgcHJpbmNpcGFscyBhbGxvd2VkIGZvciBhbiBhY3Rpb24gZ3JvdXAgaW4gdGhlIHJlc291cmNlIHBvbGljeSBhc3NvY2lhdGVkIHdpdGggdGhlIHNwZWNpZmllZCBQcm9maWxpbmcgR3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlZ3VydS9sYXRlc3QvcHJvZmlsZXItYXBpL0FQSV9QdXRQZXJtaXNzaW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1B1dFBlcm1pc3Npb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1B1dFBlcm1pc3Npb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYW4gYWxyZWFkeSBjb25maWd1cmVkIFNOU3RvcGljIGFybiBmcm9tIHRoZSBub3RpZmljYXRpb24gY29uZmlndXJhdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlZ3VydS9sYXRlc3QvcHJvZmlsZXItYXBpL0FQSV9SZW1vdmVOb3RpZmljYXRpb25DaGFubmVsLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1JlbW92ZU5vdGlmaWNhdGlvbkNoYW5uZWwoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1JlbW92ZU5vdGlmaWNhdGlvbkNoYW5uZWwnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZW1vdmUgdGhlIHBlcm1pc3Npb24gb2Ygc3BlY2lmaWVkIEFjdGlvbiBHcm91cCBmcm9tIHRoZSByZXNvdXJjZSBwb2xpY3kgYXNzb2NpYXRlZCB3aXRoIHRoZSBzcGVjaWZpZWQgUHJvZmlsaW5nIEdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUGVybWlzc2lvbnMgbWFuYWdlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29kZWd1cnUvbGF0ZXN0L3Byb2ZpbGVyLWFwaS9BUElfUmVtb3ZlUGVybWlzc2lvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9SZW1vdmVQZXJtaXNzaW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdSZW1vdmVQZXJtaXNzaW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3VibWl0IHVzZXIgZmVlZGJhY2sgZm9yIHVzZWZ1bCBvciBub24gdXNlZnVsIGFub21hbHlcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29kZWd1cnUvbGF0ZXN0L3Byb2ZpbGVyLWFwaS9BUElfU3VibWl0RmVlZGJhY2suaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU3VibWl0RmVlZGJhY2soKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1N1Ym1pdEZlZWRiYWNrJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYWRkIG9yIG92ZXJ3cml0ZSB0YWdzIHRvIGEgUHJvZmlsaW5nIEdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogVGFnZ2luZ1xuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29kZWd1cnUvbGF0ZXN0L3Byb2ZpbGVyLWFwaS9BUElfVGFnUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVGFnUmVzb3VyY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1RhZ1Jlc291cmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVtb3ZlIHRhZ3MgZnJvbSBhIFByb2ZpbGluZyBHcm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZGVndXJ1L2xhdGVzdC9wcm9maWxlci1hcGkvQVBJX1VudGFnUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVW50YWdSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVW50YWdSZXNvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhIHNwZWNpZmljIHByb2ZpbGluZyBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlZ3VydS9sYXRlc3QvcHJvZmlsZXItYXBpL0FQSV9VcGRhdGVQcm9maWxpbmdHcm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVQcm9maWxpbmdHcm91cCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlUHJvZmlsaW5nR3JvdXAnKTtcbiAgfVxuXG4gIHByb3RlY3RlZCBhY2Nlc3NMZXZlbExpc3Q6IEFjY2Vzc0xldmVsTGlzdCA9IHtcbiAgICBXcml0ZTogW1xuICAgICAgJ0FkZE5vdGlmaWNhdGlvbkNoYW5uZWxzJyxcbiAgICAgICdDb25maWd1cmVBZ2VudCcsXG4gICAgICAnQ3JlYXRlUHJvZmlsaW5nR3JvdXAnLFxuICAgICAgJ0RlbGV0ZVByb2ZpbGluZ0dyb3VwJyxcbiAgICAgICdQb3N0QWdlbnRQcm9maWxlJyxcbiAgICAgICdSZW1vdmVOb3RpZmljYXRpb25DaGFubmVsJyxcbiAgICAgICdTdWJtaXRGZWVkYmFjaycsXG4gICAgICAnVXBkYXRlUHJvZmlsaW5nR3JvdXAnXG4gICAgXSxcbiAgICBMaXN0OiBbXG4gICAgICAnQmF0Y2hHZXRGcmFtZU1ldHJpY0RhdGEnLFxuICAgICAgJ0xpc3RGaW5kaW5nc1JlcG9ydHMnLFxuICAgICAgJ0xpc3RQcm9maWxlVGltZXMnLFxuICAgICAgJ0xpc3RQcm9maWxpbmdHcm91cHMnLFxuICAgICAgJ0xpc3RUYWdzRm9yUmVzb3VyY2UnXG4gICAgXSxcbiAgICBSZWFkOiBbXG4gICAgICAnRGVzY3JpYmVQcm9maWxpbmdHcm91cCcsXG4gICAgICAnR2V0RmluZGluZ3NSZXBvcnRBY2NvdW50U3VtbWFyeScsXG4gICAgICAnR2V0Tm90aWZpY2F0aW9uQ29uZmlndXJhdGlvbicsXG4gICAgICAnR2V0UG9saWN5JyxcbiAgICAgICdHZXRQcm9maWxlJyxcbiAgICAgICdHZXRSZWNvbW1lbmRhdGlvbnMnXG4gICAgXSxcbiAgICAnUGVybWlzc2lvbnMgbWFuYWdlbWVudCc6IFtcbiAgICAgICdQdXRQZXJtaXNzaW9uJyxcbiAgICAgICdSZW1vdmVQZXJtaXNzaW9uJ1xuICAgIF0sXG4gICAgVGFnZ2luZzogW1xuICAgICAgJ1RhZ1Jlc291cmNlJyxcbiAgICAgICdVbnRhZ1Jlc291cmNlJ1xuICAgIF1cbiAgfTtcblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgUHJvZmlsaW5nR3JvdXAgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29kZWd1cnUvbGF0ZXN0L3Byb2ZpbGVyLXVnL3dvcmtpbmctd2l0aC1wcm9maWxpbmctZ3JvdXBzLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHByb2ZpbGluZ0dyb3VwTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBwcm9maWxpbmdHcm91cE5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblByb2ZpbGluZ0dyb3VwKHByb2ZpbGluZ0dyb3VwTmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmNvZGVndXJ1LXByb2ZpbGVyOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpwcm9maWxpbmdHcm91cC8keyBwcm9maWxpbmdHcm91cE5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSBwcmVzZW5jZSBvZiB0YWcga2V5LXZhbHVlIHBhaXJzIGluIHRoZSByZXF1ZXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy1yZXF1ZXN0dGFnXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9DcmVhdGVQcm9maWxpbmdHcm91cCgpXG4gICAqIC0gLnRvVGFnUmVzb3VyY2UoKVxuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzUmVxdWVzdFRhZyh0YWdLZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpSZXF1ZXN0VGFnLyR7IHRhZ0tleSB9YCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGFnIGtleS12YWx1ZSBwYWlycyBhdHRhY2hlZCB0byB0aGUgcmVzb3VyY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXJlc291cmNldGFnXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gUHJvZmlsaW5nR3JvdXBcbiAgICpcbiAgICogQHBhcmFtIHRhZ0tleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1Jlc291cmNlVGFnKHRhZ0tleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlJlc291cmNlVGFnLyR7IHRhZ0tleSB9YCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGhlIHByZXNlbmNlIG9mIHRhZyBrZXlzIGluIHRoZSByZXF1ZXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy10YWdrZXlzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9DcmVhdGVQcm9maWxpbmdHcm91cCgpXG4gICAqIC0gLnRvVGFnUmVzb3VyY2UoKVxuICAgKiAtIC50b1VudGFnUmVzb3VyY2UoKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1RhZ0tleXModmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpUYWdLZXlzYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtjb2RlZ3VydS1wcm9maWxlcl0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYW1hem9uY29kZWd1cnVwcm9maWxlci5odG1sKS5cbiAgICpcbiAgICovXG4gIGNvbnN0cnVjdG9yKHByb3BzPzogaWFtLlBvbGljeVN0YXRlbWVudFByb3BzKSB7XG4gICAgc3VwZXIocHJvcHMpO1xuICB9XG59XG4iXX0=