"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Codepipeline = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [codepipeline](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscodepipeline.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Codepipeline extends shared_1.PolicyStatement {
    /**
     * Grants permission to view information about a specified job and whether that job has been received by the job worker
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_AcknowledgeJob.html
     */
    toAcknowledgeJob() {
        return this.to('AcknowledgeJob');
    }
    /**
     * Grants permission to confirm that a job worker has received the specified job (partner actions only)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_AcknowledgeThirdPartyJob.html
     */
    toAcknowledgeThirdPartyJob() {
        return this.to('AcknowledgeThirdPartyJob');
    }
    /**
     * Grants permission to create a custom action that you can use in the pipelines associated with your AWS account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_CreateCustomActionType.html
     */
    toCreateCustomActionType() {
        return this.to('CreateCustomActionType');
    }
    /**
     * Grants permission to create a uniquely named pipeline
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_CreatePipeline.html
     */
    toCreatePipeline() {
        return this.to('CreatePipeline');
    }
    /**
     * Grants permission to delete a custom action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_DeleteCustomActionType.html
     */
    toDeleteCustomActionType() {
        return this.to('DeleteCustomActionType');
    }
    /**
     * Grants permission to delete a specified pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_DeletePipeline.html
     */
    toDeletePipeline() {
        return this.to('DeletePipeline');
    }
    /**
     * Grants permission to delete a specified webhook
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_DeleteWebhook.html
     */
    toDeleteWebhook() {
        return this.to('DeleteWebhook');
    }
    /**
     * Grants permission to remove the registration of a webhook with the third party specified in its configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_DeregisterWebhookWithThirdParty.html
     */
    toDeregisterWebhookWithThirdParty() {
        return this.to('DeregisterWebhookWithThirdParty');
    }
    /**
     * Grants permission to prevent revisions from transitioning to the next stage in a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_DisableStageTransition.html
     */
    toDisableStageTransition() {
        return this.to('DisableStageTransition');
    }
    /**
     * Grants permission to allow revisions to transition to the next stage in a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_EnableStageTransition.html
     */
    toEnableStageTransition() {
        return this.to('EnableStageTransition');
    }
    /**
     * Grants permission to view information about an action type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_GetActionType.html
     */
    toGetActionType() {
        return this.to('GetActionType');
    }
    /**
     * Grants permission to view information about a job (custom actions only)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_GetJobDetails.html
     */
    toGetJobDetails() {
        return this.to('GetJobDetails');
    }
    /**
     * Grants permission to retrieve information about a pipeline structure
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_GetPipeline.html
     */
    toGetPipeline() {
        return this.to('GetPipeline');
    }
    /**
     * Grants permission to view information about an execution of a pipeline, including details about artifacts, the pipeline execution ID, and the name, version, and status of the pipeline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_GetPipelineExecution.html
     */
    toGetPipelineExecution() {
        return this.to('GetPipelineExecution');
    }
    /**
     * Grants permission to view information about the current state of the stages and actions of a pipeline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_GetPipelineState.html
     */
    toGetPipelineState() {
        return this.to('GetPipelineState');
    }
    /**
     * Grants permission to view the details of a job for a third-party action (partner actions only)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_GetThirdPartyJobDetails.html
     */
    toGetThirdPartyJobDetails() {
        return this.to('GetThirdPartyJobDetails');
    }
    /**
     * Grants permission to list the action executions that have occurred in a pipeline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_ListActionExecutions.html
     */
    toListActionExecutions() {
        return this.to('ListActionExecutions');
    }
    /**
     * Grants permission to list a summary of all the action types available for pipelines in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_ListActionTypes.html
     */
    toListActionTypes() {
        return this.to('ListActionTypes');
    }
    /**
     * Grants permission to list a summary of the most recent executions for a pipeline
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_ListPipelineExecutions.html
     */
    toListPipelineExecutions() {
        return this.to('ListPipelineExecutions');
    }
    /**
     * Grants permission to list a summary of all the pipelines associated with your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_ListPipelines.html
     */
    toListPipelines() {
        return this.to('ListPipelines');
    }
    /**
     * Grants permission to list the rule executions that have occurred in a pipeline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_ListRuleExecutions.html
     */
    toListRuleExecutions() {
        return this.to('ListRuleExecutions');
    }
    /**
     * Grants permission to list a summary of all the rule types available for pipelines in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_ListRuleTypes.html
     */
    toListRuleTypes() {
        return this.to('ListRuleTypes');
    }
    /**
     * Grants permission to list tags for a CodePipeline resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list all of the webhooks associated with your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_ListWebhooks.html
     */
    toListWebhooks() {
        return this.to('ListWebhooks');
    }
    /**
     * Grants permission to resume the pipeline execution by overriding a condition in a stage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_OverrideStageCondition.html
     */
    toOverrideStageCondition() {
        return this.to('OverrideStageCondition');
    }
    /**
     * Grants permission to view information about any jobs for CodePipeline to act on
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PollForJobs.html
     */
    toPollForJobs() {
        return this.to('PollForJobs');
    }
    /**
     * Grants permission to determine whether there are any third-party jobs for a job worker to act on (partner actions only)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PollForThirdPartyJobs.html
     */
    toPollForThirdPartyJobs() {
        return this.to('PollForThirdPartyJobs');
    }
    /**
     * Grants permission to edit actions in a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutActionRevision.html
     */
    toPutActionRevision() {
        return this.to('PutActionRevision');
    }
    /**
     * Grants permission to provide a response (Approved or Rejected) to a manual approval request in CodePipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutApprovalResult.html
     */
    toPutApprovalResult() {
        return this.to('PutApprovalResult');
    }
    /**
     * Grants permission to represent the failure of a job as returned to the pipeline by a job worker (custom actions only)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutJobFailureResult.html
     */
    toPutJobFailureResult() {
        return this.to('PutJobFailureResult');
    }
    /**
     * Grants permission to represent the success of a job as returned to the pipeline by a job worker (custom actions only)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutJobSuccessResult.html
     */
    toPutJobSuccessResult() {
        return this.to('PutJobSuccessResult');
    }
    /**
     * Grants permission to represent the failure of a third-party job as returned to the pipeline by a job worker (partner actions only)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutThirdPartyJobFailureResult.html
     */
    toPutThirdPartyJobFailureResult() {
        return this.to('PutThirdPartyJobFailureResult');
    }
    /**
     * Grants permission to represent the success of a third-party job as returned to the pipeline by a job worker (partner actions only)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutThirdPartyJobSuccessResult.html
     */
    toPutThirdPartyJobSuccessResult() {
        return this.to('PutThirdPartyJobSuccessResult');
    }
    /**
     * Grants permission to create or update a webhook
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutWebhook.html
     */
    toPutWebhook() {
        return this.to('PutWebhook');
    }
    /**
     * Grants permission to register a webhook with the third party specified in its configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_RegisterWebhookWithThirdParty.html
     */
    toRegisterWebhookWithThirdParty() {
        return this.to('RegisterWebhookWithThirdParty');
    }
    /**
     * Grants permission to resume the pipeline execution by retrying the last failed actions in a stage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_RetryStageExecution.html
     */
    toRetryStageExecution() {
        return this.to('RetryStageExecution');
    }
    /**
     * Grants permission to rollback the stage to a previous successful execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_RollbackStage.html
     */
    toRollbackStage() {
        return this.to('RollbackStage');
    }
    /**
     * Grants permission to run the most recent revision through the pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_StartPipelineExecution.html
     */
    toStartPipelineExecution() {
        return this.to('StartPipelineExecution');
    }
    /**
     * Grants permission to stop an in-progress pipeline execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_StopPipelineExecution.html
     */
    toStopPipelineExecution() {
        return this.to('StopPipelineExecution');
    }
    /**
     * Grants permission to tag a CodePipeline resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove a tag from a CodePipeline resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an action type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_UpdateActionType.html
     */
    toUpdateActionType() {
        return this.to('UpdateActionType');
    }
    /**
     * Grants permission to update a pipeline with changes to the structure of the pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_UpdatePipeline.html
     */
    toUpdatePipeline() {
        return this.to('UpdatePipeline');
    }
    /**
     * Adds a resource of type action to the statement
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/iam-access-control-identity-based.html#ACP_ARN_Format
     *
     * @param pipelineName - Identifier for the pipelineName.
     * @param stageName - Identifier for the stageName.
     * @param actionName - Identifier for the actionName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAction(pipelineName, stageName, actionName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codepipeline:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:${pipelineName}/${stageName}/${actionName}`);
    }
    /**
     * Adds a resource of type actiontype to the statement
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/iam-access-control-identity-based.html#ACP_ARN_Format
     *
     * @param owner - Identifier for the owner.
     * @param category - Identifier for the category.
     * @param provider - Identifier for the provider.
     * @param version - Identifier for the version.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onActiontype(owner, category, provider, version, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codepipeline:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:actiontype:${owner}/${category}/${provider}/${version}`);
    }
    /**
     * Adds a resource of type pipeline to the statement
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/iam-access-control-identity-based.html#ACP_ARN_Format
     *
     * @param pipelineName - Identifier for the pipelineName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPipeline(pipelineName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codepipeline:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:${pipelineName}`);
    }
    /**
     * Adds a resource of type stage to the statement
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/iam-access-control-identity-based.html#ACP_ARN_Format
     *
     * @param pipelineName - Identifier for the pipelineName.
     * @param stageName - Identifier for the stageName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStage(pipelineName, stageName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codepipeline:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:${pipelineName}/${stageName}`);
    }
    /**
     * Adds a resource of type webhook to the statement
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/iam-access-control-identity-based.html#ACP_ARN_Format
     *
     * @param webhookName - Identifier for the webhookName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWebhook(webhookName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codepipeline:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:webhook:${webhookName}`);
    }
    /**
     * Filters actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateCustomActionType()
     * - .toCreatePipeline()
     * - .toPutWebhook()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - action
     * - actiontype
     * - pipeline
     * - stage
     * - webhook
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateCustomActionType()
     * - .toCreatePipeline()
     * - .toPutWebhook()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [codepipeline](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscodepipeline.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'codepipeline';
        this.accessLevelList = {
            Write: [
                'AcknowledgeJob',
                'AcknowledgeThirdPartyJob',
                'CreateCustomActionType',
                'CreatePipeline',
                'DeleteCustomActionType',
                'DeletePipeline',
                'DeleteWebhook',
                'DeregisterWebhookWithThirdParty',
                'DisableStageTransition',
                'EnableStageTransition',
                'OverrideStageCondition',
                'PollForJobs',
                'PollForThirdPartyJobs',
                'PutActionRevision',
                'PutApprovalResult',
                'PutJobFailureResult',
                'PutJobSuccessResult',
                'PutThirdPartyJobFailureResult',
                'PutThirdPartyJobSuccessResult',
                'PutWebhook',
                'RegisterWebhookWithThirdParty',
                'RetryStageExecution',
                'RollbackStage',
                'StartPipelineExecution',
                'StopPipelineExecution',
                'UpdateActionType',
                'UpdatePipeline'
            ],
            Read: [
                'GetActionType',
                'GetJobDetails',
                'GetPipeline',
                'GetPipelineExecution',
                'GetPipelineState',
                'GetThirdPartyJobDetails',
                'ListActionExecutions',
                'ListActionTypes',
                'ListRuleExecutions',
                'ListRuleTypes',
                'ListTagsForResource'
            ],
            List: [
                'ListPipelineExecutions',
                'ListPipelines',
                'ListWebhooks'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Codepipeline = Codepipeline;
//# sourceMappingURL=data:application/json;base64,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