"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodestarNotifications = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [codestar-notifications](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscodestarnotifications.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class CodestarNotifications extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a notification rule for a resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifNotificationsForResource()
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_CreateNotificationRule.html
     */
    toCreateNotificationRule() {
        return this.to('CreateNotificationRule');
    }
    /**
     * Grants permission to delete a notification rule for a resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifNotificationsForResource()
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_DeleteNotificationRule.html
     */
    toDeleteNotificationRule() {
        return this.to('DeleteNotificationRule');
    }
    /**
     * Grants permission to delete a target for a notification rule
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_DeleteTarget.html
     */
    toDeleteTarget() {
        return this.to('DeleteTarget');
    }
    /**
     * Grants permission to get information about a notification rule
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifNotificationsForResource()
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_DescribeNotificationRule.html
     */
    toDescribeNotificationRule() {
        return this.to('DescribeNotificationRule');
    }
    /**
     * Grants permission to list notifications event types
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_ListEventTypes.html
     */
    toListEventTypes() {
        return this.to('ListEventTypes');
    }
    /**
     * Grants permission to list notification rules in an AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_ListNotificationRules.html
     */
    toListNotificationRules() {
        return this.to('ListNotificationRules');
    }
    /**
     * Grants permission to list the tags attached to a notification rule resource ARN
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list the notification rule targets for an AWS account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_ListTargets.html
     */
    toListTargets() {
        return this.to('ListTargets');
    }
    /**
     * Grants permission to create an association between a notification rule and an Amazon SNS topic
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifNotificationsForResource()
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_Subscribe.html
     */
    toSubscribe() {
        return this.to('Subscribe');
    }
    /**
     * Grants permission to attach resource tags to a notification rule resource ARN
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove an association between a notification rule and an Amazon SNS topic
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifNotificationsForResource()
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_Unsubscribe.html
     */
    toUnsubscribe() {
        return this.to('Unsubscribe');
    }
    /**
     * Grants permission to disassociate resource tags from a notification rule resource ARN
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to change a notification rule for a resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifNotificationsForResource()
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/APIReference/API_UpdateNotificationRule.html
     */
    toUpdateNotificationRule() {
        return this.to('UpdateNotificationRule');
    }
    /**
     * Adds a resource of type notificationrule to the statement
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/userguide/security_iam_service-with-iam.html
     *
     * @param notificationRuleId - Identifier for the notificationRuleId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onNotificationrule(notificationRuleId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codestar-notifications:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:notificationrule/${notificationRuleId}`);
    }
    /**
     * Filters actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateNotificationRule()
     * - .toDeleteNotificationRule()
     * - .toDeleteTarget()
     * - .toDescribeNotificationRule()
     * - .toListTagsForResource()
     * - .toListTargets()
     * - .toSubscribe()
     * - .toTagResource()
     * - .toUnsubscribe()
     * - .toUpdateNotificationRule()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toCreateNotificationRule()
     * - .toDeleteNotificationRule()
     * - .toDescribeNotificationRule()
     * - .toSubscribe()
     * - .toTagResource()
     * - .toUnsubscribe()
     * - .toUntagResource()
     * - .toUpdateNotificationRule()
     *
     * Applies to resource types:
     * - notificationrule
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateNotificationRule()
     * - .toDeleteNotificationRule()
     * - .toDeleteTarget()
     * - .toDescribeNotificationRule()
     * - .toListTagsForResource()
     * - .toListTargets()
     * - .toSubscribe()
     * - .toTagResource()
     * - .toUnsubscribe()
     * - .toUntagResource()
     * - .toUpdateNotificationRule()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the ARN of the resource for which notifications are configured
     *
     * https://docs.aws.amazon.com/codestar-notifications/latest/userguide/security_iam_id-based-policy-examples.html
     *
     * Applies to actions:
     * - .toCreateNotificationRule()
     * - .toDeleteNotificationRule()
     * - .toDescribeNotificationRule()
     * - .toSubscribe()
     * - .toUnsubscribe()
     * - .toUpdateNotificationRule()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifNotificationsForResource(value, operator) {
        return this.if(`NotificationsForResource`, value, operator ?? 'ArnLike');
    }
    /**
     * Statement provider for service [codestar-notifications](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscodestarnotifications.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'codestar-notifications';
        this.accessLevelList = {
            Write: [
                'CreateNotificationRule',
                'DeleteNotificationRule',
                'DeleteTarget',
                'Subscribe',
                'Unsubscribe',
                'UpdateNotificationRule'
            ],
            Read: [
                'DescribeNotificationRule'
            ],
            List: [
                'ListEventTypes',
                'ListNotificationRules',
                'ListTagsForResource',
                'ListTargets'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.CodestarNotifications = CodestarNotifications;
//# sourceMappingURL=data:application/json;base64,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