"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Comprehend = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [comprehend](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncomprehend.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Comprehend extends shared_1.PolicyStatement {
    /**
     * Grants permission to detect the language or languages present in the list of text documents
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_BatchDetectDominantLanguage.html
     */
    toBatchDetectDominantLanguage() {
        return this.to('BatchDetectDominantLanguage');
    }
    /**
     * Grants permission to detect the named entities ("People", "Places", "Locations", etc) within the given list of text documents
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_BatchDetectEntities.html
     */
    toBatchDetectEntities() {
        return this.to('BatchDetectEntities');
    }
    /**
     * Grants permission to detect the phrases in the list of text documents that are most indicative of the content
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_BatchDetectKeyPhrases.html
     */
    toBatchDetectKeyPhrases() {
        return this.to('BatchDetectKeyPhrases');
    }
    /**
     * Grants permission to detect the sentiment of a text in the list of documents (Positive, Negative, Neutral, or Mixed)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_BatchDetectSentiment.html
     */
    toBatchDetectSentiment() {
        return this.to('BatchDetectSentiment');
    }
    /**
     * Grants permission to detect syntactic information (like Part of Speech, Tokens) in a list of text documents
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_BatchDetectSyntax.html
     */
    toBatchDetectSyntax() {
        return this.to('BatchDetectSyntax');
    }
    /**
     * Grants permission to detect the sentiments associated with specific entities (such as brands or products) within the given list of text documents
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_BatchDetectTargetedSentiment.html
     */
    toBatchDetectTargetedSentiment() {
        return this.to('BatchDetectTargetedSentiment');
    }
    /**
     * Grants permission to create a new document classification request to analyze a single document in real-time, using a previously created and trained custom model and an endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_ClassifyDocument.html
     */
    toClassifyDocument() {
        return this.to('ClassifyDocument');
    }
    /**
     * Grants permission to classify the personally identifiable information within given documents in real-time
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_ContainsPiiEntities.html
     */
    toContainsPiiEntities() {
        return this.to('ContainsPiiEntities');
    }
    /**
     * Grants permission to create a new dataset within a flywheel
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_CreateDataset.html
     */
    toCreateDataset() {
        return this.to('CreateDataset');
    }
    /**
     * Grants permission to create a new document classifier that you can use to categorize documents
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifVolumeKmsKey()
     * - .ifModelKmsKey()
     * - .ifOutputKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_CreateDocumentClassifier.html
     */
    toCreateDocumentClassifier() {
        return this.to('CreateDocumentClassifier');
    }
    /**
     * Grants permission to create a model-specific endpoint for synchronous inference for a previously trained custom model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_CreateEndpoint.html
     */
    toCreateEndpoint() {
        return this.to('CreateEndpoint');
    }
    /**
     * Grants permission to create an entity recognizer using submitted files
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifVolumeKmsKey()
     * - .ifModelKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_CreateEntityRecognizer.html
     */
    toCreateEntityRecognizer() {
        return this.to('CreateEntityRecognizer');
    }
    /**
     * Grants permission to create a new flywheel that you can use to train model versions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_CreateFlywheel.html
     */
    toCreateFlywheel() {
        return this.to('CreateFlywheel');
    }
    /**
     * Grants permission to delete a previously created document classifier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_DeleteDocumentClassifier.html
     */
    toDeleteDocumentClassifier() {
        return this.to('DeleteDocumentClassifier');
    }
    /**
     * Grants permission to delete a model-specific endpoint for a previously-trained custom model. All endpoints must be deleted in order for the model to be deleted
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_DeleteEndpoint.html
     */
    toDeleteEndpoint() {
        return this.to('DeleteEndpoint');
    }
    /**
     * Grants permission to delete a submitted entity recognizer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_DeleteEntityRecognizer.html
     */
    toDeleteEntityRecognizer() {
        return this.to('DeleteEntityRecognizer');
    }
    /**
     * Grants permission to Delete a flywheel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_DeleteFlywheel.html
     */
    toDeleteFlywheel() {
        return this.to('DeleteFlywheel');
    }
    /**
     * Grants permission to remove policy on resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to get the properties associated with a dataset
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_DescribeDataset.html
     */
    toDescribeDataset() {
        return this.to('DescribeDataset');
    }
    /**
     * Grants permission to get the properties associated with a document classification job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_DescribeDocumentClassificationJob.html
     */
    toDescribeDocumentClassificationJob() {
        return this.to('DescribeDocumentClassificationJob');
    }
    /**
     * Grants permission to get the properties associated with a document classifier
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_DescribeDocumentClassifier.html
     */
    toDescribeDocumentClassifier() {
        return this.to('DescribeDocumentClassifier');
    }
    /**
     * Grants permission to get the properties associated with a dominant language detection job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_DescribeDominantLanguageDetectionJob.html
     */
    toDescribeDominantLanguageDetectionJob() {
        return this.to('DescribeDominantLanguageDetectionJob');
    }
    /**
     * Grants permission to get the properties associated with a specific endpoint. Use this operation to get the status of an endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_DescribeEndpoint.html
     */
    toDescribeEndpoint() {
        return this.to('DescribeEndpoint');
    }
    /**
     * Grants permission to get the properties associated with an entities detection job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_DescribeEntitiesDetectionJob.html
     */
    toDescribeEntitiesDetectionJob() {
        return this.to('DescribeEntitiesDetectionJob');
    }
    /**
     * Grants permission to provide details about an entity recognizer including status, S3 buckets containing training data, recognizer metadata, metrics, and so on
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_DescribeEntityRecognizer.html
     */
    toDescribeEntityRecognizer() {
        return this.to('DescribeEntityRecognizer');
    }
    /**
     * Grants permission to get the properties associated with an Events detection job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_DescribeEventsDetectionJob.html
     */
    toDescribeEventsDetectionJob() {
        return this.to('DescribeEventsDetectionJob');
    }
    /**
     * Grants permission to get the properties associated with a flywheel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_DescribeFlywheel.html
     */
    toDescribeFlywheel() {
        return this.to('DescribeFlywheel');
    }
    /**
     * Grants permission to get the properties associated with a flywheel iteration for a flywheel
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifFlywheelIterationId()
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_DescribeFlywheelIteration.html
     */
    toDescribeFlywheelIteration() {
        return this.to('DescribeFlywheelIteration');
    }
    /**
     * Grants permission to get the properties associated with a key phrases detection job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_DescribeKeyPhrasesDetectionJob.html
     */
    toDescribeKeyPhrasesDetectionJob() {
        return this.to('DescribeKeyPhrasesDetectionJob');
    }
    /**
     * Grants permission to get the properties associated with a PII entities detection job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_DescribePiiEntitiesDetectionJob.html
     */
    toDescribePiiEntitiesDetectionJob() {
        return this.to('DescribePiiEntitiesDetectionJob');
    }
    /**
     * Grants permission to read attached policy on resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_DescribeResourcePolicy.html
     */
    toDescribeResourcePolicy() {
        return this.to('DescribeResourcePolicy');
    }
    /**
     * Grants permission to get the properties associated with a sentiment detection job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_DescribeSentimentDetectionJob.html
     */
    toDescribeSentimentDetectionJob() {
        return this.to('DescribeSentimentDetectionJob');
    }
    /**
     * Grants permission to get the properties associated with a targeted sentiment detection job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_DescribeTargetedSentimentDetectionJob.html
     */
    toDescribeTargetedSentimentDetectionJob() {
        return this.to('DescribeTargetedSentimentDetectionJob');
    }
    /**
     * Grants permission to get the properties associated with a topic detection job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_DescribeTopicsDetectionJob.html
     */
    toDescribeTopicsDetectionJob() {
        return this.to('DescribeTopicsDetectionJob');
    }
    /**
     * Grants permission to detect the language or languages present in the text
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_DetectDominantLanguage.html
     */
    toDetectDominantLanguage() {
        return this.to('DetectDominantLanguage');
    }
    /**
     * Grants permission to detect the named entities ("People", "Places", "Locations", etc) within the given text document
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_DetectEntities.html
     */
    toDetectEntities() {
        return this.to('DetectEntities');
    }
    /**
     * Grants permission to detect the phrases in the text that are most indicative of the content
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_DetectKeyPhrases.html
     */
    toDetectKeyPhrases() {
        return this.to('DetectKeyPhrases');
    }
    /**
     * Grants permission to detect the personally identifiable information entities ("Name", "SSN", "PIN", etc) within the given text document
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_DetectPiiEntities.html
     */
    toDetectPiiEntities() {
        return this.to('DetectPiiEntities');
    }
    /**
     * Grants permission to detect the sentiment of a text in a document (Positive, Negative, Neutral, or Mixed)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_DetectSentiment.html
     */
    toDetectSentiment() {
        return this.to('DetectSentiment');
    }
    /**
     * Grants permission to detect syntactic information (like Part of Speech, Tokens) in a text document
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_DetectSyntax.html
     */
    toDetectSyntax() {
        return this.to('DetectSyntax');
    }
    /**
     * Grants permission to detect the sentiments associated with specific entities (such as brands or products) in a document
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_DetectTargetedSentiment.html
     */
    toDetectTargetedSentiment() {
        return this.to('DetectTargetedSentiment');
    }
    /**
     * Grants permission to detect toxic content within the given list of text segments
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_DetectToxicContent.html
     */
    toDetectToxicContent() {
        return this.to('DetectToxicContent');
    }
    /**
     * Grants permission to import a trained Comprehend model
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifModelKmsKey()
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_ImportModel.html
     */
    toImportModel() {
        return this.to('ImportModel');
    }
    /**
     * Grants permission to get a list of the Datasets associated with a flywheel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_ListDatasets.html
     */
    toListDatasets() {
        return this.to('ListDatasets');
    }
    /**
     * Grants permission to get a list of the document classification jobs that you have submitted
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_ListDocumentClassificationJobs.html
     */
    toListDocumentClassificationJobs() {
        return this.to('ListDocumentClassificationJobs');
    }
    /**
     * Grants permission to get a list of summaries of the document classifiers that you have created
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_ListDocumentClassifierSummaries.html
     */
    toListDocumentClassifierSummaries() {
        return this.to('ListDocumentClassifierSummaries');
    }
    /**
     * Grants permission to get a list of the document classifiers that you have created
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_ListDocumentClassifiers.html
     */
    toListDocumentClassifiers() {
        return this.to('ListDocumentClassifiers');
    }
    /**
     * Grants permission to get a list of the dominant language detection jobs that you have submitted
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_ListDominantLanguageDetectionJobs.html
     */
    toListDominantLanguageDetectionJobs() {
        return this.to('ListDominantLanguageDetectionJobs');
    }
    /**
     * Grants permission to get a list of all existing endpoints that you've created
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_ListEndpoints.html
     */
    toListEndpoints() {
        return this.to('ListEndpoints');
    }
    /**
     * Grants permission to get a list of the entity detection jobs that you have submitted
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_ListEntitiesDetectionJobs.html
     */
    toListEntitiesDetectionJobs() {
        return this.to('ListEntitiesDetectionJobs');
    }
    /**
     * Grants permission to get a list of summaries for the entity recognizers that you have created
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_ListEntityRecognizerSummaries.html
     */
    toListEntityRecognizerSummaries() {
        return this.to('ListEntityRecognizerSummaries');
    }
    /**
     * Grants permission to get a list of the properties of all entity recognizers that you created, including recognizers currently in training
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_ListEntityRecognizers.html
     */
    toListEntityRecognizers() {
        return this.to('ListEntityRecognizers');
    }
    /**
     * Grants permission to get a list of Events detection jobs that you have submitted
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_ListEventsDetectionJobs.html
     */
    toListEventsDetectionJobs() {
        return this.to('ListEventsDetectionJobs');
    }
    /**
     * Grants permission to get a list of iterations associated for a flywheel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_ListFlywheelIterationHistory.html
     */
    toListFlywheelIterationHistory() {
        return this.to('ListFlywheelIterationHistory');
    }
    /**
     * Grants permission to get a list of the flywheels that you have created
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_ListFlywheels.html
     */
    toListFlywheels() {
        return this.to('ListFlywheels');
    }
    /**
     * Grants permission to get a list of key phrase detection jobs that you have submitted
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_ListKeyPhrasesDetectionJobs.html
     */
    toListKeyPhrasesDetectionJobs() {
        return this.to('ListKeyPhrasesDetectionJobs');
    }
    /**
     * Grants permission to get a list of PII entities detection jobs that you have submitted
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_ListPiiEntitiesDetectionJobs.html
     */
    toListPiiEntitiesDetectionJobs() {
        return this.to('ListPiiEntitiesDetectionJobs');
    }
    /**
     * Grants permission to get a list of sentiment detection jobs that you have submitted
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_ListSentimentDetectionJobs.html
     */
    toListSentimentDetectionJobs() {
        return this.to('ListSentimentDetectionJobs');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to get a list of targeted sentiment detection jobs that you have submitted
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_ListTargetedSentimentDetectionJobs.html
     */
    toListTargetedSentimentDetectionJobs() {
        return this.to('ListTargetedSentimentDetectionJobs');
    }
    /**
     * Grants permission to get a list of the topic detection jobs that you have submitted
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_ListTopicsDetectionJobs.html
     */
    toListTopicsDetectionJobs() {
        return this.to('ListTopicsDetectionJobs');
    }
    /**
     * Grants permission to attach policy to resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to start an asynchronous document classification job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_StartDocumentClassificationJob.html
     */
    toStartDocumentClassificationJob() {
        return this.to('StartDocumentClassificationJob');
    }
    /**
     * Grants permission to start an asynchronous dominant language detection job for a collection of documents
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifVolumeKmsKey()
     * - .ifOutputKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_StartDominantLanguageDetectionJob.html
     */
    toStartDominantLanguageDetectionJob() {
        return this.to('StartDominantLanguageDetectionJob');
    }
    /**
     * Grants permission to start an asynchronous entity detection job for a collection of documents
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_StartEntitiesDetectionJob.html
     */
    toStartEntitiesDetectionJob() {
        return this.to('StartEntitiesDetectionJob');
    }
    /**
     * Grants permission to start an asynchronous Events detection job for a collection of documents
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifOutputKmsKey()
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_StartEventsDetectionJob.html
     */
    toStartEventsDetectionJob() {
        return this.to('StartEventsDetectionJob');
    }
    /**
     * Grants permission to start a flywheel iteration for a flywheel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_StartFlywheelIteration.html
     */
    toStartFlywheelIteration() {
        return this.to('StartFlywheelIteration');
    }
    /**
     * Grants permission to start an asynchronous key phrase detection job for a collection of documents
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifVolumeKmsKey()
     * - .ifOutputKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_StartKeyPhrasesDetectionJob.html
     */
    toStartKeyPhrasesDetectionJob() {
        return this.to('StartKeyPhrasesDetectionJob');
    }
    /**
     * Grants permission to start an asynchronous PII entities detection job for a collection of documents
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifOutputKmsKey()
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_StartPiiEntitiesDetectionJob.html
     */
    toStartPiiEntitiesDetectionJob() {
        return this.to('StartPiiEntitiesDetectionJob');
    }
    /**
     * Grants permission to start an asynchronous sentiment detection job for a collection of documents
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifVolumeKmsKey()
     * - .ifOutputKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_StartSentimentDetectionJob.html
     */
    toStartSentimentDetectionJob() {
        return this.to('StartSentimentDetectionJob');
    }
    /**
     * Grants permission to start an asynchronous targeted sentiment detection job for a collection of documents
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifVolumeKmsKey()
     * - .ifOutputKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_StartTargetedSentimentDetectionJob.html
     */
    toStartTargetedSentimentDetectionJob() {
        return this.to('StartTargetedSentimentDetectionJob');
    }
    /**
     * Grants permission to start an asynchronous job to detect the most common topics in the collection of documents and the phrases associated with each topic
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifVolumeKmsKey()
     * - .ifOutputKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_StartTopicsDetectionJob.html
     */
    toStartTopicsDetectionJob() {
        return this.to('StartTopicsDetectionJob');
    }
    /**
     * Grants permission to stop a dominant language detection job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_StopDominantLanguageDetectionJob.html
     */
    toStopDominantLanguageDetectionJob() {
        return this.to('StopDominantLanguageDetectionJob');
    }
    /**
     * Grants permission to stop an entity detection job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_StopEntitiesDetectionJob.html
     */
    toStopEntitiesDetectionJob() {
        return this.to('StopEntitiesDetectionJob');
    }
    /**
     * Grants permission to stop an Events detection job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_StopEventsDetectionJob.html
     */
    toStopEventsDetectionJob() {
        return this.to('StopEventsDetectionJob');
    }
    /**
     * Grants permission to stop a key phrase detection job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_StopKeyPhrasesDetectionJob.html
     */
    toStopKeyPhrasesDetectionJob() {
        return this.to('StopKeyPhrasesDetectionJob');
    }
    /**
     * Grants permission to stop a PII entities detection job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_StopPiiEntitiesDetectionJob.html
     */
    toStopPiiEntitiesDetectionJob() {
        return this.to('StopPiiEntitiesDetectionJob');
    }
    /**
     * Grants permission to stop a sentiment detection job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_StopSentimentDetectionJob.html
     */
    toStopSentimentDetectionJob() {
        return this.to('StopSentimentDetectionJob');
    }
    /**
     * Grants permission to stop a targeted sentiment detection job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_StopTargetedSentimentDetectionJob.html
     */
    toStopTargetedSentimentDetectionJob() {
        return this.to('StopTargetedSentimentDetectionJob');
    }
    /**
     * Grants permission to stop a previously created document classifier training job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_StopTrainingDocumentClassifier.html
     */
    toStopTrainingDocumentClassifier() {
        return this.to('StopTrainingDocumentClassifier');
    }
    /**
     * Grants permission to stop a previously created entity recognizer training job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_StopTrainingEntityRecognizer.html
     */
    toStopTrainingEntityRecognizer() {
        return this.to('StopTrainingEntityRecognizer');
    }
    /**
     * Grants permission to tag a resource with given key value pairs
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a resource with given key
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update information about the specified endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_UpdateEndpoint.html
     */
    toUpdateEndpoint() {
        return this.to('UpdateEndpoint');
    }
    /**
     * Grants permission to Update a flywheel's configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_UpdateFlywheel.html
     */
    toUpdateFlywheel() {
        return this.to('UpdateFlywheel');
    }
    /**
     * Adds a resource of type targeted-sentiment-detection-job to the statement
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_StartTargetedSentimentDetectionJob.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTargetedSentimentDetectionJob(jobId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:comprehend:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:targeted-sentiment-detection-job/${jobId}`);
    }
    /**
     * Adds a resource of type document-classifier to the statement
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/how-document-classification-training.html
     *
     * @param documentClassifierName - Identifier for the documentClassifierName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDocumentClassifier(documentClassifierName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:comprehend:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:document-classifier/${documentClassifierName}`);
    }
    /**
     * Adds a resource of type document-classifier-endpoint to the statement
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/manage-endpoints.html
     *
     * @param documentClassifierEndpointName - Identifier for the documentClassifierEndpointName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDocumentClassifierEndpoint(documentClassifierEndpointName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:comprehend:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:document-classifier-endpoint/${documentClassifierEndpointName}`);
    }
    /**
     * Adds a resource of type entity-recognizer to the statement
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/training-recognizers.html
     *
     * @param entityRecognizerName - Identifier for the entityRecognizerName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEntityRecognizer(entityRecognizerName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:comprehend:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:entity-recognizer/${entityRecognizerName}`);
    }
    /**
     * Adds a resource of type entity-recognizer-endpoint to the statement
     *
     * https://docs.aws.amazon.com/comprehend/latest/dg/manage-endpoints.html
     *
     * @param entityRecognizerEndpointName - Identifier for the entityRecognizerEndpointName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEntityRecognizerEndpoint(entityRecognizerEndpointName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:comprehend:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:entity-recognizer-endpoint/${entityRecognizerEndpointName}`);
    }
    /**
     * Adds a resource of type dominant-language-detection-job to the statement
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_StartDominantLanguageDetectionJob.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDominantLanguageDetectionJob(jobId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:comprehend:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:dominant-language-detection-job/${jobId}`);
    }
    /**
     * Adds a resource of type entities-detection-job to the statement
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_StartEntitiesDetectionJob.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEntitiesDetectionJob(jobId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:comprehend:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:entities-detection-job/${jobId}`);
    }
    /**
     * Adds a resource of type pii-entities-detection-job to the statement
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_StartPiiEntitiesDetectionJob.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPiiEntitiesDetectionJob(jobId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:comprehend:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:pii-entities-detection-job/${jobId}`);
    }
    /**
     * Adds a resource of type events-detection-job to the statement
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_StartEventsDetectionJob.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEventsDetectionJob(jobId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:comprehend:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:events-detection-job/${jobId}`);
    }
    /**
     * Adds a resource of type key-phrases-detection-job to the statement
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_StartKeyPhrasesDetectionJob.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onKeyPhrasesDetectionJob(jobId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:comprehend:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:key-phrases-detection-job/${jobId}`);
    }
    /**
     * Adds a resource of type sentiment-detection-job to the statement
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_StartSentimentDetectionJob.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSentimentDetectionJob(jobId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:comprehend:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:sentiment-detection-job/${jobId}`);
    }
    /**
     * Adds a resource of type topics-detection-job to the statement
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_StartTopicsDetectionJob.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTopicsDetectionJob(jobId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:comprehend:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:topics-detection-job/${jobId}`);
    }
    /**
     * Adds a resource of type document-classification-job to the statement
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_StartDocumentClassificationJob.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDocumentClassificationJob(jobId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:comprehend:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:document-classification-job/${jobId}`);
    }
    /**
     * Adds a resource of type flywheel to the statement
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_CreateFlywheel.html
     *
     * @param flywheelName - Identifier for the flywheelName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFlywheel(flywheelName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:comprehend:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:flywheel/${flywheelName}`);
    }
    /**
     * Adds a resource of type flywheel-dataset to the statement
     *
     * https://docs.aws.amazon.com/comprehend/latest/APIReference/API_CreateDataset.html
     *
     * @param flywheelName - Identifier for the flywheelName.
     * @param datasetName - Identifier for the datasetName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFlywheelDataset(flywheelName, datasetName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:comprehend:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:flywheel/${flywheelName}/dataset/${datasetName}`);
    }
    /**
     * Filters access by requiring tag values present in a resource creation request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * Applies to actions:
     * - .toCreateDataset()
     * - .toCreateDocumentClassifier()
     * - .toCreateEndpoint()
     * - .toCreateEntityRecognizer()
     * - .toCreateFlywheel()
     * - .toImportModel()
     * - .toStartDocumentClassificationJob()
     * - .toStartDominantLanguageDetectionJob()
     * - .toStartEntitiesDetectionJob()
     * - .toStartEventsDetectionJob()
     * - .toStartKeyPhrasesDetectionJob()
     * - .toStartPiiEntitiesDetectionJob()
     * - .toStartSentimentDetectionJob()
     * - .toStartTargetedSentimentDetectionJob()
     * - .toStartTopicsDetectionJob()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by requiring tag value associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * Applies to resource types:
     * - targeted-sentiment-detection-job
     * - document-classifier
     * - document-classifier-endpoint
     * - entity-recognizer
     * - entity-recognizer-endpoint
     * - dominant-language-detection-job
     * - entities-detection-job
     * - pii-entities-detection-job
     * - events-detection-job
     * - key-phrases-detection-job
     * - sentiment-detection-job
     * - topics-detection-job
     * - document-classification-job
     * - flywheel
     * - flywheel-dataset
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by requiring the presence of mandatory tags in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * Applies to actions:
     * - .toCreateDataset()
     * - .toCreateDocumentClassifier()
     * - .toCreateEndpoint()
     * - .toCreateEntityRecognizer()
     * - .toCreateFlywheel()
     * - .toImportModel()
     * - .toStartDocumentClassificationJob()
     * - .toStartDominantLanguageDetectionJob()
     * - .toStartEntitiesDetectionJob()
     * - .toStartEventsDetectionJob()
     * - .toStartKeyPhrasesDetectionJob()
     * - .toStartPiiEntitiesDetectionJob()
     * - .toStartSentimentDetectionJob()
     * - .toStartTargetedSentimentDetectionJob()
     * - .toStartTopicsDetectionJob()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the DataLake Kms Key associated with the flywheel resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncomprehend.html#amazoncomprehend-policy-keys
     *
     * Applies to actions:
     * - .toCreateFlywheel()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifDataLakeKmsKey(value, operator) {
        return this.if(`DataLakeKmsKey`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by particular Iteration Id for a flywheel
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncomprehend.html#amazoncomprehend-policy-keys
     *
     * Applies to actions:
     * - .toDescribeFlywheelIteration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFlywheelIterationId(value, operator) {
        return this.if(`FlywheelIterationId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the model KMS key associated with the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncomprehend.html#amazoncomprehend-policy-keys
     *
     * Applies to actions:
     * - .toCreateDocumentClassifier()
     * - .toCreateEntityRecognizer()
     * - .toCreateFlywheel()
     * - .toImportModel()
     * - .toUpdateFlywheel()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifModelKmsKey(value, operator) {
        return this.if(`ModelKmsKey`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the output KMS key associated with the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncomprehend.html#amazoncomprehend-policy-keys
     *
     * Applies to actions:
     * - .toCreateDocumentClassifier()
     * - .toStartDocumentClassificationJob()
     * - .toStartDominantLanguageDetectionJob()
     * - .toStartEntitiesDetectionJob()
     * - .toStartEventsDetectionJob()
     * - .toStartKeyPhrasesDetectionJob()
     * - .toStartPiiEntitiesDetectionJob()
     * - .toStartSentimentDetectionJob()
     * - .toStartTargetedSentimentDetectionJob()
     * - .toStartTopicsDetectionJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifOutputKmsKey(value, operator) {
        return this.if(`OutputKmsKey`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the volume KMS key associated with the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncomprehend.html#amazoncomprehend-policy-keys
     *
     * Applies to actions:
     * - .toCreateDocumentClassifier()
     * - .toCreateEntityRecognizer()
     * - .toCreateFlywheel()
     * - .toStartDocumentClassificationJob()
     * - .toStartDominantLanguageDetectionJob()
     * - .toStartEntitiesDetectionJob()
     * - .toStartKeyPhrasesDetectionJob()
     * - .toStartSentimentDetectionJob()
     * - .toStartTargetedSentimentDetectionJob()
     * - .toStartTopicsDetectionJob()
     * - .toUpdateFlywheel()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifVolumeKmsKey(value, operator) {
        return this.if(`VolumeKmsKey`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the list of all VPC security group ids associated with the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncomprehend.html#amazoncomprehend-policy-keys
     *
     * Applies to actions:
     * - .toCreateDocumentClassifier()
     * - .toCreateEntityRecognizer()
     * - .toCreateFlywheel()
     * - .toStartDocumentClassificationJob()
     * - .toStartDominantLanguageDetectionJob()
     * - .toStartEntitiesDetectionJob()
     * - .toStartKeyPhrasesDetectionJob()
     * - .toStartSentimentDetectionJob()
     * - .toStartTargetedSentimentDetectionJob()
     * - .toStartTopicsDetectionJob()
     * - .toUpdateFlywheel()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpcSecurityGroupIds(value, operator) {
        return this.if(`VpcSecurityGroupIds`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the list of all VPC subnets associated with the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncomprehend.html#amazoncomprehend-policy-keys
     *
     * Applies to actions:
     * - .toCreateDocumentClassifier()
     * - .toCreateEntityRecognizer()
     * - .toCreateFlywheel()
     * - .toStartDocumentClassificationJob()
     * - .toStartDominantLanguageDetectionJob()
     * - .toStartEntitiesDetectionJob()
     * - .toStartKeyPhrasesDetectionJob()
     * - .toStartSentimentDetectionJob()
     * - .toStartTargetedSentimentDetectionJob()
     * - .toStartTopicsDetectionJob()
     * - .toUpdateFlywheel()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpcSubnets(value, operator) {
        return this.if(`VpcSubnets`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [comprehend](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncomprehend.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'comprehend';
        this.accessLevelList = {
            Read: [
                'BatchDetectDominantLanguage',
                'BatchDetectEntities',
                'BatchDetectKeyPhrases',
                'BatchDetectSentiment',
                'BatchDetectSyntax',
                'BatchDetectTargetedSentiment',
                'ClassifyDocument',
                'ContainsPiiEntities',
                'DescribeDataset',
                'DescribeDocumentClassificationJob',
                'DescribeDocumentClassifier',
                'DescribeDominantLanguageDetectionJob',
                'DescribeEndpoint',
                'DescribeEntitiesDetectionJob',
                'DescribeEntityRecognizer',
                'DescribeEventsDetectionJob',
                'DescribeFlywheel',
                'DescribeFlywheelIteration',
                'DescribeKeyPhrasesDetectionJob',
                'DescribePiiEntitiesDetectionJob',
                'DescribeResourcePolicy',
                'DescribeSentimentDetectionJob',
                'DescribeTargetedSentimentDetectionJob',
                'DescribeTopicsDetectionJob',
                'DetectDominantLanguage',
                'DetectEntities',
                'DetectKeyPhrases',
                'DetectPiiEntities',
                'DetectSentiment',
                'DetectSyntax',
                'DetectTargetedSentiment',
                'DetectToxicContent',
                'ListDatasets',
                'ListDocumentClassificationJobs',
                'ListDocumentClassifierSummaries',
                'ListDocumentClassifiers',
                'ListDominantLanguageDetectionJobs',
                'ListEndpoints',
                'ListEntitiesDetectionJobs',
                'ListEntityRecognizerSummaries',
                'ListEntityRecognizers',
                'ListEventsDetectionJobs',
                'ListFlywheelIterationHistory',
                'ListFlywheels',
                'ListKeyPhrasesDetectionJobs',
                'ListPiiEntitiesDetectionJobs',
                'ListSentimentDetectionJobs',
                'ListTagsForResource',
                'ListTargetedSentimentDetectionJobs',
                'ListTopicsDetectionJobs'
            ],
            Write: [
                'CreateDataset',
                'CreateDocumentClassifier',
                'CreateEndpoint',
                'CreateEntityRecognizer',
                'CreateFlywheel',
                'DeleteDocumentClassifier',
                'DeleteEndpoint',
                'DeleteEntityRecognizer',
                'DeleteFlywheel',
                'DeleteResourcePolicy',
                'ImportModel',
                'PutResourcePolicy',
                'StartDocumentClassificationJob',
                'StartDominantLanguageDetectionJob',
                'StartEntitiesDetectionJob',
                'StartEventsDetectionJob',
                'StartFlywheelIteration',
                'StartKeyPhrasesDetectionJob',
                'StartPiiEntitiesDetectionJob',
                'StartSentimentDetectionJob',
                'StartTargetedSentimentDetectionJob',
                'StartTopicsDetectionJob',
                'StopDominantLanguageDetectionJob',
                'StopEntitiesDetectionJob',
                'StopEventsDetectionJob',
                'StopKeyPhrasesDetectionJob',
                'StopPiiEntitiesDetectionJob',
                'StopSentimentDetectionJob',
                'StopTargetedSentimentDetectionJob',
                'StopTrainingDocumentClassifier',
                'StopTrainingEntityRecognizer',
                'UpdateEndpoint',
                'UpdateFlywheel'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Comprehend = Comprehend;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29tcHJlaGVuZC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImNvbXByZWhlbmQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0EseUNBQXlEO0FBR3pEOzs7O0dBSUc7QUFDSCxNQUFhLFVBQVcsU0FBUSx3QkFBZTtJQUc3Qzs7Ozs7O09BTUc7SUFDSSw2QkFBNkI7UUFDbEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7SUFDaEQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUJBQXVCO1FBQzVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0lBQzFDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQkFBc0I7UUFDM0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksOEJBQThCO1FBQ25DLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO0lBQ2pELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGVBQWU7UUFDcEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7O09BZUc7SUFDSSwwQkFBMEI7UUFDL0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSx3QkFBd0I7UUFDN0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7SUFDM0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO0lBQ25DLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQkFBc0I7UUFDM0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUNBQW1DO1FBQ3hDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQ0FBbUMsQ0FBQyxDQUFDO0lBQ3RELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw0QkFBNEI7UUFDakMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDRCQUE0QixDQUFDLENBQUM7SUFDL0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNDQUFzQztRQUMzQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0NBQXNDLENBQUMsQ0FBQztJQUN6RCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw4QkFBOEI7UUFDbkMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDhCQUE4QixDQUFDLENBQUM7SUFDakQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBCQUEwQjtRQUMvQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztJQUM3QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNEJBQTRCO1FBQ2pDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO0lBQy9DLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLDJCQUEyQjtRQUNoQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMkJBQTJCLENBQUMsQ0FBQztJQUM5QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0NBQWdDO1FBQ3JDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO0lBQ25ELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQ0FBaUM7UUFDdEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlDQUFpQyxDQUFDLENBQUM7SUFDcEQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksK0JBQStCO1FBQ3BDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1Q0FBdUM7UUFDNUMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVDQUF1QyxDQUFDLENBQUM7SUFDMUQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDRCQUE0QjtRQUNqQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNEJBQTRCLENBQUMsQ0FBQztJQUMvQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQWtCLENBQUMsQ0FBQztJQUNyQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGNBQWMsQ0FBQyxDQUFDO0lBQ2pDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUI7UUFDOUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7SUFDNUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxhQUFhO1FBQ2xCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUNoQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsY0FBYyxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdDQUFnQztRQUNyQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztJQUNuRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUNBQWlDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQ0FBaUMsQ0FBQyxDQUFDO0lBQ3BELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUI7UUFDOUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7SUFDNUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1DQUFtQztRQUN4QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUNBQW1DLENBQUMsQ0FBQztJQUN0RCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDJCQUEyQjtRQUNoQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMkJBQTJCLENBQUMsQ0FBQztJQUM5QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksK0JBQStCO1FBQ3BDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLENBQUMsQ0FBQztJQUM1QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksOEJBQThCO1FBQ25DLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO0lBQ2pELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNkJBQTZCO1FBQ2xDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO0lBQ2hELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw4QkFBOEI7UUFDbkMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDhCQUE4QixDQUFDLENBQUM7SUFDakQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDRCQUE0QjtRQUNqQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNEJBQTRCLENBQUMsQ0FBQztJQUMvQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQ0FBb0M7UUFDekMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxDQUFDLENBQUM7SUFDdkQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLENBQUMsQ0FBQztJQUM1QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQ0FBZ0M7UUFDckMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdDQUFnQyxDQUFDLENBQUM7SUFDbkQsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7OztPQWNHO0lBQ0ksbUNBQW1DO1FBQ3hDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQ0FBbUMsQ0FBQyxDQUFDO0lBQ3RELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwyQkFBMkI7UUFDaEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDJCQUEyQixDQUFDLENBQUM7SUFDOUMsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0kseUJBQXlCO1FBQzlCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx3QkFBd0I7UUFDN0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7SUFDM0MsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7OztPQWNHO0lBQ0ksNkJBQTZCO1FBQ2xDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO0lBQ2hELENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLDhCQUE4QjtRQUNuQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsOEJBQThCLENBQUMsQ0FBQztJQUNqRCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSw0QkFBNEI7UUFDakMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDRCQUE0QixDQUFDLENBQUM7SUFDL0MsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7OztPQWNHO0lBQ0ksb0NBQW9DO1FBQ3pDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQ0FBb0MsQ0FBQyxDQUFDO0lBQ3ZELENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7T0FjRztJQUNJLHlCQUF5QjtRQUM5QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLENBQUMsQ0FBQztJQUM1QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0NBQWtDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO0lBQ3JELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwwQkFBMEI7UUFDL0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNEJBQTRCO1FBQ2pDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO0lBQy9DLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw2QkFBNkI7UUFDbEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7SUFDaEQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDJCQUEyQjtRQUNoQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMkJBQTJCLENBQUMsQ0FBQztJQUM5QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUNBQW1DO1FBQ3hDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQ0FBbUMsQ0FBQyxDQUFDO0lBQ3RELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQ0FBZ0M7UUFDckMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdDQUFnQyxDQUFDLENBQUM7SUFDbkQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDhCQUE4QjtRQUNuQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsOEJBQThCLENBQUMsQ0FBQztJQUNqRCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGFBQWE7UUFDbEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO0lBQ25DLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQWdHRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSwrQkFBK0IsQ0FBQyxLQUFhLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDekcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLHFDQUFzQyxLQUFNLEVBQUUsQ0FBQyxDQUFDO0lBQzdMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxvQkFBb0IsQ0FBQyxzQkFBOEIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUMvRyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsd0JBQXlCLHNCQUF1QixFQUFFLENBQUMsQ0FBQztJQUNqTSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksNEJBQTRCLENBQUMsOEJBQXNDLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDL0gsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLGlDQUFrQyw4QkFBK0IsRUFBRSxDQUFDLENBQUM7SUFDbE4sQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGtCQUFrQixDQUFDLG9CQUE0QixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzNHLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxzQkFBdUIsb0JBQXFCLEVBQUUsQ0FBQyxDQUFDO0lBQzdMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSwwQkFBMEIsQ0FBQyw0QkFBb0MsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUMzSCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsK0JBQWdDLDRCQUE2QixFQUFFLENBQUMsQ0FBQztJQUM5TSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksOEJBQThCLENBQUMsS0FBYSxFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3hHLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxvQ0FBcUMsS0FBTSxFQUFFLENBQUMsQ0FBQztJQUM1TCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksc0JBQXNCLENBQUMsS0FBYSxFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ2hHLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSwyQkFBNEIsS0FBTSxFQUFFLENBQUMsQ0FBQztJQUNuTCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0kseUJBQXlCLENBQUMsS0FBYSxFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ25HLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSwrQkFBZ0MsS0FBTSxFQUFFLENBQUMsQ0FBQztJQUN2TCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksb0JBQW9CLENBQUMsS0FBYSxFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzlGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSx5QkFBMEIsS0FBTSxFQUFFLENBQUMsQ0FBQztJQUNqTCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksd0JBQXdCLENBQUMsS0FBYSxFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ2xHLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSw4QkFBK0IsS0FBTSxFQUFFLENBQUMsQ0FBQztJQUN0TCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksdUJBQXVCLENBQUMsS0FBYSxFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ2pHLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSw0QkFBNkIsS0FBTSxFQUFFLENBQUMsQ0FBQztJQUNwTCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksb0JBQW9CLENBQUMsS0FBYSxFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzlGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSx5QkFBMEIsS0FBTSxFQUFFLENBQUMsQ0FBQztJQUNqTCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksMkJBQTJCLENBQUMsS0FBYSxFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3JHLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxnQ0FBaUMsS0FBTSxFQUFFLENBQUMsQ0FBQztJQUN4TCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksVUFBVSxDQUFDLFlBQW9CLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDM0YsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLGFBQWMsWUFBYSxFQUFFLENBQUMsQ0FBQztJQUM1SyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLGlCQUFpQixDQUFDLFlBQW9CLEVBQUUsV0FBbUIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN2SCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsYUFBYyxZQUFhLFlBQWEsV0FBWSxFQUFFLENBQUMsQ0FBQztJQUNyTSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09BMEJHO0lBQ0ksZUFBZSxDQUFDLE1BQWMsRUFBRSxLQUF3QixFQUFFLFFBQTRCO1FBQzNGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBbUIsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNoRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0F5Qkc7SUFDSSxnQkFBZ0IsQ0FBQyxNQUFjLEVBQUUsS0FBd0IsRUFBRSxRQUE0QjtRQUM1RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW9CLE1BQU8sRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDakYsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQTBCRztJQUNJLFlBQVksQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3hFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGdCQUFnQixDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDNUUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksU0FBUyxDQUFDLENBQUM7SUFDakUsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxxQkFBcUIsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ2pGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ3pFLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7T0FjRztJQUNJLGFBQWEsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3pFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxTQUFTLENBQUMsQ0FBQztJQUM5RCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FtQkc7SUFDSSxjQUFjLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUMxRSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsY0FBYyxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksU0FBUyxDQUFDLENBQUM7SUFDL0QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQW9CRztJQUNJLGNBQWMsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQzFFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxjQUFjLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxTQUFTLENBQUMsQ0FBQztJQUMvRCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09Bb0JHO0lBQ0kscUJBQXFCLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUNqRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUN6RSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09Bb0JHO0lBQ0ksWUFBWSxDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDeEUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFlBQVksRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2hFLENBQUM7SUFFRDs7O09BR0c7SUFDSCxZQUFZLEtBQWdDO1FBQzFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztRQWxsRFIsa0JBQWEsR0FBRyxZQUFZLENBQUM7UUErL0IxQixvQkFBZSxHQUFvQjtZQUMzQyxJQUFJLEVBQUU7Z0JBQ0osNkJBQTZCO2dCQUM3QixxQkFBcUI7Z0JBQ3JCLHVCQUF1QjtnQkFDdkIsc0JBQXNCO2dCQUN0QixtQkFBbUI7Z0JBQ25CLDhCQUE4QjtnQkFDOUIsa0JBQWtCO2dCQUNsQixxQkFBcUI7Z0JBQ3JCLGlCQUFpQjtnQkFDakIsbUNBQW1DO2dCQUNuQyw0QkFBNEI7Z0JBQzVCLHNDQUFzQztnQkFDdEMsa0JBQWtCO2dCQUNsQiw4QkFBOEI7Z0JBQzlCLDBCQUEwQjtnQkFDMUIsNEJBQTRCO2dCQUM1QixrQkFBa0I7Z0JBQ2xCLDJCQUEyQjtnQkFDM0IsZ0NBQWdDO2dCQUNoQyxpQ0FBaUM7Z0JBQ2pDLHdCQUF3QjtnQkFDeEIsK0JBQStCO2dCQUMvQix1Q0FBdUM7Z0JBQ3ZDLDRCQUE0QjtnQkFDNUIsd0JBQXdCO2dCQUN4QixnQkFBZ0I7Z0JBQ2hCLGtCQUFrQjtnQkFDbEIsbUJBQW1CO2dCQUNuQixpQkFBaUI7Z0JBQ2pCLGNBQWM7Z0JBQ2QseUJBQXlCO2dCQUN6QixvQkFBb0I7Z0JBQ3BCLGNBQWM7Z0JBQ2QsZ0NBQWdDO2dCQUNoQyxpQ0FBaUM7Z0JBQ2pDLHlCQUF5QjtnQkFDekIsbUNBQW1DO2dCQUNuQyxlQUFlO2dCQUNmLDJCQUEyQjtnQkFDM0IsK0JBQStCO2dCQUMvQix1QkFBdUI7Z0JBQ3ZCLHlCQUF5QjtnQkFDekIsOEJBQThCO2dCQUM5QixlQUFlO2dCQUNmLDZCQUE2QjtnQkFDN0IsOEJBQThCO2dCQUM5Qiw0QkFBNEI7Z0JBQzVCLHFCQUFxQjtnQkFDckIsb0NBQW9DO2dCQUNwQyx5QkFBeUI7YUFDMUI7WUFDRCxLQUFLLEVBQUU7Z0JBQ0wsZUFBZTtnQkFDZiwwQkFBMEI7Z0JBQzFCLGdCQUFnQjtnQkFDaEIsd0JBQXdCO2dCQUN4QixnQkFBZ0I7Z0JBQ2hCLDBCQUEwQjtnQkFDMUIsZ0JBQWdCO2dCQUNoQix3QkFBd0I7Z0JBQ3hCLGdCQUFnQjtnQkFDaEIsc0JBQXNCO2dCQUN0QixhQUFhO2dCQUNiLG1CQUFtQjtnQkFDbkIsZ0NBQWdDO2dCQUNoQyxtQ0FBbUM7Z0JBQ25DLDJCQUEyQjtnQkFDM0IseUJBQXlCO2dCQUN6Qix3QkFBd0I7Z0JBQ3hCLDZCQUE2QjtnQkFDN0IsOEJBQThCO2dCQUM5Qiw0QkFBNEI7Z0JBQzVCLG9DQUFvQztnQkFDcEMseUJBQXlCO2dCQUN6QixrQ0FBa0M7Z0JBQ2xDLDBCQUEwQjtnQkFDMUIsd0JBQXdCO2dCQUN4Qiw0QkFBNEI7Z0JBQzVCLDZCQUE2QjtnQkFDN0IsMkJBQTJCO2dCQUMzQixtQ0FBbUM7Z0JBQ25DLGdDQUFnQztnQkFDaEMsOEJBQThCO2dCQUM5QixnQkFBZ0I7Z0JBQ2hCLGdCQUFnQjthQUNqQjtZQUNELE9BQU8sRUFBRTtnQkFDUCxhQUFhO2dCQUNiLGVBQWU7YUFDaEI7U0FDRixDQUFDO0lBd2ZGLENBQUM7Q0FDRjtBQXJsREQsZ0NBcWxEQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEFjY2Vzc0xldmVsTGlzdCB9IGZyb20gJy4uLy4uL3NoYXJlZC9hY2Nlc3MtbGV2ZWwnO1xuaW1wb3J0IHsgUG9saWN5U3RhdGVtZW50LCBPcGVyYXRvciB9IGZyb20gJy4uLy4uL3NoYXJlZCc7XG5pbXBvcnQgeyBhd3NfaWFtIGFzIGlhbSB9IGZyb20gXCJhd3MtY2RrLWxpYlwiO1xuXG4vKipcbiAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbY29tcHJlaGVuZF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYW1hem9uY29tcHJlaGVuZC5odG1sKS5cbiAqXG4gKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAqL1xuZXhwb3J0IGNsYXNzIENvbXByZWhlbmQgZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnQge1xuICBwdWJsaWMgc2VydmljZVByZWZpeCA9ICdjb21wcmVoZW5kJztcblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGV0ZWN0IHRoZSBsYW5ndWFnZSBvciBsYW5ndWFnZXMgcHJlc2VudCBpbiB0aGUgbGlzdCBvZiB0ZXh0IGRvY3VtZW50c1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQmF0Y2hEZXRlY3REb21pbmFudExhbmd1YWdlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0JhdGNoRGV0ZWN0RG9taW5hbnRMYW5ndWFnZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQmF0Y2hEZXRlY3REb21pbmFudExhbmd1YWdlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGV0ZWN0IHRoZSBuYW1lZCBlbnRpdGllcyAoXCJQZW9wbGVcIiwgXCJQbGFjZXNcIiwgXCJMb2NhdGlvbnNcIiwgZXRjKSB3aXRoaW4gdGhlIGdpdmVuIGxpc3Qgb2YgdGV4dCBkb2N1bWVudHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0JhdGNoRGV0ZWN0RW50aXRpZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQmF0Y2hEZXRlY3RFbnRpdGllcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQmF0Y2hEZXRlY3RFbnRpdGllcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRldGVjdCB0aGUgcGhyYXNlcyBpbiB0aGUgbGlzdCBvZiB0ZXh0IGRvY3VtZW50cyB0aGF0IGFyZSBtb3N0IGluZGljYXRpdmUgb2YgdGhlIGNvbnRlbnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0JhdGNoRGV0ZWN0S2V5UGhyYXNlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9CYXRjaERldGVjdEtleVBocmFzZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0JhdGNoRGV0ZWN0S2V5UGhyYXNlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRldGVjdCB0aGUgc2VudGltZW50IG9mIGEgdGV4dCBpbiB0aGUgbGlzdCBvZiBkb2N1bWVudHMgKFBvc2l0aXZlLCBOZWdhdGl2ZSwgTmV1dHJhbCwgb3IgTWl4ZWQpXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9CYXRjaERldGVjdFNlbnRpbWVudC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9CYXRjaERldGVjdFNlbnRpbWVudCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQmF0Y2hEZXRlY3RTZW50aW1lbnQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZXRlY3Qgc3ludGFjdGljIGluZm9ybWF0aW9uIChsaWtlIFBhcnQgb2YgU3BlZWNoLCBUb2tlbnMpIGluIGEgbGlzdCBvZiB0ZXh0IGRvY3VtZW50c1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQmF0Y2hEZXRlY3RTeW50YXguaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQmF0Y2hEZXRlY3RTeW50YXgoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0JhdGNoRGV0ZWN0U3ludGF4Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGV0ZWN0IHRoZSBzZW50aW1lbnRzIGFzc29jaWF0ZWQgd2l0aCBzcGVjaWZpYyBlbnRpdGllcyAoc3VjaCBhcyBicmFuZHMgb3IgcHJvZHVjdHMpIHdpdGhpbiB0aGUgZ2l2ZW4gbGlzdCBvZiB0ZXh0IGRvY3VtZW50c1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQmF0Y2hEZXRlY3RUYXJnZXRlZFNlbnRpbWVudC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9CYXRjaERldGVjdFRhcmdldGVkU2VudGltZW50KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdCYXRjaERldGVjdFRhcmdldGVkU2VudGltZW50Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgbmV3IGRvY3VtZW50IGNsYXNzaWZpY2F0aW9uIHJlcXVlc3QgdG8gYW5hbHl6ZSBhIHNpbmdsZSBkb2N1bWVudCBpbiByZWFsLXRpbWUsIHVzaW5nIGEgcHJldmlvdXNseSBjcmVhdGVkIGFuZCB0cmFpbmVkIGN1c3RvbSBtb2RlbCBhbmQgYW4gZW5kcG9pbnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NsYXNzaWZ5RG9jdW1lbnQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ2xhc3NpZnlEb2N1bWVudCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ2xhc3NpZnlEb2N1bWVudCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNsYXNzaWZ5IHRoZSBwZXJzb25hbGx5IGlkZW50aWZpYWJsZSBpbmZvcm1hdGlvbiB3aXRoaW4gZ2l2ZW4gZG9jdW1lbnRzIGluIHJlYWwtdGltZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ29udGFpbnNQaWlFbnRpdGllcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Db250YWluc1BpaUVudGl0aWVzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDb250YWluc1BpaUVudGl0aWVzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgbmV3IGRhdGFzZXQgd2l0aGluIGEgZmx5d2hlZWxcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVEYXRhc2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZURhdGFzZXQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZURhdGFzZXQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBuZXcgZG9jdW1lbnQgY2xhc3NpZmllciB0aGF0IHlvdSBjYW4gdXNlIHRvIGNhdGVnb3JpemUgZG9jdW1lbnRzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICogLSAuaWZWb2x1bWVLbXNLZXkoKVxuICAgKiAtIC5pZk1vZGVsS21zS2V5KClcbiAgICogLSAuaWZPdXRwdXRLbXNLZXkoKVxuICAgKiAtIC5pZlZwY1NlY3VyaXR5R3JvdXBJZHMoKVxuICAgKiAtIC5pZlZwY1N1Ym5ldHMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVEb2N1bWVudENsYXNzaWZpZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlRG9jdW1lbnRDbGFzc2lmaWVyKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVEb2N1bWVudENsYXNzaWZpZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBtb2RlbC1zcGVjaWZpYyBlbmRwb2ludCBmb3Igc3luY2hyb25vdXMgaW5mZXJlbmNlIGZvciBhIHByZXZpb3VzbHkgdHJhaW5lZCBjdXN0b20gbW9kZWxcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVFbmRwb2ludC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVFbmRwb2ludCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlRW5kcG9pbnQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYW4gZW50aXR5IHJlY29nbml6ZXIgdXNpbmcgc3VibWl0dGVkIGZpbGVzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICogLSAuaWZWb2x1bWVLbXNLZXkoKVxuICAgKiAtIC5pZk1vZGVsS21zS2V5KClcbiAgICogLSAuaWZWcGNTZWN1cml0eUdyb3VwSWRzKClcbiAgICogLSAuaWZWcGNTdWJuZXRzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlRW50aXR5UmVjb2duaXplci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVFbnRpdHlSZWNvZ25pemVyKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVFbnRpdHlSZWNvZ25pemVyJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgbmV3IGZseXdoZWVsIHRoYXQgeW91IGNhbiB1c2UgdG8gdHJhaW4gbW9kZWwgdmVyc2lvbnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVGbHl3aGVlbC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVGbHl3aGVlbCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlRmx5d2hlZWwnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBwcmV2aW91c2x5IGNyZWF0ZWQgZG9jdW1lbnQgY2xhc3NpZmllclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZURvY3VtZW50Q2xhc3NpZmllci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVEb2N1bWVudENsYXNzaWZpZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZURvY3VtZW50Q2xhc3NpZmllcicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIG1vZGVsLXNwZWNpZmljIGVuZHBvaW50IGZvciBhIHByZXZpb3VzbHktdHJhaW5lZCBjdXN0b20gbW9kZWwuIEFsbCBlbmRwb2ludHMgbXVzdCBiZSBkZWxldGVkIGluIG9yZGVyIGZvciB0aGUgbW9kZWwgdG8gYmUgZGVsZXRlZFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUVuZHBvaW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUVuZHBvaW50KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVFbmRwb2ludCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIHN1Ym1pdHRlZCBlbnRpdHkgcmVjb2duaXplclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUVudGl0eVJlY29nbml6ZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlRW50aXR5UmVjb2duaXplcigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlRW50aXR5UmVjb2duaXplcicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIERlbGV0ZSBhIGZseXdoZWVsXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlRmx5d2hlZWwuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlRmx5d2hlZWwoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZUZseXdoZWVsJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVtb3ZlIHBvbGljeSBvbiByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZVJlc291cmNlUG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVJlc291cmNlUG9saWN5KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVSZXNvdXJjZVBvbGljeScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCB0aGUgcHJvcGVydGllcyBhc3NvY2lhdGVkIHdpdGggYSBkYXRhc2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZURhdGFzZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVEYXRhc2V0KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZURhdGFzZXQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgdGhlIHByb3BlcnRpZXMgYXNzb2NpYXRlZCB3aXRoIGEgZG9jdW1lbnQgY2xhc3NpZmljYXRpb24gam9iXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZURvY3VtZW50Q2xhc3NpZmljYXRpb25Kb2IuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVEb2N1bWVudENsYXNzaWZpY2F0aW9uSm9iKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZURvY3VtZW50Q2xhc3NpZmljYXRpb25Kb2InKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgdGhlIHByb3BlcnRpZXMgYXNzb2NpYXRlZCB3aXRoIGEgZG9jdW1lbnQgY2xhc3NpZmllclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVEb2N1bWVudENsYXNzaWZpZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVEb2N1bWVudENsYXNzaWZpZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlRG9jdW1lbnRDbGFzc2lmaWVyJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IHRoZSBwcm9wZXJ0aWVzIGFzc29jaWF0ZWQgd2l0aCBhIGRvbWluYW50IGxhbmd1YWdlIGRldGVjdGlvbiBqb2JcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlRG9taW5hbnRMYW5ndWFnZURldGVjdGlvbkpvYi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZURvbWluYW50TGFuZ3VhZ2VEZXRlY3Rpb25Kb2IoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlRG9taW5hbnRMYW5ndWFnZURldGVjdGlvbkpvYicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCB0aGUgcHJvcGVydGllcyBhc3NvY2lhdGVkIHdpdGggYSBzcGVjaWZpYyBlbmRwb2ludC4gVXNlIHRoaXMgb3BlcmF0aW9uIHRvIGdldCB0aGUgc3RhdHVzIG9mIGFuIGVuZHBvaW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUVuZHBvaW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlRW5kcG9pbnQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlRW5kcG9pbnQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgdGhlIHByb3BlcnRpZXMgYXNzb2NpYXRlZCB3aXRoIGFuIGVudGl0aWVzIGRldGVjdGlvbiBqb2JcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlRW50aXRpZXNEZXRlY3Rpb25Kb2IuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVFbnRpdGllc0RldGVjdGlvbkpvYigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVFbnRpdGllc0RldGVjdGlvbkpvYicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHByb3ZpZGUgZGV0YWlscyBhYm91dCBhbiBlbnRpdHkgcmVjb2duaXplciBpbmNsdWRpbmcgc3RhdHVzLCBTMyBidWNrZXRzIGNvbnRhaW5pbmcgdHJhaW5pbmcgZGF0YSwgcmVjb2duaXplciBtZXRhZGF0YSwgbWV0cmljcywgYW5kIHNvIG9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUVudGl0eVJlY29nbml6ZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVFbnRpdHlSZWNvZ25pemVyKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZUVudGl0eVJlY29nbml6ZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgdGhlIHByb3BlcnRpZXMgYXNzb2NpYXRlZCB3aXRoIGFuIEV2ZW50cyBkZXRlY3Rpb24gam9iXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUV2ZW50c0RldGVjdGlvbkpvYi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUV2ZW50c0RldGVjdGlvbkpvYigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVFdmVudHNEZXRlY3Rpb25Kb2InKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgdGhlIHByb3BlcnRpZXMgYXNzb2NpYXRlZCB3aXRoIGEgZmx5d2hlZWxcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlRmx5d2hlZWwuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVGbHl3aGVlbCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVGbHl3aGVlbCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCB0aGUgcHJvcGVydGllcyBhc3NvY2lhdGVkIHdpdGggYSBmbHl3aGVlbCBpdGVyYXRpb24gZm9yIGEgZmx5d2hlZWxcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmRmx5d2hlZWxJdGVyYXRpb25JZCgpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlRmx5d2hlZWxJdGVyYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVGbHl3aGVlbEl0ZXJhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVGbHl3aGVlbEl0ZXJhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCB0aGUgcHJvcGVydGllcyBhc3NvY2lhdGVkIHdpdGggYSBrZXkgcGhyYXNlcyBkZXRlY3Rpb24gam9iXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUtleVBocmFzZXNEZXRlY3Rpb25Kb2IuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVLZXlQaHJhc2VzRGV0ZWN0aW9uSm9iKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZUtleVBocmFzZXNEZXRlY3Rpb25Kb2InKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgdGhlIHByb3BlcnRpZXMgYXNzb2NpYXRlZCB3aXRoIGEgUElJIGVudGl0aWVzIGRldGVjdGlvbiBqb2JcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlUGlpRW50aXRpZXNEZXRlY3Rpb25Kb2IuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVQaWlFbnRpdGllc0RldGVjdGlvbkpvYigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVQaWlFbnRpdGllc0RldGVjdGlvbkpvYicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJlYWQgYXR0YWNoZWQgcG9saWN5IG9uIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZVJlc291cmNlUG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlUmVzb3VyY2VQb2xpY3koKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlUmVzb3VyY2VQb2xpY3knKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgdGhlIHByb3BlcnRpZXMgYXNzb2NpYXRlZCB3aXRoIGEgc2VudGltZW50IGRldGVjdGlvbiBqb2JcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlU2VudGltZW50RGV0ZWN0aW9uSm9iLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlU2VudGltZW50RGV0ZWN0aW9uSm9iKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZVNlbnRpbWVudERldGVjdGlvbkpvYicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCB0aGUgcHJvcGVydGllcyBhc3NvY2lhdGVkIHdpdGggYSB0YXJnZXRlZCBzZW50aW1lbnQgZGV0ZWN0aW9uIGpvYlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVUYXJnZXRlZFNlbnRpbWVudERldGVjdGlvbkpvYi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZVRhcmdldGVkU2VudGltZW50RGV0ZWN0aW9uSm9iKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZVRhcmdldGVkU2VudGltZW50RGV0ZWN0aW9uSm9iJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IHRoZSBwcm9wZXJ0aWVzIGFzc29jaWF0ZWQgd2l0aCBhIHRvcGljIGRldGVjdGlvbiBqb2JcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlVG9waWNzRGV0ZWN0aW9uSm9iLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlVG9waWNzRGV0ZWN0aW9uSm9iKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZVRvcGljc0RldGVjdGlvbkpvYicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRldGVjdCB0aGUgbGFuZ3VhZ2Ugb3IgbGFuZ3VhZ2VzIHByZXNlbnQgaW4gdGhlIHRleHRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RldGVjdERvbWluYW50TGFuZ3VhZ2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGV0ZWN0RG9taW5hbnRMYW5ndWFnZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGV0ZWN0RG9taW5hbnRMYW5ndWFnZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRldGVjdCB0aGUgbmFtZWQgZW50aXRpZXMgKFwiUGVvcGxlXCIsIFwiUGxhY2VzXCIsIFwiTG9jYXRpb25zXCIsIGV0Yykgd2l0aGluIHRoZSBnaXZlbiB0ZXh0IGRvY3VtZW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXRlY3RFbnRpdGllcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXRlY3RFbnRpdGllcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGV0ZWN0RW50aXRpZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZXRlY3QgdGhlIHBocmFzZXMgaW4gdGhlIHRleHQgdGhhdCBhcmUgbW9zdCBpbmRpY2F0aXZlIG9mIHRoZSBjb250ZW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXRlY3RLZXlQaHJhc2VzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RldGVjdEtleVBocmFzZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RldGVjdEtleVBocmFzZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZXRlY3QgdGhlIHBlcnNvbmFsbHkgaWRlbnRpZmlhYmxlIGluZm9ybWF0aW9uIGVudGl0aWVzIChcIk5hbWVcIiwgXCJTU05cIiwgXCJQSU5cIiwgZXRjKSB3aXRoaW4gdGhlIGdpdmVuIHRleHQgZG9jdW1lbnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RldGVjdFBpaUVudGl0aWVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RldGVjdFBpaUVudGl0aWVzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXRlY3RQaWlFbnRpdGllcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRldGVjdCB0aGUgc2VudGltZW50IG9mIGEgdGV4dCBpbiBhIGRvY3VtZW50IChQb3NpdGl2ZSwgTmVnYXRpdmUsIE5ldXRyYWwsIG9yIE1peGVkKVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGV0ZWN0U2VudGltZW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RldGVjdFNlbnRpbWVudCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGV0ZWN0U2VudGltZW50Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGV0ZWN0IHN5bnRhY3RpYyBpbmZvcm1hdGlvbiAobGlrZSBQYXJ0IG9mIFNwZWVjaCwgVG9rZW5zKSBpbiBhIHRleHQgZG9jdW1lbnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RldGVjdFN5bnRheC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXRlY3RTeW50YXgoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RldGVjdFN5bnRheCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRldGVjdCB0aGUgc2VudGltZW50cyBhc3NvY2lhdGVkIHdpdGggc3BlY2lmaWMgZW50aXRpZXMgKHN1Y2ggYXMgYnJhbmRzIG9yIHByb2R1Y3RzKSBpbiBhIGRvY3VtZW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXRlY3RUYXJnZXRlZFNlbnRpbWVudC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXRlY3RUYXJnZXRlZFNlbnRpbWVudCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGV0ZWN0VGFyZ2V0ZWRTZW50aW1lbnQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZXRlY3QgdG94aWMgY29udGVudCB3aXRoaW4gdGhlIGdpdmVuIGxpc3Qgb2YgdGV4dCBzZWdtZW50c1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGV0ZWN0VG94aWNDb250ZW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RldGVjdFRveGljQ29udGVudCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGV0ZWN0VG94aWNDb250ZW50Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gaW1wb3J0IGEgdHJhaW5lZCBDb21wcmVoZW5kIG1vZGVsXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICogLSAuaWZNb2RlbEttc0tleSgpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0ltcG9ydE1vZGVsLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0ltcG9ydE1vZGVsKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdJbXBvcnRNb2RlbCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBhIGxpc3Qgb2YgdGhlIERhdGFzZXRzIGFzc29jaWF0ZWQgd2l0aCBhIGZseXdoZWVsXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0RGF0YXNldHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdERhdGFzZXRzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0RGF0YXNldHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYSBsaXN0IG9mIHRoZSBkb2N1bWVudCBjbGFzc2lmaWNhdGlvbiBqb2JzIHRoYXQgeW91IGhhdmUgc3VibWl0dGVkXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0RG9jdW1lbnRDbGFzc2lmaWNhdGlvbkpvYnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdERvY3VtZW50Q2xhc3NpZmljYXRpb25Kb2JzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0RG9jdW1lbnRDbGFzc2lmaWNhdGlvbkpvYnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYSBsaXN0IG9mIHN1bW1hcmllcyBvZiB0aGUgZG9jdW1lbnQgY2xhc3NpZmllcnMgdGhhdCB5b3UgaGF2ZSBjcmVhdGVkXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0RG9jdW1lbnRDbGFzc2lmaWVyU3VtbWFyaWVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3REb2N1bWVudENsYXNzaWZpZXJTdW1tYXJpZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3REb2N1bWVudENsYXNzaWZpZXJTdW1tYXJpZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYSBsaXN0IG9mIHRoZSBkb2N1bWVudCBjbGFzc2lmaWVycyB0aGF0IHlvdSBoYXZlIGNyZWF0ZWRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3REb2N1bWVudENsYXNzaWZpZXJzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3REb2N1bWVudENsYXNzaWZpZXJzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0RG9jdW1lbnRDbGFzc2lmaWVycycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBhIGxpc3Qgb2YgdGhlIGRvbWluYW50IGxhbmd1YWdlIGRldGVjdGlvbiBqb2JzIHRoYXQgeW91IGhhdmUgc3VibWl0dGVkXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0RG9taW5hbnRMYW5ndWFnZURldGVjdGlvbkpvYnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdERvbWluYW50TGFuZ3VhZ2VEZXRlY3Rpb25Kb2JzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0RG9taW5hbnRMYW5ndWFnZURldGVjdGlvbkpvYnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYSBsaXN0IG9mIGFsbCBleGlzdGluZyBlbmRwb2ludHMgdGhhdCB5b3UndmUgY3JlYXRlZFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdEVuZHBvaW50cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0RW5kcG9pbnRzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0RW5kcG9pbnRzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGEgbGlzdCBvZiB0aGUgZW50aXR5IGRldGVjdGlvbiBqb2JzIHRoYXQgeW91IGhhdmUgc3VibWl0dGVkXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0RW50aXRpZXNEZXRlY3Rpb25Kb2JzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RFbnRpdGllc0RldGVjdGlvbkpvYnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RFbnRpdGllc0RldGVjdGlvbkpvYnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYSBsaXN0IG9mIHN1bW1hcmllcyBmb3IgdGhlIGVudGl0eSByZWNvZ25pemVycyB0aGF0IHlvdSBoYXZlIGNyZWF0ZWRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RFbnRpdHlSZWNvZ25pemVyU3VtbWFyaWVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RFbnRpdHlSZWNvZ25pemVyU3VtbWFyaWVzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0RW50aXR5UmVjb2duaXplclN1bW1hcmllcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBhIGxpc3Qgb2YgdGhlIHByb3BlcnRpZXMgb2YgYWxsIGVudGl0eSByZWNvZ25pemVycyB0aGF0IHlvdSBjcmVhdGVkLCBpbmNsdWRpbmcgcmVjb2duaXplcnMgY3VycmVudGx5IGluIHRyYWluaW5nXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0RW50aXR5UmVjb2duaXplcnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEVudGl0eVJlY29nbml6ZXJzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0RW50aXR5UmVjb2duaXplcnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYSBsaXN0IG9mIEV2ZW50cyBkZXRlY3Rpb24gam9icyB0aGF0IHlvdSBoYXZlIHN1Ym1pdHRlZFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdEV2ZW50c0RldGVjdGlvbkpvYnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEV2ZW50c0RldGVjdGlvbkpvYnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RFdmVudHNEZXRlY3Rpb25Kb2JzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGEgbGlzdCBvZiBpdGVyYXRpb25zIGFzc29jaWF0ZWQgZm9yIGEgZmx5d2hlZWxcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RGbHl3aGVlbEl0ZXJhdGlvbkhpc3RvcnkuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEZseXdoZWVsSXRlcmF0aW9uSGlzdG9yeSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdEZseXdoZWVsSXRlcmF0aW9uSGlzdG9yeScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBhIGxpc3Qgb2YgdGhlIGZseXdoZWVscyB0aGF0IHlvdSBoYXZlIGNyZWF0ZWRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RGbHl3aGVlbHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEZseXdoZWVscygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdEZseXdoZWVscycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBhIGxpc3Qgb2Yga2V5IHBocmFzZSBkZXRlY3Rpb24gam9icyB0aGF0IHlvdSBoYXZlIHN1Ym1pdHRlZFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdEtleVBocmFzZXNEZXRlY3Rpb25Kb2JzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RLZXlQaHJhc2VzRGV0ZWN0aW9uSm9icygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdEtleVBocmFzZXNEZXRlY3Rpb25Kb2JzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGEgbGlzdCBvZiBQSUkgZW50aXRpZXMgZGV0ZWN0aW9uIGpvYnMgdGhhdCB5b3UgaGF2ZSBzdWJtaXR0ZWRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RQaWlFbnRpdGllc0RldGVjdGlvbkpvYnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFBpaUVudGl0aWVzRGV0ZWN0aW9uSm9icygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFBpaUVudGl0aWVzRGV0ZWN0aW9uSm9icycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBhIGxpc3Qgb2Ygc2VudGltZW50IGRldGVjdGlvbiBqb2JzIHRoYXQgeW91IGhhdmUgc3VibWl0dGVkXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0U2VudGltZW50RGV0ZWN0aW9uSm9icy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0U2VudGltZW50RGV0ZWN0aW9uSm9icygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFNlbnRpbWVudERldGVjdGlvbkpvYnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHRhZ3MgZm9yIGEgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RUYWdzRm9yUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFRhZ3NGb3JSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFRhZ3NGb3JSZXNvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBhIGxpc3Qgb2YgdGFyZ2V0ZWQgc2VudGltZW50IGRldGVjdGlvbiBqb2JzIHRoYXQgeW91IGhhdmUgc3VibWl0dGVkXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0VGFyZ2V0ZWRTZW50aW1lbnREZXRlY3Rpb25Kb2JzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RUYXJnZXRlZFNlbnRpbWVudERldGVjdGlvbkpvYnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RUYXJnZXRlZFNlbnRpbWVudERldGVjdGlvbkpvYnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYSBsaXN0IG9mIHRoZSB0b3BpYyBkZXRlY3Rpb24gam9icyB0aGF0IHlvdSBoYXZlIHN1Ym1pdHRlZFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdFRvcGljc0RldGVjdGlvbkpvYnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFRvcGljc0RldGVjdGlvbkpvYnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RUb3BpY3NEZXRlY3Rpb25Kb2JzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYXR0YWNoIHBvbGljeSB0byByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1B1dFJlc291cmNlUG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1B1dFJlc291cmNlUG9saWN5KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdQdXRSZXNvdXJjZVBvbGljeScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHN0YXJ0IGFuIGFzeW5jaHJvbm91cyBkb2N1bWVudCBjbGFzc2lmaWNhdGlvbiBqb2JcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TdGFydERvY3VtZW50Q2xhc3NpZmljYXRpb25Kb2IuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU3RhcnREb2N1bWVudENsYXNzaWZpY2F0aW9uSm9iKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdGFydERvY3VtZW50Q2xhc3NpZmljYXRpb25Kb2InKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzdGFydCBhbiBhc3luY2hyb25vdXMgZG9taW5hbnQgbGFuZ3VhZ2UgZGV0ZWN0aW9uIGpvYiBmb3IgYSBjb2xsZWN0aW9uIG9mIGRvY3VtZW50c1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmVm9sdW1lS21zS2V5KClcbiAgICogLSAuaWZPdXRwdXRLbXNLZXkoKVxuICAgKiAtIC5pZlZwY1NlY3VyaXR5R3JvdXBJZHMoKVxuICAgKiAtIC5pZlZwY1N1Ym5ldHMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TdGFydERvbWluYW50TGFuZ3VhZ2VEZXRlY3Rpb25Kb2IuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU3RhcnREb21pbmFudExhbmd1YWdlRGV0ZWN0aW9uSm9iKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdGFydERvbWluYW50TGFuZ3VhZ2VEZXRlY3Rpb25Kb2InKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzdGFydCBhbiBhc3luY2hyb25vdXMgZW50aXR5IGRldGVjdGlvbiBqb2IgZm9yIGEgY29sbGVjdGlvbiBvZiBkb2N1bWVudHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TdGFydEVudGl0aWVzRGV0ZWN0aW9uSm9iLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0YXJ0RW50aXRpZXNEZXRlY3Rpb25Kb2IoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1N0YXJ0RW50aXRpZXNEZXRlY3Rpb25Kb2InKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzdGFydCBhbiBhc3luY2hyb25vdXMgRXZlbnRzIGRldGVjdGlvbiBqb2IgZm9yIGEgY29sbGVjdGlvbiBvZiBkb2N1bWVudHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKiAtIC5pZk91dHB1dEttc0tleSgpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N0YXJ0RXZlbnRzRGV0ZWN0aW9uSm9iLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0YXJ0RXZlbnRzRGV0ZWN0aW9uSm9iKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdGFydEV2ZW50c0RldGVjdGlvbkpvYicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHN0YXJ0IGEgZmx5d2hlZWwgaXRlcmF0aW9uIGZvciBhIGZseXdoZWVsXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3RhcnRGbHl3aGVlbEl0ZXJhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TdGFydEZseXdoZWVsSXRlcmF0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdGFydEZseXdoZWVsSXRlcmF0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3RhcnQgYW4gYXN5bmNocm9ub3VzIGtleSBwaHJhc2UgZGV0ZWN0aW9uIGpvYiBmb3IgYSBjb2xsZWN0aW9uIG9mIGRvY3VtZW50c1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmVm9sdW1lS21zS2V5KClcbiAgICogLSAuaWZPdXRwdXRLbXNLZXkoKVxuICAgKiAtIC5pZlZwY1NlY3VyaXR5R3JvdXBJZHMoKVxuICAgKiAtIC5pZlZwY1N1Ym5ldHMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TdGFydEtleVBocmFzZXNEZXRlY3Rpb25Kb2IuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU3RhcnRLZXlQaHJhc2VzRGV0ZWN0aW9uSm9iKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdGFydEtleVBocmFzZXNEZXRlY3Rpb25Kb2InKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzdGFydCBhbiBhc3luY2hyb25vdXMgUElJIGVudGl0aWVzIGRldGVjdGlvbiBqb2IgZm9yIGEgY29sbGVjdGlvbiBvZiBkb2N1bWVudHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKiAtIC5pZk91dHB1dEttc0tleSgpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N0YXJ0UGlpRW50aXRpZXNEZXRlY3Rpb25Kb2IuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU3RhcnRQaWlFbnRpdGllc0RldGVjdGlvbkpvYigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU3RhcnRQaWlFbnRpdGllc0RldGVjdGlvbkpvYicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHN0YXJ0IGFuIGFzeW5jaHJvbm91cyBzZW50aW1lbnQgZGV0ZWN0aW9uIGpvYiBmb3IgYSBjb2xsZWN0aW9uIG9mIGRvY3VtZW50c1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmVm9sdW1lS21zS2V5KClcbiAgICogLSAuaWZPdXRwdXRLbXNLZXkoKVxuICAgKiAtIC5pZlZwY1NlY3VyaXR5R3JvdXBJZHMoKVxuICAgKiAtIC5pZlZwY1N1Ym5ldHMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TdGFydFNlbnRpbWVudERldGVjdGlvbkpvYi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TdGFydFNlbnRpbWVudERldGVjdGlvbkpvYigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU3RhcnRTZW50aW1lbnREZXRlY3Rpb25Kb2InKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzdGFydCBhbiBhc3luY2hyb25vdXMgdGFyZ2V0ZWQgc2VudGltZW50IGRldGVjdGlvbiBqb2IgZm9yIGEgY29sbGVjdGlvbiBvZiBkb2N1bWVudHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKiAtIC5pZlZvbHVtZUttc0tleSgpXG4gICAqIC0gLmlmT3V0cHV0S21zS2V5KClcbiAgICogLSAuaWZWcGNTZWN1cml0eUdyb3VwSWRzKClcbiAgICogLSAuaWZWcGNTdWJuZXRzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3RhcnRUYXJnZXRlZFNlbnRpbWVudERldGVjdGlvbkpvYi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TdGFydFRhcmdldGVkU2VudGltZW50RGV0ZWN0aW9uSm9iKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdGFydFRhcmdldGVkU2VudGltZW50RGV0ZWN0aW9uSm9iJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3RhcnQgYW4gYXN5bmNocm9ub3VzIGpvYiB0byBkZXRlY3QgdGhlIG1vc3QgY29tbW9uIHRvcGljcyBpbiB0aGUgY29sbGVjdGlvbiBvZiBkb2N1bWVudHMgYW5kIHRoZSBwaHJhc2VzIGFzc29jaWF0ZWQgd2l0aCBlYWNoIHRvcGljXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICogLSAuaWZWb2x1bWVLbXNLZXkoKVxuICAgKiAtIC5pZk91dHB1dEttc0tleSgpXG4gICAqIC0gLmlmVnBjU2VjdXJpdHlHcm91cElkcygpXG4gICAqIC0gLmlmVnBjU3VibmV0cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N0YXJ0VG9waWNzRGV0ZWN0aW9uSm9iLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0YXJ0VG9waWNzRGV0ZWN0aW9uSm9iKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdGFydFRvcGljc0RldGVjdGlvbkpvYicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHN0b3AgYSBkb21pbmFudCBsYW5ndWFnZSBkZXRlY3Rpb24gam9iXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3RvcERvbWluYW50TGFuZ3VhZ2VEZXRlY3Rpb25Kb2IuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU3RvcERvbWluYW50TGFuZ3VhZ2VEZXRlY3Rpb25Kb2IoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1N0b3BEb21pbmFudExhbmd1YWdlRGV0ZWN0aW9uSm9iJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3RvcCBhbiBlbnRpdHkgZGV0ZWN0aW9uIGpvYlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N0b3BFbnRpdGllc0RldGVjdGlvbkpvYi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TdG9wRW50aXRpZXNEZXRlY3Rpb25Kb2IoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1N0b3BFbnRpdGllc0RldGVjdGlvbkpvYicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHN0b3AgYW4gRXZlbnRzIGRldGVjdGlvbiBqb2JcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TdG9wRXZlbnRzRGV0ZWN0aW9uSm9iLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0b3BFdmVudHNEZXRlY3Rpb25Kb2IoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1N0b3BFdmVudHNEZXRlY3Rpb25Kb2InKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzdG9wIGEga2V5IHBocmFzZSBkZXRlY3Rpb24gam9iXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3RvcEtleVBocmFzZXNEZXRlY3Rpb25Kb2IuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU3RvcEtleVBocmFzZXNEZXRlY3Rpb25Kb2IoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1N0b3BLZXlQaHJhc2VzRGV0ZWN0aW9uSm9iJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3RvcCBhIFBJSSBlbnRpdGllcyBkZXRlY3Rpb24gam9iXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3RvcFBpaUVudGl0aWVzRGV0ZWN0aW9uSm9iLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0b3BQaWlFbnRpdGllc0RldGVjdGlvbkpvYigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU3RvcFBpaUVudGl0aWVzRGV0ZWN0aW9uSm9iJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3RvcCBhIHNlbnRpbWVudCBkZXRlY3Rpb24gam9iXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3RvcFNlbnRpbWVudERldGVjdGlvbkpvYi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TdG9wU2VudGltZW50RGV0ZWN0aW9uSm9iKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdG9wU2VudGltZW50RGV0ZWN0aW9uSm9iJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3RvcCBhIHRhcmdldGVkIHNlbnRpbWVudCBkZXRlY3Rpb24gam9iXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3RvcFRhcmdldGVkU2VudGltZW50RGV0ZWN0aW9uSm9iLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0b3BUYXJnZXRlZFNlbnRpbWVudERldGVjdGlvbkpvYigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU3RvcFRhcmdldGVkU2VudGltZW50RGV0ZWN0aW9uSm9iJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3RvcCBhIHByZXZpb3VzbHkgY3JlYXRlZCBkb2N1bWVudCBjbGFzc2lmaWVyIHRyYWluaW5nIGpvYlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N0b3BUcmFpbmluZ0RvY3VtZW50Q2xhc3NpZmllci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TdG9wVHJhaW5pbmdEb2N1bWVudENsYXNzaWZpZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1N0b3BUcmFpbmluZ0RvY3VtZW50Q2xhc3NpZmllcicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHN0b3AgYSBwcmV2aW91c2x5IGNyZWF0ZWQgZW50aXR5IHJlY29nbml6ZXIgdHJhaW5pbmcgam9iXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3RvcFRyYWluaW5nRW50aXR5UmVjb2duaXplci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TdG9wVHJhaW5pbmdFbnRpdHlSZWNvZ25pemVyKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdG9wVHJhaW5pbmdFbnRpdHlSZWNvZ25pemVyJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdGFnIGEgcmVzb3VyY2Ugd2l0aCBnaXZlbiBrZXkgdmFsdWUgcGFpcnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1RhZ1Jlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1RhZ1Jlc291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdUYWdSZXNvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVudGFnIGEgcmVzb3VyY2Ugd2l0aCBnaXZlbiBrZXlcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VudGFnUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVW50YWdSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVW50YWdSZXNvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBpbmZvcm1hdGlvbiBhYm91dCB0aGUgc3BlY2lmaWVkIGVuZHBvaW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlRW5kcG9pbnQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlRW5kcG9pbnQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUVuZHBvaW50Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gVXBkYXRlIGEgZmx5d2hlZWwncyBjb25maWd1cmF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlRmx5d2hlZWwuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlRmx5d2hlZWwoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUZseXdoZWVsJyk7XG4gIH1cblxuICBwcm90ZWN0ZWQgYWNjZXNzTGV2ZWxMaXN0OiBBY2Nlc3NMZXZlbExpc3QgPSB7XG4gICAgUmVhZDogW1xuICAgICAgJ0JhdGNoRGV0ZWN0RG9taW5hbnRMYW5ndWFnZScsXG4gICAgICAnQmF0Y2hEZXRlY3RFbnRpdGllcycsXG4gICAgICAnQmF0Y2hEZXRlY3RLZXlQaHJhc2VzJyxcbiAgICAgICdCYXRjaERldGVjdFNlbnRpbWVudCcsXG4gICAgICAnQmF0Y2hEZXRlY3RTeW50YXgnLFxuICAgICAgJ0JhdGNoRGV0ZWN0VGFyZ2V0ZWRTZW50aW1lbnQnLFxuICAgICAgJ0NsYXNzaWZ5RG9jdW1lbnQnLFxuICAgICAgJ0NvbnRhaW5zUGlpRW50aXRpZXMnLFxuICAgICAgJ0Rlc2NyaWJlRGF0YXNldCcsXG4gICAgICAnRGVzY3JpYmVEb2N1bWVudENsYXNzaWZpY2F0aW9uSm9iJyxcbiAgICAgICdEZXNjcmliZURvY3VtZW50Q2xhc3NpZmllcicsXG4gICAgICAnRGVzY3JpYmVEb21pbmFudExhbmd1YWdlRGV0ZWN0aW9uSm9iJyxcbiAgICAgICdEZXNjcmliZUVuZHBvaW50JyxcbiAgICAgICdEZXNjcmliZUVudGl0aWVzRGV0ZWN0aW9uSm9iJyxcbiAgICAgICdEZXNjcmliZUVudGl0eVJlY29nbml6ZXInLFxuICAgICAgJ0Rlc2NyaWJlRXZlbnRzRGV0ZWN0aW9uSm9iJyxcbiAgICAgICdEZXNjcmliZUZseXdoZWVsJyxcbiAgICAgICdEZXNjcmliZUZseXdoZWVsSXRlcmF0aW9uJyxcbiAgICAgICdEZXNjcmliZUtleVBocmFzZXNEZXRlY3Rpb25Kb2InLFxuICAgICAgJ0Rlc2NyaWJlUGlpRW50aXRpZXNEZXRlY3Rpb25Kb2InLFxuICAgICAgJ0Rlc2NyaWJlUmVzb3VyY2VQb2xpY3knLFxuICAgICAgJ0Rlc2NyaWJlU2VudGltZW50RGV0ZWN0aW9uSm9iJyxcbiAgICAgICdEZXNjcmliZVRhcmdldGVkU2VudGltZW50RGV0ZWN0aW9uSm9iJyxcbiAgICAgICdEZXNjcmliZVRvcGljc0RldGVjdGlvbkpvYicsXG4gICAgICAnRGV0ZWN0RG9taW5hbnRMYW5ndWFnZScsXG4gICAgICAnRGV0ZWN0RW50aXRpZXMnLFxuICAgICAgJ0RldGVjdEtleVBocmFzZXMnLFxuICAgICAgJ0RldGVjdFBpaUVudGl0aWVzJyxcbiAgICAgICdEZXRlY3RTZW50aW1lbnQnLFxuICAgICAgJ0RldGVjdFN5bnRheCcsXG4gICAgICAnRGV0ZWN0VGFyZ2V0ZWRTZW50aW1lbnQnLFxuICAgICAgJ0RldGVjdFRveGljQ29udGVudCcsXG4gICAgICAnTGlzdERhdGFzZXRzJyxcbiAgICAgICdMaXN0RG9jdW1lbnRDbGFzc2lmaWNhdGlvbkpvYnMnLFxuICAgICAgJ0xpc3REb2N1bWVudENsYXNzaWZpZXJTdW1tYXJpZXMnLFxuICAgICAgJ0xpc3REb2N1bWVudENsYXNzaWZpZXJzJyxcbiAgICAgICdMaXN0RG9taW5hbnRMYW5ndWFnZURldGVjdGlvbkpvYnMnLFxuICAgICAgJ0xpc3RFbmRwb2ludHMnLFxuICAgICAgJ0xpc3RFbnRpdGllc0RldGVjdGlvbkpvYnMnLFxuICAgICAgJ0xpc3RFbnRpdHlSZWNvZ25pemVyU3VtbWFyaWVzJyxcbiAgICAgICdMaXN0RW50aXR5UmVjb2duaXplcnMnLFxuICAgICAgJ0xpc3RFdmVudHNEZXRlY3Rpb25Kb2JzJyxcbiAgICAgICdMaXN0Rmx5d2hlZWxJdGVyYXRpb25IaXN0b3J5JyxcbiAgICAgICdMaXN0Rmx5d2hlZWxzJyxcbiAgICAgICdMaXN0S2V5UGhyYXNlc0RldGVjdGlvbkpvYnMnLFxuICAgICAgJ0xpc3RQaWlFbnRpdGllc0RldGVjdGlvbkpvYnMnLFxuICAgICAgJ0xpc3RTZW50aW1lbnREZXRlY3Rpb25Kb2JzJyxcbiAgICAgICdMaXN0VGFnc0ZvclJlc291cmNlJyxcbiAgICAgICdMaXN0VGFyZ2V0ZWRTZW50aW1lbnREZXRlY3Rpb25Kb2JzJyxcbiAgICAgICdMaXN0VG9waWNzRGV0ZWN0aW9uSm9icydcbiAgICBdLFxuICAgIFdyaXRlOiBbXG4gICAgICAnQ3JlYXRlRGF0YXNldCcsXG4gICAgICAnQ3JlYXRlRG9jdW1lbnRDbGFzc2lmaWVyJyxcbiAgICAgICdDcmVhdGVFbmRwb2ludCcsXG4gICAgICAnQ3JlYXRlRW50aXR5UmVjb2duaXplcicsXG4gICAgICAnQ3JlYXRlRmx5d2hlZWwnLFxuICAgICAgJ0RlbGV0ZURvY3VtZW50Q2xhc3NpZmllcicsXG4gICAgICAnRGVsZXRlRW5kcG9pbnQnLFxuICAgICAgJ0RlbGV0ZUVudGl0eVJlY29nbml6ZXInLFxuICAgICAgJ0RlbGV0ZUZseXdoZWVsJyxcbiAgICAgICdEZWxldGVSZXNvdXJjZVBvbGljeScsXG4gICAgICAnSW1wb3J0TW9kZWwnLFxuICAgICAgJ1B1dFJlc291cmNlUG9saWN5JyxcbiAgICAgICdTdGFydERvY3VtZW50Q2xhc3NpZmljYXRpb25Kb2InLFxuICAgICAgJ1N0YXJ0RG9taW5hbnRMYW5ndWFnZURldGVjdGlvbkpvYicsXG4gICAgICAnU3RhcnRFbnRpdGllc0RldGVjdGlvbkpvYicsXG4gICAgICAnU3RhcnRFdmVudHNEZXRlY3Rpb25Kb2InLFxuICAgICAgJ1N0YXJ0Rmx5d2hlZWxJdGVyYXRpb24nLFxuICAgICAgJ1N0YXJ0S2V5UGhyYXNlc0RldGVjdGlvbkpvYicsXG4gICAgICAnU3RhcnRQaWlFbnRpdGllc0RldGVjdGlvbkpvYicsXG4gICAgICAnU3RhcnRTZW50aW1lbnREZXRlY3Rpb25Kb2InLFxuICAgICAgJ1N0YXJ0VGFyZ2V0ZWRTZW50aW1lbnREZXRlY3Rpb25Kb2InLFxuICAgICAgJ1N0YXJ0VG9waWNzRGV0ZWN0aW9uSm9iJyxcbiAgICAgICdTdG9wRG9taW5hbnRMYW5ndWFnZURldGVjdGlvbkpvYicsXG4gICAgICAnU3RvcEVudGl0aWVzRGV0ZWN0aW9uSm9iJyxcbiAgICAgICdTdG9wRXZlbnRzRGV0ZWN0aW9uSm9iJyxcbiAgICAgICdTdG9wS2V5UGhyYXNlc0RldGVjdGlvbkpvYicsXG4gICAgICAnU3RvcFBpaUVudGl0aWVzRGV0ZWN0aW9uSm9iJyxcbiAgICAgICdTdG9wU2VudGltZW50RGV0ZWN0aW9uSm9iJyxcbiAgICAgICdTdG9wVGFyZ2V0ZWRTZW50aW1lbnREZXRlY3Rpb25Kb2InLFxuICAgICAgJ1N0b3BUcmFpbmluZ0RvY3VtZW50Q2xhc3NpZmllcicsXG4gICAgICAnU3RvcFRyYWluaW5nRW50aXR5UmVjb2duaXplcicsXG4gICAgICAnVXBkYXRlRW5kcG9pbnQnLFxuICAgICAgJ1VwZGF0ZUZseXdoZWVsJ1xuICAgIF0sXG4gICAgVGFnZ2luZzogW1xuICAgICAgJ1RhZ1Jlc291cmNlJyxcbiAgICAgICdVbnRhZ1Jlc291cmNlJ1xuICAgIF1cbiAgfTtcblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgdGFyZ2V0ZWQtc2VudGltZW50LWRldGVjdGlvbi1qb2IgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TdGFydFRhcmdldGVkU2VudGltZW50RGV0ZWN0aW9uSm9iLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGpvYklkIC0gSWRlbnRpZmllciBmb3IgdGhlIGpvYklkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25UYXJnZXRlZFNlbnRpbWVudERldGVjdGlvbkpvYihqb2JJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmNvbXByZWhlbmQ6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OnRhcmdldGVkLXNlbnRpbWVudC1kZXRlY3Rpb24tam9iLyR7IGpvYklkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBkb2N1bWVudC1jbGFzc2lmaWVyIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L2RnL2hvdy1kb2N1bWVudC1jbGFzc2lmaWNhdGlvbi10cmFpbmluZy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBkb2N1bWVudENsYXNzaWZpZXJOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGRvY3VtZW50Q2xhc3NpZmllck5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkRvY3VtZW50Q2xhc3NpZmllcihkb2N1bWVudENsYXNzaWZpZXJOYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06Y29tcHJlaGVuZDokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06ZG9jdW1lbnQtY2xhc3NpZmllci8keyBkb2N1bWVudENsYXNzaWZpZXJOYW1lIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBkb2N1bWVudC1jbGFzc2lmaWVyLWVuZHBvaW50IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L2RnL21hbmFnZS1lbmRwb2ludHMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gZG9jdW1lbnRDbGFzc2lmaWVyRW5kcG9pbnROYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGRvY3VtZW50Q2xhc3NpZmllckVuZHBvaW50TmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRG9jdW1lbnRDbGFzc2lmaWVyRW5kcG9pbnQoZG9jdW1lbnRDbGFzc2lmaWVyRW5kcG9pbnROYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06Y29tcHJlaGVuZDokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06ZG9jdW1lbnQtY2xhc3NpZmllci1lbmRwb2ludC8keyBkb2N1bWVudENsYXNzaWZpZXJFbmRwb2ludE5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGVudGl0eS1yZWNvZ25pemVyIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L2RnL3RyYWluaW5nLXJlY29nbml6ZXJzLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGVudGl0eVJlY29nbml6ZXJOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGVudGl0eVJlY29nbml6ZXJOYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25FbnRpdHlSZWNvZ25pemVyKGVudGl0eVJlY29nbml6ZXJOYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06Y29tcHJlaGVuZDokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06ZW50aXR5LXJlY29nbml6ZXIvJHsgZW50aXR5UmVjb2duaXplck5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGVudGl0eS1yZWNvZ25pemVyLWVuZHBvaW50IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L2RnL21hbmFnZS1lbmRwb2ludHMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gZW50aXR5UmVjb2duaXplckVuZHBvaW50TmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBlbnRpdHlSZWNvZ25pemVyRW5kcG9pbnROYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25FbnRpdHlSZWNvZ25pemVyRW5kcG9pbnQoZW50aXR5UmVjb2duaXplckVuZHBvaW50TmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmNvbXByZWhlbmQ6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmVudGl0eS1yZWNvZ25pemVyLWVuZHBvaW50LyR7IGVudGl0eVJlY29nbml6ZXJFbmRwb2ludE5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGRvbWluYW50LWxhbmd1YWdlLWRldGVjdGlvbi1qb2IgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TdGFydERvbWluYW50TGFuZ3VhZ2VEZXRlY3Rpb25Kb2IuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gam9iSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgam9iSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkRvbWluYW50TGFuZ3VhZ2VEZXRlY3Rpb25Kb2Ioam9iSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpjb21wcmVoZW5kOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpkb21pbmFudC1sYW5ndWFnZS1kZXRlY3Rpb24tam9iLyR7IGpvYklkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBlbnRpdGllcy1kZXRlY3Rpb24tam9iIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3RhcnRFbnRpdGllc0RldGVjdGlvbkpvYi5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBqb2JJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBqb2JJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRW50aXRpZXNEZXRlY3Rpb25Kb2Ioam9iSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpjb21wcmVoZW5kOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTplbnRpdGllcy1kZXRlY3Rpb24tam9iLyR7IGpvYklkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBwaWktZW50aXRpZXMtZGV0ZWN0aW9uLWpvYiB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb21wcmVoZW5kL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N0YXJ0UGlpRW50aXRpZXNEZXRlY3Rpb25Kb2IuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gam9iSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgam9iSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblBpaUVudGl0aWVzRGV0ZWN0aW9uSm9iKGpvYklkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06Y29tcHJlaGVuZDokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06cGlpLWVudGl0aWVzLWRldGVjdGlvbi1qb2IvJHsgam9iSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGV2ZW50cy1kZXRlY3Rpb24tam9iIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3RhcnRFdmVudHNEZXRlY3Rpb25Kb2IuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gam9iSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgam9iSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkV2ZW50c0RldGVjdGlvbkpvYihqb2JJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmNvbXByZWhlbmQ6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmV2ZW50cy1kZXRlY3Rpb24tam9iLyR7IGpvYklkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBrZXktcGhyYXNlcy1kZXRlY3Rpb24tam9iIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3RhcnRLZXlQaHJhc2VzRGV0ZWN0aW9uSm9iLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGpvYklkIC0gSWRlbnRpZmllciBmb3IgdGhlIGpvYklkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25LZXlQaHJhc2VzRGV0ZWN0aW9uSm9iKGpvYklkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06Y29tcHJlaGVuZDokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06a2V5LXBocmFzZXMtZGV0ZWN0aW9uLWpvYi8keyBqb2JJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgc2VudGltZW50LWRldGVjdGlvbi1qb2IgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TdGFydFNlbnRpbWVudERldGVjdGlvbkpvYi5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBqb2JJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBqb2JJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uU2VudGltZW50RGV0ZWN0aW9uSm9iKGpvYklkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06Y29tcHJlaGVuZDokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06c2VudGltZW50LWRldGVjdGlvbi1qb2IvJHsgam9iSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHRvcGljcy1kZXRlY3Rpb24tam9iIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3RhcnRUb3BpY3NEZXRlY3Rpb25Kb2IuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gam9iSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgam9iSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblRvcGljc0RldGVjdGlvbkpvYihqb2JJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmNvbXByZWhlbmQ6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OnRvcGljcy1kZXRlY3Rpb24tam9iLyR7IGpvYklkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBkb2N1bWVudC1jbGFzc2lmaWNhdGlvbi1qb2IgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29tcHJlaGVuZC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TdGFydERvY3VtZW50Q2xhc3NpZmljYXRpb25Kb2IuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gam9iSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgam9iSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkRvY3VtZW50Q2xhc3NpZmljYXRpb25Kb2Ioam9iSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpjb21wcmVoZW5kOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpkb2N1bWVudC1jbGFzc2lmaWNhdGlvbi1qb2IvJHsgam9iSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGZseXdoZWVsIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlRmx5d2hlZWwuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gZmx5d2hlZWxOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGZseXdoZWVsTmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRmx5d2hlZWwoZmx5d2hlZWxOYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06Y29tcHJlaGVuZDokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06Zmx5d2hlZWwvJHsgZmx5d2hlZWxOYW1lIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBmbHl3aGVlbC1kYXRhc2V0IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvbXByZWhlbmQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlRGF0YXNldC5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBmbHl3aGVlbE5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgZmx5d2hlZWxOYW1lLlxuICAgKiBAcGFyYW0gZGF0YXNldE5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgZGF0YXNldE5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkZseXdoZWVsRGF0YXNldChmbHl3aGVlbE5hbWU6IHN0cmluZywgZGF0YXNldE5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpjb21wcmVoZW5kOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpmbHl3aGVlbC8keyBmbHl3aGVlbE5hbWUgfS9kYXRhc2V0LyR7IGRhdGFzZXROYW1lIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSByZXF1aXJpbmcgdGFnIHZhbHVlcyBwcmVzZW50IGluIGEgcmVzb3VyY2UgY3JlYXRpb24gcmVxdWVzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtZ2xvYmFsbHktYXZhaWxhYmxlXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9DcmVhdGVEYXRhc2V0KClcbiAgICogLSAudG9DcmVhdGVEb2N1bWVudENsYXNzaWZpZXIoKVxuICAgKiAtIC50b0NyZWF0ZUVuZHBvaW50KClcbiAgICogLSAudG9DcmVhdGVFbnRpdHlSZWNvZ25pemVyKClcbiAgICogLSAudG9DcmVhdGVGbHl3aGVlbCgpXG4gICAqIC0gLnRvSW1wb3J0TW9kZWwoKVxuICAgKiAtIC50b1N0YXJ0RG9jdW1lbnRDbGFzc2lmaWNhdGlvbkpvYigpXG4gICAqIC0gLnRvU3RhcnREb21pbmFudExhbmd1YWdlRGV0ZWN0aW9uSm9iKClcbiAgICogLSAudG9TdGFydEVudGl0aWVzRGV0ZWN0aW9uSm9iKClcbiAgICogLSAudG9TdGFydEV2ZW50c0RldGVjdGlvbkpvYigpXG4gICAqIC0gLnRvU3RhcnRLZXlQaHJhc2VzRGV0ZWN0aW9uSm9iKClcbiAgICogLSAudG9TdGFydFBpaUVudGl0aWVzRGV0ZWN0aW9uSm9iKClcbiAgICogLSAudG9TdGFydFNlbnRpbWVudERldGVjdGlvbkpvYigpXG4gICAqIC0gLnRvU3RhcnRUYXJnZXRlZFNlbnRpbWVudERldGVjdGlvbkpvYigpXG4gICAqIC0gLnRvU3RhcnRUb3BpY3NEZXRlY3Rpb25Kb2IoKVxuICAgKiAtIC50b1RhZ1Jlc291cmNlKClcbiAgICpcbiAgICogQHBhcmFtIHRhZ0tleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1JlcXVlc3RUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6UmVxdWVzdFRhZy8keyB0YWdLZXkgfWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHJlcXVpcmluZyB0YWcgdmFsdWUgYXNzb2NpYXRlZCB3aXRoIHRoZSByZXNvdXJjZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtZ2xvYmFsbHktYXZhaWxhYmxlXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gdGFyZ2V0ZWQtc2VudGltZW50LWRldGVjdGlvbi1qb2JcbiAgICogLSBkb2N1bWVudC1jbGFzc2lmaWVyXG4gICAqIC0gZG9jdW1lbnQtY2xhc3NpZmllci1lbmRwb2ludFxuICAgKiAtIGVudGl0eS1yZWNvZ25pemVyXG4gICAqIC0gZW50aXR5LXJlY29nbml6ZXItZW5kcG9pbnRcbiAgICogLSBkb21pbmFudC1sYW5ndWFnZS1kZXRlY3Rpb24tam9iXG4gICAqIC0gZW50aXRpZXMtZGV0ZWN0aW9uLWpvYlxuICAgKiAtIHBpaS1lbnRpdGllcy1kZXRlY3Rpb24tam9iXG4gICAqIC0gZXZlbnRzLWRldGVjdGlvbi1qb2JcbiAgICogLSBrZXktcGhyYXNlcy1kZXRlY3Rpb24tam9iXG4gICAqIC0gc2VudGltZW50LWRldGVjdGlvbi1qb2JcbiAgICogLSB0b3BpY3MtZGV0ZWN0aW9uLWpvYlxuICAgKiAtIGRvY3VtZW50LWNsYXNzaWZpY2F0aW9uLWpvYlxuICAgKiAtIGZseXdoZWVsXG4gICAqIC0gZmx5d2hlZWwtZGF0YXNldFxuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzUmVzb3VyY2VUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6UmVzb3VyY2VUYWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSByZXF1aXJpbmcgdGhlIHByZXNlbmNlIG9mIG1hbmRhdG9yeSB0YWdzIGluIHRoZSByZXF1ZXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy1nbG9iYWxseS1hdmFpbGFibGVcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZURhdGFzZXQoKVxuICAgKiAtIC50b0NyZWF0ZURvY3VtZW50Q2xhc3NpZmllcigpXG4gICAqIC0gLnRvQ3JlYXRlRW5kcG9pbnQoKVxuICAgKiAtIC50b0NyZWF0ZUVudGl0eVJlY29nbml6ZXIoKVxuICAgKiAtIC50b0NyZWF0ZUZseXdoZWVsKClcbiAgICogLSAudG9JbXBvcnRNb2RlbCgpXG4gICAqIC0gLnRvU3RhcnREb2N1bWVudENsYXNzaWZpY2F0aW9uSm9iKClcbiAgICogLSAudG9TdGFydERvbWluYW50TGFuZ3VhZ2VEZXRlY3Rpb25Kb2IoKVxuICAgKiAtIC50b1N0YXJ0RW50aXRpZXNEZXRlY3Rpb25Kb2IoKVxuICAgKiAtIC50b1N0YXJ0RXZlbnRzRGV0ZWN0aW9uSm9iKClcbiAgICogLSAudG9TdGFydEtleVBocmFzZXNEZXRlY3Rpb25Kb2IoKVxuICAgKiAtIC50b1N0YXJ0UGlpRW50aXRpZXNEZXRlY3Rpb25Kb2IoKVxuICAgKiAtIC50b1N0YXJ0U2VudGltZW50RGV0ZWN0aW9uSm9iKClcbiAgICogLSAudG9TdGFydFRhcmdldGVkU2VudGltZW50RGV0ZWN0aW9uSm9iKClcbiAgICogLSAudG9TdGFydFRvcGljc0RldGVjdGlvbkpvYigpXG4gICAqIC0gLnRvVGFnUmVzb3VyY2UoKVxuICAgKiAtIC50b1VudGFnUmVzb3VyY2UoKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1RhZ0tleXModmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpUYWdLZXlzYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGhlIERhdGFMYWtlIEttcyBLZXkgYXNzb2NpYXRlZCB3aXRoIHRoZSBmbHl3aGVlbCByZXNvdXJjZSBpbiB0aGUgcmVxdWVzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hbWF6b25jb21wcmVoZW5kLmh0bWwjYW1hem9uY29tcHJlaGVuZC1wb2xpY3kta2V5c1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQ3JlYXRlRmx5d2hlZWwoKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFthcm4gb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX0FSTikuICoqRGVmYXVsdDoqKiBgQXJuTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkRhdGFMYWtlS21zS2V5KHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBEYXRhTGFrZUttc0tleWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnQXJuTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHBhcnRpY3VsYXIgSXRlcmF0aW9uIElkIGZvciBhIGZseXdoZWVsXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9saXN0X2FtYXpvbmNvbXByZWhlbmQuaHRtbCNhbWF6b25jb21wcmVoZW5kLXBvbGljeS1rZXlzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9EZXNjcmliZUZseXdoZWVsSXRlcmF0aW9uKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZGbHl3aGVlbEl0ZXJhdGlvbklkKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBGbHl3aGVlbEl0ZXJhdGlvbklkYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGhlIG1vZGVsIEtNUyBrZXkgYXNzb2NpYXRlZCB3aXRoIHRoZSByZXNvdXJjZSBpbiB0aGUgcmVxdWVzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hbWF6b25jb21wcmVoZW5kLmh0bWwjYW1hem9uY29tcHJlaGVuZC1wb2xpY3kta2V5c1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQ3JlYXRlRG9jdW1lbnRDbGFzc2lmaWVyKClcbiAgICogLSAudG9DcmVhdGVFbnRpdHlSZWNvZ25pemVyKClcbiAgICogLSAudG9DcmVhdGVGbHl3aGVlbCgpXG4gICAqIC0gLnRvSW1wb3J0TW9kZWwoKVxuICAgKiAtIC50b1VwZGF0ZUZseXdoZWVsKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbYXJuIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19BUk4pLiAqKkRlZmF1bHQ6KiogYEFybkxpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZNb2RlbEttc0tleSh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgTW9kZWxLbXNLZXlgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ0Fybkxpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgb3V0cHV0IEtNUyBrZXkgYXNzb2NpYXRlZCB3aXRoIHRoZSByZXNvdXJjZSBpbiB0aGUgcmVxdWVzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hbWF6b25jb21wcmVoZW5kLmh0bWwjYW1hem9uY29tcHJlaGVuZC1wb2xpY3kta2V5c1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQ3JlYXRlRG9jdW1lbnRDbGFzc2lmaWVyKClcbiAgICogLSAudG9TdGFydERvY3VtZW50Q2xhc3NpZmljYXRpb25Kb2IoKVxuICAgKiAtIC50b1N0YXJ0RG9taW5hbnRMYW5ndWFnZURldGVjdGlvbkpvYigpXG4gICAqIC0gLnRvU3RhcnRFbnRpdGllc0RldGVjdGlvbkpvYigpXG4gICAqIC0gLnRvU3RhcnRFdmVudHNEZXRlY3Rpb25Kb2IoKVxuICAgKiAtIC50b1N0YXJ0S2V5UGhyYXNlc0RldGVjdGlvbkpvYigpXG4gICAqIC0gLnRvU3RhcnRQaWlFbnRpdGllc0RldGVjdGlvbkpvYigpXG4gICAqIC0gLnRvU3RhcnRTZW50aW1lbnREZXRlY3Rpb25Kb2IoKVxuICAgKiAtIC50b1N0YXJ0VGFyZ2V0ZWRTZW50aW1lbnREZXRlY3Rpb25Kb2IoKVxuICAgKiAtIC50b1N0YXJ0VG9waWNzRGV0ZWN0aW9uSm9iKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbYXJuIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19BUk4pLiAqKkRlZmF1bHQ6KiogYEFybkxpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZPdXRwdXRLbXNLZXkodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYE91dHB1dEttc0tleWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnQXJuTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSB2b2x1bWUgS01TIGtleSBhc3NvY2lhdGVkIHdpdGggdGhlIHJlc291cmNlIGluIHRoZSByZXF1ZXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9saXN0X2FtYXpvbmNvbXByZWhlbmQuaHRtbCNhbWF6b25jb21wcmVoZW5kLXBvbGljeS1rZXlzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9DcmVhdGVEb2N1bWVudENsYXNzaWZpZXIoKVxuICAgKiAtIC50b0NyZWF0ZUVudGl0eVJlY29nbml6ZXIoKVxuICAgKiAtIC50b0NyZWF0ZUZseXdoZWVsKClcbiAgICogLSAudG9TdGFydERvY3VtZW50Q2xhc3NpZmljYXRpb25Kb2IoKVxuICAgKiAtIC50b1N0YXJ0RG9taW5hbnRMYW5ndWFnZURldGVjdGlvbkpvYigpXG4gICAqIC0gLnRvU3RhcnRFbnRpdGllc0RldGVjdGlvbkpvYigpXG4gICAqIC0gLnRvU3RhcnRLZXlQaHJhc2VzRGV0ZWN0aW9uSm9iKClcbiAgICogLSAudG9TdGFydFNlbnRpbWVudERldGVjdGlvbkpvYigpXG4gICAqIC0gLnRvU3RhcnRUYXJnZXRlZFNlbnRpbWVudERldGVjdGlvbkpvYigpXG4gICAqIC0gLnRvU3RhcnRUb3BpY3NEZXRlY3Rpb25Kb2IoKVxuICAgKiAtIC50b1VwZGF0ZUZseXdoZWVsKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbYXJuIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19BUk4pLiAqKkRlZmF1bHQ6KiogYEFybkxpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZWb2x1bWVLbXNLZXkodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFZvbHVtZUttc0tleWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnQXJuTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSBsaXN0IG9mIGFsbCBWUEMgc2VjdXJpdHkgZ3JvdXAgaWRzIGFzc29jaWF0ZWQgd2l0aCB0aGUgcmVzb3VyY2UgaW4gdGhlIHJlcXVlc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2xpc3RfYW1hem9uY29tcHJlaGVuZC5odG1sI2FtYXpvbmNvbXByZWhlbmQtcG9saWN5LWtleXNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZURvY3VtZW50Q2xhc3NpZmllcigpXG4gICAqIC0gLnRvQ3JlYXRlRW50aXR5UmVjb2duaXplcigpXG4gICAqIC0gLnRvQ3JlYXRlRmx5d2hlZWwoKVxuICAgKiAtIC50b1N0YXJ0RG9jdW1lbnRDbGFzc2lmaWNhdGlvbkpvYigpXG4gICAqIC0gLnRvU3RhcnREb21pbmFudExhbmd1YWdlRGV0ZWN0aW9uSm9iKClcbiAgICogLSAudG9TdGFydEVudGl0aWVzRGV0ZWN0aW9uSm9iKClcbiAgICogLSAudG9TdGFydEtleVBocmFzZXNEZXRlY3Rpb25Kb2IoKVxuICAgKiAtIC50b1N0YXJ0U2VudGltZW50RGV0ZWN0aW9uSm9iKClcbiAgICogLSAudG9TdGFydFRhcmdldGVkU2VudGltZW50RGV0ZWN0aW9uSm9iKClcbiAgICogLSAudG9TdGFydFRvcGljc0RldGVjdGlvbkpvYigpXG4gICAqIC0gLnRvVXBkYXRlRmx5d2hlZWwoKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlZwY1NlY3VyaXR5R3JvdXBJZHModmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFZwY1NlY3VyaXR5R3JvdXBJZHNgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgbGlzdCBvZiBhbGwgVlBDIHN1Ym5ldHMgYXNzb2NpYXRlZCB3aXRoIHRoZSByZXNvdXJjZSBpbiB0aGUgcmVxdWVzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hbWF6b25jb21wcmVoZW5kLmh0bWwjYW1hem9uY29tcHJlaGVuZC1wb2xpY3kta2V5c1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQ3JlYXRlRG9jdW1lbnRDbGFzc2lmaWVyKClcbiAgICogLSAudG9DcmVhdGVFbnRpdHlSZWNvZ25pemVyKClcbiAgICogLSAudG9DcmVhdGVGbHl3aGVlbCgpXG4gICAqIC0gLnRvU3RhcnREb2N1bWVudENsYXNzaWZpY2F0aW9uSm9iKClcbiAgICogLSAudG9TdGFydERvbWluYW50TGFuZ3VhZ2VEZXRlY3Rpb25Kb2IoKVxuICAgKiAtIC50b1N0YXJ0RW50aXRpZXNEZXRlY3Rpb25Kb2IoKVxuICAgKiAtIC50b1N0YXJ0S2V5UGhyYXNlc0RldGVjdGlvbkpvYigpXG4gICAqIC0gLnRvU3RhcnRTZW50aW1lbnREZXRlY3Rpb25Kb2IoKVxuICAgKiAtIC50b1N0YXJ0VGFyZ2V0ZWRTZW50aW1lbnREZXRlY3Rpb25Kb2IoKVxuICAgKiAtIC50b1N0YXJ0VG9waWNzRGV0ZWN0aW9uSm9iKClcbiAgICogLSAudG9VcGRhdGVGbHl3aGVlbCgpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmVnBjU3VibmV0cyh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgVnBjU3VibmV0c2AsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbY29tcHJlaGVuZF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYW1hem9uY29tcHJlaGVuZC5odG1sKS5cbiAgICpcbiAgICovXG4gIGNvbnN0cnVjdG9yKHByb3BzPzogaWFtLlBvbGljeVN0YXRlbWVudFByb3BzKSB7XG4gICAgc3VwZXIocHJvcHMpO1xuICB9XG59XG4iXX0=