"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConnectCampaigns = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [connect-campaigns](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonconnectoutboundcampaigns.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class ConnectCampaigns extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a campaign
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toCreateCampaign() {
        return this.to('CreateCampaign');
    }
    /**
     * Grants permission to delete a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toDeleteCampaign() {
        return this.to('DeleteCampaign');
    }
    /**
     * Grants permission to delete the channel subtype configuration of a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toDeleteCampaignChannelSubtypeConfig() {
        return this.to('DeleteCampaignChannelSubtypeConfig');
    }
    /**
     * Grants permission to delete the communication limits configuration of a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toDeleteCampaignCommunicationLimits() {
        return this.to('DeleteCampaignCommunicationLimits');
    }
    /**
     * Grants permission to delete the communication time configuration of a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toDeleteCampaignCommunicationTime() {
        return this.to('DeleteCampaignCommunicationTime');
    }
    /**
     * Grants permission to remove configuration information for an Amazon Connect instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toDeleteConnectInstanceConfig() {
        return this.to('DeleteConnectInstanceConfig');
    }
    /**
     * Grants permission to remove integration information for an Amazon Connect instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toDeleteConnectInstanceIntegration() {
        return this.to('DeleteConnectInstanceIntegration');
    }
    /**
     * Grants permission to remove onboarding job for an Amazon Connect instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toDeleteInstanceOnboardingJob() {
        return this.to('DeleteInstanceOnboardingJob');
    }
    /**
     * Grants permission to describe a specific campaign
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toDescribeCampaign() {
        return this.to('DescribeCampaign');
    }
    /**
     * Grants permission to get state of a campaign
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toGetCampaignState() {
        return this.to('GetCampaignState');
    }
    /**
     * Grants permission to get state of campaigns
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toGetCampaignStateBatch() {
        return this.to('GetCampaignStateBatch');
    }
    /**
     * Grants permission to get configuration information for an Amazon Connect instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toGetConnectInstanceConfig() {
        return this.to('GetConnectInstanceConfig');
    }
    /**
     * Grants permission to get onboarding job status for an Amazon Connect instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toGetInstanceOnboardingJobStatus() {
        return this.to('GetInstanceOnboardingJobStatus');
    }
    /**
     * Grants permission to provide summary of all campaigns
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toListCampaigns() {
        return this.to('ListCampaigns');
    }
    /**
     * Grants permission to provide summary of all integrations with an Amazon Connect instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toListConnectInstanceIntegrations() {
        return this.to('ListConnectInstanceIntegrations');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to pause a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toPauseCampaign() {
        return this.to('PauseCampaign');
    }
    /**
     * Grants permission to put an integration configuration with an Amazon Connect instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toPutConnectInstanceIntegration() {
        return this.to('PutConnectInstanceIntegration');
    }
    /**
     * Grants permission to create dial requests for the specified campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toPutDialRequestBatch() {
        return this.to('PutDialRequestBatch');
    }
    /**
     * Grants permission to create dial requests for the specified campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toPutOutboundRequestBatch() {
        return this.to('PutOutboundRequestBatch');
    }
    /**
     * Grants permission to create profile outbound requests for the specified campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toPutProfileOutboundRequestBatch() {
        return this.to('PutProfileOutboundRequestBatch');
    }
    /**
     * Grants permission to resume a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toResumeCampaign() {
        return this.to('ResumeCampaign');
    }
    /**
     * Grants permission to start a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toStartCampaign() {
        return this.to('StartCampaign');
    }
    /**
     * Grants permission to start onboarding job for an Amazon Connect instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toStartInstanceOnboardingJob() {
        return this.to('StartInstanceOnboardingJob');
    }
    /**
     * Grants permission to stop a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toStopCampaign() {
        return this.to('StopCampaign');
    }
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the channel subtype configuration of a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toUpdateCampaignChannelSubtypeConfig() {
        return this.to('UpdateCampaignChannelSubtypeConfig');
    }
    /**
     * Grants permission to update the communication limits configuration of a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toUpdateCampaignCommunicationLimits() {
        return this.to('UpdateCampaignCommunicationLimits');
    }
    /**
     * Grants permission to update the communication time configuration of a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toUpdateCampaignCommunicationTime() {
        return this.to('UpdateCampaignCommunicationTime');
    }
    /**
     * Grants permission to update the dialer configuration of a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toUpdateCampaignDialerConfig() {
        return this.to('UpdateCampaignDialerConfig');
    }
    /**
     * Grants permission to update the flow association of a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toUpdateCampaignFlowAssociation() {
        return this.to('UpdateCampaignFlowAssociation');
    }
    /**
     * Grants permission to update the name of a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toUpdateCampaignName() {
        return this.to('UpdateCampaignName');
    }
    /**
     * Grants permission to update the outbound call configuration of a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toUpdateCampaignOutboundCallConfig() {
        return this.to('UpdateCampaignOutboundCallConfig');
    }
    /**
     * Grants permission to update the schedule of a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toUpdateCampaignSchedule() {
        return this.to('UpdateCampaignSchedule');
    }
    /**
     * Grants permission to update the source of a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toUpdateCampaignSource() {
        return this.to('UpdateCampaignSource');
    }
    /**
     * Adds a resource of type campaign to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     *
     * @param campaignId - Identifier for the campaignId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCampaign(campaignId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:connect-campaigns:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:campaign/${campaignId}`);
    }
    /**
     * Filters access by actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateCampaign()
     * - .toDescribeCampaign()
     * - .toGetCampaignState()
     * - .toGetCampaignStateBatch()
     * - .toListCampaigns()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toListTagsForResource()
     *
     * Applies to resource types:
     * - campaign
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateCampaign()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [connect-campaigns](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonconnectoutboundcampaigns.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'connect-campaigns';
        this.accessLevelList = {
            Write: [
                'CreateCampaign',
                'DeleteCampaign',
                'DeleteCampaignChannelSubtypeConfig',
                'DeleteCampaignCommunicationLimits',
                'DeleteCampaignCommunicationTime',
                'DeleteConnectInstanceConfig',
                'DeleteConnectInstanceIntegration',
                'DeleteInstanceOnboardingJob',
                'PauseCampaign',
                'PutConnectInstanceIntegration',
                'PutDialRequestBatch',
                'PutOutboundRequestBatch',
                'PutProfileOutboundRequestBatch',
                'ResumeCampaign',
                'StartCampaign',
                'StartInstanceOnboardingJob',
                'StopCampaign',
                'UpdateCampaignChannelSubtypeConfig',
                'UpdateCampaignCommunicationLimits',
                'UpdateCampaignCommunicationTime',
                'UpdateCampaignDialerConfig',
                'UpdateCampaignFlowAssociation',
                'UpdateCampaignName',
                'UpdateCampaignOutboundCallConfig',
                'UpdateCampaignSchedule',
                'UpdateCampaignSource'
            ],
            Read: [
                'DescribeCampaign',
                'GetCampaignState',
                'GetCampaignStateBatch',
                'GetConnectInstanceConfig',
                'GetInstanceOnboardingJobStatus',
                'ListTagsForResource'
            ],
            List: [
                'ListCampaigns',
                'ListConnectInstanceIntegrations'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.ConnectCampaigns = ConnectCampaigns;
//# sourceMappingURL=data:application/json;base64,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