"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Deepracer = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [deepracer](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdeepracer.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Deepracer extends shared_1.PolicyStatement {
    /**
     * Grants permission to add access for a private leaderboard
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-manage-community-races.html
     */
    toAddLeaderboardAccessPermission() {
        return this.to('AddLeaderboardAccessPermission');
    }
    /**
     * Grants permission to retrieve information of the KMS key that the user currently has assigned to their account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-admin-describe-account-key.html
     */
    toAdminDescribeAccountKey() {
        return this.to('AdminDescribeAccountKey');
    }
    /**
     * Grants permission to get current admin multiuser configuration for this account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-admin-set-account-config.html
     */
    toAdminGetAccountConfig() {
        return this.to('AdminGetAccountConfig');
    }
    /**
     * Grants permission to list all deepracer users with their associated resources created under this account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-admin-list-associated-resources.html
     */
    toAdminListAssociatedResources() {
        return this.to('AdminListAssociatedResources');
    }
    /**
     * Grants permission to list user data for all users associated with this account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-admin-list-associated-users.html
     */
    toAdminListAssociatedUsers() {
        return this.to('AdminListAssociatedUsers');
    }
    /**
     * Grants permission to manage a user associated with this account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-admin-manage-user.html
     */
    toAdminManageUser() {
        return this.to('AdminManageUser');
    }
    /**
     * Grants permission to set configuration options for this account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-admin-set-account-config.html
     */
    toAdminSetAccountConfig() {
        return this.to('AdminSetAccountConfig');
    }
    /**
     * Grants permission to update the KMS key that is assigned to the user's account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-admin-update-account-key.html
     */
    toAdminUpdateAccountKey() {
        return this.to('AdminUpdateAccountKey');
    }
    /**
     * Grants permission to clone an existing DeepRacer model
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html#deepracer-clone-trained-model
     */
    toCloneReinforcementLearningModel() {
        return this.to('CloneReinforcementLearningModel');
    }
    /**
     * Grants permission to create a DeepRacer car in your garage
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-choose-race-type.html
     */
    toCreateCar() {
        return this.to('CreateCar');
    }
    /**
     * Grants permission to create a leaderboard
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-create-community-race.html
     */
    toCreateLeaderboard() {
        return this.to('CreateLeaderboard');
    }
    /**
     * Grants permission to create an access token for a private leaderboard
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-manage-community-races.html
     */
    toCreateLeaderboardAccessToken() {
        return this.to('CreateLeaderboardAccessToken');
    }
    /**
     * Grants permission to submit a DeepRacer model to be evaluated for leaderboards
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    toCreateLeaderboardSubmission() {
        return this.to('CreateLeaderboardSubmission');
    }
    /**
     * Grants permission to create ra einforcement learning model for DeepRacer
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    toCreateReinforcementLearningModel() {
        return this.to('CreateReinforcementLearningModel');
    }
    /**
     * Grants permission to delete a leaderboard
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-manage-community-races.html
     */
    toDeleteLeaderboard() {
        return this.to('DeleteLeaderboard');
    }
    /**
     * Grants permission to delete a DeepRacer model
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    toDeleteModel() {
        return this.to('DeleteModel');
    }
    /**
     * Grants permission to edit a leaderboard
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-manage-community-races.html
     */
    toEditLeaderboard() {
        return this.to('EditLeaderboard');
    }
    /**
     * Grants permission to get current multiuser configuration for this account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-admin-set-account-config.html
     */
    toGetAccountConfig() {
        return this.to('GetAccountConfig');
    }
    /**
     * Grants permission to retrieve the user's alias for submitting a DeepRacer model to leaderboards
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    toGetAlias() {
        return this.to('GetAlias');
    }
    /**
     * Grants permission to download artifacts for an existing DeepRacer model
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html
     */
    toGetAssetUrl() {
        return this.to('GetAssetUrl');
    }
    /**
     * Grants permission to retrieve a specific DeepRacer car from your garage
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-choose-race-type.html
     */
    toGetCar() {
        return this.to('GetCar');
    }
    /**
     * Grants permission to view all the DeepRacer cars in your garage
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-choose-race-type.html
     */
    toGetCars() {
        return this.to('GetCars');
    }
    /**
     * Grants permission to retrieve information about an existing DeepRacer model's evaluation jobs
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html
     */
    toGetEvaluation() {
        return this.to('GetEvaluation');
    }
    /**
     * Grants permission to retrieve information about how the latest submitted DeepRacer model for a user performed on a leaderboard
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    toGetLatestUserSubmission() {
        return this.to('GetLatestUserSubmission');
    }
    /**
     * Grants permission to retrieve information about leaderboards
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    toGetLeaderboard() {
        return this.to('GetLeaderboard');
    }
    /**
     * Grants permission to retrieve information about an existing DeepRacer model
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    toGetModel() {
        return this.to('GetModel');
    }
    /**
     * Grants permission to retrieve information about private leaderboards
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-join-community-race.html
     */
    toGetPrivateLeaderboard() {
        return this.to('GetPrivateLeaderboard');
    }
    /**
     * Grants permission to retrieve information about the performance of a user's DeepRacer model that got placed on a leaderboard
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    toGetRankedUserSubmission() {
        return this.to('GetRankedUserSubmission');
    }
    /**
     * Grants permission to retrieve information about DeepRacer tracks
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html
     */
    toGetTrack() {
        return this.to('GetTrack');
    }
    /**
     * Grants permission to retrieve information about an existing DeepRacer model's training job
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    toGetTrainingJob() {
        return this.to('GetTrainingJob');
    }
    /**
     * Grants permission to import a reinforcement learning model for DeepRacer
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-troubleshooting-service-migration-errors.html
     */
    toImportModel() {
        return this.to('ImportModel');
    }
    /**
     * Grants permission to list a DeepRacer model's evaluation jobs
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html
     */
    toListEvaluations() {
        return this.to('ListEvaluations');
    }
    /**
     * Grants permission to list all the user's leaderboard evaluation jobs for a leaderboard
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    toListLeaderboardEvaluations() {
        return this.to('ListLeaderboardEvaluations');
    }
    /**
     * Grants permission to list all the DeepRacer model submissions of a user on a leaderboard
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    toListLeaderboardSubmissions() {
        return this.to('ListLeaderboardSubmissions');
    }
    /**
     * Grants permission to list all the available leaderboards
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    toListLeaderboards() {
        return this.to('ListLeaderboards');
    }
    /**
     * Grants permission to list all existing DeepRacer models
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    toListModels() {
        return this.to('ListModels');
    }
    /**
     * Grants permission to retrieve participant information about private leaderboards
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-join-community-race.html
     */
    toListPrivateLeaderboardParticipants() {
        return this.to('ListPrivateLeaderboardParticipants');
    }
    /**
     * Grants permission to list all the available private leaderboards
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-join-community-race.html
     */
    toListPrivateLeaderboards() {
        return this.to('ListPrivateLeaderboards');
    }
    /**
     * Grants permission to list all the subscribed private leaderboards
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-join-community-race.html
     */
    toListSubscribedPrivateLeaderboards() {
        return this.to('ListSubscribedPrivateLeaderboards');
    }
    /**
     * Grants permission to lists tag for a resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-tagging.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list all DeepRacer tracks
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html
     */
    toListTracks() {
        return this.to('ListTracks');
    }
    /**
     * Grants permission to list a DeepRacer model's training jobs
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    toListTrainingJobs() {
        return this.to('ListTrainingJobs');
    }
    /**
     * Grants permission to migrate previous reinforcement learning models for DeepRacer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-troubleshooting-service-migration-errors.html
     */
    toMigrateModels() {
        return this.to('MigrateModels');
    }
    /**
     * Grants permission to performs the leaderboard operation mentioned in the operation attribute
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-perform-leaderboard-operation.html
     */
    toPerformLeaderboardOperation() {
        return this.to('PerformLeaderboardOperation');
    }
    /**
     * Grants permission to remove access for a private leaderboard
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-manage-community-races.html
     */
    toRemoveLeaderboardAccessPermission() {
        return this.to('RemoveLeaderboardAccessPermission');
    }
    /**
     * Grants permission to set the user's alias for submitting a DeepRacer model to leaderboards
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     */
    toSetAlias() {
        return this.to('SetAlias');
    }
    /**
     * Grants permission to evaluate a DeepRacer model in a simulated environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html
     */
    toStartEvaluation() {
        return this.to('StartEvaluation');
    }
    /**
     * Grants permission to stop DeepRacer model evaluations
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html
     */
    toStopEvaluation() {
        return this.to('StopEvaluation');
    }
    /**
     * Grants permission to stop training a DeepRacer model
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     */
    toStopTrainingReinforcementLearningModel() {
        return this.to('StopTrainingReinforcementLearningModel');
    }
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-tagging.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to test reward functions for correctness
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html#deepracer-train-models-define-reward-function
     */
    toTestRewardFunction() {
        return this.to('TestRewardFunction');
    }
    /**
     * Grants permission to untag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-tagging.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a DeepRacer car in your garage
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserToken()
     * - .ifMultiUser()
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-choose-race-type.html
     */
    toUpdateCar() {
        return this.to('UpdateCar');
    }
    /**
     * Adds a resource of type car to the statement
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-choose-race-type.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCar(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:deepracer:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:car/${resourceId}`);
    }
    /**
     * Adds a resource of type evaluation_job to the statement
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-test-in-simulator.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEvaluationJob(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:deepracer:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:evaluation_job/${resourceId}`);
    }
    /**
     * Adds a resource of type leaderboard to the statement
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLeaderboard(resourceId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:deepracer:${region ?? this.defaultRegion}::leaderboard/${resourceId}`);
    }
    /**
     * Adds a resource of type leaderboard_evaluation_job to the statement
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-submit-model-to-leaderboard.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLeaderboardEvaluationJob(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:deepracer:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:leaderboard_evaluation_job/${resourceId}`);
    }
    /**
     * Adds a resource of type reinforcement_learning_model to the statement
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onReinforcementLearningModel(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:deepracer:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:model/reinforcement_learning/${resourceId}`);
    }
    /**
     * Adds a resource of type track to the statement
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-console-train-evaluate-models.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onTrack(resourceId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:deepracer:${region ?? this.defaultRegion}::track/${resourceId}`);
    }
    /**
     * Adds a resource of type training_job to the statement
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/deepracer-get-started-training-model.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTrainingJob(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:deepracer:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:training_job/${resourceId}`);
    }
    /**
     * Filters access by tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCloneReinforcementLearningModel()
     * - .toCreateCar()
     * - .toCreateLeaderboard()
     * - .toCreateLeaderboardSubmission()
     * - .toCreateReinforcementLearningModel()
     * - .toStartEvaluation()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toListTagsForResource()
     * - .toTagResource()
     *
     * Applies to resource types:
     * - car
     * - evaluation_job
     * - leaderboard
     * - leaderboard_evaluation_job
     * - reinforcement_learning_model
     * - training_job
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCloneReinforcementLearningModel()
     * - .toCreateCar()
     * - .toCreateLeaderboard()
     * - .toCreateLeaderboardSubmission()
     * - .toCreateReinforcementLearningModel()
     * - .toStartEvaluation()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by multiuser flag
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/reference_policies_iam-condition-keys.html#condition-keys-multiuser
     *
     * Applies to actions:
     * - .toAddLeaderboardAccessPermission()
     * - .toAdminDescribeAccountKey()
     * - .toAdminUpdateAccountKey()
     * - .toCloneReinforcementLearningModel()
     * - .toCreateCar()
     * - .toCreateLeaderboard()
     * - .toCreateLeaderboardAccessToken()
     * - .toCreateLeaderboardSubmission()
     * - .toCreateReinforcementLearningModel()
     * - .toDeleteLeaderboard()
     * - .toDeleteModel()
     * - .toEditLeaderboard()
     * - .toGetAccountConfig()
     * - .toGetAlias()
     * - .toGetAssetUrl()
     * - .toGetCar()
     * - .toGetCars()
     * - .toGetEvaluation()
     * - .toGetLatestUserSubmission()
     * - .toGetLeaderboard()
     * - .toGetModel()
     * - .toGetPrivateLeaderboard()
     * - .toGetRankedUserSubmission()
     * - .toGetTrainingJob()
     * - .toImportModel()
     * - .toListEvaluations()
     * - .toListLeaderboardEvaluations()
     * - .toListLeaderboardSubmissions()
     * - .toListLeaderboards()
     * - .toListModels()
     * - .toListPrivateLeaderboardParticipants()
     * - .toListPrivateLeaderboards()
     * - .toListSubscribedPrivateLeaderboards()
     * - .toListTagsForResource()
     * - .toListTrainingJobs()
     * - .toPerformLeaderboardOperation()
     * - .toRemoveLeaderboardAccessPermission()
     * - .toSetAlias()
     * - .toStartEvaluation()
     * - .toStopEvaluation()
     * - .toStopTrainingReinforcementLearningModel()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateCar()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifMultiUser(value) {
        return this.if(`MultiUser`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by user token in the request
     *
     * https://docs.aws.amazon.com/deepracer/latest/developerguide/reference_policies_iam-condition-keys.html#condition-keys-usertoken
     *
     * Applies to actions:
     * - .toAddLeaderboardAccessPermission()
     * - .toAdminDescribeAccountKey()
     * - .toAdminUpdateAccountKey()
     * - .toCloneReinforcementLearningModel()
     * - .toCreateCar()
     * - .toCreateLeaderboard()
     * - .toCreateLeaderboardAccessToken()
     * - .toCreateLeaderboardSubmission()
     * - .toCreateReinforcementLearningModel()
     * - .toDeleteLeaderboard()
     * - .toDeleteModel()
     * - .toEditLeaderboard()
     * - .toGetAccountConfig()
     * - .toGetAlias()
     * - .toGetAssetUrl()
     * - .toGetCar()
     * - .toGetCars()
     * - .toGetEvaluation()
     * - .toGetLatestUserSubmission()
     * - .toGetLeaderboard()
     * - .toGetModel()
     * - .toGetPrivateLeaderboard()
     * - .toGetRankedUserSubmission()
     * - .toGetTrainingJob()
     * - .toImportModel()
     * - .toListEvaluations()
     * - .toListLeaderboardEvaluations()
     * - .toListLeaderboardSubmissions()
     * - .toListLeaderboards()
     * - .toListModels()
     * - .toListPrivateLeaderboardParticipants()
     * - .toListPrivateLeaderboards()
     * - .toListSubscribedPrivateLeaderboards()
     * - .toListTagsForResource()
     * - .toListTrainingJobs()
     * - .toPerformLeaderboardOperation()
     * - .toRemoveLeaderboardAccessPermission()
     * - .toSetAlias()
     * - .toStartEvaluation()
     * - .toStopEvaluation()
     * - .toStopTrainingReinforcementLearningModel()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateCar()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUserToken(value, operator) {
        return this.if(`UserToken`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [deepracer](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdeepracer.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'deepracer';
        this.accessLevelList = {
            Write: [
                'AddLeaderboardAccessPermission',
                'AdminManageUser',
                'AdminSetAccountConfig',
                'AdminUpdateAccountKey',
                'CloneReinforcementLearningModel',
                'CreateCar',
                'CreateLeaderboard',
                'CreateLeaderboardAccessToken',
                'CreateLeaderboardSubmission',
                'CreateReinforcementLearningModel',
                'DeleteLeaderboard',
                'DeleteModel',
                'EditLeaderboard',
                'ImportModel',
                'MigrateModels',
                'PerformLeaderboardOperation',
                'RemoveLeaderboardAccessPermission',
                'SetAlias',
                'StartEvaluation',
                'StopEvaluation',
                'StopTrainingReinforcementLearningModel',
                'TestRewardFunction',
                'UpdateCar'
            ],
            Read: [
                'AdminDescribeAccountKey',
                'AdminGetAccountConfig',
                'AdminListAssociatedResources',
                'AdminListAssociatedUsers',
                'GetAccountConfig',
                'GetAlias',
                'GetAssetUrl',
                'GetCar',
                'GetCars',
                'GetEvaluation',
                'GetLatestUserSubmission',
                'GetLeaderboard',
                'GetModel',
                'GetPrivateLeaderboard',
                'GetRankedUserSubmission',
                'GetTrack',
                'GetTrainingJob',
                'ListEvaluations',
                'ListLeaderboardEvaluations',
                'ListLeaderboardSubmissions',
                'ListLeaderboards',
                'ListModels',
                'ListPrivateLeaderboardParticipants',
                'ListPrivateLeaderboards',
                'ListSubscribedPrivateLeaderboards',
                'ListTagsForResource',
                'ListTracks',
                'ListTrainingJobs'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Deepracer = Deepracer;
//# sourceMappingURL=data:application/json;base64,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