import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [ts](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdiagnostictools.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class Ts extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to get details about specific execution within AWS Diagnostic tools
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/diagnostic-tools/latest/APIReference/API_GetExecution.html
     */
    toGetExecution(): this;
    /**
     * Grants permission to get details about specific execution output within AWS Diagnostic tools
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/diagnostic-tools/latest/APIReference/API_GetExecutionOutput.html
     */
    toGetExecutionOutput(): this;
    /**
     * Grants permission to get details about specific tool within AWS Diagnostic tools
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/diagnostic-tools/latest/APIReference/API_GetTool.html
     */
    toGetTool(): this;
    /**
     * Grants permission to list all available execution within AWS Diagnostic tools
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/diagnostic-tools/latest/APIReference/API_ListExecutions.html
     */
    toListExecutions(): this;
    /**
     * Grants permission to list the tags for an AWS Diagnostic tools resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/diagnostic-tools/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to list all available tools within AWS Diagnostic tools
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/diagnostic-tools/latest/APIReference/API_ListTools.html
     */
    toListTools(): this;
    /**
     * Grants permission to start an execution workflow of specific tool within AWS Diagnostic tools
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/diagnostic-tools/latest/APIReference/API_StartExecution.html
     */
    toStartExecution(): this;
    /**
     * Grants permission to tag an AWS Diagnostic tools resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/diagnostic-tools/latest/APIReference/API_TagResource.html
     */
    toTagResource(): this;
    /**
     * Grants permission to untag an AWS Diagnostic tools resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/diagnostic-tools/latest/APIReference/API_UntagResource.html
     */
    toUntagResource(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type execution to the statement
     *
     * https://docs.aws.amazon.com/diagnostic-tools/latest/APIReference/API_Execution.html
     *
     * @param userId - Identifier for the userId.
     * @param toolId - Identifier for the toolId.
     * @param executionId - Identifier for the executionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onExecution(userId: string, toolId: string, executionId: string, account?: string, partition?: string): this;
    /**
     * Adds a resource of type tool to the statement
     *
     * https://docs.aws.amazon.com/diagnostic-tools/latest/APIReference/API_Tool.html
     *
     * @param toolId - Identifier for the toolId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onTool(toolId: string, partition?: string): this;
    /**
     * Filters access by the allowed set of values for each of the tags
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toListTagsForResource()
     * - .toStartExecution()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by tag-value associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - execution
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the presence of mandatory tags in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toListTagsForResource()
     * - .toStartExecution()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [ts](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdiagnostictools.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
