"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcrPublic = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [ecr-public](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelasticcontainerregistrypublic.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class EcrPublic extends shared_1.PolicyStatement {
    /**
     * Grants permission to check the availability of multiple image layers in a specified registry and repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECRPublic/latest/APIReference/API_BatchCheckLayerAvailability.html
     */
    toBatchCheckLayerAvailability() {
        return this.to('BatchCheckLayerAvailability');
    }
    /**
     * Grants permission to delete a list of specified images within a specified repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECRPublic/latest/APIReference/API_BatchDeleteImage.html
     */
    toBatchDeleteImage() {
        return this.to('BatchDeleteImage');
    }
    /**
     * Grants permission to inform Amazon ECR that the image layer upload for a specified registry, repository name, and upload ID, has completed
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECRPublic/latest/APIReference/API_CompleteLayerUpload.html
     */
    toCompleteLayerUpload() {
        return this.to('CompleteLayerUpload');
    }
    /**
     * Grants permission to create an image repository
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ecr-public:TagResource
     *
     * https://docs.aws.amazon.com/AmazonECRPublic/latest/APIReference/API_CreateRepository.html
     */
    toCreateRepository() {
        return this.to('CreateRepository');
    }
    /**
     * Grants permission to delete an existing image repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECRPublic/latest/APIReference/API_DeleteRepository.html
     */
    toDeleteRepository() {
        return this.to('DeleteRepository');
    }
    /**
     * Grants permission to delete the repository policy from a specified repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECRPublic/latest/APIReference/API_DeleteRepositoryPolicy.html
     */
    toDeleteRepositoryPolicy() {
        return this.to('DeleteRepositoryPolicy');
    }
    /**
     * Grants permission to describe all the image tags for a given repository
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonECRPublic/latest/APIReference/API_DescribeImageTags.html
     */
    toDescribeImageTags() {
        return this.to('DescribeImageTags');
    }
    /**
     * Grants permission to get metadata about the images in a repository, including image size, image tags, and creation date
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECRPublic/latest/APIReference/API_DescribeImages.html
     */
    toDescribeImages() {
        return this.to('DescribeImages');
    }
    /**
     * Grants permission to retrieve the catalog data associated with a registry
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonECRPublic/latest/APIReference/API_DescribeRegistries.html
     */
    toDescribeRegistries() {
        return this.to('DescribeRegistries');
    }
    /**
     * Grants permission to describe image repositories in a registry
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonECRPublic/latest/APIReference/API_DescribeRepositories.html
     */
    toDescribeRepositories() {
        return this.to('DescribeRepositories');
    }
    /**
     * Grants permission to retrieve a token that is valid for a specified registry for 12 hours
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECRPublic/latest/APIReference/API_GetAuthorizationToken.html
     */
    toGetAuthorizationToken() {
        return this.to('GetAuthorizationToken');
    }
    /**
     * Grants permission to retrieve the catalog data associated with a registry
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECRPublic/latest/APIReference/API_GetRegistryCatalogData.html
     */
    toGetRegistryCatalogData() {
        return this.to('GetRegistryCatalogData');
    }
    /**
     * Grants permission to retrieve the catalog data associated with a repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECRPublic/latest/APIReference/API_GetRepositoryCatalogData.html
     */
    toGetRepositoryCatalogData() {
        return this.to('GetRepositoryCatalogData');
    }
    /**
     * Grants permission to retrieve the repository policy for a specified repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECRPublic/latest/APIReference/API_GetRepositoryPolicy.html
     */
    toGetRepositoryPolicy() {
        return this.to('GetRepositoryPolicy');
    }
    /**
     * Grants permission to notify Amazon ECR that you intend to upload an image layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECRPublic/latest/APIReference/API_InitiateLayerUpload.html
     */
    toInitiateLayerUpload() {
        return this.to('InitiateLayerUpload');
    }
    /**
     * Grants permission to list the tags for an Amazon ECR resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECRPublic/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to create or update the image manifest associated with an image
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECRPublic/latest/APIReference/API_PutImage.html
     */
    toPutImage() {
        return this.to('PutImage');
    }
    /**
     * Grants permission to create and update the catalog data associated with a registry
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECRPublic/latest/APIReference/API_PutRegistryCatalogData.html
     */
    toPutRegistryCatalogData() {
        return this.to('PutRegistryCatalogData');
    }
    /**
     * Grants permission to update the catalog data associated with a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECRPublic/latest/APIReference/API_PutRepositoryCatalogData.html
     */
    toPutRepositoryCatalogData() {
        return this.to('PutRepositoryCatalogData');
    }
    /**
     * Grants permission to apply a repository policy on a specified repository to control access permissions
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonECRPublic/latest/APIReference/API_SetRepositoryPolicy.html
     */
    toSetRepositoryPolicy() {
        return this.to('SetRepositoryPolicy');
    }
    /**
     * Grants permission to tag an Amazon ECR resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonECRPublic/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag an Amazon ECR resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonECRPublic/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to upload an image layer part to Amazon ECR Public
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECRPublic/latest/APIReference/API_UploadLayerPart.html
     */
    toUploadLayerPart() {
        return this.to('UploadLayerPart');
    }
    /**
     * Adds a resource of type repository to the statement
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/userguide/iam-policy-structure.html#ECR-Public_ARN_Format
     *
     * @param repositoryName - Identifier for the repositoryName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onRepository(repositoryName, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ecr-public::${account ?? this.defaultAccount}:repository/${repositoryName}`);
    }
    /**
     * Adds a resource of type registry to the statement
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/userguide/iam-policy-structure.html#ECR-Public_ARN_Format
     *
     * @param registryId - Identifier for the registryId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onRegistry(registryId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ecr-public::${account ?? this.defaultAccount}:registry/${registryId}`);
    }
    /**
     * Filters create requests based on the allowed set of values for each of the tags
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateRepository()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on tag-value associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - repository
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters create requests based on the presence of mandatory tags in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateRepository()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on tag-value associated with the resource
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to resource types:
     * - repository
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagKey, value, operator) {
        return this.if(`ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [ecr-public](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelasticcontainerregistrypublic.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'ecr-public';
        this.accessLevelList = {
            Read: [
                'BatchCheckLayerAvailability',
                'DescribeImages',
                'GetAuthorizationToken',
                'GetRegistryCatalogData',
                'GetRepositoryCatalogData',
                'GetRepositoryPolicy',
                'ListTagsForResource'
            ],
            Write: [
                'BatchDeleteImage',
                'CompleteLayerUpload',
                'CreateRepository',
                'DeleteRepository',
                'DeleteRepositoryPolicy',
                'InitiateLayerUpload',
                'PutImage',
                'PutRegistryCatalogData',
                'PutRepositoryCatalogData',
                'UploadLayerPart'
            ],
            List: [
                'DescribeImageTags',
                'DescribeRegistries',
                'DescribeRepositories'
            ],
            'Permissions management': [
                'SetRepositoryPolicy'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.EcrPublic = EcrPublic;
//# sourceMappingURL=data:application/json;base64,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