"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Drs = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [drs](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselasticdisasterrecovery.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Drs extends shared_1.PolicyStatement {
    /**
     * Grants permission to get associate failback client to recovery instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toAssociateFailbackClientToRecoveryInstanceForDrs() {
        return this.to('AssociateFailbackClientToRecoveryInstanceForDrs');
    }
    /**
     * Grants permission to associate CloudFormation stack with source network
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - cloudformation:DescribeStackResource
     * - cloudformation:DescribeStacks
     * - drs:GetLaunchConfiguration
     * - ec2:CreateLaunchTemplateVersion
     * - ec2:DescribeLaunchTemplateVersions
     * - ec2:DescribeLaunchTemplates
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - ec2:ModifyLaunchTemplate
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_AssociateSourceNetworkStack.html
     */
    toAssociateSourceNetworkStack() {
        return this.to('AssociateSourceNetworkStack');
    }
    /**
     * Grants permission to batch create volume snapshot group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toBatchCreateVolumeSnapshotGroupForDrs() {
        return this.to('BatchCreateVolumeSnapshotGroupForDrs');
    }
    /**
     * Grants permission to batch delete snapshot request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toBatchDeleteSnapshotRequestForDrs() {
        return this.to('BatchDeleteSnapshotRequestForDrs');
    }
    /**
     * Grants permission to create converted snapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toCreateConvertedSnapshotForDrs() {
        return this.to('CreateConvertedSnapshotForDrs');
    }
    /**
     * Grants permission to extend a source server
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - drs:DescribeSourceServers
     * - drs:GetReplicationConfiguration
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_CreateExtendedSourceServer.html
     */
    toCreateExtendedSourceServer() {
        return this.to('CreateExtendedSourceServer');
    }
    /**
     * Grants permission to create launch configuration template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_CreateLaunchConfigurationTemplate.html
     */
    toCreateLaunchConfigurationTemplate() {
        return this.to('CreateLaunchConfigurationTemplate');
    }
    /**
     * Grants permission to create recovery instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toCreateRecoveryInstanceForDrs() {
        return this.to('CreateRecoveryInstanceForDrs');
    }
    /**
     * Grants permission to create replication configuration template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:CreateSecurityGroup
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:GetEbsDefaultKmsKeyId
     * - ec2:GetEbsEncryptionByDefault
     * - kms:CreateGrant
     * - kms:DescribeKey
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_CreateReplicationConfigurationTemplate.html
     */
    toCreateReplicationConfigurationTemplate() {
        return this.to('CreateReplicationConfigurationTemplate');
    }
    /**
     * Grants permission to create a source network
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:DescribeInstances
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_CreateSourceNetwork.html
     */
    toCreateSourceNetwork() {
        return this.to('CreateSourceNetwork');
    }
    /**
     * Grants permission to create a source server
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toCreateSourceServerForDrs() {
        return this.to('CreateSourceServerForDrs');
    }
    /**
     * Grants permission to delete a job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_DeleteJob.html
     */
    toDeleteJob() {
        return this.to('DeleteJob');
    }
    /**
     * Grants permission to delete a launch action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_DeleteLaunchAction.html
     */
    toDeleteLaunchAction() {
        return this.to('DeleteLaunchAction');
    }
    /**
     * Grants permission to delete launch configuration template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_DeleteLaunchConfigurationTemplate.html
     */
    toDeleteLaunchConfigurationTemplate() {
        return this.to('DeleteLaunchConfigurationTemplate');
    }
    /**
     * Grants permission to delete recovery instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_DeleteRecoveryInstance.html
     */
    toDeleteRecoveryInstance() {
        return this.to('DeleteRecoveryInstance');
    }
    /**
     * Grants permission to delete replication configuration template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_DeleteReplicationConfigurationTemplate.html
     */
    toDeleteReplicationConfigurationTemplate() {
        return this.to('DeleteReplicationConfigurationTemplate');
    }
    /**
     * Grants permission to delete source network
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_DeleteSourceNetwork.html
     */
    toDeleteSourceNetwork() {
        return this.to('DeleteSourceNetwork');
    }
    /**
     * Grants permission to delete source server
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_DeleteSourceServer.html
     */
    toDeleteSourceServer() {
        return this.to('DeleteSourceServer');
    }
    /**
     * Grants permission to describe job log items
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_DescribeJobLogItems.html
     */
    toDescribeJobLogItems() {
        return this.to('DescribeJobLogItems');
    }
    /**
     * Grants permission to describe jobs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_DescribeJobs.html
     */
    toDescribeJobs() {
        return this.to('DescribeJobs');
    }
    /**
     * Grants permission to describe launch configuration template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_DescribeLaunchConfigurationTemplates.html
     */
    toDescribeLaunchConfigurationTemplates() {
        return this.to('DescribeLaunchConfigurationTemplates');
    }
    /**
     * Grants permission to describe recovery instances
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - drs:DescribeSourceServers
     * - ec2:DescribeInstances
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_DescribeRecoveryInstances.html
     */
    toDescribeRecoveryInstances() {
        return this.to('DescribeRecoveryInstances');
    }
    /**
     * Grants permission to describe recovery snapshots
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_DescribeRecoverySnapshots.html
     */
    toDescribeRecoverySnapshots() {
        return this.to('DescribeRecoverySnapshots');
    }
    /**
     * Grants permission to describe replication configuration template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_DescribeReplicationConfigurationTemplates.html
     */
    toDescribeReplicationConfigurationTemplates() {
        return this.to('DescribeReplicationConfigurationTemplates');
    }
    /**
     * Grants permission to describe replication server associations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toDescribeReplicationServerAssociationsForDrs() {
        return this.to('DescribeReplicationServerAssociationsForDrs');
    }
    /**
     * Grants permission to describe snapshot requests
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toDescribeSnapshotRequestsForDrs() {
        return this.to('DescribeSnapshotRequestsForDrs');
    }
    /**
     * Grants permission to describe source networks
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_DescribeSourceNetworks.html
     */
    toDescribeSourceNetworks() {
        return this.to('DescribeSourceNetworks');
    }
    /**
     * Grants permission to describe source servers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_DescribeSourceServers.html
     */
    toDescribeSourceServers() {
        return this.to('DescribeSourceServers');
    }
    /**
     * Grants permission to disconnect recovery instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_DisconnectRecoveryInstance.html
     */
    toDisconnectRecoveryInstance() {
        return this.to('DisconnectRecoveryInstance');
    }
    /**
     * Grants permission to disconnect source server
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_DisconnectSourceServer.html
     */
    toDisconnectSourceServer() {
        return this.to('DisconnectSourceServer');
    }
    /**
     * Grants permission to export CloudFormation template which contains source network resources
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - s3:GetBucketLocation
     * - s3:GetObject
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_ExportSourceNetworkCfnTemplate.html
     */
    toExportSourceNetworkCfnTemplate() {
        return this.to('ExportSourceNetworkCfnTemplate');
    }
    /**
     * Grants permission to get agent command
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toGetAgentCommandForDrs() {
        return this.to('GetAgentCommandForDrs');
    }
    /**
     * Grants permission to get agent confirmed resume info
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toGetAgentConfirmedResumeInfoForDrs() {
        return this.to('GetAgentConfirmedResumeInfoForDrs');
    }
    /**
     * Grants permission to get agent installation assets
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toGetAgentInstallationAssetsForDrs() {
        return this.to('GetAgentInstallationAssetsForDrs');
    }
    /**
     * Grants permission to get agent replication info
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toGetAgentReplicationInfoForDrs() {
        return this.to('GetAgentReplicationInfoForDrs');
    }
    /**
     * Grants permission to get agent runtime configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toGetAgentRuntimeConfigurationForDrs() {
        return this.to('GetAgentRuntimeConfigurationForDrs');
    }
    /**
     * Grants permission to get agent snapshot credits
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toGetAgentSnapshotCreditsForDrs() {
        return this.to('GetAgentSnapshotCreditsForDrs');
    }
    /**
     * Grants permission to get channel commands
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toGetChannelCommandsForDrs() {
        return this.to('GetChannelCommandsForDrs');
    }
    /**
     * Grants permission to get failback command
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toGetFailbackCommandForDrs() {
        return this.to('GetFailbackCommandForDrs');
    }
    /**
     * Grants permission to get failback launch requested
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toGetFailbackLaunchRequestedForDrs() {
        return this.to('GetFailbackLaunchRequestedForDrs');
    }
    /**
     * Grants permission to get failback replication configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_GetFailbackReplicationConfiguration.html
     */
    toGetFailbackReplicationConfiguration() {
        return this.to('GetFailbackReplicationConfiguration');
    }
    /**
     * Grants permission to get launch configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_GetLaunchConfiguration.html
     */
    toGetLaunchConfiguration() {
        return this.to('GetLaunchConfiguration');
    }
    /**
     * Grants permission to get replication configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_GetReplicationConfiguration.html
     */
    toGetReplicationConfiguration() {
        return this.to('GetReplicationConfiguration');
    }
    /**
     * Grants permission to get suggested failback client device mapping
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toGetSuggestedFailbackClientDeviceMappingForDrs() {
        return this.to('GetSuggestedFailbackClientDeviceMappingForDrs');
    }
    /**
     * Grants permission to initialize service
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:AddRoleToInstanceProfile
     * - iam:CreateInstanceProfile
     * - iam:CreateServiceLinkedRole
     * - iam:GetInstanceProfile
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_InitializeService.html
     */
    toInitializeService() {
        return this.to('InitializeService');
    }
    /**
     * Grants permission to issue an agent certificate
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toIssueAgentCertificateForDrs() {
        return this.to('IssueAgentCertificateForDrs');
    }
    /**
     * Grants permission to list extensible source servers
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - drs:DescribeSourceServers
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_ListExtensibleSourceServers.html
     */
    toListExtensibleSourceServers() {
        return this.to('ListExtensibleSourceServers');
    }
    /**
     * Grants permission to list launch actions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_ListLaunchActions.html
     */
    toListLaunchActions() {
        return this.to('ListLaunchActions');
    }
    /**
     * Grants permission to list staging accounts
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_ListStagingAccounts.html
     */
    toListStagingAccounts() {
        return this.to('ListStagingAccounts');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to notify agent authentication
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toNotifyAgentAuthenticationForDrs() {
        return this.to('NotifyAgentAuthenticationForDrs');
    }
    /**
     * Grants permission to notify agent is connected
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toNotifyAgentConnectedForDrs() {
        return this.to('NotifyAgentConnectedForDrs');
    }
    /**
     * Grants permission to notify agent is disconnected
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toNotifyAgentDisconnectedForDrs() {
        return this.to('NotifyAgentDisconnectedForDrs');
    }
    /**
     * Grants permission to notify agent replication progress
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toNotifyAgentReplicationProgressForDrs() {
        return this.to('NotifyAgentReplicationProgressForDrs');
    }
    /**
     * Grants permission to notify consistency attained
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toNotifyConsistencyAttainedForDrs() {
        return this.to('NotifyConsistencyAttainedForDrs');
    }
    /**
     * Grants permission to notify replication server authentication
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toNotifyReplicationServerAuthenticationForDrs() {
        return this.to('NotifyReplicationServerAuthenticationForDrs');
    }
    /**
     * Grants permission to notify replicator volume events
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toNotifyVolumeEventForDrs() {
        return this.to('NotifyVolumeEventForDrs');
    }
    /**
     * Grants permission to put a launch action
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ssm:DescribeDocument
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_PutLaunchAction.html
     */
    toPutLaunchAction() {
        return this.to('PutLaunchAction');
    }
    /**
     * Grants permission to retry data replication
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_RetryDataReplication.html
     */
    toRetryDataReplication() {
        return this.to('RetryDataReplication');
    }
    /**
     * Grants permission to reverse replication
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - drs:DescribeReplicationConfigurationTemplates
     * - drs:DescribeSourceServers
     * - ec2:DescribeInstances
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_ReverseReplication.html
     */
    toReverseReplication() {
        return this.to('ReverseReplication');
    }
    /**
     * Grants permission to send agent logs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toSendAgentLogsForDrs() {
        return this.to('SendAgentLogsForDrs');
    }
    /**
     * Grants permission to send agent metrics
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toSendAgentMetricsForDrs() {
        return this.to('SendAgentMetricsForDrs');
    }
    /**
     * Grants permission to send channel command result
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toSendChannelCommandResultForDrs() {
        return this.to('SendChannelCommandResultForDrs');
    }
    /**
     * Grants permission to send client logs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toSendClientLogsForDrs() {
        return this.to('SendClientLogsForDrs');
    }
    /**
     * Grants permission to send client metrics
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toSendClientMetricsForDrs() {
        return this.to('SendClientMetricsForDrs');
    }
    /**
     * Grants permission to send volume throughput statistics
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toSendVolumeStatsForDrs() {
        return this.to('SendVolumeStatsForDrs');
    }
    /**
     * Grants permission to start failback launch
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_StartFailbackLaunch.html
     */
    toStartFailbackLaunch() {
        return this.to('StartFailbackLaunch');
    }
    /**
     * Grants permission to start recovery
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - drs:CreateRecoveryInstanceForDrs
     * - drs:ListTagsForResource
     * - ec2:AttachVolume
     * - ec2:AuthorizeSecurityGroupEgress
     * - ec2:AuthorizeSecurityGroupIngress
     * - ec2:CreateLaunchTemplate
     * - ec2:CreateLaunchTemplateVersion
     * - ec2:CreateSnapshot
     * - ec2:CreateTags
     * - ec2:CreateVolume
     * - ec2:DeleteLaunchTemplateVersions
     * - ec2:DeleteSnapshot
     * - ec2:DeleteVolume
     * - ec2:DescribeAccountAttributes
     * - ec2:DescribeAvailabilityZones
     * - ec2:DescribeImages
     * - ec2:DescribeInstanceAttribute
     * - ec2:DescribeInstanceStatus
     * - ec2:DescribeInstanceTypes
     * - ec2:DescribeInstances
     * - ec2:DescribeLaunchTemplateVersions
     * - ec2:DescribeLaunchTemplates
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSnapshots
     * - ec2:DescribeSubnets
     * - ec2:DescribeVolumes
     * - ec2:DetachVolume
     * - ec2:ModifyInstanceAttribute
     * - ec2:ModifyLaunchTemplate
     * - ec2:RevokeSecurityGroupEgress
     * - ec2:RunInstances
     * - ec2:StartInstances
     * - ec2:StopInstances
     * - ec2:TerminateInstances
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_StartRecovery.html
     */
    toStartRecovery() {
        return this.to('StartRecovery');
    }
    /**
     * Grants permission to start replication
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_StartReplication.html
     */
    toStartReplication() {
        return this.to('StartReplication');
    }
    /**
     * Grants permission to start network recovery
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - cloudformation:CreateStack
     * - cloudformation:DescribeStackResource
     * - cloudformation:DescribeStacks
     * - cloudformation:UpdateStack
     * - drs:GetLaunchConfiguration
     * - ec2:CreateLaunchTemplateVersion
     * - ec2:DescribeLaunchTemplateVersions
     * - ec2:DescribeLaunchTemplates
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - ec2:ModifyLaunchTemplate
     * - s3:GetObject
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_StartSourceNetworkRecovery.html
     */
    toStartSourceNetworkRecovery() {
        return this.to('StartSourceNetworkRecovery');
    }
    /**
     * Grants permission to start network replication
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_StartSourceNetworkReplication.html
     */
    toStartSourceNetworkReplication() {
        return this.to('StartSourceNetworkReplication');
    }
    /**
     * Grants permission to stop failback
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_StopFailback.html
     */
    toStopFailback() {
        return this.to('StopFailback');
    }
    /**
     * Grants permission to stop replication
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_StopReplication.html
     */
    toStopReplication() {
        return this.to('StopReplication');
    }
    /**
     * Grants permission to stop network replication
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_StopSourceNetworkReplication.html
     */
    toStopSourceNetworkReplication() {
        return this.to('StopSourceNetworkReplication');
    }
    /**
     * Grants permission to assign a resource tag
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifCreateAction()
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to terminate recovery instances
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - drs:DescribeSourceServers
     * - ec2:DeleteVolume
     * - ec2:DescribeInstances
     * - ec2:DescribeVolumes
     * - ec2:TerminateInstances
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_TerminateRecoveryInstances.html
     */
    toTerminateRecoveryInstances() {
        return this.to('TerminateRecoveryInstances');
    }
    /**
     * Grants permission to untag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update agent backlog
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toUpdateAgentBacklogForDrs() {
        return this.to('UpdateAgentBacklogForDrs');
    }
    /**
     * Grants permission to update agent conversion info
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toUpdateAgentConversionInfoForDrs() {
        return this.to('UpdateAgentConversionInfoForDrs');
    }
    /**
     * Grants permission to update agent replication info
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toUpdateAgentReplicationInfoForDrs() {
        return this.to('UpdateAgentReplicationInfoForDrs');
    }
    /**
     * Grants permission to update agent replication process state
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toUpdateAgentReplicationProcessStateForDrs() {
        return this.to('UpdateAgentReplicationProcessStateForDrs');
    }
    /**
     * Grants permission to update agent source properties
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toUpdateAgentSourcePropertiesForDrs() {
        return this.to('UpdateAgentSourcePropertiesForDrs');
    }
    /**
     * Grants permission to update failback client device mapping
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toUpdateFailbackClientDeviceMappingForDrs() {
        return this.to('UpdateFailbackClientDeviceMappingForDrs');
    }
    /**
     * Grants permission to update failback client last seen
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toUpdateFailbackClientLastSeenForDrs() {
        return this.to('UpdateFailbackClientLastSeenForDrs');
    }
    /**
     * Grants permission to update failback replication configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_UpdateFailbackReplicationConfiguration.html
     */
    toUpdateFailbackReplicationConfiguration() {
        return this.to('UpdateFailbackReplicationConfiguration');
    }
    /**
     * Grants permission to update launch configuration
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DescribeInstances
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_UpdateLaunchConfiguration.html
     */
    toUpdateLaunchConfiguration() {
        return this.to('UpdateLaunchConfiguration');
    }
    /**
     * Grants permission to update launch configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_UpdateLaunchConfigurationTemplate.html
     */
    toUpdateLaunchConfigurationTemplate() {
        return this.to('UpdateLaunchConfigurationTemplate');
    }
    /**
     * Grants permission to update a replication certificate
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/drs-apis.html
     */
    toUpdateReplicationCertificateForDrs() {
        return this.to('UpdateReplicationCertificateForDrs');
    }
    /**
     * Grants permission to update replication configuration
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateSecurityGroup
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:GetEbsDefaultKmsKeyId
     * - ec2:GetEbsEncryptionByDefault
     * - kms:CreateGrant
     * - kms:DescribeKey
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_UpdateReplicationConfiguration.html
     */
    toUpdateReplicationConfiguration() {
        return this.to('UpdateReplicationConfiguration');
    }
    /**
     * Grants permission to update replication configuration template
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateSecurityGroup
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:GetEbsDefaultKmsKeyId
     * - ec2:GetEbsEncryptionByDefault
     * - kms:CreateGrant
     * - kms:DescribeKey
     *
     * https://docs.aws.amazon.com/drs/latest/APIReference/API_UpdateReplicationConfigurationTemplate.html
     */
    toUpdateReplicationConfigurationTemplate() {
        return this.to('UpdateReplicationConfigurationTemplate');
    }
    /**
     * Adds a resource of type JobResource to the statement
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/failback-overview.html
     *
     * @param jobID - Identifier for the jobID.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onJobResource(jobID, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:drs:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:job/${jobID}`);
    }
    /**
     * Adds a resource of type RecoveryInstanceResource to the statement
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/recovery-instances.html
     *
     * @param recoveryInstanceID - Identifier for the recoveryInstanceID.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifEC2InstanceARN()
     */
    onRecoveryInstanceResource(recoveryInstanceID, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:drs:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:recovery-instance/${recoveryInstanceID}`);
    }
    /**
     * Adds a resource of type ReplicationConfigurationTemplateResource to the statement
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/replication-settings-template.html
     *
     * @param replicationConfigurationTemplateID - Identifier for the replicationConfigurationTemplateID.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onReplicationConfigurationTemplateResource(replicationConfigurationTemplateID, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:drs:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:replication-configuration-template/${replicationConfigurationTemplateID}`);
    }
    /**
     * Adds a resource of type LaunchConfigurationTemplateResource to the statement
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/default-drs-launch-settings.html
     *
     * @param launchConfigurationTemplateID - Identifier for the launchConfigurationTemplateID.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLaunchConfigurationTemplateResource(launchConfigurationTemplateID, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:drs:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:launch-configuration-template/${launchConfigurationTemplateID}`);
    }
    /**
     * Adds a resource of type SourceServerResource to the statement
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/source-servers.html
     *
     * @param sourceServerID - Identifier for the sourceServerID.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSourceServerResource(sourceServerID, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:drs:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:source-server/${sourceServerID}`);
    }
    /**
     * Adds a resource of type SourceNetworkResource to the statement
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/source-networks.html
     *
     * @param sourceNetworkID - Identifier for the sourceNetworkID.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSourceNetworkResource(sourceNetworkID, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:drs:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:source-network/${sourceNetworkID}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toAssociateSourceNetworkStack()
     * - .toCreateConvertedSnapshotForDrs()
     * - .toCreateExtendedSourceServer()
     * - .toCreateLaunchConfigurationTemplate()
     * - .toCreateRecoveryInstanceForDrs()
     * - .toCreateReplicationConfigurationTemplate()
     * - .toCreateSourceNetwork()
     * - .toCreateSourceServerForDrs()
     * - .toExportSourceNetworkCfnTemplate()
     * - .toReverseReplication()
     * - .toStartFailbackLaunch()
     * - .toStartRecovery()
     * - .toStartSourceNetworkRecovery()
     * - .toTagResource()
     * - .toTerminateRecoveryInstances()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - JobResource
     * - RecoveryInstanceResource
     * - ReplicationConfigurationTemplateResource
     * - LaunchConfigurationTemplateResource
     * - SourceServerResource
     * - SourceNetworkResource
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toAssociateSourceNetworkStack()
     * - .toCreateConvertedSnapshotForDrs()
     * - .toCreateExtendedSourceServer()
     * - .toCreateLaunchConfigurationTemplate()
     * - .toCreateRecoveryInstanceForDrs()
     * - .toCreateReplicationConfigurationTemplate()
     * - .toCreateSourceNetwork()
     * - .toCreateSourceServerForDrs()
     * - .toExportSourceNetworkCfnTemplate()
     * - .toReverseReplication()
     * - .toStartFailbackLaunch()
     * - .toStartRecovery()
     * - .toStartSourceNetworkRecovery()
     * - .toTagResource()
     * - .toTerminateRecoveryInstances()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the name of a resource-creating API action
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/supported-iam-actions-tagging.html
     *
     * Applies to actions:
     * - .toTagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCreateAction(value, operator) {
        return this.if(`CreateAction`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the EC2 instance the request originated from
     *
     * https://docs.aws.amazon.com/drs/latest/userguide/security_iam_authentication.html
     *
     * Applies to resource types:
     * - RecoveryInstanceResource
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifEC2InstanceARN(value, operator) {
        return this.if(`EC2InstanceARN`, value, operator ?? 'ArnLike');
    }
    /**
     * Statement provider for service [drs](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselasticdisasterrecovery.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'drs';
        this.accessLevelList = {
            Write: [
                'AssociateFailbackClientToRecoveryInstanceForDrs',
                'AssociateSourceNetworkStack',
                'BatchCreateVolumeSnapshotGroupForDrs',
                'BatchDeleteSnapshotRequestForDrs',
                'CreateConvertedSnapshotForDrs',
                'CreateExtendedSourceServer',
                'CreateLaunchConfigurationTemplate',
                'CreateRecoveryInstanceForDrs',
                'CreateReplicationConfigurationTemplate',
                'CreateSourceNetwork',
                'CreateSourceServerForDrs',
                'DeleteJob',
                'DeleteLaunchAction',
                'DeleteLaunchConfigurationTemplate',
                'DeleteRecoveryInstance',
                'DeleteReplicationConfigurationTemplate',
                'DeleteSourceNetwork',
                'DeleteSourceServer',
                'DisconnectRecoveryInstance',
                'DisconnectSourceServer',
                'ExportSourceNetworkCfnTemplate',
                'InitializeService',
                'IssueAgentCertificateForDrs',
                'NotifyAgentAuthenticationForDrs',
                'NotifyAgentConnectedForDrs',
                'NotifyAgentDisconnectedForDrs',
                'NotifyAgentReplicationProgressForDrs',
                'NotifyConsistencyAttainedForDrs',
                'NotifyReplicationServerAuthenticationForDrs',
                'NotifyVolumeEventForDrs',
                'PutLaunchAction',
                'RetryDataReplication',
                'ReverseReplication',
                'SendAgentLogsForDrs',
                'SendAgentMetricsForDrs',
                'SendChannelCommandResultForDrs',
                'SendClientLogsForDrs',
                'SendClientMetricsForDrs',
                'SendVolumeStatsForDrs',
                'StartFailbackLaunch',
                'StartRecovery',
                'StartReplication',
                'StartSourceNetworkRecovery',
                'StartSourceNetworkReplication',
                'StopFailback',
                'StopReplication',
                'StopSourceNetworkReplication',
                'TerminateRecoveryInstances',
                'UpdateAgentBacklogForDrs',
                'UpdateAgentConversionInfoForDrs',
                'UpdateAgentReplicationInfoForDrs',
                'UpdateAgentReplicationProcessStateForDrs',
                'UpdateAgentSourcePropertiesForDrs',
                'UpdateFailbackClientDeviceMappingForDrs',
                'UpdateFailbackClientLastSeenForDrs',
                'UpdateFailbackReplicationConfiguration',
                'UpdateLaunchConfiguration',
                'UpdateLaunchConfigurationTemplate',
                'UpdateReplicationCertificateForDrs',
                'UpdateReplicationConfiguration',
                'UpdateReplicationConfigurationTemplate'
            ],
            Read: [
                'DescribeJobLogItems',
                'DescribeJobs',
                'DescribeLaunchConfigurationTemplates',
                'DescribeRecoveryInstances',
                'DescribeRecoverySnapshots',
                'DescribeReplicationConfigurationTemplates',
                'DescribeReplicationServerAssociationsForDrs',
                'DescribeSnapshotRequestsForDrs',
                'DescribeSourceNetworks',
                'DescribeSourceServers',
                'GetAgentCommandForDrs',
                'GetAgentConfirmedResumeInfoForDrs',
                'GetAgentInstallationAssetsForDrs',
                'GetAgentReplicationInfoForDrs',
                'GetAgentRuntimeConfigurationForDrs',
                'GetAgentSnapshotCreditsForDrs',
                'GetChannelCommandsForDrs',
                'GetFailbackCommandForDrs',
                'GetFailbackLaunchRequestedForDrs',
                'GetFailbackReplicationConfiguration',
                'GetLaunchConfiguration',
                'GetReplicationConfiguration',
                'GetSuggestedFailbackClientDeviceMappingForDrs',
                'ListExtensibleSourceServers',
                'ListLaunchActions',
                'ListStagingAccounts',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Drs = Drs;
//# sourceMappingURL=data:application/json;base64,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