"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mediapackagev2 = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [mediapackagev2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselementalmediapackagev2.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Mediapackagev2 extends shared_1.PolicyStatement {
    /**
     * Grants permission to cancel a harvest job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_CancelHarvestJob.html
     */
    toCancelHarvestJob() {
        return this.to('CancelHarvestJob');
    }
    /**
     * Grants permission to create a channel in a channel group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_CreateChannel.html
     */
    toCreateChannel() {
        return this.to('CreateChannel');
    }
    /**
     * Grants permission to create a channel group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_CreateChannelGroup.html
     */
    toCreateChannelGroup() {
        return this.to('CreateChannelGroup');
    }
    /**
     * Grants permission to create a harvest job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_CreateHarvestJob.html
     */
    toCreateHarvestJob() {
        return this.to('CreateHarvestJob');
    }
    /**
     * Grants permission to create an origin endpoint for a channel
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_CreateOriginEndpoint.html
     */
    toCreateOriginEndpoint() {
        return this.to('CreateOriginEndpoint');
    }
    /**
     * Grants permission to delete a channel in a channel group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_DeleteChannel.html
     */
    toDeleteChannel() {
        return this.to('DeleteChannel');
    }
    /**
     * Grants permission to delete a channel group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_DeleteChannelGroup.html
     */
    toDeleteChannelGroup() {
        return this.to('DeleteChannelGroup');
    }
    /**
     * Grants permission to delete a resource policy from a channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_DeleteChannelPolicy.html
     */
    toDeleteChannelPolicy() {
        return this.to('DeleteChannelPolicy');
    }
    /**
     * Grants permission to delete an origin endpoint of a channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_DeleteOriginEndpoint.html
     */
    toDeleteOriginEndpoint() {
        return this.to('DeleteOriginEndpoint');
    }
    /**
     * Grants permission to delete a resource policy from an origin endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_DeleteOriginEndpointPolicy.html
     */
    toDeleteOriginEndpointPolicy() {
        return this.to('DeleteOriginEndpointPolicy');
    }
    /**
     * Grants permission to retrieve details of a channel in a channel group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_GetChannel.html
     */
    toGetChannel() {
        return this.to('GetChannel');
    }
    /**
     * Grants permission to retrieve details of a channel group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_GetChannelGroup.html
     */
    toGetChannelGroup() {
        return this.to('GetChannelGroup');
    }
    /**
     * Grants permission to retrieve a resource policy for a channel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_GetChannelPolicy.html
     */
    toGetChannelPolicy() {
        return this.to('GetChannelPolicy');
    }
    /**
     * Grants permission to retrieve details of an harvest job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_GetHarvestJob.html
     */
    toGetHarvestJob() {
        return this.to('GetHarvestJob');
    }
    /**
     * Grants permission to make GetHeadObject requests to MediaPackage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage/latest/userguide/dataplane-apis.html
     */
    toGetHeadObject() {
        return this.to('GetHeadObject');
    }
    /**
     * Grants permission to make GetObject requests to MediaPackage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage/latest/userguide/dataplane-apis.html
     */
    toGetObject() {
        return this.to('GetObject');
    }
    /**
     * Grants permission to retrieve details of an origin endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_GetOriginEndpoint.html
     */
    toGetOriginEndpoint() {
        return this.to('GetOriginEndpoint');
    }
    /**
     * Grants permission to retrieve details of a resource policy for an origin endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_GetOriginEndpointPolicy.html
     */
    toGetOriginEndpointPolicy() {
        return this.to('GetOriginEndpointPolicy');
    }
    /**
     * Grants permission to make HarvestObject requests to MediaPackage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage/latest/userguide/dataplane-apis.html
     */
    toHarvestObject() {
        return this.to('HarvestObject');
    }
    /**
     * Grants permission to list all channel groups for an aws account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_ListChannelGroups.html
     */
    toListChannelGroups() {
        return this.to('ListChannelGroups');
    }
    /**
     * Grants permission to list all channels in a channel group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_ListChannels.html
     */
    toListChannels() {
        return this.to('ListChannels');
    }
    /**
     * Grants permission to list all harvest jobs in a channel group, channel, origin endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_ListHarvestJobs.html
     */
    toListHarvestJobs() {
        return this.to('ListHarvestJobs');
    }
    /**
     * Grants permission to list all origin endpoints of a channel
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_ListOriginEndpoints.html
     */
    toListOriginEndpoints() {
        return this.to('ListOriginEndpoints');
    }
    /**
     * Grants permission to list tags for the specified resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to attach a resource policy for a channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_PutChannelPolicy.html
     */
    toPutChannelPolicy() {
        return this.to('PutChannelPolicy');
    }
    /**
     * Grants permission to make PutObject requests to MediaPackage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/userguide/dataplane-apis.html
     */
    toPutObject() {
        return this.to('PutObject');
    }
    /**
     * Grants permission to attach a resource policy to an origin endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_PutOriginEndpointPolicy.html
     */
    toPutOriginEndpointPolicy() {
        return this.to('PutOriginEndpointPolicy');
    }
    /**
     * Grants permission to reset a channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_ResetChannelState.html
     */
    toResetChannelState() {
        return this.to('ResetChannelState');
    }
    /**
     * Grants permission to reset an origin endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_ResetOriginEndpointState.html
     */
    toResetOriginEndpointState() {
        return this.to('ResetOriginEndpointState');
    }
    /**
     * Grants permission to add specified tags to the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove the specified tags from the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a channel in a channel group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_UpdateChannel.html
     */
    toUpdateChannel() {
        return this.to('UpdateChannel');
    }
    /**
     * Grants permission to update a channel group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_UpdateChannelGroup.html
     */
    toUpdateChannelGroup() {
        return this.to('UpdateChannelGroup');
    }
    /**
     * Grants permission to update an origin endpoint of a channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/APIReference/API_UpdateOriginEndpoint.html
     */
    toUpdateOriginEndpoint() {
        return this.to('UpdateOriginEndpoint');
    }
    /**
     * Adds a resource of type ChannelGroup to the statement
     *
     * https://docs.aws.amazon.com/mediapackage/latest/userguide/channel-groups.html
     *
     * @param channelGroupName - Identifier for the channelGroupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onChannelGroup(channelGroupName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mediapackagev2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:channelGroup/${channelGroupName}`);
    }
    /**
     * Adds a resource of type ChannelPolicy to the statement
     *
     * https://docs.aws.amazon.com/mediapackage/latest/userguide/API_GetChannelPolicy.html
     *
     * @param channelGroupName - Identifier for the channelGroupName.
     * @param channelName - Identifier for the channelName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onChannelPolicy(channelGroupName, channelName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mediapackagev2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:channelGroup/${channelGroupName}/channel/${channelName}`);
    }
    /**
     * Adds a resource of type Channel to the statement
     *
     * https://docs.aws.amazon.com/mediapackage/latest/userguide/channels.html
     *
     * @param channelGroupName - Identifier for the channelGroupName.
     * @param channelName - Identifier for the channelName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onChannel(channelGroupName, channelName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mediapackagev2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:channelGroup/${channelGroupName}/channel/${channelName}`);
    }
    /**
     * Adds a resource of type OriginEndpointPolicy to the statement
     *
     * https://docs.aws.amazon.com/mediapackage/latest/userguide/API_GetOriginEndpointPolicy.html
     *
     * @param channelGroupName - Identifier for the channelGroupName.
     * @param channelName - Identifier for the channelName.
     * @param originEndpointName - Identifier for the originEndpointName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onOriginEndpointPolicy(channelGroupName, channelName, originEndpointName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mediapackagev2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:channelGroup/${channelGroupName}/channel/${channelName}/originEndpoint/${originEndpointName}`);
    }
    /**
     * Adds a resource of type OriginEndpoint to the statement
     *
     * https://docs.aws.amazon.com/mediapackage/latest/userguide/endpoints.html
     *
     * @param channelGroupName - Identifier for the channelGroupName.
     * @param channelName - Identifier for the channelName.
     * @param originEndpointName - Identifier for the originEndpointName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onOriginEndpoint(channelGroupName, channelName, originEndpointName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mediapackagev2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:channelGroup/${channelGroupName}/channel/${channelName}/originEndpoint/${originEndpointName}`);
    }
    /**
     * Adds a resource of type HarvestJob to the statement
     *
     * https://docs.aws.amazon.com/mediapackage/latest/userguide/API_HarvestJobListConfiguration.html
     *
     * @param channelGroupName - Identifier for the channelGroupName.
     * @param channelName - Identifier for the channelName.
     * @param originEndpointName - Identifier for the originEndpointName.
     * @param harvestJobName - Identifier for the harvestJobName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onHarvestJob(channelGroupName, channelName, originEndpointName, harvestJobName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mediapackagev2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:channelGroup/${channelGroupName}/channel/${channelName}/originEndpoint/${originEndpointName}/harvestJob/${harvestJobName}`);
    }
    /**
     * Filters access by tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateChannel()
     * - .toCreateChannelGroup()
     * - .toCreateHarvestJob()
     * - .toCreateOriginEndpoint()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - ChannelGroup
     * - Channel
     * - OriginEndpoint
     * - HarvestJob
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateChannel()
     * - .toCreateChannelGroup()
     * - .toCreateHarvestJob()
     * - .toCreateOriginEndpoint()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [mediapackagev2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselementalmediapackagev2.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'mediapackagev2';
        this.accessLevelList = {
            Write: [
                'CancelHarvestJob',
                'CreateChannel',
                'CreateChannelGroup',
                'CreateHarvestJob',
                'CreateOriginEndpoint',
                'DeleteChannel',
                'DeleteChannelGroup',
                'DeleteChannelPolicy',
                'DeleteOriginEndpoint',
                'DeleteOriginEndpointPolicy',
                'PutChannelPolicy',
                'PutObject',
                'PutOriginEndpointPolicy',
                'ResetChannelState',
                'ResetOriginEndpointState',
                'UpdateChannel',
                'UpdateChannelGroup',
                'UpdateOriginEndpoint'
            ],
            Read: [
                'GetChannel',
                'GetChannelGroup',
                'GetChannelPolicy',
                'GetHarvestJob',
                'GetHeadObject',
                'GetObject',
                'GetOriginEndpoint',
                'GetOriginEndpointPolicy',
                'HarvestObject',
                'ListTagsForResource'
            ],
            List: [
                'ListChannelGroups',
                'ListChannels',
                'ListHarvestJobs',
                'ListOriginEndpoints'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Mediapackagev2 = Mediapackagev2;
//# sourceMappingURL=data:application/json;base64,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