import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [freertos](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonfreertos.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class Freertos extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to create a software configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toCreateSoftwareConfiguration(): this;
    /**
     * Grants permission to create a subscription for FreeRTOS extended maintenance plan (EMP)
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-getting-started-emp.html
     */
    toCreateSubscription(): this;
    /**
     * Grants permission to delete the software configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toDeleteSoftwareConfiguration(): this;
    /**
     * Grants permission to describe the hardware platform
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toDescribeHardwarePlatform(): this;
    /**
     * Grants permission to describe the software configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toDescribeSoftwareConfiguration(): this;
    /**
     * Grants permission to describes the subscription for FreeRTOS extended maintenance plan (EMP)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-getting-started-emp.html
     */
    toDescribeSubscription(): this;
    /**
     * Grants permission to get URL for sotware patch-release, patch-diff and release notes under FreeRTOS extended maintenance plan (EMP)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-getting-started-emp.html
     */
    toGetEmpPatchUrl(): this;
    /**
     * Grants permission to get the URL for Amazon FreeRTOS software download
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toGetSoftwareURL(): this;
    /**
     * Grants permission to get the URL for Amazon FreeRTOS software download based on the configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toGetSoftwareURLForConfiguration(): this;
    /**
     * Grants permission to fetch the subscription billing amount for FreeRTOS extended maintenance plan (EMP)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-getting-started-emp.html
     */
    toGetSubscriptionBillingAmount(): this;
    /**
     * Grants permission to lists versions of AmazonFreeRTOS
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toListFreeRTOSVersions(): this;
    /**
     * Grants permission to list the hardware platforms
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toListHardwarePlatforms(): this;
    /**
     * Grants permission to list the hardware vendors
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toListHardwareVendors(): this;
    /**
     * Grants permission to lists the software configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toListSoftwareConfigurations(): this;
    /**
     * Grants permission to list software patches of subscription for FreeRTOS extended maintenance plan (EMP)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-getting-started-emp.html
     */
    toListSoftwarePatches(): this;
    /**
     * Grants permission to list the subscription emails for FreeRTOS extended maintenance plan (EMP)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-getting-started-emp.html
     */
    toListSubscriptionEmails(): this;
    /**
     * Grants permission to list the subscriptions for FreeRTOS extended maintenance plan (EMP)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-getting-started-emp.html
     */
    toListSubscriptions(): this;
    /**
     * Grants permission to update list of subscription email address for FreeRTOS extended maintenance plan (EMP)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-getting-started-emp.html
     */
    toUpdateEmailRecipients(): this;
    /**
     * Grants permission to update the software configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    toUpdateSoftwareConfiguration(): this;
    /**
     * Grants permission to verify the email for FreeRTOS extended maintenance plan (EMP)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-getting-started-emp.html
     */
    toVerifyEmail(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type configuration to the statement
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     *
     * @param configurationName - Identifier for the configurationName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfiguration(configurationName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type subscription to the statement
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-getting-started-emp.html
     *
     * @param subscriptionID - Identifier for the subscriptionID.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSubscription(subscriptionID: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by tag key present in the request that the user makes to Amazon FreeRTOS
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateSoftwareConfiguration()
     * - .toCreateSubscription()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by tag key component attached to an Amazon FreeRTOS resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - configuration
     * - subscription
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the list of all the tag key names associated with the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateSoftwareConfiguration()
     * - .toCreateSubscription()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [freertos](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonfreertos.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
