"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Gamelift = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [gamelift](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazongameliftservers.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Gamelift extends shared_1.PolicyStatement {
    /**
     * Grants permission to register player acceptance or rejection of a proposed FlexMatch match
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_AcceptMatch.html
     */
    toAcceptMatch() {
        return this.to('AcceptMatch');
    }
    /**
     * Grants permission to locate and reserve a game server to host a new game session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_ClaimGameServer.html
     */
    toClaimGameServer() {
        return this.to('ClaimGameServer');
    }
    /**
     * Grants permission to define a new alias for a fleet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - gamelift:TagResource
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_CreateAlias.html
     */
    toCreateAlias() {
        return this.to('CreateAlias');
    }
    /**
     * Grants permission to create a new game build using files stored in an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - gamelift:TagResource
     * - iam:PassRole
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_CreateBuild.html
     */
    toCreateBuild() {
        return this.to('CreateBuild');
    }
    /**
     * Grants permission to create a new container fleet of computing resources to run your game servers
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:DescribeAvailabilityZones
     * - ec2:DescribeRegions
     * - gamelift:TagResource
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_CreateContainerFleet.html
     */
    toCreateContainerFleet() {
        return this.to('CreateContainerFleet');
    }
    /**
     * Grants permission to create a new container group definition using images stored in an Amazon ECR repository
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ecr:BatchGetImage
     * - ecr:DescribeImages
     * - ecr:GetAuthorizationToken
     * - ecr:GetDownloadUrlForLayer
     * - gamelift:TagResource
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_CreateContainerGroupDefinition.html
     */
    toCreateContainerGroupDefinition() {
        return this.to('CreateContainerGroupDefinition');
    }
    /**
     * Grants permission to create a new fleet of computing resources to run your game servers
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:DescribeAvailabilityZones
     * - ec2:DescribeRegions
     * - gamelift:TagResource
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_CreateFleet.html
     */
    toCreateFleet() {
        return this.to('CreateFleet');
    }
    /**
     * Grants permission to specify additional locations for a fleet
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DescribeAvailabilityZones
     * - ec2:DescribeRegions
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_CreateFleetLocations.html
     */
    toCreateFleetLocations() {
        return this.to('CreateFleetLocations');
    }
    /**
     * Grants permission to create a new game server group, set up a corresponding Auto Scaling group, and launche instances to host game servers
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - autoscaling:CreateAutoScalingGroup
     * - autoscaling:DescribeAutoScalingGroups
     * - autoscaling:PutLifecycleHook
     * - autoscaling:PutScalingPolicy
     * - ec2:DescribeAvailabilityZones
     * - ec2:DescribeSubnets
     * - events:PutRule
     * - events:PutTargets
     * - gamelift:TagResource
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_CreateGameServerGroup.html
     */
    toCreateGameServerGroup() {
        return this.to('CreateGameServerGroup');
    }
    /**
     * Grants permission to start a new game session on a specified fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_CreateGameSession.html
     */
    toCreateGameSession() {
        return this.to('CreateGameSession');
    }
    /**
     * Grants permission to set up a new queue for processing game session placement requests
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - gamelift:TagResource
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_CreateGameSessionQueue.html
     */
    toCreateGameSessionQueue() {
        return this.to('CreateGameSessionQueue');
    }
    /**
     * Grants permission to define a new location for a fleet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - gamelift:TagResource
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_CreateLocation.html
     */
    toCreateLocation() {
        return this.to('CreateLocation');
    }
    /**
     * Grants permission to create a new FlexMatch matchmaker
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - gamelift:TagResource
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_CreateMatchmakingConfiguration.html
     */
    toCreateMatchmakingConfiguration() {
        return this.to('CreateMatchmakingConfiguration');
    }
    /**
     * Grants permission to create a new matchmaking rule set for FlexMatch
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - gamelift:TagResource
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_CreateMatchmakingRuleSet.html
     */
    toCreateMatchmakingRuleSet() {
        return this.to('CreateMatchmakingRuleSet');
    }
    /**
     * Grants permission to reserve an available game session slot for a player
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_CreatePlayerSession.html
     */
    toCreatePlayerSession() {
        return this.to('CreatePlayerSession');
    }
    /**
     * Grants permission to reserve available game session slots for multiple players
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_CreatePlayerSessions.html
     */
    toCreatePlayerSessions() {
        return this.to('CreatePlayerSessions');
    }
    /**
     * Grants permission to create a new Realtime Servers script
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - gamelift:TagResource
     * - iam:PassRole
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_CreateScript.html
     */
    toCreateScript() {
        return this.to('CreateScript');
    }
    /**
     * Grants permission to allow GameLift to create or delete a peering connection between a GameLift fleet VPC and a VPC on another AWS account
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:AcceptVpcPeeringConnection
     * - ec2:AuthorizeSecurityGroupEgress
     * - ec2:AuthorizeSecurityGroupIngress
     * - ec2:CreateRoute
     * - ec2:DeleteRoute
     * - ec2:DescribeRouteTables
     * - ec2:DescribeSecurityGroups
     * - ec2:RevokeSecurityGroupEgress
     * - ec2:RevokeSecurityGroupIngress
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_CreateVpcPeeringAuthorization.html
     */
    toCreateVpcPeeringAuthorization() {
        return this.to('CreateVpcPeeringAuthorization');
    }
    /**
     * Grants permission to establish a peering connection between your GameLift fleet VPC and a VPC on another account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_CreateVpcPeeringConnection.html
     */
    toCreateVpcPeeringConnection() {
        return this.to('CreateVpcPeeringConnection');
    }
    /**
     * Grants permission to delete an alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DeleteAlias.html
     */
    toDeleteAlias() {
        return this.to('DeleteAlias');
    }
    /**
     * Grants permission to delete a game build
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DeleteBuild.html
     */
    toDeleteBuild() {
        return this.to('DeleteBuild');
    }
    /**
     * Grants permission to delete a container fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DeleteContainerFleet.html
     */
    toDeleteContainerFleet() {
        return this.to('DeleteContainerFleet');
    }
    /**
     * Grants permission to delete a container group definition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DeleteContainerGroupDefinition.html
     */
    toDeleteContainerGroupDefinition() {
        return this.to('DeleteContainerGroupDefinition');
    }
    /**
     * Grants permission to delete an empty fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DeleteFleet.html
     */
    toDeleteFleet() {
        return this.to('DeleteFleet');
    }
    /**
     * Grants permission to delete locations for a fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DeleteFleetLocations.html
     */
    toDeleteFleetLocations() {
        return this.to('DeleteFleetLocations');
    }
    /**
     * Grants permission to permanently delete a game server group and terminate FleetIQ activity for the corresponding Auto Scaling group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - autoscaling:DeleteAutoScalingGroup
     * - autoscaling:DescribeAutoScalingGroups
     * - autoscaling:ExitStandby
     * - autoscaling:ResumeProcesses
     * - autoscaling:SetInstanceProtection
     * - autoscaling:UpdateAutoScalingGroup
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DeleteGameServerGroup.html
     */
    toDeleteGameServerGroup() {
        return this.to('DeleteGameServerGroup');
    }
    /**
     * Grants permission to delete an existing game session queue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DeleteGameSessionQueue.html
     */
    toDeleteGameSessionQueue() {
        return this.to('DeleteGameSessionQueue');
    }
    /**
     * Grants permission to delete a location
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DeleteLocation.html
     */
    toDeleteLocation() {
        return this.to('DeleteLocation');
    }
    /**
     * Grants permission to delete an existing FlexMatch matchmaker
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DeleteMatchmakingConfiguration.html
     */
    toDeleteMatchmakingConfiguration() {
        return this.to('DeleteMatchmakingConfiguration');
    }
    /**
     * Grants permission to delete an existing FlexMatch matchmaking rule set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DeleteMatchmakingRuleSet.html
     */
    toDeleteMatchmakingRuleSet() {
        return this.to('DeleteMatchmakingRuleSet');
    }
    /**
     * Grants permission to delete a set of auto-scaling rules
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DeleteScalingPolicy.html
     */
    toDeleteScalingPolicy() {
        return this.to('DeleteScalingPolicy');
    }
    /**
     * Grants permission to delete a Realtime Servers script
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DeleteScript.html
     */
    toDeleteScript() {
        return this.to('DeleteScript');
    }
    /**
     * Grants permission to cancel a VPC peering authorization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DeleteVpcPeeringAuthorization.html
     */
    toDeleteVpcPeeringAuthorization() {
        return this.to('DeleteVpcPeeringAuthorization');
    }
    /**
     * Grants permission to remove a peering connection between VPCs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DeleteVpcPeeringConnection.html
     */
    toDeleteVpcPeeringConnection() {
        return this.to('DeleteVpcPeeringConnection');
    }
    /**
     * Grants permission to deregister a compute against a fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DeregisterCompute.html
     */
    toDeregisterCompute() {
        return this.to('DeregisterCompute');
    }
    /**
     * Grants permission to remove a game server from a game server group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DeregisterGameServer.html
     */
    toDeregisterGameServer() {
        return this.to('DeregisterGameServer');
    }
    /**
     * Grants permission to retrieve properties for an alias
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeAlias.html
     */
    toDescribeAlias() {
        return this.to('DescribeAlias');
    }
    /**
     * Grants permission to retrieve properties for a game build
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeBuild.html
     */
    toDescribeBuild() {
        return this.to('DescribeBuild');
    }
    /**
     * Grants permission to retrieve information for a compute in a fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeCompute.html
     */
    toDescribeCompute() {
        return this.to('DescribeCompute');
    }
    /**
     * Grants permission to retrieve the properties of an existing container fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeContainerFleet.html
     */
    toDescribeContainerFleet() {
        return this.to('DescribeContainerFleet');
    }
    /**
     * Grants permission to retrieve the properties of an existing container group definition
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeContainerGroupDefinition.html
     */
    toDescribeContainerGroupDefinition() {
        return this.to('DescribeContainerGroupDefinition');
    }
    /**
     * Grants permission to retrieve the maximum allowed and current usage for EC2 instance types
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeEC2InstanceLimits.html
     */
    toDescribeEC2InstanceLimits() {
        return this.to('DescribeEC2InstanceLimits');
    }
    /**
     * Grants permission to retrieve general properties, including status, for fleets
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeFleetAttributes.html
     */
    toDescribeFleetAttributes() {
        return this.to('DescribeFleetAttributes');
    }
    /**
     * Grants permission to retrieve the current capacity settings for managed fleets
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeFleetCapacity.html
     */
    toDescribeFleetCapacity() {
        return this.to('DescribeFleetCapacity');
    }
    /**
     * Grants permission to retrieve the properties of an existing fleet deployment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeFleetDeployment.html
     */
    toDescribeFleetDeployment() {
        return this.to('DescribeFleetDeployment');
    }
    /**
     * Grants permission to retrieve entries from a fleet's event log
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeFleetEvents.html
     */
    toDescribeFleetEvents() {
        return this.to('DescribeFleetEvents');
    }
    /**
     * Grants permission to retrieve general properties, including statuses, for a fleet's locations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeFleetLocationAttributes.html
     */
    toDescribeFleetLocationAttributes() {
        return this.to('DescribeFleetLocationAttributes');
    }
    /**
     * Grants permission to retrieve the current capacity setting for a fleet's location
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeFleetLocationCapacity.html
     */
    toDescribeFleetLocationCapacity() {
        return this.to('DescribeFleetLocationCapacity');
    }
    /**
     * Grants permission to retrieve utilization statistics for fleet's location
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeFleetLocationUtilization.html
     */
    toDescribeFleetLocationUtilization() {
        return this.to('DescribeFleetLocationUtilization');
    }
    /**
     * Grants permission to retrieve the inbound connection permissions for a fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeFleetPortSettings.html
     */
    toDescribeFleetPortSettings() {
        return this.to('DescribeFleetPortSettings');
    }
    /**
     * Grants permission to retrieve utilization statistics for fleets
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeFleetUtilization.html
     */
    toDescribeFleetUtilization() {
        return this.to('DescribeFleetUtilization');
    }
    /**
     * Grants permission to retrieve properties for a game server
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeGameServer.html
     */
    toDescribeGameServer() {
        return this.to('DescribeGameServer');
    }
    /**
     * Grants permission to retrieve properties for a game server group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeGameServerGroup.html
     */
    toDescribeGameServerGroup() {
        return this.to('DescribeGameServerGroup');
    }
    /**
     * Grants permission to retrieve the status of EC2 instances in a game server group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeGameServerInstances.html
     */
    toDescribeGameServerInstances() {
        return this.to('DescribeGameServerInstances');
    }
    /**
     * Grants permission to retrieve properties for game sessions in a fleet, including the protection policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeGameSessionDetails.html
     */
    toDescribeGameSessionDetails() {
        return this.to('DescribeGameSessionDetails');
    }
    /**
     * Grants permission to retrieve details of a game session placement request
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeGameSessionPlacement.html
     */
    toDescribeGameSessionPlacement() {
        return this.to('DescribeGameSessionPlacement');
    }
    /**
     * Grants permission to retrieve properties for game session queues
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeGameSessionQueues.html
     */
    toDescribeGameSessionQueues() {
        return this.to('DescribeGameSessionQueues');
    }
    /**
     * Grants permission to retrieve properties for game sessions in a fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeGameSessions.html
     */
    toDescribeGameSessions() {
        return this.to('DescribeGameSessions');
    }
    /**
     * Grants permission to retrieve information about instances in a managed fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeInstances.html
     */
    toDescribeInstances() {
        return this.to('DescribeInstances');
    }
    /**
     * Grants permission to retrieve details of matchmaking tickets
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeMatchmaking.html
     */
    toDescribeMatchmaking() {
        return this.to('DescribeMatchmaking');
    }
    /**
     * Grants permission to retrieve properties for FlexMatch matchmakers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeMatchmakingConfigurations.html
     */
    toDescribeMatchmakingConfigurations() {
        return this.to('DescribeMatchmakingConfigurations');
    }
    /**
     * Grants permission to retrieve properties for FlexMatch matchmaking rule sets
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeMatchmakingRuleSets.html
     */
    toDescribeMatchmakingRuleSets() {
        return this.to('DescribeMatchmakingRuleSets');
    }
    /**
     * Grants permission to retrieve properties for player sessions in a game session
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribePlayerSessions.html
     */
    toDescribePlayerSessions() {
        return this.to('DescribePlayerSessions');
    }
    /**
     * Grants permission to retrieve the current runtime configuration for a fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeRuntimeConfiguration.html
     */
    toDescribeRuntimeConfiguration() {
        return this.to('DescribeRuntimeConfiguration');
    }
    /**
     * Grants permission to retrieve all scaling policies that are applied to a fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeScalingPolicies.html
     */
    toDescribeScalingPolicies() {
        return this.to('DescribeScalingPolicies');
    }
    /**
     * Grants permission to retrieve properties for a Realtime Servers script
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeScript.html
     */
    toDescribeScript() {
        return this.to('DescribeScript');
    }
    /**
     * Grants permission to retrieve valid VPC peering authorizations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeVpcPeeringAuthorizations.html
     */
    toDescribeVpcPeeringAuthorizations() {
        return this.to('DescribeVpcPeeringAuthorizations');
    }
    /**
     * Grants permission to retrieve details on active or pending VPC peering connections
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_DescribeVpcPeeringConnections.html
     */
    toDescribeVpcPeeringConnections() {
        return this.to('DescribeVpcPeeringConnections');
    }
    /**
     * Grants permission to retrieve credentials to remotely access a compute in a managed fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_GetComputeAccess.html
     */
    toGetComputeAccess() {
        return this.to('GetComputeAccess');
    }
    /**
     * Grants permission to retrieve an authentication token that allows processes on a compute to send requests to the Amazon GameLift service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_GetComputeAuthToken.html
     */
    toGetComputeAuthToken() {
        return this.to('GetComputeAuthToken');
    }
    /**
     * Grants permission to retrieve the location of stored logs for a game session
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_GetGameSessionLogUrl.html
     */
    toGetGameSessionLogUrl() {
        return this.to('GetGameSessionLogUrl');
    }
    /**
     * Grants permission to request remote access to a specified fleet instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_GetInstanceAccess.html
     */
    toGetInstanceAccess() {
        return this.to('GetInstanceAccess');
    }
    /**
     * Grants permission to retrieve all aliases that are defined in the current Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_ListAliases.html
     */
    toListAliases() {
        return this.to('ListAliases');
    }
    /**
     * Grants permission to retrieve all game build in the current Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_ListBuilds.html
     */
    toListBuilds() {
        return this.to('ListBuilds');
    }
    /**
     * Grants permission to retrieve all compute resources in the current Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_ListCompute.html
     */
    toListCompute() {
        return this.to('ListCompute');
    }
    /**
     * Grants permission to retrieve the properties of all existing container fleets in the current Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_ListContainerFleets.html
     */
    toListContainerFleets() {
        return this.to('ListContainerFleets');
    }
    /**
     * Grants permission to retrieve the properties of all versions of an existing container group definition
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_ListContainerGroupDefinitionVersions.html
     */
    toListContainerGroupDefinitionVersions() {
        return this.to('ListContainerGroupDefinitionVersions');
    }
    /**
     * Grants permission to retrieve the properties of all existing container group definitions in the current Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_ListContainerGroupDefinitions.html
     */
    toListContainerGroupDefinitions() {
        return this.to('ListContainerGroupDefinitions');
    }
    /**
     * Grants permission to retrieve the properties of all existing fleet deployments in the current Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_ListFleetDeployments.html
     */
    toListFleetDeployments() {
        return this.to('ListFleetDeployments');
    }
    /**
     * Grants permission to retrieve a list of fleet IDs for all fleets in the current Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_ListFleets.html
     */
    toListFleets() {
        return this.to('ListFleets');
    }
    /**
     * Grants permission to retrieve all game server groups that are defined in the current Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_ListGameServerGroups.html
     */
    toListGameServerGroups() {
        return this.to('ListGameServerGroups');
    }
    /**
     * Grants permission to retrieve all game servers that are currently running in a game server group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_ListGameServers.html
     */
    toListGameServers() {
        return this.to('ListGameServers');
    }
    /**
     * Grants permission to retrieve all locations in this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_ListLocations.html
     */
    toListLocations() {
        return this.to('ListLocations');
    }
    /**
     * Grants permission to retrieve properties for all Realtime Servers scripts in the current region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_ListScripts.html
     */
    toListScripts() {
        return this.to('ListScripts');
    }
    /**
     * Grants permission to retrieve tags for GameLift resources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to create or update a fleet auto-scaling policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_PutScalingPolicy.html
     */
    toPutScalingPolicy() {
        return this.to('PutScalingPolicy');
    }
    /**
     * Grants permission to register a compute against a fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_RegisterCompute.html
     */
    toRegisterCompute() {
        return this.to('RegisterCompute');
    }
    /**
     * Grants permission to notify GameLift FleetIQ when a new game server is ready to host gameplay
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_RegisterGameServer.html
     */
    toRegisterGameServer() {
        return this.to('RegisterGameServer');
    }
    /**
     * Grants permission to retrieve fresh upload credentials to use when uploading a new game build
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_RequestUploadCredentials.html
     */
    toRequestUploadCredentials() {
        return this.to('RequestUploadCredentials');
    }
    /**
     * Grants permission to retrieve the fleet ID associated with an alias
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_ResolveAlias.html
     */
    toResolveAlias() {
        return this.to('ResolveAlias');
    }
    /**
     * Grants permission to reinstate suspended FleetIQ activity for a game server group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_ResumeGameServerGroup.html
     */
    toResumeGameServerGroup() {
        return this.to('ResumeGameServerGroup');
    }
    /**
     * Grants permission to retrieve game sessions that match a set of search criteria
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_SearchGameSessions.html
     */
    toSearchGameSessions() {
        return this.to('SearchGameSessions');
    }
    /**
     * Grants permission to resume auto-scaling activity on a fleet after it was suspended with StopFleetActions()
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_StartFleetActions.html
     */
    toStartFleetActions() {
        return this.to('StartFleetActions');
    }
    /**
     * Grants permission to send a game session placement request to a game session queue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_StartGameSessionPlacement.html
     */
    toStartGameSessionPlacement() {
        return this.to('StartGameSessionPlacement');
    }
    /**
     * Grants permission to request FlexMatch matchmaking to fill available player slots in an existing game session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_StartMatchBackfill.html
     */
    toStartMatchBackfill() {
        return this.to('StartMatchBackfill');
    }
    /**
     * Grants permission to request FlexMatch matchmaking for one or a group of players and initiate game session placement
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_StartMatchmaking.html
     */
    toStartMatchmaking() {
        return this.to('StartMatchmaking');
    }
    /**
     * Grants permission to suspend auto-scaling activity on a fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_StopFleetActions.html
     */
    toStopFleetActions() {
        return this.to('StopFleetActions');
    }
    /**
     * Grants permission to cancel a game session placement request that is in progress
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_StopGameSessionPlacement.html
     */
    toStopGameSessionPlacement() {
        return this.to('StopGameSessionPlacement');
    }
    /**
     * Grants permission to cancel a matchmaking or match backfill request that is in progress
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_StopMatchmaking.html
     */
    toStopMatchmaking() {
        return this.to('StopMatchmaking');
    }
    /**
     * Grants permission to temporarily stop FleetIQ activity for a game server group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_SuspendGameServerGroup.html
     */
    toSuspendGameServerGroup() {
        return this.to('SuspendGameServerGroup');
    }
    /**
     * Grants permission to tag GameLift resources
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to shut down an existing game session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_TerminateGameSession.html
     */
    toTerminateGameSession() {
        return this.to('TerminateGameSession');
    }
    /**
     * Grants permission to untag GameLift resources
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the properties of an existing alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_UpdateAlias.html
     */
    toUpdateAlias() {
        return this.to('UpdateAlias');
    }
    /**
     * Grants permission to update an existing build's metadata
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_UpdateBuild.html
     */
    toUpdateBuild() {
        return this.to('UpdateBuild');
    }
    /**
     * Grants permission to update an existing container fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_UpdateContainerFleet.html
     */
    toUpdateContainerFleet() {
        return this.to('UpdateContainerFleet');
    }
    /**
     * Grants permission to update the properties of an existing container group definition
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ecr:BatchGetImage
     * - ecr:DescribeImages
     * - ecr:GetAuthorizationToken
     * - ecr:GetDownloadUrlForLayer
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_UpdateContainerGroupDefinition.html
     */
    toUpdateContainerGroupDefinition() {
        return this.to('UpdateContainerGroupDefinition');
    }
    /**
     * Grants permission to update the general properties of an existing fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_UpdateFleetAttributes.html
     */
    toUpdateFleetAttributes() {
        return this.to('UpdateFleetAttributes');
    }
    /**
     * Grants permission to adjust a managed fleet's capacity settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_UpdateFleetCapacity.html
     */
    toUpdateFleetCapacity() {
        return this.to('UpdateFleetCapacity');
    }
    /**
     * Grants permission to adjust a fleet's port settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_UpdateFleetPortSettings.html
     */
    toUpdateFleetPortSettings() {
        return this.to('UpdateFleetPortSettings');
    }
    /**
     * Grants permission to change game server properties, health status, or utilization status
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_UpdateGameServer.html
     */
    toUpdateGameServer() {
        return this.to('UpdateGameServer');
    }
    /**
     * Grants permission to update properties for game server group, including allowed instance types
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_UpdateGameServerGroup.html
     */
    toUpdateGameServerGroup() {
        return this.to('UpdateGameServerGroup');
    }
    /**
     * Grants permission to update the properties of an existing game session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_UpdateGameSession.html
     */
    toUpdateGameSession() {
        return this.to('UpdateGameSession');
    }
    /**
     * Grants permission to update properties of an existing game session queue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_UpdateGameSessionQueue.html
     */
    toUpdateGameSessionQueue() {
        return this.to('UpdateGameSessionQueue');
    }
    /**
     * Grants permission to update properties of an existing FlexMatch matchmaking configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_UpdateMatchmakingConfiguration.html
     */
    toUpdateMatchmakingConfiguration() {
        return this.to('UpdateMatchmakingConfiguration');
    }
    /**
     * Grants permission to update how server processes are configured on instances in an existing fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_UpdateRuntimeConfiguration.html
     */
    toUpdateRuntimeConfiguration() {
        return this.to('UpdateRuntimeConfiguration');
    }
    /**
     * Grants permission to update the metadata and content of an existing Realtime Servers script
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_UpdateScript.html
     */
    toUpdateScript() {
        return this.to('UpdateScript');
    }
    /**
     * Grants permission to validate the syntax of a FlexMatch matchmaking rule set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/apireference/API_ValidateMatchmakingRuleSet.html
     */
    toValidateMatchmakingRuleSet() {
        return this.to('ValidateMatchmakingRuleSet');
    }
    /**
     * Adds a resource of type alias to the statement
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/developerguide/gamelift-console-aliases.html
     *
     * @param aliasId - Identifier for the aliasId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAlias(aliasId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:gamelift:${region ?? this.defaultRegion}::alias/${aliasId}`);
    }
    /**
     * Adds a resource of type build to the statement
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/developerguide/gamelift-console-builds.html
     *
     * @param buildId - Identifier for the buildId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBuild(buildId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:gamelift:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:build/${buildId}`);
    }
    /**
     * Adds a resource of type containerGroupDefinition to the statement
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/developerguide/containers-intro.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onContainerGroupDefinition(name, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:gamelift:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:containergroupdefinition/${name}`);
    }
    /**
     * Adds a resource of type containerFleet to the statement
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/developerguide/containers-intro.html
     *
     * @param fleetId - Identifier for the fleetId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onContainerFleet(fleetId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:gamelift:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:containerfleet/${fleetId}`);
    }
    /**
     * Adds a resource of type fleet to the statement
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/developerguide/gamelift-console-fleets.html
     *
     * @param fleetId - Identifier for the fleetId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFleet(fleetId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:gamelift:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:fleet/${fleetId}`);
    }
    /**
     * Adds a resource of type gameServerGroup to the statement
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/fleetiqguide/gsg-integrate-gameservergroup.html
     *
     * @param gameServerGroupName - Identifier for the gameServerGroupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGameServerGroup(gameServerGroupName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:gamelift:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:gameservergroup/${gameServerGroupName}`);
    }
    /**
     * Adds a resource of type gameSessionQueue to the statement
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/developerguide/queues-console.html
     *
     * @param gameSessionQueueName - Identifier for the gameSessionQueueName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGameSessionQueue(gameSessionQueueName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:gamelift:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:gamesessionqueue/${gameSessionQueueName}`);
    }
    /**
     * Adds a resource of type location to the statement
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/developerguide/fleets-creating-anywhere.html#fleet-anywhere-location
     *
     * @param locationId - Identifier for the locationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLocation(locationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:gamelift:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:location/${locationId}`);
    }
    /**
     * Adds a resource of type matchmakingConfiguration to the statement
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/flexmatchguide/match-create-configuration.html
     *
     * @param matchmakingConfigurationName - Identifier for the matchmakingConfigurationName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMatchmakingConfiguration(matchmakingConfigurationName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:gamelift:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:matchmakingconfiguration/${matchmakingConfigurationName}`);
    }
    /**
     * Adds a resource of type matchmakingRuleSet to the statement
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/flexmatchguide/match-rulesets.html
     *
     * @param matchmakingRuleSetName - Identifier for the matchmakingRuleSetName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMatchmakingRuleSet(matchmakingRuleSetName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:gamelift:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:matchmakingruleset/${matchmakingRuleSetName}`);
    }
    /**
     * Adds a resource of type script to the statement
     *
     * https://docs.aws.amazon.com/gameliftservers/latest/developerguide/gamelift-console-scripts.html
     *
     * @param scriptId - Identifier for the scriptId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onScript(scriptId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:gamelift:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:script/${scriptId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAlias()
     * - .toCreateBuild()
     * - .toCreateContainerFleet()
     * - .toCreateContainerGroupDefinition()
     * - .toCreateFleet()
     * - .toCreateGameServerGroup()
     * - .toCreateGameSessionQueue()
     * - .toCreateLocation()
     * - .toCreateMatchmakingConfiguration()
     * - .toCreateMatchmakingRuleSet()
     * - .toCreateScript()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - alias
     * - build
     * - containerGroupDefinition
     * - containerFleet
     * - fleet
     * - gameServerGroup
     * - gameSessionQueue
     * - location
     * - matchmakingConfiguration
     * - matchmakingRuleSet
     * - script
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAlias()
     * - .toCreateBuild()
     * - .toCreateContainerFleet()
     * - .toCreateContainerGroupDefinition()
     * - .toCreateFleet()
     * - .toCreateGameServerGroup()
     * - .toCreateGameSessionQueue()
     * - .toCreateLocation()
     * - .toCreateMatchmakingConfiguration()
     * - .toCreateMatchmakingRuleSet()
     * - .toCreateScript()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [gamelift](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazongameliftservers.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'gamelift';
        this.accessLevelList = {
            Write: [
                'AcceptMatch',
                'ClaimGameServer',
                'CreateAlias',
                'CreateBuild',
                'CreateContainerFleet',
                'CreateContainerGroupDefinition',
                'CreateFleet',
                'CreateFleetLocations',
                'CreateGameServerGroup',
                'CreateGameSession',
                'CreateGameSessionQueue',
                'CreateLocation',
                'CreateMatchmakingConfiguration',
                'CreateMatchmakingRuleSet',
                'CreatePlayerSession',
                'CreatePlayerSessions',
                'CreateScript',
                'CreateVpcPeeringAuthorization',
                'CreateVpcPeeringConnection',
                'DeleteAlias',
                'DeleteBuild',
                'DeleteContainerFleet',
                'DeleteContainerGroupDefinition',
                'DeleteFleet',
                'DeleteFleetLocations',
                'DeleteGameServerGroup',
                'DeleteGameSessionQueue',
                'DeleteLocation',
                'DeleteMatchmakingConfiguration',
                'DeleteMatchmakingRuleSet',
                'DeleteScalingPolicy',
                'DeleteScript',
                'DeleteVpcPeeringAuthorization',
                'DeleteVpcPeeringConnection',
                'DeregisterCompute',
                'DeregisterGameServer',
                'PutScalingPolicy',
                'RegisterCompute',
                'RegisterGameServer',
                'ResumeGameServerGroup',
                'StartFleetActions',
                'StartGameSessionPlacement',
                'StartMatchBackfill',
                'StartMatchmaking',
                'StopFleetActions',
                'StopGameSessionPlacement',
                'StopMatchmaking',
                'SuspendGameServerGroup',
                'TerminateGameSession',
                'UpdateAlias',
                'UpdateBuild',
                'UpdateContainerFleet',
                'UpdateContainerGroupDefinition',
                'UpdateFleetAttributes',
                'UpdateFleetCapacity',
                'UpdateFleetPortSettings',
                'UpdateGameServer',
                'UpdateGameServerGroup',
                'UpdateGameSession',
                'UpdateGameSessionQueue',
                'UpdateMatchmakingConfiguration',
                'UpdateRuntimeConfiguration',
                'UpdateScript'
            ],
            Read: [
                'DescribeAlias',
                'DescribeBuild',
                'DescribeCompute',
                'DescribeContainerFleet',
                'DescribeContainerGroupDefinition',
                'DescribeEC2InstanceLimits',
                'DescribeFleetAttributes',
                'DescribeFleetCapacity',
                'DescribeFleetDeployment',
                'DescribeFleetEvents',
                'DescribeFleetLocationAttributes',
                'DescribeFleetLocationCapacity',
                'DescribeFleetLocationUtilization',
                'DescribeFleetPortSettings',
                'DescribeFleetUtilization',
                'DescribeGameServer',
                'DescribeGameServerGroup',
                'DescribeGameServerInstances',
                'DescribeGameSessionDetails',
                'DescribeGameSessionPlacement',
                'DescribeGameSessionQueues',
                'DescribeGameSessions',
                'DescribeInstances',
                'DescribeMatchmaking',
                'DescribeMatchmakingConfigurations',
                'DescribeMatchmakingRuleSets',
                'DescribePlayerSessions',
                'DescribeRuntimeConfiguration',
                'DescribeScalingPolicies',
                'DescribeScript',
                'DescribeVpcPeeringAuthorizations',
                'DescribeVpcPeeringConnections',
                'GetComputeAccess',
                'GetComputeAuthToken',
                'GetGameSessionLogUrl',
                'GetInstanceAccess',
                'ListTagsForResource',
                'RequestUploadCredentials',
                'ResolveAlias',
                'SearchGameSessions',
                'ValidateMatchmakingRuleSet'
            ],
            List: [
                'ListAliases',
                'ListBuilds',
                'ListCompute',
                'ListContainerFleets',
                'ListContainerGroupDefinitionVersions',
                'ListContainerGroupDefinitions',
                'ListFleetDeployments',
                'ListFleets',
                'ListGameServerGroups',
                'ListGameServers',
                'ListLocations',
                'ListScripts'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Gamelift = Gamelift;
//# sourceMappingURL=data:application/json;base64,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