"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Gameliftstreams = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [gameliftstreams](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazongameliftstreams.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Gameliftstreams extends shared_1.PolicyStatement {
    /**
     * Grants permission to attach a StreamGroup remote location
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DescribeRegions
     *
     * https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_AddStreamGroupLocations.html
     */
    toAddStreamGroupLocations() {
        return this.to('AddStreamGroupLocations');
    }
    /**
     * Grants permission to associate Applications to a StreamGroup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_AssociateApplications.html
     */
    toAssociateApplications() {
        return this.to('AssociateApplications');
    }
    /**
     * Grants permission to create application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - gameliftstreams:TagResource
     * - s3:GetObject
     * - s3:ListBucket
     *
     * https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_CreateApplication.html
     */
    toCreateApplication() {
        return this.to('CreateApplication');
    }
    /**
     * Grants permission to create a StreamGroup
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - gameliftstreams:TagResource
     *
     * https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_CreateStreamGroup.html
     */
    toCreateStreamGroup() {
        return this.to('CreateStreamGroup');
    }
    /**
     * Grants permission to create a stream session connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_CreateStreamSessionConnection.html
     */
    toCreateStreamSessionConnection() {
        return this.to('CreateStreamSessionConnection');
    }
    /**
     * Grants permission to delete an application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_DeleteApplication.html
     */
    toDeleteApplication() {
        return this.to('DeleteApplication');
    }
    /**
     * Grants permission to delete a StreamGroup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_DeleteStreamGroup.html
     */
    toDeleteStreamGroup() {
        return this.to('DeleteStreamGroup');
    }
    /**
     * Grants permission to disassociate Applications from a StreamGroup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_DisassociateApplications.html
     */
    toDisassociateApplications() {
        return this.to('DisassociateApplications');
    }
    /**
     * Grants permission to export stream session files that your application generates
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_ExportStreamSessionFiles.html
     */
    toExportStreamSessionFiles() {
        return this.to('ExportStreamSessionFiles');
    }
    /**
     * Grants permission to get an application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_GetApplication.html
     */
    toGetApplication() {
        return this.to('GetApplication');
    }
    /**
     * Grants `permission` to get a StreamGroup
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_GetStreamGroup.html
     */
    toGetStreamGroup() {
        return this.to('GetStreamGroup');
    }
    /**
     * Grants permission to get a stream session
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_GetStreamSession.html
     */
    toGetStreamSession() {
        return this.to('GetStreamSession');
    }
    /**
     * Grants permission to list applications
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_ListApplications.html
     */
    toListApplications() {
        return this.to('ListApplications');
    }
    /**
     * Grants permission to list StreamGroups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_ListStreamGroups.html
     */
    toListStreamGroups() {
        return this.to('ListStreamGroups');
    }
    /**
     * Grants permission to list stream sessions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_ListStreamSessions.html
     */
    toListStreamSessions() {
        return this.to('ListStreamSessions');
    }
    /**
     * Grants permission to list stream sessions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_ListStreamSessionsByAccount.html
     */
    toListStreamSessionsByAccount() {
        return this.to('ListStreamSessionsByAccount');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to detach a StreamGroup remote location
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_RemoveStreamGroupLocations.html
     */
    toRemoveStreamGroupLocations() {
        return this.to('RemoveStreamGroupLocations');
    }
    /**
     * Grants permission to create a stream session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_StartStreamSession.html
     */
    toStartStreamSession() {
        return this.to('StartStreamSession');
    }
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to terminate a stream session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_TerminateStreamSession.html
     */
    toTerminateStreamSession() {
        return this.to('TerminateStreamSession');
    }
    /**
     * Grants permission to untag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_UpdateApplication.html
     */
    toUpdateApplication() {
        return this.to('UpdateApplication');
    }
    /**
     * Grants permission to update a StreamGroup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_UpdateStreamGroup.html
     */
    toUpdateStreamGroup() {
        return this.to('UpdateStreamGroup');
    }
    /**
     * Adds a resource of type application to the statement
     *
     * https://docs.aws.amazon.com/gameliftstreams/latest/developerguide/applications.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApplication(applicationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:gameliftstreams:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:application/${applicationId}`);
    }
    /**
     * Adds a resource of type stream group to the statement
     *
     * https://docs.aws.amazon.com/gameliftstreams/latest/developerguide/stream-groups.html
     *
     * @param streamGroupId - Identifier for the streamGroupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStreamGroup(streamGroupId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:gameliftstreams:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:streamgroup/${streamGroupId}`);
    }
    /**
     * Filters access by tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateApplication()
     * - .toCreateStreamGroup()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - application
     * - stream group
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a list of tag keys that are allowed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateApplication()
     * - .toCreateStreamGroup()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [gameliftstreams](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazongameliftstreams.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'gameliftstreams';
        this.accessLevelList = {
            Write: [
                'AddStreamGroupLocations',
                'AssociateApplications',
                'CreateApplication',
                'CreateStreamGroup',
                'CreateStreamSessionConnection',
                'DeleteApplication',
                'DeleteStreamGroup',
                'DisassociateApplications',
                'ExportStreamSessionFiles',
                'RemoveStreamGroupLocations',
                'StartStreamSession',
                'TerminateStreamSession',
                'UpdateApplication',
                'UpdateStreamGroup'
            ],
            Read: [
                'GetApplication',
                'GetStreamGroup',
                'GetStreamSession',
                'ListStreamSessions',
                'ListStreamSessionsByAccount',
                'ListTagsForResource'
            ],
            List: [
                'ListApplications',
                'ListStreamGroups'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Gameliftstreams = Gameliftstreams;
//# sourceMappingURL=data:application/json;base64,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