"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Healthlake = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [healthlake](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awshealthlake.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Healthlake extends shared_1.PolicyStatement {
    /**
     * Grants permission to cancel an on going FHIR Export job with Delete
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/export-datastore-rest.html
     */
    toCancelFHIRExportJobWithDelete() {
        return this.to('CancelFHIRExportJobWithDelete');
    }
    /**
     * Grants permission to create a datastore that can ingest and export FHIR data
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/healthlake/latest/APIReference/API_CreateFHIRDatastore.html
     */
    toCreateFHIRDatastore() {
        return this.to('CreateFHIRDatastore');
    }
    /**
     * Grants permission to create resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/crud-healthlake.html
     */
    toCreateResource() {
        return this.to('CreateResource');
    }
    /**
     * Grants permission to delete a datastore
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/healthlake/latest/APIReference/API_DeleteFHIRDatastore.html
     */
    toDeleteFHIRDatastore() {
        return this.to('DeleteFHIRDatastore');
    }
    /**
     * Grants permission to delete resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/crud-healthlake.html
     */
    toDeleteResource() {
        return this.to('DeleteResource');
    }
    /**
     * Grants permission to get the properties associated with the FHIR datastore, including the datastore ID, datastore ARN, datastore name, datastore status, created at, datastore type version, and datastore endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/healthlake/latest/APIReference/API_DescribeFHIRDatastore.html
     */
    toDescribeFHIRDatastore() {
        return this.to('DescribeFHIRDatastore');
    }
    /**
     * Grants permission to display the properties of a FHIR export job, including the ID, ARN, name, and the status of the datastore
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/healthlake/latest/APIReference/API_DescribeFHIRExportJob.html
     */
    toDescribeFHIRExportJob() {
        return this.to('DescribeFHIRExportJob');
    }
    /**
     * Grants permission to display the properties of a FHIR export job, including the ID, ARN, name, and the status of the datastore with Get
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/export-datastore-rest.html
     */
    toDescribeFHIRExportJobWithGet() {
        return this.to('DescribeFHIRExportJobWithGet');
    }
    /**
     * Grants permission to display the properties of a FHIR import job, including the ID, ARN, name, and the status of the datastore
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/healthlake/latest/APIReference/API_DescribeFHIRImportJob.html
     */
    toDescribeFHIRImportJob() {
        return this.to('DescribeFHIRImportJob');
    }
    /**
     * Grants permission to get the capabilities of a FHIR datastore
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/crud-healthlake.html
     */
    toGetCapabilities() {
        return this.to('GetCapabilities');
    }
    /**
     * Grants permission to access exported files from a FHIR Export job initiated with Get
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/export-datastore-rest.html
     */
    toGetExportedFile() {
        return this.to('GetExportedFile');
    }
    /**
     * Grants permission to read resource history
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/crud-healthlake.html
     */
    toGetHistoryByResourceId() {
        return this.to('GetHistoryByResourceId');
    }
    /**
     * Grants permission to list all FHIR datastores that are in the user's account, regardless of datastore status
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/healthlake/latest/APIReference/API_ListFHIRDatastores.html
     */
    toListFHIRDatastores() {
        return this.to('ListFHIRDatastores');
    }
    /**
     * Grants permission to get a list of export jobs for the specified datastore
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/healthlake/latest/APIReference/API_ListFHIRExportJobs.html
     */
    toListFHIRExportJobs() {
        return this.to('ListFHIRExportJobs');
    }
    /**
     * Grants permission to get a list of import jobs for the specified datastore
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/healthlake/latest/APIReference/API_ListFHIRImportJobs.html
     */
    toListFHIRImportJobs() {
        return this.to('ListFHIRImportJobs');
    }
    /**
     * Grants permission to get a list of tags for the specified datastore
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/healthlake/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to bundle multiple resource operations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/crud-healthlake.html
     */
    toProcessBundle() {
        return this.to('ProcessBundle');
    }
    /**
     * Grants permission to read resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/crud-healthlake.html
     */
    toReadResource() {
        return this.to('ReadResource');
    }
    /**
     * Grants permission to search all resources related to a patient
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/search-healthlake.html
     */
    toSearchEverything() {
        return this.to('SearchEverything');
    }
    /**
     * Grants permission to search resources with GET method
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/search-healthlake.html
     */
    toSearchWithGet() {
        return this.to('SearchWithGet');
    }
    /**
     * Grants permission to search resources with POST method
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/search-healthlake.html
     */
    toSearchWithPost() {
        return this.to('SearchWithPost');
    }
    /**
     * Grants permission to begin a FHIR Export job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/healthlake/latest/APIReference/API_StartFHIRExportJob.html
     */
    toStartFHIRExportJob() {
        return this.to('StartFHIRExportJob');
    }
    /**
     * Grants permission to begin a FHIR Export job with Get
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/export-datastore-rest.html
     */
    toStartFHIRExportJobWithGet() {
        return this.to('StartFHIRExportJobWithGet');
    }
    /**
     * Grants permission to begin a FHIR Export job with Post
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/export-datastore-rest.html
     */
    toStartFHIRExportJobWithPost() {
        return this.to('StartFHIRExportJobWithPost');
    }
    /**
     * Grants permission to begin a FHIR Import job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/healthlake/latest/APIReference/API_StartFHIRImportJob.html
     */
    toStartFHIRImportJob() {
        return this.to('StartFHIRImportJob');
    }
    /**
     * Grants permission to add tags to a datastore
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/healthlake/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags associated with a datastore
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/healthlake/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/crud-healthlake.html
     */
    toUpdateResource() {
        return this.to('UpdateResource');
    }
    /**
     * Grants permission to read version of a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/crud-healthlake.html
     */
    toVersionReadResource() {
        return this.to('VersionReadResource');
    }
    /**
     * Adds a resource of type datastore to the statement
     *
     * https://docs.aws.amazon.com/healthlake/latest/APIReference/API_DatastoreProperties.html
     *
     * @param datastoreId - Identifier for the datastoreId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDatastore(datastoreId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:healthlake:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:datastore/fhir/${datastoreId}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateFHIRDatastore()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toTagResource()
     *
     * Applies to resource types:
     * - datastore
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateFHIRDatastore()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [healthlake](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awshealthlake.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'healthlake';
        this.accessLevelList = {
            Write: [
                'CancelFHIRExportJobWithDelete',
                'CreateFHIRDatastore',
                'CreateResource',
                'DeleteFHIRDatastore',
                'DeleteResource',
                'ProcessBundle',
                'StartFHIRExportJob',
                'StartFHIRExportJobWithGet',
                'StartFHIRExportJobWithPost',
                'StartFHIRImportJob',
                'UpdateResource'
            ],
            Read: [
                'DescribeFHIRDatastore',
                'DescribeFHIRExportJob',
                'DescribeFHIRExportJobWithGet',
                'DescribeFHIRImportJob',
                'GetCapabilities',
                'GetExportedFile',
                'GetHistoryByResourceId',
                'ReadResource',
                'SearchEverything',
                'SearchWithGet',
                'SearchWithPost',
                'VersionReadResource'
            ],
            List: [
                'ListFHIRDatastores',
                'ListFHIRExportJobs',
                'ListFHIRImportJobs',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Healthlake = Healthlake;
//# sourceMappingURL=data:application/json;base64,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