"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Omics = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [omics](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awshealthomics.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Omics extends shared_1.PolicyStatement {
    /**
     * Grants permission to abort multipart read set uploads
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_AbortMultipartReadSetUpload.html
     */
    toAbortMultipartReadSetUpload() {
        return this.to('AbortMultipartReadSetUpload');
    }
    /**
     * Grants permission to accept a share
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_AcceptShare.html
     */
    toAcceptShare() {
        return this.to('AcceptShare');
    }
    /**
     * Grants permission to batch delete Read Sets in the given Sequence Store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_BatchDeleteReadSet.html
     */
    toBatchDeleteReadSet() {
        return this.to('BatchDeleteReadSet');
    }
    /**
     * Grants permission to cancel an Annotation Import Job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_CancelAnnotationImportJob.html
     */
    toCancelAnnotationImportJob() {
        return this.to('CancelAnnotationImportJob');
    }
    /**
     * Grants permission to cancel a workflow run and stop all workflow tasks
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_CancelRun.html
     */
    toCancelRun() {
        return this.to('CancelRun');
    }
    /**
     * Grants permission to cancel a Variant Import Job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_CancelVariantImportJob.html
     */
    toCancelVariantImportJob() {
        return this.to('CancelVariantImportJob');
    }
    /**
     * Grants permission to complete a multipart read set upload
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_CompleteMultipartReadSetUpload.html
     */
    toCompleteMultipartReadSetUpload() {
        return this.to('CompleteMultipartReadSetUpload');
    }
    /**
     * Grants permission to create an Annotation Store
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_CreateAnnotationStore.html
     */
    toCreateAnnotationStore() {
        return this.to('CreateAnnotationStore');
    }
    /**
     * Grants permission to create a Version in an Annotation Store
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_CreateAnnotationStoreVersion.html
     */
    toCreateAnnotationStoreVersion() {
        return this.to('CreateAnnotationStoreVersion');
    }
    /**
     * Grants permission to create a multipart read set upload
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_CreateMultipartReadSetUpload.html
     */
    toCreateMultipartReadSetUpload() {
        return this.to('CreateMultipartReadSetUpload');
    }
    /**
     * Grants permission to create a Reference Store
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_CreateReferenceStore.html
     */
    toCreateReferenceStore() {
        return this.to('CreateReferenceStore');
    }
    /**
     * Grants permission to create a new workflow run cache
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_CreateRunCache.html
     */
    toCreateRunCache() {
        return this.to('CreateRunCache');
    }
    /**
     * Grants permission to create a new workflow run group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_CreateRunGroup.html
     */
    toCreateRunGroup() {
        return this.to('CreateRunGroup');
    }
    /**
     * Grants permission to create a Sequence Store
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_CreateSequenceStore.html
     */
    toCreateSequenceStore() {
        return this.to('CreateSequenceStore');
    }
    /**
     * Grants permission to create a share
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_CreateShare.html
     */
    toCreateShare() {
        return this.to('CreateShare');
    }
    /**
     * Grants permission to create a Variant Store
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_CreateVariantStore.html
     */
    toCreateVariantStore() {
        return this.to('CreateVariantStore');
    }
    /**
     * Grants permission to create a new workflow with a workflow definition and template of workflow parameters
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_CreateWorkflow.html
     */
    toCreateWorkflow() {
        return this.to('CreateWorkflow');
    }
    /**
     * Grants permission to create a new workflow version with a workflow definition and template of workflow parameters
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_CreateWorkflowVersion.html
     */
    toCreateWorkflowVersion() {
        return this.to('CreateWorkflowVersion');
    }
    /**
     * Grants permission to delete an Annotation Store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_DeleteAnnotationStore.html
     */
    toDeleteAnnotationStore() {
        return this.to('DeleteAnnotationStore');
    }
    /**
     * Grants permission to delete Versions in an Annotation Store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_DeleteAnnotationStoreVersions.html
     */
    toDeleteAnnotationStoreVersions() {
        return this.to('DeleteAnnotationStoreVersions');
    }
    /**
     * Grants permission to delete a Reference in the given Reference Store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_DeleteReference.html
     */
    toDeleteReference() {
        return this.to('DeleteReference');
    }
    /**
     * Grants permission to delete a Reference Store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_DeleteReferenceStore.html
     */
    toDeleteReferenceStore() {
        return this.to('DeleteReferenceStore');
    }
    /**
     * Grants permission to delete a workflow run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_DeleteRun.html
     */
    toDeleteRun() {
        return this.to('DeleteRun');
    }
    /**
     * Grants permission to delete a workflow run cache
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_DeleteRunCache.html
     */
    toDeleteRunCache() {
        return this.to('DeleteRunCache');
    }
    /**
     * Grants permission to delete a workflow run group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_DeleteRunGroup.html
     */
    toDeleteRunGroup() {
        return this.to('DeleteRunGroup');
    }
    /**
     * Grants permission to delete an access policy on a given store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_DeleteS3AccessPolicy.html
     */
    toDeleteS3AccessPolicy() {
        return this.to('DeleteS3AccessPolicy');
    }
    /**
     * Grants permission to delete a Sequence Store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_DeleteSequenceStore.html
     */
    toDeleteSequenceStore() {
        return this.to('DeleteSequenceStore');
    }
    /**
     * Grants permission to delete a share
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_DeleteShare.html
     */
    toDeleteShare() {
        return this.to('DeleteShare');
    }
    /**
     * Grants permission to delete a Variant Store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_DeleteVariantStore.html
     */
    toDeleteVariantStore() {
        return this.to('DeleteVariantStore');
    }
    /**
     * Grants permission to delete a workflow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_DeleteWorkflow.html
     */
    toDeleteWorkflow() {
        return this.to('DeleteWorkflow');
    }
    /**
     * Grants permission to delete a workflow version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_DeleteWorkflowVersion.html
     */
    toDeleteWorkflowVersion() {
        return this.to('DeleteWorkflowVersion');
    }
    /**
     * Grants permission to get the status of an Annotation Import Job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_GetAnnotationImportJob.html
     */
    toGetAnnotationImportJob() {
        return this.to('GetAnnotationImportJob');
    }
    /**
     * Grants permission to get detailed information about an Annotation Store
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_GetAnnotationStore.html
     */
    toGetAnnotationStore() {
        return this.to('GetAnnotationStore');
    }
    /**
     * Grants permission to get detailed information about a version in an Annotation Store
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_GetAnnotationStoreVersion.html
     */
    toGetAnnotationStoreVersion() {
        return this.to('GetAnnotationStoreVersion');
    }
    /**
     * Grants permission to get a Read Set in the given Sequence Store
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_GetReadSet.html
     */
    toGetReadSet() {
        return this.to('GetReadSet');
    }
    /**
     * Grants permission to get details about a Read Set activation job for the given Sequence Store
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_GetReadSetActivationJob.html
     */
    toGetReadSetActivationJob() {
        return this.to('GetReadSetActivationJob');
    }
    /**
     * Grants permission to get details about a Read Set export job for the given Sequence Store
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_GetReadSetExportJob.html
     */
    toGetReadSetExportJob() {
        return this.to('GetReadSetExportJob');
    }
    /**
     * Grants permission to get details about a Read Set import job for the given Sequence Store
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_GetReadSetImportJob.html
     */
    toGetReadSetImportJob() {
        return this.to('GetReadSetImportJob');
    }
    /**
     * Grants permission to get details about a Read Set in the given Sequence Store
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_GetReadSetMetadata.html
     */
    toGetReadSetMetadata() {
        return this.to('GetReadSetMetadata');
    }
    /**
     * Grants permission to get a Reference in the given Reference Store
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_GetReference.html
     */
    toGetReference() {
        return this.to('GetReference');
    }
    /**
     * Grants permission to get details about a Reference import job for the given Reference Store
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_GetReferenceImportJob.html
     */
    toGetReferenceImportJob() {
        return this.to('GetReferenceImportJob');
    }
    /**
     * Grants permission to get details about a Reference in the given Reference Store
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_GetReferenceMetadata.html
     */
    toGetReferenceMetadata() {
        return this.to('GetReferenceMetadata');
    }
    /**
     * Grants permission to get details about a Reference Store
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_GetReferenceStore.html
     */
    toGetReferenceStore() {
        return this.to('GetReferenceStore');
    }
    /**
     * Grants permission to retrieve workflow run details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_GetRun.html
     */
    toGetRun() {
        return this.to('GetRun');
    }
    /**
     * Grants permission to retrieve workflow run cache details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_GetRunCache.html
     */
    toGetRunCache() {
        return this.to('GetRunCache');
    }
    /**
     * Grants permission to retrieve workflow run group details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_GetRunGroup.html
     */
    toGetRunGroup() {
        return this.to('GetRunGroup');
    }
    /**
     * Grants permission to retrieve workflow task details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_GetRunTask.html
     */
    toGetRunTask() {
        return this.to('GetRunTask');
    }
    /**
     * Grants permission to get details about an access policy on a given store
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_GetS3AccessPolicy.html
     */
    toGetS3AccessPolicy() {
        return this.to('GetS3AccessPolicy');
    }
    /**
     * Grants permission to get details about a Sequence Store
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_GetSequenceStore.html
     */
    toGetSequenceStore() {
        return this.to('GetSequenceStore');
    }
    /**
     * Grants permission to get detailed information about a Share
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_GetShare.html
     */
    toGetShare() {
        return this.to('GetShare');
    }
    /**
     * Grants permission to get the status of a Variant Import Job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_GetVariantImportJob.html
     */
    toGetVariantImportJob() {
        return this.to('GetVariantImportJob');
    }
    /**
     * Grants permission to get detailed information about a Variant Store
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_GetVariantStore.html
     */
    toGetVariantStore() {
        return this.to('GetVariantStore');
    }
    /**
     * Grants permission to retrieve workflow details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_GetWorkflow.html
     */
    toGetWorkflow() {
        return this.to('GetWorkflow');
    }
    /**
     * Grants permission to retrieve workflow version details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_GetWorkflowVersion.html
     */
    toGetWorkflowVersion() {
        return this.to('GetWorkflowVersion');
    }
    /**
     * Grants permission to get a list of Annotation Import Jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_ListAnnotationImportJobs.html
     */
    toListAnnotationImportJobs() {
        return this.to('ListAnnotationImportJobs');
    }
    /**
     * Grants permission to retrieve a list of information about Versions in an Annotation Store
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_ListAnnotationStoreVersions.html
     */
    toListAnnotationStoreVersions() {
        return this.to('ListAnnotationStoreVersions');
    }
    /**
     * Grants permission to retrieve a list of information about Annotation Stores
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_ListAnnotationStores.html
     */
    toListAnnotationStores() {
        return this.to('ListAnnotationStores');
    }
    /**
     * Grants permission to list multipart read set uploads
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_ListMultipartReadSetUploads.html
     */
    toListMultipartReadSetUploads() {
        return this.to('ListMultipartReadSetUploads');
    }
    /**
     * Grants permission to list Read Set activation jobs for the given Sequence Store
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_ListReadSetActivationJobs.html
     */
    toListReadSetActivationJobs() {
        return this.to('ListReadSetActivationJobs');
    }
    /**
     * Grants permission to list Read Set export jobs for the given Sequence Store
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_ListReadSetExportJobs.html
     */
    toListReadSetExportJobs() {
        return this.to('ListReadSetExportJobs');
    }
    /**
     * Grants permission to list Read Set import jobs for the given Sequence Store
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_ListReadSetImportJobs.html
     */
    toListReadSetImportJobs() {
        return this.to('ListReadSetImportJobs');
    }
    /**
     * Grants permission to list read set upload parts
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_ListReadSetUploadParts.html
     */
    toListReadSetUploadParts() {
        return this.to('ListReadSetUploadParts');
    }
    /**
     * Grants permission to list Read Sets in the given Sequence Store
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_ListReadSets.html
     */
    toListReadSets() {
        return this.to('ListReadSets');
    }
    /**
     * Grants permission to list Reference import jobs for the given Reference Store
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_ListReferenceImportJobs.html
     */
    toListReferenceImportJobs() {
        return this.to('ListReferenceImportJobs');
    }
    /**
     * Grants permission to list Reference Stores
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_ListReferenceStores.html
     */
    toListReferenceStores() {
        return this.to('ListReferenceStores');
    }
    /**
     * Grants permission to list References in the given Reference Store
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_ListReferences.html
     */
    toListReferences() {
        return this.to('ListReferences');
    }
    /**
     * Grants permission to retrieve a list of workflow run caches
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_ListRunCaches.html
     */
    toListRunCaches() {
        return this.to('ListRunCaches');
    }
    /**
     * Grants permission to retrieve a list of workflow run groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_ListRunGroups.html
     */
    toListRunGroups() {
        return this.to('ListRunGroups');
    }
    /**
     * Grants permission to retrieve a list of tasks for a workflow run
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_ListRunTasks.html
     */
    toListRunTasks() {
        return this.to('ListRunTasks');
    }
    /**
     * Grants permission to retrieve a list of workflow runs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_ListRuns.html
     */
    toListRuns() {
        return this.to('ListRuns');
    }
    /**
     * Grants permission to list Sequence Stores
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_ListSequenceStores.html
     */
    toListSequenceStores() {
        return this.to('ListSequenceStores');
    }
    /**
     * Grants permission to retrieve a list of information about shares
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_ListShares.html
     */
    toListShares() {
        return this.to('ListShares');
    }
    /**
     * Grants permission to retrieve a list of resource AWS tags
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to get a list of Variant Import Jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_ListVariantImportJobs.html
     */
    toListVariantImportJobs() {
        return this.to('ListVariantImportJobs');
    }
    /**
     * Grants permission to retrieve a list of metadata for Variant Stores
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_ListVariantStores.html
     */
    toListVariantStores() {
        return this.to('ListVariantStores');
    }
    /**
     * Grants permission to retrieve a list of available versions for a workflow
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_ListWorkflowVersions.html
     */
    toListWorkflowVersions() {
        return this.to('ListWorkflowVersions');
    }
    /**
     * Grants permission to retrieve a list of available workflows
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_ListWorkflows.html
     */
    toListWorkflows() {
        return this.to('ListWorkflows');
    }
    /**
     * Grants permission to put an access policy on a given store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_PutS3AccessPolicy.html
     */
    toPutS3AccessPolicy() {
        return this.to('PutS3AccessPolicy');
    }
    /**
     * Grants permission to import a list of Annotation files to an Annotation Store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_StartAnnotationImportJob.html
     */
    toStartAnnotationImportJob() {
        return this.to('StartAnnotationImportJob');
    }
    /**
     * Grants permission to start a Read Set activation job from the given Sequence Store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_StartReadSetActivationJob.html
     */
    toStartReadSetActivationJob() {
        return this.to('StartReadSetActivationJob');
    }
    /**
     * Grants permission to start a Read Set export job from the given Sequence Store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_StartReadSetExportJob.html
     */
    toStartReadSetExportJob() {
        return this.to('StartReadSetExportJob');
    }
    /**
     * Grants permission to start a Read Set import job into the given Sequence Store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_StartReadSetImportJob.html
     */
    toStartReadSetImportJob() {
        return this.to('StartReadSetImportJob');
    }
    /**
     * Grants permission to start a Reference import job into the given Reference Store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_StartReferenceImportJob.html
     */
    toStartReferenceImportJob() {
        return this.to('StartReferenceImportJob');
    }
    /**
     * Grants permission to start a workflow run
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_StartRun.html
     */
    toStartRun() {
        return this.to('StartRun');
    }
    /**
     * Grants permission to import a list of variant files to an Variant Store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_StartVariantImportJob.html
     */
    toStartVariantImportJob() {
        return this.to('StartVariantImportJob');
    }
    /**
     * Grants permission to add AWS tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove resource AWS tags
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update information about the Annotation Store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_UpdateAnnotationStore.html
     */
    toUpdateAnnotationStore() {
        return this.to('UpdateAnnotationStore');
    }
    /**
     * Grants permission to update information about the Version in an Annotation Store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_UpdateAnnotationStoreVersion.html
     */
    toUpdateAnnotationStoreVersion() {
        return this.to('UpdateAnnotationStoreVersion');
    }
    /**
     * Grants permission to update a workflow run cache
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_UpdateRunCache.html
     */
    toUpdateRunCache() {
        return this.to('UpdateRunCache');
    }
    /**
     * Grants permission to update a workflow run group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_UpdateRunGroup.html
     */
    toUpdateRunGroup() {
        return this.to('UpdateRunGroup');
    }
    /**
     * Grants permission to update details about a Sequence Store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_UpdateSequenceStore.html
     */
    toUpdateSequenceStore() {
        return this.to('UpdateSequenceStore');
    }
    /**
     * Grants permission to update metadata about the Variant Store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_UpdateVariantStore.html
     */
    toUpdateVariantStore() {
        return this.to('UpdateVariantStore');
    }
    /**
     * Grants permission to update workflow details
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_UpdateWorkflow.html
     */
    toUpdateWorkflow() {
        return this.to('UpdateWorkflow');
    }
    /**
     * Grants permission to update workflow version details
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_UpdateWorkflowVersion.html
     */
    toUpdateWorkflowVersion() {
        return this.to('UpdateWorkflowVersion');
    }
    /**
     * Grants permission to upload read set parts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_UploadReadSetPart.html
     */
    toUploadReadSetPart() {
        return this.to('UploadReadSetPart');
    }
    /**
     * Adds a resource of type AnnotationStore to the statement
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_AnnotationStoreItem.html
     *
     * @param annotationStoreName - Identifier for the annotationStoreName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAnnotationStore(annotationStoreName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:omics:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:annotationStore/${annotationStoreName}`);
    }
    /**
     * Adds a resource of type AnnotationStoreVersion to the statement
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_AnnotationStoreVersionItem.html
     *
     * @param annotationStoreName - Identifier for the annotationStoreName.
     * @param annotationStoreVersionName - Identifier for the annotationStoreVersionName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAnnotationStoreVersion(annotationStoreName, annotationStoreVersionName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:omics:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:annotationStore/${annotationStoreName}/version/${annotationStoreVersionName}`);
    }
    /**
     * Adds a resource of type readSet to the statement
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_ReadSetFiles.html
     *
     * @param sequenceStoreId - Identifier for the sequenceStoreId.
     * @param readSetId - Identifier for the readSetId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onReadSet(sequenceStoreId, readSetId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:omics:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:sequenceStore/${sequenceStoreId}/readSet/${readSetId}`);
    }
    /**
     * Adds a resource of type reference to the statement
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_ReferenceFiles.html
     *
     * @param referenceStoreId - Identifier for the referenceStoreId.
     * @param referenceId - Identifier for the referenceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onReference(referenceStoreId, referenceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:omics:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:referenceStore/${referenceStoreId}/reference/${referenceId}`);
    }
    /**
     * Adds a resource of type referenceStore to the statement
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_ReferenceStoreDetail.html
     *
     * @param referenceStoreId - Identifier for the referenceStoreId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onReferenceStore(referenceStoreId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:omics:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:referenceStore/${referenceStoreId}`);
    }
    /**
     * Adds a resource of type run to the statement
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_RunListItem.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRun(id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:omics:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:run/${id}`);
    }
    /**
     * Adds a resource of type runCache to the statement
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_RunCacheListItem.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRunCache(id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:omics:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:runCache/${id}`);
    }
    /**
     * Adds a resource of type runGroup to the statement
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_RunGroupListItem.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRunGroup(id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:omics:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:runGroup/${id}`);
    }
    /**
     * Adds a resource of type sequenceStore to the statement
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_SequenceStoreDetail.html
     *
     * @param sequenceStoreId - Identifier for the sequenceStoreId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSequenceStore(sequenceStoreId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:omics:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:sequenceStore/${sequenceStoreId}`);
    }
    /**
     * Adds a resource of type TaggingResource to the statement
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_TagResource.html
     *
     * @param tagKey - Identifier for the tagKey.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onTaggingResource(tagKey, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:omics:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:tag/${tagKey}`);
    }
    /**
     * Adds a resource of type TaskResource to the statement
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_TaskListItem.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onTaskResource(id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:omics:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:task/${id}`);
    }
    /**
     * Adds a resource of type VariantStore to the statement
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_VariantStoreItem.html
     *
     * @param variantStoreName - Identifier for the variantStoreName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVariantStore(variantStoreName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:omics:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:variantStore/${variantStoreName}`);
    }
    /**
     * Adds a resource of type workflow to the statement
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_WorkflowListItem.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorkflow(id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:omics:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workflow/${id}`);
    }
    /**
     * Adds a resource of type WorkflowVersion to the statement
     *
     * https://docs.aws.amazon.com/omics/latest/api/API_WorkflowVersionListItem.html
     *
     * @param id - Identifier for the id.
     * @param versionName - Identifier for the versionName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorkflowVersion(id, versionName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:omics:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workflow/${id}/version/${versionName}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAnnotationStore()
     * - .toCreateAnnotationStoreVersion()
     * - .toCreateReferenceStore()
     * - .toCreateRunCache()
     * - .toCreateRunGroup()
     * - .toCreateSequenceStore()
     * - .toCreateVariantStore()
     * - .toCreateWorkflow()
     * - .toCreateWorkflowVersion()
     * - .toStartRun()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - AnnotationStore
     * - AnnotationStoreVersion
     * - readSet
     * - reference
     * - referenceStore
     * - run
     * - runCache
     * - runGroup
     * - sequenceStore
     * - VariantStore
     * - workflow
     * - WorkflowVersion
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAnnotationStore()
     * - .toCreateAnnotationStoreVersion()
     * - .toCreateReferenceStore()
     * - .toCreateRunCache()
     * - .toCreateRunGroup()
     * - .toCreateSequenceStore()
     * - .toCreateVariantStore()
     * - .toCreateWorkflow()
     * - .toCreateWorkflowVersion()
     * - .toStartRun()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [omics](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awshealthomics.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'omics';
        this.accessLevelList = {
            Write: [
                'AbortMultipartReadSetUpload',
                'AcceptShare',
                'BatchDeleteReadSet',
                'CancelAnnotationImportJob',
                'CancelRun',
                'CancelVariantImportJob',
                'CompleteMultipartReadSetUpload',
                'CreateAnnotationStore',
                'CreateAnnotationStoreVersion',
                'CreateMultipartReadSetUpload',
                'CreateReferenceStore',
                'CreateRunCache',
                'CreateRunGroup',
                'CreateSequenceStore',
                'CreateShare',
                'CreateVariantStore',
                'CreateWorkflow',
                'CreateWorkflowVersion',
                'DeleteAnnotationStore',
                'DeleteAnnotationStoreVersions',
                'DeleteReference',
                'DeleteReferenceStore',
                'DeleteRun',
                'DeleteRunCache',
                'DeleteRunGroup',
                'DeleteS3AccessPolicy',
                'DeleteSequenceStore',
                'DeleteShare',
                'DeleteVariantStore',
                'DeleteWorkflow',
                'DeleteWorkflowVersion',
                'PutS3AccessPolicy',
                'StartAnnotationImportJob',
                'StartReadSetActivationJob',
                'StartReadSetExportJob',
                'StartReadSetImportJob',
                'StartReferenceImportJob',
                'StartRun',
                'StartVariantImportJob',
                'UpdateAnnotationStore',
                'UpdateAnnotationStoreVersion',
                'UpdateRunCache',
                'UpdateRunGroup',
                'UpdateSequenceStore',
                'UpdateVariantStore',
                'UpdateWorkflow',
                'UpdateWorkflowVersion',
                'UploadReadSetPart'
            ],
            Read: [
                'GetAnnotationImportJob',
                'GetAnnotationStore',
                'GetAnnotationStoreVersion',
                'GetReadSet',
                'GetReadSetActivationJob',
                'GetReadSetExportJob',
                'GetReadSetImportJob',
                'GetReadSetMetadata',
                'GetReference',
                'GetReferenceImportJob',
                'GetReferenceMetadata',
                'GetReferenceStore',
                'GetRun',
                'GetRunCache',
                'GetRunGroup',
                'GetRunTask',
                'GetS3AccessPolicy',
                'GetSequenceStore',
                'GetShare',
                'GetVariantImportJob',
                'GetVariantStore',
                'GetWorkflow',
                'GetWorkflowVersion'
            ],
            List: [
                'ListAnnotationImportJobs',
                'ListAnnotationStoreVersions',
                'ListAnnotationStores',
                'ListMultipartReadSetUploads',
                'ListReadSetActivationJobs',
                'ListReadSetExportJobs',
                'ListReadSetImportJobs',
                'ListReadSetUploadParts',
                'ListReadSets',
                'ListReferenceImportJobs',
                'ListReferenceStores',
                'ListReferences',
                'ListRunCaches',
                'ListRunGroups',
                'ListRunTasks',
                'ListRuns',
                'ListSequenceStores',
                'ListShares',
                'ListTagsForResource',
                'ListVariantImportJobs',
                'ListVariantStores',
                'ListWorkflowVersions',
                'ListWorkflows'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Omics = Omics;
//# sourceMappingURL=data:application/json;base64,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