"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Rolesanywhere = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [rolesanywhere](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsidentityandaccessmanagementrolesanywhere.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Rolesanywhere extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a profile
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_CreateProfile.html
     */
    toCreateProfile() {
        return this.to('CreateProfile');
    }
    /**
     * Grants permission to create a trust anchor
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_CreateTrustAnchor.html
     */
    toCreateTrustAnchor() {
        return this.to('CreateTrustAnchor');
    }
    /**
     * Grants permission to delete a mapping rule from a profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_DeleteAttributeMapping.html
     */
    toDeleteAttributeMapping() {
        return this.to('DeleteAttributeMapping');
    }
    /**
     * Grants permission to delete a certificate revocation list (crl)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_DeleteCrl.html
     */
    toDeleteCrl() {
        return this.to('DeleteCrl');
    }
    /**
     * Grants permission to delete a profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_DeleteProfile.html
     */
    toDeleteProfile() {
        return this.to('DeleteProfile');
    }
    /**
     * Grants permission to delete a trust anchor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_DeleteTrustAnchor.html
     */
    toDeleteTrustAnchor() {
        return this.to('DeleteTrustAnchor');
    }
    /**
     * Grants permission to disable a certificate revocation list (crl)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_DisableCrl.html
     */
    toDisableCrl() {
        return this.to('DisableCrl');
    }
    /**
     * Grants permission to disable a profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_DisableProfile.html
     */
    toDisableProfile() {
        return this.to('DisableProfile');
    }
    /**
     * Grants permission to disable a trust anchor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_DisableTrustAnchor.html
     */
    toDisableTrustAnchor() {
        return this.to('DisableTrustAnchor');
    }
    /**
     * Grants permission to enable a certificate revocation list (crl)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_EnableCrl.html
     */
    toEnableCrl() {
        return this.to('EnableCrl');
    }
    /**
     * Grants permission to enable a profile
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_EnableProfile.html
     */
    toEnableProfile() {
        return this.to('EnableProfile');
    }
    /**
     * Grants permission to enable a trust anchor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_EnableTrustAnchor.html
     */
    toEnableTrustAnchor() {
        return this.to('EnableTrustAnchor');
    }
    /**
     * Grants permission to get a certificate revocation list (crl)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_GetCrl.html
     */
    toGetCrl() {
        return this.to('GetCrl');
    }
    /**
     * Grants permission to get a profile
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_GetProfile.html
     */
    toGetProfile() {
        return this.to('GetProfile');
    }
    /**
     * Grants permission to get a subject
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_GetSubject.html
     */
    toGetSubject() {
        return this.to('GetSubject');
    }
    /**
     * Grants permission to get a trust anchor
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_GetTrustAnchor.html
     */
    toGetTrustAnchor() {
        return this.to('GetTrustAnchor');
    }
    /**
     * Grants permission to import a certificate revocation list (crl)
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_ImportCrl.html
     */
    toImportCrl() {
        return this.to('ImportCrl');
    }
    /**
     * Grants permission to list certificate revocation lists (crls)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_ListCrls.html
     */
    toListCrls() {
        return this.to('ListCrls');
    }
    /**
     * Grants permission to list profiles
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_ListProfiles.html
     */
    toListProfiles() {
        return this.to('ListProfiles');
    }
    /**
     * Grants permission to list subjects
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_ListSubjects.html
     */
    toListSubjects() {
        return this.to('ListSubjects');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list trust anchors
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_ListTrustAnchors.html
     */
    toListTrustAnchors() {
        return this.to('ListTrustAnchors');
    }
    /**
     * Grants permission to put a mapping rule into a profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_PutAttributeMapping.html
     */
    toPutAttributeMapping() {
        return this.to('PutAttributeMapping');
    }
    /**
     * Grants permission to attach notification settings to a trust anchor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_PutNotificationSettings.html
     */
    toPutNotificationSettings() {
        return this.to('PutNotificationSettings');
    }
    /**
     * Grants permission to reset custom notification settings to IAM Roles Anywhere defined default state
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_ResetNotificationSettings.html
     */
    toResetNotificationSettings() {
        return this.to('ResetNotificationSettings');
    }
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a certificate revocation list (crl)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_UpdateCrl.html
     */
    toUpdateCrl() {
        return this.to('UpdateCrl');
    }
    /**
     * Grants permission to update a profile
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_UpdateProfile.html
     */
    toUpdateProfile() {
        return this.to('UpdateProfile');
    }
    /**
     * Grants permission to update a trust anchor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/APIReference/API_UpdateTrustAnchor.html
     */
    toUpdateTrustAnchor() {
        return this.to('UpdateTrustAnchor');
    }
    /**
     * Adds a resource of type trust-anchor to the statement
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/userguide/introduction.html#first-time-user
     *
     * @param trustAnchorId - Identifier for the trustAnchorId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTrustAnchor(trustAnchorId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rolesanywhere:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:trust-anchor/${trustAnchorId}`);
    }
    /**
     * Adds a resource of type profile to the statement
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/userguide/introduction.html#first-time-user
     *
     * @param profileId - Identifier for the profileId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProfile(profileId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rolesanywhere:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:profile/${profileId}`);
    }
    /**
     * Adds a resource of type subject to the statement
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/userguide/introduction.html#first-time-user
     *
     * @param subjectId - Identifier for the subjectId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSubject(subjectId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rolesanywhere:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:subject/${subjectId}`);
    }
    /**
     * Adds a resource of type crl to the statement
     *
     * https://docs.aws.amazon.com/rolesanywhere/latest/userguide/introduction.html#first-time-user
     *
     * @param crlId - Identifier for the crlId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCrl(crlId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:rolesanywhere:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:crl/${crlId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateProfile()
     * - .toCreateTrustAnchor()
     * - .toImportCrl()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - trust-anchor
     * - profile
     * - subject
     * - crl
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateProfile()
     * - .toCreateTrustAnchor()
     * - .toImportCrl()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [rolesanywhere](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsidentityandaccessmanagementrolesanywhere.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'rolesanywhere';
        this.accessLevelList = {
            Write: [
                'CreateProfile',
                'CreateTrustAnchor',
                'DeleteAttributeMapping',
                'DeleteCrl',
                'DeleteProfile',
                'DeleteTrustAnchor',
                'DisableCrl',
                'DisableProfile',
                'DisableTrustAnchor',
                'EnableCrl',
                'EnableProfile',
                'EnableTrustAnchor',
                'ImportCrl',
                'PutAttributeMapping',
                'PutNotificationSettings',
                'ResetNotificationSettings',
                'UpdateCrl',
                'UpdateProfile',
                'UpdateTrustAnchor'
            ],
            Read: [
                'GetCrl',
                'GetProfile',
                'GetSubject',
                'GetTrustAnchor'
            ],
            List: [
                'ListCrls',
                'ListProfiles',
                'ListSubjects',
                'ListTagsForResource',
                'ListTrustAnchors'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Rolesanywhere = Rolesanywhere;
//# sourceMappingURL=data:application/json;base64,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