"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Iotwireless = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [iotwireless](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotwireless.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Iotwireless extends shared_1.PolicyStatement {
    /**
     * Grants permission to link partner accounts with AWS account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_AssociateAwsAccountWithPartnerAccount.html
     */
    toAssociateAwsAccountWithPartnerAccount() {
        return this.to('AssociateAwsAccountWithPartnerAccount');
    }
    /**
     * Grants permission to associate the MulticastGroup with FuotaTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_AssociateMulticastGroupWithFuotaTask.html
     */
    toAssociateMulticastGroupWithFuotaTask() {
        return this.to('AssociateMulticastGroupWithFuotaTask');
    }
    /**
     * Grants permission to associate the wireless device with FuotaTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_AssociateWirelessDeviceWithFuotaTask.html
     */
    toAssociateWirelessDeviceWithFuotaTask() {
        return this.to('AssociateWirelessDeviceWithFuotaTask');
    }
    /**
     * Grants permission to associate the WirelessDevice with MulticastGroup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_AssociateWirelessDeviceWithMulticastGroup.html
     */
    toAssociateWirelessDeviceWithMulticastGroup() {
        return this.to('AssociateWirelessDeviceWithMulticastGroup');
    }
    /**
     * Grants permission to associate the wireless device with AWS IoT thing for a given wirelessDeviceId
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iot:DescribeThing
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_AssociateWirelessDeviceWithThing.html
     */
    toAssociateWirelessDeviceWithThing() {
        return this.to('AssociateWirelessDeviceWithThing');
    }
    /**
     * Grants permission to associate a WirelessGateway with the IoT Core Identity certificate
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_AssociateWirelessGatewayWithCertificate.html
     */
    toAssociateWirelessGatewayWithCertificate() {
        return this.to('AssociateWirelessGatewayWithCertificate');
    }
    /**
     * Grants permission to associate the wireless gateway with AWS IoT thing for a given wirelessGatewayId
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iot:DescribeThing
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_AssociateWirelessGatewayWithThing.html
     */
    toAssociateWirelessGatewayWithThing() {
        return this.to('AssociateWirelessGatewayWithThing');
    }
    /**
     * Grants permission to cancel the MulticastGroup session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_CancelMulticastGroupSession.html
     */
    toCancelMulticastGroupSession() {
        return this.to('CancelMulticastGroupSession');
    }
    /**
     * Grants permission to create a Destination resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_CreateDestination.html
     */
    toCreateDestination() {
        return this.to('CreateDestination');
    }
    /**
     * Grants permission to create a DeviceProfile resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_CreateDeviceProfile.html
     */
    toCreateDeviceProfile() {
        return this.to('CreateDeviceProfile');
    }
    /**
     * Grants permission to create a FuotaTask resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_CreateFuotaTask.html
     */
    toCreateFuotaTask() {
        return this.to('CreateFuotaTask');
    }
    /**
     * Grants permission to create a MulticastGroup resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_CreateMulticastGroup.html
     */
    toCreateMulticastGroup() {
        return this.to('CreateMulticastGroup');
    }
    /**
     * Grants permission to create a NetworkAnalyzerConfiguration resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_CreateNetworkAnalyzerConfiguration.html
     */
    toCreateNetworkAnalyzerConfiguration() {
        return this.to('CreateNetworkAnalyzerConfiguration');
    }
    /**
     * Grants permission to create a ServiceProfile resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_CreateServiceProfile.html
     */
    toCreateServiceProfile() {
        return this.to('CreateServiceProfile');
    }
    /**
     * Grants permission to create a WirelessDevice resource with given Destination
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_CreateWirelessDevice.html
     */
    toCreateWirelessDevice() {
        return this.to('CreateWirelessDevice');
    }
    /**
     * Grants permission to create a WirelessGateway resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_CreateWirelessGateway.html
     */
    toCreateWirelessGateway() {
        return this.to('CreateWirelessGateway');
    }
    /**
     * Grants permission to create a task for a given WirelessGateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_CreateWirelessGatewayTask.html
     */
    toCreateWirelessGatewayTask() {
        return this.to('CreateWirelessGatewayTask');
    }
    /**
     * Grants permission to create a WirelessGateway task definition
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_CreateWirelessGatewayTaskDefinition.html
     */
    toCreateWirelessGatewayTaskDefinition() {
        return this.to('CreateWirelessGatewayTaskDefinition');
    }
    /**
     * Grants permission to delete a Destination
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_DeleteDestination.html
     */
    toDeleteDestination() {
        return this.to('DeleteDestination');
    }
    /**
     * Grants permission to delete a DeviceProfile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_DeleteDeviceProfile.html
     */
    toDeleteDeviceProfile() {
        return this.to('DeleteDeviceProfile');
    }
    /**
     * Grants permission to delete the FuotaTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_DeleteFuotaTask.html
     */
    toDeleteFuotaTask() {
        return this.to('DeleteFuotaTask');
    }
    /**
     * Grants permission to delete the MulticastGroup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_DeleteMulticastGroup.html
     */
    toDeleteMulticastGroup() {
        return this.to('DeleteMulticastGroup');
    }
    /**
     * Grants permission to delete the NetworkAnalyzerConfiguration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_DeleteNetworkAnalyzerConfiguration.html
     */
    toDeleteNetworkAnalyzerConfiguration() {
        return this.to('DeleteNetworkAnalyzerConfiguration');
    }
    /**
     * Grants permission to delete QueuedMessages
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_DeleteQueuedMessages.html
     */
    toDeleteQueuedMessages() {
        return this.to('DeleteQueuedMessages');
    }
    /**
     * Grants permission to delete a ServiceProfile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_DeleteServiceProfile.html
     */
    toDeleteServiceProfile() {
        return this.to('DeleteServiceProfile');
    }
    /**
     * Grants permission to delete a WirelessDevice
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_DeleteWirelessDevice.html
     */
    toDeleteWirelessDevice() {
        return this.to('DeleteWirelessDevice');
    }
    /**
     * Grants permission to delete the wireless device import task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_DeleteWirelessDeviceImportTask.html
     */
    toDeleteWirelessDeviceImportTask() {
        return this.to('DeleteWirelessDeviceImportTask');
    }
    /**
     * Grants permission to delete a WirelessGateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_DeleteWirelessGateway.html
     */
    toDeleteWirelessGateway() {
        return this.to('DeleteWirelessGateway');
    }
    /**
     * Grants permission to delete task for a given WirelessGateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_DeleteWirelessGatewayTask.html
     */
    toDeleteWirelessGatewayTask() {
        return this.to('DeleteWirelessGatewayTask');
    }
    /**
     * Grants permission to delete a WirelessGateway task definition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_DeleteWirelessGatewayTaskDefinition.html
     */
    toDeleteWirelessGatewayTaskDefinition() {
        return this.to('DeleteWirelessGatewayTaskDefinition');
    }
    /**
     * Grants permission to deregister wireless device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_DeregisterWirelessDevice.html
     */
    toDeregisterWirelessDevice() {
        return this.to('DeregisterWirelessDevice');
    }
    /**
     * Grants permission to disassociate an AWS account from a partner account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_DisassociateAwsAccountFromPartnerAccount.html
     */
    toDisassociateAwsAccountFromPartnerAccount() {
        return this.to('DisassociateAwsAccountFromPartnerAccount');
    }
    /**
     * Grants permission to disassociate the MulticastGroup from FuotaTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_DisassociateMulticastGroupFromFuotaTask.html
     */
    toDisassociateMulticastGroupFromFuotaTask() {
        return this.to('DisassociateMulticastGroupFromFuotaTask');
    }
    /**
     * Grants permission to disassociate the wireless device from FuotaTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_DisassociateWirelessDeviceFromFuotaTask.html
     */
    toDisassociateWirelessDeviceFromFuotaTask() {
        return this.to('DisassociateWirelessDeviceFromFuotaTask');
    }
    /**
     * Grants permission to disassociate the wireless device from MulticastGroup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_DisassociateWirelessDeviceFromMulticastGroup.html
     */
    toDisassociateWirelessDeviceFromMulticastGroup() {
        return this.to('DisassociateWirelessDeviceFromMulticastGroup');
    }
    /**
     * Grants permission to disassociate a wireless device from a AWS IoT thing
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iot:DescribeThing
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_DisassociateWirelessDeviceFromThing.html
     */
    toDisassociateWirelessDeviceFromThing() {
        return this.to('DisassociateWirelessDeviceFromThing');
    }
    /**
     * Grants permission to disassociate a WirelessGateway from a IoT Core Identity certificate
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_DisassociateWirelessGatewayFromCertificate.html
     */
    toDisassociateWirelessGatewayFromCertificate() {
        return this.to('DisassociateWirelessGatewayFromCertificate');
    }
    /**
     * Grants permission to disassociate a WirelessGateway from a IoT Core thing
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iot:DescribeThing
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_DisassociateWirelessGatewayFromThing.html
     */
    toDisassociateWirelessGatewayFromThing() {
        return this.to('DisassociateWirelessGatewayFromThing');
    }
    /**
     * Grants permission to get the Destination
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_GetDestination.html
     */
    toGetDestination() {
        return this.to('GetDestination');
    }
    /**
     * Grants permission to get the DeviceProfile
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_GetDeviceProfile.html
     */
    toGetDeviceProfile() {
        return this.to('GetDeviceProfile');
    }
    /**
     * Grants permission to get event configuration by resource types
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_GetEventConfigurationByResourceTypes.html
     */
    toGetEventConfigurationByResourceTypes() {
        return this.to('GetEventConfigurationByResourceTypes');
    }
    /**
     * Grants permission to get the FuotaTask
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_GetFuotaTask.html
     */
    toGetFuotaTask() {
        return this.to('GetFuotaTask');
    }
    /**
     * Grants permission to get log levels by resource types
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_GetLogLevelsByResourceTypes.html
     */
    toGetLogLevelsByResourceTypes() {
        return this.to('GetLogLevelsByResourceTypes');
    }
    /**
     * Grants permission to get metric configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_GetMetricConfiguration.html
     */
    toGetMetricConfiguration() {
        return this.to('GetMetricConfiguration');
    }
    /**
     * Grants permission to get metrics
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_GetMetrics.html
     */
    toGetMetrics() {
        return this.to('GetMetrics');
    }
    /**
     * Grants permission to get the MulticastGroup
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_GetMulticastGroup.html
     */
    toGetMulticastGroup() {
        return this.to('GetMulticastGroup');
    }
    /**
     * Grants permission to get the MulticastGroup session
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_GetMulticastGroupSession.html
     */
    toGetMulticastGroupSession() {
        return this.to('GetMulticastGroupSession');
    }
    /**
     * Grants permission to get the NetworkAnalyzerConfiguration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_GetNetworkAnalyzerConfiguration.html
     */
    toGetNetworkAnalyzerConfiguration() {
        return this.to('GetNetworkAnalyzerConfiguration');
    }
    /**
     * Grants permission to get the associated PartnerAccount
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_GetPartnerAccount.html
     */
    toGetPartnerAccount() {
        return this.to('GetPartnerAccount');
    }
    /**
     * Grants permission to get position for a given resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_GetPosition.html
     */
    toGetPosition() {
        return this.to('GetPosition');
    }
    /**
     * Grants permission to get position configuration for a given resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_GetPositionConfiguration.html
     */
    toGetPositionConfiguration() {
        return this.to('GetPositionConfiguration');
    }
    /**
     * Grants permission to get position estimate
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_GetPositionEstimate.html
     */
    toGetPositionEstimate() {
        return this.to('GetPositionEstimate');
    }
    /**
     * Grants permission to get an event configuration for an identifier
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_GetResourceEventConfiguration.html
     */
    toGetResourceEventConfiguration() {
        return this.to('GetResourceEventConfiguration');
    }
    /**
     * Grants permission to get resource log level
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_GetResourceLogLevel.html
     */
    toGetResourceLogLevel() {
        return this.to('GetResourceLogLevel');
    }
    /**
     * Grants permission to get position for a given resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_GetResourcePosition.html
     */
    toGetResourcePosition() {
        return this.to('GetResourcePosition');
    }
    /**
     * Grants permission to retrieve the customer account specific endpoint for CUPS protocol connection or LoRaWAN Network Server (LNS) protocol connection, and optionally server trust certificate in PEM format
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_GetServiceEndpoint.html
     */
    toGetServiceEndpoint() {
        return this.to('GetServiceEndpoint');
    }
    /**
     * Grants permission to get the ServiceProfile
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_GetServiceProfile.html
     */
    toGetServiceProfile() {
        return this.to('GetServiceProfile');
    }
    /**
     * Grants permission to get the WirelessDevice
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_GetWirelessDevice.html
     */
    toGetWirelessDevice() {
        return this.to('GetWirelessDevice');
    }
    /**
     * Grants permission to get the wireless device import task
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_GetWirelessDeviceImportTask.html
     */
    toGetWirelessDeviceImportTask() {
        return this.to('GetWirelessDeviceImportTask');
    }
    /**
     * Grants permission to get statistics info for a given WirelessDevice
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_GetWirelessDeviceStatistics.html
     */
    toGetWirelessDeviceStatistics() {
        return this.to('GetWirelessDeviceStatistics');
    }
    /**
     * Grants permission to get the WirelessGateway
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_GetWirelessGateway.html
     */
    toGetWirelessGateway() {
        return this.to('GetWirelessGateway');
    }
    /**
     * Grants permission to get the IoT Core Identity certificate id associated with the WirelessGateway
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_GetWirelessGatewayCertificate.html
     */
    toGetWirelessGatewayCertificate() {
        return this.to('GetWirelessGatewayCertificate');
    }
    /**
     * Grants permission to get Current firmware version and other information for the WirelessGateway
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_GetWirelessGatewayFirmwareInformation.html
     */
    toGetWirelessGatewayFirmwareInformation() {
        return this.to('GetWirelessGatewayFirmwareInformation');
    }
    /**
     * Grants permission to get statistics info for a given WirelessGateway
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_GetWirelessGatewayStatistics.html
     */
    toGetWirelessGatewayStatistics() {
        return this.to('GetWirelessGatewayStatistics');
    }
    /**
     * Grants permission to get the task for a given WirelessGateway
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_GetWirelessGatewayTask.html
     */
    toGetWirelessGatewayTask() {
        return this.to('GetWirelessGatewayTask');
    }
    /**
     * Grants permission to get the given WirelessGateway task definition
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_GetWirelessGatewayTaskDefinition.html
     */
    toGetWirelessGatewayTaskDefinition() {
        return this.to('GetWirelessGatewayTaskDefinition');
    }
    /**
     * Grants permission to list information of available Destinations based on the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_ListDestinations.html
     */
    toListDestinations() {
        return this.to('ListDestinations');
    }
    /**
     * Grants permission to list information of available DeviceProfiles based on the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_ListDeviceProfiles.html
     */
    toListDeviceProfiles() {
        return this.to('ListDeviceProfiles');
    }
    /**
     * Grants permission to list information of devices by wireless device import task based on the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_ListDevicesForWirelessDeviceImportTask.html
     */
    toListDevicesForWirelessDeviceImportTask() {
        return this.to('ListDevicesForWirelessDeviceImportTask');
    }
    /**
     * Grants permission to list information of available event configurations based on the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_ListEventConfigurations.html
     */
    toListEventConfigurations() {
        return this.to('ListEventConfigurations');
    }
    /**
     * Grants permission to list information of available FuotaTasks based on the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_ListFuotaTasks.html
     */
    toListFuotaTasks() {
        return this.to('ListFuotaTasks');
    }
    /**
     * Grants permission to list information of available MulticastGroups based on the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_ListMulticastGroups.html
     */
    toListMulticastGroups() {
        return this.to('ListMulticastGroups');
    }
    /**
     * Grants permission to list information of available MulticastGroups by FuotaTask based on the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_ListMulticastGroupsByFuotaTask.html
     */
    toListMulticastGroupsByFuotaTask() {
        return this.to('ListMulticastGroupsByFuotaTask');
    }
    /**
     * Grants permission to list information of available NetworkAnalyzerConfigurations based on the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_ListNetworkAnalyzerConfigurations.html
     */
    toListNetworkAnalyzerConfigurations() {
        return this.to('ListNetworkAnalyzerConfigurations');
    }
    /**
     * Grants permission to list the available partner accounts
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_ListPartnerAccounts.html
     */
    toListPartnerAccounts() {
        return this.to('ListPartnerAccounts');
    }
    /**
     * Grants permission to list information of available position configurations based on the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_ListPositionConfigurations.html
     */
    toListPositionConfigurations() {
        return this.to('ListPositionConfigurations');
    }
    /**
     * Grants permission to list the Queued Messages
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_ListQueuedMessages.html
     */
    toListQueuedMessages() {
        return this.to('ListQueuedMessages');
    }
    /**
     * Grants permission to list information of available ServiceProfiles based on the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_ListServiceProfiles.html
     */
    toListServiceProfiles() {
        return this.to('ListServiceProfiles');
    }
    /**
     * Grants permission to list all tags for a given resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list wireless device import tasks information of based on the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_ListWirelessDeviceImportTasks.html
     */
    toListWirelessDeviceImportTasks() {
        return this.to('ListWirelessDeviceImportTasks');
    }
    /**
     * Grants permission to list information of available WirelessDevices based on the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_ListWirelessDevices.html
     */
    toListWirelessDevices() {
        return this.to('ListWirelessDevices');
    }
    /**
     * Grants permission to list information of available WirelessGateway task definitions based on the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_ListWirelessGatewayTaskDefinitions.html
     */
    toListWirelessGatewayTaskDefinitions() {
        return this.to('ListWirelessGatewayTaskDefinitions');
    }
    /**
     * Grants permission to list information of available WirelessGateways based on the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_ListWirelessGateways.html
     */
    toListWirelessGateways() {
        return this.to('ListWirelessGateways');
    }
    /**
     * Grants permission to put position configuration for a given resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_PutPositionConfiguration.html
     */
    toPutPositionConfiguration() {
        return this.to('PutPositionConfiguration');
    }
    /**
     * Grants permission to put resource log level
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_PutResourceLogLevel.html
     */
    toPutResourceLogLevel() {
        return this.to('PutResourceLogLevel');
    }
    /**
     * Grants permission to reset all resource log levels
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_ResetAllResourceLogLevels.html
     */
    toResetAllResourceLogLevels() {
        return this.to('ResetAllResourceLogLevels');
    }
    /**
     * Grants permission to reset resource log level
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_ResetResourceLogLevel.html
     */
    toResetResourceLogLevel() {
        return this.to('ResetResourceLogLevel');
    }
    /**
     * Grants permission to send data to the MulticastGroup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_SendDataToMulticastGroup.html
     */
    toSendDataToMulticastGroup() {
        return this.to('SendDataToMulticastGroup');
    }
    /**
     * Grants permission to send the decrypted application data frame to the target device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_SendDataToWirelessDevice.html
     */
    toSendDataToWirelessDevice() {
        return this.to('SendDataToWirelessDevice');
    }
    /**
     * Grants permission to associate the WirelessDevices with MulticastGroup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_StartBulkAssociateWirelessDeviceWithMulticastGroup.html
     */
    toStartBulkAssociateWirelessDeviceWithMulticastGroup() {
        return this.to('StartBulkAssociateWirelessDeviceWithMulticastGroup');
    }
    /**
     * Grants permission to bulk disassociate the WirelessDevices from MulticastGroup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_StartBulkDisassociateWirelessDeviceFromMulticastGroup.html
     */
    toStartBulkDisassociateWirelessDeviceFromMulticastGroup() {
        return this.to('StartBulkDisassociateWirelessDeviceFromMulticastGroup');
    }
    /**
     * Grants permission to start the FuotaTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_StartFuotaTask.html
     */
    toStartFuotaTask() {
        return this.to('StartFuotaTask');
    }
    /**
     * Grants permission to start the MulticastGroup session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_StartMulticastGroupSession.html
     */
    toStartMulticastGroupSession() {
        return this.to('StartMulticastGroupSession');
    }
    /**
     * Grants permission to start NetworkAnalyzer stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/connect-iot-lorawan-network-analyzer-api.html
     */
    toStartNetworkAnalyzerStream() {
        return this.to('StartNetworkAnalyzerStream');
    }
    /**
     * Grants permission to start the single wireless device import task
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_StartSingleWirelessDeviceImportTask.html
     */
    toStartSingleWirelessDeviceImportTask() {
        return this.to('StartSingleWirelessDeviceImportTask');
    }
    /**
     * Grants permission to start the wireless device import task
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_StartWirelessDeviceImportTask.html
     */
    toStartWirelessDeviceImportTask() {
        return this.to('StartWirelessDeviceImportTask');
    }
    /**
     * Grants permission to tag a given resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to simulate a provisioned device to send an uplink data with payload of 'Hello'
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_TestWirelessDevice.html
     */
    toTestWirelessDevice() {
        return this.to('TestWirelessDevice');
    }
    /**
     * Grants permission to remove the given tags from the resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a Destination resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_UpdateDestination.html
     */
    toUpdateDestination() {
        return this.to('UpdateDestination');
    }
    /**
     * Grants permission to update event configuration by resource types
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_UpdateEventConfigurationByResourceTypes.html
     */
    toUpdateEventConfigurationByResourceTypes() {
        return this.to('UpdateEventConfigurationByResourceTypes');
    }
    /**
     * Grants permission to update the FuotaTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_UpdateFuotaTask.html
     */
    toUpdateFuotaTask() {
        return this.to('UpdateFuotaTask');
    }
    /**
     * Grants permission to update log levels by resource types
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_UpdateLogLevelsByResourceTypes.html
     */
    toUpdateLogLevelsByResourceTypes() {
        return this.to('UpdateLogLevelsByResourceTypes');
    }
    /**
     * Grants permission to update metric configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_UpdateMetricConfiguration.html
     */
    toUpdateMetricConfiguration() {
        return this.to('UpdateMetricConfiguration');
    }
    /**
     * Grants permission to update the MulticastGroup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_UpdateMulticastGroup.html
     */
    toUpdateMulticastGroup() {
        return this.to('UpdateMulticastGroup');
    }
    /**
     * Grants permission to update the NetworkAnalyzerConfiguration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_UpdateNetworkAnalyzerConfiguration.html
     */
    toUpdateNetworkAnalyzerConfiguration() {
        return this.to('UpdateNetworkAnalyzerConfiguration');
    }
    /**
     * Grants permission to update a partner account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_UpdatePartnerAccount.html
     */
    toUpdatePartnerAccount() {
        return this.to('UpdatePartnerAccount');
    }
    /**
     * Grants permission to update position for a given resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_UpdatePosition.html
     */
    toUpdatePosition() {
        return this.to('UpdatePosition');
    }
    /**
     * Grants permission to update an event configuration for an identifier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_UpdateResourceEventConfiguration.html
     */
    toUpdateResourceEventConfiguration() {
        return this.to('UpdateResourceEventConfiguration');
    }
    /**
     * Grants permission to update position for a given resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_UpdateResourcePosition.html
     */
    toUpdateResourcePosition() {
        return this.to('UpdateResourcePosition');
    }
    /**
     * Grants permission to update a WirelessDevice resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_UpdateWirelessDevice.html
     */
    toUpdateWirelessDevice() {
        return this.to('UpdateWirelessDevice');
    }
    /**
     * Grants permission to update a wireless device import task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_UpdateWirelessDeviceImportTask.html
     */
    toUpdateWirelessDeviceImportTask() {
        return this.to('UpdateWirelessDeviceImportTask');
    }
    /**
     * Grants permission to update a WirelessGateway resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_UpdateWirelessGateway.html
     */
    toUpdateWirelessGateway() {
        return this.to('UpdateWirelessGateway');
    }
    /**
     * Adds a resource of type WirelessDevice to the statement
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_CreateWirelessDevice.html
     *
     * @param wirelessDeviceId - Identifier for the wirelessDeviceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWirelessDevice(wirelessDeviceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotwireless:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:WirelessDevice/${wirelessDeviceId}`);
    }
    /**
     * Adds a resource of type WirelessGateway to the statement
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_CreateWirelessGateway.html
     *
     * @param wirelessGatewayId - Identifier for the wirelessGatewayId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWirelessGateway(wirelessGatewayId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotwireless:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:WirelessGateway/${wirelessGatewayId}`);
    }
    /**
     * Adds a resource of type DeviceProfile to the statement
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_CreateDeviceProfile.html
     *
     * @param deviceProfileId - Identifier for the deviceProfileId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeviceProfile(deviceProfileId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotwireless:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:DeviceProfile/${deviceProfileId}`);
    }
    /**
     * Adds a resource of type ServiceProfile to the statement
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_CreateServiceProfile.html
     *
     * @param serviceProfileId - Identifier for the serviceProfileId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onServiceProfile(serviceProfileId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotwireless:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:ServiceProfile/${serviceProfileId}`);
    }
    /**
     * Adds a resource of type Destination to the statement
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_CreateDestination.html
     *
     * @param destinationName - Identifier for the destinationName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDestination(destinationName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotwireless:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:Destination/${destinationName}`);
    }
    /**
     * Adds a resource of type SidewalkAccount to the statement
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_AssociateAwsAccountWithPartnerAccount.html
     *
     * @param sidewalkAccountId - Identifier for the sidewalkAccountId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSidewalkAccount(sidewalkAccountId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotwireless:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:SidewalkAccount/${sidewalkAccountId}`);
    }
    /**
     * Adds a resource of type WirelessGatewayTaskDefinition to the statement
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_CreateWirelessGatewayTaskDefinition.html
     *
     * @param wirelessGatewayTaskDefinitionId - Identifier for the wirelessGatewayTaskDefinitionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWirelessGatewayTaskDefinition(wirelessGatewayTaskDefinitionId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotwireless:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:WirelessGatewayTaskDefinition/${wirelessGatewayTaskDefinitionId}`);
    }
    /**
     * Adds a resource of type FuotaTask to the statement
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_CreateFuotaTask.html
     *
     * @param fuotaTaskId - Identifier for the fuotaTaskId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFuotaTask(fuotaTaskId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotwireless:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:FuotaTask/${fuotaTaskId}`);
    }
    /**
     * Adds a resource of type MulticastGroup to the statement
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_CreateMulticastGroup.html
     *
     * @param multicastGroupId - Identifier for the multicastGroupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMulticastGroup(multicastGroupId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotwireless:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:MulticastGroup/${multicastGroupId}`);
    }
    /**
     * Adds a resource of type NetworkAnalyzerConfiguration to the statement
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_CreateNetworkAnalyzerConfiguration.html
     *
     * @param networkAnalyzerConfigurationName - Identifier for the networkAnalyzerConfigurationName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onNetworkAnalyzerConfiguration(networkAnalyzerConfigurationName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotwireless:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:NetworkAnalyzerConfiguration/${networkAnalyzerConfigurationName}`);
    }
    /**
     * Adds a resource of type thing to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/thing-registry.html
     *
     * @param thingName - Identifier for the thingName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onThing(thingName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iot:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:thing/${thingName}`);
    }
    /**
     * Adds a resource of type cert to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/x509-client-certs.html
     *
     * @param certificate - Identifier for the certificate.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onCert(certificate, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iot:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:cert/${certificate}`);
    }
    /**
     * Adds a resource of type ImportTask to the statement
     *
     * https://docs.aws.amazon.com/iot-wireless/2020-11-22/apireference/API_StartWirelessDeviceImportTask.html
     *
     * @param importTaskId - Identifier for the importTaskId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onImportTask(importTaskId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotwireless:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:ImportTask/${importTaskId}`);
    }
    /**
     * Filters access by a tag key that is present in the request that the user makes to IoT Wireless
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/tagging-iot-iam.html/
     *
     * Applies to actions:
     * - .toAssociateAwsAccountWithPartnerAccount()
     * - .toCreateDestination()
     * - .toCreateDeviceProfile()
     * - .toCreateFuotaTask()
     * - .toCreateMulticastGroup()
     * - .toCreateNetworkAnalyzerConfiguration()
     * - .toCreateServiceProfile()
     * - .toCreateWirelessDevice()
     * - .toCreateWirelessGateway()
     * - .toCreateWirelessGatewayTaskDefinition()
     * - .toStartSingleWirelessDeviceImportTask()
     * - .toStartWirelessDeviceImportTask()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key component of a tag attached to an IoT Wireless resource
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/tagging-iot-iam.html/
     *
     * Applies to resource types:
     * - WirelessDevice
     * - WirelessGateway
     * - DeviceProfile
     * - ServiceProfile
     * - Destination
     * - SidewalkAccount
     * - WirelessGatewayTaskDefinition
     * - FuotaTask
     * - MulticastGroup
     * - NetworkAnalyzerConfiguration
     * - ImportTask
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the list of all the tag key names associated with the resource in the request
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/tagging-iot-iam.html/
     *
     * Applies to actions:
     * - .toAssociateAwsAccountWithPartnerAccount()
     * - .toCreateDestination()
     * - .toCreateDeviceProfile()
     * - .toCreateFuotaTask()
     * - .toCreateMulticastGroup()
     * - .toCreateNetworkAnalyzerConfiguration()
     * - .toCreateServiceProfile()
     * - .toCreateWirelessDevice()
     * - .toCreateWirelessGateway()
     * - .toCreateWirelessGatewayTaskDefinition()
     * - .toStartSingleWirelessDeviceImportTask()
     * - .toStartWirelessDeviceImportTask()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [iotwireless](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotwireless.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'iotwireless';
        this.accessLevelList = {
            Write: [
                'AssociateAwsAccountWithPartnerAccount',
                'AssociateMulticastGroupWithFuotaTask',
                'AssociateWirelessDeviceWithFuotaTask',
                'AssociateWirelessDeviceWithMulticastGroup',
                'AssociateWirelessDeviceWithThing',
                'AssociateWirelessGatewayWithCertificate',
                'AssociateWirelessGatewayWithThing',
                'CancelMulticastGroupSession',
                'CreateDestination',
                'CreateDeviceProfile',
                'CreateFuotaTask',
                'CreateMulticastGroup',
                'CreateNetworkAnalyzerConfiguration',
                'CreateServiceProfile',
                'CreateWirelessDevice',
                'CreateWirelessGateway',
                'CreateWirelessGatewayTask',
                'CreateWirelessGatewayTaskDefinition',
                'DeleteDestination',
                'DeleteDeviceProfile',
                'DeleteFuotaTask',
                'DeleteMulticastGroup',
                'DeleteNetworkAnalyzerConfiguration',
                'DeleteQueuedMessages',
                'DeleteServiceProfile',
                'DeleteWirelessDevice',
                'DeleteWirelessDeviceImportTask',
                'DeleteWirelessGateway',
                'DeleteWirelessGatewayTask',
                'DeleteWirelessGatewayTaskDefinition',
                'DeregisterWirelessDevice',
                'DisassociateAwsAccountFromPartnerAccount',
                'DisassociateMulticastGroupFromFuotaTask',
                'DisassociateWirelessDeviceFromFuotaTask',
                'DisassociateWirelessDeviceFromMulticastGroup',
                'DisassociateWirelessDeviceFromThing',
                'DisassociateWirelessGatewayFromCertificate',
                'DisassociateWirelessGatewayFromThing',
                'PutPositionConfiguration',
                'PutResourceLogLevel',
                'ResetAllResourceLogLevels',
                'ResetResourceLogLevel',
                'SendDataToMulticastGroup',
                'SendDataToWirelessDevice',
                'StartBulkAssociateWirelessDeviceWithMulticastGroup',
                'StartBulkDisassociateWirelessDeviceFromMulticastGroup',
                'StartFuotaTask',
                'StartMulticastGroupSession',
                'StartNetworkAnalyzerStream',
                'StartSingleWirelessDeviceImportTask',
                'StartWirelessDeviceImportTask',
                'TestWirelessDevice',
                'UpdateDestination',
                'UpdateEventConfigurationByResourceTypes',
                'UpdateFuotaTask',
                'UpdateLogLevelsByResourceTypes',
                'UpdateMetricConfiguration',
                'UpdateMulticastGroup',
                'UpdateNetworkAnalyzerConfiguration',
                'UpdatePartnerAccount',
                'UpdatePosition',
                'UpdateResourceEventConfiguration',
                'UpdateResourcePosition',
                'UpdateWirelessDevice',
                'UpdateWirelessDeviceImportTask',
                'UpdateWirelessGateway'
            ],
            Read: [
                'GetDestination',
                'GetDeviceProfile',
                'GetEventConfigurationByResourceTypes',
                'GetFuotaTask',
                'GetLogLevelsByResourceTypes',
                'GetMetricConfiguration',
                'GetMetrics',
                'GetMulticastGroup',
                'GetMulticastGroupSession',
                'GetNetworkAnalyzerConfiguration',
                'GetPartnerAccount',
                'GetPosition',
                'GetPositionConfiguration',
                'GetPositionEstimate',
                'GetResourceEventConfiguration',
                'GetResourceLogLevel',
                'GetResourcePosition',
                'GetServiceEndpoint',
                'GetServiceProfile',
                'GetWirelessDevice',
                'GetWirelessDeviceImportTask',
                'GetWirelessDeviceStatistics',
                'GetWirelessGateway',
                'GetWirelessGatewayCertificate',
                'GetWirelessGatewayFirmwareInformation',
                'GetWirelessGatewayStatistics',
                'GetWirelessGatewayTask',
                'GetWirelessGatewayTaskDefinition',
                'ListDestinations',
                'ListDeviceProfiles',
                'ListDevicesForWirelessDeviceImportTask',
                'ListEventConfigurations',
                'ListFuotaTasks',
                'ListMulticastGroups',
                'ListMulticastGroupsByFuotaTask',
                'ListNetworkAnalyzerConfigurations',
                'ListPartnerAccounts',
                'ListPositionConfigurations',
                'ListQueuedMessages',
                'ListServiceProfiles',
                'ListTagsForResource',
                'ListWirelessDeviceImportTasks',
                'ListWirelessDevices',
                'ListWirelessGatewayTaskDefinitions',
                'ListWirelessGateways'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Iotwireless = Iotwireless;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW90d2lyZWxlc3MuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJpb3R3aXJlbGVzcy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFDQSx5Q0FBeUQ7QUFHekQ7Ozs7R0FJRztBQUNILE1BQWEsV0FBWSxTQUFRLHdCQUFlO0lBRzlDOzs7Ozs7Ozs7O09BVUc7SUFDSSx1Q0FBdUM7UUFDNUMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVDQUF1QyxDQUFDLENBQUM7SUFDMUQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNDQUFzQztRQUMzQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0NBQXNDLENBQUMsQ0FBQztJQUN6RCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0NBQXNDO1FBQzNDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDO0lBQ3pELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwyQ0FBMkM7UUFDaEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDJDQUEyQyxDQUFDLENBQUM7SUFDOUQsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGtDQUFrQztRQUN2QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0NBQWtDLENBQUMsQ0FBQztJQUNyRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUNBQXlDO1FBQzlDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5Q0FBeUMsQ0FBQyxDQUFDO0lBQzVELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxtQ0FBbUM7UUFDeEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1DQUFtQyxDQUFDLENBQUM7SUFDdEQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDZCQUE2QjtRQUNsQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztJQUNoRCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGlCQUFpQjtRQUN0QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLG9DQUFvQztRQUN6QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0NBQW9DLENBQUMsQ0FBQztJQUN2RCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLHVCQUF1QjtRQUM1QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLENBQUMsQ0FBQztJQUMxQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMkJBQTJCO1FBQ2hDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO0lBQzlDLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0kscUNBQXFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO0lBQ3hELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCO1FBQ3RCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO0lBQ3BDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQkFBc0I7UUFDM0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9DQUFvQztRQUN6QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0NBQW9DLENBQUMsQ0FBQztJQUN2RCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQkFBc0I7UUFDM0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0NBQWdDO1FBQ3JDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO0lBQ25ELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDJCQUEyQjtRQUNoQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMkJBQTJCLENBQUMsQ0FBQztJQUM5QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUNBQXFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO0lBQ3hELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwwQkFBMEI7UUFDL0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBDQUEwQztRQUMvQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMENBQTBDLENBQUMsQ0FBQztJQUM3RCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUNBQXlDO1FBQzlDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5Q0FBeUMsQ0FBQyxDQUFDO0lBQzVELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5Q0FBeUM7UUFDOUMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlDQUF5QyxDQUFDLENBQUM7SUFDNUQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDhDQUE4QztRQUNuRCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsOENBQThDLENBQUMsQ0FBQztJQUNqRSxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0kscUNBQXFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO0lBQ3hELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw0Q0FBNEM7UUFDakQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDRDQUE0QyxDQUFDLENBQUM7SUFDL0QsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHNDQUFzQztRQUMzQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0NBQXNDLENBQUMsQ0FBQztJQUN6RCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO0lBQ25DLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNDQUFzQztRQUMzQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0NBQXNDLENBQUMsQ0FBQztJQUN6RCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsY0FBYyxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDZCQUE2QjtRQUNsQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztJQUNoRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxZQUFZO1FBQ2pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxZQUFZLENBQUMsQ0FBQztJQUMvQixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwwQkFBMEI7UUFDL0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlDQUFpQztRQUN0QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUNBQWlDLENBQUMsQ0FBQztJQUNwRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUNoQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLCtCQUErQjtRQUNwQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsK0JBQStCLENBQUMsQ0FBQztJQUNsRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDZCQUE2QjtRQUNsQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztJQUNoRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNkJBQTZCO1FBQ2xDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO0lBQ2hELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLCtCQUErQjtRQUNwQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsK0JBQStCLENBQUMsQ0FBQztJQUNsRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUNBQXVDO1FBQzVDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1Q0FBdUMsQ0FBQyxDQUFDO0lBQzFELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw4QkFBOEI7UUFDbkMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDhCQUE4QixDQUFDLENBQUM7SUFDakQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0NBQWtDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO0lBQ3JELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0NBQXdDO1FBQzdDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3Q0FBd0MsQ0FBQyxDQUFDO0lBQzNELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUI7UUFDOUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7SUFDNUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQ0FBZ0M7UUFDckMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdDQUFnQyxDQUFDLENBQUM7SUFDbkQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1DQUFtQztRQUN4QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUNBQW1DLENBQUMsQ0FBQztJQUN0RCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw0QkFBNEI7UUFDakMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDRCQUE0QixDQUFDLENBQUM7SUFDL0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLCtCQUErQjtRQUNwQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsK0JBQStCLENBQUMsQ0FBQztJQUNsRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQ0FBb0M7UUFDekMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxDQUFDLENBQUM7SUFDdkQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDJCQUEyQjtRQUNoQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMkJBQTJCLENBQUMsQ0FBQztJQUM5QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUJBQXVCO1FBQzVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0lBQzFDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwwQkFBMEI7UUFDL0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBCQUEwQjtRQUMvQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztJQUM3QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0RBQW9EO1FBQ3pELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvREFBb0QsQ0FBQyxDQUFDO0lBQ3ZFLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1REFBdUQ7UUFDNUQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVEQUF1RCxDQUFDLENBQUM7SUFDMUUsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNEJBQTRCO1FBQ2pDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO0lBQy9DLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw0QkFBNEI7UUFDakMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDRCQUE0QixDQUFDLENBQUM7SUFDL0MsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxxQ0FBcUM7UUFDMUMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFDQUFxQyxDQUFDLENBQUM7SUFDeEQsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSwrQkFBK0I7UUFDcEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLCtCQUErQixDQUFDLENBQUM7SUFDbEQsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxhQUFhO1FBQ2xCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUNoQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5Q0FBeUM7UUFDOUMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlDQUF5QyxDQUFDLENBQUM7SUFDNUQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0NBQWdDO1FBQ3JDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO0lBQ25ELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwyQkFBMkI7UUFDaEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDJCQUEyQixDQUFDLENBQUM7SUFDOUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0NBQW9DO1FBQ3pDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQ0FBb0MsQ0FBQyxDQUFDO0lBQ3ZELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQkFBc0I7UUFDM0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0NBQWtDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO0lBQ3JELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx3QkFBd0I7UUFDN0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7SUFDM0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0NBQWdDO1FBQ3JDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO0lBQ25ELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQTRIRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxnQkFBZ0IsQ0FBQyxnQkFBd0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNyRyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixnQkFBaUIsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLG1CQUFvQixnQkFBaUIsRUFBRSxDQUFDLENBQUM7SUFDdkwsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGlCQUFpQixDQUFDLGlCQUF5QixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3ZHLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGdCQUFpQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsb0JBQXFCLGlCQUFrQixFQUFFLENBQUMsQ0FBQztJQUN6TCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksZUFBZSxDQUFDLGVBQXVCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDbkcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZ0JBQWlCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxrQkFBbUIsZUFBZ0IsRUFBRSxDQUFDLENBQUM7SUFDckwsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGdCQUFnQixDQUFDLGdCQUF3QixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3JHLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGdCQUFpQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsbUJBQW9CLGdCQUFpQixFQUFFLENBQUMsQ0FBQztJQUN2TCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksYUFBYSxDQUFDLGVBQXVCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDakcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZ0JBQWlCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxnQkFBaUIsZUFBZ0IsRUFBRSxDQUFDLENBQUM7SUFDbkwsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGlCQUFpQixDQUFDLGlCQUF5QixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3ZHLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGdCQUFpQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsb0JBQXFCLGlCQUFrQixFQUFFLENBQUMsQ0FBQztJQUN6TCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksK0JBQStCLENBQUMsK0JBQXVDLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDbkksT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZ0JBQWlCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxrQ0FBbUMsK0JBQWdDLEVBQUUsQ0FBQyxDQUFDO0lBQ3JOLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxXQUFXLENBQUMsV0FBbUIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUMzRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixnQkFBaUIsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLGNBQWUsV0FBWSxFQUFFLENBQUMsQ0FBQztJQUM3SyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksZ0JBQWdCLENBQUMsZ0JBQXdCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDckcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZ0JBQWlCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxtQkFBb0IsZ0JBQWlCLEVBQUUsQ0FBQyxDQUFDO0lBQ3ZMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSw4QkFBOEIsQ0FBQyxnQ0FBd0MsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNuSSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixnQkFBaUIsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLGlDQUFrQyxnQ0FBaUMsRUFBRSxDQUFDLENBQUM7SUFDck4sQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLE9BQU8sQ0FBQyxTQUFpQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3JGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLFFBQVMsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLFVBQVcsU0FBVSxFQUFFLENBQUMsQ0FBQztJQUMvSixDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksTUFBTSxDQUFDLFdBQW1CLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDdEYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsUUFBUyxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsU0FBVSxXQUFZLEVBQUUsQ0FBQyxDQUFDO0lBQ2hLLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxZQUFZLENBQUMsWUFBb0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUM3RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixnQkFBaUIsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLGVBQWdCLFlBQWEsRUFBRSxDQUFDLENBQUM7SUFDL0ssQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQXVCRztJQUNJLGVBQWUsQ0FBQyxNQUFjLEVBQUUsS0FBd0IsRUFBRSxRQUE0QjtRQUMzRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQW1CLE1BQU8sRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDaEYsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FxQkc7SUFDSSxnQkFBZ0IsQ0FBQyxNQUFjLEVBQUUsS0FBd0IsRUFBRSxRQUE0QjtRQUM1RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW9CLE1BQU8sRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDakYsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQXVCRztJQUNJLFlBQVksQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3hFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRSxDQUFDO0lBRUQ7OztPQUdHO0lBQ0gsWUFBWSxLQUFnQztRQUMxQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7UUF4c0RSLGtCQUFhLEdBQUcsYUFBYSxDQUFDO1FBZ3lDM0Isb0JBQWUsR0FBb0I7WUFDM0MsS0FBSyxFQUFFO2dCQUNMLHVDQUF1QztnQkFDdkMsc0NBQXNDO2dCQUN0QyxzQ0FBc0M7Z0JBQ3RDLDJDQUEyQztnQkFDM0Msa0NBQWtDO2dCQUNsQyx5Q0FBeUM7Z0JBQ3pDLG1DQUFtQztnQkFDbkMsNkJBQTZCO2dCQUM3QixtQkFBbUI7Z0JBQ25CLHFCQUFxQjtnQkFDckIsaUJBQWlCO2dCQUNqQixzQkFBc0I7Z0JBQ3RCLG9DQUFvQztnQkFDcEMsc0JBQXNCO2dCQUN0QixzQkFBc0I7Z0JBQ3RCLHVCQUF1QjtnQkFDdkIsMkJBQTJCO2dCQUMzQixxQ0FBcUM7Z0JBQ3JDLG1CQUFtQjtnQkFDbkIscUJBQXFCO2dCQUNyQixpQkFBaUI7Z0JBQ2pCLHNCQUFzQjtnQkFDdEIsb0NBQW9DO2dCQUNwQyxzQkFBc0I7Z0JBQ3RCLHNCQUFzQjtnQkFDdEIsc0JBQXNCO2dCQUN0QixnQ0FBZ0M7Z0JBQ2hDLHVCQUF1QjtnQkFDdkIsMkJBQTJCO2dCQUMzQixxQ0FBcUM7Z0JBQ3JDLDBCQUEwQjtnQkFDMUIsMENBQTBDO2dCQUMxQyx5Q0FBeUM7Z0JBQ3pDLHlDQUF5QztnQkFDekMsOENBQThDO2dCQUM5QyxxQ0FBcUM7Z0JBQ3JDLDRDQUE0QztnQkFDNUMsc0NBQXNDO2dCQUN0QywwQkFBMEI7Z0JBQzFCLHFCQUFxQjtnQkFDckIsMkJBQTJCO2dCQUMzQix1QkFBdUI7Z0JBQ3ZCLDBCQUEwQjtnQkFDMUIsMEJBQTBCO2dCQUMxQixvREFBb0Q7Z0JBQ3BELHVEQUF1RDtnQkFDdkQsZ0JBQWdCO2dCQUNoQiw0QkFBNEI7Z0JBQzVCLDRCQUE0QjtnQkFDNUIscUNBQXFDO2dCQUNyQywrQkFBK0I7Z0JBQy9CLG9CQUFvQjtnQkFDcEIsbUJBQW1CO2dCQUNuQix5Q0FBeUM7Z0JBQ3pDLGlCQUFpQjtnQkFDakIsZ0NBQWdDO2dCQUNoQywyQkFBMkI7Z0JBQzNCLHNCQUFzQjtnQkFDdEIsb0NBQW9DO2dCQUNwQyxzQkFBc0I7Z0JBQ3RCLGdCQUFnQjtnQkFDaEIsa0NBQWtDO2dCQUNsQyx3QkFBd0I7Z0JBQ3hCLHNCQUFzQjtnQkFDdEIsZ0NBQWdDO2dCQUNoQyx1QkFBdUI7YUFDeEI7WUFDRCxJQUFJLEVBQUU7Z0JBQ0osZ0JBQWdCO2dCQUNoQixrQkFBa0I7Z0JBQ2xCLHNDQUFzQztnQkFDdEMsY0FBYztnQkFDZCw2QkFBNkI7Z0JBQzdCLHdCQUF3QjtnQkFDeEIsWUFBWTtnQkFDWixtQkFBbUI7Z0JBQ25CLDBCQUEwQjtnQkFDMUIsaUNBQWlDO2dCQUNqQyxtQkFBbUI7Z0JBQ25CLGFBQWE7Z0JBQ2IsMEJBQTBCO2dCQUMxQixxQkFBcUI7Z0JBQ3JCLCtCQUErQjtnQkFDL0IscUJBQXFCO2dCQUNyQixxQkFBcUI7Z0JBQ3JCLG9CQUFvQjtnQkFDcEIsbUJBQW1CO2dCQUNuQixtQkFBbUI7Z0JBQ25CLDZCQUE2QjtnQkFDN0IsNkJBQTZCO2dCQUM3QixvQkFBb0I7Z0JBQ3BCLCtCQUErQjtnQkFDL0IsdUNBQXVDO2dCQUN2Qyw4QkFBOEI7Z0JBQzlCLHdCQUF3QjtnQkFDeEIsa0NBQWtDO2dCQUNsQyxrQkFBa0I7Z0JBQ2xCLG9CQUFvQjtnQkFDcEIsd0NBQXdDO2dCQUN4Qyx5QkFBeUI7Z0JBQ3pCLGdCQUFnQjtnQkFDaEIscUJBQXFCO2dCQUNyQixnQ0FBZ0M7Z0JBQ2hDLG1DQUFtQztnQkFDbkMscUJBQXFCO2dCQUNyQiw0QkFBNEI7Z0JBQzVCLG9CQUFvQjtnQkFDcEIscUJBQXFCO2dCQUNyQixxQkFBcUI7Z0JBQ3JCLCtCQUErQjtnQkFDL0IscUJBQXFCO2dCQUNyQixvQ0FBb0M7Z0JBQ3BDLHNCQUFzQjthQUN2QjtZQUNELE9BQU8sRUFBRTtnQkFDUCxhQUFhO2dCQUNiLGVBQWU7YUFDaEI7U0FDRixDQUFDO0lBaVRGLENBQUM7Q0FDRjtBQTNzREQsa0NBMnNEQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEFjY2Vzc0xldmVsTGlzdCB9IGZyb20gJy4uLy4uL3NoYXJlZC9hY2Nlc3MtbGV2ZWwnO1xuaW1wb3J0IHsgUG9saWN5U3RhdGVtZW50LCBPcGVyYXRvciB9IGZyb20gJy4uLy4uL3NoYXJlZCc7XG5pbXBvcnQgeyBhd3NfaWFtIGFzIGlhbSB9IGZyb20gXCJhd3MtY2RrLWxpYlwiO1xuXG4vKipcbiAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbaW90d2lyZWxlc3NdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlLWF1dGhvcml6YXRpb24vbGF0ZXN0L3JlZmVyZW5jZS9saXN0X2F3c2lvdHdpcmVsZXNzLmh0bWwpLlxuICpcbiAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICovXG5leHBvcnQgY2xhc3MgSW90d2lyZWxlc3MgZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnQge1xuICBwdWJsaWMgc2VydmljZVByZWZpeCA9ICdpb3R3aXJlbGVzcyc7XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpbmsgcGFydG5lciBhY2NvdW50cyB3aXRoIEFXUyBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfQXNzb2NpYXRlQXdzQWNjb3VudFdpdGhQYXJ0bmVyQWNjb3VudC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Bc3NvY2lhdGVBd3NBY2NvdW50V2l0aFBhcnRuZXJBY2NvdW50KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdBc3NvY2lhdGVBd3NBY2NvdW50V2l0aFBhcnRuZXJBY2NvdW50Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYXNzb2NpYXRlIHRoZSBNdWx0aWNhc3RHcm91cCB3aXRoIEZ1b3RhVGFza1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX0Fzc29jaWF0ZU11bHRpY2FzdEdyb3VwV2l0aEZ1b3RhVGFzay5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Bc3NvY2lhdGVNdWx0aWNhc3RHcm91cFdpdGhGdW90YVRhc2soKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Fzc29jaWF0ZU11bHRpY2FzdEdyb3VwV2l0aEZ1b3RhVGFzaycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFzc29jaWF0ZSB0aGUgd2lyZWxlc3MgZGV2aWNlIHdpdGggRnVvdGFUYXNrXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfQXNzb2NpYXRlV2lyZWxlc3NEZXZpY2VXaXRoRnVvdGFUYXNrLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Fzc29jaWF0ZVdpcmVsZXNzRGV2aWNlV2l0aEZ1b3RhVGFzaygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQXNzb2NpYXRlV2lyZWxlc3NEZXZpY2VXaXRoRnVvdGFUYXNrJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYXNzb2NpYXRlIHRoZSBXaXJlbGVzc0RldmljZSB3aXRoIE11bHRpY2FzdEdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfQXNzb2NpYXRlV2lyZWxlc3NEZXZpY2VXaXRoTXVsdGljYXN0R3JvdXAuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQXNzb2NpYXRlV2lyZWxlc3NEZXZpY2VXaXRoTXVsdGljYXN0R3JvdXAoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Fzc29jaWF0ZVdpcmVsZXNzRGV2aWNlV2l0aE11bHRpY2FzdEdyb3VwJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYXNzb2NpYXRlIHRoZSB3aXJlbGVzcyBkZXZpY2Ugd2l0aCBBV1MgSW9UIHRoaW5nIGZvciBhIGdpdmVuIHdpcmVsZXNzRGV2aWNlSWRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBpb3Q6RGVzY3JpYmVUaGluZ1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9Bc3NvY2lhdGVXaXJlbGVzc0RldmljZVdpdGhUaGluZy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Bc3NvY2lhdGVXaXJlbGVzc0RldmljZVdpdGhUaGluZygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQXNzb2NpYXRlV2lyZWxlc3NEZXZpY2VXaXRoVGhpbmcnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBhc3NvY2lhdGUgYSBXaXJlbGVzc0dhdGV3YXkgd2l0aCB0aGUgSW9UIENvcmUgSWRlbnRpdHkgY2VydGlmaWNhdGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9Bc3NvY2lhdGVXaXJlbGVzc0dhdGV3YXlXaXRoQ2VydGlmaWNhdGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQXNzb2NpYXRlV2lyZWxlc3NHYXRld2F5V2l0aENlcnRpZmljYXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdBc3NvY2lhdGVXaXJlbGVzc0dhdGV3YXlXaXRoQ2VydGlmaWNhdGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBhc3NvY2lhdGUgdGhlIHdpcmVsZXNzIGdhdGV3YXkgd2l0aCBBV1MgSW9UIHRoaW5nIGZvciBhIGdpdmVuIHdpcmVsZXNzR2F0ZXdheUlkXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gaW90OkRlc2NyaWJlVGhpbmdcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfQXNzb2NpYXRlV2lyZWxlc3NHYXRld2F5V2l0aFRoaW5nLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Fzc29jaWF0ZVdpcmVsZXNzR2F0ZXdheVdpdGhUaGluZygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQXNzb2NpYXRlV2lyZWxlc3NHYXRld2F5V2l0aFRoaW5nJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY2FuY2VsIHRoZSBNdWx0aWNhc3RHcm91cCBzZXNzaW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfQ2FuY2VsTXVsdGljYXN0R3JvdXBTZXNzaW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NhbmNlbE11bHRpY2FzdEdyb3VwU2Vzc2lvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ2FuY2VsTXVsdGljYXN0R3JvdXBTZXNzaW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgRGVzdGluYXRpb24gcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9DcmVhdGVEZXN0aW5hdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVEZXN0aW5hdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlRGVzdGluYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBEZXZpY2VQcm9maWxlIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfQ3JlYXRlRGV2aWNlUHJvZmlsZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVEZXZpY2VQcm9maWxlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVEZXZpY2VQcm9maWxlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgRnVvdGFUYXNrIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfQ3JlYXRlRnVvdGFUYXNrLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUZ1b3RhVGFzaygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlRnVvdGFUYXNrJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgTXVsdGljYXN0R3JvdXAgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9DcmVhdGVNdWx0aWNhc3RHcm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVNdWx0aWNhc3RHcm91cCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlTXVsdGljYXN0R3JvdXAnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBOZXR3b3JrQW5hbHl6ZXJDb25maWd1cmF0aW9uIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfQ3JlYXRlTmV0d29ya0FuYWx5emVyQ29uZmlndXJhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVOZXR3b3JrQW5hbHl6ZXJDb25maWd1cmF0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVOZXR3b3JrQW5hbHl6ZXJDb25maWd1cmF0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgU2VydmljZVByb2ZpbGUgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9DcmVhdGVTZXJ2aWNlUHJvZmlsZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVTZXJ2aWNlUHJvZmlsZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlU2VydmljZVByb2ZpbGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBXaXJlbGVzc0RldmljZSByZXNvdXJjZSB3aXRoIGdpdmVuIERlc3RpbmF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfQ3JlYXRlV2lyZWxlc3NEZXZpY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlV2lyZWxlc3NEZXZpY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZVdpcmVsZXNzRGV2aWNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgV2lyZWxlc3NHYXRld2F5IHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfQ3JlYXRlV2lyZWxlc3NHYXRld2F5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVdpcmVsZXNzR2F0ZXdheSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlV2lyZWxlc3NHYXRld2F5Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgdGFzayBmb3IgYSBnaXZlbiBXaXJlbGVzc0dhdGV3YXlcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9DcmVhdGVXaXJlbGVzc0dhdGV3YXlUYXNrLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVdpcmVsZXNzR2F0ZXdheVRhc2soKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZVdpcmVsZXNzR2F0ZXdheVRhc2snKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBXaXJlbGVzc0dhdGV3YXkgdGFzayBkZWZpbml0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfQ3JlYXRlV2lyZWxlc3NHYXRld2F5VGFza0RlZmluaXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlV2lyZWxlc3NHYXRld2F5VGFza0RlZmluaXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZVdpcmVsZXNzR2F0ZXdheVRhc2tEZWZpbml0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgRGVzdGluYXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9EZWxldGVEZXN0aW5hdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVEZXN0aW5hdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlRGVzdGluYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBEZXZpY2VQcm9maWxlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfRGVsZXRlRGV2aWNlUHJvZmlsZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVEZXZpY2VQcm9maWxlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVEZXZpY2VQcm9maWxlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIHRoZSBGdW90YVRhc2tcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9EZWxldGVGdW90YVRhc2suaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlRnVvdGFUYXNrKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVGdW90YVRhc2snKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgdGhlIE11bHRpY2FzdEdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfRGVsZXRlTXVsdGljYXN0R3JvdXAuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlTXVsdGljYXN0R3JvdXAoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZU11bHRpY2FzdEdyb3VwJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIHRoZSBOZXR3b3JrQW5hbHl6ZXJDb25maWd1cmF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfRGVsZXRlTmV0d29ya0FuYWx5emVyQ29uZmlndXJhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVOZXR3b3JrQW5hbHl6ZXJDb25maWd1cmF0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVOZXR3b3JrQW5hbHl6ZXJDb25maWd1cmF0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIFF1ZXVlZE1lc3NhZ2VzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfRGVsZXRlUXVldWVkTWVzc2FnZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlUXVldWVkTWVzc2FnZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZVF1ZXVlZE1lc3NhZ2VzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgU2VydmljZVByb2ZpbGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9EZWxldGVTZXJ2aWNlUHJvZmlsZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVTZXJ2aWNlUHJvZmlsZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlU2VydmljZVByb2ZpbGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBXaXJlbGVzc0RldmljZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX0RlbGV0ZVdpcmVsZXNzRGV2aWNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVdpcmVsZXNzRGV2aWNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVXaXJlbGVzc0RldmljZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSB0aGUgd2lyZWxlc3MgZGV2aWNlIGltcG9ydCB0YXNrXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfRGVsZXRlV2lyZWxlc3NEZXZpY2VJbXBvcnRUYXNrLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVdpcmVsZXNzRGV2aWNlSW1wb3J0VGFzaygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlV2lyZWxlc3NEZXZpY2VJbXBvcnRUYXNrJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgV2lyZWxlc3NHYXRld2F5XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfRGVsZXRlV2lyZWxlc3NHYXRld2F5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVdpcmVsZXNzR2F0ZXdheSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlV2lyZWxlc3NHYXRld2F5Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIHRhc2sgZm9yIGEgZ2l2ZW4gV2lyZWxlc3NHYXRld2F5XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfRGVsZXRlV2lyZWxlc3NHYXRld2F5VGFzay5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVXaXJlbGVzc0dhdGV3YXlUYXNrKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVXaXJlbGVzc0dhdGV3YXlUYXNrJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgV2lyZWxlc3NHYXRld2F5IHRhc2sgZGVmaW5pdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX0RlbGV0ZVdpcmVsZXNzR2F0ZXdheVRhc2tEZWZpbml0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVdpcmVsZXNzR2F0ZXdheVRhc2tEZWZpbml0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVXaXJlbGVzc0dhdGV3YXlUYXNrRGVmaW5pdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlcmVnaXN0ZXIgd2lyZWxlc3MgZGV2aWNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfRGVyZWdpc3RlcldpcmVsZXNzRGV2aWNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlcmVnaXN0ZXJXaXJlbGVzc0RldmljZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVyZWdpc3RlcldpcmVsZXNzRGV2aWNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGlzYXNzb2NpYXRlIGFuIEFXUyBhY2NvdW50IGZyb20gYSBwYXJ0bmVyIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9EaXNhc3NvY2lhdGVBd3NBY2NvdW50RnJvbVBhcnRuZXJBY2NvdW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rpc2Fzc29jaWF0ZUF3c0FjY291bnRGcm9tUGFydG5lckFjY291bnQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rpc2Fzc29jaWF0ZUF3c0FjY291bnRGcm9tUGFydG5lckFjY291bnQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkaXNhc3NvY2lhdGUgdGhlIE11bHRpY2FzdEdyb3VwIGZyb20gRnVvdGFUYXNrXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfRGlzYXNzb2NpYXRlTXVsdGljYXN0R3JvdXBGcm9tRnVvdGFUYXNrLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rpc2Fzc29jaWF0ZU11bHRpY2FzdEdyb3VwRnJvbUZ1b3RhVGFzaygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGlzYXNzb2NpYXRlTXVsdGljYXN0R3JvdXBGcm9tRnVvdGFUYXNrJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGlzYXNzb2NpYXRlIHRoZSB3aXJlbGVzcyBkZXZpY2UgZnJvbSBGdW90YVRhc2tcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9EaXNhc3NvY2lhdGVXaXJlbGVzc0RldmljZUZyb21GdW90YVRhc2suaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGlzYXNzb2NpYXRlV2lyZWxlc3NEZXZpY2VGcm9tRnVvdGFUYXNrKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEaXNhc3NvY2lhdGVXaXJlbGVzc0RldmljZUZyb21GdW90YVRhc2snKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkaXNhc3NvY2lhdGUgdGhlIHdpcmVsZXNzIGRldmljZSBmcm9tIE11bHRpY2FzdEdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfRGlzYXNzb2NpYXRlV2lyZWxlc3NEZXZpY2VGcm9tTXVsdGljYXN0R3JvdXAuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGlzYXNzb2NpYXRlV2lyZWxlc3NEZXZpY2VGcm9tTXVsdGljYXN0R3JvdXAoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rpc2Fzc29jaWF0ZVdpcmVsZXNzRGV2aWNlRnJvbU11bHRpY2FzdEdyb3VwJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGlzYXNzb2NpYXRlIGEgd2lyZWxlc3MgZGV2aWNlIGZyb20gYSBBV1MgSW9UIHRoaW5nXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gaW90OkRlc2NyaWJlVGhpbmdcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfRGlzYXNzb2NpYXRlV2lyZWxlc3NEZXZpY2VGcm9tVGhpbmcuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGlzYXNzb2NpYXRlV2lyZWxlc3NEZXZpY2VGcm9tVGhpbmcoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rpc2Fzc29jaWF0ZVdpcmVsZXNzRGV2aWNlRnJvbVRoaW5nJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGlzYXNzb2NpYXRlIGEgV2lyZWxlc3NHYXRld2F5IGZyb20gYSBJb1QgQ29yZSBJZGVudGl0eSBjZXJ0aWZpY2F0ZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX0Rpc2Fzc29jaWF0ZVdpcmVsZXNzR2F0ZXdheUZyb21DZXJ0aWZpY2F0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EaXNhc3NvY2lhdGVXaXJlbGVzc0dhdGV3YXlGcm9tQ2VydGlmaWNhdGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rpc2Fzc29jaWF0ZVdpcmVsZXNzR2F0ZXdheUZyb21DZXJ0aWZpY2F0ZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRpc2Fzc29jaWF0ZSBhIFdpcmVsZXNzR2F0ZXdheSBmcm9tIGEgSW9UIENvcmUgdGhpbmdcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBpb3Q6RGVzY3JpYmVUaGluZ1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9EaXNhc3NvY2lhdGVXaXJlbGVzc0dhdGV3YXlGcm9tVGhpbmcuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGlzYXNzb2NpYXRlV2lyZWxlc3NHYXRld2F5RnJvbVRoaW5nKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEaXNhc3NvY2lhdGVXaXJlbGVzc0dhdGV3YXlGcm9tVGhpbmcnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgdGhlIERlc3RpbmF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9HZXREZXN0aW5hdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXREZXN0aW5hdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0RGVzdGluYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgdGhlIERldmljZVByb2ZpbGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX0dldERldmljZVByb2ZpbGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0RGV2aWNlUHJvZmlsZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0RGV2aWNlUHJvZmlsZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBldmVudCBjb25maWd1cmF0aW9uIGJ5IHJlc291cmNlIHR5cGVzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9HZXRFdmVudENvbmZpZ3VyYXRpb25CeVJlc291cmNlVHlwZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0RXZlbnRDb25maWd1cmF0aW9uQnlSZXNvdXJjZVR5cGVzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRFdmVudENvbmZpZ3VyYXRpb25CeVJlc291cmNlVHlwZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgdGhlIEZ1b3RhVGFza1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfR2V0RnVvdGFUYXNrLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEZ1b3RhVGFzaygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0RnVvdGFUYXNrJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGxvZyBsZXZlbHMgYnkgcmVzb3VyY2UgdHlwZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX0dldExvZ0xldmVsc0J5UmVzb3VyY2VUeXBlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRMb2dMZXZlbHNCeVJlc291cmNlVHlwZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldExvZ0xldmVsc0J5UmVzb3VyY2VUeXBlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBtZXRyaWMgY29uZmlndXJhdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfR2V0TWV0cmljQ29uZmlndXJhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRNZXRyaWNDb25maWd1cmF0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRNZXRyaWNDb25maWd1cmF0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IG1ldHJpY3NcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX0dldE1ldHJpY3MuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0TWV0cmljcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0TWV0cmljcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCB0aGUgTXVsdGljYXN0R3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX0dldE11bHRpY2FzdEdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldE11bHRpY2FzdEdyb3VwKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRNdWx0aWNhc3RHcm91cCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCB0aGUgTXVsdGljYXN0R3JvdXAgc2Vzc2lvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfR2V0TXVsdGljYXN0R3JvdXBTZXNzaW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldE11bHRpY2FzdEdyb3VwU2Vzc2lvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0TXVsdGljYXN0R3JvdXBTZXNzaW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IHRoZSBOZXR3b3JrQW5hbHl6ZXJDb25maWd1cmF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9HZXROZXR3b3JrQW5hbHl6ZXJDb25maWd1cmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldE5ldHdvcmtBbmFseXplckNvbmZpZ3VyYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldE5ldHdvcmtBbmFseXplckNvbmZpZ3VyYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgdGhlIGFzc29jaWF0ZWQgUGFydG5lckFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX0dldFBhcnRuZXJBY2NvdW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFBhcnRuZXJBY2NvdW50KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRQYXJ0bmVyQWNjb3VudCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBwb3NpdGlvbiBmb3IgYSBnaXZlbiByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfR2V0UG9zaXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0UG9zaXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldFBvc2l0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IHBvc2l0aW9uIGNvbmZpZ3VyYXRpb24gZm9yIGEgZ2l2ZW4gcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX0dldFBvc2l0aW9uQ29uZmlndXJhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRQb3NpdGlvbkNvbmZpZ3VyYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldFBvc2l0aW9uQ29uZmlndXJhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBwb3NpdGlvbiBlc3RpbWF0ZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfR2V0UG9zaXRpb25Fc3RpbWF0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRQb3NpdGlvbkVzdGltYXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRQb3NpdGlvbkVzdGltYXRlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGFuIGV2ZW50IGNvbmZpZ3VyYXRpb24gZm9yIGFuIGlkZW50aWZpZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX0dldFJlc291cmNlRXZlbnRDb25maWd1cmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFJlc291cmNlRXZlbnRDb25maWd1cmF0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRSZXNvdXJjZUV2ZW50Q29uZmlndXJhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCByZXNvdXJjZSBsb2cgbGV2ZWxcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX0dldFJlc291cmNlTG9nTGV2ZWwuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0UmVzb3VyY2VMb2dMZXZlbCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0UmVzb3VyY2VMb2dMZXZlbCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBwb3NpdGlvbiBmb3IgYSBnaXZlbiByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfR2V0UmVzb3VyY2VQb3NpdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRSZXNvdXJjZVBvc2l0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRSZXNvdXJjZVBvc2l0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgdGhlIGN1c3RvbWVyIGFjY291bnQgc3BlY2lmaWMgZW5kcG9pbnQgZm9yIENVUFMgcHJvdG9jb2wgY29ubmVjdGlvbiBvciBMb1JhV0FOIE5ldHdvcmsgU2VydmVyIChMTlMpIHByb3RvY29sIGNvbm5lY3Rpb24sIGFuZCBvcHRpb25hbGx5IHNlcnZlciB0cnVzdCBjZXJ0aWZpY2F0ZSBpbiBQRU0gZm9ybWF0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9HZXRTZXJ2aWNlRW5kcG9pbnQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0U2VydmljZUVuZHBvaW50KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRTZXJ2aWNlRW5kcG9pbnQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgdGhlIFNlcnZpY2VQcm9maWxlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9HZXRTZXJ2aWNlUHJvZmlsZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRTZXJ2aWNlUHJvZmlsZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0U2VydmljZVByb2ZpbGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgdGhlIFdpcmVsZXNzRGV2aWNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9HZXRXaXJlbGVzc0RldmljZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRXaXJlbGVzc0RldmljZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0V2lyZWxlc3NEZXZpY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgdGhlIHdpcmVsZXNzIGRldmljZSBpbXBvcnQgdGFza1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfR2V0V2lyZWxlc3NEZXZpY2VJbXBvcnRUYXNrLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFdpcmVsZXNzRGV2aWNlSW1wb3J0VGFzaygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0V2lyZWxlc3NEZXZpY2VJbXBvcnRUYXNrJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IHN0YXRpc3RpY3MgaW5mbyBmb3IgYSBnaXZlbiBXaXJlbGVzc0RldmljZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfR2V0V2lyZWxlc3NEZXZpY2VTdGF0aXN0aWNzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFdpcmVsZXNzRGV2aWNlU3RhdGlzdGljcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0V2lyZWxlc3NEZXZpY2VTdGF0aXN0aWNzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IHRoZSBXaXJlbGVzc0dhdGV3YXlcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX0dldFdpcmVsZXNzR2F0ZXdheS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRXaXJlbGVzc0dhdGV3YXkoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldFdpcmVsZXNzR2F0ZXdheScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCB0aGUgSW9UIENvcmUgSWRlbnRpdHkgY2VydGlmaWNhdGUgaWQgYXNzb2NpYXRlZCB3aXRoIHRoZSBXaXJlbGVzc0dhdGV3YXlcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX0dldFdpcmVsZXNzR2F0ZXdheUNlcnRpZmljYXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFdpcmVsZXNzR2F0ZXdheUNlcnRpZmljYXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRXaXJlbGVzc0dhdGV3YXlDZXJ0aWZpY2F0ZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBDdXJyZW50IGZpcm13YXJlIHZlcnNpb24gYW5kIG90aGVyIGluZm9ybWF0aW9uIGZvciB0aGUgV2lyZWxlc3NHYXRld2F5XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9HZXRXaXJlbGVzc0dhdGV3YXlGaXJtd2FyZUluZm9ybWF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFdpcmVsZXNzR2F0ZXdheUZpcm13YXJlSW5mb3JtYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldFdpcmVsZXNzR2F0ZXdheUZpcm13YXJlSW5mb3JtYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgc3RhdGlzdGljcyBpbmZvIGZvciBhIGdpdmVuIFdpcmVsZXNzR2F0ZXdheVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfR2V0V2lyZWxlc3NHYXRld2F5U3RhdGlzdGljcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRXaXJlbGVzc0dhdGV3YXlTdGF0aXN0aWNzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRXaXJlbGVzc0dhdGV3YXlTdGF0aXN0aWNzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IHRoZSB0YXNrIGZvciBhIGdpdmVuIFdpcmVsZXNzR2F0ZXdheVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfR2V0V2lyZWxlc3NHYXRld2F5VGFzay5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRXaXJlbGVzc0dhdGV3YXlUYXNrKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRXaXJlbGVzc0dhdGV3YXlUYXNrJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IHRoZSBnaXZlbiBXaXJlbGVzc0dhdGV3YXkgdGFzayBkZWZpbml0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9HZXRXaXJlbGVzc0dhdGV3YXlUYXNrRGVmaW5pdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRXaXJlbGVzc0dhdGV3YXlUYXNrRGVmaW5pdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0V2lyZWxlc3NHYXRld2F5VGFza0RlZmluaXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGluZm9ybWF0aW9uIG9mIGF2YWlsYWJsZSBEZXN0aW5hdGlvbnMgYmFzZWQgb24gdGhlIEFXUyBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9MaXN0RGVzdGluYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3REZXN0aW5hdGlvbnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3REZXN0aW5hdGlvbnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGluZm9ybWF0aW9uIG9mIGF2YWlsYWJsZSBEZXZpY2VQcm9maWxlcyBiYXNlZCBvbiB0aGUgQVdTIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX0xpc3REZXZpY2VQcm9maWxlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0RGV2aWNlUHJvZmlsZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3REZXZpY2VQcm9maWxlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgaW5mb3JtYXRpb24gb2YgZGV2aWNlcyBieSB3aXJlbGVzcyBkZXZpY2UgaW1wb3J0IHRhc2sgYmFzZWQgb24gdGhlIEFXUyBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9MaXN0RGV2aWNlc0ZvcldpcmVsZXNzRGV2aWNlSW1wb3J0VGFzay5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0RGV2aWNlc0ZvcldpcmVsZXNzRGV2aWNlSW1wb3J0VGFzaygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdERldmljZXNGb3JXaXJlbGVzc0RldmljZUltcG9ydFRhc2snKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGluZm9ybWF0aW9uIG9mIGF2YWlsYWJsZSBldmVudCBjb25maWd1cmF0aW9ucyBiYXNlZCBvbiB0aGUgQVdTIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX0xpc3RFdmVudENvbmZpZ3VyYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RFdmVudENvbmZpZ3VyYXRpb25zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0RXZlbnRDb25maWd1cmF0aW9ucycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgaW5mb3JtYXRpb24gb2YgYXZhaWxhYmxlIEZ1b3RhVGFza3MgYmFzZWQgb24gdGhlIEFXUyBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9MaXN0RnVvdGFUYXNrcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0RnVvdGFUYXNrcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdEZ1b3RhVGFza3MnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGluZm9ybWF0aW9uIG9mIGF2YWlsYWJsZSBNdWx0aWNhc3RHcm91cHMgYmFzZWQgb24gdGhlIEFXUyBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9MaXN0TXVsdGljYXN0R3JvdXBzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RNdWx0aWNhc3RHcm91cHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RNdWx0aWNhc3RHcm91cHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGluZm9ybWF0aW9uIG9mIGF2YWlsYWJsZSBNdWx0aWNhc3RHcm91cHMgYnkgRnVvdGFUYXNrIGJhc2VkIG9uIHRoZSBBV1MgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfTGlzdE11bHRpY2FzdEdyb3Vwc0J5RnVvdGFUYXNrLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RNdWx0aWNhc3RHcm91cHNCeUZ1b3RhVGFzaygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdE11bHRpY2FzdEdyb3Vwc0J5RnVvdGFUYXNrJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBpbmZvcm1hdGlvbiBvZiBhdmFpbGFibGUgTmV0d29ya0FuYWx5emVyQ29uZmlndXJhdGlvbnMgYmFzZWQgb24gdGhlIEFXUyBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9MaXN0TmV0d29ya0FuYWx5emVyQ29uZmlndXJhdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdE5ldHdvcmtBbmFseXplckNvbmZpZ3VyYXRpb25zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0TmV0d29ya0FuYWx5emVyQ29uZmlndXJhdGlvbnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHRoZSBhdmFpbGFibGUgcGFydG5lciBhY2NvdW50c1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfTGlzdFBhcnRuZXJBY2NvdW50cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0UGFydG5lckFjY291bnRzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0UGFydG5lckFjY291bnRzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBpbmZvcm1hdGlvbiBvZiBhdmFpbGFibGUgcG9zaXRpb24gY29uZmlndXJhdGlvbnMgYmFzZWQgb24gdGhlIEFXUyBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9MaXN0UG9zaXRpb25Db25maWd1cmF0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0UG9zaXRpb25Db25maWd1cmF0aW9ucygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFBvc2l0aW9uQ29uZmlndXJhdGlvbnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHRoZSBRdWV1ZWQgTWVzc2FnZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX0xpc3RRdWV1ZWRNZXNzYWdlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0UXVldWVkTWVzc2FnZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RRdWV1ZWRNZXNzYWdlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgaW5mb3JtYXRpb24gb2YgYXZhaWxhYmxlIFNlcnZpY2VQcm9maWxlcyBiYXNlZCBvbiB0aGUgQVdTIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX0xpc3RTZXJ2aWNlUHJvZmlsZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFNlcnZpY2VQcm9maWxlcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFNlcnZpY2VQcm9maWxlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgYWxsIHRhZ3MgZm9yIGEgZ2l2ZW4gcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX0xpc3RUYWdzRm9yUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFRhZ3NGb3JSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFRhZ3NGb3JSZXNvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3Qgd2lyZWxlc3MgZGV2aWNlIGltcG9ydCB0YXNrcyBpbmZvcm1hdGlvbiBvZiBiYXNlZCBvbiB0aGUgQVdTIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX0xpc3RXaXJlbGVzc0RldmljZUltcG9ydFRhc2tzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RXaXJlbGVzc0RldmljZUltcG9ydFRhc2tzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0V2lyZWxlc3NEZXZpY2VJbXBvcnRUYXNrcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgaW5mb3JtYXRpb24gb2YgYXZhaWxhYmxlIFdpcmVsZXNzRGV2aWNlcyBiYXNlZCBvbiB0aGUgQVdTIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX0xpc3RXaXJlbGVzc0RldmljZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFdpcmVsZXNzRGV2aWNlcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFdpcmVsZXNzRGV2aWNlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgaW5mb3JtYXRpb24gb2YgYXZhaWxhYmxlIFdpcmVsZXNzR2F0ZXdheSB0YXNrIGRlZmluaXRpb25zIGJhc2VkIG9uIHRoZSBBV1MgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfTGlzdFdpcmVsZXNzR2F0ZXdheVRhc2tEZWZpbml0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0V2lyZWxlc3NHYXRld2F5VGFza0RlZmluaXRpb25zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0V2lyZWxlc3NHYXRld2F5VGFza0RlZmluaXRpb25zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBpbmZvcm1hdGlvbiBvZiBhdmFpbGFibGUgV2lyZWxlc3NHYXRld2F5cyBiYXNlZCBvbiB0aGUgQVdTIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX0xpc3RXaXJlbGVzc0dhdGV3YXlzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RXaXJlbGVzc0dhdGV3YXlzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0V2lyZWxlc3NHYXRld2F5cycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHB1dCBwb3NpdGlvbiBjb25maWd1cmF0aW9uIGZvciBhIGdpdmVuIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfUHV0UG9zaXRpb25Db25maWd1cmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1B1dFBvc2l0aW9uQ29uZmlndXJhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnUHV0UG9zaXRpb25Db25maWd1cmF0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcHV0IHJlc291cmNlIGxvZyBsZXZlbFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX1B1dFJlc291cmNlTG9nTGV2ZWwuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUHV0UmVzb3VyY2VMb2dMZXZlbCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnUHV0UmVzb3VyY2VMb2dMZXZlbCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJlc2V0IGFsbCByZXNvdXJjZSBsb2cgbGV2ZWxzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfUmVzZXRBbGxSZXNvdXJjZUxvZ0xldmVscy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9SZXNldEFsbFJlc291cmNlTG9nTGV2ZWxzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdSZXNldEFsbFJlc291cmNlTG9nTGV2ZWxzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVzZXQgcmVzb3VyY2UgbG9nIGxldmVsXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfUmVzZXRSZXNvdXJjZUxvZ0xldmVsLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1Jlc2V0UmVzb3VyY2VMb2dMZXZlbCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnUmVzZXRSZXNvdXJjZUxvZ0xldmVsJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc2VuZCBkYXRhIHRvIHRoZSBNdWx0aWNhc3RHcm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX1NlbmREYXRhVG9NdWx0aWNhc3RHcm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TZW5kRGF0YVRvTXVsdGljYXN0R3JvdXAoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1NlbmREYXRhVG9NdWx0aWNhc3RHcm91cCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHNlbmQgdGhlIGRlY3J5cHRlZCBhcHBsaWNhdGlvbiBkYXRhIGZyYW1lIHRvIHRoZSB0YXJnZXQgZGV2aWNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfU2VuZERhdGFUb1dpcmVsZXNzRGV2aWNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1NlbmREYXRhVG9XaXJlbGVzc0RldmljZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU2VuZERhdGFUb1dpcmVsZXNzRGV2aWNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYXNzb2NpYXRlIHRoZSBXaXJlbGVzc0RldmljZXMgd2l0aCBNdWx0aWNhc3RHcm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX1N0YXJ0QnVsa0Fzc29jaWF0ZVdpcmVsZXNzRGV2aWNlV2l0aE11bHRpY2FzdEdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0YXJ0QnVsa0Fzc29jaWF0ZVdpcmVsZXNzRGV2aWNlV2l0aE11bHRpY2FzdEdyb3VwKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdGFydEJ1bGtBc3NvY2lhdGVXaXJlbGVzc0RldmljZVdpdGhNdWx0aWNhc3RHcm91cCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGJ1bGsgZGlzYXNzb2NpYXRlIHRoZSBXaXJlbGVzc0RldmljZXMgZnJvbSBNdWx0aWNhc3RHcm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX1N0YXJ0QnVsa0Rpc2Fzc29jaWF0ZVdpcmVsZXNzRGV2aWNlRnJvbU11bHRpY2FzdEdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0YXJ0QnVsa0Rpc2Fzc29jaWF0ZVdpcmVsZXNzRGV2aWNlRnJvbU11bHRpY2FzdEdyb3VwKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdGFydEJ1bGtEaXNhc3NvY2lhdGVXaXJlbGVzc0RldmljZUZyb21NdWx0aWNhc3RHcm91cCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHN0YXJ0IHRoZSBGdW90YVRhc2tcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9TdGFydEZ1b3RhVGFzay5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TdGFydEZ1b3RhVGFzaygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU3RhcnRGdW90YVRhc2snKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzdGFydCB0aGUgTXVsdGljYXN0R3JvdXAgc2Vzc2lvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX1N0YXJ0TXVsdGljYXN0R3JvdXBTZXNzaW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0YXJ0TXVsdGljYXN0R3JvdXBTZXNzaW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdGFydE11bHRpY2FzdEdyb3VwU2Vzc2lvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHN0YXJ0IE5ldHdvcmtBbmFseXplciBzdHJlYW1cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9jb25uZWN0LWlvdC1sb3Jhd2FuLW5ldHdvcmstYW5hbHl6ZXItYXBpLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0YXJ0TmV0d29ya0FuYWx5emVyU3RyZWFtKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdGFydE5ldHdvcmtBbmFseXplclN0cmVhbScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHN0YXJ0IHRoZSBzaW5nbGUgd2lyZWxlc3MgZGV2aWNlIGltcG9ydCB0YXNrXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfU3RhcnRTaW5nbGVXaXJlbGVzc0RldmljZUltcG9ydFRhc2suaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU3RhcnRTaW5nbGVXaXJlbGVzc0RldmljZUltcG9ydFRhc2soKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1N0YXJ0U2luZ2xlV2lyZWxlc3NEZXZpY2VJbXBvcnRUYXNrJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3RhcnQgdGhlIHdpcmVsZXNzIGRldmljZSBpbXBvcnQgdGFza1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX1N0YXJ0V2lyZWxlc3NEZXZpY2VJbXBvcnRUYXNrLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0YXJ0V2lyZWxlc3NEZXZpY2VJbXBvcnRUYXNrKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdGFydFdpcmVsZXNzRGV2aWNlSW1wb3J0VGFzaycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHRhZyBhIGdpdmVuIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogVGFnZ2luZ1xuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9UYWdSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9UYWdSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVGFnUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzaW11bGF0ZSBhIHByb3Zpc2lvbmVkIGRldmljZSB0byBzZW5kIGFuIHVwbGluayBkYXRhIHdpdGggcGF5bG9hZCBvZiAnSGVsbG8nXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfVGVzdFdpcmVsZXNzRGV2aWNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1Rlc3RXaXJlbGVzc0RldmljZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVGVzdFdpcmVsZXNzRGV2aWNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVtb3ZlIHRoZSBnaXZlbiB0YWdzIGZyb20gdGhlIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogVGFnZ2luZ1xuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9VbnRhZ1Jlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VudGFnUmVzb3VyY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VudGFnUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSBEZXN0aW5hdGlvbiByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX1VwZGF0ZURlc3RpbmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZURlc3RpbmF0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVEZXN0aW5hdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBldmVudCBjb25maWd1cmF0aW9uIGJ5IHJlc291cmNlIHR5cGVzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfVXBkYXRlRXZlbnRDb25maWd1cmF0aW9uQnlSZXNvdXJjZVR5cGVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUV2ZW50Q29uZmlndXJhdGlvbkJ5UmVzb3VyY2VUeXBlcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlRXZlbnRDb25maWd1cmF0aW9uQnlSZXNvdXJjZVR5cGVzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIHRoZSBGdW90YVRhc2tcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9VcGRhdGVGdW90YVRhc2suaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlRnVvdGFUYXNrKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVGdW90YVRhc2snKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgbG9nIGxldmVscyBieSByZXNvdXJjZSB0eXBlc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX1VwZGF0ZUxvZ0xldmVsc0J5UmVzb3VyY2VUeXBlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVMb2dMZXZlbHNCeVJlc291cmNlVHlwZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUxvZ0xldmVsc0J5UmVzb3VyY2VUeXBlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBtZXRyaWMgY29uZmlndXJhdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX1VwZGF0ZU1ldHJpY0NvbmZpZ3VyYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlTWV0cmljQ29uZmlndXJhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlTWV0cmljQ29uZmlndXJhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSB0aGUgTXVsdGljYXN0R3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9VcGRhdGVNdWx0aWNhc3RHcm91cC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVNdWx0aWNhc3RHcm91cCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlTXVsdGljYXN0R3JvdXAnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgdGhlIE5ldHdvcmtBbmFseXplckNvbmZpZ3VyYXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9VcGRhdGVOZXR3b3JrQW5hbHl6ZXJDb25maWd1cmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZU5ldHdvcmtBbmFseXplckNvbmZpZ3VyYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZU5ldHdvcmtBbmFseXplckNvbmZpZ3VyYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSBwYXJ0bmVyIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9VcGRhdGVQYXJ0bmVyQWNjb3VudC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVQYXJ0bmVyQWNjb3VudCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlUGFydG5lckFjY291bnQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgcG9zaXRpb24gZm9yIGEgZ2l2ZW4gcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9VcGRhdGVQb3NpdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVQb3NpdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlUG9zaXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYW4gZXZlbnQgY29uZmlndXJhdGlvbiBmb3IgYW4gaWRlbnRpZmllclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX1VwZGF0ZVJlc291cmNlRXZlbnRDb25maWd1cmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZVJlc291cmNlRXZlbnRDb25maWd1cmF0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVSZXNvdXJjZUV2ZW50Q29uZmlndXJhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBwb3NpdGlvbiBmb3IgYSBnaXZlbiByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX1VwZGF0ZVJlc291cmNlUG9zaXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlUmVzb3VyY2VQb3NpdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlUmVzb3VyY2VQb3NpdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhIFdpcmVsZXNzRGV2aWNlIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfVXBkYXRlV2lyZWxlc3NEZXZpY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlV2lyZWxlc3NEZXZpY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZVdpcmVsZXNzRGV2aWNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGEgd2lyZWxlc3MgZGV2aWNlIGltcG9ydCB0YXNrXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfVXBkYXRlV2lyZWxlc3NEZXZpY2VJbXBvcnRUYXNrLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZVdpcmVsZXNzRGV2aWNlSW1wb3J0VGFzaygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlV2lyZWxlc3NEZXZpY2VJbXBvcnRUYXNrJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGEgV2lyZWxlc3NHYXRld2F5IHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfVXBkYXRlV2lyZWxlc3NHYXRld2F5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZVdpcmVsZXNzR2F0ZXdheSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlV2lyZWxlc3NHYXRld2F5Jyk7XG4gIH1cblxuICBwcm90ZWN0ZWQgYWNjZXNzTGV2ZWxMaXN0OiBBY2Nlc3NMZXZlbExpc3QgPSB7XG4gICAgV3JpdGU6IFtcbiAgICAgICdBc3NvY2lhdGVBd3NBY2NvdW50V2l0aFBhcnRuZXJBY2NvdW50JyxcbiAgICAgICdBc3NvY2lhdGVNdWx0aWNhc3RHcm91cFdpdGhGdW90YVRhc2snLFxuICAgICAgJ0Fzc29jaWF0ZVdpcmVsZXNzRGV2aWNlV2l0aEZ1b3RhVGFzaycsXG4gICAgICAnQXNzb2NpYXRlV2lyZWxlc3NEZXZpY2VXaXRoTXVsdGljYXN0R3JvdXAnLFxuICAgICAgJ0Fzc29jaWF0ZVdpcmVsZXNzRGV2aWNlV2l0aFRoaW5nJyxcbiAgICAgICdBc3NvY2lhdGVXaXJlbGVzc0dhdGV3YXlXaXRoQ2VydGlmaWNhdGUnLFxuICAgICAgJ0Fzc29jaWF0ZVdpcmVsZXNzR2F0ZXdheVdpdGhUaGluZycsXG4gICAgICAnQ2FuY2VsTXVsdGljYXN0R3JvdXBTZXNzaW9uJyxcbiAgICAgICdDcmVhdGVEZXN0aW5hdGlvbicsXG4gICAgICAnQ3JlYXRlRGV2aWNlUHJvZmlsZScsXG4gICAgICAnQ3JlYXRlRnVvdGFUYXNrJyxcbiAgICAgICdDcmVhdGVNdWx0aWNhc3RHcm91cCcsXG4gICAgICAnQ3JlYXRlTmV0d29ya0FuYWx5emVyQ29uZmlndXJhdGlvbicsXG4gICAgICAnQ3JlYXRlU2VydmljZVByb2ZpbGUnLFxuICAgICAgJ0NyZWF0ZVdpcmVsZXNzRGV2aWNlJyxcbiAgICAgICdDcmVhdGVXaXJlbGVzc0dhdGV3YXknLFxuICAgICAgJ0NyZWF0ZVdpcmVsZXNzR2F0ZXdheVRhc2snLFxuICAgICAgJ0NyZWF0ZVdpcmVsZXNzR2F0ZXdheVRhc2tEZWZpbml0aW9uJyxcbiAgICAgICdEZWxldGVEZXN0aW5hdGlvbicsXG4gICAgICAnRGVsZXRlRGV2aWNlUHJvZmlsZScsXG4gICAgICAnRGVsZXRlRnVvdGFUYXNrJyxcbiAgICAgICdEZWxldGVNdWx0aWNhc3RHcm91cCcsXG4gICAgICAnRGVsZXRlTmV0d29ya0FuYWx5emVyQ29uZmlndXJhdGlvbicsXG4gICAgICAnRGVsZXRlUXVldWVkTWVzc2FnZXMnLFxuICAgICAgJ0RlbGV0ZVNlcnZpY2VQcm9maWxlJyxcbiAgICAgICdEZWxldGVXaXJlbGVzc0RldmljZScsXG4gICAgICAnRGVsZXRlV2lyZWxlc3NEZXZpY2VJbXBvcnRUYXNrJyxcbiAgICAgICdEZWxldGVXaXJlbGVzc0dhdGV3YXknLFxuICAgICAgJ0RlbGV0ZVdpcmVsZXNzR2F0ZXdheVRhc2snLFxuICAgICAgJ0RlbGV0ZVdpcmVsZXNzR2F0ZXdheVRhc2tEZWZpbml0aW9uJyxcbiAgICAgICdEZXJlZ2lzdGVyV2lyZWxlc3NEZXZpY2UnLFxuICAgICAgJ0Rpc2Fzc29jaWF0ZUF3c0FjY291bnRGcm9tUGFydG5lckFjY291bnQnLFxuICAgICAgJ0Rpc2Fzc29jaWF0ZU11bHRpY2FzdEdyb3VwRnJvbUZ1b3RhVGFzaycsXG4gICAgICAnRGlzYXNzb2NpYXRlV2lyZWxlc3NEZXZpY2VGcm9tRnVvdGFUYXNrJyxcbiAgICAgICdEaXNhc3NvY2lhdGVXaXJlbGVzc0RldmljZUZyb21NdWx0aWNhc3RHcm91cCcsXG4gICAgICAnRGlzYXNzb2NpYXRlV2lyZWxlc3NEZXZpY2VGcm9tVGhpbmcnLFxuICAgICAgJ0Rpc2Fzc29jaWF0ZVdpcmVsZXNzR2F0ZXdheUZyb21DZXJ0aWZpY2F0ZScsXG4gICAgICAnRGlzYXNzb2NpYXRlV2lyZWxlc3NHYXRld2F5RnJvbVRoaW5nJyxcbiAgICAgICdQdXRQb3NpdGlvbkNvbmZpZ3VyYXRpb24nLFxuICAgICAgJ1B1dFJlc291cmNlTG9nTGV2ZWwnLFxuICAgICAgJ1Jlc2V0QWxsUmVzb3VyY2VMb2dMZXZlbHMnLFxuICAgICAgJ1Jlc2V0UmVzb3VyY2VMb2dMZXZlbCcsXG4gICAgICAnU2VuZERhdGFUb011bHRpY2FzdEdyb3VwJyxcbiAgICAgICdTZW5kRGF0YVRvV2lyZWxlc3NEZXZpY2UnLFxuICAgICAgJ1N0YXJ0QnVsa0Fzc29jaWF0ZVdpcmVsZXNzRGV2aWNlV2l0aE11bHRpY2FzdEdyb3VwJyxcbiAgICAgICdTdGFydEJ1bGtEaXNhc3NvY2lhdGVXaXJlbGVzc0RldmljZUZyb21NdWx0aWNhc3RHcm91cCcsXG4gICAgICAnU3RhcnRGdW90YVRhc2snLFxuICAgICAgJ1N0YXJ0TXVsdGljYXN0R3JvdXBTZXNzaW9uJyxcbiAgICAgICdTdGFydE5ldHdvcmtBbmFseXplclN0cmVhbScsXG4gICAgICAnU3RhcnRTaW5nbGVXaXJlbGVzc0RldmljZUltcG9ydFRhc2snLFxuICAgICAgJ1N0YXJ0V2lyZWxlc3NEZXZpY2VJbXBvcnRUYXNrJyxcbiAgICAgICdUZXN0V2lyZWxlc3NEZXZpY2UnLFxuICAgICAgJ1VwZGF0ZURlc3RpbmF0aW9uJyxcbiAgICAgICdVcGRhdGVFdmVudENvbmZpZ3VyYXRpb25CeVJlc291cmNlVHlwZXMnLFxuICAgICAgJ1VwZGF0ZUZ1b3RhVGFzaycsXG4gICAgICAnVXBkYXRlTG9nTGV2ZWxzQnlSZXNvdXJjZVR5cGVzJyxcbiAgICAgICdVcGRhdGVNZXRyaWNDb25maWd1cmF0aW9uJyxcbiAgICAgICdVcGRhdGVNdWx0aWNhc3RHcm91cCcsXG4gICAgICAnVXBkYXRlTmV0d29ya0FuYWx5emVyQ29uZmlndXJhdGlvbicsXG4gICAgICAnVXBkYXRlUGFydG5lckFjY291bnQnLFxuICAgICAgJ1VwZGF0ZVBvc2l0aW9uJyxcbiAgICAgICdVcGRhdGVSZXNvdXJjZUV2ZW50Q29uZmlndXJhdGlvbicsXG4gICAgICAnVXBkYXRlUmVzb3VyY2VQb3NpdGlvbicsXG4gICAgICAnVXBkYXRlV2lyZWxlc3NEZXZpY2UnLFxuICAgICAgJ1VwZGF0ZVdpcmVsZXNzRGV2aWNlSW1wb3J0VGFzaycsXG4gICAgICAnVXBkYXRlV2lyZWxlc3NHYXRld2F5J1xuICAgIF0sXG4gICAgUmVhZDogW1xuICAgICAgJ0dldERlc3RpbmF0aW9uJyxcbiAgICAgICdHZXREZXZpY2VQcm9maWxlJyxcbiAgICAgICdHZXRFdmVudENvbmZpZ3VyYXRpb25CeVJlc291cmNlVHlwZXMnLFxuICAgICAgJ0dldEZ1b3RhVGFzaycsXG4gICAgICAnR2V0TG9nTGV2ZWxzQnlSZXNvdXJjZVR5cGVzJyxcbiAgICAgICdHZXRNZXRyaWNDb25maWd1cmF0aW9uJyxcbiAgICAgICdHZXRNZXRyaWNzJyxcbiAgICAgICdHZXRNdWx0aWNhc3RHcm91cCcsXG4gICAgICAnR2V0TXVsdGljYXN0R3JvdXBTZXNzaW9uJyxcbiAgICAgICdHZXROZXR3b3JrQW5hbHl6ZXJDb25maWd1cmF0aW9uJyxcbiAgICAgICdHZXRQYXJ0bmVyQWNjb3VudCcsXG4gICAgICAnR2V0UG9zaXRpb24nLFxuICAgICAgJ0dldFBvc2l0aW9uQ29uZmlndXJhdGlvbicsXG4gICAgICAnR2V0UG9zaXRpb25Fc3RpbWF0ZScsXG4gICAgICAnR2V0UmVzb3VyY2VFdmVudENvbmZpZ3VyYXRpb24nLFxuICAgICAgJ0dldFJlc291cmNlTG9nTGV2ZWwnLFxuICAgICAgJ0dldFJlc291cmNlUG9zaXRpb24nLFxuICAgICAgJ0dldFNlcnZpY2VFbmRwb2ludCcsXG4gICAgICAnR2V0U2VydmljZVByb2ZpbGUnLFxuICAgICAgJ0dldFdpcmVsZXNzRGV2aWNlJyxcbiAgICAgICdHZXRXaXJlbGVzc0RldmljZUltcG9ydFRhc2snLFxuICAgICAgJ0dldFdpcmVsZXNzRGV2aWNlU3RhdGlzdGljcycsXG4gICAgICAnR2V0V2lyZWxlc3NHYXRld2F5JyxcbiAgICAgICdHZXRXaXJlbGVzc0dhdGV3YXlDZXJ0aWZpY2F0ZScsXG4gICAgICAnR2V0V2lyZWxlc3NHYXRld2F5RmlybXdhcmVJbmZvcm1hdGlvbicsXG4gICAgICAnR2V0V2lyZWxlc3NHYXRld2F5U3RhdGlzdGljcycsXG4gICAgICAnR2V0V2lyZWxlc3NHYXRld2F5VGFzaycsXG4gICAgICAnR2V0V2lyZWxlc3NHYXRld2F5VGFza0RlZmluaXRpb24nLFxuICAgICAgJ0xpc3REZXN0aW5hdGlvbnMnLFxuICAgICAgJ0xpc3REZXZpY2VQcm9maWxlcycsXG4gICAgICAnTGlzdERldmljZXNGb3JXaXJlbGVzc0RldmljZUltcG9ydFRhc2snLFxuICAgICAgJ0xpc3RFdmVudENvbmZpZ3VyYXRpb25zJyxcbiAgICAgICdMaXN0RnVvdGFUYXNrcycsXG4gICAgICAnTGlzdE11bHRpY2FzdEdyb3VwcycsXG4gICAgICAnTGlzdE11bHRpY2FzdEdyb3Vwc0J5RnVvdGFUYXNrJyxcbiAgICAgICdMaXN0TmV0d29ya0FuYWx5emVyQ29uZmlndXJhdGlvbnMnLFxuICAgICAgJ0xpc3RQYXJ0bmVyQWNjb3VudHMnLFxuICAgICAgJ0xpc3RQb3NpdGlvbkNvbmZpZ3VyYXRpb25zJyxcbiAgICAgICdMaXN0UXVldWVkTWVzc2FnZXMnLFxuICAgICAgJ0xpc3RTZXJ2aWNlUHJvZmlsZXMnLFxuICAgICAgJ0xpc3RUYWdzRm9yUmVzb3VyY2UnLFxuICAgICAgJ0xpc3RXaXJlbGVzc0RldmljZUltcG9ydFRhc2tzJyxcbiAgICAgICdMaXN0V2lyZWxlc3NEZXZpY2VzJyxcbiAgICAgICdMaXN0V2lyZWxlc3NHYXRld2F5VGFza0RlZmluaXRpb25zJyxcbiAgICAgICdMaXN0V2lyZWxlc3NHYXRld2F5cydcbiAgICBdLFxuICAgIFRhZ2dpbmc6IFtcbiAgICAgICdUYWdSZXNvdXJjZScsXG4gICAgICAnVW50YWdSZXNvdXJjZSdcbiAgICBdXG4gIH07XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIFdpcmVsZXNzRGV2aWNlIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfQ3JlYXRlV2lyZWxlc3NEZXZpY2UuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gd2lyZWxlc3NEZXZpY2VJZCAtIElkZW50aWZpZXIgZm9yIHRoZSB3aXJlbGVzc0RldmljZUlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25XaXJlbGVzc0RldmljZSh3aXJlbGVzc0RldmljZUlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06aW90d2lyZWxlc3M6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OldpcmVsZXNzRGV2aWNlLyR7IHdpcmVsZXNzRGV2aWNlSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIFdpcmVsZXNzR2F0ZXdheSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX0NyZWF0ZVdpcmVsZXNzR2F0ZXdheS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSB3aXJlbGVzc0dhdGV3YXlJZCAtIElkZW50aWZpZXIgZm9yIHRoZSB3aXJlbGVzc0dhdGV3YXlJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uV2lyZWxlc3NHYXRld2F5KHdpcmVsZXNzR2F0ZXdheUlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06aW90d2lyZWxlc3M6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OldpcmVsZXNzR2F0ZXdheS8keyB3aXJlbGVzc0dhdGV3YXlJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgRGV2aWNlUHJvZmlsZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX0NyZWF0ZURldmljZVByb2ZpbGUuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gZGV2aWNlUHJvZmlsZUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGRldmljZVByb2ZpbGVJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRGV2aWNlUHJvZmlsZShkZXZpY2VQcm9maWxlSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTppb3R3aXJlbGVzczokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06RGV2aWNlUHJvZmlsZS8keyBkZXZpY2VQcm9maWxlSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIFNlcnZpY2VQcm9maWxlIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfQ3JlYXRlU2VydmljZVByb2ZpbGUuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gc2VydmljZVByb2ZpbGVJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBzZXJ2aWNlUHJvZmlsZUlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25TZXJ2aWNlUHJvZmlsZShzZXJ2aWNlUHJvZmlsZUlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06aW90d2lyZWxlc3M6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OlNlcnZpY2VQcm9maWxlLyR7IHNlcnZpY2VQcm9maWxlSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIERlc3RpbmF0aW9uIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfQ3JlYXRlRGVzdGluYXRpb24uaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gZGVzdGluYXRpb25OYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGRlc3RpbmF0aW9uTmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRGVzdGluYXRpb24oZGVzdGluYXRpb25OYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06aW90d2lyZWxlc3M6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OkRlc3RpbmF0aW9uLyR7IGRlc3RpbmF0aW9uTmFtZSB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgU2lkZXdhbGtBY2NvdW50IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfQXNzb2NpYXRlQXdzQWNjb3VudFdpdGhQYXJ0bmVyQWNjb3VudC5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBzaWRld2Fsa0FjY291bnRJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBzaWRld2Fsa0FjY291bnRJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uU2lkZXdhbGtBY2NvdW50KHNpZGV3YWxrQWNjb3VudElkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06aW90d2lyZWxlc3M6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OlNpZGV3YWxrQWNjb3VudC8keyBzaWRld2Fsa0FjY291bnRJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgV2lyZWxlc3NHYXRld2F5VGFza0RlZmluaXRpb24gdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90LXdpcmVsZXNzLzIwMjAtMTEtMjIvYXBpcmVmZXJlbmNlL0FQSV9DcmVhdGVXaXJlbGVzc0dhdGV3YXlUYXNrRGVmaW5pdGlvbi5odG1sXG4gICAqXG4gICAqIEBwYXJhbSB3aXJlbGVzc0dhdGV3YXlUYXNrRGVmaW5pdGlvbklkIC0gSWRlbnRpZmllciBmb3IgdGhlIHdpcmVsZXNzR2F0ZXdheVRhc2tEZWZpbml0aW9uSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbldpcmVsZXNzR2F0ZXdheVRhc2tEZWZpbml0aW9uKHdpcmVsZXNzR2F0ZXdheVRhc2tEZWZpbml0aW9uSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTppb3R3aXJlbGVzczokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06V2lyZWxlc3NHYXRld2F5VGFza0RlZmluaXRpb24vJHsgd2lyZWxlc3NHYXRld2F5VGFza0RlZmluaXRpb25JZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgRnVvdGFUYXNrIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfQ3JlYXRlRnVvdGFUYXNrLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGZ1b3RhVGFza0lkIC0gSWRlbnRpZmllciBmb3IgdGhlIGZ1b3RhVGFza0lkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25GdW90YVRhc2soZnVvdGFUYXNrSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTppb3R3aXJlbGVzczokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06RnVvdGFUYXNrLyR7IGZ1b3RhVGFza0lkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBNdWx0aWNhc3RHcm91cCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3Qtd2lyZWxlc3MvMjAyMC0xMS0yMi9hcGlyZWZlcmVuY2UvQVBJX0NyZWF0ZU11bHRpY2FzdEdyb3VwLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIG11bHRpY2FzdEdyb3VwSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgbXVsdGljYXN0R3JvdXBJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uTXVsdGljYXN0R3JvdXAobXVsdGljYXN0R3JvdXBJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmlvdHdpcmVsZXNzOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpNdWx0aWNhc3RHcm91cC8keyBtdWx0aWNhc3RHcm91cElkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBOZXR3b3JrQW5hbHl6ZXJDb25maWd1cmF0aW9uIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfQ3JlYXRlTmV0d29ya0FuYWx5emVyQ29uZmlndXJhdGlvbi5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBuZXR3b3JrQW5hbHl6ZXJDb25maWd1cmF0aW9uTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBuZXR3b3JrQW5hbHl6ZXJDb25maWd1cmF0aW9uTmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uTmV0d29ya0FuYWx5emVyQ29uZmlndXJhdGlvbihuZXR3b3JrQW5hbHl6ZXJDb25maWd1cmF0aW9uTmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmlvdHdpcmVsZXNzOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpOZXR3b3JrQW5hbHl6ZXJDb25maWd1cmF0aW9uLyR7IG5ldHdvcmtBbmFseXplckNvbmZpZ3VyYXRpb25OYW1lIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSB0aGluZyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3QvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3RoaW5nLXJlZ2lzdHJ5Lmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHRoaW5nTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSB0aGluZ05hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKi9cbiAgcHVibGljIG9uVGhpbmcodGhpbmdOYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06aW90OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTp0aGluZy8keyB0aGluZ05hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGNlcnQgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90L2xhdGVzdC9kZXZlbG9wZXJndWlkZS94NTA5LWNsaWVudC1jZXJ0cy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBjZXJ0aWZpY2F0ZSAtIElkZW50aWZpZXIgZm9yIHRoZSBjZXJ0aWZpY2F0ZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqL1xuICBwdWJsaWMgb25DZXJ0KGNlcnRpZmljYXRlOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06aW90OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpjZXJ0LyR7IGNlcnRpZmljYXRlIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBJbXBvcnRUYXNrIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC13aXJlbGVzcy8yMDIwLTExLTIyL2FwaXJlZmVyZW5jZS9BUElfU3RhcnRXaXJlbGVzc0RldmljZUltcG9ydFRhc2suaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gaW1wb3J0VGFza0lkIC0gSWRlbnRpZmllciBmb3IgdGhlIGltcG9ydFRhc2tJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uSW1wb3J0VGFzayhpbXBvcnRUYXNrSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTppb3R3aXJlbGVzczokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06SW1wb3J0VGFzay8keyBpbXBvcnRUYXNrSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IGEgdGFnIGtleSB0aGF0IGlzIHByZXNlbnQgaW4gdGhlIHJlcXVlc3QgdGhhdCB0aGUgdXNlciBtYWtlcyB0byBJb1QgV2lyZWxlc3NcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2lvdC9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvdGFnZ2luZy1pb3QtaWFtLmh0bWwvXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9Bc3NvY2lhdGVBd3NBY2NvdW50V2l0aFBhcnRuZXJBY2NvdW50KClcbiAgICogLSAudG9DcmVhdGVEZXN0aW5hdGlvbigpXG4gICAqIC0gLnRvQ3JlYXRlRGV2aWNlUHJvZmlsZSgpXG4gICAqIC0gLnRvQ3JlYXRlRnVvdGFUYXNrKClcbiAgICogLSAudG9DcmVhdGVNdWx0aWNhc3RHcm91cCgpXG4gICAqIC0gLnRvQ3JlYXRlTmV0d29ya0FuYWx5emVyQ29uZmlndXJhdGlvbigpXG4gICAqIC0gLnRvQ3JlYXRlU2VydmljZVByb2ZpbGUoKVxuICAgKiAtIC50b0NyZWF0ZVdpcmVsZXNzRGV2aWNlKClcbiAgICogLSAudG9DcmVhdGVXaXJlbGVzc0dhdGV3YXkoKVxuICAgKiAtIC50b0NyZWF0ZVdpcmVsZXNzR2F0ZXdheVRhc2tEZWZpbml0aW9uKClcbiAgICogLSAudG9TdGFydFNpbmdsZVdpcmVsZXNzRGV2aWNlSW1wb3J0VGFzaygpXG4gICAqIC0gLnRvU3RhcnRXaXJlbGVzc0RldmljZUltcG9ydFRhc2soKVxuICAgKiAtIC50b1RhZ1Jlc291cmNlKClcbiAgICpcbiAgICogQHBhcmFtIHRhZ0tleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1JlcXVlc3RUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6UmVxdWVzdFRhZy8keyB0YWdLZXkgfWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRhZyBrZXkgY29tcG9uZW50IG9mIGEgdGFnIGF0dGFjaGVkIHRvIGFuIElvVCBXaXJlbGVzcyByZXNvdXJjZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vaW90L2xhdGVzdC9kZXZlbG9wZXJndWlkZS90YWdnaW5nLWlvdC1pYW0uaHRtbC9cbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBXaXJlbGVzc0RldmljZVxuICAgKiAtIFdpcmVsZXNzR2F0ZXdheVxuICAgKiAtIERldmljZVByb2ZpbGVcbiAgICogLSBTZXJ2aWNlUHJvZmlsZVxuICAgKiAtIERlc3RpbmF0aW9uXG4gICAqIC0gU2lkZXdhbGtBY2NvdW50XG4gICAqIC0gV2lyZWxlc3NHYXRld2F5VGFza0RlZmluaXRpb25cbiAgICogLSBGdW90YVRhc2tcbiAgICogLSBNdWx0aWNhc3RHcm91cFxuICAgKiAtIE5ldHdvcmtBbmFseXplckNvbmZpZ3VyYXRpb25cbiAgICogLSBJbXBvcnRUYXNrXG4gICAqXG4gICAqIEBwYXJhbSB0YWdLZXkgVGhlIHRhZyBrZXkgdG8gY2hlY2tcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NSZXNvdXJjZVRhZyh0YWdLZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpSZXNvdXJjZVRhZy8keyB0YWdLZXkgfWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSBsaXN0IG9mIGFsbCB0aGUgdGFnIGtleSBuYW1lcyBhc3NvY2lhdGVkIHdpdGggdGhlIHJlc291cmNlIGluIHRoZSByZXF1ZXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9pb3QvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3RhZ2dpbmctaW90LWlhbS5odG1sL1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQXNzb2NpYXRlQXdzQWNjb3VudFdpdGhQYXJ0bmVyQWNjb3VudCgpXG4gICAqIC0gLnRvQ3JlYXRlRGVzdGluYXRpb24oKVxuICAgKiAtIC50b0NyZWF0ZURldmljZVByb2ZpbGUoKVxuICAgKiAtIC50b0NyZWF0ZUZ1b3RhVGFzaygpXG4gICAqIC0gLnRvQ3JlYXRlTXVsdGljYXN0R3JvdXAoKVxuICAgKiAtIC50b0NyZWF0ZU5ldHdvcmtBbmFseXplckNvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b0NyZWF0ZVNlcnZpY2VQcm9maWxlKClcbiAgICogLSAudG9DcmVhdGVXaXJlbGVzc0RldmljZSgpXG4gICAqIC0gLnRvQ3JlYXRlV2lyZWxlc3NHYXRld2F5KClcbiAgICogLSAudG9DcmVhdGVXaXJlbGVzc0dhdGV3YXlUYXNrRGVmaW5pdGlvbigpXG4gICAqIC0gLnRvU3RhcnRTaW5nbGVXaXJlbGVzc0RldmljZUltcG9ydFRhc2soKVxuICAgKiAtIC50b1N0YXJ0V2lyZWxlc3NEZXZpY2VJbXBvcnRUYXNrKClcbiAgICogLSAudG9UYWdSZXNvdXJjZSgpXG4gICAqIC0gLnRvVW50YWdSZXNvdXJjZSgpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzVGFnS2V5cyh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlRhZ0tleXNgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW2lvdHdpcmVsZXNzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hd3Npb3R3aXJlbGVzcy5odG1sKS5cbiAgICpcbiAgICovXG4gIGNvbnN0cnVjdG9yKHByb3BzPzogaWFtLlBvbGljeVN0YXRlbWVudFByb3BzKSB7XG4gICAgc3VwZXIocHJvcHMpO1xuICB9XG59XG4iXX0=