"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Kinesis = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [kinesis](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonkinesisdatastreams.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Kinesis extends shared_1.PolicyStatement {
    /**
     * Grants permission to add or update tags for the specified Amazon Kinesis stream. Each stream can have up to 50 tags
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_AddTagsToStream.html
     */
    toAddTagsToStream() {
        return this.to('AddTagsToStream');
    }
    /**
     * Grants permission to create a Amazon Kinesis stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_CreateStream.html
     */
    toCreateStream() {
        return this.to('CreateStream');
    }
    /**
     * Grants permission to decrease the stream's retention period, which is the length of time data records are accessible after they are added to the stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DecreaseStreamRetentionPeriod.html
     */
    toDecreaseStreamRetentionPeriod() {
        return this.to('DecreaseStreamRetentionPeriod');
    }
    /**
     * Grants permission to delete a resource policy associated with a specified stream or consumer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete a stream and all its shards and data
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DeleteStream.html
     */
    toDeleteStream() {
        return this.to('DeleteStream');
    }
    /**
     * Grants permission to deregister a stream consumer with a Kinesis data stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DeregisterStreamConsumer.html
     */
    toDeregisterStreamConsumer() {
        return this.to('DeregisterStreamConsumer');
    }
    /**
     * Grants permission to describe the shard limits and usage for the account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DescribeLimits.html
     */
    toDescribeLimits() {
        return this.to('DescribeLimits');
    }
    /**
     * Grants permission to describe the specified stream
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DescribeStream.html
     */
    toDescribeStream() {
        return this.to('DescribeStream');
    }
    /**
     * Grants permission to get the description of a registered stream consumer
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DescribeStreamConsumer.html
     */
    toDescribeStreamConsumer() {
        return this.to('DescribeStreamConsumer');
    }
    /**
     * Grants permission to provide a summarized description of the specified Kinesis data stream without the shard list
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DescribeStreamSummary.html
     */
    toDescribeStreamSummary() {
        return this.to('DescribeStreamSummary');
    }
    /**
     * Grants permission to disables enhanced monitoring
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_DisableEnhancedMonitoring.html
     */
    toDisableEnhancedMonitoring() {
        return this.to('DisableEnhancedMonitoring');
    }
    /**
     * Grants permission to enable enhanced Kinesis data stream monitoring for shard-level metrics
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_EnableEnhancedMonitoring.html
     */
    toEnableEnhancedMonitoring() {
        return this.to('EnableEnhancedMonitoring');
    }
    /**
     * Grants permission to get data records from a shard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_GetRecords.html
     */
    toGetRecords() {
        return this.to('GetRecords');
    }
    /**
     * Grants permission to get a resource policy associated with a specified stream or consumer
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_GetResourcePolicy.html
     */
    toGetResourcePolicy() {
        return this.to('GetResourcePolicy');
    }
    /**
     * Grants permission to get a shard iterator. A shard iterator expires five minutes after it is returned to the requester
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_GetShardIterator.html
     */
    toGetShardIterator() {
        return this.to('GetShardIterator');
    }
    /**
     * Grants permission to increase the stream's retention period, which is the length of time data records are accessible after they are added to the stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_IncreaseStreamRetentionPeriod.html
     */
    toIncreaseStreamRetentionPeriod() {
        return this.to('IncreaseStreamRetentionPeriod');
    }
    /**
     * Grants permission to list the shards in a stream and provides information about each shard
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_ListShards.html
     */
    toListShards() {
        return this.to('ListShards');
    }
    /**
     * Grants permission to list the stream consumers registered to receive data from a Kinesis stream using enhanced fan-out, and provides information about each consumer
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_ListStreamConsumers.html
     */
    toListStreamConsumers() {
        return this.to('ListStreamConsumers');
    }
    /**
     * Grants permission to list your streams
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_ListStreams.html
     */
    toListStreams() {
        return this.to('ListStreams');
    }
    /**
     * Grants permission to list the tags for the specified Amazon Kinesis resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list the tags for the specified Amazon Kinesis stream
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_ListTagsForStream.html
     */
    toListTagsForStream() {
        return this.to('ListTagsForStream');
    }
    /**
     * Grants permission to merge two adjacent shards in a stream and combines them into a single shard to reduce the stream's capacity to ingest and transport data
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_MergeShards.html
     */
    toMergeShards() {
        return this.to('MergeShards');
    }
    /**
     * Grants permission to write a single data record from a producer into an Amazon Kinesis stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_PutRecord.html
     */
    toPutRecord() {
        return this.to('PutRecord');
    }
    /**
     * Grants permission to write multiple data records from a producer into an Amazon Kinesis stream in a single call (also referred to as a PutRecords request)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_PutRecords.html
     */
    toPutRecords() {
        return this.to('PutRecords');
    }
    /**
     * Grants permission to attach a resource policy to a specified stream or consumer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to register a stream consumer with a Kinesis data stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_RegisterStreamConsumer.html
     */
    toRegisterStreamConsumer() {
        return this.to('RegisterStreamConsumer');
    }
    /**
     * Grants permission to remove tags from the specified Kinesis data stream. Removed tags are deleted and cannot be recovered after this operation successfully completes
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_RemoveTagsFromStream.html
     */
    toRemoveTagsFromStream() {
        return this.to('RemoveTagsFromStream');
    }
    /**
     * Grants permission to split a shard into two new shards in the Kinesis data stream, to increase the stream's capacity to ingest and transport data
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_SplitShard.html
     */
    toSplitShard() {
        return this.to('SplitShard');
    }
    /**
     * Grants permission to enable or update server-side encryption using an AWS KMS key for a specified stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_StartStreamEncryption.html
     */
    toStartStreamEncryption() {
        return this.to('StartStreamEncryption');
    }
    /**
     * Grants permission to disable server-side encryption for a specified stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_StopStreamEncryption.html
     */
    toStopStreamEncryption() {
        return this.to('StopStreamEncryption');
    }
    /**
     * Grants permission to listen to a specific shard with enhanced fan-out
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_SubscribeToShard.html
     */
    toSubscribeToShard() {
        return this.to('SubscribeToShard');
    }
    /**
     * Grants permission to add or update tags for the specified Amazon Kinesis resource. Each resource can have up to 50 tags
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from the specified Kinesis data resource. Removed tags are deleted and cannot be recovered after this operation successfully completes
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the shard count of the specified stream to the specified number of shards
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_UpdateShardCount.html
     */
    toUpdateShardCount() {
        return this.to('UpdateShardCount');
    }
    /**
     * Grants permission to update the capacity mode of the data stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesis/latest/APIReference/API_UpdateStreamMode.html
     */
    toUpdateStreamMode() {
        return this.to('UpdateStreamMode');
    }
    /**
     * Adds a resource of type stream to the statement
     *
     * https://docs.aws.amazon.com/kinesis/latest/dev/amazon-kinesis-streams.html
     *
     * @param streamName - Identifier for the streamName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStream(streamName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:kinesis:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:stream/${streamName}`);
    }
    /**
     * Adds a resource of type consumer to the statement
     *
     * https://docs.aws.amazon.com/kinesis/latest/dev/amazon-kinesis-consumers.html
     *
     * @param streamType - Identifier for the streamType.
     * @param streamName - Identifier for the streamName.
     * @param consumerName - Identifier for the consumerName.
     * @param consumerCreationTimpstamp - Identifier for the consumerCreationTimpstamp.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onConsumer(streamType, streamName, consumerName, consumerCreationTimpstamp, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:kinesis:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:${streamType}/${streamName}/consumer/${consumerName}:${consumerCreationTimpstamp}`);
    }
    /**
     * Adds a resource of type kmsKey to the statement
     *
     * https://docs.aws.amazon.com/kinesis/latest/dev/concepts.html#kms_keys
     *
     * @param keyId - Identifier for the keyId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onKmsKey(keyId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:kms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:key/${keyId}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toAddTagsToStream()
     * - .toCreateStream()
     * - .toRegisterStreamConsumer()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toAddTagsToStream()
     * - .toCreateStream()
     * - .toDecreaseStreamRetentionPeriod()
     * - .toDeleteResourcePolicy()
     * - .toDeleteStream()
     * - .toDeregisterStreamConsumer()
     * - .toDescribeStream()
     * - .toDescribeStreamConsumer()
     * - .toDescribeStreamSummary()
     * - .toGetRecords()
     * - .toGetResourcePolicy()
     * - .toGetShardIterator()
     * - .toIncreaseStreamRetentionPeriod()
     * - .toListShards()
     * - .toListStreamConsumers()
     * - .toListTagsForResource()
     * - .toListTagsForStream()
     * - .toMergeShards()
     * - .toPutRecord()
     * - .toPutRecords()
     * - .toPutResourcePolicy()
     * - .toRegisterStreamConsumer()
     * - .toRemoveTagsFromStream()
     * - .toSplitShard()
     * - .toStartStreamEncryption()
     * - .toStopStreamEncryption()
     * - .toSubscribeToShard()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - stream
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toAddTagsToStream()
     * - .toCreateStream()
     * - .toRegisterStreamConsumer()
     * - .toRemoveTagsFromStream()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [kinesis](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonkinesisdatastreams.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'kinesis';
        this.accessLevelList = {
            Tagging: [
                'AddTagsToStream',
                'RemoveTagsFromStream',
                'TagResource',
                'UntagResource'
            ],
            Write: [
                'CreateStream',
                'DecreaseStreamRetentionPeriod',
                'DeleteResourcePolicy',
                'DeleteStream',
                'DeregisterStreamConsumer',
                'DisableEnhancedMonitoring',
                'EnableEnhancedMonitoring',
                'IncreaseStreamRetentionPeriod',
                'MergeShards',
                'PutRecord',
                'PutRecords',
                'PutResourcePolicy',
                'RegisterStreamConsumer',
                'SplitShard',
                'StartStreamEncryption',
                'StopStreamEncryption',
                'UpdateShardCount',
                'UpdateStreamMode'
            ],
            Read: [
                'DescribeLimits',
                'DescribeStream',
                'DescribeStreamConsumer',
                'DescribeStreamSummary',
                'GetRecords',
                'GetResourcePolicy',
                'GetShardIterator',
                'ListTagsForResource',
                'ListTagsForStream',
                'SubscribeToShard'
            ],
            List: [
                'ListShards',
                'ListStreamConsumers',
                'ListStreams'
            ]
        };
    }
}
exports.Kinesis = Kinesis;
//# sourceMappingURL=data:application/json;base64,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