"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VendorInsights = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [vendor-insights](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsmarketplacevendorinsights.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class VendorInsights extends shared_1.PolicyStatement {
    /**
     * Grants permission to activate the security profile
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/marketplace/latest/userguide/vendor-insights-seller-controlling-access.html
     */
    toActivateSecurityProfile() {
        return this.to('ActivateSecurityProfile');
    }
    /**
     * Grants permission to associate security profile with a data source
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - vendor-insights:GetDataSource
     *
     * https://docs.aws.amazon.com/marketplace/latest/userguide/vendor-insights-seller-controlling-access.html
     */
    toAssociateDataSource() {
        return this.to('AssociateDataSource');
    }
    /**
     * Grants permission to create a new data source
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - vendor-insights:TagResource
     *
     * https://docs.aws.amazon.com/marketplace/latest/userguide/vendor-insights-seller-controlling-access.html
     */
    toCreateDataSource() {
        return this.to('CreateDataSource');
    }
    /**
     * Grants permission to create a new security profile
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - vendor-insights:TagResource
     *
     * https://docs.aws.amazon.com/marketplace/latest/userguide/vendor-insights-seller-controlling-access.html
     */
    toCreateSecurityProfile() {
        return this.to('CreateSecurityProfile');
    }
    /**
     * Grants permission to deactivate the security profile
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/marketplace/latest/userguide/vendor-insights-seller-controlling-access.html
     */
    toDeactivateSecurityProfile() {
        return this.to('DeactivateSecurityProfile');
    }
    /**
     * Grants permission to delete a data source
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/marketplace/latest/userguide/vendor-insights-seller-controlling-access.html
     */
    toDeleteDataSource() {
        return this.to('DeleteDataSource');
    }
    /**
     * Grants permission to disassociate security profile from a data source
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - vendor-insights:GetDataSource
     *
     * https://docs.aws.amazon.com/marketplace/latest/userguide/vendor-insights-seller-controlling-access.html
     */
    toDisassociateDataSource() {
        return this.to('DisassociateDataSource');
    }
    /**
     * Grants permission to retrieve the details of an existing data source
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/marketplace/latest/userguide/vendor-insights-seller-controlling-access.html
     */
    toGetDataSource() {
        return this.to('GetDataSource');
    }
    /**
     * Grants permission to return the details of a security profile snapshot that requester is entitled to read
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/marketplace/latest/buyerguide/buyer-vendor-insights-controlling-access.html
     */
    toGetEntitledSecurityProfileSnapshot() {
        return this.to('GetEntitledSecurityProfileSnapshot');
    }
    /**
     * Grants permission to get the access terms for a vendor insights profile
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/marketplace/latest/buyerguide/buyer-vendor-insights-controlling-access.html
     */
    toGetProfileAccessTerms() {
        return this.to('GetProfileAccessTerms');
    }
    /**
     * Grants permission to return the details of an existing security profile
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/marketplace/latest/userguide/vendor-insights-seller-controlling-access.html
     */
    toGetSecurityProfile() {
        return this.to('GetSecurityProfile');
    }
    /**
     * Grants permission to return the details of a security profile snapshot
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/marketplace/latest/userguide/vendor-insights-seller-controlling-access.html
     */
    toGetSecurityProfileSnapshot() {
        return this.to('GetSecurityProfileSnapshot');
    }
    /**
     * Grants permission to list existing data sources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/marketplace/latest/userguide/vendor-insights-seller-controlling-access.html
     */
    toListDataSources() {
        return this.to('ListDataSources');
    }
    /**
     * Grants permission to return the snapshot summary list for an existing security profile that requester is entitled to list
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/marketplace/latest/buyerguide/buyer-vendor-insights-controlling-access.html
     */
    toListEntitledSecurityProfileSnapshots() {
        return this.to('ListEntitledSecurityProfileSnapshots');
    }
    /**
     * Grants permission to list entitled security profiles
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/marketplace/latest/buyerguide/buyer-vendor-insights-controlling-access.html
     */
    toListEntitledSecurityProfiles() {
        return this.to('ListEntitledSecurityProfiles');
    }
    /**
     * Grants permission to return the snapshot summary list for an existing security profile
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/marketplace/latest/userguide/vendor-insights-seller-controlling-access.html
     */
    toListSecurityProfileSnapshots() {
        return this.to('ListSecurityProfileSnapshots');
    }
    /**
     * Grants permission to list existing security profiles
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/marketplace/latest/userguide/vendor-insights-seller-controlling-access.html
     */
    toListSecurityProfiles() {
        return this.to('ListSecurityProfiles');
    }
    /**
     * Grants permission to list tags for vendor insights resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/marketplace/latest/userguide/vendor-insights-seller-controlling-access.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to tag vendor insights resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/marketplace/latest/userguide/vendor-insights-seller-controlling-access.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag vendor insights resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/marketplace/latest/userguide/vendor-insights-seller-controlling-access.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an existing data source
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/marketplace/latest/userguide/vendor-insights-seller-controlling-access.html
     */
    toUpdateDataSource() {
        return this.to('UpdateDataSource');
    }
    /**
     * Grants permission to update the security profile
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/marketplace/latest/userguide/vendor-insights-seller-controlling-access.html
     */
    toUpdateSecurityProfile() {
        return this.to('UpdateSecurityProfile');
    }
    /**
     * Grants permission to update the security profile snapshot creation configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/marketplace/latest/userguide/vendor-insights-seller-controlling-access.html
     */
    toUpdateSecurityProfileSnapshotCreationConfiguration() {
        return this.to('UpdateSecurityProfileSnapshotCreationConfiguration');
    }
    /**
     * Grants permission to update the security profile snapshot release configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/marketplace/latest/userguide/vendor-insights-seller-controlling-access.html
     */
    toUpdateSecurityProfileSnapshotReleaseConfiguration() {
        return this.to('UpdateSecurityProfileSnapshotReleaseConfiguration');
    }
    /**
     * Adds a resource of type DataSource to the statement
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsmarketplacevendorinsights.html#awsmarketplacevendorinsights-resources-for-iam-policies
     *
     * @param resourceId - Identifier for the resourceId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     */
    onDataSource(resourceId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:vendor-insights:::data-source:${resourceId}`);
    }
    /**
     * Adds a resource of type SecurityProfile to the statement
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsmarketplacevendorinsights.html#awsmarketplacevendorinsights-resources-for-iam-policies
     *
     * @param resourceId - Identifier for the resourceId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     */
    onSecurityProfile(resourceId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:vendor-insights:::security-profile:${resourceId}`);
    }
    /**
     * Filters access by tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateDataSource()
     * - .toCreateSecurityProfile()
     * - .toTagResource()
     *
     * Applies to resource types:
     * - DataSource
     * - SecurityProfile
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toActivateSecurityProfile()
     * - .toAssociateDataSource()
     * - .toCreateDataSource()
     * - .toCreateSecurityProfile()
     * - .toDeactivateSecurityProfile()
     * - .toDeleteDataSource()
     * - .toDisassociateDataSource()
     * - .toGetDataSource()
     * - .toGetSecurityProfile()
     * - .toGetSecurityProfileSnapshot()
     * - .toListSecurityProfileSnapshots()
     * - .toListTagsForResource()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateDataSource()
     * - .toUpdateSecurityProfile()
     * - .toUpdateSecurityProfileSnapshotCreationConfiguration()
     * - .toUpdateSecurityProfileSnapshotReleaseConfiguration()
     *
     * Applies to resource types:
     * - DataSource
     * - SecurityProfile
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateDataSource()
     * - .toCreateSecurityProfile()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - DataSource
     * - SecurityProfile
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [vendor-insights](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsmarketplacevendorinsights.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'vendor-insights';
        this.accessLevelList = {
            Write: [
                'ActivateSecurityProfile',
                'AssociateDataSource',
                'CreateDataSource',
                'CreateSecurityProfile',
                'DeactivateSecurityProfile',
                'DeleteDataSource',
                'DisassociateDataSource',
                'UpdateDataSource',
                'UpdateSecurityProfile',
                'UpdateSecurityProfileSnapshotCreationConfiguration',
                'UpdateSecurityProfileSnapshotReleaseConfiguration'
            ],
            Read: [
                'GetDataSource',
                'GetEntitledSecurityProfileSnapshot',
                'GetProfileAccessTerms',
                'GetSecurityProfile',
                'GetSecurityProfileSnapshot',
                'ListTagsForResource'
            ],
            List: [
                'ListDataSources',
                'ListEntitledSecurityProfileSnapshots',
                'ListEntitledSecurityProfiles',
                'ListSecurityProfileSnapshots',
                'ListSecurityProfiles'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.VendorInsights = VendorInsights;
//# sourceMappingURL=data:application/json;base64,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