import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [ec2messages](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmessagedeliveryservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class Ec2messages extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to acknowledge a message, ensuring it will not be delivered again
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-setting-up-messageAPIs.html
     */
    toAcknowledgeMessage(): this;
    /**
     * Grants permission to delete a message
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-setting-up-messageAPIs.html
     */
    toDeleteMessage(): this;
    /**
     * Grants permission to fail a message, signifying the message could not be processed successfully, ensuring it cannot be replied to or delivered again
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-setting-up-messageAPIs.html
     */
    toFailMessage(): this;
    /**
     * Grants permission to route traffic to the correct endpoint based on the given destination for the messages
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-setting-up-messageAPIs.html
     */
    toGetEndpoint(): this;
    /**
     * Grants permission to deliver messages to clients/instances using long polling
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifSsmSourceInstanceARN()
     * - .ifEc2SourceInstanceARN()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-setting-up-messageAPIs.html
     */
    toGetMessages(): this;
    /**
     * Grants permission to send replies from clients/instances to upstream service
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSsmSourceInstanceARN()
     * - .ifEc2SourceInstanceARN()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/systems-manager-setting-up-messageAPIs.html
     */
    toSendReply(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Filters access by the ARN of the instance from which the request originated
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toGetMessages()
     * - .toSendReply()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifEc2SourceInstanceARN(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by verifying the Amazon Resource Name (ARN) of the AWS Systems Manager's managed instance from which the request is made. This key is not present when the request comes from the managed instance authenticated with an IAM role associated with EC2 instance profile
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssystemsmanager.html#awssystemsmanager-policy-keys
     *
     * Applies to actions:
     * - .toGetMessages()
     * - .toSendReply()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifSsmSourceInstanceARN(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [ec2messages](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmessagedeliveryservice.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
