"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RefactorSpaces = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [refactor-spaces](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsmigrationhubrefactorspaces.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class RefactorSpaces extends shared_1.PolicyStatement {
    /**
     * Grants permission to create an application within an environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_CreateApplication.html
     */
    toCreateApplication() {
        return this.to('CreateApplication');
    }
    /**
     * Grants permission to create an environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_CreateEnvironment.html
     */
    toCreateEnvironment() {
        return this.to('CreateEnvironment');
    }
    /**
     * Grants permission to create a route within an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifRouteCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifSourcePath()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_CreateRoute.html
     */
    toCreateRoute() {
        return this.to('CreateRoute');
    }
    /**
     * Grants permission to create a service within an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_CreateService.html
     */
    toCreateService() {
        return this.to('CreateService');
    }
    /**
     * Grants permission to delete an application from an environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_DeleteApplication.html
     */
    toDeleteApplication() {
        return this.to('DeleteApplication');
    }
    /**
     * Grants permission to delete an environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_DeleteEnvironment.html
     */
    toDeleteEnvironment() {
        return this.to('DeleteEnvironment');
    }
    /**
     * Grants permission to delete a resource policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete a route from an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifRouteCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifSourcePath()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_DeleteRoute.html
     */
    toDeleteRoute() {
        return this.to('DeleteRoute');
    }
    /**
     * Grants permission to delete a service from an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_DeleteService.html
     */
    toDeleteService() {
        return this.to('DeleteService');
    }
    /**
     * Grants permission to get more information about an application
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_GetApplication.html
     */
    toGetApplication() {
        return this.to('GetApplication');
    }
    /**
     * Grants permission to get more information for an environment
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_GetEnvironment.html
     */
    toGetEnvironment() {
        return this.to('GetEnvironment');
    }
    /**
     * Grants permission to get the details about a resource policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_GetResourcePolicy.html
     */
    toGetResourcePolicy() {
        return this.to('GetResourcePolicy');
    }
    /**
     * Grants permission to get more information about a route
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifRouteCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifSourcePath()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_GetRoute.html
     */
    toGetRoute() {
        return this.to('GetRoute');
    }
    /**
     * Grants permission to get more information about a service
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_GetService.html
     */
    toGetService() {
        return this.to('GetService');
    }
    /**
     * Grants permission to list all the applications in an environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_ListApplications.html
     */
    toListApplications() {
        return this.to('ListApplications');
    }
    /**
     * Grants permission to list all the VPCs for the environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_ListEnvironmentVpcs.html
     */
    toListEnvironmentVpcs() {
        return this.to('ListEnvironmentVpcs');
    }
    /**
     * Grants permission to list all environments
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_ListEnvironments.html
     */
    toListEnvironments() {
        return this.to('ListEnvironments');
    }
    /**
     * Grants permission to list all the routes in an application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_ListRoutes.html
     */
    toListRoutes() {
        return this.to('ListRoutes');
    }
    /**
     * Grants permission to list all the services in an environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_ListServices.html
     */
    toListServices() {
        return this.to('ListServices');
    }
    /**
     * Grants permission to list all the tags for a given resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to add a resource policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifRouteCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifSourcePath()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove a tag from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifRouteCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifSourcePath()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a route from an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifRouteCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifSourcePath()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/APIReference/API_UpdateRoute.html
     */
    toUpdateRoute() {
        return this.to('UpdateRoute');
    }
    /**
     * Adds a resource of type environment to the statement
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-resources
     *
     * @param environmentId - Identifier for the environmentId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEnvironment(environmentId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:refactor-spaces:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:environment/${environmentId}`);
    }
    /**
     * Adds a resource of type application to the statement
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-resources
     *
     * @param environmentId - Identifier for the environmentId.
     * @param applicationId - Identifier for the applicationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifApplicationCreatedByAccount()
     * - .ifCreatedByAccountIds()
     */
    onApplication(environmentId, applicationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:refactor-spaces:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:environment/${environmentId}/application/${applicationId}`);
    }
    /**
     * Adds a resource of type service to the statement
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-resources
     *
     * @param environmentId - Identifier for the environmentId.
     * @param applicationId - Identifier for the applicationId.
     * @param serviceId - Identifier for the serviceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifApplicationCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifServiceCreatedByAccount()
     */
    onService(environmentId, applicationId, serviceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:refactor-spaces:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:environment/${environmentId}/application/${applicationId}/service/${serviceId}`);
    }
    /**
     * Adds a resource of type route to the statement
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-resources
     *
     * @param environmentId - Identifier for the environmentId.
     * @param applicationId - Identifier for the applicationId.
     * @param routeId - Identifier for the routeId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifApplicationCreatedByAccount()
     * - .ifCreatedByAccountIds()
     * - .ifRouteCreatedByAccount()
     * - .ifServiceCreatedByAccount()
     * - .ifSourcePath()
     */
    onRoute(environmentId, applicationId, routeId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:refactor-spaces:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:environment/${environmentId}/application/${applicationId}/route/${routeId}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateApplication()
     * - .toCreateEnvironment()
     * - .toCreateRoute()
     * - .toCreateService()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toDeleteApplication()
     * - .toDeleteEnvironment()
     * - .toDeleteRoute()
     * - .toDeleteService()
     * - .toGetApplication()
     * - .toGetEnvironment()
     * - .toGetRoute()
     * - .toGetService()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateRoute()
     *
     * Applies to resource types:
     * - environment
     * - application
     * - service
     * - route
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateApplication()
     * - .toCreateEnvironment()
     * - .toCreateRoute()
     * - .toCreateService()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by restricting the action to only those accounts that created the application within an environment
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to actions:
     * - .toCreateApplication()
     * - .toCreateRoute()
     * - .toCreateService()
     * - .toDeleteApplication()
     * - .toDeleteRoute()
     * - .toDeleteService()
     * - .toGetApplication()
     * - .toGetRoute()
     * - .toGetService()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateRoute()
     *
     * Applies to resource types:
     * - application
     * - service
     * - route
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifApplicationCreatedByAccount(value, operator) {
        return this.if(`ApplicationCreatedByAccount`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the accounts that created the resource
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to actions:
     * - .toCreateApplication()
     * - .toCreateRoute()
     * - .toCreateService()
     * - .toDeleteApplication()
     * - .toDeleteRoute()
     * - .toDeleteService()
     * - .toGetApplication()
     * - .toGetRoute()
     * - .toGetService()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateRoute()
     *
     * Applies to resource types:
     * - application
     * - service
     * - route
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCreatedByAccountIds(value, operator) {
        return this.if(`CreatedByAccountIds`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by restricting the action to only those accounts that created the route within an application
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to actions:
     * - .toCreateRoute()
     * - .toDeleteRoute()
     * - .toGetRoute()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateRoute()
     *
     * Applies to resource types:
     * - route
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRouteCreatedByAccount(value, operator) {
        return this.if(`RouteCreatedByAccount`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by restricting the action to only those accounts that created the service within an application
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to actions:
     * - .toCreateRoute()
     * - .toCreateService()
     * - .toDeleteRoute()
     * - .toDeleteService()
     * - .toGetRoute()
     * - .toGetService()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateRoute()
     *
     * Applies to resource types:
     * - service
     * - route
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifServiceCreatedByAccount(value, operator) {
        return this.if(`ServiceCreatedByAccount`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the path of the route
     *
     * https://docs.aws.amazon.com/migrationhub-refactor-spaces/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to actions:
     * - .toCreateRoute()
     * - .toDeleteRoute()
     * - .toGetRoute()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateRoute()
     *
     * Applies to resource types:
     * - route
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSourcePath(value, operator) {
        return this.if(`SourcePath`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [refactor-spaces](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsmigrationhubrefactorspaces.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'refactor-spaces';
        this.accessLevelList = {
            Write: [
                'CreateApplication',
                'CreateEnvironment',
                'CreateRoute',
                'CreateService',
                'DeleteApplication',
                'DeleteEnvironment',
                'DeleteResourcePolicy',
                'DeleteRoute',
                'DeleteService',
                'PutResourcePolicy',
                'UpdateRoute'
            ],
            Read: [
                'GetApplication',
                'GetEnvironment',
                'GetResourcePolicy',
                'GetRoute',
                'GetService',
                'ListApplications',
                'ListEnvironmentVpcs',
                'ListEnvironments',
                'ListRoutes',
                'ListServices',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.RefactorSpaces = RefactorSpaces;
//# sourceMappingURL=data:application/json;base64,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