"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Es = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [es](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonopensearchservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Es extends shared_1.PolicyStatement {
    /**
     * Grants permission to the destination domain owner to accept an inbound cross-cluster search connection request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_AcceptInboundConnection.html
     */
    toAcceptInboundConnection() {
        return this.to('AcceptInboundConnection');
    }
    /**
     * Grants permission to the destination domain owner to accept an inbound cross-cluster search connection request. This permission is deprecated. Use AcceptInboundConnection instead
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_AcceptInboundConnection.html
     */
    toAcceptInboundCrossClusterSearchConnection() {
        return this.to('AcceptInboundCrossClusterSearchConnection');
    }
    /**
     * Grants permission to add the data source for the OpenSearch Service domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_AddDataSource.html
     */
    toAddDataSource() {
        return this.to('AddDataSource');
    }
    /**
     * Grants permission to add the data source for the provided OpenSearch arns
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_AddDirectQueryDataSource.html
     */
    toAddDirectQueryDataSource() {
        return this.to('AddDirectQueryDataSource');
    }
    /**
     * Grants permission to attach resource tags to an OpenSearch Service domain, data source, or application
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_AddTags.html
     */
    toAddTags() {
        return this.to('AddTags');
    }
    /**
     * Grants permission to associate a package with an OpenSearch Service domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_AssociatePackage.html
     */
    toAssociatePackage() {
        return this.to('AssociatePackage');
    }
    /**
     * Grants permission to associate multiple packages with an OpenSearch Service domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_AssociatePackages.html
     */
    toAssociatePackages() {
        return this.to('AssociatePackages');
    }
    /**
     * Grants permission to provide access to an Amazon OpenSearch Service domain through the use of an interface VPC endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_AuthorizeVpcEndpointAccess.html
     */
    toAuthorizeVpcEndpointAccess() {
        return this.to('AuthorizeVpcEndpointAccess');
    }
    /**
     * Grants permission to cancel a change on an OpenSearch Service domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_CancelDomainConfigChange.html
     */
    toCancelDomainConfigChange() {
        return this.to('CancelDomainConfigChange');
    }
    /**
     * Grants permission to cancel a service software update of a domain. This permission is deprecated. Use CancelServiceSoftwareUpdate instead
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_CancelServiceSoftwareUpdate.html
     */
    toCancelElasticsearchServiceSoftwareUpdate() {
        return this.to('CancelElasticsearchServiceSoftwareUpdate');
    }
    /**
     * Grants permission to cancel a service software update of a domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_CancelServiceSoftwareUpdate.html
     */
    toCancelServiceSoftwareUpdate() {
        return this.to('CancelServiceSoftwareUpdate');
    }
    /**
     * Grants permission to create an OpenSearch Application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_CreateApplication.html
     */
    toCreateApplication() {
        return this.to('CreateApplication');
    }
    /**
     * Grants permission to create an Amazon OpenSearch Service domain
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_CreateDomain.html
     */
    toCreateDomain() {
        return this.to('CreateDomain');
    }
    /**
     * Grants permission to create an OpenSearch Service domain. This permission is deprecated. Use CreateDomain instead
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_CreateDomain.html
     */
    toCreateElasticsearchDomain() {
        return this.to('CreateElasticsearchDomain');
    }
    /**
     * Grants permission to create the service-linked role required for OpenSearch Service domains that use VPC access. This permission is deprecated. OpenSearch Service creates the service-linked role for you
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/Welcome.html
     */
    toCreateElasticsearchServiceRole() {
        return this.to('CreateElasticsearchServiceRole');
    }
    /**
     * Grants permission to create a new cross-cluster search connection from a source domain to a destination domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_CreateOutboundConnection.html
     */
    toCreateOutboundConnection() {
        return this.to('CreateOutboundConnection');
    }
    /**
     * Grants permission to create a new cross-cluster search connection from a source domain to a destination domain. This permission is deprecated. Use CreateOutboundConnection instead
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_CreateOutboundConnection.html
     */
    toCreateOutboundCrossClusterSearchConnection() {
        return this.to('CreateOutboundCrossClusterSearchConnection');
    }
    /**
     * Grants permission to add a package for use with OpenSearch Service domains
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_CreatePackage.html
     */
    toCreatePackage() {
        return this.to('CreatePackage');
    }
    /**
     * Grants permission to create the service-linked role required for Amazon OpenSearch Service domains that use VPC access
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/Welcome.html
     */
    toCreateServiceRole() {
        return this.to('CreateServiceRole');
    }
    /**
     * Grants permission to create an Amazon OpenSearch Service-managed VPC endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_CreateVpcEndpoint.html
     */
    toCreateVpcEndpoint() {
        return this.to('CreateVpcEndpoint');
    }
    /**
     * Grants permission to delete an OpenSearch Application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DeleteApplication.html
     */
    toDeleteApplication() {
        return this.to('DeleteApplication');
    }
    /**
     * Grants permission to delete the data source for the OpenSearch Service domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DeleteDataSource.html
     */
    toDeleteDataSource() {
        return this.to('DeleteDataSource');
    }
    /**
     * Grants permission to delete the data source for the provided OpenSearch arns
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DeleteDirectQueryDataSource.html
     */
    toDeleteDirectQueryDataSource() {
        return this.to('DeleteDirectQueryDataSource');
    }
    /**
     * Grants permission to delete an Amazon OpenSearch Service domain and all of its data
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DeleteDomain.html
     */
    toDeleteDomain() {
        return this.to('DeleteDomain');
    }
    /**
     * Grants permission to delete an OpenSearch Service domain and all of its data. This permission is deprecated. Use DeleteDomain instead
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DeleteDomain.html
     */
    toDeleteElasticsearchDomain() {
        return this.to('DeleteElasticsearchDomain');
    }
    /**
     * Grants permission to delete the service-linked role required for OpenSearch Service domains that use VPC access. This permission is deprecated. Use the IAM API to delete service-linked roles
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_Welcome.html
     */
    toDeleteElasticsearchServiceRole() {
        return this.to('DeleteElasticsearchServiceRole');
    }
    /**
     * Grants permission to the destination domain owner to delete an existing inbound cross-cluster search connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DeleteInboundConnection.html
     */
    toDeleteInboundConnection() {
        return this.to('DeleteInboundConnection');
    }
    /**
     * Grants permission to the destination domain owner to delete an existing inbound cross-cluster search connection. This permission is deprecated. Use DeleteInboundConnection instead
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DeleteInboundConnection.html
     */
    toDeleteInboundCrossClusterSearchConnection() {
        return this.to('DeleteInboundCrossClusterSearchConnection');
    }
    /**
     * Grants permission to the source domain owner to delete an existing outbound cross-cluster search connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DeleteOutboundConnection.html
     */
    toDeleteOutboundConnection() {
        return this.to('DeleteOutboundConnection');
    }
    /**
     * Grants permission to the source domain owner to delete an existing outbound cross-cluster search connection. This permission is deprecated. Use DeleteOutboundConnection instead
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DeleteOutboundConnection.html
     */
    toDeleteOutboundCrossClusterSearchConnection() {
        return this.to('DeleteOutboundCrossClusterSearchConnection');
    }
    /**
     * Grants permission to delete a package from OpenSearch Service. The package cannot be associated with any domains
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DeletePackage.html
     */
    toDeletePackage() {
        return this.to('DeletePackage');
    }
    /**
     * Grants permission to delete an Amazon OpenSearch Service-managed interface VPC endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DeleteVpcEndpoint.html
     */
    toDeleteVpcEndpoint() {
        return this.to('DeleteVpcEndpoint');
    }
    /**
     * Grants permission to view a description of the domain configuration for the specified OpenSearch Service domain, including the domain ID, service endpoint, and ARN
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DescribeDomain.html
     */
    toDescribeDomain() {
        return this.to('DescribeDomain');
    }
    /**
     * Grants permission to view the Auto-Tune configuration of the domain for the specified OpenSearch Service domain, including the Auto-Tune state and maintenance schedules
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DescribeDomainAutoTunes.html
     */
    toDescribeDomainAutoTunes() {
        return this.to('DescribeDomainAutoTunes');
    }
    /**
     * Grants permission to view detail stage progress of an OpenSearch Service domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DescribeDomainChangeProgress.html
     */
    toDescribeDomainChangeProgress() {
        return this.to('DescribeDomainChangeProgress');
    }
    /**
     * Grants permission to view a description of the configuration options and status of an OpenSearch Service domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DescribeDomainConfig.html
     */
    toDescribeDomainConfig() {
        return this.to('DescribeDomainConfig');
    }
    /**
     * Grants permission to view information about domain and node health, the standby Availability Zone, number of nodes per Availability Zone, and shard count per node
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DescribeDomainHealth.html
     */
    toDescribeDomainHealth() {
        return this.to('DescribeDomainHealth');
    }
    /**
     * Grants permission to view information about nodes configured for the domain and their configurations- the node id, type of node, status of node, Availability Zone, instance type and storage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DescribeDomainNodes.html
     */
    toDescribeDomainNodes() {
        return this.to('DescribeDomainNodes');
    }
    /**
     * Grants permission to view a description of the domain configuration for up to five specified OpenSearch Service domains
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DescribeDomains.html
     */
    toDescribeDomains() {
        return this.to('DescribeDomains');
    }
    /**
     * Grants permission to describe the status of a pre-update validation check on an OpenSearch Service domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DescribeDryRunProgress.html
     */
    toDescribeDryRunProgress() {
        return this.to('DescribeDryRunProgress');
    }
    /**
     * Grants permission to view a description of the domain configuration for the specified OpenSearch Service domain, including the domain ID, service endpoint, and ARN. This permission is deprecated. Use DescribeDomain instead
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DescribeDomain.html
     */
    toDescribeElasticsearchDomain() {
        return this.to('DescribeElasticsearchDomain');
    }
    /**
     * Grants permission to view a description of the configuration and status of an OpenSearch Service domain. This permission is deprecated. Use DescribeDomainConfig instead
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DescribeDomainConfig.html
     */
    toDescribeElasticsearchDomainConfig() {
        return this.to('DescribeElasticsearchDomainConfig');
    }
    /**
     * Grants permission to view a description of the domain configuration for up to five specified Amazon OpenSearch domains. This permission is deprecated. Use DescribeDomains instead
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DescribeDomains.html
     */
    toDescribeElasticsearchDomains() {
        return this.to('DescribeElasticsearchDomains');
    }
    /**
     * Grants permission to view the instance count, storage, and master node limits for a given OpenSearch version and instance type. This permission is deprecated. Use DescribeInstanceTypeLimits instead
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DescribeInstanceTypeLimits.html
     */
    toDescribeElasticsearchInstanceTypeLimits() {
        return this.to('DescribeElasticsearchInstanceTypeLimits');
    }
    /**
     * Grants permission to list all the inbound cross-cluster search connections for a destination domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DescribeInboundConnections.html
     */
    toDescribeInboundConnections() {
        return this.to('DescribeInboundConnections');
    }
    /**
     * Grants permission to list all the inbound cross-cluster search connections for a destination domain. This permission is deprecated. Use DescribeInboundConnections instead
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DescribeInboundConnections.html
     */
    toDescribeInboundCrossClusterSearchConnections() {
        return this.to('DescribeInboundCrossClusterSearchConnections');
    }
    /**
     * Grants permission to view the instance count, storage, and master node limits for a given engine version and instance type
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DescribeInstanceTypeLimits.html
     */
    toDescribeInstanceTypeLimits() {
        return this.to('DescribeInstanceTypeLimits');
    }
    /**
     * Grants permission to list all the outbound cross-cluster search connections for a source domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DescribeOutboundConnections.html
     */
    toDescribeOutboundConnections() {
        return this.to('DescribeOutboundConnections');
    }
    /**
     * Grants permission to list all the outbound cross-cluster search connections for a source domain. This permission is deprecated. Use DescribeOutboundConnections instead
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DescribeOutboundConnections.html
     */
    toDescribeOutboundCrossClusterSearchConnections() {
        return this.to('DescribeOutboundCrossClusterSearchConnections');
    }
    /**
     * Grants permission to describe all packages available to OpenSearch Service domains
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DescribePackages.html
     */
    toDescribePackages() {
        return this.to('DescribePackages');
    }
    /**
     * Grants permission to fetch Reserved Instance offerings for Amazon OpenSearch Service. This permission is deprecated. Use DescribeReservedInstanceOfferings instead
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DescribeReservedInstanceOfferings.html
     */
    toDescribeReservedElasticsearchInstanceOfferings() {
        return this.to('DescribeReservedElasticsearchInstanceOfferings');
    }
    /**
     * Grants permission to fetch OpenSearch Service Reserved Instances that have already been purchased. This permission is deprecated. Use DescribeReservedInstances instead
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DescribeReservedInstances.html
     */
    toDescribeReservedElasticsearchInstances() {
        return this.to('DescribeReservedElasticsearchInstances');
    }
    /**
     * Grants permission to fetch Reserved Instance offerings for OpenSearch Service
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DescribeReservedInstanceOfferings.html
     */
    toDescribeReservedInstanceOfferings() {
        return this.to('DescribeReservedInstanceOfferings');
    }
    /**
     * Grants permission to fetch OpenSearch Service Reserved Instances that have already been purchased
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DescribeReservedInstances.html
     */
    toDescribeReservedInstances() {
        return this.to('DescribeReservedInstances');
    }
    /**
     * Grants permission to describe one or more Amazon OpenSearch Service-managed VPC endpoints
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DescribeVpcEndpoints.html
     */
    toDescribeVpcEndpoints() {
        return this.to('DescribeVpcEndpoints');
    }
    /**
     * Grants permission to disassociate a package from the specified OpenSearch Service domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DissociatePackage.html
     */
    toDissociatePackage() {
        return this.to('DissociatePackage');
    }
    /**
     * Grants permission to disassociate multiple packages from the specified OpenSearch Service domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_DissociatePackages.html
     */
    toDissociatePackages() {
        return this.to('DissociatePackages');
    }
    /**
     * Grants permission to send cross-cluster requests to a destination domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/Welcome.html
     */
    toESCrossClusterGet() {
        return this.to('ESCrossClusterGet');
    }
    /**
     * Grants permission to send HTTP DELETE requests to the OpenSearch APIs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/Welcome.html
     */
    toESHttpDelete() {
        return this.to('ESHttpDelete');
    }
    /**
     * Grants permission to send HTTP GET requests to the OpenSearch APIs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/Welcome.html
     */
    toESHttpGet() {
        return this.to('ESHttpGet');
    }
    /**
     * Grants permission to send HTTP HEAD requests to the OpenSearch APIs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/Welcome.html
     */
    toESHttpHead() {
        return this.to('ESHttpHead');
    }
    /**
     * Grants permission to send HTTP PATCH requests to the OpenSearch APIs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/Welcome.html
     */
    toESHttpPatch() {
        return this.to('ESHttpPatch');
    }
    /**
     * Grants permission to send HTTP POST requests to the OpenSearch APIs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/Welcome.html
     */
    toESHttpPost() {
        return this.to('ESHttpPost');
    }
    /**
     * Grants permission to send HTTP PUT requests to the OpenSearch APIs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/Welcome.html
     */
    toESHttpPut() {
        return this.to('ESHttpPut');
    }
    /**
     * Grants permission to get information about an OpenSearch Application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_GetApplication.html
     */
    toGetApplication() {
        return this.to('GetApplication');
    }
    /**
     * Grants permission to fetch a list of compatible OpenSearch and Elasticsearch versions to which an OpenSearch Service domain can be upgraded. This permission is deprecated. Use GetCompatibleVersions instead
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_GetCompatibleVersions.html
     */
    toGetCompatibleElasticsearchVersions() {
        return this.to('GetCompatibleElasticsearchVersions');
    }
    /**
     * Grants permission to fetch list of compatible engine versions to which an OpenSearch Service domain can be upgraded
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_GetCompatibleVersions.html
     */
    toGetCompatibleVersions() {
        return this.to('GetCompatibleVersions');
    }
    /**
     * Grants permission to get the data source for the OpenSearch Service domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_GetDataSource.html
     */
    toGetDataSource() {
        return this.to('GetDataSource');
    }
    /**
     * Grants permission to get the data source for the provided OpenSearch arns
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_GetDirectQueryDataSource.html
     */
    toGetDirectQueryDataSource() {
        return this.to('GetDirectQueryDataSource');
    }
    /**
     * Grants permission to retrieve the status of maintenance action for the node
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_GetDomainMaintenanceStatus.html
     */
    toGetDomainMaintenanceStatus() {
        return this.to('GetDomainMaintenanceStatus');
    }
    /**
     * Grants permission to fetch the version history for a package
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_GetPackageVersionHistory.html
     */
    toGetPackageVersionHistory() {
        return this.to('GetPackageVersionHistory');
    }
    /**
     * Grants permission to fetch the upgrade history of a given OpenSearch Service domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_GetUpgradeHistory.html
     */
    toGetUpgradeHistory() {
        return this.to('GetUpgradeHistory');
    }
    /**
     * Grants permission to fetch the upgrade status of a given OpenSearch Service domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_GetUpgradeStatus.html
     */
    toGetUpgradeStatus() {
        return this.to('GetUpgradeStatus');
    }
    /**
     * Grants permission to list OpenSearch Applications
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_ListApplications.html
     */
    toListApplications() {
        return this.to('ListApplications');
    }
    /**
     * Grants permission to retrieve a list of data source for the OpenSearch Service domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_ListDataSources.html
     */
    toListDataSources() {
        return this.to('ListDataSources');
    }
    /**
     * Grants permission to retrieve a list of data source for the provided OpenSearch arns
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_ListDirectQueryDataSources.html
     */
    toListDirectQueryDataSources() {
        return this.to('ListDirectQueryDataSources');
    }
    /**
     * Grants permission to retrieve a list of maintenance actions for the OpenSearch Service domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_ListDomainMaintenances.html
     */
    toListDomainMaintenances() {
        return this.to('ListDomainMaintenances');
    }
    /**
     * Grants permission to display the names of all OpenSearch Service domains that the current user owns
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_ListDomainNames.html
     */
    toListDomainNames() {
        return this.to('ListDomainNames');
    }
    /**
     * Grants permission to list all OpenSearch Service domains that a package is associated with
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_ListDomainsForPackage.html
     */
    toListDomainsForPackage() {
        return this.to('ListDomainsForPackage');
    }
    /**
     * Grants permission to list all instance types and available features for a given OpenSearch version. This permission is deprecated. Use ListInstanceTypeDetails instead
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_ListInstanceTypeDetails.html
     */
    toListElasticsearchInstanceTypeDetails() {
        return this.to('ListElasticsearchInstanceTypeDetails');
    }
    /**
     * Grants permission to list all EC2 instance types that are supported for a given OpenSearch version
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_ListInstanceTypeDetails.html
     */
    toListElasticsearchInstanceTypes() {
        return this.to('ListElasticsearchInstanceTypes');
    }
    /**
     * Grants permission to list all supported OpenSearch versions on Amazon OpenSearch Service. This permission is deprecated. Use ListVersions instead
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_ListVersions.html
     */
    toListElasticsearchVersions() {
        return this.to('ListElasticsearchVersions');
    }
    /**
     * Grants permission to list all instance types and available features for a given OpenSearch or Elasticsearch version
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_ListInstanceTypeDetails.html
     */
    toListInstanceTypeDetails() {
        return this.to('ListInstanceTypeDetails');
    }
    /**
     * Grants permission to list all packages associated with the OpenSearch Service domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_ListPackagesForDomain.html
     */
    toListPackagesForDomain() {
        return this.to('ListPackagesForDomain');
    }
    /**
     * Grants permission to retrieve a list of configuration changes that are scheduled for a OpenSearch Service domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_ListScheduledActions.html
     */
    toListScheduledActions() {
        return this.to('ListScheduledActions');
    }
    /**
     * Grants permission to display all resource tags for an OpenSearch Service domain, data source, or application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_ListTags.html
     */
    toListTags() {
        return this.to('ListTags');
    }
    /**
     * Grants permission to list all supported OpenSearch and Elasticsearch versions in Amazon OpenSearch Service
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_ListVersions.html
     */
    toListVersions() {
        return this.to('ListVersions');
    }
    /**
     * Grants permission to retrieve information about each AWS principal that is allowed to access a given Amazon OpenSearch Service domain through the use of an interface VPC endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_ListVpcEndpointAccess.html
     */
    toListVpcEndpointAccess() {
        return this.to('ListVpcEndpointAccess');
    }
    /**
     * Grants permission to retrieve all Amazon OpenSearch Service-managed VPC endpoints in the current AWS account and Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_ListVpcEndpoints.html
     */
    toListVpcEndpoints() {
        return this.to('ListVpcEndpoints');
    }
    /**
     * Grants permission to retrieve all Amazon OpenSearch Service-managed VPC endpoints associated with a particular domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_ListVpcEndpointsForDomain.html
     */
    toListVpcEndpointsForDomain() {
        return this.to('ListVpcEndpointsForDomain');
    }
    /**
     * Grants permission to purchase OpenSearch Service Reserved Instances. This permission is deprecated. Use PurchaseReservedInstanceOffering instead
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_PurchaseReservedInstanceOffering.html
     */
    toPurchaseReservedElasticsearchInstanceOffering() {
        return this.to('PurchaseReservedElasticsearchInstanceOffering');
    }
    /**
     * Grants permission to purchase OpenSearch reserved instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_PurchaseReservedInstanceOffering.html
     */
    toPurchaseReservedInstanceOffering() {
        return this.to('PurchaseReservedInstanceOffering');
    }
    /**
     * Grants permission to the destination domain owner to reject an inbound cross-cluster search connection request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_RejectInboundConnection.html
     */
    toRejectInboundConnection() {
        return this.to('RejectInboundConnection');
    }
    /**
     * Grants permission to the destination domain owner to reject an inbound cross-cluster search connection request. This permission is deprecated. Use RejectInboundConnection instead
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_RejectInboundConnection.html
     */
    toRejectInboundCrossClusterSearchConnection() {
        return this.to('RejectInboundCrossClusterSearchConnection');
    }
    /**
     * Grants permission to remove resource tags from an OpenSearch Service domain, data source, or application
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_RemoveTags.html
     */
    toRemoveTags() {
        return this.to('RemoveTags');
    }
    /**
     * Grants permission to revoke access to an Amazon OpenSearch Service domain that was provided through an interface VPC endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_RevokeVpcEndpointAccess.html
     */
    toRevokeVpcEndpointAccess() {
        return this.to('RevokeVpcEndpointAccess');
    }
    /**
     * Grants permission to initiate the maintenance on the node
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_StartDomainMaintenance.html
     */
    toStartDomainMaintenance() {
        return this.to('StartDomainMaintenance');
    }
    /**
     * Grants permission to start a service software update of a domain. This permission is deprecated. Use StartServiceSoftwareUpdate instead
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_StartServiceSoftwareUpdate.html
     */
    toStartElasticsearchServiceSoftwareUpdate() {
        return this.to('StartElasticsearchServiceSoftwareUpdate');
    }
    /**
     * Grants permission to start a service software update of a domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_StartServiceSoftwareUpdate.html
     */
    toStartServiceSoftwareUpdate() {
        return this.to('StartServiceSoftwareUpdate');
    }
    /**
     * Grants permission to update an OpenSearch Application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_UpdateApplication.html
     */
    toUpdateApplication() {
        return this.to('UpdateApplication');
    }
    /**
     * Grants permission to update the data source for the OpenSearch Service domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_UpdateDataSource.html
     */
    toUpdateDataSource() {
        return this.to('UpdateDataSource');
    }
    /**
     * Grants permission to update the data source for the provided OpenSearch arns
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_UpdateDirectQueryDataSource.html
     */
    toUpdateDirectQueryDataSource() {
        return this.to('UpdateDirectQueryDataSource');
    }
    /**
     * Grants permission to modify the configuration of an OpenSearch Service domain, such as the instance type or number of instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_UpdateDomainConfig.html
     */
    toUpdateDomainConfig() {
        return this.to('UpdateDomainConfig');
    }
    /**
     * Grants permission to modify the configuration of an OpenSearch Service domain, such as the instance type or number of instances. This permission is deprecated. Use UpdateDomainConfig instead
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_UpdateDomainConfig.html
     */
    toUpdateElasticsearchDomainConfig() {
        return this.to('UpdateElasticsearchDomainConfig');
    }
    /**
     * Grants permission to update a package for use with OpenSearch Service domains
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_UpdatePackage.html
     */
    toUpdatePackage() {
        return this.to('UpdatePackage');
    }
    /**
     * Grants permission to update scope a package
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_UpdatePackageScope.html
     */
    toUpdatePackageScope() {
        return this.to('UpdatePackageScope');
    }
    /**
     * Grants permission to reschedule a planned OpenSearch Service domain configuration change for a later time
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_UpdateScheduledAction.html
     */
    toUpdateScheduledAction() {
        return this.to('UpdateScheduledAction');
    }
    /**
     * Grants permission to modify an Amazon OpenSearch Service-managed interface VPC endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_UpdateVpcEndpoint.html
     */
    toUpdateVpcEndpoint() {
        return this.to('UpdateVpcEndpoint');
    }
    /**
     * Grants permission to initiate upgrade of an OpenSearch Service domain to a given version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_UpgradeDomain.html
     */
    toUpgradeDomain() {
        return this.to('UpgradeDomain');
    }
    /**
     * Grants permission to initiate upgrade of an OpenSearch Service domain to a specified version. This permission is deprecated. Use UpgradeDomain instead
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/APIReference/API_UpgradeDomain.html
     */
    toUpgradeElasticsearchDomain() {
        return this.to('UpgradeElasticsearchDomain');
    }
    /**
     * Adds a resource of type domain to the statement
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/developerguide/ac.html
     *
     * @param domainName - Identifier for the domainName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDomain(domainName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:es:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:domain/${domainName}`);
    }
    /**
     * Adds a resource of type application to the statement
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/developerguide/ac.html
     *
     * @param appId - Identifier for the appId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApplication(appId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:opensearch:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:application/${appId}`);
    }
    /**
     * Adds a resource of type es_role to the statement
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/developerguide/slr.html
     *
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEsRole(account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iam::${account ?? this.defaultAccount}:role/aws-service-role/es.amazonaws.com/AWSServiceRoleForAmazonOpenSearchService`);
    }
    /**
     * Adds a resource of type opensearchservice_role to the statement
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/developerguide/slr.html
     *
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onOpensearchserviceRole(account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iam::${account ?? this.defaultAccount}:role/aws-service-role/opensearchservice.amazonaws.com/AWSServiceRoleForAmazonOpenSearchService`);
    }
    /**
     * Adds a resource of type datasource to the statement
     *
     * https://docs.aws.amazon.com/opensearch-service/latest/developerguide/datasource.html
     *
     * @param dataSourceName - Identifier for the dataSourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDatasource(dataSourceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:opensearch:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:datasource/${dataSourceName}`);
    }
    /**
     * Filters access based on the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toAddDirectQueryDataSource()
     * - .toAddTags()
     * - .toCreateApplication()
     * - .toCreateDomain()
     * - .toCreateElasticsearchDomain()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - domain
     * - application
     * - es_role
     * - opensearchservice_role
     * - datasource
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toAddDirectQueryDataSource()
     * - .toAddTags()
     * - .toCreateApplication()
     * - .toCreateDomain()
     * - .toCreateElasticsearchDomain()
     * - .toRemoveTags()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [es](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonopensearchservice.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'es';
        this.accessLevelList = {
            Write: [
                'AcceptInboundConnection',
                'AcceptInboundCrossClusterSearchConnection',
                'AddDataSource',
                'AddDirectQueryDataSource',
                'AssociatePackage',
                'AssociatePackages',
                'AuthorizeVpcEndpointAccess',
                'CancelDomainConfigChange',
                'CancelElasticsearchServiceSoftwareUpdate',
                'CancelServiceSoftwareUpdate',
                'CreateApplication',
                'CreateDomain',
                'CreateElasticsearchDomain',
                'CreateElasticsearchServiceRole',
                'CreateOutboundConnection',
                'CreateOutboundCrossClusterSearchConnection',
                'CreatePackage',
                'CreateServiceRole',
                'CreateVpcEndpoint',
                'DeleteApplication',
                'DeleteDataSource',
                'DeleteDirectQueryDataSource',
                'DeleteDomain',
                'DeleteElasticsearchDomain',
                'DeleteElasticsearchServiceRole',
                'DeleteInboundConnection',
                'DeleteInboundCrossClusterSearchConnection',
                'DeleteOutboundConnection',
                'DeleteOutboundCrossClusterSearchConnection',
                'DeletePackage',
                'DeleteVpcEndpoint',
                'DissociatePackage',
                'DissociatePackages',
                'ESHttpDelete',
                'ESHttpPatch',
                'ESHttpPost',
                'ESHttpPut',
                'PurchaseReservedElasticsearchInstanceOffering',
                'PurchaseReservedInstanceOffering',
                'RejectInboundConnection',
                'RejectInboundCrossClusterSearchConnection',
                'RevokeVpcEndpointAccess',
                'StartDomainMaintenance',
                'StartElasticsearchServiceSoftwareUpdate',
                'StartServiceSoftwareUpdate',
                'UpdateApplication',
                'UpdateDataSource',
                'UpdateDirectQueryDataSource',
                'UpdateDomainConfig',
                'UpdateElasticsearchDomainConfig',
                'UpdatePackage',
                'UpdatePackageScope',
                'UpdateScheduledAction',
                'UpdateVpcEndpoint',
                'UpgradeDomain',
                'UpgradeElasticsearchDomain'
            ],
            Tagging: [
                'AddTags',
                'RemoveTags'
            ],
            Read: [
                'DescribeDomain',
                'DescribeDomainAutoTunes',
                'DescribeDomainChangeProgress',
                'DescribeDomainConfig',
                'DescribeDomainHealth',
                'DescribeDomainNodes',
                'DescribeDryRunProgress',
                'DescribeElasticsearchDomain',
                'DescribeElasticsearchDomainConfig',
                'DescribePackages',
                'ESCrossClusterGet',
                'ESHttpGet',
                'ESHttpHead',
                'GetApplication',
                'GetDataSource',
                'GetDirectQueryDataSource',
                'GetDomainMaintenanceStatus',
                'GetPackageVersionHistory',
                'GetUpgradeHistory',
                'GetUpgradeStatus',
                'ListTags'
            ],
            List: [
                'DescribeDomains',
                'DescribeElasticsearchDomains',
                'DescribeElasticsearchInstanceTypeLimits',
                'DescribeInboundConnections',
                'DescribeInboundCrossClusterSearchConnections',
                'DescribeInstanceTypeLimits',
                'DescribeOutboundConnections',
                'DescribeOutboundCrossClusterSearchConnections',
                'DescribeReservedElasticsearchInstanceOfferings',
                'DescribeReservedElasticsearchInstances',
                'DescribeReservedInstanceOfferings',
                'DescribeReservedInstances',
                'DescribeVpcEndpoints',
                'GetCompatibleElasticsearchVersions',
                'GetCompatibleVersions',
                'ListApplications',
                'ListDataSources',
                'ListDirectQueryDataSources',
                'ListDomainMaintenances',
                'ListDomainNames',
                'ListDomainsForPackage',
                'ListElasticsearchInstanceTypeDetails',
                'ListElasticsearchInstanceTypes',
                'ListElasticsearchVersions',
                'ListInstanceTypeDetails',
                'ListPackagesForDomain',
                'ListScheduledActions',
                'ListVersions',
                'ListVpcEndpointAccess',
                'ListVpcEndpoints',
                'ListVpcEndpointsForDomain'
            ]
        };
    }
}
exports.Es = Es;
//# sourceMappingURL=data:application/json;base64,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