"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SesPinpoint = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [ses-pinpoint](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonpinpointemailservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class SesPinpoint extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_CreateConfigurationSet.html
     */
    toCreateConfigurationSet() {
        return this.to('CreateConfigurationSet');
    }
    /**
     * Grants permission to create a configuration set event destination
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_CreateConfigurationSetEventDestination.html
     */
    toCreateConfigurationSetEventDestination() {
        return this.to('CreateConfigurationSetEventDestination');
    }
    /**
     * Grants permission to create a new pool of dedicated IP addresses
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_CreateDedicatedIpPool.html
     */
    toCreateDedicatedIpPool() {
        return this.to('CreateDedicatedIpPool');
    }
    /**
     * Grants permission to create a new predictive inbox placement test
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_CreateDeliverabilityTestReport.html
     */
    toCreateDeliverabilityTestReport() {
        return this.to('CreateDeliverabilityTestReport');
    }
    /**
     * Grants permission to start the process of verifying an email identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_CreateEmailIdentity.html
     */
    toCreateEmailIdentity() {
        return this.to('CreateEmailIdentity');
    }
    /**
     * Grants permission to delete an existing configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_DeleteConfigurationSet.html
     */
    toDeleteConfigurationSet() {
        return this.to('DeleteConfigurationSet');
    }
    /**
     * Grants permission to delete an event destination
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_DeleteConfigurationSetEventDestination.html
     */
    toDeleteConfigurationSetEventDestination() {
        return this.to('DeleteConfigurationSetEventDestination');
    }
    /**
     * Grants permission to delete a dedicated IP pool
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_DeleteDedicatedIpPool.html
     */
    toDeleteDedicatedIpPool() {
        return this.to('DeleteDedicatedIpPool');
    }
    /**
     * Grants permission to delete an email identity that you previously verified
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_DeleteEmailIdentity.html
     */
    toDeleteEmailIdentity() {
        return this.to('DeleteEmailIdentity');
    }
    /**
     * Grants permission to get information about the email-sending status and capabilities
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetAccount.html
     */
    toGetAccount() {
        return this.to('GetAccount');
    }
    /**
     * Grants permission to retrieve a list of the deny lists on which your dedicated IP addresses appear
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetBlacklistReports.html
     */
    toGetBlacklistReports() {
        return this.to('GetBlacklistReports');
    }
    /**
     * Grants permission to get information about an existing configuration set
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetConfigurationSet.html
     */
    toGetConfigurationSet() {
        return this.to('GetConfigurationSet');
    }
    /**
     * Grants permission to retrieve a list of event destinations that are associated with a configuration set
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetConfigurationSetEventDestinations.html
     */
    toGetConfigurationSetEventDestinations() {
        return this.to('GetConfigurationSetEventDestinations');
    }
    /**
     * Grants permission to get information about a dedicated IP address
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetDedicatedIp.html
     */
    toGetDedicatedIp() {
        return this.to('GetDedicatedIp');
    }
    /**
     * Grants permission to list the dedicated IP addresses that are associated with your account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetDedicatedIps.html
     */
    toGetDedicatedIps() {
        return this.to('GetDedicatedIps');
    }
    /**
     * Grants permission to get the status of the Deliverability dashboard
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetDeliverabilityDashboardOptions.html
     */
    toGetDeliverabilityDashboardOptions() {
        return this.to('GetDeliverabilityDashboardOptions');
    }
    /**
     * Grants permission to retrieve the results of a predictive inbox placement test
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetDeliverabilityTestReport.html
     */
    toGetDeliverabilityTestReport() {
        return this.to('GetDeliverabilityTestReport');
    }
    /**
     * Grants permission to retrieve all the deliverability data for a specific campaign
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetDomainDeliverabilityCampaign.html
     */
    toGetDomainDeliverabilityCampaign() {
        return this.to('GetDomainDeliverabilityCampaign');
    }
    /**
     * Grants permission to retrieve inbox placement and engagement rates for the domains that you use to send email
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetDomainStatisticsReport.html
     */
    toGetDomainStatisticsReport() {
        return this.to('GetDomainStatisticsReport');
    }
    /**
     * Grants permission to get information about a specific identity associated with your account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_GetEmailIdentity.html
     */
    toGetEmailIdentity() {
        return this.to('GetEmailIdentity');
    }
    /**
     * Grants permission to list all of the configuration sets associated with your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_ListConfigurationSets.html
     */
    toListConfigurationSets() {
        return this.to('ListConfigurationSets');
    }
    /**
     * Grants permission to list all of the dedicated IP pools that exist in your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_ListDedicatedIpPools.html
     */
    toListDedicatedIpPools() {
        return this.to('ListDedicatedIpPools');
    }
    /**
     * Grants permission to retrieve a list of the predictive inbox placement tests that you've performed, regardless of their statuses
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_ListDeliverabilityTestReports.html
     */
    toListDeliverabilityTestReports() {
        return this.to('ListDeliverabilityTestReports');
    }
    /**
     * Grants permission to retrieve deliverability data for all the campaigns that used a specific domain to send email during a specified time range
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_ListDomainDeliverabilityCampaigns.html
     */
    toListDomainDeliverabilityCampaigns() {
        return this.to('ListDomainDeliverabilityCampaigns');
    }
    /**
     * Grants permission to list all of the email identities that are associated with your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_ListEmailIdentities.html
     */
    toListEmailIdentities() {
        return this.to('ListEmailIdentities');
    }
    /**
     * Grants permission to retrieve a list of the tags (keys and values) that are associated with a specific resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to enable or disable the automatic warm-up feature for dedicated IP addresses
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutAccountDedicatedIpWarmupAttributes.html
     */
    toPutAccountDedicatedIpWarmupAttributes() {
        return this.to('PutAccountDedicatedIpWarmupAttributes');
    }
    /**
     * Grants permission to enable or disable the ability of your account to send email
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutAccountSendingAttributes.html
     */
    toPutAccountSendingAttributes() {
        return this.to('PutAccountSendingAttributes');
    }
    /**
     * Grants permission to associate a configuration set with a dedicated IP pool
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutConfigurationSetDeliveryOptions.html
     */
    toPutConfigurationSetDeliveryOptions() {
        return this.to('PutConfigurationSetDeliveryOptions');
    }
    /**
     * Grants permission to enable or disable collection of reputation metrics for emails that you send using a particular configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutConfigurationSetReputationOptions.html
     */
    toPutConfigurationSetReputationOptions() {
        return this.to('PutConfigurationSetReputationOptions');
    }
    /**
     * Grants permission to enable or disable email sending for messages that use a particular configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutConfigurationSetSendingOptions.html
     */
    toPutConfigurationSetSendingOptions() {
        return this.to('PutConfigurationSetSendingOptions');
    }
    /**
     * Grants permission to specify a custom domain to use for open and click tracking elements in email that you send using a particular configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutConfigurationSetTrackingOptions.html
     */
    toPutConfigurationSetTrackingOptions() {
        return this.to('PutConfigurationSetTrackingOptions');
    }
    /**
     * Grants permission to move a dedicated IP address to an existing dedicated IP pool
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutDedicatedIpInPool.html
     */
    toPutDedicatedIpInPool() {
        return this.to('PutDedicatedIpInPool');
    }
    /**
     * Grants permission to enable dedicated IP warm up attributes
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutDedicatedIpWarmupAttributes.html
     */
    toPutDedicatedIpWarmupAttributes() {
        return this.to('PutDedicatedIpWarmupAttributes');
    }
    /**
     * Grants permission to enable or disable the Deliverability dashboard
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutDeliverabilityDashboardOption.html
     */
    toPutDeliverabilityDashboardOption() {
        return this.to('PutDeliverabilityDashboardOption');
    }
    /**
     * Grants permission to enable or disable DKIM authentication for an email identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutEmailIdentityDkimAttributes.html
     */
    toPutEmailIdentityDkimAttributes() {
        return this.to('PutEmailIdentityDkimAttributes');
    }
    /**
     * Grants permission to enable or disable feedback forwarding for an identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutEmailIdentityFeedbackAttributes.html
     */
    toPutEmailIdentityFeedbackAttributes() {
        return this.to('PutEmailIdentityFeedbackAttributes');
    }
    /**
     * Grants permission to enable or disable the custom MAIL FROM domain configuration for an email identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_PutEmailIdentityMailFromAttributes.html
     */
    toPutEmailIdentityMailFromAttributes() {
        return this.to('PutEmailIdentityMailFromAttributes');
    }
    /**
     * Grants permission to send an email message
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifFeedbackAddress()
     * - .ifFromAddress()
     * - .ifFromDisplayName()
     * - .ifRecipients()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_SendEmail.html
     */
    toSendEmail() {
        return this.to('SendEmail');
    }
    /**
     * Grants permission to add one or more tags (keys and values) to a specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove one or more tags (keys and values) from a specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the configuration of an event destination for a configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_UpdateConfigurationSetEventDestination.html
     */
    toUpdateConfigurationSetEventDestination() {
        return this.to('UpdateConfigurationSetEventDestination');
    }
    /**
     * Adds a resource of type configuration-set to the statement
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_CreateConfigurationSet.html
     *
     * @param configurationSetName - Identifier for the configurationSetName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfigurationSet(configurationSetName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:configuration-set/${configurationSetName}`);
    }
    /**
     * Adds a resource of type dedicated-ip-pool to the statement
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_DedicatedIp.html
     *
     * @param dedicatedIPPool - Identifier for the dedicatedIPPool.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDedicatedIpPool(dedicatedIPPool, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:dedicated-ip-pool/${dedicatedIPPool}`);
    }
    /**
     * Adds a resource of type deliverability-test-report to the statement
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_DeliverabilityTestReport.html
     *
     * @param reportId - Identifier for the reportId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeliverabilityTestReport(reportId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:deliverability-test-report/${reportId}`);
    }
    /**
     * Adds a resource of type identity to the statement
     *
     * https://docs.aws.amazon.com/pinpoint-email/latest/APIReference/API_IdentityInfo.html
     *
     * @param identityName - Identifier for the identityName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIdentity(identityName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:identity/${identityName}`);
    }
    /**
     * Filters actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateConfigurationSet()
     * - .toCreateDedicatedIpPool()
     * - .toCreateDeliverabilityTestReport()
     * - .toCreateEmailIdentity()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toCreateConfigurationSetEventDestination()
     * - .toDeleteConfigurationSet()
     * - .toDeleteConfigurationSetEventDestination()
     * - .toDeleteDedicatedIpPool()
     * - .toDeleteEmailIdentity()
     * - .toGetConfigurationSet()
     * - .toGetConfigurationSetEventDestinations()
     * - .toGetDedicatedIps()
     * - .toGetDeliverabilityTestReport()
     * - .toGetDomainStatisticsReport()
     * - .toGetEmailIdentity()
     * - .toPutConfigurationSetDeliveryOptions()
     * - .toPutConfigurationSetReputationOptions()
     * - .toPutConfigurationSetSendingOptions()
     * - .toPutConfigurationSetTrackingOptions()
     * - .toPutDedicatedIpInPool()
     * - .toPutEmailIdentityDkimAttributes()
     * - .toPutEmailIdentityFeedbackAttributes()
     * - .toPutEmailIdentityMailFromAttributes()
     * - .toUpdateConfigurationSetEventDestination()
     *
     * Applies to resource types:
     * - configuration-set
     * - dedicated-ip-pool
     * - deliverability-test-report
     * - identity
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateConfigurationSet()
     * - .toCreateDedicatedIpPool()
     * - .toCreateDeliverabilityTestReport()
     * - .toCreateEmailIdentity()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the SES API version
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toCreateConfigurationSet()
     * - .toCreateConfigurationSetEventDestination()
     * - .toCreateDedicatedIpPool()
     * - .toCreateDeliverabilityTestReport()
     * - .toCreateEmailIdentity()
     * - .toDeleteConfigurationSet()
     * - .toDeleteConfigurationSetEventDestination()
     * - .toDeleteDedicatedIpPool()
     * - .toDeleteEmailIdentity()
     * - .toGetAccount()
     * - .toGetBlacklistReports()
     * - .toGetConfigurationSet()
     * - .toGetConfigurationSetEventDestinations()
     * - .toGetDedicatedIp()
     * - .toGetDedicatedIps()
     * - .toGetDeliverabilityDashboardOptions()
     * - .toGetDeliverabilityTestReport()
     * - .toGetDomainDeliverabilityCampaign()
     * - .toGetDomainStatisticsReport()
     * - .toGetEmailIdentity()
     * - .toListConfigurationSets()
     * - .toListDedicatedIpPools()
     * - .toListDeliverabilityTestReports()
     * - .toListDomainDeliverabilityCampaigns()
     * - .toListEmailIdentities()
     * - .toListTagsForResource()
     * - .toPutAccountDedicatedIpWarmupAttributes()
     * - .toPutAccountSendingAttributes()
     * - .toPutConfigurationSetDeliveryOptions()
     * - .toPutConfigurationSetReputationOptions()
     * - .toPutConfigurationSetSendingOptions()
     * - .toPutConfigurationSetTrackingOptions()
     * - .toPutDedicatedIpInPool()
     * - .toPutDedicatedIpWarmupAttributes()
     * - .toPutDeliverabilityDashboardOption()
     * - .toPutEmailIdentityDkimAttributes()
     * - .toPutEmailIdentityFeedbackAttributes()
     * - .toPutEmailIdentityMailFromAttributes()
     * - .toSendEmail()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateConfigurationSetEventDestination()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifApiVersion(value, operator) {
        return this.if(`ApiVersion`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the "Return-Path" address, which specifies where bounces and complaints are sent by email feedback forwarding
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toSendEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFeedbackAddress(value, operator) {
        return this.if(`FeedbackAddress`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the "From" address of a message
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toSendEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFromAddress(value, operator) {
        return this.if(`FromAddress`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the "From" address that is used as the display name of a message
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toSendEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFromDisplayName(value, operator) {
        return this.if(`FromDisplayName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the recipient addresses of a message, which include the "To", "CC", and "BCC" addresses
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toSendEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRecipients(value, operator) {
        return this.if(`Recipients`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [ses-pinpoint](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonpinpointemailservice.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'ses';
        this.accessLevelList = {
            Write: [
                'CreateConfigurationSet',
                'CreateConfigurationSetEventDestination',
                'CreateDedicatedIpPool',
                'CreateDeliverabilityTestReport',
                'CreateEmailIdentity',
                'DeleteConfigurationSet',
                'DeleteConfigurationSetEventDestination',
                'DeleteDedicatedIpPool',
                'DeleteEmailIdentity',
                'PutAccountDedicatedIpWarmupAttributes',
                'PutAccountSendingAttributes',
                'PutConfigurationSetDeliveryOptions',
                'PutConfigurationSetReputationOptions',
                'PutConfigurationSetSendingOptions',
                'PutConfigurationSetTrackingOptions',
                'PutDedicatedIpInPool',
                'PutDedicatedIpWarmupAttributes',
                'PutDeliverabilityDashboardOption',
                'PutEmailIdentityDkimAttributes',
                'PutEmailIdentityFeedbackAttributes',
                'PutEmailIdentityMailFromAttributes',
                'SendEmail',
                'UpdateConfigurationSetEventDestination'
            ],
            Read: [
                'GetAccount',
                'GetBlacklistReports',
                'GetConfigurationSet',
                'GetConfigurationSetEventDestinations',
                'GetDedicatedIp',
                'GetDedicatedIps',
                'GetDeliverabilityDashboardOptions',
                'GetDeliverabilityTestReport',
                'GetDomainDeliverabilityCampaign',
                'GetDomainStatisticsReport',
                'GetEmailIdentity',
                'ListDomainDeliverabilityCampaigns',
                'ListTagsForResource'
            ],
            List: [
                'ListConfigurationSets',
                'ListDedicatedIpPools',
                'ListDeliverabilityTestReports',
                'ListEmailIdentities'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.SesPinpoint = SesPinpoint;
//# sourceMappingURL=data:application/json;base64,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