"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PcaConnectorAd = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [pca-connector-ad](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsprivatecaconnectorforactivedirectory.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class PcaConnectorAd extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a Connector in your account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - acm-pca:DescribeCertificateAuthority
     * - acm-pca:GetCertificate
     * - acm-pca:GetCertificateAuthorityCertificate
     * - acm-pca:IssueCertificate
     * - ec2:CreateTags
     * - ec2:CreateVpcEndpoint
     * - ec2:DescribeVpcEndpoints
     *
     * https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_CreateConnector.html
     */
    toCreateConnector() {
        return this.to('CreateConnector');
    }
    /**
     * Grants permission to create a DirectoryRegistration in your account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ds:AuthorizeApplication
     * - ds:DescribeDirectories
     *
     * https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_CreateDirectoryRegistration.html
     */
    toCreateDirectoryRegistration() {
        return this.to('CreateDirectoryRegistration');
    }
    /**
     * Grants permission to create a ServicePrincipalName for a DirectoryRegistration
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ds:UpdateAuthorizedApplication
     *
     * https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_CreateServicePrincipalName.html
     */
    toCreateServicePrincipalName() {
        return this.to('CreateServicePrincipalName');
    }
    /**
     * Grants permission to create a Template for a Connector
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_CreateTemplate.html
     */
    toCreateTemplate() {
        return this.to('CreateTemplate');
    }
    /**
     * Grants permission to create a TemplateGroupAccessControlEntry for a Template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_CreateTemplateGroupAccessControlEntry.html
     */
    toCreateTemplateGroupAccessControlEntry() {
        return this.to('CreateTemplateGroupAccessControlEntry');
    }
    /**
     * Grants permission to delete a Connector in your account
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DeleteVpcEndpoints
     * - ec2:DescribeVpcEndpoints
     *
     * https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_DeleteConnector.html
     */
    toDeleteConnector() {
        return this.to('DeleteConnector');
    }
    /**
     * Grants permission to delete a DirectoryRegistration in your account
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ds:UnauthorizeApplication
     * - ds:UpdateAuthorizedApplication
     *
     * https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_DeleteDirectoryRegistration.html
     */
    toDeleteDirectoryRegistration() {
        return this.to('DeleteDirectoryRegistration');
    }
    /**
     * Grants permission to delete a ServicePrincipalName for a DirectoryRegistration
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ds:UpdateAuthorizedApplication
     *
     * https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_DeleteServicePrincipalName.html
     */
    toDeleteServicePrincipalName() {
        return this.to('DeleteServicePrincipalName');
    }
    /**
     * Grants permission to delete a Template for a Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_DeleteTemplate.html
     */
    toDeleteTemplate() {
        return this.to('DeleteTemplate');
    }
    /**
     * Grants permission to delete a TemplateGroupAccessControlEntry for a Template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_DeleteTemplateGroupAccessControlEntry.html
     */
    toDeleteTemplateGroupAccessControlEntry() {
        return this.to('DeleteTemplateGroupAccessControlEntry');
    }
    /**
     * Grants permission to get a Connector in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_GetConnector.html
     */
    toGetConnector() {
        return this.to('GetConnector');
    }
    /**
     * Grants permission to get a DirectoryRegistration in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_GetDirectoryRegistration.html
     */
    toGetDirectoryRegistration() {
        return this.to('GetDirectoryRegistration');
    }
    /**
     * Grants permission to get a ServicePrincipalName for a DirectoryRegistration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_GetServicePrincipalName.html
     */
    toGetServicePrincipalName() {
        return this.to('GetServicePrincipalName');
    }
    /**
     * Grants permission to get a Template for a Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_GetTemplate.html
     */
    toGetTemplate() {
        return this.to('GetTemplate');
    }
    /**
     * Grants permission to get a TemplateGroupAccessControlEntry for a Template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_GetTemplateGroupAccessControlEntry.html
     */
    toGetTemplateGroupAccessControlEntry() {
        return this.to('GetTemplateGroupAccessControlEntry');
    }
    /**
     * Grants permission to list the Connectors in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_ListConnectors.html
     */
    toListConnectors() {
        return this.to('ListConnectors');
    }
    /**
     * Grants permission to list the DirectoryRegistrations in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_ListDirectoryRegistrations.html
     */
    toListDirectoryRegistrations() {
        return this.to('ListDirectoryRegistrations');
    }
    /**
     * Grants permission to list the ServicePrincipalNames for a DirectoryRegistration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_ListServicePrincipalNames.html
     */
    toListServicePrincipalNames() {
        return this.to('ListServicePrincipalNames');
    }
    /**
     * Grants permission to list the tags for a pca-connector-ad resource in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list the TemplateGroupAccessControlEntries for a Template
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_ListTemplateGroupAccessControlEntries.html
     */
    toListTemplateGroupAccessControlEntries() {
        return this.to('ListTemplateGroupAccessControlEntries');
    }
    /**
     * Grants permission to list the Templates for a Connector
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_ListTemplates.html
     */
    toListTemplates() {
        return this.to('ListTemplates');
    }
    /**
     * Grants permission to tag a pca-connector-ad resource in your account
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a pca-connector-ad resource in your account
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a Template for a Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_UpdateTemplate.html
     */
    toUpdateTemplate() {
        return this.to('UpdateTemplate');
    }
    /**
     * Grants permission to update a TemplateGroupAccessControlEntry for a Template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_UpdateTemplateGroupAccessControlEntry.html
     */
    toUpdateTemplateGroupAccessControlEntry() {
        return this.to('UpdateTemplateGroupAccessControlEntry');
    }
    /**
     * Adds a resource of type Connector to the statement
     *
     * https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_Connector.html
     *
     * @param connectorId - Identifier for the connectorId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConnector(connectorId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:pca-connector-ad:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:connector/${connectorId}`);
    }
    /**
     * Adds a resource of type DirectoryRegistration to the statement
     *
     * https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_DirectoryRegistration.html
     *
     * @param directoryId - Identifier for the directoryId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDirectoryRegistration(directoryId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:pca-connector-ad:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:directory-registration/${directoryId}`);
    }
    /**
     * Adds a resource of type Template to the statement
     *
     * https://docs.aws.amazon.com/pca-connector-ad/latest/APIReference/API_Template.html
     *
     * @param connectorId - Identifier for the connectorId.
     * @param templateId - Identifier for the templateId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTemplate(connectorId, templateId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:pca-connector-ad:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:connector/${connectorId}/template/${templateId}`);
    }
    /**
     * Filters access by on the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsprivatecaconnectorforactivedirectory.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateConnector()
     * - .toCreateDirectoryRegistration()
     * - .toCreateTemplate()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsprivatecaconnectorforactivedirectory.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toCreateConnector()
     * - .toCreateDirectoryRegistration()
     * - .toCreateTemplate()
     *
     * Applies to resource types:
     * - Connector
     * - DirectoryRegistration
     * - Template
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by on the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsprivatecaconnectorforactivedirectory.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateConnector()
     * - .toCreateDirectoryRegistration()
     * - .toCreateTemplate()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [pca-connector-ad](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsprivatecaconnectorforactivedirectory.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'pca-connector-ad';
        this.accessLevelList = {
            Write: [
                'CreateConnector',
                'CreateDirectoryRegistration',
                'CreateServicePrincipalName',
                'CreateTemplate',
                'CreateTemplateGroupAccessControlEntry',
                'DeleteConnector',
                'DeleteDirectoryRegistration',
                'DeleteServicePrincipalName',
                'DeleteTemplate',
                'DeleteTemplateGroupAccessControlEntry',
                'UpdateTemplate',
                'UpdateTemplateGroupAccessControlEntry'
            ],
            Read: [
                'GetConnector',
                'GetDirectoryRegistration',
                'GetServicePrincipalName',
                'GetTemplate',
                'GetTemplateGroupAccessControlEntry',
                'ListTagsForResource'
            ],
            List: [
                'ListConnectors',
                'ListDirectoryRegistrations',
                'ListServicePrincipalNames',
                'ListTemplateGroupAccessControlEntries',
                'ListTemplates'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.PcaConnectorAd = PcaConnectorAd;
//# sourceMappingURL=data:application/json;base64,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