import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [redshift-data](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonredshiftdataapi.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class RedshiftData extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to execute multiple queries under a single connection
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSessionOwnerIamUserid()
     * - .ifGlueCatalogArn()
     *
     * https://docs.aws.amazon.com/redshift-data/latest/APIReference/API_BatchExecuteStatement.html
     */
    toBatchExecuteStatement(): this;
    /**
     * Grants permission to cancel a running query
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifStatementOwnerIamUserid()
     *
     * https://docs.aws.amazon.com/redshift-data/latest/APIReference/API_CancelStatement.html
     */
    toCancelStatement(): this;
    /**
     * Grants permission to retrieve detailed information about a statement execution
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifStatementOwnerIamUserid()
     *
     * https://docs.aws.amazon.com/redshift-data/latest/APIReference/API_DescribeStatement.html
     */
    toDescribeStatement(): this;
    /**
     * Grants permission to retrieve metadata about a particular table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift-data/latest/APIReference/API_DescribeTable.html
     */
    toDescribeTable(): this;
    /**
     * Grants permission to execute a query
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSessionOwnerIamUserid()
     * - .ifGlueCatalogArn()
     *
     * https://docs.aws.amazon.com/redshift-data/latest/APIReference/API_ExecuteStatement.html
     */
    toExecuteStatement(): this;
    /**
     * Grants permission to get staging bucket location for a given managed workgroup
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift-data/latest/APIReference/API_GetStagingBucketLocation.html
     */
    toGetStagingBucketLocation(): this;
    /**
     * Grants permission to fetch the result of a query
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifStatementOwnerIamUserid()
     *
     * https://docs.aws.amazon.com/redshift-data/latest/APIReference/API_GetStatementResult.html
     */
    toGetStatementResult(): this;
    /**
     * Grants permission to list databases for a given cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift-data/latest/APIReference/API_ListDatabases.html
     */
    toListDatabases(): this;
    /**
     * Grants permission to list schemas for a given cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift-data/latest/APIReference/API_ListSchemas.html
     */
    toListSchemas(): this;
    /**
     * Grants permission to list queries for a given principal
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifStatementOwnerIamUserid()
     *
     * https://docs.aws.amazon.com/redshift-data/latest/APIReference/API_ListStatements.html
     */
    toListStatements(): this;
    /**
     * Grants permission to list tables for a given cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift-data/latest/APIReference/API_ListTables.html
     */
    toListTables(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-clusters.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCluster(clusterName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type workgroup to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-serverless.html
     *
     * @param workgroupId - Identifier for the workgroupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorkgroup(workgroupId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type managed-workgroup to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-serverless.html
     *
     * @param managedWorkgroupId - Identifier for the managedWorkgroupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onManagedWorkgroup(managedWorkgroupId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by tag-value associated with the resource
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-iam-access-control-overview.html#redshift-policy-resources.conditions
     *
     * Applies to resource types:
     * - cluster
     * - workgroup
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by glue catalog arn
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-iam-access-control-overview.html#redshift-policy-resources.conditions
     *
     * Applies to actions:
     * - .toBatchExecuteStatement()
     * - .toExecuteStatement()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifGlueCatalogArn(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by session owner iam userid
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-iam-access-control-overview.html#redshift-policy-resources.conditions
     *
     * Applies to actions:
     * - .toBatchExecuteStatement()
     * - .toExecuteStatement()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSessionOwnerIamUserid(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by statement owner iam userid
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-iam-access-control-overview.html#redshift-policy-resources.conditions
     *
     * Applies to actions:
     * - .toCancelStatement()
     * - .toDescribeStatement()
     * - .toGetStatementResult()
     * - .toListStatements()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifStatementOwnerIamUserid(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [redshift-data](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonredshiftdataapi.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
