"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Resiliencehub = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [resiliencehub](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsresiliencehub.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Resiliencehub extends shared_1.PolicyStatement {
    /**
     * Grants permission to accept resource grouping recommendations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_AcceptResourceGroupingRecommendations.html
     */
    toAcceptResourceGroupingRecommendations() {
        return this.to('AcceptResourceGroupingRecommendations');
    }
    /**
     * Grants permission to add draft application version resource mappings
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - cloudformation:DescribeStacks
     * - cloudformation:ListStackResources
     * - resource-groups:GetGroup
     * - resource-groups:ListGroupResources
     * - servicecatalog:GetApplication
     * - servicecatalog:ListAssociatedResources
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_AddDraftAppVersionResourceMappings.html
     */
    toAddDraftAppVersionResourceMappings() {
        return this.to('AddDraftAppVersionResourceMappings');
    }
    /**
     * Grants permission to include or exclude one or more operational recommendations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_BatchUpdateRecommendationStatus.html
     */
    toBatchUpdateRecommendationStatus() {
        return this.to('BatchUpdateRecommendationStatus');
    }
    /**
     * Grants permission to create application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_CreateApp.html
     */
    toCreateApp() {
        return this.to('CreateApp');
    }
    /**
     * Grants permission to create application app component
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_CreateAppVersionAppComponent.html
     */
    toCreateAppVersionAppComponent() {
        return this.to('CreateAppVersionAppComponent');
    }
    /**
     * Grants permission to create application resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_CreateAppVersionResource.html
     */
    toCreateAppVersionResource() {
        return this.to('CreateAppVersionResource');
    }
    /**
     * Grants permission to create recommendation template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - s3:CreateBucket
     * - s3:ListBucket
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_CreateRecommendationTemplate.html
     */
    toCreateRecommendationTemplate() {
        return this.to('CreateRecommendationTemplate');
    }
    /**
     * Grants permission to create resiliency policy
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_CreateResiliencyPolicy.html
     */
    toCreateResiliencyPolicy() {
        return this.to('CreateResiliencyPolicy');
    }
    /**
     * Grants permission to batch delete application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_DeleteApp.html
     */
    toDeleteApp() {
        return this.to('DeleteApp');
    }
    /**
     * Grants permission to batch delete application assessment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_DeleteAppAssessment.html
     */
    toDeleteAppAssessment() {
        return this.to('DeleteAppAssessment');
    }
    /**
     * Grants permission to remove application input source
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_DeleteAppInputSource.html
     */
    toDeleteAppInputSource() {
        return this.to('DeleteAppInputSource');
    }
    /**
     * Grants permission to delete application app component
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_DeleteAppVersionAppComponent.html
     */
    toDeleteAppVersionAppComponent() {
        return this.to('DeleteAppVersionAppComponent');
    }
    /**
     * Grants permission to delete application resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_DeleteAppVersionResource.html
     */
    toDeleteAppVersionResource() {
        return this.to('DeleteAppVersionResource');
    }
    /**
     * Grants permission to batch delete recommendation template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_DeleteRecommendationTemplate.html
     */
    toDeleteRecommendationTemplate() {
        return this.to('DeleteRecommendationTemplate');
    }
    /**
     * Grants permission to batch delete resiliency policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_DeleteResiliencyPolicy.html
     */
    toDeleteResiliencyPolicy() {
        return this.to('DeleteResiliencyPolicy');
    }
    /**
     * Grants permission to describe application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_DescribeApp.html
     */
    toDescribeApp() {
        return this.to('DescribeApp');
    }
    /**
     * Grants permission to describe application assessment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_DescribeAppAssessment.html
     */
    toDescribeAppAssessment() {
        return this.to('DescribeAppAssessment');
    }
    /**
     * Grants permission to describe application version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_DescribeAppVersion.html
     */
    toDescribeAppVersion() {
        return this.to('DescribeAppVersion');
    }
    /**
     * Grants permission to describe application version app component
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_DescribeAppVersionAppComponent.html
     */
    toDescribeAppVersionAppComponent() {
        return this.to('DescribeAppVersionAppComponent');
    }
    /**
     * Grants permission to describe application version resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_DescribeAppVersionResource.html
     */
    toDescribeAppVersionResource() {
        return this.to('DescribeAppVersionResource');
    }
    /**
     * Grants permission to describe application resolution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_DescribeAppVersionResourcesResolutionStatus.html
     */
    toDescribeAppVersionResourcesResolutionStatus() {
        return this.to('DescribeAppVersionResourcesResolutionStatus');
    }
    /**
     * Grants permission to describe application version template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_DescribeAppVersionTemplate.html
     */
    toDescribeAppVersionTemplate() {
        return this.to('DescribeAppVersionTemplate');
    }
    /**
     * Grants permission to describe draft application version resources import status
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_DescribeDraftAppVersionResourcesImportStatus.html
     */
    toDescribeDraftAppVersionResourcesImportStatus() {
        return this.to('DescribeDraftAppVersionResourcesImportStatus');
    }
    /**
     * Grants permission to describe metrics export
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_DescribeMetricsExport.html
     */
    toDescribeMetricsExport() {
        return this.to('DescribeMetricsExport');
    }
    /**
     * Grants permission to describe resiliency policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_DescribeResiliencyPolicy.html
     */
    toDescribeResiliencyPolicy() {
        return this.to('DescribeResiliencyPolicy');
    }
    /**
     * Grants permission to describe the latest status of the grouping recommendation process
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_DescribeResourceGroupingRecommendationTask.html
     */
    toDescribeResourceGroupingRecommendationTask() {
        return this.to('DescribeResourceGroupingRecommendationTask');
    }
    /**
     * Grants permission to import resources to draft application version
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - cloudformation:DescribeStacks
     * - cloudformation:ListStackResources
     * - resource-groups:GetGroup
     * - resource-groups:ListGroupResources
     * - servicecatalog:GetApplication
     * - servicecatalog:ListAssociatedResources
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_ImportResourcesToDraftAppVersion.html
     */
    toImportResourcesToDraftAppVersion() {
        return this.to('ImportResourcesToDraftAppVersion');
    }
    /**
     * Grants permission to list alarm recommendation
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_ListAlarmRecommendations.html
     */
    toListAlarmRecommendations() {
        return this.to('ListAlarmRecommendations');
    }
    /**
     * Grants permission to list compliance drifts that were detected while running an assessment
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_ListAppAssessmentComplianceDrifts.html
     */
    toListAppAssessmentComplianceDrifts() {
        return this.to('ListAppAssessmentComplianceDrifts');
    }
    /**
     * Grants permission to list resource drifts that were detected while running an assessment
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_ListAppAssessmentResourceDrifts.html
     */
    toListAppAssessmentResourceDrifts() {
        return this.to('ListAppAssessmentResourceDrifts');
    }
    /**
     * Grants permission to list application assessment
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_ListAppAssessments.html
     */
    toListAppAssessments() {
        return this.to('ListAppAssessments');
    }
    /**
     * Grants permission to list app component compliances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_ListAppComponentCompliances.html
     */
    toListAppComponentCompliances() {
        return this.to('ListAppComponentCompliances');
    }
    /**
     * Grants permission to list app component recommendations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_ListAppComponentRecommendations.html
     */
    toListAppComponentRecommendations() {
        return this.to('ListAppComponentRecommendations');
    }
    /**
     * Grants permission to list application input sources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_ListAppInputSources.html
     */
    toListAppInputSources() {
        return this.to('ListAppInputSources');
    }
    /**
     * Grants permission to list application version app components
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_ListAppVersionAppComponents.html
     */
    toListAppVersionAppComponents() {
        return this.to('ListAppVersionAppComponents');
    }
    /**
     * Grants permission to application version resource mappings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_ListAppVersionResourceMappings.html
     */
    toListAppVersionResourceMappings() {
        return this.to('ListAppVersionResourceMappings');
    }
    /**
     * Grants permission to list application resources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_ListAppVersionResources.html
     */
    toListAppVersionResources() {
        return this.to('ListAppVersionResources');
    }
    /**
     * Grants permission to list application version
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_ListAppVersions.html
     */
    toListAppVersions() {
        return this.to('ListAppVersions');
    }
    /**
     * Grants permission to list applications
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_ListApps.html
     */
    toListApps() {
        return this.to('ListApps');
    }
    /**
     * Grants permission to list metrics
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_ListMetrics.html
     */
    toListMetrics() {
        return this.to('ListMetrics');
    }
    /**
     * Grants permission to list recommendation templates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_ListRecommendationTemplates.html
     */
    toListRecommendationTemplates() {
        return this.to('ListRecommendationTemplates');
    }
    /**
     * Grants permission to list resiliency policies
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_ListResiliencyPolicies.html
     */
    toListResiliencyPolicies() {
        return this.to('ListResiliencyPolicies');
    }
    /**
     * Grants permission to list resource grouping recommendations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_ListResourceGroupingRecommendations.html
     */
    toListResourceGroupingRecommendations() {
        return this.to('ListResourceGroupingRecommendations');
    }
    /**
     * Grants permission to list SOP recommendations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_ListSopRecommendations.html
     */
    toListSopRecommendations() {
        return this.to('ListSopRecommendations');
    }
    /**
     * Grants permission to list suggested resiliency policies
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_ListSuggestedResiliencyPolicies.html
     */
    toListSuggestedResiliencyPolicies() {
        return this.to('ListSuggestedResiliencyPolicies');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list test recommendations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_ListTestRecommendations.html
     */
    toListTestRecommendations() {
        return this.to('ListTestRecommendations');
    }
    /**
     * Grants permission to list unsupported application version resources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_ListUnsupportedAppVersionResources.html
     */
    toListUnsupportedAppVersionResources() {
        return this.to('ListUnsupportedAppVersionResources');
    }
    /**
     * Grants permission to publish application version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_PublishAppVersion.html
     */
    toPublishAppVersion() {
        return this.to('PublishAppVersion');
    }
    /**
     * Grants permission to put draft application version template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_PutDraftAppVersionTemplate.html
     */
    toPutDraftAppVersionTemplate() {
        return this.to('PutDraftAppVersionTemplate');
    }
    /**
     * Grants permission to reject resource grouping recommendations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_RejectResourceGroupingRecommendations.html
     */
    toRejectResourceGroupingRecommendations() {
        return this.to('RejectResourceGroupingRecommendations');
    }
    /**
     * Grants permission to remove draft application version mappings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_RemoveDraftAppVersionResourceMappings.html
     */
    toRemoveDraftAppVersionResourceMappings() {
        return this.to('RemoveDraftAppVersionResourceMappings');
    }
    /**
     * Grants permission to resolve application version resources
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - cloudformation:DescribeStacks
     * - cloudformation:ListStackResources
     * - resource-groups:GetGroup
     * - resource-groups:ListGroupResources
     * - servicecatalog:GetApplication
     * - servicecatalog:ListAssociatedResources
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_ResolveAppVersionResources.html
     */
    toResolveAppVersionResources() {
        return this.to('ResolveAppVersionResources');
    }
    /**
     * Grants permission to create application assessment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - cloudformation:DescribeStacks
     * - cloudformation:ListStackResources
     * - cloudwatch:DescribeAlarms
     * - cloudwatch:GetMetricData
     * - cloudwatch:GetMetricStatistics
     * - cloudwatch:PutMetricData
     * - ec2:DescribeRegions
     * - fis:GetExperimentTemplate
     * - fis:ListExperimentTemplates
     * - fis:ListExperiments
     * - resource-groups:GetGroup
     * - resource-groups:ListGroupResources
     * - servicecatalog:GetApplication
     * - servicecatalog:ListAssociatedResources
     * - ssm:GetParametersByPath
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_StartAppAssessment.html
     */
    toStartAppAssessment() {
        return this.to('StartAppAssessment');
    }
    /**
     * Grants permission to start the metrics export
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_StartMetricsExport.html
     */
    toStartMetricsExport() {
        return this.to('StartMetricsExport');
    }
    /**
     * Grants permission to start the grouping recommendation generation process
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_StartResourceGroupingRecommendationTask.html
     */
    toStartResourceGroupingRecommendationTask() {
        return this.to('StartResourceGroupingRecommendationTask');
    }
    /**
     * Grants permission to assign a resource tag
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update application
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_UpdateApp.html
     */
    toUpdateApp() {
        return this.to('UpdateApp');
    }
    /**
     * Grants permission to update application version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_UpdateAppVersion.html
     */
    toUpdateAppVersion() {
        return this.to('UpdateAppVersion');
    }
    /**
     * Grants permission to update application app component
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_UpdateAppVersionAppComponent.html
     */
    toUpdateAppVersionAppComponent() {
        return this.to('UpdateAppVersionAppComponent');
    }
    /**
     * Grants permission to update application resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_UpdateAppVersionResource.html
     */
    toUpdateAppVersionResource() {
        return this.to('UpdateAppVersionResource');
    }
    /**
     * Grants permission to update resiliency policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_UpdateResiliencyPolicy.html
     */
    toUpdateResiliencyPolicy() {
        return this.to('UpdateResiliencyPolicy');
    }
    /**
     * Adds a resource of type resiliency-policy to the statement
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_ResiliencyPolicy.html
     *
     * @param resiliencyPolicyId - Identifier for the resiliencyPolicyId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onResiliencyPolicy(resiliencyPolicyId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:resiliencehub:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:resiliency-policy/${resiliencyPolicyId}`);
    }
    /**
     * Adds a resource of type application to the statement
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_App.html
     *
     * @param appId - Identifier for the appId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApplication(appId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:resiliencehub:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:app/${appId}`);
    }
    /**
     * Adds a resource of type app-assessment to the statement
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_AppAssessment.html
     *
     * @param appAssessmentId - Identifier for the appAssessmentId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAppAssessment(appAssessmentId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:resiliencehub:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:app-assessment/${appAssessmentId}`);
    }
    /**
     * Adds a resource of type recommendation-template to the statement
     *
     * https://docs.aws.amazon.com/resilience-hub/latest/APIReference/API_RecommendationTemplate.html
     *
     * @param recommendationTemplateId - Identifier for the recommendationTemplateId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRecommendationTemplate(recommendationTemplateId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:resiliencehub:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:recommendation-template/${recommendationTemplateId}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateApp()
     * - .toCreateRecommendationTemplate()
     * - .toCreateResiliencyPolicy()
     * - .toStartAppAssessment()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - resiliency-policy
     * - application
     * - app-assessment
     * - recommendation-template
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateApp()
     * - .toCreateRecommendationTemplate()
     * - .toCreateResiliencyPolicy()
     * - .toStartAppAssessment()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [resiliencehub](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsresiliencehub.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'resiliencehub';
        this.accessLevelList = {
            Write: [
                'AcceptResourceGroupingRecommendations',
                'AddDraftAppVersionResourceMappings',
                'BatchUpdateRecommendationStatus',
                'CreateApp',
                'CreateAppVersionAppComponent',
                'CreateAppVersionResource',
                'CreateRecommendationTemplate',
                'CreateResiliencyPolicy',
                'DeleteApp',
                'DeleteAppAssessment',
                'DeleteAppInputSource',
                'DeleteAppVersionAppComponent',
                'DeleteAppVersionResource',
                'DeleteRecommendationTemplate',
                'DeleteResiliencyPolicy',
                'ImportResourcesToDraftAppVersion',
                'PublishAppVersion',
                'PutDraftAppVersionTemplate',
                'RejectResourceGroupingRecommendations',
                'RemoveDraftAppVersionResourceMappings',
                'ResolveAppVersionResources',
                'StartAppAssessment',
                'StartMetricsExport',
                'StartResourceGroupingRecommendationTask',
                'UpdateApp',
                'UpdateAppVersion',
                'UpdateAppVersionAppComponent',
                'UpdateAppVersionResource',
                'UpdateResiliencyPolicy'
            ],
            Read: [
                'DescribeApp',
                'DescribeAppAssessment',
                'DescribeAppVersion',
                'DescribeAppVersionAppComponent',
                'DescribeAppVersionResource',
                'DescribeAppVersionResourcesResolutionStatus',
                'DescribeAppVersionTemplate',
                'DescribeDraftAppVersionResourcesImportStatus',
                'DescribeMetricsExport',
                'DescribeResiliencyPolicy',
                'DescribeResourceGroupingRecommendationTask',
                'ListTagsForResource'
            ],
            List: [
                'ListAlarmRecommendations',
                'ListAppAssessmentComplianceDrifts',
                'ListAppAssessmentResourceDrifts',
                'ListAppAssessments',
                'ListAppComponentCompliances',
                'ListAppComponentRecommendations',
                'ListAppInputSources',
                'ListAppVersionAppComponents',
                'ListAppVersionResourceMappings',
                'ListAppVersionResources',
                'ListAppVersions',
                'ListApps',
                'ListMetrics',
                'ListRecommendationTemplates',
                'ListResiliencyPolicies',
                'ListResourceGroupingRecommendations',
                'ListSopRecommendations',
                'ListSuggestedResiliencyPolicies',
                'ListTestRecommendations',
                'ListUnsupportedAppVersionResources'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Resiliencehub = Resiliencehub;
//# sourceMappingURL=data:application/json;base64,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