"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ResourceExplorer2 = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [resource-explorer-2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsresourceexplorer.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class ResourceExplorer2 extends shared_1.PolicyStatement {
    /**
     * Grants permission to set the specified view as the default for this AWS Region in this AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_AssociateDefaultView.html
     */
    toAssociateDefaultView() {
        return this.to('AssociateDefaultView');
    }
    /**
     * Grants permission to retrieve details about views that you specify by a list of ARNs
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - resource-explorer-2:GetView
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_BatchGetView.html
     */
    toBatchGetView() {
        return this.to('BatchGetView');
    }
    /**
     * Grants permission to turn on Resource Explorer in the AWS Region in which you called this operation by creating an index
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_CreateIndex.html
     */
    toCreateIndex() {
        return this.to('CreateIndex');
    }
    /**
     * Grants permission to create managed view
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/userguide/API_ManagedView.html
     */
    toCreateManagedView() {
        return this.to('CreateManagedView');
    }
    /**
     * Grants permission to create a view that users can query
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_CreateView.html
     */
    toCreateView() {
        return this.to('CreateView');
    }
    /**
     * Grants permission to turn off Resource Explorer in the specified AWS Region by deleting the index
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_DeleteIndex.html
     */
    toDeleteIndex() {
        return this.to('DeleteIndex');
    }
    /**
     * Grants permission to delete the specified view's resource policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/userguide/manage-views-share.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete a view
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_DeleteView.html
     */
    toDeleteView() {
        return this.to('DeleteView');
    }
    /**
     * Grants permission to remove the default view for the AWS Region in which you call this operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_DisassociateDefaultView.html
     */
    toDisassociateDefaultView() {
        return this.to('DisassociateDefaultView');
    }
    /**
     * Grants permission to Resource Explorer to access account level data within your AWS Organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_GetAccountLevelServiceConfiguration.html
     */
    toGetAccountLevelServiceConfiguration() {
        return this.to('GetAccountLevelServiceConfiguration');
    }
    /**
     * Grants permission to retrieve the Amazon resource name (ARN) of the view that is the default for the AWS Region in which you call this operation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_GetDefaultView.html
     */
    toGetDefaultView() {
        return this.to('GetDefaultView');
    }
    /**
     * Grants permission to retrieve information about the index in the AWS Region in which you call this operation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_GetIndex.html
     */
    toGetIndex() {
        return this.to('GetIndex');
    }
    /**
     * Grants permission to get managed view
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_GetManagedView.html
     */
    toGetManagedView() {
        return this.to('GetManagedView');
    }
    /**
     * Grants permission to retrieve information about the specified view's resource policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/userguide/manage-views-share.html
     */
    toGetResourcePolicy() {
        return this.to('GetResourcePolicy');
    }
    /**
     * Grants permission to retrieve information about the specified view
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_GetView.html
     */
    toGetView() {
        return this.to('GetView');
    }
    /**
     * Grants permission to list the indexes in all AWS Regions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_ListIndexes.html
     */
    toListIndexes() {
        return this.to('ListIndexes');
    }
    /**
     * Grants permission to list the organization member account's indexes in all AWS Regions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_ListIndexesForMembers.html
     */
    toListIndexesForMembers() {
        return this.to('ListIndexesForMembers');
    }
    /**
     * Grants permission to list managed views
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_ListManagedViews.html
     */
    toListManagedViews() {
        return this.to('ListManagedViews');
    }
    /**
     * Grants permission to retrieve a list of all resource types currently supported by Resource Explorer
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_ListSupportedResourceTypes.html
     */
    toListSupportedResourceTypes() {
        return this.to('ListSupportedResourceTypes');
    }
    /**
     * Grants permission to list the tags that are attached to the specified resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list the Amazon resource names (ARNs) of all of the views available in the AWS Region in which you call this operation
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_ListViews.html
     */
    toListViews() {
        return this.to('ListViews');
    }
    /**
     * Grants permission to update the specified view's resource policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/userguide/manage-views-share.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to search for resources and display details about all resources that match the specified criteria
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifOperation()
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_Search.html
     */
    toSearch() {
        return this.to('Search');
    }
    /**
     * Grants permission to add one or more tag key and value pairs to the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove one or more tag key and value pairs from the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to change the type of the index from LOCAL to AGGREGATOR or back
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_UpdateIndexType.html
     */
    toUpdateIndexType() {
        return this.to('UpdateIndexType');
    }
    /**
     * Grants permission to modify some of the details of a view
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_UpdateView.html
     */
    toUpdateView() {
        return this.to('UpdateView');
    }
    /**
     * Adds a resource of type view to the statement
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_View.html
     *
     * @param viewName - Identifier for the viewName.
     * @param viewUuid - Identifier for the viewUuid.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onView(viewName, viewUuid, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:resource-explorer-2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:view/${viewName}/${viewUuid}`);
    }
    /**
     * Adds a resource of type index to the statement
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_Index.html
     *
     * @param indexUuid - Identifier for the indexUuid.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIndex(indexUuid, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:resource-explorer-2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:index/${indexUuid}`);
    }
    /**
     * Adds a resource of type managed-view to the statement
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/userguide/API_ManagedView.html
     *
     * @param managedViewName - Identifier for the managedViewName.
     * @param managedViewUuid - Identifier for the managedViewUuid.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onManagedView(managedViewName, managedViewUuid, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:resource-explorer-2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:managed-view/${managedViewName}/${managedViewUuid}`);
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateIndex()
     * - .toCreateView()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keyss attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - view
     * - index
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateIndex()
     * - .toCreateView()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the actual operation that is being invoked, available values: Search, ListResources
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsresourceexplorer.html
     *
     * Applies to actions:
     * - .toSearch()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOperation(value, operator) {
        return this.if(`Operation`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [resource-explorer-2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsresourceexplorer.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'resource-explorer-2';
        this.accessLevelList = {
            Write: [
                'AssociateDefaultView',
                'CreateIndex',
                'CreateManagedView',
                'CreateView',
                'DeleteIndex',
                'DeleteResourcePolicy',
                'DeleteView',
                'DisassociateDefaultView',
                'PutResourcePolicy',
                'UpdateIndexType',
                'UpdateView'
            ],
            Read: [
                'BatchGetView',
                'GetAccountLevelServiceConfiguration',
                'GetDefaultView',
                'GetIndex',
                'GetManagedView',
                'GetResourcePolicy',
                'GetView',
                'ListTagsForResource',
                'Search'
            ],
            List: [
                'ListIndexes',
                'ListIndexesForMembers',
                'ListManagedViews',
                'ListSupportedResourceTypes',
                'ListViews'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.ResourceExplorer2 = ResourceExplorer2;
//# sourceMappingURL=data:application/json;base64,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