"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Route53 = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [route53](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonroute53.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Route53 extends shared_1.PolicyStatement {
    /**
     * Grants permission to activate a key-signing key so that it can be used for signing by DNSSEC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ActivateKeySigningKey.html
     */
    toActivateKeySigningKey() {
        return this.to('ActivateKeySigningKey');
    }
    /**
     * Grants permission to associate an additional Amazon VPC with a private hosted zone
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifVPCs()
     *
     * Dependent actions:
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_AssociateVPCWithHostedZone.html
     */
    toAssociateVPCWithHostedZone() {
        return this.to('AssociateVPCWithHostedZone');
    }
    /**
     * Grants permission to create or delete CIDR blocks within a CIDR collection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ChangeCidrCollection.html
     */
    toChangeCidrCollection() {
        return this.to('ChangeCidrCollection');
    }
    /**
     * Grants permission to create, update, or delete a record, which contains authoritative DNS information for a specified domain or subdomain name
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifChangeResourceRecordSetsNormalizedRecordNames()
     * - .ifChangeResourceRecordSetsRecordTypes()
     * - .ifChangeResourceRecordSetsActions()
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ChangeResourceRecordSets.html
     */
    toChangeResourceRecordSets() {
        return this.to('ChangeResourceRecordSets');
    }
    /**
     * Grants permission to add, edit, or delete tags for a health check or a hosted zone
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ChangeTagsForResource.html
     */
    toChangeTagsForResource() {
        return this.to('ChangeTagsForResource');
    }
    /**
     * Grants permission to create a new CIDR collection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateCidrCollection.html
     */
    toCreateCidrCollection() {
        return this.to('CreateCidrCollection');
    }
    /**
     * Grants permission to create a new health check, which monitors the health and performance of your web applications, web servers, and other resources
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateHealthCheck.html
     */
    toCreateHealthCheck() {
        return this.to('CreateHealthCheck');
    }
    /**
     * Grants permission to create a public hosted zone, which you use to specify how the Domain Name System (DNS) routes traffic on the Internet for a domain, such as example.com, and its subdomains
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifVPCs()
     *
     * Dependent actions:
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateHostedZone.html
     */
    toCreateHostedZone() {
        return this.to('CreateHostedZone');
    }
    /**
     * Grants permission to create a new key-signing key associated with a hosted zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateKeySigningKey.html
     */
    toCreateKeySigningKey() {
        return this.to('CreateKeySigningKey');
    }
    /**
     * Grants permission to create a configuration for DNS query logging
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateQueryLoggingConfig.html
     */
    toCreateQueryLoggingConfig() {
        return this.to('CreateQueryLoggingConfig');
    }
    /**
     * Grants permission to create a delegation set (a group of four name servers) that can be reused by multiple hosted zones
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateReusableDelegationSet.html
     */
    toCreateReusableDelegationSet() {
        return this.to('CreateReusableDelegationSet');
    }
    /**
     * Grants permission to create a traffic policy, which you use to create multiple DNS records for one domain name (such as example.com) or one subdomain name (such as www.example.com)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateTrafficPolicy.html
     */
    toCreateTrafficPolicy() {
        return this.to('CreateTrafficPolicy');
    }
    /**
     * Grants permission to create records in a specified hosted zone based on the settings in a specified traffic policy version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateTrafficPolicyInstance.html
     */
    toCreateTrafficPolicyInstance() {
        return this.to('CreateTrafficPolicyInstance');
    }
    /**
     * Grants permission to create a new version of an existing traffic policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateTrafficPolicyVersion.html
     */
    toCreateTrafficPolicyVersion() {
        return this.to('CreateTrafficPolicyVersion');
    }
    /**
     * Grants permission to authorize the AWS account that created a specified VPC to submit an AssociateVPCWithHostedZone request, which associates the VPC with a specified hosted zone that was created by a different account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifVPCs()
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateVPCAssociationAuthorization.html
     */
    toCreateVPCAssociationAuthorization() {
        return this.to('CreateVPCAssociationAuthorization');
    }
    /**
     * Grants permission to deactivate a key-signing key so that it will not be used for signing by DNSSEC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeactivateKeySigningKey.html
     */
    toDeactivateKeySigningKey() {
        return this.to('DeactivateKeySigningKey');
    }
    /**
     * Grants permission to delete a CIDR collection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteCidrCollection.html
     */
    toDeleteCidrCollection() {
        return this.to('DeleteCidrCollection');
    }
    /**
     * Grants permission to delete a health check
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteHealthCheck.html
     */
    toDeleteHealthCheck() {
        return this.to('DeleteHealthCheck');
    }
    /**
     * Grants permission to delete a hosted zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteHostedZone.html
     */
    toDeleteHostedZone() {
        return this.to('DeleteHostedZone');
    }
    /**
     * Grants permission to delete a key-signing key
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteKeySigningKey.html
     */
    toDeleteKeySigningKey() {
        return this.to('DeleteKeySigningKey');
    }
    /**
     * Grants permission to delete a configuration for DNS query logging
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteQueryLoggingConfig.html
     */
    toDeleteQueryLoggingConfig() {
        return this.to('DeleteQueryLoggingConfig');
    }
    /**
     * Grants permission to delete a reusable delegation set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteReusableDelegationSet.html
     */
    toDeleteReusableDelegationSet() {
        return this.to('DeleteReusableDelegationSet');
    }
    /**
     * Grants permission to delete a traffic policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteTrafficPolicy.html
     */
    toDeleteTrafficPolicy() {
        return this.to('DeleteTrafficPolicy');
    }
    /**
     * Grants permission to delete a traffic policy instance and all the records that Route 53 created when you created the instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteTrafficPolicyInstance.html
     */
    toDeleteTrafficPolicyInstance() {
        return this.to('DeleteTrafficPolicyInstance');
    }
    /**
     * Grants permission to remove authorization for associating an Amazon Virtual Private Cloud with a Route 53 private hosted zone
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifVPCs()
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteVPCAssociationAuthorization.html
     */
    toDeleteVPCAssociationAuthorization() {
        return this.to('DeleteVPCAssociationAuthorization');
    }
    /**
     * Grants permission to disable DNSSEC signing in a specific hosted zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DisableHostedZoneDNSSEC.html
     */
    toDisableHostedZoneDNSSEC() {
        return this.to('DisableHostedZoneDNSSEC');
    }
    /**
     * Grants permission to disassociate an Amazon Virtual Private Cloud from a Route 53 private hosted zone
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifVPCs()
     *
     * Dependent actions:
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DisassociateVPCFromHostedZone.html
     */
    toDisassociateVPCFromHostedZone() {
        return this.to('DisassociateVPCFromHostedZone');
    }
    /**
     * Grants permission to enable DNSSEC signing in a specific hosted zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_EnableHostedZoneDNSSEC.html
     */
    toEnableHostedZoneDNSSEC() {
        return this.to('EnableHostedZoneDNSSEC');
    }
    /**
     * Grants permission to get the specified limit for the current account, for example, the maximum number of health checks that you can create using the account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetAccountLimit.html
     */
    toGetAccountLimit() {
        return this.to('GetAccountLimit');
    }
    /**
     * Grants permission to get the current status of a request to create, update, or delete one or more records
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetChange.html
     */
    toGetChange() {
        return this.to('GetChange');
    }
    /**
     * Grants permission to get a list of the IP ranges that are used by Route 53 health checkers to check the health of your resources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetCheckerIpRanges.html
     */
    toGetCheckerIpRanges() {
        return this.to('GetCheckerIpRanges');
    }
    /**
     * Grants permission to get information about DNSSEC for a specific hosted zone, including the key-signing keys in the hosted zone
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetDNSSEC.html
     */
    toGetDNSSEC() {
        return this.to('GetDNSSEC');
    }
    /**
     * Grants permission to get information about whether a specified geographic location is supported for Route 53 geolocation records
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetGeoLocation.html
     */
    toGetGeoLocation() {
        return this.to('GetGeoLocation');
    }
    /**
     * Grants permission to get information about a specified health check
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHealthCheck.html
     */
    toGetHealthCheck() {
        return this.to('GetHealthCheck');
    }
    /**
     * Grants permission to get the number of health checks that are associated with the current AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHealthCheckCount.html
     */
    toGetHealthCheckCount() {
        return this.to('GetHealthCheckCount');
    }
    /**
     * Grants permission to get the reason that a specified health check failed most recently
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHealthCheckLastFailureReason.html
     */
    toGetHealthCheckLastFailureReason() {
        return this.to('GetHealthCheckLastFailureReason');
    }
    /**
     * Grants permission to get the status of a specified health check
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHealthCheckStatus.html
     */
    toGetHealthCheckStatus() {
        return this.to('GetHealthCheckStatus');
    }
    /**
     * Grants permission to get information about a specified hosted zone including the four name servers that Route 53 assigned to the hosted zone
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHostedZone.html
     */
    toGetHostedZone() {
        return this.to('GetHostedZone');
    }
    /**
     * Grants permission to get the number of hosted zones that are associated with the current AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHostedZoneCount.html
     */
    toGetHostedZoneCount() {
        return this.to('GetHostedZoneCount');
    }
    /**
     * Grants permission to get the specified limit for a specified hosted zone
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHostedZoneLimit.html
     */
    toGetHostedZoneLimit() {
        return this.to('GetHostedZoneLimit');
    }
    /**
     * Grants permission to get information about a specified configuration for DNS query logging
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetQueryLoggingConfig.html
     */
    toGetQueryLoggingConfig() {
        return this.to('GetQueryLoggingConfig');
    }
    /**
     * Grants permission to get information about a specified reusable delegation set, including the four name servers that are assigned to the delegation set
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetReusableDelegationSet.html
     */
    toGetReusableDelegationSet() {
        return this.to('GetReusableDelegationSet');
    }
    /**
     * Grants permission to get the maximum number of hosted zones that you can associate with the specified reusable delegation set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetReusableDelegationSetLimit.html
     */
    toGetReusableDelegationSetLimit() {
        return this.to('GetReusableDelegationSetLimit');
    }
    /**
     * Grants permission to get information about a specified traffic policy version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetTrafficPolicy.html
     */
    toGetTrafficPolicy() {
        return this.to('GetTrafficPolicy');
    }
    /**
     * Grants permission to get information about a specified traffic policy instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetTrafficPolicyInstance.html
     */
    toGetTrafficPolicyInstance() {
        return this.to('GetTrafficPolicyInstance');
    }
    /**
     * Grants permission to get the number of traffic policy instances that are associated with the current AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetTrafficPolicyInstanceCount.html
     */
    toGetTrafficPolicyInstanceCount() {
        return this.to('GetTrafficPolicyInstanceCount');
    }
    /**
     * Grants permission to get a list of the CIDR blocks within a specified CIDR collection
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListCidrBlocks.html
     */
    toListCidrBlocks() {
        return this.to('ListCidrBlocks');
    }
    /**
     * Grants permission to get a list of the CIDR collections that are associated with the current AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListCidrCollections.html
     */
    toListCidrCollections() {
        return this.to('ListCidrCollections');
    }
    /**
     * Grants permission to get a list of the CIDR locations that belong to a specified CIDR collection
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListCidrLocations.html
     */
    toListCidrLocations() {
        return this.to('ListCidrLocations');
    }
    /**
     * Grants permission to get a list of geographic locations that Route 53 supports for geolocation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListGeoLocations.html
     */
    toListGeoLocations() {
        return this.to('ListGeoLocations');
    }
    /**
     * Grants permission to get a list of the health checks that are associated with the current AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListHealthChecks.html
     */
    toListHealthChecks() {
        return this.to('ListHealthChecks');
    }
    /**
     * Grants permission to get a list of the public and private hosted zones that are associated with the current AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListHostedZones.html
     */
    toListHostedZones() {
        return this.to('ListHostedZones');
    }
    /**
     * Grants permission to get a list of your hosted zones in lexicographic order. Hosted zones are sorted by name with the labels reversed, for example, com.example.www
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListHostedZonesByName.html
     */
    toListHostedZonesByName() {
        return this.to('ListHostedZonesByName');
    }
    /**
     * Grants permission to get a list of all the private hosted zones that a specified VPC is associated with
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifVPCs()
     *
     * Dependent actions:
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListHostedZonesByVPC.html
     */
    toListHostedZonesByVPC() {
        return this.to('ListHostedZonesByVPC');
    }
    /**
     * Grants permission to list the configurations for DNS query logging that are associated with the current AWS account or the configuration that is associated with a specified hosted zone
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListQueryLoggingConfigs.html
     */
    toListQueryLoggingConfigs() {
        return this.to('ListQueryLoggingConfigs');
    }
    /**
     * Grants permission to list the records in a specified hosted zone
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListResourceRecordSets.html
     */
    toListResourceRecordSets() {
        return this.to('ListResourceRecordSets');
    }
    /**
     * Grants permission to list the reusable delegation sets that are associated with the current AWS account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListReusableDelegationSets.html
     */
    toListReusableDelegationSets() {
        return this.to('ListReusableDelegationSets');
    }
    /**
     * Grants permission to list tags for one health check or hosted zone
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list tags for up to 10 health checks or hosted zones
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTagsForResources.html
     */
    toListTagsForResources() {
        return this.to('ListTagsForResources');
    }
    /**
     * Grants permission to get information about the latest version for every traffic policy that is associated with the current AWS account. Policies are listed in the order in which they were created
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicies.html
     */
    toListTrafficPolicies() {
        return this.to('ListTrafficPolicies');
    }
    /**
     * Grants permission to get information about the traffic policy instances that you created by using the current AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicyInstances.html
     */
    toListTrafficPolicyInstances() {
        return this.to('ListTrafficPolicyInstances');
    }
    /**
     * Grants permission to get information about the traffic policy instances that you created in a specified hosted zone
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicyInstancesByHostedZone.html
     */
    toListTrafficPolicyInstancesByHostedZone() {
        return this.to('ListTrafficPolicyInstancesByHostedZone');
    }
    /**
     * Grants permission to get information about the traffic policy instances that you created using a specified traffic policy version
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicyInstancesByPolicy.html
     */
    toListTrafficPolicyInstancesByPolicy() {
        return this.to('ListTrafficPolicyInstancesByPolicy');
    }
    /**
     * Grants permission to get information about all the versions for a specified traffic policy
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicyVersions.html
     */
    toListTrafficPolicyVersions() {
        return this.to('ListTrafficPolicyVersions');
    }
    /**
     * Grants permission to get a list of the VPCs that were created by other accounts and that can be associated with a specified hosted zone
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListVPCAssociationAuthorizations.html
     */
    toListVPCAssociationAuthorizations() {
        return this.to('ListVPCAssociationAuthorizations');
    }
    /**
     * Grants permission to get the value that Route 53 returns in response to a DNS query for a specified record name and type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_TestDNSAnswer.html
     */
    toTestDNSAnswer() {
        return this.to('TestDNSAnswer');
    }
    /**
     * Grants permission to update an existing health check
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_UpdateHealthCheck.html
     */
    toUpdateHealthCheck() {
        return this.to('UpdateHealthCheck');
    }
    /**
     * Grants permission to update the comment for a specified hosted zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_UpdateHostedZoneComment.html
     */
    toUpdateHostedZoneComment() {
        return this.to('UpdateHostedZoneComment');
    }
    /**
     * Grants permission to update the comment for a specified traffic policy version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_UpdateTrafficPolicyComment.html
     */
    toUpdateTrafficPolicyComment() {
        return this.to('UpdateTrafficPolicyComment');
    }
    /**
     * Grants permission to update the records in a specified hosted zone that were created based on the settings in a specified traffic policy version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_UpdateTrafficPolicyInstance.html
     */
    toUpdateTrafficPolicyInstance() {
        return this.to('UpdateTrafficPolicyInstance');
    }
    /**
     * Adds a resource of type cidrcollection to the statement
     *
     * @param id - Identifier for the id.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onCidrcollection(id, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:route53:::cidrcollection/${id}`);
    }
    /**
     * Adds a resource of type change to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_Change.html
     *
     * @param id - Identifier for the id.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onChange(id, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:route53:::change/${id}`);
    }
    /**
     * Adds a resource of type delegationset to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/route-53-concepts.html#route-53-concepts-reusable-delegation-set
     *
     * @param id - Identifier for the id.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onDelegationset(id, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:route53:::delegationset/${id}`);
    }
    /**
     * Adds a resource of type healthcheck to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/route-53-concepts.html#route-53-concepts-health-check
     *
     * @param id - Identifier for the id.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onHealthcheck(id, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:route53:::healthcheck/${id}`);
    }
    /**
     * Adds a resource of type hostedzone to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/route-53-concepts.html#route-53-concepts-hosted-zone
     *
     * @param id - Identifier for the id.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onHostedzone(id, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:route53:::hostedzone/${id}`);
    }
    /**
     * Adds a resource of type trafficpolicy to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/traffic-policies.html
     *
     * @param id - Identifier for the id.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onTrafficpolicy(id, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:route53:::trafficpolicy/${id}`);
    }
    /**
     * Adds a resource of type trafficpolicyinstance to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/traffic-policy-records.html
     *
     * @param id - Identifier for the id.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onTrafficpolicyinstance(id, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:route53:::trafficpolicyinstance/${id}`);
    }
    /**
     * Adds a resource of type queryloggingconfig to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/query-logs.html
     *
     * @param id - Identifier for the id.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onQueryloggingconfig(id, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:route53:::queryloggingconfig/${id}`);
    }
    /**
     * Adds a resource of type vpc to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html
     *
     * @param vpcId - Identifier for the vpcId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onVpc(vpcId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ec2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:vpc/${vpcId}`);
    }
    /**
     * Filters access by the change actions, CREATE, UPSERT, or DELETE, in a ChangeResourceRecordSets request
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/specifying-conditions-route53.html#route53_rrsetConditionKeys
     *
     * Applies to actions:
     * - .toChangeResourceRecordSets()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifChangeResourceRecordSetsActions(value, operator) {
        return this.if(`ChangeResourceRecordSetsActions`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the normalized DNS record names in a ChangeResourceRecordSets request
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/specifying-conditions-route53.html#route53_rrsetConditionKeys
     *
     * Applies to actions:
     * - .toChangeResourceRecordSets()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifChangeResourceRecordSetsNormalizedRecordNames(value, operator) {
        return this.if(`ChangeResourceRecordSetsNormalizedRecordNames`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the DNS record types in a ChangeResourceRecordSets request
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/specifying-conditions-route53.html#route53_rrsetConditionKeys
     *
     * Applies to actions:
     * - .toChangeResourceRecordSets()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifChangeResourceRecordSetsRecordTypes(value, operator) {
        return this.if(`ChangeResourceRecordSetsRecordTypes`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by VPCs in request
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/specifying-conditions-route53.html#route53_rrsetConditionKeys
     *
     * Applies to actions:
     * - .toAssociateVPCWithHostedZone()
     * - .toCreateHostedZone()
     * - .toCreateVPCAssociationAuthorization()
     * - .toDeleteVPCAssociationAuthorization()
     * - .toDisassociateVPCFromHostedZone()
     * - .toListHostedZonesByVPC()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVPCs(value, operator) {
        return this.if(`VPCs`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [route53](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonroute53.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'route53';
        this.accessLevelList = {
            Write: [
                'ActivateKeySigningKey',
                'AssociateVPCWithHostedZone',
                'ChangeCidrCollection',
                'ChangeResourceRecordSets',
                'CreateCidrCollection',
                'CreateHealthCheck',
                'CreateHostedZone',
                'CreateKeySigningKey',
                'CreateQueryLoggingConfig',
                'CreateReusableDelegationSet',
                'CreateTrafficPolicy',
                'CreateTrafficPolicyInstance',
                'CreateTrafficPolicyVersion',
                'CreateVPCAssociationAuthorization',
                'DeactivateKeySigningKey',
                'DeleteCidrCollection',
                'DeleteHealthCheck',
                'DeleteHostedZone',
                'DeleteKeySigningKey',
                'DeleteQueryLoggingConfig',
                'DeleteReusableDelegationSet',
                'DeleteTrafficPolicy',
                'DeleteTrafficPolicyInstance',
                'DeleteVPCAssociationAuthorization',
                'DisableHostedZoneDNSSEC',
                'DisassociateVPCFromHostedZone',
                'EnableHostedZoneDNSSEC',
                'UpdateHealthCheck',
                'UpdateHostedZoneComment',
                'UpdateTrafficPolicyComment',
                'UpdateTrafficPolicyInstance'
            ],
            Tagging: [
                'ChangeTagsForResource'
            ],
            Read: [
                'GetAccountLimit',
                'GetDNSSEC',
                'GetHealthCheck',
                'GetHostedZoneLimit',
                'GetQueryLoggingConfig',
                'GetReusableDelegationSetLimit',
                'GetTrafficPolicy',
                'GetTrafficPolicyInstance',
                'GetTrafficPolicyInstanceCount',
                'ListGeoLocations',
                'ListHealthChecks',
                'ListReusableDelegationSets',
                'ListTagsForResource',
                'ListTagsForResources',
                'ListTrafficPolicyInstances',
                'TestDNSAnswer'
            ],
            List: [
                'GetChange',
                'GetCheckerIpRanges',
                'GetGeoLocation',
                'GetHealthCheckCount',
                'GetHealthCheckLastFailureReason',
                'GetHealthCheckStatus',
                'GetHostedZone',
                'GetHostedZoneCount',
                'GetReusableDelegationSet',
                'ListCidrBlocks',
                'ListCidrCollections',
                'ListCidrLocations',
                'ListHostedZones',
                'ListHostedZonesByName',
                'ListHostedZonesByVPC',
                'ListQueryLoggingConfigs',
                'ListResourceRecordSets',
                'ListTrafficPolicies',
                'ListTrafficPolicyInstancesByHostedZone',
                'ListTrafficPolicyInstancesByPolicy',
                'ListTrafficPolicyVersions',
                'ListVPCAssociationAuthorizations'
            ]
        };
    }
}
exports.Route53 = Route53;
//# sourceMappingURL=data:application/json;base64,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