"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Route53RecoveryCluster = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [route53-recovery-cluster](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonroute53recoverycluster.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Route53RecoveryCluster extends shared_1.PolicyStatement {
    /**
     * Grants permission to get a routing control state
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/routing-control/latest/APIReference/API_GetRoutingControlState.html
     */
    toGetRoutingControlState() {
        return this.to('GetRoutingControlState');
    }
    /**
     * Grants permission to list routing controls
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/routing-control/latest/APIReference/API_ListRoutingControls.html
     */
    toListRoutingControls() {
        return this.to('ListRoutingControls');
    }
    /**
     * Grants permission to update a routing control state
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAllowSafetyRulesOverrides()
     *
     * https://docs.aws.amazon.com/routing-control/latest/APIReference/API_UpdateRoutingControlState.html
     */
    toUpdateRoutingControlState() {
        return this.to('UpdateRoutingControlState');
    }
    /**
     * Grants permission to update a batch of routing control states
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAllowSafetyRulesOverrides()
     *
     * https://docs.aws.amazon.com/routing-control/latest/APIReference/API_UpdateRoutingControlStates.html
     */
    toUpdateRoutingControlStates() {
        return this.to('UpdateRoutingControlStates');
    }
    /**
     * Adds a resource of type routingcontrol to the statement
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/routingcontrol.html
     *
     * @param controlPanelId - Identifier for the controlPanelId.
     * @param routingControlId - Identifier for the routingControlId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onRoutingcontrol(controlPanelId, routingControlId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:route53-recovery-control::${account ?? this.defaultAccount}:controlpanel/${controlPanelId}/routingcontrol/${routingControlId}`);
    }
    /**
     * Override safety rules to allow routing control state updates
     *
     * https://docs.aws.amazon.com/routing-control/latest/APIReference/API_UpdateRoutingControlState.html
     *
     * Applies to actions:
     * - .toUpdateRoutingControlState()
     * - .toUpdateRoutingControlStates()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifAllowSafetyRulesOverrides(value) {
        return this.if(`AllowSafetyRulesOverrides`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Statement provider for service [route53-recovery-cluster](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonroute53recoverycluster.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'route53-recovery-cluster';
        this.accessLevelList = {
            Read: [
                'GetRoutingControlState',
                'ListRoutingControls'
            ],
            Write: [
                'UpdateRoutingControlState',
                'UpdateRoutingControlStates'
            ]
        };
    }
}
exports.Route53RecoveryCluster = Route53RecoveryCluster;
//# sourceMappingURL=data:application/json;base64,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