"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3 = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [s3](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazons3.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class S3 extends shared_1.PolicyStatement {
    /**
     * Grants permission to abort a multipart upload
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointArn()
     * - .ifAccessGrantsInstanceArn()
     * - .ifDataAccessPointAccount()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_AbortMultipartUpload.html
     */
    toAbortMultipartUpload() {
        return this.to('AbortMultipartUpload');
    }
    /**
     * Grants permission to associate Access Grants identity center
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_AssociateAccessGrantsIdentityCenter.html
     */
    toAssociateAccessGrantsIdentityCenter() {
        return this.to('AssociateAccessGrantsIdentityCenter');
    }
    /**
     * Grants permission to allow circumvention of governance-mode object retention settings
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifRequestObjectTag()
     * - .ifRequestObjectTagKeys()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzCopySource()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     * - .ifXAmzMetadataDirective()
     * - .ifXAmzServerSideEncryption()
     * - .ifXAmzServerSideEncryptionAwsKmsKeyId()
     * - .ifXAmzServerSideEncryptionCustomerAlgorithm()
     * - .ifXAmzStorageClass()
     * - .ifXAmzWebsiteRedirectLocation()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/object-lock-managing.html#object-lock-managing-bypass
     */
    toBypassGovernanceRetention() {
        return this.to('BypassGovernanceRetention');
    }
    /**
     * Grants permission to create Access Grant
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateAccessGrant.html
     */
    toCreateAccessGrant() {
        return this.to('CreateAccessGrant');
    }
    /**
     * Grants permission to Create Access Grants Instance
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateAccessGrantsInstance.html
     */
    toCreateAccessGrantsInstance() {
        return this.to('CreateAccessGrantsInstance');
    }
    /**
     * Grants permission to create Access Grants location
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateAccessGrantsLocation.html
     */
    toCreateAccessGrantsLocation() {
        return this.to('CreateAccessGrantsLocation');
    }
    /**
     * Grants permission to create a new access point
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifLocationconstraint()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateAccessPoint.html
     */
    toCreateAccessPoint() {
        return this.to('CreateAccessPoint');
    }
    /**
     * Grants permission to create an object lambda enabled accesspoint
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateAccessPointForObjectLambda.html
     */
    toCreateAccessPointForObjectLambda() {
        return this.to('CreateAccessPointForObjectLambda');
    }
    /**
     * Grants permission to create a new bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifLocationconstraint()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     * - .ifXAmzObjectOwnership()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_CreateBucket.html
     */
    toCreateBucket() {
        return this.to('CreateBucket');
    }
    /**
     * Grants permission to create a new S3 Metadata configuration for a specified bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * Dependent actions:
     * - s3tables:CreateNamespace
     * - s3tables:CreateTable
     * - s3tables:GetTable
     * - s3tables:PutTablePolicy
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_CreateBucketMetadataTableConfiguration.html
     */
    toCreateBucketMetadataTableConfiguration() {
        return this.to('CreateBucketMetadataTableConfiguration');
    }
    /**
     * Grants permission to create a new Amazon S3 Batch Operations job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifRequestJobPriority()
     * - .ifRequestJobOperation()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateJob.html
     */
    toCreateJob() {
        return this.to('CreateJob');
    }
    /**
     * Grants permission to create a new Multi-Region Access Point
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateMultiRegionAccessPoint.html
     */
    toCreateMultiRegionAccessPoint() {
        return this.to('CreateMultiRegionAccessPoint');
    }
    /**
     * Grants permission to create an Amazon S3 Storage Lens group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateStorageLensGroup.html
     */
    toCreateStorageLensGroup() {
        return this.to('CreateStorageLensGroup');
    }
    /**
     * Grants permission to delete Access Grant
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessGrant.html
     */
    toDeleteAccessGrant() {
        return this.to('DeleteAccessGrant');
    }
    /**
     * Grants permission to Delete Access Grants Instance
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessGrantsInstance.html
     */
    toDeleteAccessGrantsInstance() {
        return this.to('DeleteAccessGrantsInstance');
    }
    /**
     * Grants permission to read Access grants instance resource policy
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessGrantsInstanceResourcePolicy.html
     */
    toDeleteAccessGrantsInstanceResourcePolicy() {
        return this.to('DeleteAccessGrantsInstanceResourcePolicy');
    }
    /**
     * Grants permission to delete Access Grants location
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessGrantsLocation.html
     */
    toDeleteAccessGrantsLocation() {
        return this.to('DeleteAccessGrantsLocation');
    }
    /**
     * Grants permission to delete the access point named in the URI
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointArn()
     * - .ifDataAccessPointAccount()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPoint.html
     */
    toDeleteAccessPoint() {
        return this.to('DeleteAccessPoint');
    }
    /**
     * Grants permission to delete the object lambda enabled access point named in the URI
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointArn()
     * - .ifDataAccessPointAccount()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointForObjectLambda.html
     */
    toDeleteAccessPointForObjectLambda() {
        return this.to('DeleteAccessPointForObjectLambda');
    }
    /**
     * Grants permission to delete the policy on a specified access point
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointArn()
     * - .ifDataAccessPointAccount()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointPolicy.html
     */
    toDeleteAccessPointPolicy() {
        return this.to('DeleteAccessPointPolicy');
    }
    /**
     * Grants permission to delete the policy on a specified object lambda enabled access point
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointArn()
     * - .ifDataAccessPointAccount()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointPolicyForObjectLambda.html
     */
    toDeleteAccessPointPolicyForObjectLambda() {
        return this.to('DeleteAccessPointPolicyForObjectLambda');
    }
    /**
     * Grants permission to delete the bucket named in the URI
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucket.html
     */
    toDeleteBucket() {
        return this.to('DeleteBucket');
    }
    /**
     * Grants permission to delete the S3 Metadata configuration for a specified bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucketMetadataTableConfiguration.html
     */
    toDeleteBucketMetadataTableConfiguration() {
        return this.to('DeleteBucketMetadataTableConfiguration');
    }
    /**
     * Grants permission to delete the policy on a specified bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucketPolicy.html
     */
    toDeleteBucketPolicy() {
        return this.to('DeleteBucketPolicy');
    }
    /**
     * Grants permission to remove the website configuration for a bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucketWebsite.html
     */
    toDeleteBucketWebsite() {
        return this.to('DeleteBucketWebsite');
    }
    /**
     * Grants permission to remove tags from an existing Amazon S3 Batch Operations job
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifExistingJobPriority()
     * - .ifExistingJobOperation()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteJobTagging.html
     */
    toDeleteJobTagging() {
        return this.to('DeleteJobTagging');
    }
    /**
     * Grants permission to delete the Multi-Region Access Point named in the URI
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteMultiRegionAccessPoint.html
     */
    toDeleteMultiRegionAccessPoint() {
        return this.to('DeleteMultiRegionAccessPoint');
    }
    /**
     * Grants permission to remove the null version of an object and insert a delete marker, which becomes the current version of the object
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAccessGrantsInstanceArn()
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObject.html
     */
    toDeleteObject() {
        return this.to('DeleteObject');
    }
    /**
     * Grants permission to use the tagging subresource to remove the entire tag set from the specified object
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObjectTagging.html
     */
    toDeleteObjectTagging() {
        return this.to('DeleteObjectTagging');
    }
    /**
     * Grants permission to remove a specific version of an object
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAccessGrantsInstanceArn()
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObject.html
     */
    toDeleteObjectVersion() {
        return this.to('DeleteObjectVersion');
    }
    /**
     * Grants permission to remove the entire tag set for a specific version of the object
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObjectTagging.html
     */
    toDeleteObjectVersionTagging() {
        return this.to('DeleteObjectVersionTagging');
    }
    /**
     * Grants permission to delete an existing Amazon S3 Storage Lens configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteStorageLensConfiguration.html
     */
    toDeleteStorageLensConfiguration() {
        return this.to('DeleteStorageLensConfiguration');
    }
    /**
     * Grants permission to remove tags from an existing Amazon S3 Storage Lens configuration
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteStorageLensConfigurationTagging.html
     */
    toDeleteStorageLensConfigurationTagging() {
        return this.to('DeleteStorageLensConfigurationTagging');
    }
    /**
     * Grants permission to delete an existing S3 Storage Lens group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteStorageLensGroup.html
     */
    toDeleteStorageLensGroup() {
        return this.to('DeleteStorageLensGroup');
    }
    /**
     * Grants permission to retrieve the configuration parameters and status for a batch operations job
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DescribeJob.html
     */
    toDescribeJob() {
        return this.to('DescribeJob');
    }
    /**
     * Grants permission to retrieve the configurations for a Multi-Region Access Point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DescribeMultiRegionAccessPointOperation.html
     */
    toDescribeMultiRegionAccessPointOperation() {
        return this.to('DescribeMultiRegionAccessPointOperation');
    }
    /**
     * Grants permission to disassociate Access Grants identity center
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DissociateAccessGrantsIdentityCenter.html
     */
    toDissociateAccessGrantsIdentityCenter() {
        return this.to('DissociateAccessGrantsIdentityCenter');
    }
    /**
     * Grants permission to uses the accelerate subresource to return the Transfer Acceleration state of a bucket, which is either Enabled or Suspended
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAccelerateConfiguration.html
     */
    toGetAccelerateConfiguration() {
        return this.to('GetAccelerateConfiguration');
    }
    /**
     * Grants permission to read Access Grant
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessGrant.html
     */
    toGetAccessGrant() {
        return this.to('GetAccessGrant');
    }
    /**
     * Grants permission to Read Access Grants Instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessGrantsInstance.html
     */
    toGetAccessGrantsInstance() {
        return this.to('GetAccessGrantsInstance');
    }
    /**
     * Grants permission to Read Access Grants Instance by prefix
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessGrantsInstanceForPrefix.html
     */
    toGetAccessGrantsInstanceForPrefix() {
        return this.to('GetAccessGrantsInstanceForPrefix');
    }
    /**
     * Grants permission to read Access grants instance resource policy
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessGrantsInstanceResourcePolicy.html
     */
    toGetAccessGrantsInstanceResourcePolicy() {
        return this.to('GetAccessGrantsInstanceResourcePolicy');
    }
    /**
     * Grants permission to read Access Grants location
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessGrantsLocation.html
     */
    toGetAccessGrantsLocation() {
        return this.to('GetAccessGrantsLocation');
    }
    /**
     * Grants permission to return configuration information about the specified access point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPoint.html
     */
    toGetAccessPoint() {
        return this.to('GetAccessPoint');
    }
    /**
     * Grants permission to retrieve the configuration of the object lambda enabled access point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointArn()
     * - .ifDataAccessPointAccount()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointConfigurationForObjectLambda.html
     */
    toGetAccessPointConfigurationForObjectLambda() {
        return this.to('GetAccessPointConfigurationForObjectLambda');
    }
    /**
     * Grants permission to create an object lambda enabled accesspoint
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointForObjectLambda.html
     */
    toGetAccessPointForObjectLambda() {
        return this.to('GetAccessPointForObjectLambda');
    }
    /**
     * Grants permission to return the access point policy associated with the specified access point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicy.html
     */
    toGetAccessPointPolicy() {
        return this.to('GetAccessPointPolicy');
    }
    /**
     * Grants permission to return the access point policy associated with the specified object lambda enabled access point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicyForObjectLambda.html
     */
    toGetAccessPointPolicyForObjectLambda() {
        return this.to('GetAccessPointPolicyForObjectLambda');
    }
    /**
     * Grants permission to return the policy status for a specific access point policy
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicyStatus.html
     */
    toGetAccessPointPolicyStatus() {
        return this.to('GetAccessPointPolicyStatus');
    }
    /**
     * Grants permission to return the policy status for a specific object lambda access point policy
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicyStatusForObjectLambda.html
     */
    toGetAccessPointPolicyStatusForObjectLambda() {
        return this.to('GetAccessPointPolicyStatusForObjectLambda');
    }
    /**
     * Grants permission to retrieve the PublicAccessBlock configuration for an AWS account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetPublicAccessBlock.html
     */
    toGetAccountPublicAccessBlock() {
        return this.to('GetAccountPublicAccessBlock');
    }
    /**
     * Grants permission to get an analytics configuration from an Amazon S3 bucket, identified by the analytics configuration ID
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAnalyticsConfiguration.html
     */
    toGetAnalyticsConfiguration() {
        return this.to('GetAnalyticsConfiguration');
    }
    /**
     * Grants permission to use the acl subresource to return the access control list (ACL) of an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAcl.html
     */
    toGetBucketAcl() {
        return this.to('GetBucketAcl');
    }
    /**
     * Grants permission to return the CORS configuration information set for an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketCors.html
     */
    toGetBucketCORS() {
        return this.to('GetBucketCORS');
    }
    /**
     * Grants permission to return the Region that an Amazon S3 bucket resides in
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLocation.html
     */
    toGetBucketLocation() {
        return this.to('GetBucketLocation');
    }
    /**
     * Grants permission to return the logging status of an Amazon S3 bucket and the permissions users have to view or modify that status
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLogging.html
     */
    toGetBucketLogging() {
        return this.to('GetBucketLogging');
    }
    /**
     * Grants permission to return the S3 Metadata configuration for a specified bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketMetadataTableConfiguration.html
     */
    toGetBucketMetadataTableConfiguration() {
        return this.to('GetBucketMetadataTableConfiguration');
    }
    /**
     * Grants permission to get the notification configuration of an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketNotification.html
     */
    toGetBucketNotification() {
        return this.to('GetBucketNotification');
    }
    /**
     * Grants permission to get the Object Lock configuration of an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifSignatureversion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectLockConfiguration.html
     */
    toGetBucketObjectLockConfiguration() {
        return this.to('GetBucketObjectLockConfiguration');
    }
    /**
     * Grants permission to retrieve ownership controls on a bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketOwnershipControls.html
     */
    toGetBucketOwnershipControls() {
        return this.to('GetBucketOwnershipControls');
    }
    /**
     * Grants permission to return the policy of the specified bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketPolicy.html
     */
    toGetBucketPolicy() {
        return this.to('GetBucketPolicy');
    }
    /**
     * Grants permission to retrieve the policy status for a specific Amazon S3 bucket, which indicates whether the bucket is public
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketPolicyStatus.html
     */
    toGetBucketPolicyStatus() {
        return this.to('GetBucketPolicyStatus');
    }
    /**
     * Grants permission to retrieve the PublicAccessBlock configuration for an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetPublicAccessBlock.html
     */
    toGetBucketPublicAccessBlock() {
        return this.to('GetBucketPublicAccessBlock');
    }
    /**
     * Grants permission to return the request payment configuration for an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketRequestPayment.html
     */
    toGetBucketRequestPayment() {
        return this.to('GetBucketRequestPayment');
    }
    /**
     * Grants permission to return the tag set associated with an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketTagging.html
     */
    toGetBucketTagging() {
        return this.to('GetBucketTagging');
    }
    /**
     * Grants permission to return the versioning state of an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketVersioning.html
     */
    toGetBucketVersioning() {
        return this.to('GetBucketVersioning');
    }
    /**
     * Grants permission to return the website configuration for an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketWebsite.html
     */
    toGetBucketWebsite() {
        return this.to('GetBucketWebsite');
    }
    /**
     * Grants permission to get Access
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetDataAccess.html
     */
    toGetDataAccess() {
        return this.to('GetDataAccess');
    }
    /**
     * Grants permission to return the default encryption configuration an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketEncryption.html
     */
    toGetEncryptionConfiguration() {
        return this.to('GetEncryptionConfiguration');
    }
    /**
     * Grants permission to get an or list all Amazon S3 Intelligent Tiering configuration in a S3 Bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketIntelligentTieringConfiguration.html
     */
    toGetIntelligentTieringConfiguration() {
        return this.to('GetIntelligentTieringConfiguration');
    }
    /**
     * Grants permission to return an inventory configuration from an Amazon S3 bucket, identified by the inventory configuration ID
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketInventoryConfiguration.html
     */
    toGetInventoryConfiguration() {
        return this.to('GetInventoryConfiguration');
    }
    /**
     * Grants permission to return the tag set of an existing Amazon S3 Batch Operations job
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetJobTagging.html
     */
    toGetJobTagging() {
        return this.to('GetJobTagging');
    }
    /**
     * Grants permission to return the lifecycle configuration information set on an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLifecycleConfiguration.html
     */
    toGetLifecycleConfiguration() {
        return this.to('GetLifecycleConfiguration');
    }
    /**
     * Grants permission to get a metrics configuration from an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketMetricsConfiguration.html
     */
    toGetMetricsConfiguration() {
        return this.to('GetMetricsConfiguration');
    }
    /**
     * Grants permission to return configuration information about the specified Multi-Region Access Point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetMultiRegionAccessPoint.html
     */
    toGetMultiRegionAccessPoint() {
        return this.to('GetMultiRegionAccessPoint');
    }
    /**
     * Grants permission to return the access point policy associated with the specified Multi-Region Access Point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetMultiRegionAccessPointPolicy.html
     */
    toGetMultiRegionAccessPointPolicy() {
        return this.to('GetMultiRegionAccessPointPolicy');
    }
    /**
     * Grants permission to return the policy status for a specific Multi-Region Access Point policy
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetMultiRegionAccessPointPolicyStatus.html
     */
    toGetMultiRegionAccessPointPolicyStatus() {
        return this.to('GetMultiRegionAccessPointPolicyStatus');
    }
    /**
     * Grants permission to return the route configuration for a Multi-Region Access Point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetMultiRegionAccessPointRoutes.html
     */
    toGetMultiRegionAccessPointRoutes() {
        return this.to('GetMultiRegionAccessPointRoutes');
    }
    /**
     * Grants permission to retrieve objects from Amazon S3
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAccessGrantsInstanceArn()
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifIfMatch()
     * - .ifIfNoneMatch()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObject.html
     */
    toGetObject() {
        return this.to('GetObject');
    }
    /**
     * Grants permission to return the access control list (ACL) of an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAccessGrantsInstanceArn()
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectAcl.html
     */
    toGetObjectAcl() {
        return this.to('GetObjectAcl');
    }
    /**
     * Grants permission to retrieve attributes related to a specific object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectAttributes.html
     */
    toGetObjectAttributes() {
        return this.to('GetObjectAttributes');
    }
    /**
     * Grants permission to get an object's current Legal Hold status
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectLegalHold.html
     */
    toGetObjectLegalHold() {
        return this.to('GetObjectLegalHold');
    }
    /**
     * Grants permission to retrieve the retention settings for an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectRetention.html
     */
    toGetObjectRetention() {
        return this.to('GetObjectRetention');
    }
    /**
     * Grants permission to return the tag set of an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTagging.html
     */
    toGetObjectTagging() {
        return this.to('GetObjectTagging');
    }
    /**
     * Grants permission to return torrent files from an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTorrent.html
     */
    toGetObjectTorrent() {
        return this.to('GetObjectTorrent');
    }
    /**
     * Grants permission to retrieve a specific version of an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAccessGrantsInstanceArn()
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObject.html
     */
    toGetObjectVersion() {
        return this.to('GetObjectVersion');
    }
    /**
     * Grants permission to return the access control list (ACL) of a specific object version
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAccessGrantsInstanceArn()
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectAcl.html
     */
    toGetObjectVersionAcl() {
        return this.to('GetObjectVersionAcl');
    }
    /**
     * Grants permission to retrieve attributes related to a specific version of an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectAttributes.html
     */
    toGetObjectVersionAttributes() {
        return this.to('GetObjectVersionAttributes');
    }
    /**
     * Grants permission to replicate both unencrypted objects and objects encrypted with SSE-S3 or SSE-KMS
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/replication-config-for-kms-objects.html
     */
    toGetObjectVersionForReplication() {
        return this.to('GetObjectVersionForReplication');
    }
    /**
     * Grants permission to return the tag set for a specific version of the object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/setting-repl-config-perm-overview.html
     */
    toGetObjectVersionTagging() {
        return this.to('GetObjectVersionTagging');
    }
    /**
     * Grants permission to get Torrent files about a different version using the versionId subresource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTorrent.html
     */
    toGetObjectVersionTorrent() {
        return this.to('GetObjectVersionTorrent');
    }
    /**
     * Grants permission to get the replication configuration information set on an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketReplication.html
     */
    toGetReplicationConfiguration() {
        return this.to('GetReplicationConfiguration');
    }
    /**
     * Grants permission to get an Amazon S3 Storage Lens configuration
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetStorageLensConfiguration.html
     */
    toGetStorageLensConfiguration() {
        return this.to('GetStorageLensConfiguration');
    }
    /**
     * Grants permission to get the tag set of an existing Amazon S3 Storage Lens configuration
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetStorageLensConfigurationTagging.html
     */
    toGetStorageLensConfigurationTagging() {
        return this.to('GetStorageLensConfigurationTagging');
    }
    /**
     * Grants permission to get an Amazon S3 Storage Lens dashboard
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/storage_lens_dashboard.html
     */
    toGetStorageLensDashboard() {
        return this.to('GetStorageLensDashboard');
    }
    /**
     * Grants permission to get an Amazon S3 Storage Lens group
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetStorageLensGroup.html
     */
    toGetStorageLensGroup() {
        return this.to('GetStorageLensGroup');
    }
    /**
     * Grants permission to initiate the replication process by setting replication status of an object to pending
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourceAccount()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/setting-repl-config-perm-overview.html
     */
    toInitiateReplication() {
        return this.to('InitiateReplication');
    }
    /**
     * Grants permission to list Access Grant
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListAccessGrants.html
     */
    toListAccessGrants() {
        return this.to('ListAccessGrants');
    }
    /**
     * Grants permission to List Access Grants Instances
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListAccessGrantsInstances.html
     */
    toListAccessGrantsInstances() {
        return this.to('ListAccessGrantsInstances');
    }
    /**
     * Grants permission to list Access Grants locations
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListAccessGrantsLocations.html
     */
    toListAccessGrantsLocations() {
        return this.to('ListAccessGrantsLocations');
    }
    /**
     * Grants permission to list access points
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListAccessPoints.html
     */
    toListAccessPoints() {
        return this.to('ListAccessPoints');
    }
    /**
     * Grants permission to list object lambda enabled accesspoints
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListAccessPointsForObjectLambda.html
     */
    toListAccessPointsForObjectLambda() {
        return this.to('ListAccessPointsForObjectLambda');
    }
    /**
     * Grants permission to list all buckets owned by the authenticated sender of the request
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListBuckets.html
     */
    toListAllMyBuckets() {
        return this.to('ListAllMyBuckets');
    }
    /**
     * Grants permission to list some or all of the objects in an Amazon S3 bucket (up to 1000)
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAccessGrantsInstanceArn()
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifDelimiter()
     * - .ifMaxKeys()
     * - .ifPrefix()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListObjectsV2.html
     */
    toListBucket() {
        return this.to('ListBucket');
    }
    /**
     * Grants permission to list in-progress multipart uploads
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAccessGrantsInstanceArn()
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListMultipartUploads.html
     */
    toListBucketMultipartUploads() {
        return this.to('ListBucketMultipartUploads');
    }
    /**
     * Grants permission to list metadata about all the versions of objects in an Amazon S3 bucket
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAccessGrantsInstanceArn()
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifDelimiter()
     * - .ifMaxKeys()
     * - .ifPrefix()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListObjectVersions.html
     */
    toListBucketVersions() {
        return this.to('ListBucketVersions');
    }
    /**
     * Grants permission to list caller's Access Grant
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListCallerAccessGrants.html
     */
    toListCallerAccessGrants() {
        return this.to('ListCallerAccessGrants');
    }
    /**
     * Grants permission to list current jobs and jobs that have ended recently
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListJobs.html
     */
    toListJobs() {
        return this.to('ListJobs');
    }
    /**
     * Grants permission to list Multi-Region Access Points
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListMultiRegionAccessPoints.html
     */
    toListMultiRegionAccessPoints() {
        return this.to('ListMultiRegionAccessPoints');
    }
    /**
     * Grants permission to list the parts that have been uploaded for a specific multipart upload
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAccessGrantsInstanceArn()
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListParts.html
     */
    toListMultipartUploadParts() {
        return this.to('ListMultipartUploadParts');
    }
    /**
     * Grants permission to list Amazon S3 Storage Lens configurations
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListStorageLensConfigurations.html
     */
    toListStorageLensConfigurations() {
        return this.to('ListStorageLensConfigurations');
    }
    /**
     * Grants permission to list S3 Storage Lens groups
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListStorageLensGroups.html
     */
    toListStorageLensGroups() {
        return this.to('ListStorageLensGroups');
    }
    /**
     * Grants permission to list the tags attached to the specified resource
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to change replica ownership
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/replication-change-owner.html#repl-ownership-add-role-permission
     */
    toObjectOwnerOverrideToBucketOwner() {
        return this.to('ObjectOwnerOverrideToBucketOwner');
    }
    /**
     * Grants permission to pause S3 Replication from target source buckets to destination buckets
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDestinationRegion()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * Dependent actions:
     * - s3:GetReplicationConfiguration
     * - s3:PutReplicationConfiguration
     *
     * https://docs.aws.amazon.com/fis/latest/userguide/fis-actions-reference.html#bucket-pause-replication
     */
    toPauseReplication() {
        return this.to('PauseReplication');
    }
    /**
     * Grants permission to use the accelerate subresource to set the Transfer Acceleration state of an existing S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAccelerateConfiguration.html
     */
    toPutAccelerateConfiguration() {
        return this.to('PutAccelerateConfiguration');
    }
    /**
     * Grants permission to put Access grants instance resource policy
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutAccessGrantsInstanceResourcePolicy.html
     */
    toPutAccessGrantsInstanceResourcePolicy() {
        return this.to('PutAccessGrantsInstanceResourcePolicy');
    }
    /**
     * Grants permission to set the configuration of the object lambda enabled access point
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointArn()
     * - .ifDataAccessPointAccount()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutAccessPointConfigurationForObjectLambda.html
     */
    toPutAccessPointConfigurationForObjectLambda() {
        return this.to('PutAccessPointConfigurationForObjectLambda');
    }
    /**
     * Grants permission to associate an access policy with a specified access point
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutAccessPointPolicy.html
     */
    toPutAccessPointPolicy() {
        return this.to('PutAccessPointPolicy');
    }
    /**
     * Grants permission to associate an access policy with a specified object lambda enabled access point
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutAccessPointPolicyForObjectLambda.html
     */
    toPutAccessPointPolicyForObjectLambda() {
        return this.to('PutAccessPointPolicyForObjectLambda');
    }
    /**
     * Grants permission to associate public access block configurations with a specified access point, while creating a access point
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/access-control-block-public-access.html#access-control-block-public-access-examples-access-point
     */
    toPutAccessPointPublicAccessBlock() {
        return this.to('PutAccessPointPublicAccessBlock');
    }
    /**
     * Grants permission to create or modify the PublicAccessBlock configuration for an AWS account
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutPublicAccessBlock.html
     */
    toPutAccountPublicAccessBlock() {
        return this.to('PutAccountPublicAccessBlock');
    }
    /**
     * Grants permission to set an analytics configuration for the bucket, specified by the analytics configuration ID
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAnalyticsConfiguration.html
     */
    toPutAnalyticsConfiguration() {
        return this.to('PutAnalyticsConfiguration');
    }
    /**
     * Grants permission to set the permissions on an existing bucket using access control lists (ACLs)
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAcl.html
     */
    toPutBucketAcl() {
        return this.to('PutBucketAcl');
    }
    /**
     * Grants permission to set the CORS configuration for an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketCors.html
     */
    toPutBucketCORS() {
        return this.to('PutBucketCORS');
    }
    /**
     * Grants permission to set the logging parameters for an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketLogging.html
     */
    toPutBucketLogging() {
        return this.to('PutBucketLogging');
    }
    /**
     * Grants permission to receive notifications when certain events happen in an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketNotification.html
     */
    toPutBucketNotification() {
        return this.to('PutBucketNotification');
    }
    /**
     * Grants permission to put Object Lock configuration on a specific bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     * - .ifSignatureversion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectLockConfiguration.html
     */
    toPutBucketObjectLockConfiguration() {
        return this.to('PutBucketObjectLockConfiguration');
    }
    /**
     * Grants permission to add, replace or delete ownership controls on a bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketOwnershipControls.html
     */
    toPutBucketOwnershipControls() {
        return this.to('PutBucketOwnershipControls');
    }
    /**
     * Grants permission to add or replace a bucket policy on a bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketPolicy.html
     */
    toPutBucketPolicy() {
        return this.to('PutBucketPolicy');
    }
    /**
     * Grants permission to create or modify the PublicAccessBlock configuration for a specific Amazon S3 bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutPublicAccessBlock.html
     */
    toPutBucketPublicAccessBlock() {
        return this.to('PutBucketPublicAccessBlock');
    }
    /**
     * Grants permission to set the request payment configuration of a bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketRequestPayment.html
     */
    toPutBucketRequestPayment() {
        return this.to('PutBucketRequestPayment');
    }
    /**
     * Grants permission to add a set of tags to an existing Amazon S3 bucket
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketTagging.html
     */
    toPutBucketTagging() {
        return this.to('PutBucketTagging');
    }
    /**
     * Grants permission to set the versioning state of an existing Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketVersioning.html
     */
    toPutBucketVersioning() {
        return this.to('PutBucketVersioning');
    }
    /**
     * Grants permission to set the configuration of the website that is specified in the website subresource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketWebsite.html
     */
    toPutBucketWebsite() {
        return this.to('PutBucketWebsite');
    }
    /**
     * Grants permission to set the encryption configuration for an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketEncryption.html
     */
    toPutEncryptionConfiguration() {
        return this.to('PutEncryptionConfiguration');
    }
    /**
     * Grants permission to create new or update or delete an existing Amazon S3 Intelligent Tiering configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketIntelligentTieringConfiguration.html
     */
    toPutIntelligentTieringConfiguration() {
        return this.to('PutIntelligentTieringConfiguration');
    }
    /**
     * Grants permission to add an inventory configuration to the bucket, identified by the inventory ID
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifInventoryAccessibleOptionalFields()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketInventoryConfiguration.html
     */
    toPutInventoryConfiguration() {
        return this.to('PutInventoryConfiguration');
    }
    /**
     * Grants permission to replace tags on an existing Amazon S3 Batch Operations job
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifExistingJobPriority()
     * - .ifExistingJobOperation()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutJobTagging.html
     */
    toPutJobTagging() {
        return this.to('PutJobTagging');
    }
    /**
     * Grants permission to create a new lifecycle configuration for the bucket or replace an existing lifecycle configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketLifecycleConfiguration.html
     */
    toPutLifecycleConfiguration() {
        return this.to('PutLifecycleConfiguration');
    }
    /**
     * Grants permission to set or update a metrics configuration for the CloudWatch request metrics from an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketMetricsConfiguration.html
     */
    toPutMetricsConfiguration() {
        return this.to('PutMetricsConfiguration');
    }
    /**
     * Grants permission to associate an access policy with a specified Multi-Region Access Point
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutMultiRegionAccessPointPolicy.html
     */
    toPutMultiRegionAccessPointPolicy() {
        return this.to('PutMultiRegionAccessPointPolicy');
    }
    /**
     * Grants permission to add an object to a bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAccessGrantsInstanceArn()
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifRequestObjectTag()
     * - .ifRequestObjectTagKeys()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzCopySource()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     * - .ifXAmzMetadataDirective()
     * - .ifXAmzServerSideEncryption()
     * - .ifXAmzServerSideEncryptionAwsKmsKeyId()
     * - .ifXAmzServerSideEncryptionCustomerAlgorithm()
     * - .ifXAmzStorageClass()
     * - .ifXAmzWebsiteRedirectLocation()
     * - .ifObjectLockMode()
     * - .ifObjectLockRetainUntilDate()
     * - .ifObjectLockRemainingRetentionDays()
     * - .ifObjectLockLegalHold()
     * - .ifIfMatch()
     * - .ifIfNoneMatch()
     * - .ifObjectCreationOperation()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObject.html
     */
    toPutObject() {
        return this.to('PutObject');
    }
    /**
     * Grants permission to set the access control list (ACL) permissions for new or existing objects in an S3 bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAccessGrantsInstanceArn()
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     * - .ifXAmzStorageClass()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectAcl.html
     */
    toPutObjectAcl() {
        return this.to('PutObjectAcl');
    }
    /**
     * Grants permission to apply a Legal Hold configuration to the specified object
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifObjectLockLegalHold()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectLegalHold.html
     */
    toPutObjectLegalHold() {
        return this.to('PutObjectLegalHold');
    }
    /**
     * Grants permission to place an Object Retention configuration on an object
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifObjectLockMode()
     * - .ifObjectLockRetainUntilDate()
     * - .ifObjectLockRemainingRetentionDays()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectRetention.html
     */
    toPutObjectRetention() {
        return this.to('PutObjectRetention');
    }
    /**
     * Grants permission to set the supplied tag-set to an object that already exists in a bucket
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifRequestObjectTag()
     * - .ifRequestObjectTagKeys()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectTagging.html
     */
    toPutObjectTagging() {
        return this.to('PutObjectTagging');
    }
    /**
     * Grants permission to use the acl subresource to set the access control list (ACL) permissions for an object that already exists in a bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAccessGrantsInstanceArn()
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifVersionid()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     * - .ifXAmzStorageClass()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectAcl.html
     */
    toPutObjectVersionAcl() {
        return this.to('PutObjectVersionAcl');
    }
    /**
     * Grants permission to set the supplied tag-set for a specific version of an object
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifRequestObjectTag()
     * - .ifRequestObjectTagKeys()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectTagging.html
     */
    toPutObjectVersionTagging() {
        return this.to('PutObjectVersionTagging');
    }
    /**
     * Grants permission to create a new replication configuration or replace an existing one
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifIsReplicationPauseRequest()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketReplication.html
     */
    toPutReplicationConfiguration() {
        return this.to('PutReplicationConfiguration');
    }
    /**
     * Grants permission to create or update an Amazon S3 Storage Lens configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutStorageLensConfiguration.html
     */
    toPutStorageLensConfiguration() {
        return this.to('PutStorageLensConfiguration');
    }
    /**
     * Grants permission to put or replace tags on an existing Amazon S3 Storage Lens configuration
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutStorageLensConfigurationTagging.html
     */
    toPutStorageLensConfigurationTagging() {
        return this.to('PutStorageLensConfigurationTagging');
    }
    /**
     * Grants permission to replicate delete markers to the destination bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/setting-repl-config-perm-overview.html
     */
    toReplicateDelete() {
        return this.to('ReplicateDelete');
    }
    /**
     * Grants permission to replicate objects and object tags to the destination bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifXAmzServerSideEncryption()
     * - .ifXAmzServerSideEncryptionAwsKmsKeyId()
     * - .ifXAmzServerSideEncryptionCustomerAlgorithm()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/setting-repl-config-perm-overview.html
     */
    toReplicateObject() {
        return this.to('ReplicateObject');
    }
    /**
     * Grants permission to replicate object tags to the destination bucket
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/setting-repl-config-perm-overview.html
     */
    toReplicateTags() {
        return this.to('ReplicateTags');
    }
    /**
     * Grants permission to restore an archived copy of an object back into Amazon S3
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_RestoreObject.html
     */
    toRestoreObject() {
        return this.to('RestoreObject');
    }
    /**
     * Grants permission to submit a route configuration update for a Multi-Region Access Point
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifSignatureAge()
     * - .ifTlsVersion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_SubmitMultiRegionAccessPointRoutes.html
     */
    toSubmitMultiRegionAccessPointRoutes() {
        return this.to('SubmitMultiRegionAccessPointRoutes');
    }
    /**
     * Grants permission to add tags to the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update Access Grants location
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateAccessGrantsLocation.html
     */
    toUpdateAccessGrantsLocation() {
        return this.to('UpdateAccessGrantsLocation');
    }
    /**
     * Grants permission to update the priority of an existing job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifRequestJobPriority()
     * - .ifExistingJobPriority()
     * - .ifExistingJobOperation()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobPriority.html
     */
    toUpdateJobPriority() {
        return this.to('UpdateJobPriority');
    }
    /**
     * Grants permission to update the status for the specified job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifExistingJobPriority()
     * - .ifExistingJobOperation()
     * - .ifJobSuspendedCause()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobStatus.html
     */
    toUpdateJobStatus() {
        return this.to('UpdateJobStatus');
    }
    /**
     * Grants permission to update an existing S3 Storage Lens group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateStorageLensGroup.html
     */
    toUpdateStorageLensGroup() {
        return this.to('UpdateStorageLensGroup');
    }
    /**
     * Adds a resource of type accesspoint to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/access-points.html
     *
     * @param accessPointName - Identifier for the accessPointName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAccesspoint(accessPointName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:s3:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:accesspoint/${accessPointName}`);
    }
    /**
     * Adds a resource of type bucket to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/UsingBucket.html
     *
     * @param bucketName - Identifier for the bucketName.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onBucket(bucketName, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:s3:::${bucketName}`);
    }
    /**
     * Adds a resource of type object to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/UsingObjects.html
     *
     * @param bucketName - Identifier for the bucketName.
     * @param objectName - Identifier for the objectName.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onObject(bucketName, objectName, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:s3:::${bucketName}/${objectName}`);
    }
    /**
     * Adds a resource of type job to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/batch-ops-managing-jobs.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     */
    onJob(jobId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:s3:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:job/${jobId}`);
    }
    /**
     * Adds a resource of type storagelensconfiguration to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/storage_lens.html
     *
     * @param configId - Identifier for the configId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     */
    onStoragelensconfiguration(configId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:s3:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:storage-lens/${configId}`);
    }
    /**
     * Adds a resource of type storagelensgroup to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/storage_lens_group.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     */
    onStoragelensgroup(name, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:s3:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:storage-lens-group/${name}`);
    }
    /**
     * Adds a resource of type objectlambdaaccesspoint to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/transforming-objects.html
     *
     * @param accessPointName - Identifier for the accessPointName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onObjectlambdaaccesspoint(accessPointName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:s3-object-lambda:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:accesspoint/${accessPointName}`);
    }
    /**
     * Adds a resource of type multiregionaccesspoint to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/MultiRegionAccessPointRequests.html
     *
     * @param accessPointAlias - Identifier for the accessPointAlias.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onMultiregionaccesspoint(accessPointAlias, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:s3::${account ?? this.defaultAccount}:accesspoint/${accessPointAlias}`);
    }
    /**
     * Adds a resource of type multiregionaccesspointrequestarn to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/MultiRegionAccessPointRequests.html
     *
     * @param operation - Identifier for the operation.
     * @param token - Identifier for the token.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onMultiregionaccesspointrequestarn(operation, token, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:s3:us-west-2:${account ?? this.defaultAccount}:async-request/mrap/${operation}/${token}`);
    }
    /**
     * Adds a resource of type accessgrantsinstance to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/access-grants-instance.html
     *
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     */
    onAccessgrantsinstance(account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:s3:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:access-grants/default`);
    }
    /**
     * Adds a resource of type accessgrantslocation to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/access-grants-location.html
     *
     * @param token - Identifier for the token.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     */
    onAccessgrantslocation(token, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:s3:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:access-grants/default/location/${token}`);
    }
    /**
     * Adds a resource of type accessgrant to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/access-grants-grant.html
     *
     * @param token - Identifier for the token.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     */
    onAccessgrant(token, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:s3:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:access-grants/default/grant/${token}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAccessGrant()
     * - .toCreateAccessGrantsInstance()
     * - .toCreateAccessGrantsLocation()
     * - .toCreateJob()
     * - .toCreateStorageLensGroup()
     * - .toPutJobTagging()
     * - .toPutStorageLensConfiguration()
     * - .toPutStorageLensConfigurationTagging()
     * - .toTagResource()
     *
     * Applies to resource types:
     * - job
     * - storagelensconfiguration
     * - storagelensgroup
     * - accessgrantsinstance
     * - accessgrantslocation
     * - accessgrant
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toAssociateAccessGrantsIdentityCenter()
     * - .toCreateAccessGrant()
     * - .toCreateAccessGrantsInstance()
     * - .toCreateAccessGrantsLocation()
     * - .toDeleteAccessGrant()
     * - .toDeleteAccessGrantsInstance()
     * - .toDeleteAccessGrantsInstanceResourcePolicy()
     * - .toDeleteAccessGrantsLocation()
     * - .toDissociateAccessGrantsIdentityCenter()
     * - .toGetAccessGrant()
     * - .toGetAccessGrantsInstance()
     * - .toGetAccessGrantsInstanceForPrefix()
     * - .toGetAccessGrantsInstanceResourcePolicy()
     * - .toGetAccessGrantsLocation()
     * - .toGetDataAccess()
     * - .toListAccessGrants()
     * - .toListAccessGrantsLocations()
     * - .toListCallerAccessGrants()
     * - .toPutAccessGrantsInstanceResourcePolicy()
     * - .toUpdateAccessGrantsLocation()
     *
     * Applies to resource types:
     * - job
     * - storagelensconfiguration
     * - storagelensgroup
     * - accessgrantsinstance
     * - accessgrantslocation
     * - accessgrant
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAccessGrant()
     * - .toCreateAccessGrantsInstance()
     * - .toCreateAccessGrantsLocation()
     * - .toCreateJob()
     * - .toCreateStorageLensGroup()
     * - .toPutJobTagging()
     * - .toPutStorageLensConfiguration()
     * - .toPutStorageLensConfigurationTagging()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - job
     * - storagelensconfiguration
     * - storagelensgroup
     * - accessgrantsinstance
     * - accessgrantslocation
     * - accessgrant
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by access grants instance ARN
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/access-grants-instance.html
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toDeleteObject()
     * - .toDeleteObjectVersion()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListMultipartUploadParts()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifAccessGrantsInstanceArn(value, operator) {
        return this.if(`AccessGrantsInstanceArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the network origin (Internet or VPC)
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/creating-access-points.html#access-points-policies
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toCreateAccessPointForObjectLambda()
     * - .toCreateMultiRegionAccessPoint()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointForObjectLambda()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointPolicyForObjectLambda()
     * - .toDeleteMultiRegionAccessPoint()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toGetAccessPoint()
     * - .toGetAccessPointConfigurationForObjectLambda()
     * - .toGetAccessPointForObjectLambda()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyForObjectLambda()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetAccessPointPolicyStatusForObjectLambda()
     * - .toGetMultiRegionAccessPoint()
     * - .toGetMultiRegionAccessPointPolicy()
     * - .toGetMultiRegionAccessPointPolicyStatus()
     * - .toGetMultiRegionAccessPointRoutes()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectAttributes()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionAttributes()
     * - .toGetObjectVersionTagging()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListMultipartUploadParts()
     * - .toPutAccessPointConfigurationForObjectLambda()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointPolicyForObjectLambda()
     * - .toPutMultiRegionAccessPointPolicy()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toRestoreObject()
     * - .toSubmitMultiRegionAccessPointRoutes()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAccessPointNetworkOrigin(value, operator) {
        return this.if(`AccessPointNetworkOrigin`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the AWS Account ID that owns the access point
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/creating-access-points.html#access-points-policies
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toCreateAccessPointForObjectLambda()
     * - .toCreateMultiRegionAccessPoint()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointForObjectLambda()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointPolicyForObjectLambda()
     * - .toDeleteMultiRegionAccessPoint()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toGetAccessPoint()
     * - .toGetAccessPointConfigurationForObjectLambda()
     * - .toGetAccessPointForObjectLambda()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyForObjectLambda()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetAccessPointPolicyStatusForObjectLambda()
     * - .toGetMultiRegionAccessPoint()
     * - .toGetMultiRegionAccessPointPolicy()
     * - .toGetMultiRegionAccessPointPolicyStatus()
     * - .toGetMultiRegionAccessPointRoutes()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectAttributes()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionAttributes()
     * - .toGetObjectVersionTagging()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListMultipartUploadParts()
     * - .toPutAccessPointConfigurationForObjectLambda()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointPolicyForObjectLambda()
     * - .toPutMultiRegionAccessPointPolicy()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toRestoreObject()
     * - .toSubmitMultiRegionAccessPointRoutes()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDataAccessPointAccount(value, operator) {
        return this.if(`DataAccessPointAccount`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by an access point Amazon Resource Name (ARN)
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/creating-access-points.html#access-points-policies
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toCreateAccessPointForObjectLambda()
     * - .toCreateMultiRegionAccessPoint()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointForObjectLambda()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointPolicyForObjectLambda()
     * - .toDeleteMultiRegionAccessPoint()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toGetAccessPoint()
     * - .toGetAccessPointConfigurationForObjectLambda()
     * - .toGetAccessPointForObjectLambda()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyForObjectLambda()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetAccessPointPolicyStatusForObjectLambda()
     * - .toGetMultiRegionAccessPoint()
     * - .toGetMultiRegionAccessPointPolicy()
     * - .toGetMultiRegionAccessPointPolicyStatus()
     * - .toGetMultiRegionAccessPointRoutes()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectAttributes()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionAttributes()
     * - .toGetObjectVersionTagging()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListMultipartUploadParts()
     * - .toPutAccessPointConfigurationForObjectLambda()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointPolicyForObjectLambda()
     * - .toPutMultiRegionAccessPointPolicy()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toRestoreObject()
     * - .toSubmitMultiRegionAccessPointRoutes()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifDataAccessPointArn(value, operator) {
        return this.if(`DataAccessPointArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by operation to updating the job priority
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .toDeleteJobTagging()
     * - .toPutJobTagging()
     * - .toUpdateJobPriority()
     * - .toUpdateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExistingJobOperation(value, operator) {
        return this.if(`ExistingJobOperation`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by priority range to cancelling existing jobs
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .toDeleteJobTagging()
     * - .toPutJobTagging()
     * - .toUpdateJobPriority()
     * - .toUpdateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifExistingJobPriority(value, operator) {
        return this.if(`ExistingJobPriority`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by existing object tag key and value
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/object-tagging.html#tagging-and-policies
     *
     * Applies to actions:
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersionTagging()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectAttributes()
     * - .toGetObjectTagging()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionAttributes()
     * - .toGetObjectVersionTagging()
     * - .toPutObjectAcl()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     *
     * @param key The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExistingObjectTag(key, value, operator) {
        return this.if(`ExistingObjectTag/${key}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by restricting which optional metadata fields a user can add when configuring S3 Inventory reports
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/example-bucket-policies.html#example-bucket-policies-s3-inventory-2
     *
     * Applies to actions:
     * - .toPutInventoryConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInventoryAccessibleOptionalFields(value, operator) {
        return this.if(`InventoryAccessibleOptionalFields`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a specific job suspended cause (for example, AWAITING_CONFIRMATION) to cancelling suspended jobs
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .toUpdateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifJobSuspendedCause(value, operator) {
        return this.if(`JobSuspendedCause`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by whether or not the operation creates an object
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/conditional-writes-enforce.html
     *
     * Applies to actions:
     * - .toPutObject()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifObjectCreationOperation(value) {
        return this.if(`ObjectCreationOperation`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by operation to creating jobs
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .toCreateJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestJobOperation(value, operator) {
        return this.if(`RequestJobOperation`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by priority range to creating new jobs
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .toCreateJob()
     * - .toUpdateJobPriority()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifRequestJobPriority(value, operator) {
        return this.if(`RequestJobPriority`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the tag keys and values to be added to objects
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/object-tagging.html#tagging-and-policies
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionTagging()
     *
     * @param key The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestObjectTag(key, value, operator) {
        return this.if(`RequestObjectTag/${key}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys to be added to objects
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/object-tagging.html#tagging-and-policies
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionTagging()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestObjectTagKeys(value, operator) {
        return this.if(`RequestObjectTagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the resource owner AWS account ID
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/amazon-s3-policy-keys.html#example-object-resource-account
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toAssociateAccessGrantsIdentityCenter()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessGrant()
     * - .toCreateAccessGrantsInstance()
     * - .toCreateAccessGrantsLocation()
     * - .toCreateAccessPoint()
     * - .toCreateAccessPointForObjectLambda()
     * - .toCreateBucket()
     * - .toCreateBucketMetadataTableConfiguration()
     * - .toCreateJob()
     * - .toCreateMultiRegionAccessPoint()
     * - .toCreateStorageLensGroup()
     * - .toDeleteAccessGrant()
     * - .toDeleteAccessGrantsInstance()
     * - .toDeleteAccessGrantsInstanceResourcePolicy()
     * - .toDeleteAccessGrantsLocation()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointForObjectLambda()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointPolicyForObjectLambda()
     * - .toDeleteBucket()
     * - .toDeleteBucketMetadataTableConfiguration()
     * - .toDeleteBucketPolicy()
     * - .toDeleteBucketWebsite()
     * - .toDeleteJobTagging()
     * - .toDeleteMultiRegionAccessPoint()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toDeleteStorageLensConfiguration()
     * - .toDeleteStorageLensConfigurationTagging()
     * - .toDeleteStorageLensGroup()
     * - .toDescribeJob()
     * - .toDescribeMultiRegionAccessPointOperation()
     * - .toDissociateAccessGrantsIdentityCenter()
     * - .toGetAccelerateConfiguration()
     * - .toGetAccessGrant()
     * - .toGetAccessGrantsInstance()
     * - .toGetAccessGrantsInstanceForPrefix()
     * - .toGetAccessGrantsInstanceResourcePolicy()
     * - .toGetAccessGrantsLocation()
     * - .toGetAccessPoint()
     * - .toGetAccessPointConfigurationForObjectLambda()
     * - .toGetAccessPointForObjectLambda()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyForObjectLambda()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetAccessPointPolicyStatusForObjectLambda()
     * - .toGetAccountPublicAccessBlock()
     * - .toGetAnalyticsConfiguration()
     * - .toGetBucketAcl()
     * - .toGetBucketCORS()
     * - .toGetBucketLocation()
     * - .toGetBucketLogging()
     * - .toGetBucketMetadataTableConfiguration()
     * - .toGetBucketNotification()
     * - .toGetBucketObjectLockConfiguration()
     * - .toGetBucketOwnershipControls()
     * - .toGetBucketPolicy()
     * - .toGetBucketPolicyStatus()
     * - .toGetBucketPublicAccessBlock()
     * - .toGetBucketRequestPayment()
     * - .toGetBucketTagging()
     * - .toGetBucketVersioning()
     * - .toGetBucketWebsite()
     * - .toGetDataAccess()
     * - .toGetEncryptionConfiguration()
     * - .toGetIntelligentTieringConfiguration()
     * - .toGetInventoryConfiguration()
     * - .toGetJobTagging()
     * - .toGetLifecycleConfiguration()
     * - .toGetMetricsConfiguration()
     * - .toGetMultiRegionAccessPoint()
     * - .toGetMultiRegionAccessPointPolicy()
     * - .toGetMultiRegionAccessPointPolicyStatus()
     * - .toGetMultiRegionAccessPointRoutes()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectAttributes()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectTorrent()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionAttributes()
     * - .toGetObjectVersionForReplication()
     * - .toGetObjectVersionTagging()
     * - .toGetObjectVersionTorrent()
     * - .toGetReplicationConfiguration()
     * - .toGetStorageLensConfiguration()
     * - .toGetStorageLensConfigurationTagging()
     * - .toGetStorageLensDashboard()
     * - .toGetStorageLensGroup()
     * - .toInitiateReplication()
     * - .toListAccessGrants()
     * - .toListAccessGrantsInstances()
     * - .toListAccessGrantsLocations()
     * - .toListAccessPoints()
     * - .toListAccessPointsForObjectLambda()
     * - .toListAllMyBuckets()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListCallerAccessGrants()
     * - .toListJobs()
     * - .toListMultiRegionAccessPoints()
     * - .toListMultipartUploadParts()
     * - .toListStorageLensConfigurations()
     * - .toListStorageLensGroups()
     * - .toListTagsForResource()
     * - .toObjectOwnerOverrideToBucketOwner()
     * - .toPauseReplication()
     * - .toPutAccelerateConfiguration()
     * - .toPutAccessGrantsInstanceResourcePolicy()
     * - .toPutAccessPointConfigurationForObjectLambda()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointPolicyForObjectLambda()
     * - .toPutAccountPublicAccessBlock()
     * - .toPutAnalyticsConfiguration()
     * - .toPutBucketAcl()
     * - .toPutBucketCORS()
     * - .toPutBucketLogging()
     * - .toPutBucketNotification()
     * - .toPutBucketObjectLockConfiguration()
     * - .toPutBucketOwnershipControls()
     * - .toPutBucketPolicy()
     * - .toPutBucketPublicAccessBlock()
     * - .toPutBucketRequestPayment()
     * - .toPutBucketTagging()
     * - .toPutBucketVersioning()
     * - .toPutBucketWebsite()
     * - .toPutEncryptionConfiguration()
     * - .toPutIntelligentTieringConfiguration()
     * - .toPutInventoryConfiguration()
     * - .toPutJobTagging()
     * - .toPutLifecycleConfiguration()
     * - .toPutMetricsConfiguration()
     * - .toPutMultiRegionAccessPointPolicy()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toPutReplicationConfiguration()
     * - .toPutStorageLensConfiguration()
     * - .toPutStorageLensConfigurationTagging()
     * - .toReplicateDelete()
     * - .toReplicateObject()
     * - .toReplicateTags()
     * - .toRestoreObject()
     * - .toSubmitMultiRegionAccessPointRoutes()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateAccessGrantsLocation()
     * - .toUpdateJobPriority()
     * - .toUpdateJobStatus()
     * - .toUpdateStorageLensGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceAccount(value, operator) {
        return this.if(`ResourceAccount`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the TLS version used by the client
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/amazon-s3-policy-keys.html#example-object-tls-version
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toAssociateAccessGrantsIdentityCenter()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessGrant()
     * - .toCreateAccessGrantsInstance()
     * - .toCreateAccessGrantsLocation()
     * - .toCreateAccessPoint()
     * - .toCreateAccessPointForObjectLambda()
     * - .toCreateBucket()
     * - .toCreateBucketMetadataTableConfiguration()
     * - .toCreateJob()
     * - .toCreateMultiRegionAccessPoint()
     * - .toCreateStorageLensGroup()
     * - .toDeleteAccessGrant()
     * - .toDeleteAccessGrantsInstance()
     * - .toDeleteAccessGrantsInstanceResourcePolicy()
     * - .toDeleteAccessGrantsLocation()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointForObjectLambda()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointPolicyForObjectLambda()
     * - .toDeleteBucket()
     * - .toDeleteBucketMetadataTableConfiguration()
     * - .toDeleteBucketPolicy()
     * - .toDeleteBucketWebsite()
     * - .toDeleteJobTagging()
     * - .toDeleteMultiRegionAccessPoint()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toDeleteStorageLensConfiguration()
     * - .toDeleteStorageLensConfigurationTagging()
     * - .toDeleteStorageLensGroup()
     * - .toDescribeJob()
     * - .toDescribeMultiRegionAccessPointOperation()
     * - .toDissociateAccessGrantsIdentityCenter()
     * - .toGetAccelerateConfiguration()
     * - .toGetAccessGrant()
     * - .toGetAccessGrantsInstance()
     * - .toGetAccessGrantsInstanceForPrefix()
     * - .toGetAccessGrantsInstanceResourcePolicy()
     * - .toGetAccessGrantsLocation()
     * - .toGetAccessPoint()
     * - .toGetAccessPointConfigurationForObjectLambda()
     * - .toGetAccessPointForObjectLambda()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyForObjectLambda()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetAccessPointPolicyStatusForObjectLambda()
     * - .toGetAccountPublicAccessBlock()
     * - .toGetAnalyticsConfiguration()
     * - .toGetBucketAcl()
     * - .toGetBucketCORS()
     * - .toGetBucketLocation()
     * - .toGetBucketLogging()
     * - .toGetBucketMetadataTableConfiguration()
     * - .toGetBucketNotification()
     * - .toGetBucketObjectLockConfiguration()
     * - .toGetBucketOwnershipControls()
     * - .toGetBucketPolicy()
     * - .toGetBucketPolicyStatus()
     * - .toGetBucketPublicAccessBlock()
     * - .toGetBucketRequestPayment()
     * - .toGetBucketTagging()
     * - .toGetBucketVersioning()
     * - .toGetBucketWebsite()
     * - .toGetDataAccess()
     * - .toGetEncryptionConfiguration()
     * - .toGetIntelligentTieringConfiguration()
     * - .toGetInventoryConfiguration()
     * - .toGetJobTagging()
     * - .toGetLifecycleConfiguration()
     * - .toGetMetricsConfiguration()
     * - .toGetMultiRegionAccessPoint()
     * - .toGetMultiRegionAccessPointPolicy()
     * - .toGetMultiRegionAccessPointPolicyStatus()
     * - .toGetMultiRegionAccessPointRoutes()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectAttributes()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectTorrent()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionAttributes()
     * - .toGetObjectVersionForReplication()
     * - .toGetObjectVersionTagging()
     * - .toGetObjectVersionTorrent()
     * - .toGetReplicationConfiguration()
     * - .toGetStorageLensConfiguration()
     * - .toGetStorageLensConfigurationTagging()
     * - .toGetStorageLensDashboard()
     * - .toGetStorageLensGroup()
     * - .toListAccessGrants()
     * - .toListAccessGrantsInstances()
     * - .toListAccessGrantsLocations()
     * - .toListAccessPoints()
     * - .toListAccessPointsForObjectLambda()
     * - .toListAllMyBuckets()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListCallerAccessGrants()
     * - .toListJobs()
     * - .toListMultiRegionAccessPoints()
     * - .toListMultipartUploadParts()
     * - .toListStorageLensConfigurations()
     * - .toListStorageLensGroups()
     * - .toListTagsForResource()
     * - .toObjectOwnerOverrideToBucketOwner()
     * - .toPauseReplication()
     * - .toPutAccelerateConfiguration()
     * - .toPutAccessGrantsInstanceResourcePolicy()
     * - .toPutAccessPointConfigurationForObjectLambda()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointPolicyForObjectLambda()
     * - .toPutAccountPublicAccessBlock()
     * - .toPutAnalyticsConfiguration()
     * - .toPutBucketAcl()
     * - .toPutBucketCORS()
     * - .toPutBucketLogging()
     * - .toPutBucketNotification()
     * - .toPutBucketObjectLockConfiguration()
     * - .toPutBucketOwnershipControls()
     * - .toPutBucketPolicy()
     * - .toPutBucketPublicAccessBlock()
     * - .toPutBucketRequestPayment()
     * - .toPutBucketTagging()
     * - .toPutBucketVersioning()
     * - .toPutBucketWebsite()
     * - .toPutEncryptionConfiguration()
     * - .toPutIntelligentTieringConfiguration()
     * - .toPutInventoryConfiguration()
     * - .toPutJobTagging()
     * - .toPutLifecycleConfiguration()
     * - .toPutMetricsConfiguration()
     * - .toPutMultiRegionAccessPointPolicy()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toPutReplicationConfiguration()
     * - .toPutStorageLensConfiguration()
     * - .toPutStorageLensConfigurationTagging()
     * - .toReplicateDelete()
     * - .toReplicateObject()
     * - .toReplicateTags()
     * - .toRestoreObject()
     * - .toSubmitMultiRegionAccessPointRoutes()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateAccessGrantsLocation()
     * - .toUpdateJobPriority()
     * - .toUpdateJobStatus()
     * - .toUpdateStorageLensGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifTlsVersion(value, operator) {
        return this.if(`TlsVersion`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by authentication method
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toAssociateAccessGrantsIdentityCenter()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessGrant()
     * - .toCreateAccessGrantsInstance()
     * - .toCreateAccessGrantsLocation()
     * - .toCreateAccessPoint()
     * - .toCreateAccessPointForObjectLambda()
     * - .toCreateBucket()
     * - .toCreateBucketMetadataTableConfiguration()
     * - .toCreateJob()
     * - .toCreateMultiRegionAccessPoint()
     * - .toCreateStorageLensGroup()
     * - .toDeleteAccessGrant()
     * - .toDeleteAccessGrantsInstance()
     * - .toDeleteAccessGrantsInstanceResourcePolicy()
     * - .toDeleteAccessGrantsLocation()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointForObjectLambda()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointPolicyForObjectLambda()
     * - .toDeleteBucket()
     * - .toDeleteBucketMetadataTableConfiguration()
     * - .toDeleteBucketPolicy()
     * - .toDeleteBucketWebsite()
     * - .toDeleteJobTagging()
     * - .toDeleteMultiRegionAccessPoint()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toDeleteStorageLensConfiguration()
     * - .toDeleteStorageLensConfigurationTagging()
     * - .toDeleteStorageLensGroup()
     * - .toDescribeJob()
     * - .toDescribeMultiRegionAccessPointOperation()
     * - .toDissociateAccessGrantsIdentityCenter()
     * - .toGetAccelerateConfiguration()
     * - .toGetAccessGrant()
     * - .toGetAccessGrantsInstance()
     * - .toGetAccessGrantsInstanceForPrefix()
     * - .toGetAccessGrantsInstanceResourcePolicy()
     * - .toGetAccessGrantsLocation()
     * - .toGetAccessPoint()
     * - .toGetAccessPointConfigurationForObjectLambda()
     * - .toGetAccessPointForObjectLambda()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyForObjectLambda()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetAccessPointPolicyStatusForObjectLambda()
     * - .toGetAccountPublicAccessBlock()
     * - .toGetAnalyticsConfiguration()
     * - .toGetBucketAcl()
     * - .toGetBucketCORS()
     * - .toGetBucketLocation()
     * - .toGetBucketLogging()
     * - .toGetBucketMetadataTableConfiguration()
     * - .toGetBucketNotification()
     * - .toGetBucketObjectLockConfiguration()
     * - .toGetBucketOwnershipControls()
     * - .toGetBucketPolicy()
     * - .toGetBucketPolicyStatus()
     * - .toGetBucketPublicAccessBlock()
     * - .toGetBucketRequestPayment()
     * - .toGetBucketTagging()
     * - .toGetBucketVersioning()
     * - .toGetBucketWebsite()
     * - .toGetDataAccess()
     * - .toGetEncryptionConfiguration()
     * - .toGetIntelligentTieringConfiguration()
     * - .toGetInventoryConfiguration()
     * - .toGetJobTagging()
     * - .toGetLifecycleConfiguration()
     * - .toGetMetricsConfiguration()
     * - .toGetMultiRegionAccessPoint()
     * - .toGetMultiRegionAccessPointPolicy()
     * - .toGetMultiRegionAccessPointPolicyStatus()
     * - .toGetMultiRegionAccessPointRoutes()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectAttributes()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectTorrent()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionAttributes()
     * - .toGetObjectVersionForReplication()
     * - .toGetObjectVersionTagging()
     * - .toGetObjectVersionTorrent()
     * - .toGetReplicationConfiguration()
     * - .toGetStorageLensConfiguration()
     * - .toGetStorageLensConfigurationTagging()
     * - .toGetStorageLensDashboard()
     * - .toGetStorageLensGroup()
     * - .toListAccessGrants()
     * - .toListAccessGrantsInstances()
     * - .toListAccessGrantsLocations()
     * - .toListAccessPoints()
     * - .toListAccessPointsForObjectLambda()
     * - .toListAllMyBuckets()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListCallerAccessGrants()
     * - .toListJobs()
     * - .toListMultiRegionAccessPoints()
     * - .toListMultipartUploadParts()
     * - .toListStorageLensConfigurations()
     * - .toListStorageLensGroups()
     * - .toListTagsForResource()
     * - .toObjectOwnerOverrideToBucketOwner()
     * - .toPauseReplication()
     * - .toPutAccelerateConfiguration()
     * - .toPutAccessGrantsInstanceResourcePolicy()
     * - .toPutAccessPointConfigurationForObjectLambda()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointPolicyForObjectLambda()
     * - .toPutAccountPublicAccessBlock()
     * - .toPutAnalyticsConfiguration()
     * - .toPutBucketAcl()
     * - .toPutBucketCORS()
     * - .toPutBucketLogging()
     * - .toPutBucketNotification()
     * - .toPutBucketObjectLockConfiguration()
     * - .toPutBucketOwnershipControls()
     * - .toPutBucketPolicy()
     * - .toPutBucketPublicAccessBlock()
     * - .toPutBucketRequestPayment()
     * - .toPutBucketTagging()
     * - .toPutBucketVersioning()
     * - .toPutBucketWebsite()
     * - .toPutEncryptionConfiguration()
     * - .toPutIntelligentTieringConfiguration()
     * - .toPutInventoryConfiguration()
     * - .toPutJobTagging()
     * - .toPutLifecycleConfiguration()
     * - .toPutMetricsConfiguration()
     * - .toPutMultiRegionAccessPointPolicy()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toPutReplicationConfiguration()
     * - .toPutStorageLensConfiguration()
     * - .toPutStorageLensConfigurationTagging()
     * - .toReplicateDelete()
     * - .toReplicateObject()
     * - .toReplicateTags()
     * - .toRestoreObject()
     * - .toSubmitMultiRegionAccessPointRoutes()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateAccessGrantsLocation()
     * - .toUpdateJobPriority()
     * - .toUpdateJobStatus()
     * - .toUpdateStorageLensGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAuthType(value, operator) {
        return this.if(`authType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by delimiter parameter
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/walkthrough1.html
     *
     * Applies to actions:
     * - .toListBucket()
     * - .toListBucketVersions()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDelimiter(value, operator) {
        return this.if(`delimiter`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a specific replication destination region for targeted buckets of the AWS FIS action aws:s3:bucket-pause-replication
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/replication.html
     *
     * Applies to actions:
     * - .toPauseReplication()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDestinationRegion(value, operator) {
        return this.if(`destinationRegion`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the request's 'If-Match' conditional header
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/conditional-writes-enforce.html
     *
     * Applies to actions:
     * - .toGetObject()
     * - .toPutObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifIfMatch(value, operator) {
        return this.if(`if-match`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the request's 'If-None-Match' conditional header
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/conditional-writes-enforce.html
     *
     * Applies to actions:
     * - .toGetObject()
     * - .toPutObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifIfNoneMatch(value, operator) {
        return this.if(`if-none-match`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by request made via AWS FIS action aws:s3:bucket-pause-replication
     *
     * https://docs.aws.amazon.com/fis/latest/userguide/security_iam_id-based-policy-examples.html#security-iam-policy-examples-s3
     *
     * Applies to actions:
     * - .toPutReplicationConfiguration()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifIsReplicationPauseRequest(value) {
        return this.if(`isReplicationPauseRequest`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by a specific Region
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/amazon-s3-policy-keys.html#condition-key-bucket-ops-1
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLocationconstraint(value, operator) {
        return this.if(`locationconstraint`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by maximum number of keys returned in a ListBucket request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/amazon-s3-policy-keys.html#example-numeric-condition-operators
     *
     * Applies to actions:
     * - .toListBucket()
     * - .toListBucketVersions()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMaxKeys(value, operator) {
        return this.if(`max-keys`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by object legal hold status
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/object-lock-overview.html#object-lock-legal-holds
     *
     * Applies to actions:
     * - .toPutObject()
     * - .toPutObjectLegalHold()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifObjectLockLegalHold(value, operator) {
        return this.if(`object-lock-legal-hold`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by object retention mode (COMPLIANCE or GOVERNANCE)
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/object-lock-overview.html#object-lock-retention-modes
     *
     * Applies to actions:
     * - .toPutObject()
     * - .toPutObjectRetention()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifObjectLockMode(value, operator) {
        return this.if(`object-lock-mode`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by remaining object retention days
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/object-lock-managing.html#object-lock-managing-retention-limits
     *
     * Applies to actions:
     * - .toPutObject()
     * - .toPutObjectRetention()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifObjectLockRemainingRetentionDays(value, operator) {
        return this.if(`object-lock-remaining-retention-days`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by object retain-until date
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/object-lock-overview.html#object-lock-retention-periods
     *
     * Applies to actions:
     * - .toPutObject()
     * - .toPutObjectRetention()
     *
     * @param value The value(s) to check
     * @param operator Works with [date operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date). **Default:** `DateEquals`
     */
    ifObjectLockRetainUntilDate(value, operator) {
        if (typeof value.getMonth === "function") {
            value = value.toISOString();
        }
        else if (Array.isArray(value)) {
            value = value.map((item) => {
                if (typeof item.getMonth === "function") {
                    item = item.toISOString();
                }
                return item;
            });
        }
        return this.if(`object-lock-retain-until-date`, value, operator ?? 'DateEquals');
    }
    /**
     * Filters access by key name prefix
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/amazon-s3-policy-keys.html#condition-key-bucket-ops-2
     *
     * Applies to actions:
     * - .toListBucket()
     * - .toListBucketVersions()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPrefix(value, operator) {
        return this.if(`prefix`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the age in milliseconds of the request signature
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toAssociateAccessGrantsIdentityCenter()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessGrant()
     * - .toCreateAccessGrantsInstance()
     * - .toCreateAccessGrantsLocation()
     * - .toCreateAccessPoint()
     * - .toCreateAccessPointForObjectLambda()
     * - .toCreateBucket()
     * - .toCreateBucketMetadataTableConfiguration()
     * - .toCreateJob()
     * - .toCreateMultiRegionAccessPoint()
     * - .toCreateStorageLensGroup()
     * - .toDeleteAccessGrant()
     * - .toDeleteAccessGrantsInstance()
     * - .toDeleteAccessGrantsInstanceResourcePolicy()
     * - .toDeleteAccessGrantsLocation()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointForObjectLambda()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointPolicyForObjectLambda()
     * - .toDeleteBucket()
     * - .toDeleteBucketMetadataTableConfiguration()
     * - .toDeleteBucketPolicy()
     * - .toDeleteBucketWebsite()
     * - .toDeleteJobTagging()
     * - .toDeleteMultiRegionAccessPoint()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toDeleteStorageLensConfiguration()
     * - .toDeleteStorageLensConfigurationTagging()
     * - .toDeleteStorageLensGroup()
     * - .toDescribeJob()
     * - .toDescribeMultiRegionAccessPointOperation()
     * - .toDissociateAccessGrantsIdentityCenter()
     * - .toGetAccelerateConfiguration()
     * - .toGetAccessGrant()
     * - .toGetAccessGrantsInstance()
     * - .toGetAccessGrantsInstanceForPrefix()
     * - .toGetAccessGrantsInstanceResourcePolicy()
     * - .toGetAccessGrantsLocation()
     * - .toGetAccessPoint()
     * - .toGetAccessPointConfigurationForObjectLambda()
     * - .toGetAccessPointForObjectLambda()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyForObjectLambda()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetAccessPointPolicyStatusForObjectLambda()
     * - .toGetAccountPublicAccessBlock()
     * - .toGetAnalyticsConfiguration()
     * - .toGetBucketAcl()
     * - .toGetBucketCORS()
     * - .toGetBucketLocation()
     * - .toGetBucketLogging()
     * - .toGetBucketMetadataTableConfiguration()
     * - .toGetBucketNotification()
     * - .toGetBucketObjectLockConfiguration()
     * - .toGetBucketOwnershipControls()
     * - .toGetBucketPolicy()
     * - .toGetBucketPolicyStatus()
     * - .toGetBucketPublicAccessBlock()
     * - .toGetBucketRequestPayment()
     * - .toGetBucketTagging()
     * - .toGetBucketVersioning()
     * - .toGetBucketWebsite()
     * - .toGetDataAccess()
     * - .toGetEncryptionConfiguration()
     * - .toGetIntelligentTieringConfiguration()
     * - .toGetInventoryConfiguration()
     * - .toGetJobTagging()
     * - .toGetLifecycleConfiguration()
     * - .toGetMetricsConfiguration()
     * - .toGetMultiRegionAccessPoint()
     * - .toGetMultiRegionAccessPointPolicy()
     * - .toGetMultiRegionAccessPointPolicyStatus()
     * - .toGetMultiRegionAccessPointRoutes()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectAttributes()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectTorrent()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionAttributes()
     * - .toGetObjectVersionForReplication()
     * - .toGetObjectVersionTagging()
     * - .toGetObjectVersionTorrent()
     * - .toGetReplicationConfiguration()
     * - .toGetStorageLensConfiguration()
     * - .toGetStorageLensConfigurationTagging()
     * - .toGetStorageLensDashboard()
     * - .toGetStorageLensGroup()
     * - .toListAccessGrants()
     * - .toListAccessGrantsInstances()
     * - .toListAccessGrantsLocations()
     * - .toListAccessPoints()
     * - .toListAccessPointsForObjectLambda()
     * - .toListAllMyBuckets()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListCallerAccessGrants()
     * - .toListJobs()
     * - .toListMultiRegionAccessPoints()
     * - .toListMultipartUploadParts()
     * - .toListStorageLensConfigurations()
     * - .toListStorageLensGroups()
     * - .toListTagsForResource()
     * - .toObjectOwnerOverrideToBucketOwner()
     * - .toPauseReplication()
     * - .toPutAccelerateConfiguration()
     * - .toPutAccessGrantsInstanceResourcePolicy()
     * - .toPutAccessPointConfigurationForObjectLambda()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointPolicyForObjectLambda()
     * - .toPutAccountPublicAccessBlock()
     * - .toPutAnalyticsConfiguration()
     * - .toPutBucketAcl()
     * - .toPutBucketCORS()
     * - .toPutBucketLogging()
     * - .toPutBucketNotification()
     * - .toPutBucketObjectLockConfiguration()
     * - .toPutBucketOwnershipControls()
     * - .toPutBucketPolicy()
     * - .toPutBucketPublicAccessBlock()
     * - .toPutBucketRequestPayment()
     * - .toPutBucketTagging()
     * - .toPutBucketVersioning()
     * - .toPutBucketWebsite()
     * - .toPutEncryptionConfiguration()
     * - .toPutIntelligentTieringConfiguration()
     * - .toPutInventoryConfiguration()
     * - .toPutJobTagging()
     * - .toPutLifecycleConfiguration()
     * - .toPutMetricsConfiguration()
     * - .toPutMultiRegionAccessPointPolicy()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toPutReplicationConfiguration()
     * - .toPutStorageLensConfiguration()
     * - .toPutStorageLensConfigurationTagging()
     * - .toReplicateDelete()
     * - .toReplicateObject()
     * - .toReplicateTags()
     * - .toRestoreObject()
     * - .toSubmitMultiRegionAccessPointRoutes()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateAccessGrantsLocation()
     * - .toUpdateJobPriority()
     * - .toUpdateJobStatus()
     * - .toUpdateStorageLensGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifSignatureAge(value, operator) {
        return this.if(`signatureAge`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the version of AWS Signature used on the request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toAssociateAccessGrantsIdentityCenter()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessGrant()
     * - .toCreateAccessGrantsInstance()
     * - .toCreateAccessGrantsLocation()
     * - .toCreateAccessPoint()
     * - .toCreateAccessPointForObjectLambda()
     * - .toCreateBucket()
     * - .toCreateBucketMetadataTableConfiguration()
     * - .toCreateJob()
     * - .toCreateMultiRegionAccessPoint()
     * - .toCreateStorageLensGroup()
     * - .toDeleteAccessGrant()
     * - .toDeleteAccessGrantsInstance()
     * - .toDeleteAccessGrantsInstanceResourcePolicy()
     * - .toDeleteAccessGrantsLocation()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointForObjectLambda()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointPolicyForObjectLambda()
     * - .toDeleteBucket()
     * - .toDeleteBucketMetadataTableConfiguration()
     * - .toDeleteBucketPolicy()
     * - .toDeleteBucketWebsite()
     * - .toDeleteJobTagging()
     * - .toDeleteMultiRegionAccessPoint()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toDeleteStorageLensConfiguration()
     * - .toDeleteStorageLensConfigurationTagging()
     * - .toDeleteStorageLensGroup()
     * - .toDescribeJob()
     * - .toDescribeMultiRegionAccessPointOperation()
     * - .toDissociateAccessGrantsIdentityCenter()
     * - .toGetAccelerateConfiguration()
     * - .toGetAccessGrant()
     * - .toGetAccessGrantsInstance()
     * - .toGetAccessGrantsInstanceForPrefix()
     * - .toGetAccessGrantsInstanceResourcePolicy()
     * - .toGetAccessGrantsLocation()
     * - .toGetAccessPoint()
     * - .toGetAccessPointConfigurationForObjectLambda()
     * - .toGetAccessPointForObjectLambda()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyForObjectLambda()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetAccessPointPolicyStatusForObjectLambda()
     * - .toGetAccountPublicAccessBlock()
     * - .toGetAnalyticsConfiguration()
     * - .toGetBucketAcl()
     * - .toGetBucketCORS()
     * - .toGetBucketLocation()
     * - .toGetBucketLogging()
     * - .toGetBucketMetadataTableConfiguration()
     * - .toGetBucketNotification()
     * - .toGetBucketObjectLockConfiguration()
     * - .toGetBucketOwnershipControls()
     * - .toGetBucketPolicy()
     * - .toGetBucketPolicyStatus()
     * - .toGetBucketPublicAccessBlock()
     * - .toGetBucketRequestPayment()
     * - .toGetBucketTagging()
     * - .toGetBucketVersioning()
     * - .toGetBucketWebsite()
     * - .toGetDataAccess()
     * - .toGetEncryptionConfiguration()
     * - .toGetIntelligentTieringConfiguration()
     * - .toGetInventoryConfiguration()
     * - .toGetJobTagging()
     * - .toGetLifecycleConfiguration()
     * - .toGetMetricsConfiguration()
     * - .toGetMultiRegionAccessPoint()
     * - .toGetMultiRegionAccessPointPolicy()
     * - .toGetMultiRegionAccessPointPolicyStatus()
     * - .toGetMultiRegionAccessPointRoutes()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectAttributes()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectTorrent()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionAttributes()
     * - .toGetObjectVersionForReplication()
     * - .toGetObjectVersionTagging()
     * - .toGetObjectVersionTorrent()
     * - .toGetReplicationConfiguration()
     * - .toGetStorageLensConfiguration()
     * - .toGetStorageLensConfigurationTagging()
     * - .toGetStorageLensDashboard()
     * - .toGetStorageLensGroup()
     * - .toListAccessGrants()
     * - .toListAccessGrantsInstances()
     * - .toListAccessGrantsLocations()
     * - .toListAccessPoints()
     * - .toListAccessPointsForObjectLambda()
     * - .toListAllMyBuckets()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListCallerAccessGrants()
     * - .toListJobs()
     * - .toListMultiRegionAccessPoints()
     * - .toListMultipartUploadParts()
     * - .toListStorageLensConfigurations()
     * - .toListStorageLensGroups()
     * - .toListTagsForResource()
     * - .toObjectOwnerOverrideToBucketOwner()
     * - .toPauseReplication()
     * - .toPutAccelerateConfiguration()
     * - .toPutAccessGrantsInstanceResourcePolicy()
     * - .toPutAccessPointConfigurationForObjectLambda()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointPolicyForObjectLambda()
     * - .toPutAccountPublicAccessBlock()
     * - .toPutAnalyticsConfiguration()
     * - .toPutBucketAcl()
     * - .toPutBucketCORS()
     * - .toPutBucketLogging()
     * - .toPutBucketNotification()
     * - .toPutBucketObjectLockConfiguration()
     * - .toPutBucketOwnershipControls()
     * - .toPutBucketPolicy()
     * - .toPutBucketPublicAccessBlock()
     * - .toPutBucketRequestPayment()
     * - .toPutBucketTagging()
     * - .toPutBucketVersioning()
     * - .toPutBucketWebsite()
     * - .toPutEncryptionConfiguration()
     * - .toPutIntelligentTieringConfiguration()
     * - .toPutInventoryConfiguration()
     * - .toPutJobTagging()
     * - .toPutLifecycleConfiguration()
     * - .toPutMetricsConfiguration()
     * - .toPutMultiRegionAccessPointPolicy()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toPutReplicationConfiguration()
     * - .toPutStorageLensConfiguration()
     * - .toPutStorageLensConfigurationTagging()
     * - .toReplicateDelete()
     * - .toReplicateObject()
     * - .toReplicateTags()
     * - .toRestoreObject()
     * - .toSubmitMultiRegionAccessPointRoutes()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateAccessGrantsLocation()
     * - .toUpdateJobPriority()
     * - .toUpdateJobStatus()
     * - .toUpdateStorageLensGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSignatureversion(value, operator) {
        return this.if(`signatureversion`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a specific object version
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/amazon-s3-policy-keys.html#getobjectversion-limit-access-to-specific-version-3
     *
     * Applies to actions:
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionAttributes()
     * - .toGetObjectVersionTagging()
     * - .toGetObjectVersionTorrent()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVersionid(value, operator) {
        return this.if(`versionid`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by canned ACL in the request's x-amz-acl header
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toPutBucketAcl()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzAcl(value, operator) {
        return this.if(`x-amz-acl`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by unsigned content in your bucket
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toAssociateAccessGrantsIdentityCenter()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessGrant()
     * - .toCreateAccessGrantsInstance()
     * - .toCreateAccessGrantsLocation()
     * - .toCreateAccessPoint()
     * - .toCreateAccessPointForObjectLambda()
     * - .toCreateBucket()
     * - .toCreateBucketMetadataTableConfiguration()
     * - .toCreateJob()
     * - .toCreateStorageLensGroup()
     * - .toDeleteAccessGrant()
     * - .toDeleteAccessGrantsInstance()
     * - .toDeleteAccessGrantsInstanceResourcePolicy()
     * - .toDeleteAccessGrantsLocation()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointForObjectLambda()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointPolicyForObjectLambda()
     * - .toDeleteBucket()
     * - .toDeleteBucketMetadataTableConfiguration()
     * - .toDeleteBucketPolicy()
     * - .toDeleteBucketWebsite()
     * - .toDeleteJobTagging()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toDeleteStorageLensConfiguration()
     * - .toDeleteStorageLensConfigurationTagging()
     * - .toDeleteStorageLensGroup()
     * - .toDescribeJob()
     * - .toDissociateAccessGrantsIdentityCenter()
     * - .toGetAccelerateConfiguration()
     * - .toGetAccessGrant()
     * - .toGetAccessGrantsInstance()
     * - .toGetAccessGrantsInstanceForPrefix()
     * - .toGetAccessGrantsInstanceResourcePolicy()
     * - .toGetAccessGrantsLocation()
     * - .toGetAccessPoint()
     * - .toGetAccessPointConfigurationForObjectLambda()
     * - .toGetAccessPointForObjectLambda()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyForObjectLambda()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetAccessPointPolicyStatusForObjectLambda()
     * - .toGetAccountPublicAccessBlock()
     * - .toGetAnalyticsConfiguration()
     * - .toGetBucketAcl()
     * - .toGetBucketCORS()
     * - .toGetBucketLocation()
     * - .toGetBucketLogging()
     * - .toGetBucketMetadataTableConfiguration()
     * - .toGetBucketNotification()
     * - .toGetBucketOwnershipControls()
     * - .toGetBucketPolicy()
     * - .toGetBucketPolicyStatus()
     * - .toGetBucketPublicAccessBlock()
     * - .toGetBucketRequestPayment()
     * - .toGetBucketTagging()
     * - .toGetBucketVersioning()
     * - .toGetBucketWebsite()
     * - .toGetDataAccess()
     * - .toGetEncryptionConfiguration()
     * - .toGetIntelligentTieringConfiguration()
     * - .toGetInventoryConfiguration()
     * - .toGetJobTagging()
     * - .toGetLifecycleConfiguration()
     * - .toGetMetricsConfiguration()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectAttributes()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectTorrent()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionAttributes()
     * - .toGetObjectVersionForReplication()
     * - .toGetObjectVersionTagging()
     * - .toGetObjectVersionTorrent()
     * - .toGetReplicationConfiguration()
     * - .toGetStorageLensConfiguration()
     * - .toGetStorageLensConfigurationTagging()
     * - .toGetStorageLensDashboard()
     * - .toGetStorageLensGroup()
     * - .toListAccessGrants()
     * - .toListAccessGrantsInstances()
     * - .toListAccessGrantsLocations()
     * - .toListAccessPoints()
     * - .toListAccessPointsForObjectLambda()
     * - .toListAllMyBuckets()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListCallerAccessGrants()
     * - .toListJobs()
     * - .toListMultipartUploadParts()
     * - .toListStorageLensConfigurations()
     * - .toListStorageLensGroups()
     * - .toListTagsForResource()
     * - .toObjectOwnerOverrideToBucketOwner()
     * - .toPauseReplication()
     * - .toPutAccelerateConfiguration()
     * - .toPutAccessGrantsInstanceResourcePolicy()
     * - .toPutAccessPointConfigurationForObjectLambda()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointPolicyForObjectLambda()
     * - .toPutAccountPublicAccessBlock()
     * - .toPutAnalyticsConfiguration()
     * - .toPutBucketAcl()
     * - .toPutBucketCORS()
     * - .toPutBucketLogging()
     * - .toPutBucketNotification()
     * - .toPutBucketOwnershipControls()
     * - .toPutBucketPolicy()
     * - .toPutBucketPublicAccessBlock()
     * - .toPutBucketRequestPayment()
     * - .toPutBucketTagging()
     * - .toPutBucketVersioning()
     * - .toPutBucketWebsite()
     * - .toPutEncryptionConfiguration()
     * - .toPutIntelligentTieringConfiguration()
     * - .toPutInventoryConfiguration()
     * - .toPutJobTagging()
     * - .toPutLifecycleConfiguration()
     * - .toPutMetricsConfiguration()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toPutReplicationConfiguration()
     * - .toPutStorageLensConfiguration()
     * - .toPutStorageLensConfigurationTagging()
     * - .toReplicateDelete()
     * - .toReplicateObject()
     * - .toReplicateTags()
     * - .toRestoreObject()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateAccessGrantsLocation()
     * - .toUpdateJobPriority()
     * - .toUpdateJobStatus()
     * - .toUpdateStorageLensGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzContentSha256(value, operator) {
        return this.if(`x-amz-content-sha256`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by copy source bucket, prefix, or object in the copy object requests
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/amazon-s3-policy-keys.html#putobject-limit-copy-source-3
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzCopySource(value, operator) {
        return this.if(`x-amz-copy-source`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by x-amz-grant-full-control (full control) header
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toCreateBucket()
     * - .toPutBucketAcl()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantFullControl(value, operator) {
        return this.if(`x-amz-grant-full-control`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by x-amz-grant-read (read access) header
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toCreateBucket()
     * - .toPutBucketAcl()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantRead(value, operator) {
        return this.if(`x-amz-grant-read`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the x-amz-grant-read-acp (read permissions for the ACL) header
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toCreateBucket()
     * - .toPutBucketAcl()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantReadAcp(value, operator) {
        return this.if(`x-amz-grant-read-acp`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the x-amz-grant-write (write access) header
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toCreateBucket()
     * - .toPutBucketAcl()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantWrite(value, operator) {
        return this.if(`x-amz-grant-write`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the x-amz-grant-write-acp (write permissions for the ACL) header
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toCreateBucket()
     * - .toPutBucketAcl()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantWriteAcp(value, operator) {
        return this.if(`x-amz-grant-write-acp`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by object metadata behavior (COPY or REPLACE) when objects are copied
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_CopyObject.html
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzMetadataDirective(value, operator) {
        return this.if(`x-amz-metadata-directive`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by Object Ownership
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/ensure-object-ownership.html#object-ownership-requiring-bucket-owner-enforced
     *
     * Applies to actions:
     * - .toCreateBucket()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzObjectOwnership(value, operator) {
        return this.if(`x-amz-object-ownership`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by server-side encryption
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/UsingServerSideEncryption.html
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toReplicateObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzServerSideEncryption(value, operator) {
        return this.if(`x-amz-server-side-encryption`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by AWS KMS customer managed CMK for server-side encryption
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/UsingKMSEncryption.html#require-sse-kms
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toReplicateObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifXAmzServerSideEncryptionAwsKmsKeyId(value, operator) {
        return this.if(`x-amz-server-side-encryption-aws-kms-key-id`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by customer specified algorithm for server-side encryption
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/ServerSideEncryptionCustomerKeys.html
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toReplicateObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzServerSideEncryptionCustomerAlgorithm(value, operator) {
        return this.if(`x-amz-server-side-encryption-customer-algorithm`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by storage class
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/storage-class-intro.html#sc-howtoset
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzStorageClass(value, operator) {
        return this.if(`x-amz-storage-class`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a specific website redirect location for buckets that are configured as static websites
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/how-to-page-redirect.html#page-redirect-using-rest-api
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzWebsiteRedirectLocation(value, operator) {
        return this.if(`x-amz-website-redirect-location`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [s3](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazons3.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 's3';
        this.accessLevelList = {
            Write: [
                'AbortMultipartUpload',
                'CreateAccessPoint',
                'CreateAccessPointForObjectLambda',
                'CreateBucket',
                'CreateBucketMetadataTableConfiguration',
                'CreateJob',
                'CreateMultiRegionAccessPoint',
                'CreateStorageLensGroup',
                'DeleteAccessPoint',
                'DeleteAccessPointForObjectLambda',
                'DeleteBucket',
                'DeleteBucketMetadataTableConfiguration',
                'DeleteBucketWebsite',
                'DeleteMultiRegionAccessPoint',
                'DeleteObject',
                'DeleteObjectVersion',
                'DeleteStorageLensConfiguration',
                'DeleteStorageLensGroup',
                'InitiateReplication',
                'PauseReplication',
                'PutAccelerateConfiguration',
                'PutAccessPointConfigurationForObjectLambda',
                'PutAnalyticsConfiguration',
                'PutBucketCORS',
                'PutBucketLogging',
                'PutBucketNotification',
                'PutBucketObjectLockConfiguration',
                'PutBucketRequestPayment',
                'PutBucketVersioning',
                'PutBucketWebsite',
                'PutEncryptionConfiguration',
                'PutIntelligentTieringConfiguration',
                'PutInventoryConfiguration',
                'PutLifecycleConfiguration',
                'PutMetricsConfiguration',
                'PutObject',
                'PutObjectLegalHold',
                'PutObjectRetention',
                'PutReplicationConfiguration',
                'PutStorageLensConfiguration',
                'ReplicateDelete',
                'ReplicateObject',
                'RestoreObject',
                'SubmitMultiRegionAccessPointRoutes',
                'UpdateJobPriority',
                'UpdateJobStatus',
                'UpdateStorageLensGroup'
            ],
            'Permissions management': [
                'AssociateAccessGrantsIdentityCenter',
                'BypassGovernanceRetention',
                'CreateAccessGrant',
                'CreateAccessGrantsInstance',
                'CreateAccessGrantsLocation',
                'DeleteAccessGrant',
                'DeleteAccessGrantsInstance',
                'DeleteAccessGrantsInstanceResourcePolicy',
                'DeleteAccessGrantsLocation',
                'DeleteAccessPointPolicy',
                'DeleteAccessPointPolicyForObjectLambda',
                'DeleteBucketPolicy',
                'DissociateAccessGrantsIdentityCenter',
                'ObjectOwnerOverrideToBucketOwner',
                'PutAccessGrantsInstanceResourcePolicy',
                'PutAccessPointPolicy',
                'PutAccessPointPolicyForObjectLambda',
                'PutAccessPointPublicAccessBlock',
                'PutAccountPublicAccessBlock',
                'PutBucketAcl',
                'PutBucketOwnershipControls',
                'PutBucketPolicy',
                'PutBucketPublicAccessBlock',
                'PutMultiRegionAccessPointPolicy',
                'PutObjectAcl',
                'PutObjectVersionAcl',
                'UpdateAccessGrantsLocation'
            ],
            Tagging: [
                'DeleteJobTagging',
                'DeleteObjectTagging',
                'DeleteObjectVersionTagging',
                'DeleteStorageLensConfigurationTagging',
                'PutBucketTagging',
                'PutJobTagging',
                'PutObjectTagging',
                'PutObjectVersionTagging',
                'PutStorageLensConfigurationTagging',
                'ReplicateTags',
                'TagResource',
                'UntagResource'
            ],
            Read: [
                'DescribeJob',
                'DescribeMultiRegionAccessPointOperation',
                'GetAccelerateConfiguration',
                'GetAccessGrant',
                'GetAccessGrantsInstance',
                'GetAccessGrantsInstanceForPrefix',
                'GetAccessGrantsInstanceResourcePolicy',
                'GetAccessGrantsLocation',
                'GetAccessPoint',
                'GetAccessPointConfigurationForObjectLambda',
                'GetAccessPointForObjectLambda',
                'GetAccessPointPolicy',
                'GetAccessPointPolicyForObjectLambda',
                'GetAccessPointPolicyStatus',
                'GetAccessPointPolicyStatusForObjectLambda',
                'GetAccountPublicAccessBlock',
                'GetAnalyticsConfiguration',
                'GetBucketAcl',
                'GetBucketCORS',
                'GetBucketLocation',
                'GetBucketLogging',
                'GetBucketMetadataTableConfiguration',
                'GetBucketNotification',
                'GetBucketObjectLockConfiguration',
                'GetBucketOwnershipControls',
                'GetBucketPolicy',
                'GetBucketPolicyStatus',
                'GetBucketPublicAccessBlock',
                'GetBucketRequestPayment',
                'GetBucketTagging',
                'GetBucketVersioning',
                'GetBucketWebsite',
                'GetDataAccess',
                'GetEncryptionConfiguration',
                'GetIntelligentTieringConfiguration',
                'GetInventoryConfiguration',
                'GetJobTagging',
                'GetLifecycleConfiguration',
                'GetMetricsConfiguration',
                'GetMultiRegionAccessPoint',
                'GetMultiRegionAccessPointPolicy',
                'GetMultiRegionAccessPointPolicyStatus',
                'GetMultiRegionAccessPointRoutes',
                'GetObject',
                'GetObjectAcl',
                'GetObjectAttributes',
                'GetObjectLegalHold',
                'GetObjectRetention',
                'GetObjectTagging',
                'GetObjectTorrent',
                'GetObjectVersion',
                'GetObjectVersionAcl',
                'GetObjectVersionAttributes',
                'GetObjectVersionForReplication',
                'GetObjectVersionTagging',
                'GetObjectVersionTorrent',
                'GetReplicationConfiguration',
                'GetStorageLensConfiguration',
                'GetStorageLensConfigurationTagging',
                'GetStorageLensDashboard',
                'GetStorageLensGroup'
            ],
            List: [
                'ListAccessGrants',
                'ListAccessGrantsInstances',
                'ListAccessGrantsLocations',
                'ListAccessPoints',
                'ListAccessPointsForObjectLambda',
                'ListAllMyBuckets',
                'ListBucket',
                'ListBucketMultipartUploads',
                'ListBucketVersions',
                'ListCallerAccessGrants',
                'ListJobs',
                'ListMultiRegionAccessPoints',
                'ListMultipartUploadParts',
                'ListStorageLensConfigurations',
                'ListStorageLensGroups',
                'ListTagsForResource'
            ]
        };
    }
}
exports.S3 = S3;
//# sourceMappingURL=data:application/json;base64,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