"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SagemakerGeospatial = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [sagemaker-geospatial](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsagemakergeospatialcapabilities.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class SagemakerGeospatial extends shared_1.PolicyStatement {
    /**
     * Grants permission to the DeleteEarthObservationJob operation which deletes an existing earth observation job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_DeleteEarthObservationJob.html
     */
    toDeleteEarthObservationJob() {
        return this.to('DeleteEarthObservationJob');
    }
    /**
     * Grants permission to the DeleteVectorEnrichmentJob operation which deletes an existing vector enrichment job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_DeleteVectorEnrichmentJob.html
     */
    toDeleteVectorEnrichmentJob() {
        return this.to('DeleteVectorEnrichmentJob');
    }
    /**
     * Grants permission to copy results of an earth observation job to an S3 location
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_ExportEarthObservationJob.html
     */
    toExportEarthObservationJob() {
        return this.to('ExportEarthObservationJob');
    }
    /**
     * Grants permission to copy results of an VectorEnrichmentJob to an S3 location
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_ExportVectorEnrichmentJob.html
     */
    toExportVectorEnrichmentJob() {
        return this.to('ExportVectorEnrichmentJob');
    }
    /**
     * Grants permission to return details about the earth observation job
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_GetEarthObservationJob.html
     */
    toGetEarthObservationJob() {
        return this.to('GetEarthObservationJob');
    }
    /**
     * Grants permission to return details about the raster data collection
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_GetRasterDataCollection.html
     */
    toGetRasterDataCollection() {
        return this.to('GetRasterDataCollection');
    }
    /**
     * Grants permission to get the tile of an earth observation job
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_GetTile.html
     */
    toGetTile() {
        return this.to('GetTile');
    }
    /**
     * Grants permission to return details about the vector enrichment job
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_GetVectorEnrichmentJob.html
     */
    toGetVectorEnrichmentJob() {
        return this.to('GetVectorEnrichmentJob');
    }
    /**
     * Grants permission to return an array of earth observation jobs associated with the current account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_ListEarthObservationJobs.html
     */
    toListEarthObservationJobs() {
        return this.to('ListEarthObservationJobs');
    }
    /**
     * Grants permission to return an array of aster data collections associated with the given model name
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_ListRasterDataCollections.html
     */
    toListRasterDataCollections() {
        return this.to('ListRasterDataCollections');
    }
    /**
     * Grants permission to lists tag for an SageMaker Geospatial resource
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to return an array of vector enrichment jobs associated with the current account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_ListVectorEnrichmentJobs.html
     */
    toListVectorEnrichmentJobs() {
        return this.to('ListVectorEnrichmentJobs');
    }
    /**
     * Grants permission to query raster data collections
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_SearchRasterDataCollection.html
     */
    toSearchRasterDataCollection() {
        return this.to('SearchRasterDataCollection');
    }
    /**
     * Grants permission to the StartEarthObservationJob operation which starts a new earth observation job to your account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker-geospatial:TagResource
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_StartEarthObservationJob.html
     */
    toStartEarthObservationJob() {
        return this.to('StartEarthObservationJob');
    }
    /**
     * Grants permission to the StartVectorEnrichmentJob operation which starts a new vector enrichment job to your account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker-geospatial:TagResource
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_StartVectorEnrichmentJob.html
     */
    toStartVectorEnrichmentJob() {
        return this.to('StartVectorEnrichmentJob');
    }
    /**
     * Grants permission to the StopEarthObservationJob operation which stops an existing earth observation job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_StopEarthObservationJob.html
     */
    toStopEarthObservationJob() {
        return this.to('StopEarthObservationJob');
    }
    /**
     * Grants permission to the StopVectorEnrichmentJob operation which stops an existing vector enrichment job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_StopVectorEnrichmentJob.html
     */
    toStopVectorEnrichmentJob() {
        return this.to('StopVectorEnrichmentJob');
    }
    /**
     * Grants permission to tag an SageMaker Geospatial resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag an SageMaker Geospatial resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_geospatial_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Adds a resource of type EarthObservationJob to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/geospatial-eoj.html
     *
     * @param jobID - Identifier for the jobID.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEarthObservationJob(jobID, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker-geospatial:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:earth-observation-job/${jobID}`);
    }
    /**
     * Adds a resource of type RasterDataCollection to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/geospatial-data-collections.html
     *
     * @param collectionID - Identifier for the collectionID.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRasterDataCollection(collectionID, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker-geospatial:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:raster-data-collection/${collectionID}`);
    }
    /**
     * Adds a resource of type VectorEnrichmentJob to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/geospatial-vej.html
     *
     * @param jobID - Identifier for the jobID.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVectorEnrichmentJob(jobID, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker-geospatial:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:vector-enrichment-job/${jobID}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toStartEarthObservationJob()
     * - .toStartVectorEnrichmentJob()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toDeleteEarthObservationJob()
     * - .toDeleteVectorEnrichmentJob()
     * - .toExportEarthObservationJob()
     * - .toExportVectorEnrichmentJob()
     * - .toGetEarthObservationJob()
     * - .toGetRasterDataCollection()
     * - .toGetVectorEnrichmentJob()
     * - .toListTagsForResource()
     * - .toStopEarthObservationJob()
     * - .toStopVectorEnrichmentJob()
     * - .toTagResource()
     *
     * Applies to resource types:
     * - EarthObservationJob
     * - RasterDataCollection
     * - VectorEnrichmentJob
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toStartEarthObservationJob()
     * - .toStartVectorEnrichmentJob()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [sagemaker-geospatial](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsagemakergeospatialcapabilities.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'sagemaker-geospatial';
        this.accessLevelList = {
            Write: [
                'DeleteEarthObservationJob',
                'DeleteVectorEnrichmentJob',
                'ExportEarthObservationJob',
                'ExportVectorEnrichmentJob',
                'StartEarthObservationJob',
                'StartVectorEnrichmentJob',
                'StopEarthObservationJob',
                'StopVectorEnrichmentJob'
            ],
            Read: [
                'GetEarthObservationJob',
                'GetRasterDataCollection',
                'GetTile',
                'GetVectorEnrichmentJob',
                'SearchRasterDataCollection'
            ],
            List: [
                'ListEarthObservationJobs',
                'ListRasterDataCollections',
                'ListTagsForResource',
                'ListVectorEnrichmentJobs'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.SagemakerGeospatial = SagemakerGeospatial;
//# sourceMappingURL=data:application/json;base64,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