"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Servicecatalog = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [servicecatalog](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsservicecatalog.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Servicecatalog extends shared_1.PolicyStatement {
    /**
     * Grants permission to accept a portfolio that has been shared with you
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AcceptPortfolioShare.html
     */
    toAcceptPortfolioShare() {
        return this.to('AcceptPortfolioShare');
    }
    /**
     * Grants permission to associate an attribute group with an application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_app-registry_AssociateAttributeGroup.html
     */
    toAssociateAttributeGroup() {
        return this.to('AssociateAttributeGroup');
    }
    /**
     * Grants permission to associate a budget with a resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AssociateBudgetWithResource.html
     */
    toAssociateBudgetWithResource() {
        return this.to('AssociateBudgetWithResource');
    }
    /**
     * Grants permission to associate an IAM principal with a portfolio, giving the specified principal access to any products associated with the specified portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AssociatePrincipalWithPortfolio.html
     */
    toAssociatePrincipalWithPortfolio() {
        return this.to('AssociatePrincipalWithPortfolio');
    }
    /**
     * Grants permission to associate a product with a portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AssociateProductWithPortfolio.html
     */
    toAssociateProductWithPortfolio() {
        return this.to('AssociateProductWithPortfolio');
    }
    /**
     * Grants permission to associate a resource with an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourceType()
     * - .ifResource()
     *
     * Dependent actions:
     * - cloudformation:DescribeStacks
     * - resource-groups:CreateGroup
     * - resource-groups:GetGroup
     * - resource-groups:Tag
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_app-registry_AssociateResource.html
     */
    toAssociateResource() {
        return this.to('AssociateResource');
    }
    /**
     * Grants permission to associate an action with a provisioning artifact
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AssociateServiceActionWithProvisioningArtifact.html
     */
    toAssociateServiceActionWithProvisioningArtifact() {
        return this.to('AssociateServiceActionWithProvisioningArtifact');
    }
    /**
     * Grants permission to associate the specified TagOption with the specified portfolio or product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_AssociateTagOptionWithResource.html
     */
    toAssociateTagOptionWithResource() {
        return this.to('AssociateTagOptionWithResource');
    }
    /**
     * Grants permission to associate multiple self-service actions with provisioning artifacts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_BatchAssociateServiceActionWithProvisioningArtifact.html
     */
    toBatchAssociateServiceActionWithProvisioningArtifact() {
        return this.to('BatchAssociateServiceActionWithProvisioningArtifact');
    }
    /**
     * Grants permission to disassociate a batch of self-service actions from the specified provisioning artifact
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_BatchDisassociateServiceActionFromProvisioningArtifact.html
     */
    toBatchDisassociateServiceActionFromProvisioningArtifact() {
        return this.to('BatchDisassociateServiceActionFromProvisioningArtifact');
    }
    /**
     * Grants permission to copy the specified source product to the specified target product or a new product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CopyProduct.html
     */
    toCopyProduct() {
        return this.to('CopyProduct');
    }
    /**
     * Grants permission to create an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_app-registry_CreateApplication.html
     */
    toCreateApplication() {
        return this.to('CreateApplication');
    }
    /**
     * Grants permission to create an attribute group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_app-registry_CreateAttributeGroup.html
     */
    toCreateAttributeGroup() {
        return this.to('CreateAttributeGroup');
    }
    /**
     * Grants permission to create a constraint on an associated product and portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateConstraint.html
     */
    toCreateConstraint() {
        return this.to('CreateConstraint');
    }
    /**
     * Grants permission to create a portfolio
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreatePortfolio.html
     */
    toCreatePortfolio() {
        return this.to('CreatePortfolio');
    }
    /**
     * Grants permission to share a portfolio you own with another AWS account
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreatePortfolioShare.html
     */
    toCreatePortfolioShare() {
        return this.to('CreatePortfolioShare');
    }
    /**
     * Grants permission to create a product and that product's first provisioning artifact
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateProduct.html
     */
    toCreateProduct() {
        return this.to('CreateProduct');
    }
    /**
     * Grants permission to add a new provisioned product plan
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateProvisionedProductPlan.html
     */
    toCreateProvisionedProductPlan() {
        return this.to('CreateProvisionedProductPlan');
    }
    /**
     * Grants permission to add a new provisioning artifact to an existing product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateProvisioningArtifact.html
     */
    toCreateProvisioningArtifact() {
        return this.to('CreateProvisioningArtifact');
    }
    /**
     * Grants permission to create a self-service action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateServiceAction.html
     */
    toCreateServiceAction() {
        return this.to('CreateServiceAction');
    }
    /**
     * Grants permission to create a TagOption
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_CreateTagOption.html
     */
    toCreateTagOption() {
        return this.to('CreateTagOption');
    }
    /**
     * Grants permission to delete an application if all associations have been removed from the application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_app-registry_DeleteApplication.html
     */
    toDeleteApplication() {
        return this.to('DeleteApplication');
    }
    /**
     * Grants permission to delete an attribute group if all associations have been removed from the attribute group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_app-registry_DeleteAttributeGroup.html
     */
    toDeleteAttributeGroup() {
        return this.to('DeleteAttributeGroup');
    }
    /**
     * Grants permission to remove and delete an existing constraint from an associated product and portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteConstraint.html
     */
    toDeleteConstraint() {
        return this.to('DeleteConstraint');
    }
    /**
     * Grants permission to delete a portfolio if all associations and shares have been removed from the portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeletePortfolio.html
     */
    toDeletePortfolio() {
        return this.to('DeletePortfolio');
    }
    /**
     * Grants permission to unshare a portfolio you own from an AWS account you previously shared the portfolio with
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeletePortfolioShare.html
     */
    toDeletePortfolioShare() {
        return this.to('DeletePortfolioShare');
    }
    /**
     * Grants permission to delete a product if all associations have been removed from the product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteProduct.html
     */
    toDeleteProduct() {
        return this.to('DeleteProduct');
    }
    /**
     * Grants permission to delete a provisioned product plan
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteProvisionedProductPlan.html
     */
    toDeleteProvisionedProductPlan() {
        return this.to('DeleteProvisionedProductPlan');
    }
    /**
     * Grants permission to delete a provisioning artifact from a product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteProvisioningArtifact.html
     */
    toDeleteProvisioningArtifact() {
        return this.to('DeleteProvisioningArtifact');
    }
    /**
     * Grants permission to delete a resource-based policy for the specified resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/arguide/sharing-definitions.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete a self-service action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteServiceAction.html
     */
    toDeleteServiceAction() {
        return this.to('DeleteServiceAction');
    }
    /**
     * Grants permission to delete the specified TagOption
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DeleteTagOption.html
     */
    toDeleteTagOption() {
        return this.to('DeleteTagOption');
    }
    /**
     * Grants permission to describe a constraint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeConstraint.html
     */
    toDescribeConstraint() {
        return this.to('DescribeConstraint');
    }
    /**
     * Grants permission to get the status of the specified copy product operation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeCopyProductStatus.html
     */
    toDescribeCopyProductStatus() {
        return this.to('DescribeCopyProductStatus');
    }
    /**
     * Grants permission to describe a portfolio
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribePortfolio.html
     */
    toDescribePortfolio() {
        return this.to('DescribePortfolio');
    }
    /**
     * Grants permission to get the status of the specified portfolio share operation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribePortfolioShareStatus.html
     */
    toDescribePortfolioShareStatus() {
        return this.to('DescribePortfolioShareStatus');
    }
    /**
     * Grants permission to view a summary of each of the portfolio shares that were created for the specified portfolio
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribePortfolioShares.html
     */
    toDescribePortfolioShares() {
        return this.to('DescribePortfolioShares');
    }
    /**
     * Grants permission to describe a product as an end-user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProduct.html
     */
    toDescribeProduct() {
        return this.to('DescribeProduct');
    }
    /**
     * Grants permission to describe a product as an admin
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProductAsAdmin.html
     */
    toDescribeProductAsAdmin() {
        return this.to('DescribeProductAsAdmin');
    }
    /**
     * Grants permission to describe a product as an end-user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProductView.html
     */
    toDescribeProductView() {
        return this.to('DescribeProductView');
    }
    /**
     * Grants permission to describe a provisioned product
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProvisionedProduct.html
     */
    toDescribeProvisionedProduct() {
        return this.to('DescribeProvisionedProduct');
    }
    /**
     * Grants permission to describe a provisioned product plan
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProvisionedProductPlan.html
     */
    toDescribeProvisionedProductPlan() {
        return this.to('DescribeProvisionedProductPlan');
    }
    /**
     * Grants permission to describe a provisioning artifact
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProvisioningArtifact.html
     */
    toDescribeProvisioningArtifact() {
        return this.to('DescribeProvisioningArtifact');
    }
    /**
     * Grants permission to describe the parameters that you need to specify to successfully provision a specified provisioning artifact
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeProvisioningParameters.html
     */
    toDescribeProvisioningParameters() {
        return this.to('DescribeProvisioningParameters');
    }
    /**
     * Grants permission to describe a record and lists any outputs
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeRecord.html
     */
    toDescribeRecord() {
        return this.to('DescribeRecord');
    }
    /**
     * Grants permission to describe a self-service action
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeServiceAction.html
     */
    toDescribeServiceAction() {
        return this.to('DescribeServiceAction');
    }
    /**
     * Grants permission to get the default parameters if you executed the specified Service Action on the specified Provisioned Product
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeServiceActionExecutionParameters.html
     */
    toDescribeServiceActionExecutionParameters() {
        return this.to('DescribeServiceActionExecutionParameters');
    }
    /**
     * Grants permission to get information about the specified TagOption
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DescribeTagOption.html
     */
    toDescribeTagOption() {
        return this.to('DescribeTagOption');
    }
    /**
     * Grants permission to disable portfolio sharing through AWS Organizations feature
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisableAWSOrganizationsAccess.html
     */
    toDisableAWSOrganizationsAccess() {
        return this.to('DisableAWSOrganizationsAccess');
    }
    /**
     * Grants permission to disassociate an attribute group from an application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_app-registry_DisassociateAttributeGroup.html
     */
    toDisassociateAttributeGroup() {
        return this.to('DisassociateAttributeGroup');
    }
    /**
     * Grants permission to disassociate a budget from a resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisassociateBudgetFromResource.html
     */
    toDisassociateBudgetFromResource() {
        return this.to('DisassociateBudgetFromResource');
    }
    /**
     * Grants permission to disassociate an IAM principal from a portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisassociatePrincipalFromPortfolio.html
     */
    toDisassociatePrincipalFromPortfolio() {
        return this.to('DisassociatePrincipalFromPortfolio');
    }
    /**
     * Grants permission to disassociate a product from a portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisassociateProductFromPortfolio.html
     */
    toDisassociateProductFromPortfolio() {
        return this.to('DisassociateProductFromPortfolio');
    }
    /**
     * Grants permission to disassociate a resource from an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourceType()
     * - .ifResource()
     *
     * Dependent actions:
     * - resource-groups:DeleteGroup
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_app-registry_DisassociateResource.html
     */
    toDisassociateResource() {
        return this.to('DisassociateResource');
    }
    /**
     * Grants permission to disassociate the specified self-service action association from the specified provisioning artifact
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisassociateServiceActionFromProvisioningArtifact.html
     */
    toDisassociateServiceActionFromProvisioningArtifact() {
        return this.to('DisassociateServiceActionFromProvisioningArtifact');
    }
    /**
     * Grants permission to disassociate the specified TagOption from the specified resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_DisassociateTagOptionFromResource.html
     */
    toDisassociateTagOptionFromResource() {
        return this.to('DisassociateTagOptionFromResource');
    }
    /**
     * Grants permission to enable portfolio sharing feature through AWS Organizations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_EnableAWSOrganizationsAccess.html
     */
    toEnableAWSOrganizationsAccess() {
        return this.to('EnableAWSOrganizationsAccess');
    }
    /**
     * Grants permission to execute a provisioned product plan
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ExecuteProvisionedProductPlan.html
     */
    toExecuteProvisionedProductPlan() {
        return this.to('ExecuteProvisionedProductPlan');
    }
    /**
     * Grants permission to executes a provisioned product plan
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ExecuteProvisionedProductServiceAction.html
     */
    toExecuteProvisionedProductServiceAction() {
        return this.to('ExecuteProvisionedProductServiceAction');
    }
    /**
     * Grants permission to get the access status of AWS Organization portfolio share feature
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_GetAWSOrganizationsAccessStatus.html
     */
    toGetAWSOrganizationsAccessStatus() {
        return this.to('GetAWSOrganizationsAccessStatus');
    }
    /**
     * Grants permission to get an application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_app-registry_GetApplication.html
     */
    toGetApplication() {
        return this.to('GetApplication');
    }
    /**
     * Grants permission to get information about a resource associated to an application
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifResourceType()
     * - .ifResource()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_app-registry_GetAssociatedResource.html
     */
    toGetAssociatedResource() {
        return this.to('GetAssociatedResource');
    }
    /**
     * Grants permission to get an attribute group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_app-registry_GetAttributeGroup.html
     */
    toGetAttributeGroup() {
        return this.to('GetAttributeGroup');
    }
    /**
     * Grants permission to read AppRegistry configurations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_app-registry_GetConfiguration.html
     */
    toGetConfiguration() {
        return this.to('GetConfiguration');
    }
    /**
     * Grants permission to get the provisioned product output with either provisioned product id or name
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_GetProvisionedProductOutputs.html
     */
    toGetProvisionedProductOutputs() {
        return this.to('GetProvisionedProductOutputs');
    }
    /**
     * Grants permission to get a resource-based policy for the specified resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/arguide/sharing-definitions.html
     */
    toGetResourcePolicy() {
        return this.to('GetResourcePolicy');
    }
    /**
     * Grants permission to import a resource into a provisioned product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ImportAsProvisionedProduct.html
     */
    toImportAsProvisionedProduct() {
        return this.to('ImportAsProvisionedProduct');
    }
    /**
     * Grants permission to list the portfolios that have been shared with you and you have accepted
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListAcceptedPortfolioShares.html
     */
    toListAcceptedPortfolioShares() {
        return this.to('ListAcceptedPortfolioShares');
    }
    /**
     * Grants permission to list your applications
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_app-registry_ListApplications.html
     */
    toListApplications() {
        return this.to('ListApplications');
    }
    /**
     * Grants permission to list the attribute groups associated with an application
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_app-registry_ListAssociatedAttributeGroups.html
     */
    toListAssociatedAttributeGroups() {
        return this.to('ListAssociatedAttributeGroups');
    }
    /**
     * Grants permission to list the resources associated with an application
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_app-registry_ListAssociatedResources.html
     */
    toListAssociatedResources() {
        return this.to('ListAssociatedResources');
    }
    /**
     * Grants permission to list your attribute groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_app-registry_ListAttributeGroups.html
     */
    toListAttributeGroups() {
        return this.to('ListAttributeGroups');
    }
    /**
     * Grants permission to list the associated attribute groups for a given application
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_app-registry_ListAttributeGroupsForApplication.html
     */
    toListAttributeGroupsForApplication() {
        return this.to('ListAttributeGroupsForApplication');
    }
    /**
     * Grants permission to list all the budgets associated to a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListBudgetsForResource.html
     */
    toListBudgetsForResource() {
        return this.to('ListBudgetsForResource');
    }
    /**
     * Grants permission to list constraints associated with a given portfolio
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListConstraintsForPortfolio.html
     */
    toListConstraintsForPortfolio() {
        return this.to('ListConstraintsForPortfolio');
    }
    /**
     * Grants permission to list the different ways to launch a given product as an end-user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListLaunchPaths.html
     */
    toListLaunchPaths() {
        return this.to('ListLaunchPaths');
    }
    /**
     * Grants permission to list the organization nodes that have access to the specified portfolio
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListOrganizationPortfolioAccess.html
     */
    toListOrganizationPortfolioAccess() {
        return this.to('ListOrganizationPortfolioAccess');
    }
    /**
     * Grants permission to list the AWS accounts you have shared a given portfolio with
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListPortfolioAccess.html
     */
    toListPortfolioAccess() {
        return this.to('ListPortfolioAccess');
    }
    /**
     * Grants permission to list the portfolios in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListPortfolios.html
     */
    toListPortfolios() {
        return this.to('ListPortfolios');
    }
    /**
     * Grants permission to list the portfolios associated with a given product
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListPortfoliosForProduct.html
     */
    toListPortfoliosForProduct() {
        return this.to('ListPortfoliosForProduct');
    }
    /**
     * Grants permission to list the IAM principals associated with a given portfolio
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListPrincipalsForPortfolio.html
     */
    toListPrincipalsForPortfolio() {
        return this.to('ListPrincipalsForPortfolio');
    }
    /**
     * Grants permission to list the provisioned product plans
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListProvisionedProductPlans.html
     */
    toListProvisionedProductPlans() {
        return this.to('ListProvisionedProductPlans');
    }
    /**
     * Grants permission to list the provisioning artifacts associated with a given product
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListProvisioningArtifacts.html
     */
    toListProvisioningArtifacts() {
        return this.to('ListProvisioningArtifacts');
    }
    /**
     * Grants permission to list all provisioning artifacts for the specified self-service action
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListProvisioningArtifactsForServiceAction.html
     */
    toListProvisioningArtifactsForServiceAction() {
        return this.to('ListProvisioningArtifactsForServiceAction');
    }
    /**
     * Grants permission to list all the records in your account or all the records related to a given provisioned product
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListRecordHistory.html
     */
    toListRecordHistory() {
        return this.to('ListRecordHistory');
    }
    /**
     * Grants permission to list the resources associated with the specified TagOption
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListResourcesForTagOption.html
     */
    toListResourcesForTagOption() {
        return this.to('ListResourcesForTagOption');
    }
    /**
     * Grants permission to list all self-service actions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListServiceActions.html
     */
    toListServiceActions() {
        return this.to('ListServiceActions');
    }
    /**
     * Grants permission to list all the service actions associated with the specified provisioning artifact in your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListServiceActionsForProvisioningArtifact.html
     */
    toListServiceActionsForProvisioningArtifact() {
        return this.to('ListServiceActionsForProvisioningArtifact');
    }
    /**
     * Grants permission to list account, region and status of each stack instances that are associated with a CFN_STACKSET type provisioned product
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListStackInstancesForProvisionedProduct.html
     */
    toListStackInstancesForProvisionedProduct() {
        return this.to('ListStackInstancesForProvisionedProduct');
    }
    /**
     * Grants permission to list the specified TagOptions or all TagOptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ListTagOptions.html
     */
    toListTagOptions() {
        return this.to('ListTagOptions');
    }
    /**
     * Grants permission to list the tags for a service catalog appregistry resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_app-registry_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to notify the result of the provisioning engine execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_NotifyProvisionProductEngineWorkflowResult.html
     */
    toNotifyProvisionProductEngineWorkflowResult() {
        return this.to('NotifyProvisionProductEngineWorkflowResult');
    }
    /**
     * Grants permission to notify the result of the terminate engine execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_NotifyTerminateProvisionedProductEngineWorkflowResult.html
     */
    toNotifyTerminateProvisionedProductEngineWorkflowResult() {
        return this.to('NotifyTerminateProvisionedProductEngineWorkflowResult');
    }
    /**
     * Grants permission to notify the result of the update engine execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_NotifyUpdateProvisionedProductEngineWorkflowResult.html
     */
    toNotifyUpdateProvisionedProductEngineWorkflowResult() {
        return this.to('NotifyUpdateProvisionedProductEngineWorkflowResult');
    }
    /**
     * Grants permission to provision a product with a specified provisioning artifact and launch parameters
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ProvisionProduct.html
     */
    toProvisionProduct() {
        return this.to('ProvisionProduct');
    }
    /**
     * Grants permission to assign AppRegistry configurations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_app-registry_PutConfiguration.html
     */
    toPutConfiguration() {
        return this.to('PutConfiguration');
    }
    /**
     * Grants permission to add a resource-based policy for the specified resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/arguide/sharing-definitions.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to reject a portfolio that has been shared with you that you previously accepted
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_RejectPortfolioShare.html
     */
    toRejectPortfolioShare() {
        return this.to('RejectPortfolioShare');
    }
    /**
     * Grants permission to list all the provisioned products in your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ScanProvisionedProducts.html
     */
    toScanProvisionedProducts() {
        return this.to('ScanProvisionedProducts');
    }
    /**
     * Grants permission to list the products available to you as an end-user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_SearchProducts.html
     */
    toSearchProducts() {
        return this.to('SearchProducts');
    }
    /**
     * Grants permission to list all the products in your account or all the products associated with a given portfolio
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_SearchProductsAsAdmin.html
     */
    toSearchProductsAsAdmin() {
        return this.to('SearchProductsAsAdmin');
    }
    /**
     * Grants permission to list all the provisioned products in your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_SearchProvisionedProducts.html
     */
    toSearchProvisionedProducts() {
        return this.to('SearchProvisionedProducts');
    }
    /**
     * Grants permission to sync a resource with its current state in AppRegistry
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - cloudformation:UpdateStack
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_app-registry_SyncResource.html
     */
    toSyncResource() {
        return this.to('SyncResource');
    }
    /**
     * Grants permission to tag a service catalog appregistry resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_app-registry_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to terminate an existing provisioned product
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_TerminateProvisionedProduct.html
     */
    toTerminateProvisionedProduct() {
        return this.to('TerminateProvisionedProduct');
    }
    /**
     * Grants permission to remove a tag from a service catalog appregistry resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_app-registry_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the attributes of an existing application
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_app-registry_UpdateApplication.html
     */
    toUpdateApplication() {
        return this.to('UpdateApplication');
    }
    /**
     * Grants permission to update the attributes of an existing attribute group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_app-registry_UpdateAttributeGroup.html
     */
    toUpdateAttributeGroup() {
        return this.to('UpdateAttributeGroup');
    }
    /**
     * Grants permission to update the metadata fields of an existing constraint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateConstraint.html
     */
    toUpdateConstraint() {
        return this.to('UpdateConstraint');
    }
    /**
     * Grants permission to update the metadata fields and/or tags of an existing portfolio
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdatePortfolio.html
     */
    toUpdatePortfolio() {
        return this.to('UpdatePortfolio');
    }
    /**
     * Grants permission to enable or disable resource sharing for an existing portfolio share
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdatePortfolioShare.html
     */
    toUpdatePortfolioShare() {
        return this.to('UpdatePortfolioShare');
    }
    /**
     * Grants permission to update the metadata fields and/or tags of an existing product
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateProduct.html
     */
    toUpdateProduct() {
        return this.to('UpdateProduct');
    }
    /**
     * Grants permission to update an existing provisioned product
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAccountLevel()
     * - .ifRoleLevel()
     * - .ifUserLevel()
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateProvisionedProduct.html
     */
    toUpdateProvisionedProduct() {
        return this.to('UpdateProvisionedProduct');
    }
    /**
     * Grants permission to update the properties of an existing provisioned product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateProvisionedProductProperties.html
     */
    toUpdateProvisionedProductProperties() {
        return this.to('UpdateProvisionedProductProperties');
    }
    /**
     * Grants permission to update the metadata fields of an existing provisioning artifact
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateProvisioningArtifact.html
     */
    toUpdateProvisioningArtifact() {
        return this.to('UpdateProvisioningArtifact');
    }
    /**
     * Grants permission to update a self-service action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateServiceAction.html
     */
    toUpdateServiceAction() {
        return this.to('UpdateServiceAction');
    }
    /**
     * Grants permission to update the specified TagOption
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_UpdateTagOption.html
     */
    toUpdateTagOption() {
        return this.to('UpdateTagOption');
    }
    /**
     * Adds a resource of type Application to the statement
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_app-registry_CreateApplication.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApplication(applicationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:servicecatalog:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/applications/${applicationId}`);
    }
    /**
     * Adds a resource of type AttributeGroup to the statement
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_app-registry_CreateAttributeGroup.html
     *
     * @param attributeGroupId - Identifier for the attributeGroupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAttributeGroup(attributeGroupId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:servicecatalog:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/attribute-groups/${attributeGroupId}`);
    }
    /**
     * Adds a resource of type Portfolio to the statement
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_PortfolioDetail.html
     *
     * @param portfolioId - Identifier for the portfolioId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPortfolio(portfolioId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:catalog:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:portfolio/${portfolioId}`);
    }
    /**
     * Adds a resource of type Product to the statement
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/dg/API_ProductViewDetail.html
     *
     * @param productId - Identifier for the productId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProduct(productId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:catalog:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:product/${productId}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateApplication()
     * - .toCreateAttributeGroup()
     * - .toCreatePortfolio()
     * - .toCreateProduct()
     * - .toTagResource()
     * - .toUpdatePortfolio()
     * - .toUpdateProduct()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - Application
     * - AttributeGroup
     * - Portfolio
     * - Product
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateApplication()
     * - .toCreateAttributeGroup()
     * - .toCreatePortfolio()
     * - .toCreateProduct()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdatePortfolio()
     * - .toUpdateProduct()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by controlling what value can be specified as the Resource parameter in an AppRegistry associate resource API
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/adminguide/permissions-examples.html
     *
     * Applies to actions:
     * - .toAssociateResource()
     * - .toDisassociateResource()
     * - .toGetAssociatedResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResource(value, operator) {
        return this.if(`Resource`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by controlling what value can be specified as the ResourceType parameter in an AppRegistry associate resource API
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/adminguide/permissions-examples.html
     *
     * Applies to actions:
     * - .toAssociateResource()
     * - .toDisassociateResource()
     * - .toGetAssociatedResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceType(value, operator) {
        return this.if(`ResourceType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by user to see and perform actions on resources created by anyone in the account
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/adminguide/permissions-examples.html
     *
     * Applies to actions:
     * - .toCreateProvisionedProductPlan()
     * - .toDeleteProvisionedProductPlan()
     * - .toDescribeProvisionedProduct()
     * - .toDescribeProvisionedProductPlan()
     * - .toDescribeRecord()
     * - .toDescribeServiceActionExecutionParameters()
     * - .toExecuteProvisionedProductPlan()
     * - .toExecuteProvisionedProductServiceAction()
     * - .toListProvisionedProductPlans()
     * - .toListRecordHistory()
     * - .toListServiceActionsForProvisioningArtifact()
     * - .toListStackInstancesForProvisionedProduct()
     * - .toScanProvisionedProducts()
     * - .toSearchProvisionedProducts()
     * - .toTerminateProvisionedProduct()
     * - .toUpdateProvisionedProduct()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAccountLevel(value, operator) {
        return this.if(`accountLevel`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by user to see and perform actions on resources created either by them or by anyone federating into the same role as them
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/adminguide/permissions-examples.html
     *
     * Applies to actions:
     * - .toCreateProvisionedProductPlan()
     * - .toDeleteProvisionedProductPlan()
     * - .toDescribeProvisionedProduct()
     * - .toDescribeProvisionedProductPlan()
     * - .toDescribeRecord()
     * - .toDescribeServiceActionExecutionParameters()
     * - .toExecuteProvisionedProductPlan()
     * - .toExecuteProvisionedProductServiceAction()
     * - .toListProvisionedProductPlans()
     * - .toListRecordHistory()
     * - .toListServiceActionsForProvisioningArtifact()
     * - .toListStackInstancesForProvisionedProduct()
     * - .toScanProvisionedProducts()
     * - .toSearchProvisionedProducts()
     * - .toTerminateProvisionedProduct()
     * - .toUpdateProvisionedProduct()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRoleLevel(value, operator) {
        return this.if(`roleLevel`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by user to see and perform actions on only resources that they created
     *
     * https://docs.aws.amazon.com/servicecatalog/latest/adminguide/permissions-examples.html
     *
     * Applies to actions:
     * - .toCreateProvisionedProductPlan()
     * - .toDeleteProvisionedProductPlan()
     * - .toDescribeProvisionedProduct()
     * - .toDescribeProvisionedProductPlan()
     * - .toDescribeRecord()
     * - .toDescribeServiceActionExecutionParameters()
     * - .toExecuteProvisionedProductPlan()
     * - .toExecuteProvisionedProductServiceAction()
     * - .toListProvisionedProductPlans()
     * - .toListRecordHistory()
     * - .toListServiceActionsForProvisioningArtifact()
     * - .toListStackInstancesForProvisionedProduct()
     * - .toScanProvisionedProducts()
     * - .toSearchProvisionedProducts()
     * - .toTerminateProvisionedProduct()
     * - .toUpdateProvisionedProduct()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUserLevel(value, operator) {
        return this.if(`userLevel`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [servicecatalog](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsservicecatalog.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'servicecatalog';
        this.accessLevelList = {
            Write: [
                'AcceptPortfolioShare',
                'AssociateAttributeGroup',
                'AssociateBudgetWithResource',
                'AssociatePrincipalWithPortfolio',
                'AssociateProductWithPortfolio',
                'AssociateResource',
                'AssociateServiceActionWithProvisioningArtifact',
                'AssociateTagOptionWithResource',
                'BatchAssociateServiceActionWithProvisioningArtifact',
                'BatchDisassociateServiceActionFromProvisioningArtifact',
                'CopyProduct',
                'CreateApplication',
                'CreateAttributeGroup',
                'CreateConstraint',
                'CreatePortfolio',
                'CreateProduct',
                'CreateProvisionedProductPlan',
                'CreateProvisioningArtifact',
                'CreateServiceAction',
                'CreateTagOption',
                'DeleteApplication',
                'DeleteAttributeGroup',
                'DeleteConstraint',
                'DeletePortfolio',
                'DeleteProduct',
                'DeleteProvisionedProductPlan',
                'DeleteProvisioningArtifact',
                'DeleteResourcePolicy',
                'DeleteServiceAction',
                'DeleteTagOption',
                'DisableAWSOrganizationsAccess',
                'DisassociateAttributeGroup',
                'DisassociateBudgetFromResource',
                'DisassociatePrincipalFromPortfolio',
                'DisassociateProductFromPortfolio',
                'DisassociateResource',
                'DisassociateServiceActionFromProvisioningArtifact',
                'DisassociateTagOptionFromResource',
                'EnableAWSOrganizationsAccess',
                'ExecuteProvisionedProductPlan',
                'ExecuteProvisionedProductServiceAction',
                'ImportAsProvisionedProduct',
                'NotifyProvisionProductEngineWorkflowResult',
                'NotifyTerminateProvisionedProductEngineWorkflowResult',
                'NotifyUpdateProvisionedProductEngineWorkflowResult',
                'ProvisionProduct',
                'PutConfiguration',
                'PutResourcePolicy',
                'RejectPortfolioShare',
                'SyncResource',
                'TerminateProvisionedProduct',
                'UpdateApplication',
                'UpdateAttributeGroup',
                'UpdateConstraint',
                'UpdatePortfolio',
                'UpdateProduct',
                'UpdateProvisionedProduct',
                'UpdateProvisionedProductProperties',
                'UpdateProvisioningArtifact',
                'UpdateServiceAction',
                'UpdateTagOption'
            ],
            'Permissions management': [
                'CreatePortfolioShare',
                'DeletePortfolioShare',
                'UpdatePortfolioShare'
            ],
            Read: [
                'DescribeConstraint',
                'DescribeCopyProductStatus',
                'DescribePortfolio',
                'DescribePortfolioShareStatus',
                'DescribeProduct',
                'DescribeProductAsAdmin',
                'DescribeProductView',
                'DescribeProvisionedProduct',
                'DescribeProvisionedProductPlan',
                'DescribeProvisioningArtifact',
                'DescribeProvisioningParameters',
                'DescribeRecord',
                'DescribeServiceAction',
                'DescribeServiceActionExecutionParameters',
                'DescribeTagOption',
                'GetAWSOrganizationsAccessStatus',
                'GetApplication',
                'GetAssociatedResource',
                'GetAttributeGroup',
                'GetConfiguration',
                'GetProvisionedProductOutputs',
                'GetResourcePolicy',
                'ListTagsForResource'
            ],
            List: [
                'DescribePortfolioShares',
                'ListAcceptedPortfolioShares',
                'ListApplications',
                'ListAssociatedAttributeGroups',
                'ListAssociatedResources',
                'ListAttributeGroups',
                'ListAttributeGroupsForApplication',
                'ListBudgetsForResource',
                'ListConstraintsForPortfolio',
                'ListLaunchPaths',
                'ListOrganizationPortfolioAccess',
                'ListPortfolioAccess',
                'ListPortfolios',
                'ListPortfoliosForProduct',
                'ListPrincipalsForPortfolio',
                'ListProvisionedProductPlans',
                'ListProvisioningArtifacts',
                'ListProvisioningArtifactsForServiceAction',
                'ListRecordHistory',
                'ListResourcesForTagOption',
                'ListServiceActions',
                'ListServiceActionsForProvisioningArtifact',
                'ListStackInstancesForProvisionedProduct',
                'ListTagOptions',
                'ScanProvisionedProducts',
                'SearchProducts',
                'SearchProductsAsAdmin',
                'SearchProvisionedProducts'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Servicecatalog = Servicecatalog;
//# sourceMappingURL=data:application/json;base64,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