"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PrivateNetworks = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [private-networks](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsserviceprovidingmanagedprivatenetworks.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class PrivateNetworks extends shared_1.PolicyStatement {
    /**
     * Grants permission to acknowledge that an order has been received
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/private-networks/latest/APIReference/API_AcknowledgeOrderReceipt.html
     */
    toAcknowledgeOrderReceipt() {
        return this.to('AcknowledgeOrderReceipt');
    }
    /**
     * Grants permission to activate a device identifier
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/private-networks/latest/APIReference/API_ActivateDeviceIdentifier.html
     */
    toActivateDeviceIdentifier() {
        return this.to('ActivateDeviceIdentifier');
    }
    /**
     * Grants permission to activate a network site
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/private-networks/latest/APIReference/API_ActivateNetworkSite.html
     */
    toActivateNetworkSite() {
        return this.to('ActivateNetworkSite');
    }
    /**
     * Grants permission to configure an access point
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/private-networks/latest/APIReference/API_ConfigureAccessPoint.html
     */
    toConfigureAccessPoint() {
        return this.to('ConfigureAccessPoint');
    }
    /**
     * Grants permission to create a network
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/private-networks/latest/APIReference/API_CreateNetwork.html
     */
    toCreateNetwork() {
        return this.to('CreateNetwork');
    }
    /**
     * Grants permission to create a network site
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/private-networks/latest/APIReference/API_CreateNetworkSite.html
     */
    toCreateNetworkSite() {
        return this.to('CreateNetworkSite');
    }
    /**
     * Grants permission to deactivate a device identifier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/private-networks/latest/APIReference/API_DeactivateDeviceIdentifier.html
     */
    toDeactivateDeviceIdentifier() {
        return this.to('DeactivateDeviceIdentifier');
    }
    /**
     * Grants permission to delete a network
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/private-networks/latest/APIReference/API_DeleteNetwork.html
     */
    toDeleteNetwork() {
        return this.to('DeleteNetwork');
    }
    /**
     * Grants permission to delete a network site
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/private-networks/latest/APIReference/API_DeleteNetworkSite.html
     */
    toDeleteNetworkSite() {
        return this.to('DeleteNetworkSite');
    }
    /**
     * Grants permission to get a device identifier
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/private-networks/latest/APIReference/API_GetDeviceIdentifier.html
     */
    toGetDeviceIdentifier() {
        return this.to('GetDeviceIdentifier');
    }
    /**
     * Grants permission to get a network
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/private-networks/latest/APIReference/API_GetNetwork.html
     */
    toGetNetwork() {
        return this.to('GetNetwork');
    }
    /**
     * Grants permission to get a network resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/private-networks/latest/APIReference/API_GetNetworkResource.html
     */
    toGetNetworkResource() {
        return this.to('GetNetworkResource');
    }
    /**
     * Grants permission to get a network site
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/private-networks/latest/APIReference/API_GetNetworkSite.html
     */
    toGetNetworkSite() {
        return this.to('GetNetworkSite');
    }
    /**
     * Grants permission to get a network order
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/private-networks/latest/APIReference/API_GetOrder.html
     */
    toGetOrder() {
        return this.to('GetOrder');
    }
    /**
     * Grants permission to list device identifiers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/private-networks/latest/APIReference/API_ListDeviceIdentifiers.html
     */
    toListDeviceIdentifiers() {
        return this.to('ListDeviceIdentifiers');
    }
    /**
     * Grants permission to list network resources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/private-networks/latest/APIReference/API_ListNetworkResources.html
     */
    toListNetworkResources() {
        return this.to('ListNetworkResources');
    }
    /**
     * Grants permission to list network sites
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/private-networks/latest/APIReference/API_ListNetworkSites.html
     */
    toListNetworkSites() {
        return this.to('ListNetworkSites');
    }
    /**
     * Grants permission to list networks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/private-networks/latest/APIReference/API_ListNetworks.html
     */
    toListNetworks() {
        return this.to('ListNetworks');
    }
    /**
     * Grants permission to list network orders
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/private-networks/latest/APIReference/API_ListOrders.html
     */
    toListOrders() {
        return this.to('ListOrders');
    }
    /**
     * Grants permission to return a list of tags for a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/private-networks/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to check the health of the service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/private-networks/latest/APIReference/API_Ping.html
     */
    toPing() {
        return this.to('Ping');
    }
    /**
     * Grants permission to start an update on the specified network resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/private-networks/latest/APIReference/API_StartNetworkResourceUpdate.html
     */
    toStartNetworkResourceUpdate() {
        return this.to('StartNetworkResourceUpdate');
    }
    /**
     * Grants permission to adds tags to the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/private-networks/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to removes tags from the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/private-networks/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a network site
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/private-networks/latest/APIReference/API_UpdateNetworkSite.html
     */
    toUpdateNetworkSite() {
        return this.to('UpdateNetworkSite');
    }
    /**
     * Grants permission to update a plan at a network site
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/private-networks/latest/APIReference/API_UpdateNetworkSitePlan.html
     */
    toUpdateNetworkSitePlan() {
        return this.to('UpdateNetworkSitePlan');
    }
    /**
     * Adds a resource of type network to the statement
     *
     * https://docs.aws.amazon.com/private-networks/latest/userguide/identity-access-management.html
     *
     * @param networkName - Identifier for the networkName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onNetwork(networkName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:private-networks:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:network/${networkName}`);
    }
    /**
     * Adds a resource of type network-site to the statement
     *
     * https://docs.aws.amazon.com/private-networks/latest/userguide/identity-access-management.html
     *
     * @param networkName - Identifier for the networkName.
     * @param networkSiteName - Identifier for the networkSiteName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onNetworkSite(networkName, networkSiteName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:private-networks:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:network-site/${networkName}/${networkSiteName}`);
    }
    /**
     * Adds a resource of type network-resource to the statement
     *
     * https://docs.aws.amazon.com/private-networks/latest/userguide/identity-access-management.html
     *
     * @param networkName - Identifier for the networkName.
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onNetworkResource(networkName, resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:private-networks:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:network-resource/${networkName}/${resourceId}`);
    }
    /**
     * Adds a resource of type order to the statement
     *
     * https://docs.aws.amazon.com/private-networks/latest/userguide/identity-access-management.html
     *
     * @param networkName - Identifier for the networkName.
     * @param orderId - Identifier for the orderId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onOrder(networkName, orderId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:private-networks:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:order/${networkName}/${orderId}`);
    }
    /**
     * Adds a resource of type device-identifier to the statement
     *
     * https://docs.aws.amazon.com/private-networks/latest/userguide/identity-access-management.html
     *
     * @param networkName - Identifier for the networkName.
     * @param deviceId - Identifier for the deviceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeviceIdentifier(networkName, deviceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:private-networks:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:device-identifier/${networkName}/${deviceId}`);
    }
    /**
     * Filters access by checking the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toActivateNetworkSite()
     * - .toCreateNetwork()
     * - .toCreateNetworkSite()
     * - .toStartNetworkResourceUpdate()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by checking tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toActivateDeviceIdentifier()
     * - .toGetDeviceIdentifier()
     * - .toGetNetwork()
     * - .toGetNetworkResource()
     * - .toGetNetworkSite()
     * - .toGetOrder()
     *
     * Applies to resource types:
     * - network
     * - network-site
     * - network-resource
     * - order
     * - device-identifier
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toActivateNetworkSite()
     * - .toCreateNetwork()
     * - .toCreateNetworkSite()
     * - .toStartNetworkResourceUpdate()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [private-networks](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsserviceprovidingmanagedprivatenetworks.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'private-networks';
        this.accessLevelList = {
            Write: [
                'AcknowledgeOrderReceipt',
                'ActivateDeviceIdentifier',
                'ActivateNetworkSite',
                'ConfigureAccessPoint',
                'CreateNetwork',
                'CreateNetworkSite',
                'DeactivateDeviceIdentifier',
                'DeleteNetwork',
                'DeleteNetworkSite',
                'StartNetworkResourceUpdate',
                'UpdateNetworkSite',
                'UpdateNetworkSitePlan'
            ],
            Read: [
                'GetDeviceIdentifier',
                'GetNetwork',
                'GetNetworkResource',
                'GetNetworkSite',
                'GetOrder',
                'Ping'
            ],
            List: [
                'ListDeviceIdentifiers',
                'ListNetworkResources',
                'ListNetworkSites',
                'ListNetworks',
                'ListOrders',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.PrivateNetworks = PrivateNetworks;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2VydmljZXByb3ZpZGluZ21hbmFnZWRwcml2YXRlbmV0d29ya3MuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJzZXJ2aWNlcHJvdmlkaW5nbWFuYWdlZHByaXZhdGVuZXR3b3Jrcy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFDQSx5Q0FBeUQ7QUFHekQ7Ozs7R0FJRztBQUNILE1BQWEsZUFBZ0IsU0FBUSx3QkFBZTtJQUdsRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUI7UUFDOUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7SUFDNUMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLDBCQUEwQjtRQUMvQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztJQUM3QyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDRCQUE0QjtRQUNqQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNEJBQTRCLENBQUMsQ0FBQztJQUMvQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxZQUFZO1FBQ2pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxZQUFZLENBQUMsQ0FBQztJQUMvQixDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksb0JBQW9CO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLFVBQVU7UUFDZixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsVUFBVSxDQUFDLENBQUM7SUFDN0IsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLENBQUMsQ0FBQztJQUMxQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGNBQWMsQ0FBQyxDQUFDO0lBQ2pDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxZQUFZO1FBQ2pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxZQUFZLENBQUMsQ0FBQztJQUMvQixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxNQUFNO1FBQ1gsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxDQUFDO0lBQ3pCLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksNEJBQTRCO1FBQ2pDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO0lBQy9DLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksYUFBYTtRQUNsQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDaEMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGVBQWU7UUFDcEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLENBQUMsQ0FBQztJQUMxQyxDQUFDO0lBdUNEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLFNBQVMsQ0FBQyxXQUFtQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3pGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLHFCQUFzQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsWUFBYSxXQUFZLEVBQUUsQ0FBQyxDQUFDO0lBQ2hMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksYUFBYSxDQUFDLFdBQW1CLEVBQUUsZUFBdUIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN0SCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixxQkFBc0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLGlCQUFrQixXQUFZLElBQUssZUFBZ0IsRUFBRSxDQUFDLENBQUM7SUFDMU0sQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxpQkFBaUIsQ0FBQyxXQUFtQixFQUFFLFVBQWtCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDckgsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIscUJBQXNCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxxQkFBc0IsV0FBWSxJQUFLLFVBQVcsRUFBRSxDQUFDLENBQUM7SUFDek0sQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxPQUFPLENBQUMsV0FBbUIsRUFBRSxPQUFlLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDeEcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIscUJBQXNCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxVQUFXLFdBQVksSUFBSyxPQUFRLEVBQUUsQ0FBQyxDQUFDO0lBQzNMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksa0JBQWtCLENBQUMsV0FBbUIsRUFBRSxRQUFnQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3BILE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLHFCQUFzQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsc0JBQXVCLFdBQVksSUFBSyxRQUFTLEVBQUUsQ0FBQyxDQUFDO0lBQ3hNLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7O09BZUc7SUFDSSxlQUFlLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDM0YsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFtQixNQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2hGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0F1Qkc7SUFDSSxnQkFBZ0IsQ0FBQyxNQUFjLEVBQUUsS0FBd0IsRUFBRSxRQUE0QjtRQUM1RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW9CLE1BQU8sRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDakYsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7T0FlRztJQUNJLFlBQVksQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3hFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRSxDQUFDO0lBRUQ7OztPQUdHO0lBQ0gsWUFBWSxLQUFnQztRQUMxQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7UUFoaEJSLGtCQUFhLEdBQUcsa0JBQWtCLENBQUM7UUF5VWhDLG9CQUFlLEdBQW9CO1lBQzNDLEtBQUssRUFBRTtnQkFDTCx5QkFBeUI7Z0JBQ3pCLDBCQUEwQjtnQkFDMUIscUJBQXFCO2dCQUNyQixzQkFBc0I7Z0JBQ3RCLGVBQWU7Z0JBQ2YsbUJBQW1CO2dCQUNuQiw0QkFBNEI7Z0JBQzVCLGVBQWU7Z0JBQ2YsbUJBQW1CO2dCQUNuQiw0QkFBNEI7Z0JBQzVCLG1CQUFtQjtnQkFDbkIsdUJBQXVCO2FBQ3hCO1lBQ0QsSUFBSSxFQUFFO2dCQUNKLHFCQUFxQjtnQkFDckIsWUFBWTtnQkFDWixvQkFBb0I7Z0JBQ3BCLGdCQUFnQjtnQkFDaEIsVUFBVTtnQkFDVixNQUFNO2FBQ1A7WUFDRCxJQUFJLEVBQUU7Z0JBQ0osdUJBQXVCO2dCQUN2QixzQkFBc0I7Z0JBQ3RCLGtCQUFrQjtnQkFDbEIsY0FBYztnQkFDZCxZQUFZO2dCQUNaLHFCQUFxQjthQUN0QjtZQUNELE9BQU8sRUFBRTtnQkFDUCxhQUFhO2dCQUNiLGVBQWU7YUFDaEI7U0FDRixDQUFDO0lBcUtGLENBQUM7Q0FDRjtBQW5oQkQsMENBbWhCQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEFjY2Vzc0xldmVsTGlzdCB9IGZyb20gJy4uLy4uL3NoYXJlZC9hY2Nlc3MtbGV2ZWwnO1xuaW1wb3J0IHsgUG9saWN5U3RhdGVtZW50LCBPcGVyYXRvciB9IGZyb20gJy4uLy4uL3NoYXJlZCc7XG5pbXBvcnQgeyBhd3NfaWFtIGFzIGlhbSB9IGZyb20gXCJhd3MtY2RrLWxpYlwiO1xuXG4vKipcbiAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbcHJpdmF0ZS1uZXR3b3Jrc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYXdzc2VydmljZXByb3ZpZGluZ21hbmFnZWRwcml2YXRlbmV0d29ya3MuaHRtbCkuXG4gKlxuICogQHBhcmFtIHNpZCBbU0lEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3NpZC5odG1sKSBvZiB0aGUgc3RhdGVtZW50XG4gKi9cbmV4cG9ydCBjbGFzcyBQcml2YXRlTmV0d29ya3MgZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnQge1xuICBwdWJsaWMgc2VydmljZVByZWZpeCA9ICdwcml2YXRlLW5ldHdvcmtzJztcblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYWNrbm93bGVkZ2UgdGhhdCBhbiBvcmRlciBoYXMgYmVlbiByZWNlaXZlZFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9wcml2YXRlLW5ldHdvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Fja25vd2xlZGdlT3JkZXJSZWNlaXB0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Fja25vd2xlZGdlT3JkZXJSZWNlaXB0KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdBY2tub3dsZWRnZU9yZGVyUmVjZWlwdCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFjdGl2YXRlIGEgZGV2aWNlIGlkZW50aWZpZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3ByaXZhdGUtbmV0d29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQWN0aXZhdGVEZXZpY2VJZGVudGlmaWVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0FjdGl2YXRlRGV2aWNlSWRlbnRpZmllcigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQWN0aXZhdGVEZXZpY2VJZGVudGlmaWVyJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYWN0aXZhdGUgYSBuZXR3b3JrIHNpdGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcHJpdmF0ZS1uZXR3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9BY3RpdmF0ZU5ldHdvcmtTaXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0FjdGl2YXRlTmV0d29ya1NpdGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0FjdGl2YXRlTmV0d29ya1NpdGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjb25maWd1cmUgYW4gYWNjZXNzIHBvaW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3ByaXZhdGUtbmV0d29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ29uZmlndXJlQWNjZXNzUG9pbnQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ29uZmlndXJlQWNjZXNzUG9pbnQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NvbmZpZ3VyZUFjY2Vzc1BvaW50Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgbmV0d29ya1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9wcml2YXRlLW5ldHdvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZU5ldHdvcmsuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlTmV0d29yaygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlTmV0d29yaycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIG5ldHdvcmsgc2l0ZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9wcml2YXRlLW5ldHdvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZU5ldHdvcmtTaXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZU5ldHdvcmtTaXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVOZXR3b3JrU2l0ZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlYWN0aXZhdGUgYSBkZXZpY2UgaWRlbnRpZmllclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9wcml2YXRlLW5ldHdvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlYWN0aXZhdGVEZXZpY2VJZGVudGlmaWVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlYWN0aXZhdGVEZXZpY2VJZGVudGlmaWVyKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWFjdGl2YXRlRGV2aWNlSWRlbnRpZmllcicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIG5ldHdvcmtcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcHJpdmF0ZS1uZXR3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVOZXR3b3JrLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZU5ldHdvcmsoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZU5ldHdvcmsnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBuZXR3b3JrIHNpdGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcHJpdmF0ZS1uZXR3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVOZXR3b3JrU2l0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVOZXR3b3JrU2l0ZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlTmV0d29ya1NpdGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYSBkZXZpY2UgaWRlbnRpZmllclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9wcml2YXRlLW5ldHdvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldERldmljZUlkZW50aWZpZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0RGV2aWNlSWRlbnRpZmllcigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0RGV2aWNlSWRlbnRpZmllcicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBhIG5ldHdvcmtcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcHJpdmF0ZS1uZXR3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXROZXR3b3JrLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldE5ldHdvcmsoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldE5ldHdvcmsnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYSBuZXR3b3JrIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3ByaXZhdGUtbmV0d29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0TmV0d29ya1Jlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldE5ldHdvcmtSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0TmV0d29ya1Jlc291cmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGEgbmV0d29yayBzaXRlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3ByaXZhdGUtbmV0d29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0TmV0d29ya1NpdGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0TmV0d29ya1NpdGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldE5ldHdvcmtTaXRlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGEgbmV0d29yayBvcmRlclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9wcml2YXRlLW5ldHdvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldE9yZGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldE9yZGVyKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRPcmRlcicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgZGV2aWNlIGlkZW50aWZpZXJzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcHJpdmF0ZS1uZXR3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0RGV2aWNlSWRlbnRpZmllcnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdERldmljZUlkZW50aWZpZXJzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0RGV2aWNlSWRlbnRpZmllcnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IG5ldHdvcmsgcmVzb3VyY2VzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcHJpdmF0ZS1uZXR3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0TmV0d29ya1Jlc291cmNlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0TmV0d29ya1Jlc291cmNlcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdE5ldHdvcmtSZXNvdXJjZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IG5ldHdvcmsgc2l0ZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9wcml2YXRlLW5ldHdvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3ROZXR3b3JrU2l0ZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdE5ldHdvcmtTaXRlcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdE5ldHdvcmtTaXRlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgbmV0d29ya3NcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9wcml2YXRlLW5ldHdvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3ROZXR3b3Jrcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0TmV0d29ya3MoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3ROZXR3b3JrcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgbmV0d29yayBvcmRlcnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9wcml2YXRlLW5ldHdvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RPcmRlcnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdE9yZGVycygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdE9yZGVycycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHVybiBhIGxpc3Qgb2YgdGFncyBmb3IgYSByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3ByaXZhdGUtbmV0d29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdFRhZ3NGb3JSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0VGFnc0ZvclJlc291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0VGFnc0ZvclJlc291cmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY2hlY2sgdGhlIGhlYWx0aCBvZiB0aGUgc2VydmljZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3ByaXZhdGUtbmV0d29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUGluZy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QaW5nKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdQaW5nJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3RhcnQgYW4gdXBkYXRlIG9uIHRoZSBzcGVjaWZpZWQgbmV0d29yayByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9wcml2YXRlLW5ldHdvcmtzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N0YXJ0TmV0d29ya1Jlc291cmNlVXBkYXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0YXJ0TmV0d29ya1Jlc291cmNlVXBkYXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdGFydE5ldHdvcmtSZXNvdXJjZVVwZGF0ZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFkZHMgdGFncyB0byB0aGUgc3BlY2lmaWVkIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogVGFnZ2luZ1xuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcHJpdmF0ZS1uZXR3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9UYWdSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9UYWdSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVGFnUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZW1vdmVzIHRhZ3MgZnJvbSB0aGUgc3BlY2lmaWVkIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogVGFnZ2luZ1xuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcHJpdmF0ZS1uZXR3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VbnRhZ1Jlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VudGFnUmVzb3VyY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VudGFnUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSBuZXR3b3JrIHNpdGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcHJpdmF0ZS1uZXR3b3Jrcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVOZXR3b3JrU2l0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVOZXR3b3JrU2l0ZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlTmV0d29ya1NpdGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSBwbGFuIGF0IGEgbmV0d29yayBzaXRlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3ByaXZhdGUtbmV0d29ya3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlTmV0d29ya1NpdGVQbGFuLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZU5ldHdvcmtTaXRlUGxhbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlTmV0d29ya1NpdGVQbGFuJyk7XG4gIH1cblxuICBwcm90ZWN0ZWQgYWNjZXNzTGV2ZWxMaXN0OiBBY2Nlc3NMZXZlbExpc3QgPSB7XG4gICAgV3JpdGU6IFtcbiAgICAgICdBY2tub3dsZWRnZU9yZGVyUmVjZWlwdCcsXG4gICAgICAnQWN0aXZhdGVEZXZpY2VJZGVudGlmaWVyJyxcbiAgICAgICdBY3RpdmF0ZU5ldHdvcmtTaXRlJyxcbiAgICAgICdDb25maWd1cmVBY2Nlc3NQb2ludCcsXG4gICAgICAnQ3JlYXRlTmV0d29yaycsXG4gICAgICAnQ3JlYXRlTmV0d29ya1NpdGUnLFxuICAgICAgJ0RlYWN0aXZhdGVEZXZpY2VJZGVudGlmaWVyJyxcbiAgICAgICdEZWxldGVOZXR3b3JrJyxcbiAgICAgICdEZWxldGVOZXR3b3JrU2l0ZScsXG4gICAgICAnU3RhcnROZXR3b3JrUmVzb3VyY2VVcGRhdGUnLFxuICAgICAgJ1VwZGF0ZU5ldHdvcmtTaXRlJyxcbiAgICAgICdVcGRhdGVOZXR3b3JrU2l0ZVBsYW4nXG4gICAgXSxcbiAgICBSZWFkOiBbXG4gICAgICAnR2V0RGV2aWNlSWRlbnRpZmllcicsXG4gICAgICAnR2V0TmV0d29yaycsXG4gICAgICAnR2V0TmV0d29ya1Jlc291cmNlJyxcbiAgICAgICdHZXROZXR3b3JrU2l0ZScsXG4gICAgICAnR2V0T3JkZXInLFxuICAgICAgJ1BpbmcnXG4gICAgXSxcbiAgICBMaXN0OiBbXG4gICAgICAnTGlzdERldmljZUlkZW50aWZpZXJzJyxcbiAgICAgICdMaXN0TmV0d29ya1Jlc291cmNlcycsXG4gICAgICAnTGlzdE5ldHdvcmtTaXRlcycsXG4gICAgICAnTGlzdE5ldHdvcmtzJyxcbiAgICAgICdMaXN0T3JkZXJzJyxcbiAgICAgICdMaXN0VGFnc0ZvclJlc291cmNlJ1xuICAgIF0sXG4gICAgVGFnZ2luZzogW1xuICAgICAgJ1RhZ1Jlc291cmNlJyxcbiAgICAgICdVbnRhZ1Jlc291cmNlJ1xuICAgIF1cbiAgfTtcblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgbmV0d29yayB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9wcml2YXRlLW5ldHdvcmtzL2xhdGVzdC91c2VyZ3VpZGUvaWRlbnRpdHktYWNjZXNzLW1hbmFnZW1lbnQuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gbmV0d29ya05hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgbmV0d29ya05hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbk5ldHdvcmsobmV0d29ya05hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpwcml2YXRlLW5ldHdvcmtzOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpuZXR3b3JrLyR7IG5ldHdvcmtOYW1lIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBuZXR3b3JrLXNpdGUgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcHJpdmF0ZS1uZXR3b3Jrcy9sYXRlc3QvdXNlcmd1aWRlL2lkZW50aXR5LWFjY2Vzcy1tYW5hZ2VtZW50Lmh0bWxcbiAgICpcbiAgICogQHBhcmFtIG5ldHdvcmtOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIG5ldHdvcmtOYW1lLlxuICAgKiBAcGFyYW0gbmV0d29ya1NpdGVOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIG5ldHdvcmtTaXRlTmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uTmV0d29ya1NpdGUobmV0d29ya05hbWU6IHN0cmluZywgbmV0d29ya1NpdGVOYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06cHJpdmF0ZS1uZXR3b3JrczokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06bmV0d29yay1zaXRlLyR7IG5ldHdvcmtOYW1lIH0vJHsgbmV0d29ya1NpdGVOYW1lIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBuZXR3b3JrLXJlc291cmNlIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3ByaXZhdGUtbmV0d29ya3MvbGF0ZXN0L3VzZXJndWlkZS9pZGVudGl0eS1hY2Nlc3MtbWFuYWdlbWVudC5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBuZXR3b3JrTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBuZXR3b3JrTmFtZS5cbiAgICogQHBhcmFtIHJlc291cmNlSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzb3VyY2VJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uTmV0d29ya1Jlc291cmNlKG5ldHdvcmtOYW1lOiBzdHJpbmcsIHJlc291cmNlSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpwcml2YXRlLW5ldHdvcmtzOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpuZXR3b3JrLXJlc291cmNlLyR7IG5ldHdvcmtOYW1lIH0vJHsgcmVzb3VyY2VJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgb3JkZXIgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vcHJpdmF0ZS1uZXR3b3Jrcy9sYXRlc3QvdXNlcmd1aWRlL2lkZW50aXR5LWFjY2Vzcy1tYW5hZ2VtZW50Lmh0bWxcbiAgICpcbiAgICogQHBhcmFtIG5ldHdvcmtOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIG5ldHdvcmtOYW1lLlxuICAgKiBAcGFyYW0gb3JkZXJJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBvcmRlcklkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25PcmRlcihuZXR3b3JrTmFtZTogc3RyaW5nLCBvcmRlcklkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06cHJpdmF0ZS1uZXR3b3JrczokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06b3JkZXIvJHsgbmV0d29ya05hbWUgfS8keyBvcmRlcklkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBkZXZpY2UtaWRlbnRpZmllciB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9wcml2YXRlLW5ldHdvcmtzL2xhdGVzdC91c2VyZ3VpZGUvaWRlbnRpdHktYWNjZXNzLW1hbmFnZW1lbnQuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gbmV0d29ya05hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgbmV0d29ya05hbWUuXG4gICAqIEBwYXJhbSBkZXZpY2VJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBkZXZpY2VJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRGV2aWNlSWRlbnRpZmllcihuZXR3b3JrTmFtZTogc3RyaW5nLCBkZXZpY2VJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OnByaXZhdGUtbmV0d29ya3M6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmRldmljZS1pZGVudGlmaWVyLyR7IG5ldHdvcmtOYW1lIH0vJHsgZGV2aWNlSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IGNoZWNraW5nIHRoZSBwcmVzZW5jZSBvZiB0YWcga2V5LXZhbHVlIHBhaXJzIGluIHRoZSByZXF1ZXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy1yZXF1ZXN0dGFnXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9BY3RpdmF0ZU5ldHdvcmtTaXRlKClcbiAgICogLSAudG9DcmVhdGVOZXR3b3JrKClcbiAgICogLSAudG9DcmVhdGVOZXR3b3JrU2l0ZSgpXG4gICAqIC0gLnRvU3RhcnROZXR3b3JrUmVzb3VyY2VVcGRhdGUoKVxuICAgKiAtIC50b1RhZ1Jlc291cmNlKClcbiAgICpcbiAgICogQHBhcmFtIHRhZ0tleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1JlcXVlc3RUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6UmVxdWVzdFRhZy8keyB0YWdLZXkgfWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IGNoZWNraW5nIHRhZyBrZXktdmFsdWUgcGFpcnMgYXR0YWNoZWQgdG8gdGhlIHJlc291cmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy1yZXNvdXJjZXRhZ1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQWN0aXZhdGVEZXZpY2VJZGVudGlmaWVyKClcbiAgICogLSAudG9HZXREZXZpY2VJZGVudGlmaWVyKClcbiAgICogLSAudG9HZXROZXR3b3JrKClcbiAgICogLSAudG9HZXROZXR3b3JrUmVzb3VyY2UoKVxuICAgKiAtIC50b0dldE5ldHdvcmtTaXRlKClcbiAgICogLSAudG9HZXRPcmRlcigpXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gbmV0d29ya1xuICAgKiAtIG5ldHdvcmstc2l0ZVxuICAgKiAtIG5ldHdvcmstcmVzb3VyY2VcbiAgICogLSBvcmRlclxuICAgKiAtIGRldmljZS1pZGVudGlmaWVyXG4gICAqXG4gICAqIEBwYXJhbSB0YWdLZXkgVGhlIHRhZyBrZXkgdG8gY2hlY2tcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NSZXNvdXJjZVRhZyh0YWdLZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpSZXNvdXJjZVRhZy8keyB0YWdLZXkgfWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHByZXNlbmNlIG9mIHRhZyBrZXlzIGluIHRoZSByZXF1ZXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy10YWdrZXlzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9BY3RpdmF0ZU5ldHdvcmtTaXRlKClcbiAgICogLSAudG9DcmVhdGVOZXR3b3JrKClcbiAgICogLSAudG9DcmVhdGVOZXR3b3JrU2l0ZSgpXG4gICAqIC0gLnRvU3RhcnROZXR3b3JrUmVzb3VyY2VVcGRhdGUoKVxuICAgKiAtIC50b1RhZ1Jlc291cmNlKClcbiAgICogLSAudG9VbnRhZ1Jlc291cmNlKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NUYWdLZXlzKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6VGFnS2V5c2AsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbcHJpdmF0ZS1uZXR3b3Jrc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYXdzc2VydmljZXByb3ZpZGluZ21hbmFnZWRwcml2YXRlbmV0d29ya3MuaHRtbCkuXG4gICAqXG4gICAqL1xuICBjb25zdHJ1Y3Rvcihwcm9wcz86IGlhbS5Qb2xpY3lTdGF0ZW1lbnRQcm9wcykge1xuICAgIHN1cGVyKHByb3BzKTtcbiAgfVxufVxuIl19