"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ses = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [ses](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonses.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ses extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a receipt rule set by cloning an existing one
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CloneReceiptRuleSet.html
     */
    toCloneReceiptRuleSet() {
        return this.to('CloneReceiptRuleSet');
    }
    /**
     * Grants permission to create a new configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateConfigurationSet.html
     */
    toCreateConfigurationSet() {
        return this.to('CreateConfigurationSet');
    }
    /**
     * Grants permission to create a configuration set event destination
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateConfigurationSetEventDestination.html
     */
    toCreateConfigurationSetEventDestination() {
        return this.to('CreateConfigurationSetEventDestination');
    }
    /**
     * Grants permission to creates an association between a configuration set and a custom domain for open and click event tracking
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateConfigurationSetTrackingOptions.html
     */
    toCreateConfigurationSetTrackingOptions() {
        return this.to('CreateConfigurationSetTrackingOptions');
    }
    /**
     * Grants permission to create a new custom verification email template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateCustomVerificationEmailTemplate.html
     */
    toCreateCustomVerificationEmailTemplate() {
        return this.to('CreateCustomVerificationEmailTemplate');
    }
    /**
     * Grants permission to create a new IP address filter
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateReceiptFilter.html
     */
    toCreateReceiptFilter() {
        return this.to('CreateReceiptFilter');
    }
    /**
     * Grants permission to create a receipt rule
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateReceiptRule.html
     */
    toCreateReceiptRule() {
        return this.to('CreateReceiptRule');
    }
    /**
     * Grants permission to create an empty receipt rule set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateReceiptRuleSet.html
     */
    toCreateReceiptRuleSet() {
        return this.to('CreateReceiptRuleSet');
    }
    /**
     * Grants permission to creates an email template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateTemplate.html
     */
    toCreateTemplate() {
        return this.to('CreateTemplate');
    }
    /**
     * Grants permission to delete an existing configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteConfigurationSet.html
     */
    toDeleteConfigurationSet() {
        return this.to('DeleteConfigurationSet');
    }
    /**
     * Grants permission to delete an event destination
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteConfigurationSetEventDestination.html
     */
    toDeleteConfigurationSetEventDestination() {
        return this.to('DeleteConfigurationSetEventDestination');
    }
    /**
     * Grants permission to delete an association between a configuration set and a custom domain for open and click event tracking
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteConfigurationSetTrackingOptions.html
     */
    toDeleteConfigurationSetTrackingOptions() {
        return this.to('DeleteConfigurationSetTrackingOptions');
    }
    /**
     * Grants permission to delete an existing custom verification email template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteCustomVerificationEmailTemplate.html
     */
    toDeleteCustomVerificationEmailTemplate() {
        return this.to('DeleteCustomVerificationEmailTemplate');
    }
    /**
     * Grants permission to delete the specified identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteIdentity.html
     */
    toDeleteIdentity() {
        return this.to('DeleteIdentity');
    }
    /**
     * Grants permission to delete the specified sending authorization policy for the given identity (an email address or a domain)
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteIdentityPolicy.html
     */
    toDeleteIdentityPolicy() {
        return this.to('DeleteIdentityPolicy');
    }
    /**
     * Grants permission to delete the specified IP address filter
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteReceiptFilter.html
     */
    toDeleteReceiptFilter() {
        return this.to('DeleteReceiptFilter');
    }
    /**
     * Grants permission to delete the specified receipt rule
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteReceiptRule.html
     */
    toDeleteReceiptRule() {
        return this.to('DeleteReceiptRule');
    }
    /**
     * Grants permission to delete the specified receipt rule set and all of the receipt rules it contains
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteReceiptRuleSet.html
     */
    toDeleteReceiptRuleSet() {
        return this.to('DeleteReceiptRuleSet');
    }
    /**
     * Grants permission to delete an email template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteTemplate.html
     */
    toDeleteTemplate() {
        return this.to('DeleteTemplate');
    }
    /**
     * Grants permission to delete the specified email address from the list of verified addresses
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteVerifiedEmailAddress.html
     */
    toDeleteVerifiedEmailAddress() {
        return this.to('DeleteVerifiedEmailAddress');
    }
    /**
     * Grants permission to return the metadata and receipt rules for the receipt rule set that is currently active
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeActiveReceiptRuleSet.html
     */
    toDescribeActiveReceiptRuleSet() {
        return this.to('DescribeActiveReceiptRuleSet');
    }
    /**
     * Grants permission to return the details of the specified configuration set
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeConfigurationSet.html
     */
    toDescribeConfigurationSet() {
        return this.to('DescribeConfigurationSet');
    }
    /**
     * Grants permission to return the details of the specified receipt rule
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeReceiptRule.html
     */
    toDescribeReceiptRule() {
        return this.to('DescribeReceiptRule');
    }
    /**
     * Grants permission to return the details of the specified receipt rule set
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeReceiptRuleSet.html
     */
    toDescribeReceiptRuleSet() {
        return this.to('DescribeReceiptRuleSet');
    }
    /**
     * Grants permission to return the email sending status of your account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetAccountSendingEnabled.html
     */
    toGetAccountSendingEnabled() {
        return this.to('GetAccountSendingEnabled');
    }
    /**
     * Grants permission to return the custom email verification template for the template name you specify
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetCustomVerificationEmailTemplate.html
     */
    toGetCustomVerificationEmailTemplate() {
        return this.to('GetCustomVerificationEmailTemplate');
    }
    /**
     * Grants permission to return the current status of Easy DKIM signing for an entity
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityDkimAttributes.html
     */
    toGetIdentityDkimAttributes() {
        return this.to('GetIdentityDkimAttributes');
    }
    /**
     * Grants permission to return the custom MAIL FROM attributes for a list of identities (email addresses and/or domains)
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityMailFromDomainAttributes.html
     */
    toGetIdentityMailFromDomainAttributes() {
        return this.to('GetIdentityMailFromDomainAttributes');
    }
    /**
     * Grants permission to return a structure describing identity notification attributes for a list of verified identities (email addresses and/or domains),
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityNotificationAttributes.html
     */
    toGetIdentityNotificationAttributes() {
        return this.to('GetIdentityNotificationAttributes');
    }
    /**
     * Grants permission to return the requested sending authorization policies for the given identity (an email address or a domain)
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityPolicies.html
     */
    toGetIdentityPolicies() {
        return this.to('GetIdentityPolicies');
    }
    /**
     * Grants permission to return the verification status and (for domain identities) the verification token for a list of identities
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityVerificationAttributes.html
     */
    toGetIdentityVerificationAttributes() {
        return this.to('GetIdentityVerificationAttributes');
    }
    /**
     * Grants permission to return the user's current sending limits
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetSendQuota.html
     */
    toGetSendQuota() {
        return this.to('GetSendQuota');
    }
    /**
     * Grants permission to returns the user's sending statistics
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetSendStatistics.html
     */
    toGetSendStatistics() {
        return this.to('GetSendStatistics');
    }
    /**
     * Grants permission to return the template object, which includes the subject line, HTML par, and text part for the template you specify
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetTemplate.html
     */
    toGetTemplate() {
        return this.to('GetTemplate');
    }
    /**
     * Grants permission to list all of the configuration sets for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListConfigurationSets.html
     */
    toListConfigurationSets() {
        return this.to('ListConfigurationSets');
    }
    /**
     * Grants permission to list all of the existing custom verification email templates for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListCustomVerificationEmailTemplates.html
     */
    toListCustomVerificationEmailTemplates() {
        return this.to('ListCustomVerificationEmailTemplates');
    }
    /**
     * Grants permission to list the email identities for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListIdentities.html
     */
    toListIdentities() {
        return this.to('ListIdentities');
    }
    /**
     * Grants permission to list all of the email templates for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListIdentityPolicies.html
     */
    toListIdentityPolicies() {
        return this.to('ListIdentityPolicies');
    }
    /**
     * Grants permission to list the IP address filters associated with your account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListReceiptFilters.html
     */
    toListReceiptFilters() {
        return this.to('ListReceiptFilters');
    }
    /**
     * Grants permission to list the receipt rule sets that exist under your account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListReceiptRuleSets.html
     */
    toListReceiptRuleSets() {
        return this.to('ListReceiptRuleSets');
    }
    /**
     * Grants permission to list the email templates present in your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListTemplates.html
     */
    toListTemplates() {
        return this.to('ListTemplates');
    }
    /**
     * Grants permission to list all of the email addresses that have been verified in your account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListVerifiedEmailAddresses.html
     */
    toListVerifiedEmailAddresses() {
        return this.to('ListVerifiedEmailAddresses');
    }
    /**
     * Grants permission to add or update the delivery options for a configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_PutConfigurationSetDeliveryOptions.html
     */
    toPutConfigurationSetDeliveryOptions() {
        return this.to('PutConfigurationSetDeliveryOptions');
    }
    /**
     * Grants permission to add or update a sending authorization policy for the specified identity (an email address or a domain)
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_PutIdentityPolicy.html
     */
    toPutIdentityPolicy() {
        return this.to('PutIdentityPolicy');
    }
    /**
     * Grants permission to reorder the receipt rules within a receipt rule set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ReorderReceiptRuleSet.html
     */
    toReorderReceiptRuleSet() {
        return this.to('ReorderReceiptRuleSet');
    }
    /**
     * Grants permission to generate and send a bounce message to the sender of an email you received through Amazon SES
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifFromAddress()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendBounce.html
     */
    toSendBounce() {
        return this.to('SendBounce');
    }
    /**
     * Grants permission to compose an email message to multiple destinations
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifFeedbackAddress()
     * - .ifFromAddress()
     * - .ifFromDisplayName()
     * - .ifRecipients()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendBulkTemplatedEmail.html
     */
    toSendBulkTemplatedEmail() {
        return this.to('SendBulkTemplatedEmail');
    }
    /**
     * Grants permission to add an email address to the list of identities and attempts to verify it for your account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifFeedbackAddress()
     * - .ifFromAddress()
     * - .ifFromDisplayName()
     * - .ifRecipients()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendCustomVerificationEmail.html
     */
    toSendCustomVerificationEmail() {
        return this.to('SendCustomVerificationEmail');
    }
    /**
     * Grants permission to send an email message
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifFeedbackAddress()
     * - .ifFromAddress()
     * - .ifFromDisplayName()
     * - .ifRecipients()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendEmail.html
     */
    toSendEmail() {
        return this.to('SendEmail');
    }
    /**
     * Grants permission to send an email message, with header and content specified by the client
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifFeedbackAddress()
     * - .ifFromAddress()
     * - .ifFromDisplayName()
     * - .ifRecipients()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendRawEmail.html
     */
    toSendRawEmail() {
        return this.to('SendRawEmail');
    }
    /**
     * Grants permission to compose an email message using an email template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifFeedbackAddress()
     * - .ifFromAddress()
     * - .ifFromDisplayName()
     * - .ifRecipients()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendTemplatedEmail.html
     */
    toSendTemplatedEmail() {
        return this.to('SendTemplatedEmail');
    }
    /**
     * Grants permission to set the specified receipt rule set as the active receipt rule set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetActiveReceiptRuleSet.html
     */
    toSetActiveReceiptRuleSet() {
        return this.to('SetActiveReceiptRuleSet');
    }
    /**
     * Grants permission to enable or disable Easy DKIM signing of email sent from an identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityDkimEnabled.html
     */
    toSetIdentityDkimEnabled() {
        return this.to('SetIdentityDkimEnabled');
    }
    /**
     * Grants permission to enable or disable whether Amazon SES forwards bounce and complaint notifications for an identity (an email address or a domain)
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityFeedbackForwardingEnabled.html
     */
    toSetIdentityFeedbackForwardingEnabled() {
        return this.to('SetIdentityFeedbackForwardingEnabled');
    }
    /**
     * Grants permission to set whether Amazon SES includes the original email headers in the Amazon Simple Notification Service (Amazon SNS) notifications of a specified type for a given identity (an email address or a domain)
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityHeadersInNotificationsEnabled.html
     */
    toSetIdentityHeadersInNotificationsEnabled() {
        return this.to('SetIdentityHeadersInNotificationsEnabled');
    }
    /**
     * Grants permission to enable or disable the custom MAIL FROM domain setup for a verified identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityMailFromDomain.html
     */
    toSetIdentityMailFromDomain() {
        return this.to('SetIdentityMailFromDomain');
    }
    /**
     * Grants permission to set an Amazon Simple Notification Service (Amazon SNS) topic to use when delivering notifications for a verified identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityNotificationTopic.html
     */
    toSetIdentityNotificationTopic() {
        return this.to('SetIdentityNotificationTopic');
    }
    /**
     * Grants permission to set the position of the specified receipt rule in the receipt rule set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetReceiptRulePosition.html
     */
    toSetReceiptRulePosition() {
        return this.to('SetReceiptRulePosition');
    }
    /**
     * Grants permission to create a preview of the MIME content of an email when provided with a template and a set of replacement data
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_TestRenderTemplate.html
     */
    toTestRenderTemplate() {
        return this.to('TestRenderTemplate');
    }
    /**
     * Grants permission to enable or disable email sending for your account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateAccountSendingEnabled.html
     */
    toUpdateAccountSendingEnabled() {
        return this.to('UpdateAccountSendingEnabled');
    }
    /**
     * Grants permission to update the event destination of a configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetEventDestination.html
     */
    toUpdateConfigurationSetEventDestination() {
        return this.to('UpdateConfigurationSetEventDestination');
    }
    /**
     * Grants permission to enable or disable the publishing of reputation metrics for emails sent using a specific configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetReputationMetricsEnabled.html
     */
    toUpdateConfigurationSetReputationMetricsEnabled() {
        return this.to('UpdateConfigurationSetReputationMetricsEnabled');
    }
    /**
     * Grants permission to enable or disable email sending for messages sent using a specific configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetSendingEnabled.html
     */
    toUpdateConfigurationSetSendingEnabled() {
        return this.to('UpdateConfigurationSetSendingEnabled');
    }
    /**
     * Grants permission to modify an association between a configuration set and a custom domain for open and click event tracking
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetTrackingOptions.html
     */
    toUpdateConfigurationSetTrackingOptions() {
        return this.to('UpdateConfigurationSetTrackingOptions');
    }
    /**
     * Grants permission to update an existing custom verification email template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateCustomVerificationEmailTemplate.html
     */
    toUpdateCustomVerificationEmailTemplate() {
        return this.to('UpdateCustomVerificationEmailTemplate');
    }
    /**
     * Grants permission to update a receipt rule
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateReceiptRule.html
     */
    toUpdateReceiptRule() {
        return this.to('UpdateReceiptRule');
    }
    /**
     * Grants permission to update an email template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateTemplate.html
     */
    toUpdateTemplate() {
        return this.to('UpdateTemplate');
    }
    /**
     * Grants permission to return a set of DKIM tokens for a domain
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyDomainDkim.html
     */
    toVerifyDomainDkim() {
        return this.to('VerifyDomainDkim');
    }
    /**
     * Grants permission to verify a domain
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyDomainIdentity.html
     */
    toVerifyDomainIdentity() {
        return this.to('VerifyDomainIdentity');
    }
    /**
     * Grants permission to verify an email address
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyEmailAddress.html
     */
    toVerifyEmailAddress() {
        return this.to('VerifyEmailAddress');
    }
    /**
     * Grants permission to verify an email identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyEmailIdentity.html
     */
    toVerifyEmailIdentity() {
        return this.to('VerifyEmailIdentity');
    }
    /**
     * Adds a resource of type configuration-set to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ConfigurationSet.html
     *
     * @param configurationSetName - Identifier for the configurationSetName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onConfigurationSet(configurationSetName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:configuration-set/${configurationSetName}`);
    }
    /**
     * Adds a resource of type custom-verification-email-template to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CustomVerificationEmailTemplate.html
     *
     * @param templateName - Identifier for the templateName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onCustomVerificationEmailTemplate(templateName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:custom-verification-email-template/${templateName}`);
    }
    /**
     * Adds a resource of type identity to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_IdentityInfo.html
     *
     * @param identityName - Identifier for the identityName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onIdentity(identityName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:identity/${identityName}`);
    }
    /**
     * Adds a resource of type template to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_Template.html
     *
     * @param templateName - Identifier for the templateName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onTemplate(templateName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:template/${templateName}`);
    }
    /**
     * Filters actions based on the SES API version
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toCloneReceiptRuleSet()
     * - .toCreateConfigurationSet()
     * - .toCreateConfigurationSetEventDestination()
     * - .toCreateConfigurationSetTrackingOptions()
     * - .toCreateCustomVerificationEmailTemplate()
     * - .toCreateReceiptFilter()
     * - .toCreateReceiptRule()
     * - .toCreateReceiptRuleSet()
     * - .toCreateTemplate()
     * - .toDeleteConfigurationSet()
     * - .toDeleteConfigurationSetEventDestination()
     * - .toDeleteConfigurationSetTrackingOptions()
     * - .toDeleteCustomVerificationEmailTemplate()
     * - .toDeleteIdentity()
     * - .toDeleteIdentityPolicy()
     * - .toDeleteReceiptFilter()
     * - .toDeleteReceiptRule()
     * - .toDeleteReceiptRuleSet()
     * - .toDeleteTemplate()
     * - .toDeleteVerifiedEmailAddress()
     * - .toDescribeActiveReceiptRuleSet()
     * - .toDescribeConfigurationSet()
     * - .toDescribeReceiptRule()
     * - .toDescribeReceiptRuleSet()
     * - .toGetAccountSendingEnabled()
     * - .toGetCustomVerificationEmailTemplate()
     * - .toGetIdentityDkimAttributes()
     * - .toGetIdentityMailFromDomainAttributes()
     * - .toGetIdentityNotificationAttributes()
     * - .toGetIdentityPolicies()
     * - .toGetIdentityVerificationAttributes()
     * - .toGetSendQuota()
     * - .toGetSendStatistics()
     * - .toGetTemplate()
     * - .toListConfigurationSets()
     * - .toListCustomVerificationEmailTemplates()
     * - .toListIdentities()
     * - .toListIdentityPolicies()
     * - .toListReceiptFilters()
     * - .toListReceiptRuleSets()
     * - .toListTemplates()
     * - .toListVerifiedEmailAddresses()
     * - .toPutConfigurationSetDeliveryOptions()
     * - .toPutIdentityPolicy()
     * - .toReorderReceiptRuleSet()
     * - .toSendBounce()
     * - .toSendBulkTemplatedEmail()
     * - .toSendCustomVerificationEmail()
     * - .toSendEmail()
     * - .toSendRawEmail()
     * - .toSendTemplatedEmail()
     * - .toSetActiveReceiptRuleSet()
     * - .toSetIdentityDkimEnabled()
     * - .toSetIdentityFeedbackForwardingEnabled()
     * - .toSetIdentityHeadersInNotificationsEnabled()
     * - .toSetIdentityMailFromDomain()
     * - .toSetIdentityNotificationTopic()
     * - .toSetReceiptRulePosition()
     * - .toTestRenderTemplate()
     * - .toUpdateAccountSendingEnabled()
     * - .toUpdateConfigurationSetEventDestination()
     * - .toUpdateConfigurationSetReputationMetricsEnabled()
     * - .toUpdateConfigurationSetSendingEnabled()
     * - .toUpdateConfigurationSetTrackingOptions()
     * - .toUpdateCustomVerificationEmailTemplate()
     * - .toUpdateReceiptRule()
     * - .toUpdateTemplate()
     * - .toVerifyDomainDkim()
     * - .toVerifyDomainIdentity()
     * - .toVerifyEmailAddress()
     * - .toVerifyEmailIdentity()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifApiVersion(value, operator) {
        return this.if(`ApiVersion`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the "Return-Path" address, which specifies where bounces and complaints are sent by email feedback forwarding
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toSendBulkTemplatedEmail()
     * - .toSendCustomVerificationEmail()
     * - .toSendEmail()
     * - .toSendRawEmail()
     * - .toSendTemplatedEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFeedbackAddress(value, operator) {
        return this.if(`FeedbackAddress`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the "From" address of a message
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toSendBounce()
     * - .toSendBulkTemplatedEmail()
     * - .toSendCustomVerificationEmail()
     * - .toSendEmail()
     * - .toSendRawEmail()
     * - .toSendTemplatedEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFromAddress(value, operator) {
        return this.if(`FromAddress`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the "From" address that is used as the display name of a message
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toSendBulkTemplatedEmail()
     * - .toSendCustomVerificationEmail()
     * - .toSendEmail()
     * - .toSendRawEmail()
     * - .toSendTemplatedEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFromDisplayName(value, operator) {
        return this.if(`FromDisplayName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the recipient addresses of a message, which include the "To", "CC", and "BCC" addresses
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toSendBulkTemplatedEmail()
     * - .toSendCustomVerificationEmail()
     * - .toSendEmail()
     * - .toSendRawEmail()
     * - .toSendTemplatedEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRecipients(value, operator) {
        return this.if(`Recipients`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [ses](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonses.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'ses';
        this.accessLevelList = {
            Write: [
                'CloneReceiptRuleSet',
                'CreateConfigurationSet',
                'CreateConfigurationSetEventDestination',
                'CreateConfigurationSetTrackingOptions',
                'CreateCustomVerificationEmailTemplate',
                'CreateReceiptFilter',
                'CreateReceiptRule',
                'CreateReceiptRuleSet',
                'CreateTemplate',
                'DeleteConfigurationSet',
                'DeleteConfigurationSetEventDestination',
                'DeleteConfigurationSetTrackingOptions',
                'DeleteCustomVerificationEmailTemplate',
                'DeleteIdentity',
                'DeleteReceiptFilter',
                'DeleteReceiptRule',
                'DeleteReceiptRuleSet',
                'DeleteTemplate',
                'DeleteVerifiedEmailAddress',
                'PutConfigurationSetDeliveryOptions',
                'ReorderReceiptRuleSet',
                'SendBounce',
                'SendBulkTemplatedEmail',
                'SendCustomVerificationEmail',
                'SendEmail',
                'SendRawEmail',
                'SendTemplatedEmail',
                'SetActiveReceiptRuleSet',
                'SetIdentityDkimEnabled',
                'SetIdentityFeedbackForwardingEnabled',
                'SetIdentityHeadersInNotificationsEnabled',
                'SetIdentityMailFromDomain',
                'SetIdentityNotificationTopic',
                'SetReceiptRulePosition',
                'TestRenderTemplate',
                'UpdateAccountSendingEnabled',
                'UpdateConfigurationSetEventDestination',
                'UpdateConfigurationSetReputationMetricsEnabled',
                'UpdateConfigurationSetSendingEnabled',
                'UpdateConfigurationSetTrackingOptions',
                'UpdateCustomVerificationEmailTemplate',
                'UpdateReceiptRule',
                'UpdateTemplate',
                'VerifyDomainDkim',
                'VerifyDomainIdentity',
                'VerifyEmailAddress',
                'VerifyEmailIdentity'
            ],
            'Permissions management': [
                'DeleteIdentityPolicy',
                'PutIdentityPolicy'
            ],
            Read: [
                'DescribeActiveReceiptRuleSet',
                'DescribeConfigurationSet',
                'DescribeReceiptRule',
                'DescribeReceiptRuleSet',
                'GetAccountSendingEnabled',
                'GetCustomVerificationEmailTemplate',
                'GetIdentityDkimAttributes',
                'GetIdentityMailFromDomainAttributes',
                'GetIdentityNotificationAttributes',
                'GetIdentityPolicies',
                'GetIdentityVerificationAttributes',
                'GetSendQuota',
                'GetSendStatistics',
                'GetTemplate',
                'ListReceiptFilters',
                'ListReceiptRuleSets',
                'ListVerifiedEmailAddresses'
            ],
            List: [
                'ListConfigurationSets',
                'ListCustomVerificationEmailTemplates',
                'ListIdentities',
                'ListIdentityPolicies',
                'ListTemplates'
            ]
        };
    }
}
exports.Ses = Ses;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2VzLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsic2VzLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUNBLHlDQUF5RDtBQUd6RDs7OztHQUlHO0FBQ0gsTUFBYSxHQUFJLFNBQVEsd0JBQWU7SUFHdEM7Ozs7Ozs7OztPQVNHO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSx3QkFBd0I7UUFDN0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7SUFDM0MsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHdDQUF3QztRQUM3QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0NBQXdDLENBQUMsQ0FBQztJQUMzRCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksdUNBQXVDO1FBQzVDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1Q0FBdUMsQ0FBQyxDQUFDO0lBQzFELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSx1Q0FBdUM7UUFDNUMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVDQUF1QyxDQUFDLENBQUM7SUFDMUQsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksbUJBQW1CO1FBQ3hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxzQkFBc0I7UUFDM0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGdCQUFnQjtRQUNyQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksd0JBQXdCO1FBQzdCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSx3Q0FBd0M7UUFDN0MsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHdDQUF3QyxDQUFDLENBQUM7SUFDM0QsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHVDQUF1QztRQUM1QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsdUNBQXVDLENBQUMsQ0FBQztJQUMxRCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksdUNBQXVDO1FBQzVDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1Q0FBdUMsQ0FBQyxDQUFDO0lBQzFELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksZ0JBQWdCO1FBQ3JCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO0lBQ25DLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSw0QkFBNEI7UUFDakMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDRCQUE0QixDQUFDLENBQUM7SUFDL0MsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLDhCQUE4QjtRQUNuQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsOEJBQThCLENBQUMsQ0FBQztJQUNqRCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksMEJBQTBCO1FBQy9CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksMEJBQTBCO1FBQy9CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxvQ0FBb0M7UUFDekMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxDQUFDLENBQUM7SUFDdkQsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLDJCQUEyQjtRQUNoQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMkJBQTJCLENBQUMsQ0FBQztJQUM5QyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0kscUNBQXFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO0lBQ3hELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxtQ0FBbUM7UUFDeEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1DQUFtQyxDQUFDLENBQUM7SUFDdEQsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksbUNBQW1DO1FBQ3hDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQ0FBbUMsQ0FBQyxDQUFDO0lBQ3RELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxjQUFjO1FBQ25CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUNqQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksbUJBQW1CO1FBQ3hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxhQUFhO1FBQ2xCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUNoQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksdUJBQXVCO1FBQzVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0lBQzFDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxzQ0FBc0M7UUFDM0MsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHNDQUFzQyxDQUFDLENBQUM7SUFDekQsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGdCQUFnQjtRQUNyQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksc0JBQXNCO1FBQzNCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxvQkFBb0I7UUFDekIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLDRCQUE0QjtRQUNqQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNEJBQTRCLENBQUMsQ0FBQztJQUMvQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksb0NBQW9DO1FBQ3pDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQ0FBb0MsQ0FBQyxDQUFDO0lBQ3ZELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHVCQUF1QjtRQUM1QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLENBQUMsQ0FBQztJQUMxQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLFlBQVk7UUFDakIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFlBQVksQ0FBQyxDQUFDO0lBQy9CLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksd0JBQXdCO1FBQzdCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksNkJBQTZCO1FBQ2xDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO0lBQ2hELENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksV0FBVztRQUNoQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsV0FBVyxDQUFDLENBQUM7SUFDOUIsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxjQUFjO1FBQ25CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUNqQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLG9CQUFvQjtRQUN6QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0kseUJBQXlCO1FBQzlCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSx3QkFBd0I7UUFDN0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7SUFDM0MsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHNDQUFzQztRQUMzQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0NBQXNDLENBQUMsQ0FBQztJQUN6RCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksMENBQTBDO1FBQy9DLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywwQ0FBMEMsQ0FBQyxDQUFDO0lBQzdELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSwyQkFBMkI7UUFDaEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDJCQUEyQixDQUFDLENBQUM7SUFDOUMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLDhCQUE4QjtRQUNuQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsOEJBQThCLENBQUMsQ0FBQztJQUNqRCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksd0JBQXdCO1FBQzdCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxvQkFBb0I7UUFDekIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLDZCQUE2QjtRQUNsQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztJQUNoRCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksd0NBQXdDO1FBQzdDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3Q0FBd0MsQ0FBQyxDQUFDO0lBQzNELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxnREFBZ0Q7UUFDckQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdEQUFnRCxDQUFDLENBQUM7SUFDbkUsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHNDQUFzQztRQUMzQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0NBQXNDLENBQUMsQ0FBQztJQUN6RCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksdUNBQXVDO1FBQzVDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1Q0FBdUMsQ0FBQyxDQUFDO0lBQzFELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSx1Q0FBdUM7UUFDNUMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVDQUF1QyxDQUFDLENBQUM7SUFDMUQsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksZ0JBQWdCO1FBQ3JCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO0lBQ25DLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksb0JBQW9CO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQW9GRDs7Ozs7Ozs7O09BU0c7SUFDSSxrQkFBa0IsQ0FBQyxvQkFBNEIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUMzRyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixRQUFTLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxzQkFBdUIsb0JBQXFCLEVBQUUsQ0FBQyxDQUFDO0lBQ3RMLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxpQ0FBaUMsQ0FBQyxZQUFvQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ2xILE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLFFBQVMsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLHVDQUF3QyxZQUFhLEVBQUUsQ0FBQyxDQUFDO0lBQy9MLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxVQUFVLENBQUMsWUFBb0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUMzRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixRQUFTLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxhQUFjLFlBQWEsRUFBRSxDQUFDLENBQUM7SUFDckssQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLFVBQVUsQ0FBQyxZQUFvQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzNGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLFFBQVMsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLGFBQWMsWUFBYSxFQUFFLENBQUMsQ0FBQztJQUNySyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09BZ0ZHO0lBQ0ksWUFBWSxDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDeEUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFlBQVksRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2hFLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7T0FjRztJQUNJLGlCQUFpQixDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDN0UsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlCQUFpQixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDckUsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7T0FlRztJQUNJLGFBQWEsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3pFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSxpQkFBaUIsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQzdFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQkFBaUIsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ3JFLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7T0FjRztJQUNJLFlBQVksQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3hFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxZQUFZLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNoRSxDQUFDO0lBRUQ7OztPQUdHO0lBQ0gsWUFBWSxLQUFnQztRQUMxQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7UUExeUNSLGtCQUFhLEdBQUcsS0FBSyxDQUFDO1FBeS9CbkIsb0JBQWUsR0FBb0I7WUFDM0MsS0FBSyxFQUFFO2dCQUNMLHFCQUFxQjtnQkFDckIsd0JBQXdCO2dCQUN4Qix3Q0FBd0M7Z0JBQ3hDLHVDQUF1QztnQkFDdkMsdUNBQXVDO2dCQUN2QyxxQkFBcUI7Z0JBQ3JCLG1CQUFtQjtnQkFDbkIsc0JBQXNCO2dCQUN0QixnQkFBZ0I7Z0JBQ2hCLHdCQUF3QjtnQkFDeEIsd0NBQXdDO2dCQUN4Qyx1Q0FBdUM7Z0JBQ3ZDLHVDQUF1QztnQkFDdkMsZ0JBQWdCO2dCQUNoQixxQkFBcUI7Z0JBQ3JCLG1CQUFtQjtnQkFDbkIsc0JBQXNCO2dCQUN0QixnQkFBZ0I7Z0JBQ2hCLDRCQUE0QjtnQkFDNUIsb0NBQW9DO2dCQUNwQyx1QkFBdUI7Z0JBQ3ZCLFlBQVk7Z0JBQ1osd0JBQXdCO2dCQUN4Qiw2QkFBNkI7Z0JBQzdCLFdBQVc7Z0JBQ1gsY0FBYztnQkFDZCxvQkFBb0I7Z0JBQ3BCLHlCQUF5QjtnQkFDekIsd0JBQXdCO2dCQUN4QixzQ0FBc0M7Z0JBQ3RDLDBDQUEwQztnQkFDMUMsMkJBQTJCO2dCQUMzQiw4QkFBOEI7Z0JBQzlCLHdCQUF3QjtnQkFDeEIsb0JBQW9CO2dCQUNwQiw2QkFBNkI7Z0JBQzdCLHdDQUF3QztnQkFDeEMsZ0RBQWdEO2dCQUNoRCxzQ0FBc0M7Z0JBQ3RDLHVDQUF1QztnQkFDdkMsdUNBQXVDO2dCQUN2QyxtQkFBbUI7Z0JBQ25CLGdCQUFnQjtnQkFDaEIsa0JBQWtCO2dCQUNsQixzQkFBc0I7Z0JBQ3RCLG9CQUFvQjtnQkFDcEIscUJBQXFCO2FBQ3RCO1lBQ0Qsd0JBQXdCLEVBQUU7Z0JBQ3hCLHNCQUFzQjtnQkFDdEIsbUJBQW1CO2FBQ3BCO1lBQ0QsSUFBSSxFQUFFO2dCQUNKLDhCQUE4QjtnQkFDOUIsMEJBQTBCO2dCQUMxQixxQkFBcUI7Z0JBQ3JCLHdCQUF3QjtnQkFDeEIsMEJBQTBCO2dCQUMxQixvQ0FBb0M7Z0JBQ3BDLDJCQUEyQjtnQkFDM0IscUNBQXFDO2dCQUNyQyxtQ0FBbUM7Z0JBQ25DLHFCQUFxQjtnQkFDckIsbUNBQW1DO2dCQUNuQyxjQUFjO2dCQUNkLG1CQUFtQjtnQkFDbkIsYUFBYTtnQkFDYixvQkFBb0I7Z0JBQ3BCLHFCQUFxQjtnQkFDckIsNEJBQTRCO2FBQzdCO1lBQ0QsSUFBSSxFQUFFO2dCQUNKLHVCQUF1QjtnQkFDdkIsc0NBQXNDO2dCQUN0QyxnQkFBZ0I7Z0JBQ2hCLHNCQUFzQjtnQkFDdEIsZUFBZTthQUNoQjtTQUNGLENBQUM7SUFrT0YsQ0FBQztDQUNGO0FBN3lDRCxrQkE2eUNDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQWNjZXNzTGV2ZWxMaXN0IH0gZnJvbSAnLi4vLi4vc2hhcmVkL2FjY2Vzcy1sZXZlbCc7XG5pbXBvcnQgeyBQb2xpY3lTdGF0ZW1lbnQsIE9wZXJhdG9yIH0gZnJvbSAnLi4vLi4vc2hhcmVkJztcbmltcG9ydCB7IGF3c19pYW0gYXMgaWFtIH0gZnJvbSBcImF3cy1jZGstbGliXCI7XG5cbi8qKlxuICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtzZXNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlLWF1dGhvcml6YXRpb24vbGF0ZXN0L3JlZmVyZW5jZS9saXN0X2FtYXpvbnNlcy5odG1sKS5cbiAqXG4gKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAqL1xuZXhwb3J0IGNsYXNzIFNlcyBleHRlbmRzIFBvbGljeVN0YXRlbWVudCB7XG4gIHB1YmxpYyBzZXJ2aWNlUHJlZml4ID0gJ3Nlcyc7XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIHJlY2VpcHQgcnVsZSBzZXQgYnkgY2xvbmluZyBhbiBleGlzdGluZyBvbmVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkFwaVZlcnNpb24oKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Nsb25lUmVjZWlwdFJ1bGVTZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ2xvbmVSZWNlaXB0UnVsZVNldCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ2xvbmVSZWNlaXB0UnVsZVNldCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIG5ldyBjb25maWd1cmF0aW9uIHNldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXBpVmVyc2lvbigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlQ29uZmlndXJhdGlvblNldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVDb25maWd1cmF0aW9uU2V0KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVDb25maWd1cmF0aW9uU2V0Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgY29uZmlndXJhdGlvbiBzZXQgZXZlbnQgZGVzdGluYXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkFwaVZlcnNpb24oKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZUNvbmZpZ3VyYXRpb25TZXRFdmVudERlc3RpbmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUNvbmZpZ3VyYXRpb25TZXRFdmVudERlc3RpbmF0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVDb25maWd1cmF0aW9uU2V0RXZlbnREZXN0aW5hdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZXMgYW4gYXNzb2NpYXRpb24gYmV0d2VlbiBhIGNvbmZpZ3VyYXRpb24gc2V0IGFuZCBhIGN1c3RvbSBkb21haW4gZm9yIG9wZW4gYW5kIGNsaWNrIGV2ZW50IHRyYWNraW5nXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBcGlWZXJzaW9uKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3Nlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVDb25maWd1cmF0aW9uU2V0VHJhY2tpbmdPcHRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUNvbmZpZ3VyYXRpb25TZXRUcmFja2luZ09wdGlvbnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZUNvbmZpZ3VyYXRpb25TZXRUcmFja2luZ09wdGlvbnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBuZXcgY3VzdG9tIHZlcmlmaWNhdGlvbiBlbWFpbCB0ZW1wbGF0ZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXBpVmVyc2lvbigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlQ3VzdG9tVmVyaWZpY2F0aW9uRW1haWxUZW1wbGF0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVDdXN0b21WZXJpZmljYXRpb25FbWFpbFRlbXBsYXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVDdXN0b21WZXJpZmljYXRpb25FbWFpbFRlbXBsYXRlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgbmV3IElQIGFkZHJlc3MgZmlsdGVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBcGlWZXJzaW9uKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3Nlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVSZWNlaXB0RmlsdGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVJlY2VpcHRGaWx0ZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZVJlY2VpcHRGaWx0ZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSByZWNlaXB0IHJ1bGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkFwaVZlcnNpb24oKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZVJlY2VpcHRSdWxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVJlY2VpcHRSdWxlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVSZWNlaXB0UnVsZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhbiBlbXB0eSByZWNlaXB0IHJ1bGUgc2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBcGlWZXJzaW9uKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3Nlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVSZWNlaXB0UnVsZVNldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVSZWNlaXB0UnVsZVNldCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlUmVjZWlwdFJ1bGVTZXQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGVzIGFuIGVtYWlsIHRlbXBsYXRlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBcGlWZXJzaW9uKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3Nlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVUZW1wbGF0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVUZW1wbGF0ZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlVGVtcGxhdGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYW4gZXhpc3RpbmcgY29uZmlndXJhdGlvbiBzZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkFwaVZlcnNpb24oKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUNvbmZpZ3VyYXRpb25TZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlQ29uZmlndXJhdGlvblNldCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlQ29uZmlndXJhdGlvblNldCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhbiBldmVudCBkZXN0aW5hdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXBpVmVyc2lvbigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlQ29uZmlndXJhdGlvblNldEV2ZW50RGVzdGluYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlQ29uZmlndXJhdGlvblNldEV2ZW50RGVzdGluYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZUNvbmZpZ3VyYXRpb25TZXRFdmVudERlc3RpbmF0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGFuIGFzc29jaWF0aW9uIGJldHdlZW4gYSBjb25maWd1cmF0aW9uIHNldCBhbmQgYSBjdXN0b20gZG9tYWluIGZvciBvcGVuIGFuZCBjbGljayBldmVudCB0cmFja2luZ1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXBpVmVyc2lvbigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlQ29uZmlndXJhdGlvblNldFRyYWNraW5nT3B0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVDb25maWd1cmF0aW9uU2V0VHJhY2tpbmdPcHRpb25zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVDb25maWd1cmF0aW9uU2V0VHJhY2tpbmdPcHRpb25zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGFuIGV4aXN0aW5nIGN1c3RvbSB2ZXJpZmljYXRpb24gZW1haWwgdGVtcGxhdGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkFwaVZlcnNpb24oKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUN1c3RvbVZlcmlmaWNhdGlvbkVtYWlsVGVtcGxhdGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlQ3VzdG9tVmVyaWZpY2F0aW9uRW1haWxUZW1wbGF0ZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlQ3VzdG9tVmVyaWZpY2F0aW9uRW1haWxUZW1wbGF0ZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSB0aGUgc3BlY2lmaWVkIGlkZW50aXR5XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBcGlWZXJzaW9uKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3Nlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVJZGVudGl0eS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVJZGVudGl0eSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlSWRlbnRpdHknKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgdGhlIHNwZWNpZmllZCBzZW5kaW5nIGF1dGhvcml6YXRpb24gcG9saWN5IGZvciB0aGUgZ2l2ZW4gaWRlbnRpdHkgKGFuIGVtYWlsIGFkZHJlc3Mgb3IgYSBkb21haW4pXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUGVybWlzc2lvbnMgbWFuYWdlbWVudFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkFwaVZlcnNpb24oKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUlkZW50aXR5UG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUlkZW50aXR5UG9saWN5KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVJZGVudGl0eVBvbGljeScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSB0aGUgc3BlY2lmaWVkIElQIGFkZHJlc3MgZmlsdGVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBcGlWZXJzaW9uKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3Nlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVSZWNlaXB0RmlsdGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVJlY2VpcHRGaWx0ZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZVJlY2VpcHRGaWx0ZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgdGhlIHNwZWNpZmllZCByZWNlaXB0IHJ1bGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkFwaVZlcnNpb24oKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZVJlY2VpcHRSdWxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVJlY2VpcHRSdWxlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVSZWNlaXB0UnVsZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSB0aGUgc3BlY2lmaWVkIHJlY2VpcHQgcnVsZSBzZXQgYW5kIGFsbCBvZiB0aGUgcmVjZWlwdCBydWxlcyBpdCBjb250YWluc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXBpVmVyc2lvbigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlUmVjZWlwdFJ1bGVTZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlUmVjZWlwdFJ1bGVTZXQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZVJlY2VpcHRSdWxlU2V0Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGFuIGVtYWlsIHRlbXBsYXRlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBcGlWZXJzaW9uKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3Nlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVUZW1wbGF0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVUZW1wbGF0ZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlVGVtcGxhdGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgdGhlIHNwZWNpZmllZCBlbWFpbCBhZGRyZXNzIGZyb20gdGhlIGxpc3Qgb2YgdmVyaWZpZWQgYWRkcmVzc2VzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBcGlWZXJzaW9uKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3Nlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVWZXJpZmllZEVtYWlsQWRkcmVzcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVWZXJpZmllZEVtYWlsQWRkcmVzcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlVmVyaWZpZWRFbWFpbEFkZHJlc3MnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gdGhlIG1ldGFkYXRhIGFuZCByZWNlaXB0IHJ1bGVzIGZvciB0aGUgcmVjZWlwdCBydWxlIHNldCB0aGF0IGlzIGN1cnJlbnRseSBhY3RpdmVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXBpVmVyc2lvbigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVBY3RpdmVSZWNlaXB0UnVsZVNldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUFjdGl2ZVJlY2VpcHRSdWxlU2V0KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZUFjdGl2ZVJlY2VpcHRSdWxlU2V0Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIHRoZSBkZXRhaWxzIG9mIHRoZSBzcGVjaWZpZWQgY29uZmlndXJhdGlvbiBzZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXBpVmVyc2lvbigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVDb25maWd1cmF0aW9uU2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlQ29uZmlndXJhdGlvblNldCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVDb25maWd1cmF0aW9uU2V0Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIHRoZSBkZXRhaWxzIG9mIHRoZSBzcGVjaWZpZWQgcmVjZWlwdCBydWxlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkFwaVZlcnNpb24oKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlUmVjZWlwdFJ1bGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVSZWNlaXB0UnVsZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVSZWNlaXB0UnVsZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHVybiB0aGUgZGV0YWlscyBvZiB0aGUgc3BlY2lmaWVkIHJlY2VpcHQgcnVsZSBzZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXBpVmVyc2lvbigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVSZWNlaXB0UnVsZVNldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZVJlY2VpcHRSdWxlU2V0KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZVJlY2VpcHRSdWxlU2V0Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIHRoZSBlbWFpbCBzZW5kaW5nIHN0YXR1cyBvZiB5b3VyIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXBpVmVyc2lvbigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0QWNjb3VudFNlbmRpbmdFbmFibGVkLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEFjY291bnRTZW5kaW5nRW5hYmxlZCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0QWNjb3VudFNlbmRpbmdFbmFibGVkJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIHRoZSBjdXN0b20gZW1haWwgdmVyaWZpY2F0aW9uIHRlbXBsYXRlIGZvciB0aGUgdGVtcGxhdGUgbmFtZSB5b3Ugc3BlY2lmeVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBcGlWZXJzaW9uKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3Nlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRDdXN0b21WZXJpZmljYXRpb25FbWFpbFRlbXBsYXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEN1c3RvbVZlcmlmaWNhdGlvbkVtYWlsVGVtcGxhdGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldEN1c3RvbVZlcmlmaWNhdGlvbkVtYWlsVGVtcGxhdGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gdGhlIGN1cnJlbnQgc3RhdHVzIG9mIEVhc3kgREtJTSBzaWduaW5nIGZvciBhbiBlbnRpdHlcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXBpVmVyc2lvbigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0SWRlbnRpdHlEa2ltQXR0cmlidXRlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRJZGVudGl0eURraW1BdHRyaWJ1dGVzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRJZGVudGl0eURraW1BdHRyaWJ1dGVzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIHRoZSBjdXN0b20gTUFJTCBGUk9NIGF0dHJpYnV0ZXMgZm9yIGEgbGlzdCBvZiBpZGVudGl0aWVzIChlbWFpbCBhZGRyZXNzZXMgYW5kL29yIGRvbWFpbnMpXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkFwaVZlcnNpb24oKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldElkZW50aXR5TWFpbEZyb21Eb21haW5BdHRyaWJ1dGVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldElkZW50aXR5TWFpbEZyb21Eb21haW5BdHRyaWJ1dGVzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRJZGVudGl0eU1haWxGcm9tRG9tYWluQXR0cmlidXRlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHVybiBhIHN0cnVjdHVyZSBkZXNjcmliaW5nIGlkZW50aXR5IG5vdGlmaWNhdGlvbiBhdHRyaWJ1dGVzIGZvciBhIGxpc3Qgb2YgdmVyaWZpZWQgaWRlbnRpdGllcyAoZW1haWwgYWRkcmVzc2VzIGFuZC9vciBkb21haW5zKSxcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXBpVmVyc2lvbigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0SWRlbnRpdHlOb3RpZmljYXRpb25BdHRyaWJ1dGVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldElkZW50aXR5Tm90aWZpY2F0aW9uQXR0cmlidXRlcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0SWRlbnRpdHlOb3RpZmljYXRpb25BdHRyaWJ1dGVzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIHRoZSByZXF1ZXN0ZWQgc2VuZGluZyBhdXRob3JpemF0aW9uIHBvbGljaWVzIGZvciB0aGUgZ2l2ZW4gaWRlbnRpdHkgKGFuIGVtYWlsIGFkZHJlc3Mgb3IgYSBkb21haW4pXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkFwaVZlcnNpb24oKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldElkZW50aXR5UG9saWNpZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0SWRlbnRpdHlQb2xpY2llcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0SWRlbnRpdHlQb2xpY2llcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHVybiB0aGUgdmVyaWZpY2F0aW9uIHN0YXR1cyBhbmQgKGZvciBkb21haW4gaWRlbnRpdGllcykgdGhlIHZlcmlmaWNhdGlvbiB0b2tlbiBmb3IgYSBsaXN0IG9mIGlkZW50aXRpZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXBpVmVyc2lvbigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0SWRlbnRpdHlWZXJpZmljYXRpb25BdHRyaWJ1dGVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldElkZW50aXR5VmVyaWZpY2F0aW9uQXR0cmlidXRlcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0SWRlbnRpdHlWZXJpZmljYXRpb25BdHRyaWJ1dGVzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIHRoZSB1c2VyJ3MgY3VycmVudCBzZW5kaW5nIGxpbWl0c1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBcGlWZXJzaW9uKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3Nlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRTZW5kUXVvdGEuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0U2VuZFF1b3RhKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRTZW5kUXVvdGEnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm5zIHRoZSB1c2VyJ3Mgc2VuZGluZyBzdGF0aXN0aWNzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkFwaVZlcnNpb24oKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldFNlbmRTdGF0aXN0aWNzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFNlbmRTdGF0aXN0aWNzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRTZW5kU3RhdGlzdGljcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHVybiB0aGUgdGVtcGxhdGUgb2JqZWN0LCB3aGljaCBpbmNsdWRlcyB0aGUgc3ViamVjdCBsaW5lLCBIVE1MIHBhciwgYW5kIHRleHQgcGFydCBmb3IgdGhlIHRlbXBsYXRlIHlvdSBzcGVjaWZ5XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkFwaVZlcnNpb24oKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldFRlbXBsYXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFRlbXBsYXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRUZW1wbGF0ZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgYWxsIG9mIHRoZSBjb25maWd1cmF0aW9uIHNldHMgZm9yIHlvdXIgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBcGlWZXJzaW9uKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3Nlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0Q29uZmlndXJhdGlvblNldHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdENvbmZpZ3VyYXRpb25TZXRzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0Q29uZmlndXJhdGlvblNldHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGFsbCBvZiB0aGUgZXhpc3RpbmcgY3VzdG9tIHZlcmlmaWNhdGlvbiBlbWFpbCB0ZW1wbGF0ZXMgZm9yIHlvdXIgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBcGlWZXJzaW9uKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3Nlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0Q3VzdG9tVmVyaWZpY2F0aW9uRW1haWxUZW1wbGF0ZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEN1c3RvbVZlcmlmaWNhdGlvbkVtYWlsVGVtcGxhdGVzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0Q3VzdG9tVmVyaWZpY2F0aW9uRW1haWxUZW1wbGF0ZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHRoZSBlbWFpbCBpZGVudGl0aWVzIGZvciB5b3VyIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXBpVmVyc2lvbigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdElkZW50aXRpZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdElkZW50aXRpZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RJZGVudGl0aWVzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBhbGwgb2YgdGhlIGVtYWlsIHRlbXBsYXRlcyBmb3IgeW91ciBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkFwaVZlcnNpb24oKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RJZGVudGl0eVBvbGljaWVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RJZGVudGl0eVBvbGljaWVzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0SWRlbnRpdHlQb2xpY2llcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGhlIElQIGFkZHJlc3MgZmlsdGVycyBhc3NvY2lhdGVkIHdpdGggeW91ciBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkFwaVZlcnNpb24oKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RSZWNlaXB0RmlsdGVycy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0UmVjZWlwdEZpbHRlcnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RSZWNlaXB0RmlsdGVycycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGhlIHJlY2VpcHQgcnVsZSBzZXRzIHRoYXQgZXhpc3QgdW5kZXIgeW91ciBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkFwaVZlcnNpb24oKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RSZWNlaXB0UnVsZVNldHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFJlY2VpcHRSdWxlU2V0cygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFJlY2VpcHRSdWxlU2V0cycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGhlIGVtYWlsIHRlbXBsYXRlcyBwcmVzZW50IGluIHlvdXIgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBcGlWZXJzaW9uKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3Nlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0VGVtcGxhdGVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RUZW1wbGF0ZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RUZW1wbGF0ZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGFsbCBvZiB0aGUgZW1haWwgYWRkcmVzc2VzIHRoYXQgaGF2ZSBiZWVuIHZlcmlmaWVkIGluIHlvdXIgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBcGlWZXJzaW9uKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3Nlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0VmVyaWZpZWRFbWFpbEFkZHJlc3Nlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0VmVyaWZpZWRFbWFpbEFkZHJlc3NlcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFZlcmlmaWVkRW1haWxBZGRyZXNzZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBhZGQgb3IgdXBkYXRlIHRoZSBkZWxpdmVyeSBvcHRpb25zIGZvciBhIGNvbmZpZ3VyYXRpb24gc2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBcGlWZXJzaW9uKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3Nlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9QdXRDb25maWd1cmF0aW9uU2V0RGVsaXZlcnlPcHRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1B1dENvbmZpZ3VyYXRpb25TZXREZWxpdmVyeU9wdGlvbnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1B1dENvbmZpZ3VyYXRpb25TZXREZWxpdmVyeU9wdGlvbnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBhZGQgb3IgdXBkYXRlIGEgc2VuZGluZyBhdXRob3JpemF0aW9uIHBvbGljeSBmb3IgdGhlIHNwZWNpZmllZCBpZGVudGl0eSAoYW4gZW1haWwgYWRkcmVzcyBvciBhIGRvbWFpbilcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXBpVmVyc2lvbigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUHV0SWRlbnRpdHlQb2xpY3kuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUHV0SWRlbnRpdHlQb2xpY3koKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1B1dElkZW50aXR5UG9saWN5Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVvcmRlciB0aGUgcmVjZWlwdCBydWxlcyB3aXRoaW4gYSByZWNlaXB0IHJ1bGUgc2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBcGlWZXJzaW9uKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3Nlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9SZW9yZGVyUmVjZWlwdFJ1bGVTZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUmVvcmRlclJlY2VpcHRSdWxlU2V0KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdSZW9yZGVyUmVjZWlwdFJ1bGVTZXQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZW5lcmF0ZSBhbmQgc2VuZCBhIGJvdW5jZSBtZXNzYWdlIHRvIHRoZSBzZW5kZXIgb2YgYW4gZW1haWwgeW91IHJlY2VpdmVkIHRocm91Z2ggQW1hem9uIFNFU1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXBpVmVyc2lvbigpXG4gICAqIC0gLmlmRnJvbUFkZHJlc3MoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1NlbmRCb3VuY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU2VuZEJvdW5jZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU2VuZEJvdW5jZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNvbXBvc2UgYW4gZW1haWwgbWVzc2FnZSB0byBtdWx0aXBsZSBkZXN0aW5hdGlvbnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkFwaVZlcnNpb24oKVxuICAgKiAtIC5pZkZlZWRiYWNrQWRkcmVzcygpXG4gICAqIC0gLmlmRnJvbUFkZHJlc3MoKVxuICAgKiAtIC5pZkZyb21EaXNwbGF5TmFtZSgpXG4gICAqIC0gLmlmUmVjaXBpZW50cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU2VuZEJ1bGtUZW1wbGF0ZWRFbWFpbC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TZW5kQnVsa1RlbXBsYXRlZEVtYWlsKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTZW5kQnVsa1RlbXBsYXRlZEVtYWlsJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYWRkIGFuIGVtYWlsIGFkZHJlc3MgdG8gdGhlIGxpc3Qgb2YgaWRlbnRpdGllcyBhbmQgYXR0ZW1wdHMgdG8gdmVyaWZ5IGl0IGZvciB5b3VyIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkFwaVZlcnNpb24oKVxuICAgKiAtIC5pZkZlZWRiYWNrQWRkcmVzcygpXG4gICAqIC0gLmlmRnJvbUFkZHJlc3MoKVxuICAgKiAtIC5pZkZyb21EaXNwbGF5TmFtZSgpXG4gICAqIC0gLmlmUmVjaXBpZW50cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU2VuZEN1c3RvbVZlcmlmaWNhdGlvbkVtYWlsLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1NlbmRDdXN0b21WZXJpZmljYXRpb25FbWFpbCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU2VuZEN1c3RvbVZlcmlmaWNhdGlvbkVtYWlsJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc2VuZCBhbiBlbWFpbCBtZXNzYWdlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBcGlWZXJzaW9uKClcbiAgICogLSAuaWZGZWVkYmFja0FkZHJlc3MoKVxuICAgKiAtIC5pZkZyb21BZGRyZXNzKClcbiAgICogLSAuaWZGcm9tRGlzcGxheU5hbWUoKVxuICAgKiAtIC5pZlJlY2lwaWVudHMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1NlbmRFbWFpbC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TZW5kRW1haWwoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1NlbmRFbWFpbCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHNlbmQgYW4gZW1haWwgbWVzc2FnZSwgd2l0aCBoZWFkZXIgYW5kIGNvbnRlbnQgc3BlY2lmaWVkIGJ5IHRoZSBjbGllbnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkFwaVZlcnNpb24oKVxuICAgKiAtIC5pZkZlZWRiYWNrQWRkcmVzcygpXG4gICAqIC0gLmlmRnJvbUFkZHJlc3MoKVxuICAgKiAtIC5pZkZyb21EaXNwbGF5TmFtZSgpXG4gICAqIC0gLmlmUmVjaXBpZW50cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU2VuZFJhd0VtYWlsLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1NlbmRSYXdFbWFpbCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU2VuZFJhd0VtYWlsJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY29tcG9zZSBhbiBlbWFpbCBtZXNzYWdlIHVzaW5nIGFuIGVtYWlsIHRlbXBsYXRlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBcGlWZXJzaW9uKClcbiAgICogLSAuaWZGZWVkYmFja0FkZHJlc3MoKVxuICAgKiAtIC5pZkZyb21BZGRyZXNzKClcbiAgICogLSAuaWZGcm9tRGlzcGxheU5hbWUoKVxuICAgKiAtIC5pZlJlY2lwaWVudHMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1NlbmRUZW1wbGF0ZWRFbWFpbC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TZW5kVGVtcGxhdGVkRW1haWwoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1NlbmRUZW1wbGF0ZWRFbWFpbCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHNldCB0aGUgc3BlY2lmaWVkIHJlY2VpcHQgcnVsZSBzZXQgYXMgdGhlIGFjdGl2ZSByZWNlaXB0IHJ1bGUgc2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBcGlWZXJzaW9uKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3Nlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TZXRBY3RpdmVSZWNlaXB0UnVsZVNldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TZXRBY3RpdmVSZWNlaXB0UnVsZVNldCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU2V0QWN0aXZlUmVjZWlwdFJ1bGVTZXQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBlbmFibGUgb3IgZGlzYWJsZSBFYXN5IERLSU0gc2lnbmluZyBvZiBlbWFpbCBzZW50IGZyb20gYW4gaWRlbnRpdHlcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkFwaVZlcnNpb24oKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1NldElkZW50aXR5RGtpbUVuYWJsZWQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU2V0SWRlbnRpdHlEa2ltRW5hYmxlZCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU2V0SWRlbnRpdHlEa2ltRW5hYmxlZCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGVuYWJsZSBvciBkaXNhYmxlIHdoZXRoZXIgQW1hem9uIFNFUyBmb3J3YXJkcyBib3VuY2UgYW5kIGNvbXBsYWludCBub3RpZmljYXRpb25zIGZvciBhbiBpZGVudGl0eSAoYW4gZW1haWwgYWRkcmVzcyBvciBhIGRvbWFpbilcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkFwaVZlcnNpb24oKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1NldElkZW50aXR5RmVlZGJhY2tGb3J3YXJkaW5nRW5hYmxlZC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TZXRJZGVudGl0eUZlZWRiYWNrRm9yd2FyZGluZ0VuYWJsZWQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1NldElkZW50aXR5RmVlZGJhY2tGb3J3YXJkaW5nRW5hYmxlZCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHNldCB3aGV0aGVyIEFtYXpvbiBTRVMgaW5jbHVkZXMgdGhlIG9yaWdpbmFsIGVtYWlsIGhlYWRlcnMgaW4gdGhlIEFtYXpvbiBTaW1wbGUgTm90aWZpY2F0aW9uIFNlcnZpY2UgKEFtYXpvbiBTTlMpIG5vdGlmaWNhdGlvbnMgb2YgYSBzcGVjaWZpZWQgdHlwZSBmb3IgYSBnaXZlbiBpZGVudGl0eSAoYW4gZW1haWwgYWRkcmVzcyBvciBhIGRvbWFpbilcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkFwaVZlcnNpb24oKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1NldElkZW50aXR5SGVhZGVyc0luTm90aWZpY2F0aW9uc0VuYWJsZWQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU2V0SWRlbnRpdHlIZWFkZXJzSW5Ob3RpZmljYXRpb25zRW5hYmxlZCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU2V0SWRlbnRpdHlIZWFkZXJzSW5Ob3RpZmljYXRpb25zRW5hYmxlZCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGVuYWJsZSBvciBkaXNhYmxlIHRoZSBjdXN0b20gTUFJTCBGUk9NIGRvbWFpbiBzZXR1cCBmb3IgYSB2ZXJpZmllZCBpZGVudGl0eVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXBpVmVyc2lvbigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU2V0SWRlbnRpdHlNYWlsRnJvbURvbWFpbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TZXRJZGVudGl0eU1haWxGcm9tRG9tYWluKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTZXRJZGVudGl0eU1haWxGcm9tRG9tYWluJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc2V0IGFuIEFtYXpvbiBTaW1wbGUgTm90aWZpY2F0aW9uIFNlcnZpY2UgKEFtYXpvbiBTTlMpIHRvcGljIHRvIHVzZSB3aGVuIGRlbGl2ZXJpbmcgbm90aWZpY2F0aW9ucyBmb3IgYSB2ZXJpZmllZCBpZGVudGl0eVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXBpVmVyc2lvbigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU2V0SWRlbnRpdHlOb3RpZmljYXRpb25Ub3BpYy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TZXRJZGVudGl0eU5vdGlmaWNhdGlvblRvcGljKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTZXRJZGVudGl0eU5vdGlmaWNhdGlvblRvcGljJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc2V0IHRoZSBwb3NpdGlvbiBvZiB0aGUgc3BlY2lmaWVkIHJlY2VpcHQgcnVsZSBpbiB0aGUgcmVjZWlwdCBydWxlIHNldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXBpVmVyc2lvbigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU2V0UmVjZWlwdFJ1bGVQb3NpdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TZXRSZWNlaXB0UnVsZVBvc2l0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTZXRSZWNlaXB0UnVsZVBvc2l0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgcHJldmlldyBvZiB0aGUgTUlNRSBjb250ZW50IG9mIGFuIGVtYWlsIHdoZW4gcHJvdmlkZWQgd2l0aCBhIHRlbXBsYXRlIGFuZCBhIHNldCBvZiByZXBsYWNlbWVudCBkYXRhXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBcGlWZXJzaW9uKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3Nlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9UZXN0UmVuZGVyVGVtcGxhdGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVGVzdFJlbmRlclRlbXBsYXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdUZXN0UmVuZGVyVGVtcGxhdGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBlbmFibGUgb3IgZGlzYWJsZSBlbWFpbCBzZW5kaW5nIGZvciB5b3VyIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkFwaVZlcnNpb24oKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZUFjY291bnRTZW5kaW5nRW5hYmxlZC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVBY2NvdW50U2VuZGluZ0VuYWJsZWQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUFjY291bnRTZW5kaW5nRW5hYmxlZCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSB0aGUgZXZlbnQgZGVzdGluYXRpb24gb2YgYSBjb25maWd1cmF0aW9uIHNldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXBpVmVyc2lvbigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlQ29uZmlndXJhdGlvblNldEV2ZW50RGVzdGluYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlQ29uZmlndXJhdGlvblNldEV2ZW50RGVzdGluYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUNvbmZpZ3VyYXRpb25TZXRFdmVudERlc3RpbmF0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZW5hYmxlIG9yIGRpc2FibGUgdGhlIHB1Ymxpc2hpbmcgb2YgcmVwdXRhdGlvbiBtZXRyaWNzIGZvciBlbWFpbHMgc2VudCB1c2luZyBhIHNwZWNpZmljIGNvbmZpZ3VyYXRpb24gc2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBcGlWZXJzaW9uKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3Nlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVDb25maWd1cmF0aW9uU2V0UmVwdXRhdGlvbk1ldHJpY3NFbmFibGVkLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUNvbmZpZ3VyYXRpb25TZXRSZXB1dGF0aW9uTWV0cmljc0VuYWJsZWQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUNvbmZpZ3VyYXRpb25TZXRSZXB1dGF0aW9uTWV0cmljc0VuYWJsZWQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBlbmFibGUgb3IgZGlzYWJsZSBlbWFpbCBzZW5kaW5nIGZvciBtZXNzYWdlcyBzZW50IHVzaW5nIGEgc3BlY2lmaWMgY29uZmlndXJhdGlvbiBzZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkFwaVZlcnNpb24oKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZUNvbmZpZ3VyYXRpb25TZXRTZW5kaW5nRW5hYmxlZC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVDb25maWd1cmF0aW9uU2V0U2VuZGluZ0VuYWJsZWQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUNvbmZpZ3VyYXRpb25TZXRTZW5kaW5nRW5hYmxlZCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIG1vZGlmeSBhbiBhc3NvY2lhdGlvbiBiZXR3ZWVuIGEgY29uZmlndXJhdGlvbiBzZXQgYW5kIGEgY3VzdG9tIGRvbWFpbiBmb3Igb3BlbiBhbmQgY2xpY2sgZXZlbnQgdHJhY2tpbmdcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkFwaVZlcnNpb24oKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZUNvbmZpZ3VyYXRpb25TZXRUcmFja2luZ09wdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlQ29uZmlndXJhdGlvblNldFRyYWNraW5nT3B0aW9ucygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlQ29uZmlndXJhdGlvblNldFRyYWNraW5nT3B0aW9ucycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhbiBleGlzdGluZyBjdXN0b20gdmVyaWZpY2F0aW9uIGVtYWlsIHRlbXBsYXRlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBcGlWZXJzaW9uKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3Nlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVDdXN0b21WZXJpZmljYXRpb25FbWFpbFRlbXBsYXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUN1c3RvbVZlcmlmaWNhdGlvbkVtYWlsVGVtcGxhdGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUN1c3RvbVZlcmlmaWNhdGlvbkVtYWlsVGVtcGxhdGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSByZWNlaXB0IHJ1bGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkFwaVZlcnNpb24oKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZVJlY2VpcHRSdWxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZVJlY2VpcHRSdWxlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVSZWNlaXB0UnVsZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhbiBlbWFpbCB0ZW1wbGF0ZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXBpVmVyc2lvbigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlVGVtcGxhdGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlVGVtcGxhdGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZVRlbXBsYXRlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIGEgc2V0IG9mIERLSU0gdG9rZW5zIGZvciBhIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXBpVmVyc2lvbigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVmVyaWZ5RG9tYWluRGtpbS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9WZXJpZnlEb21haW5Ea2ltKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdWZXJpZnlEb21haW5Ea2ltJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdmVyaWZ5IGEgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBcGlWZXJzaW9uKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3Nlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9WZXJpZnlEb21haW5JZGVudGl0eS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9WZXJpZnlEb21haW5JZGVudGl0eSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVmVyaWZ5RG9tYWluSWRlbnRpdHknKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB2ZXJpZnkgYW4gZW1haWwgYWRkcmVzc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXBpVmVyc2lvbigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVmVyaWZ5RW1haWxBZGRyZXNzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1ZlcmlmeUVtYWlsQWRkcmVzcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVmVyaWZ5RW1haWxBZGRyZXNzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdmVyaWZ5IGFuIGVtYWlsIGlkZW50aXR5XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBcGlWZXJzaW9uKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3Nlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9WZXJpZnlFbWFpbElkZW50aXR5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1ZlcmlmeUVtYWlsSWRlbnRpdHkoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1ZlcmlmeUVtYWlsSWRlbnRpdHknKTtcbiAgfVxuXG4gIHByb3RlY3RlZCBhY2Nlc3NMZXZlbExpc3Q6IEFjY2Vzc0xldmVsTGlzdCA9IHtcbiAgICBXcml0ZTogW1xuICAgICAgJ0Nsb25lUmVjZWlwdFJ1bGVTZXQnLFxuICAgICAgJ0NyZWF0ZUNvbmZpZ3VyYXRpb25TZXQnLFxuICAgICAgJ0NyZWF0ZUNvbmZpZ3VyYXRpb25TZXRFdmVudERlc3RpbmF0aW9uJyxcbiAgICAgICdDcmVhdGVDb25maWd1cmF0aW9uU2V0VHJhY2tpbmdPcHRpb25zJyxcbiAgICAgICdDcmVhdGVDdXN0b21WZXJpZmljYXRpb25FbWFpbFRlbXBsYXRlJyxcbiAgICAgICdDcmVhdGVSZWNlaXB0RmlsdGVyJyxcbiAgICAgICdDcmVhdGVSZWNlaXB0UnVsZScsXG4gICAgICAnQ3JlYXRlUmVjZWlwdFJ1bGVTZXQnLFxuICAgICAgJ0NyZWF0ZVRlbXBsYXRlJyxcbiAgICAgICdEZWxldGVDb25maWd1cmF0aW9uU2V0JyxcbiAgICAgICdEZWxldGVDb25maWd1cmF0aW9uU2V0RXZlbnREZXN0aW5hdGlvbicsXG4gICAgICAnRGVsZXRlQ29uZmlndXJhdGlvblNldFRyYWNraW5nT3B0aW9ucycsXG4gICAgICAnRGVsZXRlQ3VzdG9tVmVyaWZpY2F0aW9uRW1haWxUZW1wbGF0ZScsXG4gICAgICAnRGVsZXRlSWRlbnRpdHknLFxuICAgICAgJ0RlbGV0ZVJlY2VpcHRGaWx0ZXInLFxuICAgICAgJ0RlbGV0ZVJlY2VpcHRSdWxlJyxcbiAgICAgICdEZWxldGVSZWNlaXB0UnVsZVNldCcsXG4gICAgICAnRGVsZXRlVGVtcGxhdGUnLFxuICAgICAgJ0RlbGV0ZVZlcmlmaWVkRW1haWxBZGRyZXNzJyxcbiAgICAgICdQdXRDb25maWd1cmF0aW9uU2V0RGVsaXZlcnlPcHRpb25zJyxcbiAgICAgICdSZW9yZGVyUmVjZWlwdFJ1bGVTZXQnLFxuICAgICAgJ1NlbmRCb3VuY2UnLFxuICAgICAgJ1NlbmRCdWxrVGVtcGxhdGVkRW1haWwnLFxuICAgICAgJ1NlbmRDdXN0b21WZXJpZmljYXRpb25FbWFpbCcsXG4gICAgICAnU2VuZEVtYWlsJyxcbiAgICAgICdTZW5kUmF3RW1haWwnLFxuICAgICAgJ1NlbmRUZW1wbGF0ZWRFbWFpbCcsXG4gICAgICAnU2V0QWN0aXZlUmVjZWlwdFJ1bGVTZXQnLFxuICAgICAgJ1NldElkZW50aXR5RGtpbUVuYWJsZWQnLFxuICAgICAgJ1NldElkZW50aXR5RmVlZGJhY2tGb3J3YXJkaW5nRW5hYmxlZCcsXG4gICAgICAnU2V0SWRlbnRpdHlIZWFkZXJzSW5Ob3RpZmljYXRpb25zRW5hYmxlZCcsXG4gICAgICAnU2V0SWRlbnRpdHlNYWlsRnJvbURvbWFpbicsXG4gICAgICAnU2V0SWRlbnRpdHlOb3RpZmljYXRpb25Ub3BpYycsXG4gICAgICAnU2V0UmVjZWlwdFJ1bGVQb3NpdGlvbicsXG4gICAgICAnVGVzdFJlbmRlclRlbXBsYXRlJyxcbiAgICAgICdVcGRhdGVBY2NvdW50U2VuZGluZ0VuYWJsZWQnLFxuICAgICAgJ1VwZGF0ZUNvbmZpZ3VyYXRpb25TZXRFdmVudERlc3RpbmF0aW9uJyxcbiAgICAgICdVcGRhdGVDb25maWd1cmF0aW9uU2V0UmVwdXRhdGlvbk1ldHJpY3NFbmFibGVkJyxcbiAgICAgICdVcGRhdGVDb25maWd1cmF0aW9uU2V0U2VuZGluZ0VuYWJsZWQnLFxuICAgICAgJ1VwZGF0ZUNvbmZpZ3VyYXRpb25TZXRUcmFja2luZ09wdGlvbnMnLFxuICAgICAgJ1VwZGF0ZUN1c3RvbVZlcmlmaWNhdGlvbkVtYWlsVGVtcGxhdGUnLFxuICAgICAgJ1VwZGF0ZVJlY2VpcHRSdWxlJyxcbiAgICAgICdVcGRhdGVUZW1wbGF0ZScsXG4gICAgICAnVmVyaWZ5RG9tYWluRGtpbScsXG4gICAgICAnVmVyaWZ5RG9tYWluSWRlbnRpdHknLFxuICAgICAgJ1ZlcmlmeUVtYWlsQWRkcmVzcycsXG4gICAgICAnVmVyaWZ5RW1haWxJZGVudGl0eSdcbiAgICBdLFxuICAgICdQZXJtaXNzaW9ucyBtYW5hZ2VtZW50JzogW1xuICAgICAgJ0RlbGV0ZUlkZW50aXR5UG9saWN5JyxcbiAgICAgICdQdXRJZGVudGl0eVBvbGljeSdcbiAgICBdLFxuICAgIFJlYWQ6IFtcbiAgICAgICdEZXNjcmliZUFjdGl2ZVJlY2VpcHRSdWxlU2V0JyxcbiAgICAgICdEZXNjcmliZUNvbmZpZ3VyYXRpb25TZXQnLFxuICAgICAgJ0Rlc2NyaWJlUmVjZWlwdFJ1bGUnLFxuICAgICAgJ0Rlc2NyaWJlUmVjZWlwdFJ1bGVTZXQnLFxuICAgICAgJ0dldEFjY291bnRTZW5kaW5nRW5hYmxlZCcsXG4gICAgICAnR2V0Q3VzdG9tVmVyaWZpY2F0aW9uRW1haWxUZW1wbGF0ZScsXG4gICAgICAnR2V0SWRlbnRpdHlEa2ltQXR0cmlidXRlcycsXG4gICAgICAnR2V0SWRlbnRpdHlNYWlsRnJvbURvbWFpbkF0dHJpYnV0ZXMnLFxuICAgICAgJ0dldElkZW50aXR5Tm90aWZpY2F0aW9uQXR0cmlidXRlcycsXG4gICAgICAnR2V0SWRlbnRpdHlQb2xpY2llcycsXG4gICAgICAnR2V0SWRlbnRpdHlWZXJpZmljYXRpb25BdHRyaWJ1dGVzJyxcbiAgICAgICdHZXRTZW5kUXVvdGEnLFxuICAgICAgJ0dldFNlbmRTdGF0aXN0aWNzJyxcbiAgICAgICdHZXRUZW1wbGF0ZScsXG4gICAgICAnTGlzdFJlY2VpcHRGaWx0ZXJzJyxcbiAgICAgICdMaXN0UmVjZWlwdFJ1bGVTZXRzJyxcbiAgICAgICdMaXN0VmVyaWZpZWRFbWFpbEFkZHJlc3NlcydcbiAgICBdLFxuICAgIExpc3Q6IFtcbiAgICAgICdMaXN0Q29uZmlndXJhdGlvblNldHMnLFxuICAgICAgJ0xpc3RDdXN0b21WZXJpZmljYXRpb25FbWFpbFRlbXBsYXRlcycsXG4gICAgICAnTGlzdElkZW50aXRpZXMnLFxuICAgICAgJ0xpc3RJZGVudGl0eVBvbGljaWVzJyxcbiAgICAgICdMaXN0VGVtcGxhdGVzJ1xuICAgIF1cbiAgfTtcblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgY29uZmlndXJhdGlvbi1zZXQgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NvbmZpZ3VyYXRpb25TZXQuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gY29uZmlndXJhdGlvblNldE5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgY29uZmlndXJhdGlvblNldE5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKi9cbiAgcHVibGljIG9uQ29uZmlndXJhdGlvblNldChjb25maWd1cmF0aW9uU2V0TmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OnNlczokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06Y29uZmlndXJhdGlvbi1zZXQvJHsgY29uZmlndXJhdGlvblNldE5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGN1c3RvbS12ZXJpZmljYXRpb24tZW1haWwtdGVtcGxhdGUgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0N1c3RvbVZlcmlmaWNhdGlvbkVtYWlsVGVtcGxhdGUuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gdGVtcGxhdGVOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIHRlbXBsYXRlTmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqL1xuICBwdWJsaWMgb25DdXN0b21WZXJpZmljYXRpb25FbWFpbFRlbXBsYXRlKHRlbXBsYXRlTmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OnNlczokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06Y3VzdG9tLXZlcmlmaWNhdGlvbi1lbWFpbC10ZW1wbGF0ZS8keyB0ZW1wbGF0ZU5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGlkZW50aXR5IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3Nlcy9sYXRlc3QvQVBJUmVmZXJlbmNlLVYyL0FQSV9JZGVudGl0eUluZm8uaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gaWRlbnRpdHlOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGlkZW50aXR5TmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqL1xuICBwdWJsaWMgb25JZGVudGl0eShpZGVudGl0eU5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpzZXM6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmlkZW50aXR5LyR7IGlkZW50aXR5TmFtZSB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgdGVtcGxhdGUgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1RlbXBsYXRlLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHRlbXBsYXRlTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSB0ZW1wbGF0ZU5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKi9cbiAgcHVibGljIG9uVGVtcGxhdGUodGVtcGxhdGVOYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06c2VzOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTp0ZW1wbGF0ZS8keyB0ZW1wbGF0ZU5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWN0aW9ucyBiYXNlZCBvbiB0aGUgU0VTIEFQSSB2ZXJzaW9uXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9saXN0X2FtYXpvbnNlcy5odG1sI2FtYXpvbnNlcy1wb2xpY3kta2V5c1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQ2xvbmVSZWNlaXB0UnVsZVNldCgpXG4gICAqIC0gLnRvQ3JlYXRlQ29uZmlndXJhdGlvblNldCgpXG4gICAqIC0gLnRvQ3JlYXRlQ29uZmlndXJhdGlvblNldEV2ZW50RGVzdGluYXRpb24oKVxuICAgKiAtIC50b0NyZWF0ZUNvbmZpZ3VyYXRpb25TZXRUcmFja2luZ09wdGlvbnMoKVxuICAgKiAtIC50b0NyZWF0ZUN1c3RvbVZlcmlmaWNhdGlvbkVtYWlsVGVtcGxhdGUoKVxuICAgKiAtIC50b0NyZWF0ZVJlY2VpcHRGaWx0ZXIoKVxuICAgKiAtIC50b0NyZWF0ZVJlY2VpcHRSdWxlKClcbiAgICogLSAudG9DcmVhdGVSZWNlaXB0UnVsZVNldCgpXG4gICAqIC0gLnRvQ3JlYXRlVGVtcGxhdGUoKVxuICAgKiAtIC50b0RlbGV0ZUNvbmZpZ3VyYXRpb25TZXQoKVxuICAgKiAtIC50b0RlbGV0ZUNvbmZpZ3VyYXRpb25TZXRFdmVudERlc3RpbmF0aW9uKClcbiAgICogLSAudG9EZWxldGVDb25maWd1cmF0aW9uU2V0VHJhY2tpbmdPcHRpb25zKClcbiAgICogLSAudG9EZWxldGVDdXN0b21WZXJpZmljYXRpb25FbWFpbFRlbXBsYXRlKClcbiAgICogLSAudG9EZWxldGVJZGVudGl0eSgpXG4gICAqIC0gLnRvRGVsZXRlSWRlbnRpdHlQb2xpY3koKVxuICAgKiAtIC50b0RlbGV0ZVJlY2VpcHRGaWx0ZXIoKVxuICAgKiAtIC50b0RlbGV0ZVJlY2VpcHRSdWxlKClcbiAgICogLSAudG9EZWxldGVSZWNlaXB0UnVsZVNldCgpXG4gICAqIC0gLnRvRGVsZXRlVGVtcGxhdGUoKVxuICAgKiAtIC50b0RlbGV0ZVZlcmlmaWVkRW1haWxBZGRyZXNzKClcbiAgICogLSAudG9EZXNjcmliZUFjdGl2ZVJlY2VpcHRSdWxlU2V0KClcbiAgICogLSAudG9EZXNjcmliZUNvbmZpZ3VyYXRpb25TZXQoKVxuICAgKiAtIC50b0Rlc2NyaWJlUmVjZWlwdFJ1bGUoKVxuICAgKiAtIC50b0Rlc2NyaWJlUmVjZWlwdFJ1bGVTZXQoKVxuICAgKiAtIC50b0dldEFjY291bnRTZW5kaW5nRW5hYmxlZCgpXG4gICAqIC0gLnRvR2V0Q3VzdG9tVmVyaWZpY2F0aW9uRW1haWxUZW1wbGF0ZSgpXG4gICAqIC0gLnRvR2V0SWRlbnRpdHlEa2ltQXR0cmlidXRlcygpXG4gICAqIC0gLnRvR2V0SWRlbnRpdHlNYWlsRnJvbURvbWFpbkF0dHJpYnV0ZXMoKVxuICAgKiAtIC50b0dldElkZW50aXR5Tm90aWZpY2F0aW9uQXR0cmlidXRlcygpXG4gICAqIC0gLnRvR2V0SWRlbnRpdHlQb2xpY2llcygpXG4gICAqIC0gLnRvR2V0SWRlbnRpdHlWZXJpZmljYXRpb25BdHRyaWJ1dGVzKClcbiAgICogLSAudG9HZXRTZW5kUXVvdGEoKVxuICAgKiAtIC50b0dldFNlbmRTdGF0aXN0aWNzKClcbiAgICogLSAudG9HZXRUZW1wbGF0ZSgpXG4gICAqIC0gLnRvTGlzdENvbmZpZ3VyYXRpb25TZXRzKClcbiAgICogLSAudG9MaXN0Q3VzdG9tVmVyaWZpY2F0aW9uRW1haWxUZW1wbGF0ZXMoKVxuICAgKiAtIC50b0xpc3RJZGVudGl0aWVzKClcbiAgICogLSAudG9MaXN0SWRlbnRpdHlQb2xpY2llcygpXG4gICAqIC0gLnRvTGlzdFJlY2VpcHRGaWx0ZXJzKClcbiAgICogLSAudG9MaXN0UmVjZWlwdFJ1bGVTZXRzKClcbiAgICogLSAudG9MaXN0VGVtcGxhdGVzKClcbiAgICogLSAudG9MaXN0VmVyaWZpZWRFbWFpbEFkZHJlc3NlcygpXG4gICAqIC0gLnRvUHV0Q29uZmlndXJhdGlvblNldERlbGl2ZXJ5T3B0aW9ucygpXG4gICAqIC0gLnRvUHV0SWRlbnRpdHlQb2xpY3koKVxuICAgKiAtIC50b1Jlb3JkZXJSZWNlaXB0UnVsZVNldCgpXG4gICAqIC0gLnRvU2VuZEJvdW5jZSgpXG4gICAqIC0gLnRvU2VuZEJ1bGtUZW1wbGF0ZWRFbWFpbCgpXG4gICAqIC0gLnRvU2VuZEN1c3RvbVZlcmlmaWNhdGlvbkVtYWlsKClcbiAgICogLSAudG9TZW5kRW1haWwoKVxuICAgKiAtIC50b1NlbmRSYXdFbWFpbCgpXG4gICAqIC0gLnRvU2VuZFRlbXBsYXRlZEVtYWlsKClcbiAgICogLSAudG9TZXRBY3RpdmVSZWNlaXB0UnVsZVNldCgpXG4gICAqIC0gLnRvU2V0SWRlbnRpdHlEa2ltRW5hYmxlZCgpXG4gICAqIC0gLnRvU2V0SWRlbnRpdHlGZWVkYmFja0ZvcndhcmRpbmdFbmFibGVkKClcbiAgICogLSAudG9TZXRJZGVudGl0eUhlYWRlcnNJbk5vdGlmaWNhdGlvbnNFbmFibGVkKClcbiAgICogLSAudG9TZXRJZGVudGl0eU1haWxGcm9tRG9tYWluKClcbiAgICogLSAudG9TZXRJZGVudGl0eU5vdGlmaWNhdGlvblRvcGljKClcbiAgICogLSAudG9TZXRSZWNlaXB0UnVsZVBvc2l0aW9uKClcbiAgICogLSAudG9UZXN0UmVuZGVyVGVtcGxhdGUoKVxuICAgKiAtIC50b1VwZGF0ZUFjY291bnRTZW5kaW5nRW5hYmxlZCgpXG4gICAqIC0gLnRvVXBkYXRlQ29uZmlndXJhdGlvblNldEV2ZW50RGVzdGluYXRpb24oKVxuICAgKiAtIC50b1VwZGF0ZUNvbmZpZ3VyYXRpb25TZXRSZXB1dGF0aW9uTWV0cmljc0VuYWJsZWQoKVxuICAgKiAtIC50b1VwZGF0ZUNvbmZpZ3VyYXRpb25TZXRTZW5kaW5nRW5hYmxlZCgpXG4gICAqIC0gLnRvVXBkYXRlQ29uZmlndXJhdGlvblNldFRyYWNraW5nT3B0aW9ucygpXG4gICAqIC0gLnRvVXBkYXRlQ3VzdG9tVmVyaWZpY2F0aW9uRW1haWxUZW1wbGF0ZSgpXG4gICAqIC0gLnRvVXBkYXRlUmVjZWlwdFJ1bGUoKVxuICAgKiAtIC50b1VwZGF0ZVRlbXBsYXRlKClcbiAgICogLSAudG9WZXJpZnlEb21haW5Ea2ltKClcbiAgICogLSAudG9WZXJpZnlEb21haW5JZGVudGl0eSgpXG4gICAqIC0gLnRvVmVyaWZ5RW1haWxBZGRyZXNzKClcbiAgICogLSAudG9WZXJpZnlFbWFpbElkZW50aXR5KClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBcGlWZXJzaW9uKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBBcGlWZXJzaW9uYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY3Rpb25zIGJhc2VkIG9uIHRoZSBcIlJldHVybi1QYXRoXCIgYWRkcmVzcywgd2hpY2ggc3BlY2lmaWVzIHdoZXJlIGJvdW5jZXMgYW5kIGNvbXBsYWludHMgYXJlIHNlbnQgYnkgZW1haWwgZmVlZGJhY2sgZm9yd2FyZGluZ1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hbWF6b25zZXMuaHRtbCNhbWF6b25zZXMtcG9saWN5LWtleXNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b1NlbmRCdWxrVGVtcGxhdGVkRW1haWwoKVxuICAgKiAtIC50b1NlbmRDdXN0b21WZXJpZmljYXRpb25FbWFpbCgpXG4gICAqIC0gLnRvU2VuZEVtYWlsKClcbiAgICogLSAudG9TZW5kUmF3RW1haWwoKVxuICAgKiAtIC50b1NlbmRUZW1wbGF0ZWRFbWFpbCgpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmRmVlZGJhY2tBZGRyZXNzKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBGZWVkYmFja0FkZHJlc3NgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjdGlvbnMgYmFzZWQgb24gdGhlIFwiRnJvbVwiIGFkZHJlc3Mgb2YgYSBtZXNzYWdlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9saXN0X2FtYXpvbnNlcy5odG1sI2FtYXpvbnNlcy1wb2xpY3kta2V5c1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvU2VuZEJvdW5jZSgpXG4gICAqIC0gLnRvU2VuZEJ1bGtUZW1wbGF0ZWRFbWFpbCgpXG4gICAqIC0gLnRvU2VuZEN1c3RvbVZlcmlmaWNhdGlvbkVtYWlsKClcbiAgICogLSAudG9TZW5kRW1haWwoKVxuICAgKiAtIC50b1NlbmRSYXdFbWFpbCgpXG4gICAqIC0gLnRvU2VuZFRlbXBsYXRlZEVtYWlsKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZGcm9tQWRkcmVzcyh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgRnJvbUFkZHJlc3NgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjdGlvbnMgYmFzZWQgb24gdGhlIFwiRnJvbVwiIGFkZHJlc3MgdGhhdCBpcyB1c2VkIGFzIHRoZSBkaXNwbGF5IG5hbWUgb2YgYSBtZXNzYWdlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9saXN0X2FtYXpvbnNlcy5odG1sI2FtYXpvbnNlcy1wb2xpY3kta2V5c1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvU2VuZEJ1bGtUZW1wbGF0ZWRFbWFpbCgpXG4gICAqIC0gLnRvU2VuZEN1c3RvbVZlcmlmaWNhdGlvbkVtYWlsKClcbiAgICogLSAudG9TZW5kRW1haWwoKVxuICAgKiAtIC50b1NlbmRSYXdFbWFpbCgpXG4gICAqIC0gLnRvU2VuZFRlbXBsYXRlZEVtYWlsKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZGcm9tRGlzcGxheU5hbWUodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYEZyb21EaXNwbGF5TmFtZWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWN0aW9ucyBiYXNlZCBvbiB0aGUgcmVjaXBpZW50IGFkZHJlc3NlcyBvZiBhIG1lc3NhZ2UsIHdoaWNoIGluY2x1ZGUgdGhlIFwiVG9cIiwgXCJDQ1wiLCBhbmQgXCJCQ0NcIiBhZGRyZXNzZXNcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2xpc3RfYW1hem9uc2VzLmh0bWwjYW1hem9uc2VzLXBvbGljeS1rZXlzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9TZW5kQnVsa1RlbXBsYXRlZEVtYWlsKClcbiAgICogLSAudG9TZW5kQ3VzdG9tVmVyaWZpY2F0aW9uRW1haWwoKVxuICAgKiAtIC50b1NlbmRFbWFpbCgpXG4gICAqIC0gLnRvU2VuZFJhd0VtYWlsKClcbiAgICogLSAudG9TZW5kVGVtcGxhdGVkRW1haWwoKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlY2lwaWVudHModmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFJlY2lwaWVudHNgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW3Nlc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYW1hem9uc2VzLmh0bWwpLlxuICAgKlxuICAgKi9cbiAgY29uc3RydWN0b3IocHJvcHM/OiBpYW0uUG9saWN5U3RhdGVtZW50UHJvcHMpIHtcbiAgICBzdXBlcihwcm9wcyk7XG4gIH1cbn1cbiJdfQ==