"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SsmIncidents = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [ssm-incidents](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssystemsmanagerincidentmanager.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class SsmIncidents extends shared_1.PolicyStatement {
    /**
     * Grants permission to retrieve details about specified findings for an incident record
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_BatchGetIncidentFindings.html
     */
    toBatchGetIncidentFindings() {
        return this.to('BatchGetIncidentFindings');
    }
    /**
     * Grants permission to create a replication set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - ssm-incidents:TagResource
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_CreateReplicationSet.html
     */
    toCreateReplicationSet() {
        return this.to('CreateReplicationSet');
    }
    /**
     * Grants permission to create a response plan
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:PassRole
     * - ssm-incidents:TagResource
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_CreateResponsePlan.html
     */
    toCreateResponsePlan() {
        return this.to('CreateResponsePlan');
    }
    /**
     * Grants permission to create a timeline event for an incident record
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_CreateTimelineEvent.html
     */
    toCreateTimelineEvent() {
        return this.to('CreateTimelineEvent');
    }
    /**
     * Grants permission to delete an incident record
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_DeleteIncidentRecord.html
     */
    toDeleteIncidentRecord() {
        return this.to('DeleteIncidentRecord');
    }
    /**
     * Grants permission to delete a replication set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_DeleteReplicationSet.html
     */
    toDeleteReplicationSet() {
        return this.to('DeleteReplicationSet');
    }
    /**
     * Grants permission to delete resource policy from a response plan
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete a response plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_DeleteResponsePlan.html
     */
    toDeleteResponsePlan() {
        return this.to('DeleteResponsePlan');
    }
    /**
     * Grants permission to delete a timeline event
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_DeleteTimelineEvent.html
     */
    toDeleteTimelineEvent() {
        return this.to('DeleteTimelineEvent');
    }
    /**
     * Grants permission to view the contents of an incident record
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_GetIncidentRecord.html
     */
    toGetIncidentRecord() {
        return this.to('GetIncidentRecord');
    }
    /**
     * Grants permission to view the replication set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_GetReplicationSet.html
     */
    toGetReplicationSet() {
        return this.to('GetReplicationSet');
    }
    /**
     * Grants permission to view resource policies of a response plan
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_GetResourcePolicies.html
     */
    toGetResourcePolicies() {
        return this.to('GetResourcePolicies');
    }
    /**
     * Grants permission to view the contents of a specified response plan
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_GetResponsePlan.html
     */
    toGetResponsePlan() {
        return this.to('GetResponsePlan');
    }
    /**
     * Grants permission to view a timeline event
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_GetTimelineEvent.html
     */
    toGetTimelineEvent() {
        return this.to('GetTimelineEvent');
    }
    /**
     * Grants permission to list findings for an incident record
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_ListIncidentFindings.html
     */
    toListIncidentFindings() {
        return this.to('ListIncidentFindings');
    }
    /**
     * Grants permission to list the contents of all incident records
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_ListIncidentRecords.html
     */
    toListIncidentRecords() {
        return this.to('ListIncidentRecords');
    }
    /**
     * Grants permission to list related items of an incident record
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_ListRelatedItems.html
     */
    toListRelatedItems() {
        return this.to('ListRelatedItems');
    }
    /**
     * Grants permission to list all replication sets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_ListReplicationSets.html
     */
    toListReplicationSets() {
        return this.to('ListReplicationSets');
    }
    /**
     * Grants permission to list all response plans
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_ListResponsePlans.html
     */
    toListResponsePlans() {
        return this.to('ListResponsePlans');
    }
    /**
     * Grants permission to view a list of resource tags for a specified resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list all timeline events for an incident record
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_ListTimelineEvents.html
     */
    toListTimelineEvents() {
        return this.to('ListTimelineEvents');
    }
    /**
     * Grants permission to put resource policy on a response plan
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to start a new incident using a response plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_StartIncident.html
     */
    toStartIncident() {
        return this.to('StartIncident');
    }
    /**
     * Grants permission to add tags to a response plan
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a response plan
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update replication set deletion protection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_UpdateDeletionProtection.html
     */
    toUpdateDeletionProtection() {
        return this.to('UpdateDeletionProtection');
    }
    /**
     * Grants permission to update the contents of an incident record
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_UpdateIncidentRecord.html
     */
    toUpdateIncidentRecord() {
        return this.to('UpdateIncidentRecord');
    }
    /**
     * Grants permission to update related items of an incident record
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_UpdateRelatedItems.html
     */
    toUpdateRelatedItems() {
        return this.to('UpdateRelatedItems');
    }
    /**
     * Grants permission to update a replication set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_UpdateReplicationSet.html
     */
    toUpdateReplicationSet() {
        return this.to('UpdateReplicationSet');
    }
    /**
     * Grants permission to update the contents of a response plan
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:PassRole
     * - ssm-incidents:TagResource
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_UpdateResponsePlan.html
     */
    toUpdateResponsePlan() {
        return this.to('UpdateResponsePlan');
    }
    /**
     * Grants permission to update a timeline event
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_UpdateTimelineEvent.html
     */
    toUpdateTimelineEvent() {
        return this.to('UpdateTimelineEvent');
    }
    /**
     * Adds a resource of type response-plan to the statement
     *
     * https://docs.aws.amazon.com/incident-manager/latest/userguide/response-plans.html
     *
     * @param responsePlan - Identifier for the responsePlan.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onResponsePlan(responsePlan, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ssm-incidents::${account ?? this.defaultAccount}:response-plan/${responsePlan}`);
    }
    /**
     * Adds a resource of type incident-record to the statement
     *
     * https://docs.aws.amazon.com/incident-manager/latest/userguide/tracking-details.html
     *
     * @param responsePlan - Identifier for the responsePlan.
     * @param incidentRecord - Identifier for the incidentRecord.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIncidentRecord(responsePlan, incidentRecord, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ssm-incidents::${account ?? this.defaultAccount}:incident-record/${responsePlan}/${incidentRecord}`);
    }
    /**
     * Adds a resource of type replication-set to the statement
     *
     * https://docs.aws.amazon.com/incident-manager/latest/userguide/disaster-recovery-resiliency.html#replication
     *
     * @param replicationSet - Identifier for the replicationSet.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onReplicationSet(replicationSet, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ssm-incidents::${account ?? this.defaultAccount}:replication-set/${replicationSet}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateReplicationSet()
     * - .toCreateResponsePlan()
     * - .toTagResource()
     * - .toUpdateResponsePlan()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - response-plan
     * - incident-record
     * - replication-set
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateReplicationSet()
     * - .toCreateResponsePlan()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateResponsePlan()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [ssm-incidents](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssystemsmanagerincidentmanager.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'ssm-incidents';
        this.accessLevelList = {
            Read: [
                'BatchGetIncidentFindings',
                'GetIncidentRecord',
                'GetReplicationSet',
                'GetResourcePolicies',
                'GetResponsePlan',
                'GetTimelineEvent',
                'ListTagsForResource'
            ],
            Write: [
                'CreateReplicationSet',
                'CreateResponsePlan',
                'CreateTimelineEvent',
                'DeleteIncidentRecord',
                'DeleteReplicationSet',
                'DeleteResponsePlan',
                'DeleteTimelineEvent',
                'StartIncident',
                'UpdateDeletionProtection',
                'UpdateIncidentRecord',
                'UpdateRelatedItems',
                'UpdateReplicationSet',
                'UpdateResponsePlan',
                'UpdateTimelineEvent'
            ],
            'Permissions management': [
                'DeleteResourcePolicy',
                'PutResourcePolicy'
            ],
            List: [
                'ListIncidentFindings',
                'ListIncidentRecords',
                'ListRelatedItems',
                'ListReplicationSets',
                'ListResponsePlans',
                'ListTimelineEvents'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.SsmIncidents = SsmIncidents;
//# sourceMappingURL=data:application/json;base64,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