"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Transfer = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [transfer](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awstransferfamily.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Transfer extends shared_1.PolicyStatement {
    /**
     * Grants permission to add an access associated with a server
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_CreateAccess.html
     */
    toCreateAccess() {
        return this.to('CreateAccess');
    }
    /**
     * Grants permission to add an agreement associated with a server
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_CreateAgreement.html
     */
    toCreateAgreement() {
        return this.to('CreateAgreement');
    }
    /**
     * Grants permission to create a connector
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_CreateConnector.html
     */
    toCreateConnector() {
        return this.to('CreateConnector');
    }
    /**
     * Grants permission to create a profile
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_CreateProfile.html
     */
    toCreateProfile() {
        return this.to('CreateProfile');
    }
    /**
     * Grants permission to create a server
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_CreateServer.html
     */
    toCreateServer() {
        return this.to('CreateServer');
    }
    /**
     * Grants permission to add a user associated with a server
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_CreateUser.html
     */
    toCreateUser() {
        return this.to('CreateUser');
    }
    /**
     * Grants permission to create a webapp
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_CreateWebApp.html
     */
    toCreateWebApp() {
        return this.to('CreateWebApp');
    }
    /**
     * Grants permission to create a workflow
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_CreateWorkflow.html
     */
    toCreateWorkflow() {
        return this.to('CreateWorkflow');
    }
    /**
     * Grants permission to delete access
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DeleteAccess.html
     */
    toDeleteAccess() {
        return this.to('DeleteAccess');
    }
    /**
     * Grants permission to delete agreement
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DeleteAgreement.html
     */
    toDeleteAgreement() {
        return this.to('DeleteAgreement');
    }
    /**
     * Grants permission to delete certificate
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DeleteCertificate.html
     */
    toDeleteCertificate() {
        return this.to('DeleteCertificate');
    }
    /**
     * Grants permission to delete connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DeleteConnector.html
     */
    toDeleteConnector() {
        return this.to('DeleteConnector');
    }
    /**
     * Grants permission to delete a host key associated with a server
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DeleteHostKey.html
     */
    toDeleteHostKey() {
        return this.to('DeleteHostKey');
    }
    /**
     * Grants permission to delete profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DeleteProfile.html
     */
    toDeleteProfile() {
        return this.to('DeleteProfile');
    }
    /**
     * Grants permission to delete a server
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DeleteServer.html
     */
    toDeleteServer() {
        return this.to('DeleteServer');
    }
    /**
     * Grants permission to delete an SSH public key from a user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DeleteSshPublicKey.html
     */
    toDeleteSshPublicKey() {
        return this.to('DeleteSshPublicKey');
    }
    /**
     * Grants permission to delete a user associated with a server
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DeleteUser.html
     */
    toDeleteUser() {
        return this.to('DeleteUser');
    }
    /**
     * Grants permission to delete webapp
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DeleteWebApp.html
     */
    toDeleteWebApp() {
        return this.to('DeleteWebApp');
    }
    /**
     * Grants permission to delete webapp customization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DeleteWebAppCustomization.html
     */
    toDeleteWebAppCustomization() {
        return this.to('DeleteWebAppCustomization');
    }
    /**
     * Grants permission to delete a workflow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DeleteWorkflow.html
     */
    toDeleteWorkflow() {
        return this.to('DeleteWorkflow');
    }
    /**
     * Grants permission to describe an access assigned to a server
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DescribeAccess.html
     */
    toDescribeAccess() {
        return this.to('DescribeAccess');
    }
    /**
     * Grants permission to describe an agreement assigned to a server
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DescribeAgreement.html
     */
    toDescribeAgreement() {
        return this.to('DescribeAgreement');
    }
    /**
     * Grants permission to describe a certificate
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DescribeCertificate.html
     */
    toDescribeCertificate() {
        return this.to('DescribeCertificate');
    }
    /**
     * Grants permission to describe a connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DescribeConnector.html
     */
    toDescribeConnector() {
        return this.to('DescribeConnector');
    }
    /**
     * Grants permission to describe an execution associated with a workflow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DescribeExecution.html
     */
    toDescribeExecution() {
        return this.to('DescribeExecution');
    }
    /**
     * Grants permission to describe a host key associated with a server
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DescribeHostKey.html
     */
    toDescribeHostKey() {
        return this.to('DescribeHostKey');
    }
    /**
     * Grants permission to describe a profile
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DescribeProfile.html
     */
    toDescribeProfile() {
        return this.to('DescribeProfile');
    }
    /**
     * Grants permission to describe a security policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DescribeSecurityPolicy.html
     */
    toDescribeSecurityPolicy() {
        return this.to('DescribeSecurityPolicy');
    }
    /**
     * Grants permission to describe a server
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DescribeServer.html
     */
    toDescribeServer() {
        return this.to('DescribeServer');
    }
    /**
     * Grants permission to describe a user associated with a server
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DescribeUser.html
     */
    toDescribeUser() {
        return this.to('DescribeUser');
    }
    /**
     * Grants permission to describe a webapp
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DescribeWebApp.html
     */
    toDescribeWebApp() {
        return this.to('DescribeWebApp');
    }
    /**
     * Grants permission to describe a webapp customization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DescribeWebAppCustomization.html
     */
    toDescribeWebAppCustomization() {
        return this.to('DescribeWebAppCustomization');
    }
    /**
     * Grants permission to describe a workflow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_DescribeWorkflow.html
     */
    toDescribeWorkflow() {
        return this.to('DescribeWorkflow');
    }
    /**
     * Grants permission to add a certificate
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_ImportCertificate.html
     */
    toImportCertificate() {
        return this.to('ImportCertificate');
    }
    /**
     * Grants permission to add a host key to a server
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_ImportHostKey.html
     */
    toImportHostKey() {
        return this.to('ImportHostKey');
    }
    /**
     * Grants permission to add an SSH public key to a user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_ImportSshPublicKey.html
     */
    toImportSshPublicKey() {
        return this.to('ImportSshPublicKey');
    }
    /**
     * Grants permission to list accesses
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_ListAccesses.html
     */
    toListAccesses() {
        return this.to('ListAccesses');
    }
    /**
     * Grants permission to list agreements
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_ListAgreements.html
     */
    toListAgreements() {
        return this.to('ListAgreements');
    }
    /**
     * Grants permission to list certificates
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_ListCertificates.html
     */
    toListCertificates() {
        return this.to('ListCertificates');
    }
    /**
     * Grants permission to list connectors
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_ListConnectors.html
     */
    toListConnectors() {
        return this.to('ListConnectors');
    }
    /**
     * Grants permission to list executions associated with a workflow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_ListExecutions.html
     */
    toListExecutions() {
        return this.to('ListExecutions');
    }
    /**
     * Grants permission to list file transfer statuses for connectors
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_ListFileTransferResults.html
     */
    toListFileTransferResults() {
        return this.to('ListFileTransferResults');
    }
    /**
     * Grants permission to list host keys associated with a server
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_ListHostKeys.html
     */
    toListHostKeys() {
        return this.to('ListHostKeys');
    }
    /**
     * Grants permission to list profiles
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_ListProfiles.html
     */
    toListProfiles() {
        return this.to('ListProfiles');
    }
    /**
     * Grants permission to list security policies
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_ListSecurityPolicies.html
     */
    toListSecurityPolicies() {
        return this.to('ListSecurityPolicies');
    }
    /**
     * Grants permission to list servers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_ListServers.html
     */
    toListServers() {
        return this.to('ListServers');
    }
    /**
     * Grants permission to list tags for an AWS Transfer Family resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list users associated with a server
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_ListUsers.html
     */
    toListUsers() {
        return this.to('ListUsers');
    }
    /**
     * Grants permission to list webapps
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_ListWebApps.html
     */
    toListWebApps() {
        return this.to('ListWebApps');
    }
    /**
     * Grants permission to list workflows
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_ListWorkflows.html
     */
    toListWorkflows() {
        return this.to('ListWorkflows');
    }
    /**
     * Grants permission to send a callback for asynchronous custom steps
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_SendWorkflowStepState.html
     */
    toSendWorkflowStepState() {
        return this.to('SendWorkflowStepState');
    }
    /**
     * Grants permission to initiate a list operation on a remote server using a connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_StartDirectoryListing.html
     */
    toStartDirectoryListing() {
        return this.to('StartDirectoryListing');
    }
    /**
     * Grants permission to initiate a connector file transfer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_StartFileTransfer.html
     */
    toStartFileTransfer() {
        return this.to('StartFileTransfer');
    }
    /**
     * Grants permission to initiate a connector delete operation on remote server
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_StartRemoteDelete.html
     */
    toStartRemoteDelete() {
        return this.to('StartRemoteDelete');
    }
    /**
     * Grants permission to initiate a connector move operation on remote server
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_StartRemoteMove.html
     */
    toStartRemoteMove() {
        return this.to('StartRemoteMove');
    }
    /**
     * Grants permission to start a server
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_StartServer.html
     */
    toStartServer() {
        return this.to('StartServer');
    }
    /**
     * Grants permission to stop a server
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_StopServer.html
     */
    toStopServer() {
        return this.to('StopServer');
    }
    /**
     * Grants permission to tag an AWS Transfer Family resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to test a connector's connection to remote server
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_TestConnection.html
     */
    toTestConnection() {
        return this.to('TestConnection');
    }
    /**
     * Grants permission to test a server's custom identity provider
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_TestIdentityProvider.html
     */
    toTestIdentityProvider() {
        return this.to('TestIdentityProvider');
    }
    /**
     * Grants permission to untag an AWS Transfer Family resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update access
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_UpdateAccess.html
     */
    toUpdateAccess() {
        return this.to('UpdateAccess');
    }
    /**
     * Grants permission to update an agreement
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_UpdateAgreement.html
     */
    toUpdateAgreement() {
        return this.to('UpdateAgreement');
    }
    /**
     * Grants permission to update a certificate
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_UpdateCertificate.html
     */
    toUpdateCertificate() {
        return this.to('UpdateCertificate');
    }
    /**
     * Grants permission to update a connector
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_UpdateConnector.html
     */
    toUpdateConnector() {
        return this.to('UpdateConnector');
    }
    /**
     * Grants permission to update a host key
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_UpdateHostKey.html
     */
    toUpdateHostKey() {
        return this.to('UpdateHostKey');
    }
    /**
     * Grants permission to update a profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_UpdateProfile.html
     */
    toUpdateProfile() {
        return this.to('UpdateProfile');
    }
    /**
     * Grants permission to update the configuration of a server
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_UpdateServer.html
     */
    toUpdateServer() {
        return this.to('UpdateServer');
    }
    /**
     * Grants permission to update the configuration of a user
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_UpdateUser.html
     */
    toUpdateUser() {
        return this.to('UpdateUser');
    }
    /**
     * Grants permission to update the configuration of a webapp
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_UpdateWebApp.html
     */
    toUpdateWebApp() {
        return this.to('UpdateWebApp');
    }
    /**
     * Grants permission to update the configuration of a webapp cutomization
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/API_UpdateWebAppCustomization.html
     */
    toUpdateWebAppCustomization() {
        return this.to('UpdateWebAppCustomization');
    }
    /**
     * Adds a resource of type user to the statement
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/create-user.html
     *
     * @param serverId - Identifier for the serverId.
     * @param userName - Identifier for the userName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onUser(serverId, userName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:transfer:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:user/${serverId}/${userName}`);
    }
    /**
     * Adds a resource of type server to the statement
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/configuring-servers.html
     *
     * @param serverId - Identifier for the serverId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onServer(serverId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:transfer:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:server/${serverId}`);
    }
    /**
     * Adds a resource of type workflow to the statement
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/transfer-workflows.html
     *
     * @param workflowId - Identifier for the workflowId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorkflow(workflowId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:transfer:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workflow/${workflowId}`);
    }
    /**
     * Adds a resource of type certificate to the statement
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/create-b2b-server.html
     *
     * @param certificateId - Identifier for the certificateId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCertificate(certificateId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:transfer:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:certificate/${certificateId}`);
    }
    /**
     * Adds a resource of type connector to the statement
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/create-b2b-server.html
     *
     * @param connectorId - Identifier for the connectorId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConnector(connectorId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:transfer:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:connector/${connectorId}`);
    }
    /**
     * Adds a resource of type profile to the statement
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/create-b2b-server.html
     *
     * @param profileId - Identifier for the profileId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProfile(profileId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:transfer:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:profile/${profileId}`);
    }
    /**
     * Adds a resource of type agreement to the statement
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/create-b2b-server.html
     *
     * @param serverId - Identifier for the serverId.
     * @param agreementId - Identifier for the agreementId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAgreement(serverId, agreementId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:transfer:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:agreement/${serverId}/${agreementId}`);
    }
    /**
     * Adds a resource of type host-key to the statement
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/edit-server-config.html
     *
     * @param serverId - Identifier for the serverId.
     * @param hostKeyId - Identifier for the hostKeyId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onHostKey(serverId, hostKeyId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:transfer:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:host-key/${serverId}/${hostKeyId}`);
    }
    /**
     * Adds a resource of type webapp to the statement
     *
     * https://docs.aws.amazon.com/transfer/latest/userguide/web-app.html
     *
     * @param webAppId - Identifier for the webAppId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWebapp(webAppId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:transfer:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:webapp/${webAppId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAgreement()
     * - .toCreateConnector()
     * - .toCreateProfile()
     * - .toCreateServer()
     * - .toCreateUser()
     * - .toCreateWebApp()
     * - .toCreateWorkflow()
     * - .toImportCertificate()
     * - .toImportHostKey()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - user
     * - server
     * - workflow
     * - certificate
     * - connector
     * - profile
     * - agreement
     * - host-key
     * - webapp
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAgreement()
     * - .toCreateConnector()
     * - .toCreateProfile()
     * - .toCreateServer()
     * - .toCreateUser()
     * - .toCreateWebApp()
     * - .toCreateWorkflow()
     * - .toImportCertificate()
     * - .toImportHostKey()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [transfer](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awstransferfamily.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'transfer';
        this.accessLevelList = {
            Write: [
                'CreateAccess',
                'CreateAgreement',
                'CreateConnector',
                'CreateProfile',
                'CreateServer',
                'CreateUser',
                'CreateWebApp',
                'CreateWorkflow',
                'DeleteAccess',
                'DeleteAgreement',
                'DeleteCertificate',
                'DeleteConnector',
                'DeleteHostKey',
                'DeleteProfile',
                'DeleteServer',
                'DeleteSshPublicKey',
                'DeleteUser',
                'DeleteWebApp',
                'DeleteWebAppCustomization',
                'DeleteWorkflow',
                'ImportCertificate',
                'ImportHostKey',
                'ImportSshPublicKey',
                'SendWorkflowStepState',
                'StartDirectoryListing',
                'StartFileTransfer',
                'StartRemoteDelete',
                'StartRemoteMove',
                'StartServer',
                'StopServer',
                'TestConnection',
                'UpdateAccess',
                'UpdateAgreement',
                'UpdateCertificate',
                'UpdateConnector',
                'UpdateHostKey',
                'UpdateProfile',
                'UpdateServer',
                'UpdateUser',
                'UpdateWebApp',
                'UpdateWebAppCustomization'
            ],
            Read: [
                'DescribeAccess',
                'DescribeAgreement',
                'DescribeCertificate',
                'DescribeConnector',
                'DescribeExecution',
                'DescribeHostKey',
                'DescribeProfile',
                'DescribeSecurityPolicy',
                'DescribeServer',
                'DescribeUser',
                'DescribeWebApp',
                'DescribeWebAppCustomization',
                'DescribeWorkflow',
                'ListAccesses',
                'ListAgreements',
                'ListCertificates',
                'ListConnectors',
                'ListExecutions',
                'ListFileTransferResults',
                'ListHostKeys',
                'ListProfiles',
                'ListTagsForResource',
                'TestIdentityProvider'
            ],
            List: [
                'ListSecurityPolicies',
                'ListServers',
                'ListUsers',
                'ListWebApps',
                'ListWorkflows'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Transfer = Transfer;
//# sourceMappingURL=data:application/json;base64,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