"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.verifyCodeConfig = exports.Function = exports.Tracing = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const aws_codeguruprofiler_1 = require("@aws-cdk/aws-codeguruprofiler");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const sqs = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const architecture_1 = require("./architecture");
const function_base_1 = require("./function-base");
const function_hash_1 = require("./function-hash");
const handler_1 = require("./handler");
const lambda_version_1 = require("./lambda-version");
const lambda_generated_1 = require("./lambda.generated");
const layers_1 = require("./layers");
const runtime_1 = require("./runtime");
/**
 * X-Ray Tracing Modes (https://docs.aws.amazon.com/lambda/latest/dg/API_TracingConfig.html)
 */
var Tracing;
(function (Tracing) {
    /**
     * Lambda will respect any tracing header it receives from an upstream service.
     * If no tracing header is received, Lambda will call X-Ray for a tracing decision.
     */
    Tracing["ACTIVE"] = "Active";
    /**
     * Lambda will only trace the request from an upstream service
     * if it contains a tracing header with "sampled=1"
     */
    Tracing["PASS_THROUGH"] = "PassThrough";
    /**
     * Lambda will not trace any request.
     */
    Tracing["DISABLED"] = "Disabled";
})(Tracing = exports.Tracing || (exports.Tracing = {}));
/**
 * Deploys a file from inside the construct library as a function.
 *
 * The supplied file is subject to the 4096 bytes limit of being embedded in a
 * CloudFormation template.
 *
 * The construct includes an associated role with the lambda.
 *
 * This construct does not yet reproduce all features from the underlying resource
 * library.
 */
class Function extends function_base_1.FunctionBase {
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k;
        super(scope, id, {
            physicalName: props.functionName,
        });
        this.permissionsNode = this.node;
        this.canCreatePermissions = true;
        this.layers = [];
        /**
         * Environment variables for this function
         */
        this.environment = {};
        jsiiDeprecationWarnings._aws_cdk_aws_lambda_FunctionProps(props);
        if (props.functionName && !core_1.Token.isUnresolved(props.functionName)) {
            if (props.functionName.length > 64) {
                throw new Error(`Function name can not be longer than 64 characters but has ${props.functionName.length} characters.`);
            }
            if (!/^[a-zA-Z0-9-_]+$/.test(props.functionName)) {
                throw new Error(`Function name ${props.functionName} can contain only letters, numbers, hyphens, or underscores with no spaces.`);
            }
        }
        const managedPolicies = new Array();
        // the arn is in the form of - arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole
        managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'));
        if (props.vpc) {
            // Policy that will have ENI creation permissions
            managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaVPCAccessExecutionRole'));
        }
        this.role = props.role || new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies,
        });
        this.grantPrincipal = this.role;
        // add additional managed policies when necessary
        if (props.filesystem) {
            const config = props.filesystem.config;
            if (config.policies) {
                config.policies.forEach(p => {
                    var _b;
                    (_b = this.role) === null || _b === void 0 ? void 0 : _b.addToPrincipalPolicy(p);
                });
            }
        }
        for (const statement of (props.initialPolicy || [])) {
            this.role.addToPrincipalPolicy(statement);
        }
        const code = props.code.bind(this);
        verifyCodeConfig(code, props);
        let profilingGroupEnvironmentVariables = {};
        if (props.profilingGroup && props.profiling !== false) {
            this.validateProfiling(props);
            props.profilingGroup.grantPublish(this.role);
            profilingGroupEnvironmentVariables = {
                AWS_CODEGURU_PROFILER_GROUP_ARN: core_1.Stack.of(scope).formatArn({
                    service: 'codeguru-profiler',
                    resource: 'profilingGroup',
                    resourceName: props.profilingGroup.profilingGroupName,
                }),
                AWS_CODEGURU_PROFILER_ENABLED: 'TRUE',
            };
        }
        else if (props.profiling) {
            this.validateProfiling(props);
            const profilingGroup = new aws_codeguruprofiler_1.ProfilingGroup(this, 'ProfilingGroup', {
                computePlatform: aws_codeguruprofiler_1.ComputePlatform.AWS_LAMBDA,
            });
            profilingGroup.grantPublish(this.role);
            profilingGroupEnvironmentVariables = {
                AWS_CODEGURU_PROFILER_GROUP_ARN: profilingGroup.profilingGroupArn,
                AWS_CODEGURU_PROFILER_ENABLED: 'TRUE',
            };
        }
        const env = { ...profilingGroupEnvironmentVariables, ...props.environment };
        for (const [key, value] of Object.entries(env)) {
            this.addEnvironment(key, value);
        }
        // DLQ can be either sns.ITopic or sqs.IQueue
        const dlqTopicOrQueue = this.buildDeadLetterQueue(props);
        if (dlqTopicOrQueue !== undefined) {
            if (this.isQueue(dlqTopicOrQueue)) {
                this.deadLetterQueue = dlqTopicOrQueue;
            }
            else {
                this.deadLetterTopic = dlqTopicOrQueue;
            }
        }
        let fileSystemConfigs = undefined;
        if (props.filesystem) {
            fileSystemConfigs = [{
                    arn: props.filesystem.config.arn,
                    localMountPath: props.filesystem.config.localMountPath,
                }];
        }
        if (props.architecture && props.architectures !== undefined) {
            throw new Error('Either architecture or architectures must be specified but not both.');
        }
        if (props.architectures && props.architectures.length > 1) {
            throw new Error('Only one architecture must be specified.');
        }
        this._architecture = (_b = props.architecture) !== null && _b !== void 0 ? _b : (props.architectures && props.architectures[0]);
        const resource = new lambda_generated_1.CfnFunction(this, 'Resource', {
            functionName: this.physicalName,
            description: props.description,
            code: {
                s3Bucket: code.s3Location && code.s3Location.bucketName,
                s3Key: code.s3Location && code.s3Location.objectKey,
                s3ObjectVersion: code.s3Location && code.s3Location.objectVersion,
                zipFile: code.inlineCode,
                imageUri: (_c = code.image) === null || _c === void 0 ? void 0 : _c.imageUri,
            },
            layers: core_1.Lazy.list({ produce: () => this.layers.map(layer => layer.layerVersionArn) }, { omitEmpty: true }),
            handler: props.handler === handler_1.Handler.FROM_IMAGE ? undefined : props.handler,
            timeout: props.timeout && props.timeout.toSeconds(),
            packageType: props.runtime === runtime_1.Runtime.FROM_IMAGE ? 'Image' : undefined,
            runtime: props.runtime === runtime_1.Runtime.FROM_IMAGE ? undefined : props.runtime.name,
            role: this.role.roleArn,
            // Uncached because calling '_checkEdgeCompatibility', which gets called in the resolve of another
            // Token, actually *modifies* the 'environment' map.
            environment: core_1.Lazy.uncachedAny({ produce: () => this.renderEnvironment() }),
            memorySize: props.memorySize,
            vpcConfig: this.configureVpc(props),
            deadLetterConfig: this.buildDeadLetterConfig(dlqTopicOrQueue),
            tracingConfig: this.buildTracingConfig(props),
            reservedConcurrentExecutions: props.reservedConcurrentExecutions,
            imageConfig: undefinedIfNoKeys({
                command: (_d = code.image) === null || _d === void 0 ? void 0 : _d.cmd,
                entryPoint: (_e = code.image) === null || _e === void 0 ? void 0 : _e.entrypoint,
                workingDirectory: (_f = code.image) === null || _f === void 0 ? void 0 : _f.workingDirectory,
            }),
            kmsKeyArn: (_g = props.environmentEncryption) === null || _g === void 0 ? void 0 : _g.keyArn,
            fileSystemConfigs,
            codeSigningConfigArn: (_h = props.codeSigningConfig) === null || _h === void 0 ? void 0 : _h.codeSigningConfigArn,
            architectures: this._architecture ? [this._architecture.name] : undefined,
        });
        resource.node.addDependency(this.role);
        this.functionName = this.getResourceNameAttribute(resource.ref);
        this.functionArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'lambda',
            resource: 'function',
            resourceName: this.physicalName,
            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
        });
        this.runtime = props.runtime;
        this.timeout = props.timeout;
        this.architecture = (_j = props.architecture) !== null && _j !== void 0 ? _j : architecture_1.Architecture.X86_64;
        if (props.layers) {
            if (props.runtime === runtime_1.Runtime.FROM_IMAGE) {
                throw new Error('Layers are not supported for container image functions');
            }
            this.addLayers(...props.layers);
        }
        for (const event of props.events || []) {
            this.addEventSource(event);
        }
        // Log retention
        if (props.logRetention) {
            const logRetention = new logs.LogRetention(this, 'LogRetention', {
                logGroupName: `/aws/lambda/${this.functionName}`,
                retention: props.logRetention,
                role: props.logRetentionRole,
                logRetentionRetryOptions: props.logRetentionRetryOptions,
            });
            this._logGroup = logs.LogGroup.fromLogGroupArn(this, 'LogGroup', logRetention.logGroupArn);
        }
        props.code.bindToResource(resource);
        // Event Invoke Config
        if (props.onFailure || props.onSuccess || props.maxEventAge || props.retryAttempts !== undefined) {
            this.configureAsyncInvoke({
                onFailure: props.onFailure,
                onSuccess: props.onSuccess,
                maxEventAge: props.maxEventAge,
                retryAttempts: props.retryAttempts,
            });
        }
        this.currentVersionOptions = props.currentVersionOptions;
        if (props.filesystem) {
            if (!props.vpc) {
                throw new Error('Cannot configure \'filesystem\' without configuring a VPC.');
            }
            const config = props.filesystem.config;
            if (config.dependency) {
                this.node.addDependency(...config.dependency);
            }
            // There could be a race if the Lambda is used in a CustomResource. It is possible for the Lambda to
            // fail to attach to a given FileSystem if we do not have a dependency on the SecurityGroup ingress/egress
            // rules that were created between this Lambda's SG & the Filesystem SG.
            this.connections.securityGroups.forEach(sg => {
                sg.node.findAll().forEach(child => {
                    if (child instanceof core_1.CfnResource && child.cfnResourceType === 'AWS::EC2::SecurityGroupEgress') {
                        resource.node.addDependency(child);
                    }
                });
            });
            (_k = config.connections) === null || _k === void 0 ? void 0 : _k.securityGroups.forEach(sg => {
                sg.node.findAll().forEach(child => {
                    if (child instanceof core_1.CfnResource && child.cfnResourceType === 'AWS::EC2::SecurityGroupIngress') {
                        resource.node.addDependency(child);
                    }
                });
            });
        }
        // Configure Lambda insights
        this.configureLambdaInsights(props);
    }
    /**
     * Returns a `lambda.Version` which represents the current version of this
     * Lambda function. A new version will be created every time the function's
     * configuration changes.
     *
     * You can specify options for this version using the `currentVersionOptions`
     * prop when initializing the `lambda.Function`.
     */
    get currentVersion() {
        if (this._currentVersion) {
            return this._currentVersion;
        }
        this._currentVersion = new lambda_version_1.Version(this, 'CurrentVersion', {
            lambda: this,
            ...this.currentVersionOptions,
        });
        // override the version's logical ID with a lazy string which includes the
        // hash of the function itself, so a new version resource is created when
        // the function configuration changes.
        const cfn = this._currentVersion.node.defaultChild;
        const originalLogicalId = this.stack.resolve(cfn.logicalId);
        cfn.overrideLogicalId(core_1.Lazy.uncachedString({
            produce: () => {
                const hash = function_hash_1.calculateFunctionHash(this);
                const logicalId = function_hash_1.trimFromStart(originalLogicalId, 255 - 32);
                return `${logicalId}${hash}`;
            },
        }));
        return this._currentVersion;
    }
    /**
     * Record whether specific properties in the `AWS::Lambda::Function` resource should
     * also be associated to the Version resource.
     * See 'currentVersion' section in the module README for more details.
     * @param propertyName The property to classify
     * @param locked whether the property should be associated to the version or not.
     */
    static classifyVersionProperty(propertyName, locked) {
        this._VER_PROPS[propertyName] = locked;
    }
    /**
     * Import a lambda function into the CDK using its name
     */
    static fromFunctionName(scope, id, functionName) {
        return Function.fromFunctionAttributes(scope, id, {
            functionArn: core_1.Stack.of(scope).formatArn({
                service: 'lambda',
                resource: 'function',
                resourceName: functionName,
                arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
            }),
        });
    }
    /**
     * Import a lambda function into the CDK using its ARN
     */
    static fromFunctionArn(scope, id, functionArn) {
        return Function.fromFunctionAttributes(scope, id, { functionArn });
    }
    /**
     * Creates a Lambda function object which represents a function not defined
     * within this stack.
     *
     * @param scope The parent construct
     * @param id The name of the lambda construct
     * @param attrs the attributes of the function to import
     */
    static fromFunctionAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_lambda_FunctionAttributes(attrs);
        const functionArn = attrs.functionArn;
        const functionName = extractNameFromArn(attrs.functionArn);
        const role = attrs.role;
        class Import extends function_base_1.FunctionBase {
            constructor(s, i) {
                var _b, _c, _d;
                super(s, i, {
                    environmentFromArn: functionArn,
                });
                this.functionName = functionName;
                this.functionArn = functionArn;
                this.role = role;
                this.permissionsNode = this.node;
                this.architecture = (_b = attrs.architecture) !== null && _b !== void 0 ? _b : architecture_1.Architecture.X86_64;
                this.canCreatePermissions = (_c = attrs.sameEnvironment) !== null && _c !== void 0 ? _c : this._isStackAccount();
                this._skipPermissions = (_d = attrs.skipPermissions) !== null && _d !== void 0 ? _d : false;
                this.grantPrincipal = role || new iam.UnknownPrincipal({ resource: this });
                if (attrs.securityGroup) {
                    this._connections = new ec2.Connections({
                        securityGroups: [attrs.securityGroup],
                    });
                }
                else if (attrs.securityGroupId) {
                    this._connections = new ec2.Connections({
                        securityGroups: [ec2.SecurityGroup.fromSecurityGroupId(scope, 'SecurityGroup', attrs.securityGroupId)],
                    });
                }
            }
        }
        return new Import(scope, id);
    }
    /**
     * Return the given named metric for this Lambda
     */
    static metricAll(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/Lambda',
            metricName,
            ...props,
        });
    }
    /**
     * Metric for the number of Errors executing all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllErrors(props) {
        return this.metricAll('Errors', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the Duration executing all Lambdas
     *
     * @default average over 5 minutes
     */
    static metricAllDuration(props) {
        return this.metricAll('Duration', props);
    }
    /**
     * Metric for the number of invocations of all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllInvocations(props) {
        return this.metricAll('Invocations', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the number of throttled invocations of all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllThrottles(props) {
        return this.metricAll('Throttles', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the number of concurrent executions across all Lambdas
     *
     * @default max over 5 minutes
     */
    static metricAllConcurrentExecutions(props) {
        // Mini-FAQ: why max? This metric is a gauge that is emitted every
        // minute, so either max or avg or a percentile make sense (but sum
        // doesn't). Max is more sensitive to spiky load changes which is
        // probably what you're interested in if you're looking at this metric
        // (Load spikes may lead to concurrent execution errors that would
        // otherwise not be visible in the avg)
        return this.metricAll('ConcurrentExecutions', { statistic: 'max', ...props });
    }
    /**
     * Metric for the number of unreserved concurrent executions across all Lambdas
     *
     * @default max over 5 minutes
     */
    static metricAllUnreservedConcurrentExecutions(props) {
        return this.metricAll('UnreservedConcurrentExecutions', { statistic: 'max', ...props });
    }
    /**
     * Adds an environment variable to this Lambda function.
     * If this is a ref to a Lambda function, this operation results in a no-op.
     * @param key The environment variable key.
     * @param value The environment variable's value.
     * @param options Environment variable options.
     */
    addEnvironment(key, value, options) {
        jsiiDeprecationWarnings._aws_cdk_aws_lambda_EnvironmentOptions(options);
        this.environment[key] = { value, ...options };
        return this;
    }
    /**
     * Adds one or more Lambda Layers to this Lambda function.
     *
     * @param layers the layers to be added.
     *
     * @throws if there are already 5 layers on this function, or the layer is incompatible with this function's runtime.
     */
    addLayers(...layers) {
        jsiiDeprecationWarnings._aws_cdk_aws_lambda_ILayerVersion(layers);
        for (const layer of layers) {
            if (this.layers.length === 5) {
                throw new Error('Unable to add layer: this lambda function already uses 5 layers.');
            }
            if (layer.compatibleRuntimes && !layer.compatibleRuntimes.find(runtime => runtime.runtimeEquals(this.runtime))) {
                const runtimes = layer.compatibleRuntimes.map(runtime => runtime.name).join(', ');
                throw new Error(`This lambda function uses a runtime that is incompatible with this layer (${this.runtime.name} is not in [${runtimes}])`);
            }
            // Currently no validations for compatible architectures since Lambda service
            // allows layers configured with one architecture to be used with a Lambda function
            // from another architecture.
            this.layers.push(layer);
        }
    }
    /**
     * Add a new version for this Lambda
     *
     * If you want to deploy through CloudFormation and use aliases, you need to
     * add a new version (with a new name) to your Lambda every time you want to
     * deploy an update. An alias can then refer to the newly created Version.
     *
     * All versions should have distinct names, and you should not delete versions
     * as long as your Alias needs to refer to them.
     *
     * @param name A unique name for this version.
     * @param codeSha256 The SHA-256 hash of the most recently deployed Lambda
     *  source code, or omit to skip validation.
     * @param description A description for this version.
     * @param provisionedExecutions A provisioned concurrency configuration for a
     * function's version.
     * @param asyncInvokeConfig configuration for this version when it is invoked
     * asynchronously.
     * @returns A new Version object.
     *
     * @deprecated This method will create an AWS::Lambda::Version resource which
     * snapshots the AWS Lambda function *at the time of its creation* and it
     * won't get updated when the function changes. Instead, use
     * `this.currentVersion` to obtain a reference to a version resource that gets
     * automatically recreated when the function configuration (or code) changes.
     */
    addVersion(name, codeSha256, description, provisionedExecutions, asyncInvokeConfig = {}) {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-lambda.Function#addVersion", "This method will create an AWS::Lambda::Version resource which\nsnapshots the AWS Lambda function *at the time of its creation* and it\nwon't get updated when the function changes. Instead, use\n`this.currentVersion` to obtain a reference to a version resource that gets\nautomatically recreated when the function configuration (or code) changes.");
        jsiiDeprecationWarnings._aws_cdk_aws_lambda_EventInvokeConfigOptions(asyncInvokeConfig);
        return new lambda_version_1.Version(this, 'Version' + name, {
            lambda: this,
            codeSha256,
            description,
            provisionedConcurrentExecutions: provisionedExecutions,
            ...asyncInvokeConfig,
        });
    }
    /**
     * The LogGroup where the Lambda function's logs are made available.
     *
     * If either `logRetention` is set or this property is called, a CloudFormation custom resource is added to the stack that
     * pre-creates the log group as part of the stack deployment, if it already doesn't exist, and sets the correct log retention
     * period (never expire, by default).
     *
     * Further, if the log group already exists and the `logRetention` is not set, the custom resource will reset the log retention
     * to never expire even if it was configured with a different value.
     */
    get logGroup() {
        if (!this._logGroup) {
            const logRetention = new logs.LogRetention(this, 'LogRetention', {
                logGroupName: `/aws/lambda/${this.functionName}`,
                retention: logs.RetentionDays.INFINITE,
            });
            this._logGroup = logs.LogGroup.fromLogGroupArn(this, `${this.node.id}-LogGroup`, logRetention.logGroupArn);
        }
        return this._logGroup;
    }
    /** @internal */
    _checkEdgeCompatibility() {
        // Check env vars
        const envEntries = Object.entries(this.environment);
        for (const [key, config] of envEntries) {
            if (config.removeInEdge) {
                delete this.environment[key];
                core_1.Annotations.of(this).addInfo(`Removed ${key} environment variable for Lambda@Edge compatibility`);
            }
        }
        const envKeys = Object.keys(this.environment);
        if (envKeys.length !== 0) {
            throw new Error(`The function ${this.node.path} contains environment variables [${envKeys}] and is not compatible with Lambda@Edge. \
Environment variables can be marked for removal when used in Lambda@Edge by setting the \'removeInEdge\' property in the \'addEnvironment()\' API.`);
        }
        return;
    }
    /**
     * Configured lambda insights on the function if specified. This is acheived by adding an imported layer which is added to the
     * list of lambda layers on synthesis.
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Lambda-Insights-extension-versions.html
     */
    configureLambdaInsights(props) {
        var _b;
        if (props.insightsVersion === undefined) {
            return;
        }
        if (props.runtime !== runtime_1.Runtime.FROM_IMAGE) {
            // Layers cannot be added to Lambda container images. The image should have the insights agent installed.
            // See https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Lambda-Insights-Getting-Started-docker.html
            this.addLayers(layers_1.LayerVersion.fromLayerVersionArn(this, 'LambdaInsightsLayer', props.insightsVersion._bind(this, this).arn));
        }
        (_b = this.role) === null || _b === void 0 ? void 0 : _b.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('CloudWatchLambdaInsightsExecutionRolePolicy'));
    }
    renderEnvironment() {
        if (!this.environment || Object.keys(this.environment).length === 0) {
            return undefined;
        }
        const variables = {};
        // Sort environment so the hash of the function used to create
        // `currentVersion` is not affected by key order (this is how lambda does
        // it). For backwards compatibility we do not sort environment variables in case
        // _currentVersion is not defined. Otherwise, this would have invalidated
        // the template, and for example, may cause unneeded updates for nested
        // stacks.
        const keys = this._currentVersion
            ? Object.keys(this.environment).sort()
            : Object.keys(this.environment);
        for (const key of keys) {
            variables[key] = this.environment[key].value;
        }
        return { variables };
    }
    /**
     * If configured, set up the VPC-related properties
     *
     * Returns the VpcConfig that should be added to the
     * Lambda creation properties.
     */
    configureVpc(props) {
        var _b;
        if ((props.securityGroup || props.allowAllOutbound !== undefined) && !props.vpc) {
            throw new Error('Cannot configure \'securityGroup\' or \'allowAllOutbound\' without configuring a VPC');
        }
        if (!props.vpc) {
            return undefined;
        }
        if (props.securityGroup && props.allowAllOutbound !== undefined) {
            throw new Error('Configure \'allowAllOutbound\' directly on the supplied SecurityGroup.');
        }
        let securityGroups;
        if (props.securityGroup && props.securityGroups) {
            throw new Error('Only one of the function props, securityGroup or securityGroups, is allowed');
        }
        if (props.securityGroups) {
            securityGroups = props.securityGroups;
        }
        else {
            const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                description: 'Automatic security group for Lambda Function ' + core_1.Names.uniqueId(this),
                allowAllOutbound: props.allowAllOutbound,
            });
            securityGroups = [securityGroup];
        }
        this._connections = new ec2.Connections({ securityGroups });
        if (props.filesystem) {
            if (props.filesystem.config.connections) {
                props.filesystem.config.connections.allowDefaultPortFrom(this);
            }
        }
        const allowPublicSubnet = (_b = props.allowPublicSubnet) !== null && _b !== void 0 ? _b : false;
        const { subnetIds } = props.vpc.selectSubnets(props.vpcSubnets);
        const publicSubnetIds = new Set(props.vpc.publicSubnets.map(s => s.subnetId));
        for (const subnetId of subnetIds) {
            if (publicSubnetIds.has(subnetId) && !allowPublicSubnet) {
                throw new Error('Lambda Functions in a public subnet can NOT access the internet. ' +
                    'If you are aware of this limitation and would still like to place the function int a public subnet, set `allowPublicSubnet` to true');
            }
        }
        // List can't be empty here, if we got this far you intended to put your Lambda
        // in subnets. We're going to guarantee that we get the nice error message by
        // making VpcNetwork do the selection again.
        return {
            subnetIds,
            securityGroupIds: securityGroups.map(sg => sg.securityGroupId),
        };
    }
    isQueue(deadLetterQueue) {
        return deadLetterQueue.queueArn !== undefined;
    }
    buildDeadLetterQueue(props) {
        if (!props.deadLetterQueue && !props.deadLetterQueueEnabled && !props.deadLetterTopic) {
            return undefined;
        }
        if (props.deadLetterQueue && props.deadLetterQueueEnabled === false) {
            throw Error('deadLetterQueue defined but deadLetterQueueEnabled explicitly set to false');
        }
        if (props.deadLetterTopic && (props.deadLetterQueue || props.deadLetterQueueEnabled !== undefined)) {
            throw new Error('deadLetterQueue and deadLetterTopic cannot be specified together at the same time');
        }
        let deadLetterQueue;
        if (props.deadLetterTopic) {
            deadLetterQueue = props.deadLetterTopic;
            this.addToRolePolicy(new iam.PolicyStatement({
                actions: ['sns:Publish'],
                resources: [deadLetterQueue.topicArn],
            }));
        }
        else {
            deadLetterQueue = props.deadLetterQueue || new sqs.Queue(this, 'DeadLetterQueue', {
                retentionPeriod: core_1.Duration.days(14),
            });
            this.addToRolePolicy(new iam.PolicyStatement({
                actions: ['sqs:SendMessage'],
                resources: [deadLetterQueue.queueArn],
            }));
        }
        return deadLetterQueue;
    }
    buildDeadLetterConfig(deadLetterQueue) {
        if (deadLetterQueue) {
            return {
                targetArn: this.isQueue(deadLetterQueue) ? deadLetterQueue.queueArn : deadLetterQueue.topicArn,
            };
        }
        else {
            return undefined;
        }
    }
    buildTracingConfig(props) {
        if (props.tracing === undefined || props.tracing === Tracing.DISABLED) {
            return undefined;
        }
        this.addToRolePolicy(new iam.PolicyStatement({
            actions: ['xray:PutTraceSegments', 'xray:PutTelemetryRecords'],
            resources: ['*'],
        }));
        return {
            mode: props.tracing,
        };
    }
    validateProfiling(props) {
        if (!props.runtime.supportsCodeGuruProfiling) {
            throw new Error(`CodeGuru profiling is not supported by runtime ${props.runtime.name}`);
        }
        if (props.environment && (props.environment.AWS_CODEGURU_PROFILER_GROUP_ARN || props.environment.AWS_CODEGURU_PROFILER_ENABLED)) {
            throw new Error('AWS_CODEGURU_PROFILER_GROUP_ARN and AWS_CODEGURU_PROFILER_ENABLED must not be set when profiling options enabled');
        }
    }
}
exports.Function = Function;
_a = JSII_RTTI_SYMBOL_1;
Function[_a] = { fqn: "@aws-cdk/aws-lambda.Function", version: "1.149.0" };
/** @internal */
Function._VER_PROPS = {};
/**
 * Given an opaque (token) ARN, returns a CloudFormation expression that extracts the function
 * name from the ARN.
 *
 * Function ARNs look like this:
 *
 *   arn:aws:lambda:region:account-id:function:function-name
 *
 * ..which means that in order to extract the `function-name` component from the ARN, we can
 * split the ARN using ":" and select the component in index 6.
 *
 * @returns `FnSelect(6, FnSplit(':', arn))`
 */
function extractNameFromArn(arn) {
    return core_1.Fn.select(6, core_1.Fn.split(':', arn));
}
function verifyCodeConfig(code, props) {
    // mutually exclusive
    const codeType = [code.inlineCode, code.s3Location, code.image];
    if (codeType.filter(x => !!x).length !== 1) {
        throw new Error('lambda.Code must specify exactly one of: "inlineCode", "s3Location", or "image"');
    }
    if (!!code.image === (props.handler !== handler_1.Handler.FROM_IMAGE)) {
        throw new Error('handler must be `Handler.FROM_IMAGE` when using image asset for Lambda function');
    }
    if (!!code.image === (props.runtime !== runtime_1.Runtime.FROM_IMAGE)) {
        throw new Error('runtime must be `Runtime.FROM_IMAGE` when using image asset for Lambda function');
    }
    // if this is inline code, check that the runtime supports
    if (code.inlineCode && !props.runtime.supportsInlineCode) {
        throw new Error(`Inline source not allowed for ${props.runtime.name}`);
    }
}
exports.verifyCodeConfig = verifyCodeConfig;
function undefinedIfNoKeys(struct) {
    const allUndefined = Object.values(struct).every(val => val === undefined);
    return allUndefined ? undefined : struct;
}
//# sourceMappingURL=data:application/json;base64,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