import * as React from 'react';
import { classes } from 'typestyle';
import List from '@material-ui/core/List';
import ListItem from '@material-ui/core/ListItem';
import ClearIcon from '@material-ui/icons/Clear';
import { showErrorMessage } from '@jupyterlab/apputils';
import { activeListItemClass, filterClass, filterClearClass, filterInputClass, filterWrapperClass, listItemClass, listItemIconClass, listWrapperClass, newBranchButtonClass, wrapperClass } from '../style/BranchMenu';
import { NewBranchDialog } from './NewBranchDialog';
const CHANGES_ERR_MSG = 'The current branch contains files with uncommitted changes. Please commit or discard these changes before switching to or creating another branch.';
/**
 * React component for rendering a branch menu.
 */
export class BranchMenu extends React.Component {
    /**
     * Returns a React component for rendering a branch menu.
     *
     * @param props - component properties
     * @returns React component
     */
    constructor(props) {
        super(props);
        /**
         * Callback invoked upon a change to the menu filter.
         *
         * @param event - event object
         */
        this._onFilterChange = (event) => {
            this.setState({
                filter: event.target.value
            });
        };
        /**
         * Callback invoked to reset the menu filter.
         */
        this._resetFilter = () => {
            this.setState({
                filter: ''
            });
        };
        /**
         * Callback invoked upon clicking a button to create a new branch.
         *
         * @param event - event object
         */
        this._onNewBranchClick = () => {
            if (!this.props.branching) {
                showErrorMessage('Creating a new branch is disabled', CHANGES_ERR_MSG);
                return;
            }
            this.setState({
                branchDialog: true
            });
        };
        /**
         * Callback invoked upon closing a dialog to create a new branch.
         */
        this._onNewBranchDialogClose = () => {
            this.setState({
                branchDialog: false
            });
        };
        const repo = this.props.model.pathRepository;
        this.state = {
            filter: '',
            branchDialog: false,
            current: repo ? this.props.model.currentBranch.name : '',
            branches: repo ? this.props.model.branches : []
        };
    }
    /**
     * Callback invoked immediately after mounting a component (i.e., inserting into a tree).
     */
    componentDidMount() {
        this._addListeners();
    }
    /**
     * Callback invoked when a component will no longer be mounted.
     */
    componentWillUnmount() {
        this._removeListeners();
    }
    /**
     * Renders the component.
     *
     * @returns React element
     */
    render() {
        return (React.createElement("div", { className: wrapperClass },
            React.createElement("div", { className: filterWrapperClass },
                React.createElement("div", { className: filterClass },
                    React.createElement("input", { className: filterInputClass, type: "text", onChange: this._onFilterChange, value: this.state.filter, placeholder: "Filter", title: "Filter branch menu" }),
                    this.state.filter ? (React.createElement("button", { className: filterClearClass },
                        React.createElement(ClearIcon, { titleAccess: "Clear the current filter", fontSize: "small", onClick: this._resetFilter }))) : null),
                React.createElement("input", { className: newBranchButtonClass, type: "button", title: "Create a new branch", value: "New Branch", onClick: this._onNewBranchClick })),
            React.createElement("div", { className: listWrapperClass },
                React.createElement(List, { disablePadding: true }, this._renderItems())),
            React.createElement(NewBranchDialog, { open: this.state.branchDialog, model: this.props.model, onClose: this._onNewBranchDialogClose })));
    }
    /**
     * Renders menu items.
     *
     * @returns array of React elements
     */
    _renderItems() {
        return this.state.branches.map(this._renderItem, this);
    }
    /**
     * Renders a menu item.
     *
     * @param branch - branch
     * @param idx - item index
     * @returns React element
     */
    _renderItem(branch, idx) {
        // Perform a "simple" filter... (TODO: consider implementing fuzzy filtering)
        if (this.state.filter && !branch.name.includes(this.state.filter)) {
            return null;
        }
        return (React.createElement(ListItem, { button: true, title: `Switch to branch: ${branch.name}`, className: classes(listItemClass, branch.name === this.state.current ? activeListItemClass : null), key: branch.name, onClick: this._onBranchClickFactory(branch.name) },
            React.createElement("span", { className: listItemIconClass }),
            branch.name));
    }
    /**
     * Adds model listeners.
     */
    _addListeners() {
        // When the HEAD changes, decent probability that we've switched branches:
        this.props.model.headChanged.connect(this._syncState, this);
        // When the status changes, we may have checked out a new branch (e.g., via the command-line and not via the extension) or changed repositories:
        this.props.model.statusChanged.connect(this._syncState, this);
    }
    /**
     * Removes model listeners.
     */
    _removeListeners() {
        this.props.model.headChanged.disconnect(this._syncState, this);
        this.props.model.statusChanged.disconnect(this._syncState, this);
    }
    /**
     * Syncs the component state with the underlying model.
     */
    _syncState() {
        const repo = this.props.model.pathRepository;
        this.setState({
            current: repo ? this.props.model.currentBranch.name : '',
            branches: repo ? this.props.model.branches : []
        });
    }
    /**
     * Returns a callback which is invoked upon clicking a branch name.
     *
     * @param branch - branch name
     * @returns callback
     */
    _onBranchClickFactory(branch) {
        const self = this;
        return onClick;
        /**
         * Callback invoked upon clicking a branch name.
         *
         * @private
         * @param event - event object
         */
        function onClick() {
            if (!self.props.branching) {
                showErrorMessage('Switching branches is disabled', CHANGES_ERR_MSG);
                return;
            }
            const opts = {
                branchname: branch
            };
            self.props.model
                .checkout(opts)
                .then(onResolve)
                .catch(onError);
        }
        /**
         * Callback invoked upon promise resolution.
         *
         * @private
         * @param result - result
         */
        function onResolve(result) {
            if (result.code !== 0) {
                showErrorMessage('Error switching branch', result.message);
            }
        }
        /**
         * Callback invoked upon encountering an error.
         *
         * @private
         * @param err - error
         */
        function onError(err) {
            showErrorMessage('Error switching branch', err.message);
        }
    }
}
//# sourceMappingURL=BranchMenu.js.map