import * as React from 'react';
import { classes } from 'typestyle';
import { Dialog, showDialog } from '@jupyterlab/apputils';
import { PathExt } from '@jupyterlab/coreutils';
import { 
// NOTE: keep in alphabetical order
branchIconClass, closeMenuIconClass, openMenuIconClass, pullButtonClass, pushButtonClass, refreshButtonClass, repoIconClass, toolbarButtonClass, toolbarClass, toolbarMenuButtonClass, toolbarMenuButtonEnabledClass, toolbarMenuButtonIconClass, toolbarMenuButtonSubtitleClass, toolbarMenuButtonTitleClass, toolbarMenuButtonTitleWrapperClass, toolbarMenuWrapperClass, toolbarNavClass } from '../style/Toolbar';
import { GitCredentialsForm } from '../widgets/CredentialsBox';
import { GitPullPushDialog, Operation } from '../widgets/gitPushPull';
import { BranchMenu } from './BranchMenu';
/**
 * Displays an error dialog when a Git operation fails.
 *
 * @private
 * @param model - Git extension model
 * @param operation - Git operation name
 * @returns Promise for displaying a dialog
 */
async function showGitOperationDialog(model, operation) {
    const title = `Git ${operation}`;
    let result = await showDialog({
        title: title,
        body: new GitPullPushDialog(model, operation),
        buttons: [Dialog.okButton({ label: 'DISMISS' })]
    });
    let retry = false;
    while (!result.button.accept) {
        const credentials = await showDialog({
            title: 'Git credentials required',
            body: new GitCredentialsForm('Enter credentials for remote repository', retry ? 'Incorrect username or password.' : ''),
            buttons: [Dialog.cancelButton(), Dialog.okButton({ label: 'OK' })]
        });
        if (!credentials.button.accept) {
            break;
        }
        result = await showDialog({
            title: title,
            body: new GitPullPushDialog(model, operation, credentials.value),
            buttons: [Dialog.okButton({ label: 'DISMISS' })]
        });
        retry = true;
    }
}
/**
 * React component for rendering a panel toolbar.
 */
export class Toolbar extends React.Component {
    /**
     * Returns a React component for rendering a panel toolbar.
     *
     * @param props - component properties
     * @returns React component
     */
    constructor(props) {
        super(props);
        /**
         * Callback invoked upon clicking a button to pull the latest changes.
         *
         * @param event - event object
         */
        this._onPullClick = () => {
            showGitOperationDialog(this.props.model, Operation.Pull).catch(reason => {
                console.error(`Encountered an error when pulling changes. Error: ${reason}`);
            });
        };
        /**
         * Callback invoked upon clicking a button to push the latest changes.
         *
         * @param event - event object
         */
        this._onPushClick = () => {
            showGitOperationDialog(this.props.model, Operation.Push).catch(reason => {
                console.error(`Encountered an error when pushing changes. Error: ${reason}`);
            });
        };
        /**
         * Callback invoked upon clicking a button to change the current repository.
         *
         * @param event - event object
         */
        this._onRepositoryClick = () => {
            // Toggle the repository menu:
            this.setState({
                repoMenu: !this.state.repoMenu
            });
        };
        /**
         * Callback invoked upon clicking a button to change the current branch.
         *
         * @param event - event object
         */
        this._onBranchClick = () => {
            // Toggle the branch menu:
            this.setState({
                branchMenu: !this.state.branchMenu
            });
        };
        /**
         * Callback invoked upon clicking a button to refresh a repository.
         *
         * @param event - event object
         */
        this._onRefreshClick = () => {
            this.props.refresh();
        };
        const repo = this.props.model.pathRepository;
        this.state = {
            branchMenu: false,
            repoMenu: false,
            repository: repo || '',
            branch: repo ? this.props.model.currentBranch.name : ''
        };
    }
    /**
     * Callback invoked immediately after mounting a component (i.e., inserting into a tree).
     */
    componentDidMount() {
        this._addListeners();
    }
    /**
     * Callback invoked when a component will no longer be mounted.
     */
    componentWillUnmount() {
        this._removeListeners();
    }
    /**
     * Renders the component.
     *
     * @returns React element
     */
    render() {
        return (React.createElement("div", { className: toolbarClass },
            this._renderTopNav(),
            this._renderRepoMenu(),
            this._renderBranchMenu()));
    }
    /**
     * Renders the top navigation.
     *
     * @returns React element
     */
    _renderTopNav() {
        return (React.createElement("div", { className: toolbarNavClass },
            React.createElement("button", { className: classes(toolbarButtonClass, pullButtonClass, 'jp-Icon-16'), title: 'Pull latest changes', onClick: this._onPullClick }),
            React.createElement("button", { className: classes(toolbarButtonClass, pushButtonClass, 'jp-Icon-16'), title: 'Push committed changes', onClick: this._onPushClick }),
            React.createElement("button", { className: classes(toolbarButtonClass, refreshButtonClass, 'jp-Icon-16'), title: 'Refresh the repository to detect local and remote changes', onClick: this._onRefreshClick })));
    }
    /**
     * Renders a repository menu.
     *
     * @returns React element
     */
    _renderRepoMenu() {
        return (React.createElement("div", { className: toolbarMenuWrapperClass },
            React.createElement("button", { disabled: true, className: toolbarMenuButtonClass, title: `Current repository: ${this.state.repository}`, onClick: this._onRepositoryClick },
                React.createElement("span", { className: classes(toolbarMenuButtonIconClass, repoIconClass, 'jp-Icon-16') }),
                React.createElement("div", { className: toolbarMenuButtonTitleWrapperClass },
                    React.createElement("p", { className: toolbarMenuButtonTitleClass }, "Current Repository"),
                    React.createElement("p", { className: toolbarMenuButtonSubtitleClass }, PathExt.basename(this.state.repository)))),
            this.state.repoMenu ? null : null));
    }
    /**
     * Renders a branch menu.
     *
     * @returns React element
     */
    _renderBranchMenu() {
        if (!this.state.repository) {
            return null;
        }
        return (React.createElement("div", { className: toolbarMenuWrapperClass },
            React.createElement("button", { className: classes(toolbarMenuButtonClass, toolbarMenuButtonEnabledClass), title: `Change the current branch: ${this.state.branch}`, onClick: this._onBranchClick },
                React.createElement("span", { className: classes(toolbarMenuButtonIconClass, branchIconClass, 'jp-Icon-16') }),
                React.createElement("div", { className: toolbarMenuButtonTitleWrapperClass },
                    React.createElement("p", { className: toolbarMenuButtonTitleClass }, "Current Branch"),
                    React.createElement("p", { className: toolbarMenuButtonSubtitleClass }, this.state.branch)),
                React.createElement("span", { className: classes(toolbarMenuButtonIconClass, this.state.branchMenu ? closeMenuIconClass : openMenuIconClass, 'jp-Icon-16') })),
            this.state.branchMenu ? (React.createElement(BranchMenu, { model: this.props.model, branching: this.props.branching })) : null));
    }
    /**
     * Adds model listeners.
     */
    _addListeners() {
        // When the HEAD changes, decent probability that we've switched branches:
        this.props.model.headChanged.connect(this._syncState, this);
        // When the status changes, we may have checked out a new branch (e.g., via the command-line and not via the extension) or changed repositories:
        this.props.model.statusChanged.connect(this._syncState, this);
    }
    /**
     * Removes model listeners.
     */
    _removeListeners() {
        this.props.model.headChanged.disconnect(this._syncState, this);
        this.props.model.statusChanged.disconnect(this._syncState, this);
    }
    /**
     * Syncs the component state with the underlying model.
     */
    _syncState() {
        const repo = this.props.model.pathRepository;
        this.setState({
            repository: repo || '',
            branch: repo ? this.props.model.currentBranch.name : ''
        });
    }
}
//# sourceMappingURL=Toolbar.js.map