/*
 * Copyright 2018-2020 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { FormGroup, Intent, ResizeSensor } from '@blueprintjs/core';
import { FrontendServices } from '@elyra/application';
import { DropDown } from '@elyra/ui-components';
import { ReactWidget, showDialog, Dialog } from '@jupyterlab/apputils';
import { CodeEditor } from '@jupyterlab/codeeditor';
import { InputGroup, Button } from '@jupyterlab/ui-components';
import { find } from '@lumino/algorithm';
import * as React from 'react';
const ELYRA_METADATA_EDITOR_CLASS = 'elyra-metadataEditor';
const DIRTY_CLASS = 'jp-mod-dirty';
/**
 * Metadata editor widget
 */
export class MetadataEditor extends ReactWidget {
    constructor(props) {
        super();
        this.schema = {};
        this.allMetadata = [];
        this.metadata = {};
        this.handleDropdownChange = (schemaField, value) => {
            this.handleDirtyState(true);
            this.metadata[schemaField] = value;
            if (schemaField == 'language') {
                const getMimeTypeByLanguage = this.editorServices.mimeTypeService
                    .getMimeTypeByLanguage;
                this.editor.model.mimeType = getMimeTypeByLanguage({
                    name: value,
                    codemirror_mode: value
                });
            }
            this.update();
        };
        this.editorServices = props.editorServices;
        this.namespace = props.namespace;
        this.schemaName = props.schema;
        this.onSave = props.onSave;
        this.name = props.name;
        this.handleTextInputChange = this.handleTextInputChange.bind(this);
        this.handleDropdownChange = this.handleDropdownChange.bind(this);
        this.renderField = this.renderField.bind(this);
        this.invalidForm = false;
        this.initializeMetadata();
    }
    async initializeMetadata() {
        const schemas = await FrontendServices.getSchema(this.namespace);
        for (const schema of schemas) {
            if (this.schemaName == schema.name) {
                this.schema = schema.properties.metadata.properties;
                // All metadata has a display_name field
                this.displayName = schema.properties.display_name;
                this.requiredFields = schema.properties.metadata.required;
                break;
            }
        }
        this.allMetadata = await FrontendServices.getMetadata(this.namespace);
        if (this.name) {
            for (const metadata of this.allMetadata) {
                if (this.name == metadata.name) {
                    this.metadata = metadata['metadata'];
                    this.displayName = metadata['display_name'];
                    this.title.label = this.displayName;
                    break;
                }
            }
        }
        else {
            this.displayName = '';
        }
        this.update();
    }
    /**
     * Checks that all required fields have a value before submitting the form.
     * Returns false if the form is valid. Sets any invalid fields' intent to danger
     * so that the form will highlight the input(s) causing issues in red.
     */
    hasInvalidFields() {
        this.invalidForm = false;
        if (this.displayName == null || this.displayName == '') {
            this.invalidForm = true;
        }
        for (const schemaField in this.schema) {
            if (this.requiredFields.includes(schemaField) &&
                (this.metadata[schemaField] == null ||
                    this.metadata[schemaField] == '' ||
                    this.metadata[schemaField] == [] ||
                    this.metadata[schemaField] == '(No selection)')) {
                this.invalidForm = true;
                this.schema[schemaField].uihints.intent = Intent.DANGER;
            }
            else {
                this.schema[schemaField].uihints.intent = Intent.NONE;
            }
        }
        return this.invalidForm;
    }
    onCloseRequest(msg) {
        if (this.dirty) {
            showDialog({
                title: 'Close without saving?',
                body: (React.createElement("p", null,
                    ' ',
                    `"${this.displayName}" has unsaved changes, close without saving?`,
                    ' ')),
                buttons: [Dialog.cancelButton(), Dialog.okButton()]
            }).then((response) => {
                if (response.button.accept) {
                    this.dispose();
                    super.onCloseRequest(msg);
                }
            });
        }
        else {
            this.dispose();
            super.onCloseRequest(msg);
        }
    }
    saveMetadata() {
        const newMetadata = {
            schema_name: this.schemaName,
            display_name: this.displayName,
            metadata: this.metadata
        };
        if (this.hasInvalidFields()) {
            this.update();
            return;
        }
        if (!this.name) {
            FrontendServices.postMetadata(this.namespace, JSON.stringify(newMetadata)).then((response) => {
                this.handleDirtyState(false);
                this.onSave();
                this.close();
            });
        }
        else {
            FrontendServices.putMetadata(this.namespace, this.name, JSON.stringify(newMetadata)).then((response) => {
                this.handleDirtyState(false);
                this.onSave();
                this.close();
            });
        }
    }
    handleTextInputChange(event, schemaField) {
        this.handleDirtyState(true);
        // Special case because all metadata has a display name
        if (schemaField == 'display_name') {
            this.displayName = event.nativeEvent.srcElement.value;
        }
        else {
            this.metadata[schemaField] = event.nativeEvent.srcElement.value;
        }
    }
    handleDirtyState(dirty) {
        this.dirty = dirty;
        if (this.dirty && !this.title.className.includes(DIRTY_CLASS)) {
            this.title.className += DIRTY_CLASS;
        }
        else if (!this.dirty) {
            this.title.className = this.title.className.replace(DIRTY_CLASS, '');
        }
    }
    onUpdateRequest(msg) {
        super.onUpdateRequest(msg);
        // If the update request triggered rendering a 'code' input, and the editor hasn't
        // been initialized yet, create the editor and attach it to the 'code' node
        if (!this.editor && document.getElementById('code:' + this.id) != null) {
            let initialCodeValue;
            const getMimeTypeByLanguage = this.editorServices.mimeTypeService
                .getMimeTypeByLanguage;
            // If the file already exists, initialize the code editor with the existing code
            if (this.name) {
                initialCodeValue = this.metadata['code'].join('\n');
            }
            else {
                initialCodeValue = '';
            }
            this.editor = this.editorServices.factoryService.newInlineEditor({
                host: document.getElementById('code:' + this.id),
                model: new CodeEditor.Model({
                    value: initialCodeValue,
                    mimeType: getMimeTypeByLanguage({
                        name: this.metadata['language'],
                        codemirror_mode: this.metadata['language']
                    })
                })
            });
            this.editor.model.value.changed.connect((args) => {
                this.metadata['code'] = args.text.split('\n');
                this.handleDirtyState(true);
            });
        }
    }
    getDefaultChoices(fieldName) {
        let defaultChoices = this.schema[fieldName].uihints.default_choices;
        if (defaultChoices == undefined) {
            defaultChoices = [];
        }
        for (const otherMetadata of this.allMetadata) {
            if (!find(defaultChoices, (choice) => {
                return (choice.toLowerCase() ==
                    otherMetadata.metadata[fieldName].toLowerCase());
            })) {
                defaultChoices.push(otherMetadata.metadata[fieldName]);
            }
        }
        return defaultChoices;
    }
    renderTextInput(label, description, fieldName, defaultValue, required, intent) {
        let helperText = description;
        if (intent == Intent.DANGER) {
            helperText += 'This field is required.';
        }
        return (React.createElement(FormGroup, { key: fieldName, label: label, labelInfo: required, helperText: helperText, intent: intent },
            React.createElement(InputGroup, { onChange: (event) => {
                    this.handleTextInputChange(event, fieldName);
                }, defaultValue: defaultValue, type: "text-input" })));
    }
    renderField(fieldName) {
        const uihints = this.schema[fieldName].uihints;
        let required = '(optional)';
        if (this.requiredFields && this.requiredFields.includes(fieldName)) {
            required = '(required)';
        }
        if (uihints == undefined) {
            return;
        }
        else if (uihints.field_type == 'textinput' ||
            uihints.field_type == undefined) {
            return this.renderTextInput(uihints.label, uihints.description, fieldName, this.metadata[fieldName], required, this.schema[fieldName].uihints.intent);
        }
        else if (uihints.field_type == 'dropdown') {
            return (React.createElement(DropDown, { label: uihints.label, schemaField: fieldName, description: uihints.description, required: required, intent: this.schema[fieldName].uihints.intent, choice: this.metadata[fieldName], defaultChoices: this.getDefaultChoices(fieldName), handleDropdownChange: this.handleDropdownChange }));
        }
        else if (uihints.field_type == 'code') {
            let helperText;
            if (this.schema[fieldName].uihints.intent == Intent.DANGER) {
                helperText = 'This field is required.';
            }
            return (React.createElement(FormGroup, { className: 'elyra-metadataEditor-code', labelInfo: required, label: 'Code', intent: this.schema[fieldName].uihints.intent, helperText: helperText },
                React.createElement(ResizeSensor, { onResize: () => {
                        this.editor.refresh();
                    } },
                    React.createElement("div", { id: 'code:' + this.id, className: "elyra-form-code va-va" }))));
        }
        else {
            return;
        }
    }
    render() {
        const inputElements = [];
        for (const schemaProperty in this.schema) {
            inputElements.push(this.renderField(schemaProperty));
        }
        let headerText = `Edit "${this.displayName}"`;
        if (!this.name) {
            headerText = `Add new ${this.schemaName}`;
        }
        let intent = Intent.NONE;
        if (this.displayName == '' && this.invalidForm) {
            intent = Intent.DANGER;
        }
        return (React.createElement("div", { className: ELYRA_METADATA_EDITOR_CLASS },
            React.createElement("h3", null,
                " ",
                headerText,
                " "),
            this.renderTextInput('Name', '', 'display_name', this.displayName, '(required)', intent),
            inputElements,
            React.createElement(FormGroup, null,
                React.createElement(Button, { onClick: () => {
                        this.saveMetadata();
                    } }, "Save & Close"))));
    }
}
//# sourceMappingURL=MetadataEditor.js.map