/*
 * Copyright 2018-2020 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import '../style/index.css';
import { codeSnippetIcon, ExpandableComponent, trashIcon, importIcon } from '@elyra/ui-components';
import { Clipboard, Dialog, showDialog, ReactWidget, UseSignal } from '@jupyterlab/apputils';
import { CodeCell, MarkdownCell } from '@jupyterlab/cells';
import { CodeEditor } from '@jupyterlab/codeeditor';
import { PathExt } from '@jupyterlab/coreutils';
import { DocumentWidget } from '@jupyterlab/docregistry';
import { FileEditor } from '@jupyterlab/fileeditor';
import { NotebookPanel } from '@jupyterlab/notebook';
import { addIcon, copyIcon, editIcon } from '@jupyterlab/ui-components';
import { find } from '@lumino/algorithm';
import { Signal } from '@lumino/signaling';
import React from 'react';
import { CodeSnippetService } from './CodeSnippetService';
/**
 * The CSS class added to code snippet widget.
 */
const CODE_SNIPPETS_HEADER_CLASS = 'elyra-codeSnippetsHeader';
const CODE_SNIPPETS_HEADER_BUTTON_CLASS = 'elyra-codeSnippetHeader-button';
const CODE_SNIPPET_ITEM = 'elyra-codeSnippet-item';
const METADATA_EDITOR_ID = 'elyra-metadata-editor';
const commands = {
    OPEN_METADATA_EDITOR: `${METADATA_EDITOR_ID}:open`
};
const CODE_SNIPPET_NAMESPACE = 'code-snippets';
const CODE_SNIPPET_SCHEMA = 'code-snippet';
/**
 * A React Component for code-snippets display list.
 */
class CodeSnippetDisplay extends React.Component {
    constructor() {
        super(...arguments);
        this.editors = {};
        // Handle code snippet insert into an editor
        this.insertCodeSnippet = async (snippet) => {
            var _a, _b, _c;
            const widget = this.props.getCurrentWidget();
            const snippetStr = snippet.code.join('\n');
            if (widget instanceof DocumentWidget &&
                widget.content instanceof FileEditor) {
                const documentWidget = widget;
                const fileEditor = documentWidget.content.editor;
                const markdownRegex = /^\.(md|mkdn?|mdown|markdown)$/;
                if (PathExt.extname(widget.context.path).match(markdownRegex) !== null &&
                    snippet.language.toLowerCase() !== 'markdown') {
                    // Wrap snippet into a code block when inserting it into a markdown file
                    fileEditor.replaceSelection('```' + snippet.language + '\n' + snippetStr + '\n```');
                }
                else if (widget.constructor.name == 'PythonFileEditor') {
                    this.verifyLanguageAndInsert(snippet, 'python', fileEditor);
                }
                else {
                    fileEditor.replaceSelection(snippetStr);
                }
            }
            else if (widget instanceof NotebookPanel) {
                const notebookWidget = widget;
                const notebookCell = notebookWidget.content.activeCell;
                const notebookCellEditor = notebookCell.editor;
                if (notebookCell instanceof CodeCell) {
                    const kernelInfo = await ((_b = (_a = notebookWidget.sessionContext.session) === null || _a === void 0 ? void 0 : _a.kernel) === null || _b === void 0 ? void 0 : _b.info);
                    const kernelLanguage = ((_c = kernelInfo) === null || _c === void 0 ? void 0 : _c.language_info.name) || '';
                    this.verifyLanguageAndInsert(snippet, kernelLanguage, notebookCellEditor);
                }
                else if (notebookCell instanceof MarkdownCell &&
                    snippet.language.toLowerCase() !== 'markdown') {
                    // Wrap snippet into a code block when inserting it into a markdown cell
                    notebookCellEditor.replaceSelection('```' + snippet.language + '\n' + snippetStr + '\n```');
                }
                else {
                    notebookCellEditor.replaceSelection(snippetStr);
                }
            }
            else {
                this.showErrDialog('Code snippet insert failed: Unsupported widget');
            }
        };
        // Handle language compatibility between code snippet and editor
        this.verifyLanguageAndInsert = async (snippet, editorLanguage, editor) => {
            const snippetStr = snippet.code.join('\n');
            if (editorLanguage &&
                snippet.language.toLowerCase() !== editorLanguage.toLowerCase()) {
                const result = await this.showWarnDialog(editorLanguage, snippet.displayName);
                if (result.button.accept) {
                    editor.replaceSelection(snippetStr);
                }
            }
            else {
                // Language match or editorLanguage is unavailable
                editor.replaceSelection(snippetStr);
            }
        };
        // Display warning dialog when inserting a code snippet incompatible with editor's language
        this.showWarnDialog = async (editorLanguage, snippetName) => {
            return showDialog({
                title: 'Warning',
                body: `Code snippet "${snippetName}" is incompatible with ${editorLanguage}. Continue?`,
                buttons: [Dialog.cancelButton(), Dialog.okButton()]
            });
        };
        // Display error dialog when inserting a code snippet into unsupported widget (i.e. not an editor)
        this.showErrDialog = (errMsg) => {
            return showDialog({
                title: 'Error',
                body: errMsg,
                buttons: [Dialog.okButton()]
            });
        };
        // Render display of code snippet list
        this.renderCodeSnippet = (codeSnippet) => {
            const displayName = `[${codeSnippet.language}] ${codeSnippet.displayName}`;
            const actionButtons = [
                {
                    title: 'Copy',
                    icon: copyIcon,
                    feedback: 'Copied!',
                    onClick: () => {
                        Clipboard.copyToSystem(codeSnippet.code.join('\n'));
                    }
                },
                {
                    title: 'Insert',
                    icon: importIcon,
                    onClick: () => {
                        this.insertCodeSnippet(codeSnippet);
                    }
                },
                {
                    title: 'Edit',
                    icon: editIcon,
                    onClick: () => {
                        this.props.openCodeSnippetEditor({
                            onSave: this.props.updateSnippets,
                            namespace: CODE_SNIPPET_NAMESPACE,
                            schema: CODE_SNIPPET_SCHEMA,
                            name: codeSnippet.name
                        });
                    }
                },
                {
                    title: 'Delete',
                    icon: trashIcon,
                    onClick: () => {
                        CodeSnippetService.deleteCodeSnippet(codeSnippet).then((deleted) => {
                            if (deleted) {
                                this.props.updateSnippets();
                                delete this.editors[codeSnippet.name];
                                const editorWidget = find(this.props.shell.widgets('main'), (value, index) => {
                                    return (value.id ==
                                        `${METADATA_EDITOR_ID}:${CODE_SNIPPET_NAMESPACE}:${CODE_SNIPPET_SCHEMA}:${codeSnippet.name}`);
                                });
                                if (editorWidget) {
                                    editorWidget.dispose();
                                }
                            }
                        });
                    }
                }
            ];
            return (React.createElement("div", { key: codeSnippet.name, className: CODE_SNIPPET_ITEM },
                React.createElement(ExpandableComponent, { displayName: displayName, tooltip: codeSnippet.description, actionButtons: actionButtons, onExpand: () => {
                        this.editors[codeSnippet.name].refresh();
                    } },
                    React.createElement("div", { id: codeSnippet.name }))));
        };
    }
    componentDidUpdate() {
        const editorFactory = this.props.editorServices.factoryService
            .newInlineEditor;
        const getMimeTypeByLanguage = this.props.editorServices.mimeTypeService
            .getMimeTypeByLanguage;
        this.props.codeSnippets.map((codeSnippet) => {
            if (codeSnippet.name in this.editors) {
                // Make sure code is up to date
                this.editors[codeSnippet.name].model.value.text = codeSnippet.code.join('\n');
            }
            else {
                // Add new snippets
                this.editors[codeSnippet.name] = editorFactory({
                    config: { readOnly: true },
                    host: document.getElementById(codeSnippet.name),
                    model: new CodeEditor.Model({
                        value: codeSnippet.code.join('\n'),
                        mimeType: getMimeTypeByLanguage({
                            name: codeSnippet.language,
                            codemirror_mode: codeSnippet.language
                        })
                    })
                });
            }
        });
    }
    render() {
        return (React.createElement("div", null,
            React.createElement("div", { id: "codeSnippets" },
                React.createElement("div", null, this.props.codeSnippets.map(this.renderCodeSnippet)))));
    }
}
/**
 * A widget for Code Snippets.
 */
export class CodeSnippetWidget extends ReactWidget {
    constructor(getCurrentWidget, app, editorServices) {
        super();
        this.getCurrentWidget = getCurrentWidget;
        this.renderCodeSnippetsSignal = new Signal(this);
        this.app = app;
        this.editorServices = editorServices;
        this.fetchData = this.fetchData.bind(this);
        this.updateSnippets = this.updateSnippets.bind(this);
        this.openCodeSnippetEditor = this.openCodeSnippetEditor.bind(this);
    }
    // Request code snippets from server
    async fetchData() {
        return await CodeSnippetService.findAll();
    }
    updateSnippets() {
        this.fetchData().then((codeSnippets) => {
            this.renderCodeSnippetsSignal.emit(codeSnippets);
        });
    }
    // Triggered when the widget button on side panel is clicked
    onAfterShow(msg) {
        this.updateSnippets();
    }
    addCodeSnippet() {
        this.openCodeSnippetEditor({
            onSave: this.updateSnippets,
            namespace: CODE_SNIPPET_NAMESPACE,
            schema: CODE_SNIPPET_SCHEMA
        });
    }
    openCodeSnippetEditor(args) {
        this.app.commands.execute(commands.OPEN_METADATA_EDITOR, args);
    }
    render() {
        return (React.createElement("div", null,
            React.createElement("header", { className: CODE_SNIPPETS_HEADER_CLASS },
                React.createElement("div", { style: { display: 'flex' } },
                    React.createElement(codeSnippetIcon.react, { tag: "span", width: "24px", height: "auto", verticalAlign: "middle", marginRight: "5px", paddingBottom: "2px" }),
                    React.createElement("p", null, " Code Snippets ")),
                React.createElement("button", { className: CODE_SNIPPETS_HEADER_BUTTON_CLASS, onClick: this.addCodeSnippet.bind(this), title: "Create new Code Snippet" },
                    React.createElement(addIcon.react, { tag: "span", elementPosition: "center", width: "16px" }))),
            React.createElement(UseSignal, { signal: this.renderCodeSnippetsSignal, initialArgs: [] }, (_, codeSnippets) => (React.createElement(CodeSnippetDisplay, { codeSnippets: codeSnippets, openCodeSnippetEditor: this.openCodeSnippetEditor, getCurrentWidget: this.getCurrentWidget, editorServices: this.editorServices, updateSnippets: this.updateSnippets, shell: this.app.shell })))));
    }
}
//# sourceMappingURL=CodeSnippetWidget.js.map