"use strict";
/*
 * Copyright 2018-2020 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (Object.hasOwnProperty.call(mod, k)) result[k] = mod[k];
    result["default"] = mod;
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const application_1 = require("@elyra/application");
const apputils_1 = require("@jupyterlab/apputils");
const React = __importStar(require("react"));
const utils_1 = __importDefault(require("./utils"));
class PipelineService {
    /**
     * Returns a list of external runtime configurations available as
     * `runtimes metadata`. This is used to submit the pipeline to be
     * executed on these runtimes.
     */
    static getRuntimes() {
        return __awaiter(this, void 0, void 0, function* () {
            const runtimes = yield application_1.FrontendServices.getMetadata('runtimes');
            if (Object.keys(runtimes).length === 0) {
                return application_1.FrontendServices.noMetadataError('runtimes');
            }
            return runtimes;
        });
    }
    /**
     * Return a list of configured docker images that are used as runtimes environments
     * to run the pipeline nodes.
     */
    static getRuntimeImages() {
        return __awaiter(this, void 0, void 0, function* () {
            const runtimeImages = yield application_1.FrontendServices.getMetadata('runtime-images');
            if (Object.keys(runtimeImages).length === 0) {
                return application_1.FrontendServices.noMetadataError('runtime-images');
            }
            const images = {};
            for (const image in runtimeImages) {
                const imageName = runtimeImages[image]['metadata']['image_name'];
                images[imageName] = runtimeImages[image]['display_name'];
            }
            return images;
        });
    }
    /**
     * Submit the pipeline to be executed on an external runtime (e.g. Kbeflow Pipelines)
     *
     * @param pipeline
     * @param runtime_config
     */
    static submitPipeline(pipeline, runtime_config) {
        return __awaiter(this, void 0, void 0, function* () {
            console.log('Pipeline definition:');
            console.log(pipeline);
            const response = yield application_1.RequestHandler.makePostRequest('elyra/pipeline/schedule', JSON.stringify(pipeline), true);
            const dialogTitle = 'Job submission to ' + runtime_config + ' succeeded';
            const dialogBody = (React.createElement("p", null,
                "Check the status of your pipeline at",
                ' ',
                React.createElement("a", { href: response['run_url'], target: "_blank", rel: "noopener noreferrer" }, "Run Details."),
                React.createElement("br", null),
                "The results and outputs are in the ",
                response['object_storage_path'],
                ' ',
                "working directory in",
                ' ',
                React.createElement("a", { href: response['object_storage_url'], target: "_blank", rel: "noopener noreferrer" }, "object storage"),
                "."));
            return apputils_1.showDialog({
                title: dialogTitle,
                body: dialogBody,
                buttons: [apputils_1.Dialog.okButton()]
            });
        });
    }
    /**
     * Export a pipeline to different formats (e.g. DSL, YAML, etc). These formats
     * are understood by a given runtime.
     *
     * @param pipeline
     * @param pipeline_export_format
     * @param pipeline_export_path
     * @param overwrite
     */
    static exportPipeline(pipeline, pipeline_export_format, pipeline_export_path, overwrite) {
        return __awaiter(this, void 0, void 0, function* () {
            console.log('Pipeline definition:');
            console.log(pipeline);
            console.log('Exporting pipeline to [' + pipeline_export_format + '] format');
            console.log('Overwriting existing file: ' + overwrite);
            const body = {
                pipeline: pipeline,
                export_format: pipeline_export_format,
                export_path: pipeline_export_path,
                overwrite: overwrite
            };
            yield application_1.RequestHandler.makePostRequest('elyra/pipeline/export', JSON.stringify(body), true);
            return apputils_1.showDialog({
                title: 'Pipeline export succeeded',
                body: React.createElement("p", null),
                buttons: [apputils_1.Dialog.okButton()]
            });
        });
    }
    /**
     * Verify if the given pipeline is "current" by looking on it's version, and perform
     * any conversion if needed.
     *
     * @param pipelineDefinition
     */
    static convertPipeline(pipelineDefinition) {
        const pipelineJSON = JSON.parse(JSON.stringify(pipelineDefinition));
        const currentVersion = utils_1.default.getPipelineVersion(pipelineJSON);
        if (currentVersion < 1) {
            // original pipeline definition without a version
            console.info('Migrating pipeline to the current version.');
            return this.convertPipelineV0toV1(pipelineJSON);
        }
    }
    static convertPipelineV0toV1(pipelineJSON) {
        utils_1.default.renamePipelineAppdataField(pipelineJSON.pipelines[0], 'title', 'name');
        utils_1.default.deletePipelineAppdataField(pipelineJSON.pipelines[0], 'export');
        utils_1.default.deletePipelineAppdataField(pipelineJSON.pipelines[0], 'export_format');
        utils_1.default.deletePipelineAppdataField(pipelineJSON.pipelines[0], 'export_path');
        // look into nodes
        for (const nodeKey in pipelineJSON.pipelines[0]['nodes']) {
            const node = pipelineJSON.pipelines[0]['nodes'][nodeKey];
            utils_1.default.renamePipelineAppdataField(node, 'artifact', 'filename');
            utils_1.default.renamePipelineAppdataField(node, 'image', 'runtime_image');
            utils_1.default.renamePipelineAppdataField(node, 'vars', 'env_vars');
            utils_1.default.renamePipelineAppdataField(node, 'file_dependencies', 'dependencies');
            utils_1.default.renamePipelineAppdataField(node, 'recursive_dependencies', 'include_subdirectories');
        }
        pipelineJSON.pipelines[0]['app_data']['version'] = 1;
        return pipelineJSON;
    }
}
exports.PipelineService = PipelineService;
//# sourceMappingURL=PipelineService.js.map