#!/usr/bin/env python

# pmx  Copyright Notice
# ============================
#
# The pmx source code is copyrighted, but you can freely use and
# copy it as long as you don't change or remove any of the copyright
# notices.
#
# ----------------------------------------------------------------------
# pmx is Copyright (C) 2006-2013 by Daniel Seeliger
#
#                        All Rights Reserved
#
# Permission to use, copy, modify, distribute, and distribute modified
# versions of this software and its documentation for any purpose and
# without fee is hereby granted, provided that the above copyright
# notice appear in all copies and that both the copyright notice and
# this permission notice appear in supporting documentation, and that
# the name of Daniel Seeliger not be used in advertising or publicity
# pertaining to distribution of the software without specific, written
# prior permission.
#
# DANIEL SEELIGER DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
# SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
# FITNESS.  IN NO EVENT SHALL DANIEL SEELIGER BE LIABLE FOR ANY
# SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
# RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
# CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
# CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
# ----------------------------------------------------------------------

"""Library for useful and less useful things needed by the pmx packages.
"""
import sys
import os
try:
    import cPickle as pickle
except:
    import pickle

__all__ = ['pdb_format', 'pdb_format2', 'pmx_data_file', 'PMX_DATA']

pdb_format = "%6s%5d %-4s%1s%4s%1s%4d    %8.3f%8.3f%8.3f%6.2f%6.2f"
pdb_format2 = "%6s%5d %-4s%1s%4s%1s%4s    %8.3f%8.3f%8.3f%6.2f%6.2f"

_extended = (-139., 135., 180.)
_helix = (-57., -47., 180.)
_anti_beta = (-119., 113., 180.)

PMX_DATA = os.path.join(os.path.dirname(__file__), 'data')


def pmx_data_file(filename, verbose=False):
    if os.path.isfile(filename):
        data_file = filename
    else:
        pth = PMX_DATA
        data_file = os.path.join(pth, filename)
    if not os.path.isfile(data_file):
        raise IOError("data file \"%s\" not found ".format(data_file), file=sys.stderr)
    if verbose:
        print("pmx__> Loading data file \"%s\"" % data_file, file=sys.stderr)
    if data_file.split('.')[-1] == 'pkl':
        return pickle.load(open(data_file))
    else:
        return data_file


def pmx_aamut_pairs(lst):
    """Create an array with all possible combinations of protein residue
    mutants. e.g. A2D, A2E, E2A, D2A etc etc
    """
    pairs = []
    for i in lst:
        for j in lst:
#            if i != j:
            mut = "{0}2{1}".format(i, j)
            if mut not in pairs:
                pairs.append(mut)
    return pairs


# ==============================================================================
#                       Dicts and Sets of residue names
# ==============================================================================

# -----------
# Amino Acids
# -----------
_one_letter = {
    'ALA':'A',
    'ARG':'R',
    'ASN':'N',
    'ASP':'D',
    'ASPP':'D',
    'ASPH':'D',
    'CYS':'C',
    'CYS2':'C',
    'CYN':'C',
    'CYX':'C',
    'CYM':'C',
    'GLU':'E',
    'GLUP':'E',
    'GLN':'Q',
    'GLY':'G',
    'HIS':'H',
    'HIE':'H',
    'HSE':'H',
    'HIP':'H',
    'HSP':'H',
    'HID':'H',
    'HSD':'H',
    'ILE':'I',
    'LEU':'L',
    'LYS':'K',
    'LSN':'K',
    'LYSH':'K',
    'LYP':'K',
    'MET':'M',
    'PHE':'F',
    'PRO':'P',
    'SER':'S',
    'THR':'T',
    'TRP':'W',
    'TYR':'Y',
    'VAL':'V',
    'NALA':'A',
    'NGLY':'G',
    'NSER':'S',
    'NTHR':'T',
    'NLEU':'L',
    'NILE':'I',
    'NVAL':'V',
    'NASN':'N',
    'NGLN':'Q',
    'NARG':'R',
    'NHID':'H',
    'NHIE':'H',
    'NHIP':'H',
    'NTRP':'W',
    'NPHE':'F',
    'NTYR':'Y',
    'NGLU':'E',
    'NASP':'D',
    'NLYP':'K',
    'NPRO':'P',
    'NCYN':'C',
    'NCYX':'C',
    'NMET':'M',
    'CALA':'A',
    'CGLY':'G',
    'CSER':'S',
    'CTHR':'T',
    'CLEU':'L',
    'CILE':'I',
    'CVAL':'V',
    'CASN':'N',
    'CGLN':'Q',
    'CARG':'R',
    'CHID':'H',
    'CHIE':'H',
    'CHIP':'H',
    'CTRP':'W',
    'CPHE':'F',
    'CTYR':'Y',
    'CGLU':'E',
    'CASP':'D',
    'CLYP':'K',
    'CPRO':'P',
    'CCYN':'C',
    'CCYX':'C',
    'CMET':'M'
}

_ext_one_letter = {
    'ALA':  'A',
    'ARG':  'R',
    'ASN':  'N',
    'ASP':  'D',
    'ASPH': 'B',
    'ASPP': 'B',
    'ASH':  'B',
    'CYS':  'C',
    'CYS2': 'C',
    'CYN':  'C',
    'CYX':  'C',
    'CYM':  'CM',
    'CYSH': 'C',
    'GLU':  'E',
    'GLUH': 'J',
    'GLUP': 'J',
    'GLH':  'J',
    'GLN':  'Q',
    'GLY':  'G',
    'HIS':  'H',
    'HIE':  'X',
    'HISE': 'X',
    'HSE':  'X',
    'HIP':  'Z',
    'HSP':  'Z',
    'HISH': 'Z',
    'HID':  'H',
    'HSD':  'H',
    'ILE':  'I',
    'LEU':  'L',
    'LYS':  'K',
    'LYSH': 'K',
    'LYP':  'K',
    'LYN':  'O',
    'LSN':  'O',
    'MET':  'M',
    'PHE':  'F',
    'PRO':  'P',
    'SER':  'S',
    'THR':  'T',
    'TRP':  'W',
    'TYR':  'Y',
    'VAL':  'V',
}

_aacids_dic = {
    'A':'ALA',
    'B':'ASH',
    'R':'ARG',
    'N':'ASN',
    'D':'ASP',
    'C':'CYS',
    'CM':'CYM',
    'E':'GLU',
    'J':'GLH',
    'Q':'GLN',
    'G':'GLY',
    'H':'HIS',
    'X':'HIE',
    'Z':'HIP',
    'I':'ILE',
    'L':'LEU',
    'K':'LYS',
    'O':'LYN',
    'M':'MET',
    'F':'PHE',
    'P':'PRO',
    'S':'SER',
    'T':'THR',
    'W':'TRP',
    'Y':'TYR',
    'V':'VAL'
    }

_aacids_ext_charmm = {
    'A':'ALA',
    'B':'ASPP',
    'R':'ARG',
    'N':'ASN',
    'D':'ASP',
    'C':'CYS',
    'CM':'CYM',
    'E':'GLU',
    'Q':'GLN',
    'G':'GLY',
    'H':'HSD',
    'X':'HSE',
    'Z':'HSP',
    'I':'ILE',
    'J':'GLUP',
    'L':'LEU',
    'K':'LYS',
    'O':'LSN',
    'M':'MET',
    'F':'PHE',
    'P':'PRO',
    'S':'SER',
    'T':'THR',
    'W':'TRP',
    'Y':'TYR',
    'V':'VAL'
    }

_aacids_ext_amber = {
    'A':'ALA',
    'B':'ASH',
    'R':'ARG',
    'N':'ASN',
    'D':'ASP',
    'CM':'CYM',
    'C':'CYS',
    'E':'GLU',
    'J':'GLH',
    'Q':'GLN',
    'G':'GLY',
    'H':'HID',
    'X':'HIE',
    'Z':'HIP',
    'I':'ILE',
    'L':'LEU',
    'K':'LYS',
    'O':'LYN',
    'M':'MET',
    'F':'PHE',
    'P':'PRO',
    'S':'SER',
    'T':'THR',
    'W':'TRP',
    'Y':'TYR',
    'V':'VAL'
    }

_aacids_ext_oplsaa = {
    'A':'ALA',
    'B':'ASPH',
    'R':'ARG',
    'N':'ASN',
    'D':'ASP',
    'C':'CYSH',
    'E':'GLU',
    'J':'GLUH',
    'Q':'GLN',
    'G':'GLY',
    'H':'HIS1',
    'X':'HISE',
    'Z':'HISH',
    'I':'ILE',
    'L':'LEU',
    'K':'LYSH',
    'O':'LYS',
    'M':'MET',
    'F':'PHE',
    'P':'PRO',
    'S':'SER',
    'T':'THR',
    'W':'TRP',
    'Y':'TYR',
    'V':'VAL'
    }

# ----------------
# Protein Residues
# ----------------
_protein_residues = set([
    'ALA','ARG','ASN','ASP','ASPH','ASH','ASPP','GLH','CYS','CYS2','CYN','CYX','CYM','CYSH',
    'GLU','GLUP','GLN','GLY','HIS','HIE','HIP','HID','HSE','HSD','HSP','ILE','LEU','LYS','LYN',
    'LYSH','LYP','LSN','MET','PHE','PRO','SER','THR','TRP','TYR','VAL',
    'NALA','NGLY','NSER','NTHR','NLEU','NILE','NVAL','NASN','NGLN',
    'NARG','NHID','NHIE','NHIP','NTRP','NPHE','NTYR','NGLU','NASP',
    'NLYP','NPRO','NCYN','NCYX','NMET','CALA','CGLY','CSER','CTHR',
    'CLEU','CILE','CVAL','CASN','CGLN','CARG','CHID','CHIE','CHIP',
    'CTRP','CPHE','CTYR','CGLU','CASP','CLYP','CPRO','CCYN','CCYX',
    'CMET','SEP','HISH','HIS1','HISE','GLUH'
    ])

_pmx_mut_residues = set(pmx_aamut_pairs(lst=_ext_one_letter.values()))

_protein_residues_incl_pmx_mut = _protein_residues.union(_pmx_mut_residues)

# -------------
# Nucleic Acids
# -------------
_dna_residues = set(['DA', 'DT', 'DC', 'DG', 'DA3', 'DT3', 'DC3', 'DG3', 'DA5',
                     'DT5', 'DC5', 'DG5'])

_rna_residues = set(['RA', 'RU', 'RC', 'RG', 'RA3', 'RU3', 'RC3', 'RG3', 'RA5',
                     'RU5', 'RC5', 'RG5'])

# TODO: make nucleic acids the union of dna and rna
_nucleic_acids = set([
    'G','T','A','C','DG','DT','DA','DC','DA5','DA3','DT5','DT3',
    'DG5','DG3','DA3','DA5','DC3','DC5','U','RU','RA','RC','RG',
    'RG3','RG5','RU3','RU5','RC3','RC5','RA3','RA5','DAN','RAN',
    'DTN','RUN','RGN','DCN','RCN'])

_perturbed_dna = set(['DAT', 'DAC', 'DAG', 'DCT', 'DCG', 'DCA',
                      'DTA', 'DTG', 'DTC', 'DGA', 'DGC', 'DGT',
                      'D5K', 'D5L', 'D5M', 'D5N', 'D5O', 'D5P',
                      'D5R', 'D5S', 'D5T', 'D5X', 'D5Y', 'D5Z',
                      'D3K', 'D3L', 'D3M', 'D3N', 'D3O', 'D3P',
                      'D3R', 'D3S', 'D3T', 'D3X', 'D3Y', 'D3Z'
                      ])

_perturbed_rna = set(['RAU', 'RAC', 'RAG', 'RCU', 'RCG', 'RCA',
                      'RUA', 'RUG', 'RUC', 'RGA', 'RGC', 'RGU',
                      'R5K', 'R5L', 'R5M', 'R5N', 'R5O', 'R5P',
                      'R5R', 'R5S', 'R5T', 'R5X', 'R5Y', 'R5Z',
                      'R3K', 'R3L', 'R3M', 'R3N', 'R3O', 'R3P',
                      'R3R', 'R3S', 'R3T', 'R3X', 'R3Y', 'R3Z',
                      ])

_perturbed_nucleotides = _perturbed_dna.union(_perturbed_rna)

_dna_residues_incl_pmx_mut = _dna_residues.union(_perturbed_dna)
_rna_residues_incl_pmx_mut = _rna_residues.union(_perturbed_rna)

# ------------
# Solvent/Ions
# ------------
_ions = set([
    'CA', 'Ca',
    'MG', 'Mg',
    'FE', 'Fe',
    'MN', 'Mn',
    'K', 'K+',
    'CL', 'Cl', 'ClJ', 'Cl-', 'CL-', 'ClS',
    'NA', 'Na', 'Na+', 'NaJ', 'NA+', 'NaS',
    'ZN', 'Zn', 'Zn+'
    ])

_water = set(['SOL', 'HOH', 'HO4'])

_water_and_ions = _water.union(_ions)


# ==============================================================================
#                           Atoms/Residues Parameters
# ==============================================================================
_atommass = {
    'H'  :   1.00794,
    'He' :   4.002602,
    'HE' :   4.002602,
    'Li' :   6.941,
    'LI' :   6.941,
    'Be' :   9.012182,
    'BE' :   9.012182,
    'B'  :  10.811,
    'C'  :  12.0107,
    'N'  :  14.0067,
    'O'  :  15.9994,
    'F'  :  18.9984032,
    'Ne' :  20.1797,
    'NE' :  20.1797,
    'Na' :  22.989770,
    'NA' :  22.989770,
    'Mg' :  24.3050,
    'MG' :  24.3050,
    'Al' :  26.981538,
    'AL' :  26.981538,
    'Si' :  28.0855,
    'SI' :  28.0855,
    'P'  :  30.973761,
    'S'  :  32.065,
    'Cl' :  35.453,
    'CL' :  35.453,
    'Ar' :  39.948,
    'AR' :  39.948,
    'K'  :  39.0983,
    'Ca' :  40.078,
    'CA' :  40.078,
    'Sc' :  44.955910,
    'SC' :  44.955910,
    'Ti' :  47.867,
    'TI' :  47.867,
    'V'  :  50.9415,
    'Cr' :  51.9961,
    'CR' :  51.9961,
    'Mn' :  54.938049,
    'MN' :  54.938049,
    'Fe' :  55.845,
    'FE' :  55.845,
    'Co' :  58.933200,
    'CO' :  58.933200,
    'Ni' :  58.6934,
    'NI' :  58.6934,
    'Cu' :  63.546,
    'CU' :  63.546,
    'Zn' :  65.39,
    'ZN' :  65.39,
    'Ga' :  69.723,
    'GA' :  69.723,
    'Ge' :  72.64,
    'GE' :  72.64,
    'As' :  74.92160,
    'AS' :  74.92160,
    'Se' :  78.96,
    'SE' :  78.96,
    'Br' :  79.904,
    'BR' :  79.904,
    'Kr' :  83.80,
    'KR' :  83.80,
    'Rb' :  85.4678,
    'RB' :  85.4678,
    'Sr' :  87.62,
    'SR' :  87.62,
    'Y'  :  88.90585,
    'Zr' :  91.224,
    'ZR' :  91.224,
    'Nb' :  92.90638,
    'NB' :  92.90638,
    'Mo' :  95.94,
    'MO' :  95.94,
    'Tc' :  98,
    'TC' :  98,
    'Ru' : 101.07,
    'RU' : 101.07,
    'Rh' : 102.90550,
    'RH' : 102.90550,
    'Pd' : 106.42,
    'PD' : 106.42,
    'Ag' : 107.8682,
    'AG' : 107.8682,
    'Cd' : 112.411,
    'CD' : 112.411,
    'In' : 114.818,
    'IN' : 114.818,
    'Sn' : 118.710,
    'SN' : 118.710,
    'Sb' : 121.760,
    'SB' : 121.760,
    'Te' : 127.60,
    'TE' : 127.60,
    'I'  : 126.90447,
    'Xe' : 131.293,
    'XE' : 131.293,
    'Cs' : 132.90545,
    'CS' : 132.90545,
    'Ba' : 137.327,
    'BA' : 137.327,
    'La' : 138.9055,
    'LA' : 138.9055,
    'Ce' : 140.116,
    'CE' : 140.116,
    'Pr' : 140.90765,
    'PR' : 140.90765,
    'Nd' : 144.24,
    'ND' : 144.24,
    'Pm' : 145,
    'PM' : 145,
    'Sm' : 150.36,
    'SM' : 150.36,
    'Eu' : 151.964,
    'EU' : 151.964,
    'Gd' : 157.25,
    'GD' : 157.25,
    'Tb' : 158.92534,
    'TB' : 158.92534,
    'Dy' : 162.50,
    'DY' : 162.50,
    'Ho' : 164.93032,
    'HO' : 164.93032,
    'Er' : 167.259,
    'ER' : 167.259,
    'Tm' : 168.93421,
    'TM' : 168.93421,
    'Yb' : 173.04,
    'YB' : 173.04,
    'Lu' : 174.967,
    'LU' : 174.967,
    'Hf' : 178.49,
    'HF' : 178.49,
    'Ta' : 180.9479,
    'TA' : 180.9479,
    'W'  : 183.84,
    'Re' : 186.207,
    'RE' : 186.207,
    'Os' : 190.23,
    'OS' : 190.23,
    'Ir' : 192.217,
    'IR' : 192.217,
    'Pt' : 195.078,
    'PT' : 195.078,
    'Au' : 196.96655,
    'AU' : 196.96655,
    'Hg' : 200.59,
    'HG' : 200.59,
    'Tl' : 204.3833,
    'TL' : 204.3833,
    'Pb' : 207.2,
    'PB' : 207.2,
    'Bi' : 208.98038,
    'BI' : 208.98038,
    'Po' : 208.98,
    'PO' : 208.98,
    'At' : 209.99,
    'AT' : 209.99,
    'Rn' : 222.02,
    'RN' : 222.02,
    'Fr' : 223.02,
    'FR' : 223.02,
    'Ra' : 226.03,
    'RA' : 226.03,
    'Ac' : 227.03,
    'AC' : 227.03,
    'Th' : 232.0381,
    'TH' : 232.0381,
    'Pa' : 231.03588,
    'PA' : 231.03588,
    'U'  : 238.02891,
    'Np' : 237.05,
    'NP' : 237.05,
    'Pu' : 244.06,
    'PU' : 244.06,
    'Am' : 243.06,
    'AM' : 243.06,
    'Cm' : 247.07,
    'CM' : 247.07,
    'Bk' : 247.07,
    'BK' : 247.07,
    'Cf' : 251.08,
    'CF' : 251.08,
    'Es' : 252.08,
    'ES' : 252.08,
    'Fm' : 257.10,
    'FM' : 257.10,
    'Md' : 258.10,
    'MD' : 258.10,
    'No' : 259.10,
    'NO' : 259.10,
    'Lr' : 262.11,
    'LR' : 262.11,
    'Rf' : 261.11,
    'RF' : 261.11,
    'Db' : 262.11,
    'DB' : 262.11,
    'Sg' : 266.12,
    'SG' : 266.12,
    'Bh' : 264.12,
    'BH' : 264.12,
    'Hs' : 269.13,
    'HS' : 269.13,
    'Mt' : 268.14,
    'MT' : 268.14,
    }

_bond_contr = {
    'C':0.9,
    'H':0.35,
    'N':0.9,
    'O':0.9,
    'S':1.1,
    'P':1.1,
    'M':1.5,
    'I':1.5,
    'F':0.7,
    'CL':1.0,
    'BR':1.25
    }

_aacids = {

    'ALA' :
    (
    ('N', [   -2.983,  -4.446,   6.956 ]),
    ('H', [   -3.447,  -5.331,   7.094 ]),
    ('CA', [   -1.682,  -4.415,   7.599 ]),
    ('HA', [   -1.770,  -4.912,   8.567 ]),
    ('CB', [   -0.699,  -5.179,   6.703 ]),
    ('HB1', [    0.294,  -5.231,   7.152 ]),
    ('HB2', [   -1.031,  -6.203,   6.536 ]),
    ('HB3', [   -0.595,  -4.696,   5.730 ]),
    ('C', [   -1.219,  -2.972,   7.862 ]),
    ('O', [   -1.895,  -2.016,   7.478 ])
    ),
    'ARG' :
    (
    ('N', [   -0.095,  -2.820,   8.564 ]),
    ('H', [    0.440,  -3.638,   8.837 ]),
    ('CA', [    0.554,  -1.566,   8.925 ]),
    ('HA', [    0.713,  -0.997,   8.006 ]),
    ('CB', [   -0.329,  -0.766,   9.917 ]),
    ('HB1', [   -1.368,  -0.779,   9.590 ]),
    ('HB2', [   -0.319,  -1.266,  10.887 ]),
    ('CG', [    0.081,   0.703,  10.085 ]),
    ('HG1', [    1.087,   0.778,  10.497 ]),
    ('HG2', [    0.104,   1.178,   9.102 ]),
    ('CD', [   -0.877,   1.461,  11.005 ]),
    ('HD1', [   -1.890,   1.431,  10.598 ]),
    ('HD2', [   -0.899,   0.987,  11.987 ]),
    ('NE', [   -0.444,   2.857,  11.153 ]),
    ('HE', [    0.476,   3.016,  10.758 ]),
    ('CZ', [   -1.052,   3.809,  11.870 ]),
    ('NH1', [   -2.156,   3.532,  12.568 ]),
    ('HH11', [   -2.533,   2.595,  12.570 ]),
    ('HH12', [   -2.636,   4.213,  13.140 ]),
    ('NH2', [   -0.551,   5.045,  11.878 ]),
    ('HH21', [    0.268,   5.283,  11.340 ]),
    ('HH22', [   -0.963,   5.794,  12.417 ]),
    ('C', [    1.922,  -1.920,   9.531 ]),
    ('O', [    2.133,  -3.070,   9.916 ])
    ),
    'ASP' :
    (
    ('N', [    2.810,  -0.929,   9.643 ]),
    ('H', [    2.577,   0.002,   9.335 ]),
    ('CA', [    4.107,  -0.961,  10.319 ]),
    ('HA', [    3.965,  -1.440,  11.292 ]),
    ('CB', [    5.138,  -1.770,   9.502 ]),
    ('HB1', [    4.775,  -2.779,   9.310 ]),
    ('HB2', [    5.295,  -1.294,   8.536 ]),
    ('CG', [    6.485,  -1.891,  10.205 ]),
    ('OD1', [    6.469,  -1.920,  11.449 ]),
    ('OD2', [    7.505,  -1.873,   9.480 ]),
    ('C', [    4.529,   0.498,  10.585 ]),
    ('O', [    3.960,   1.407,   9.962 ])
    ),
    'ASH' :
    (
    ('N', [    2.810,  -0.929,   9.643 ]),
    ('H', [    2.577,   0.002,   9.335 ]),
    ('CA', [    4.107,  -0.961,  10.319 ]),
    ('HA', [    3.965,  -1.440,  11.292 ]),
    ('CB', [    5.138,  -1.770,   9.502 ]),
    ('HB1', [    4.775,  -2.779,   9.310 ]),
    ('HB2', [    5.295,  -1.294,   8.536 ]),
    ('CG', [    6.485,  -1.891,  10.205 ]),
    ('OD1', [    6.469,  -1.920,  11.449 ]),
    ('OD2', [    7.505,  -1.873,   9.480 ]),
    ('HD2', [    8.321,  -1.959,   10.051 ]),
    ('C', [    4.529,   0.498,  10.585 ]),
    ('O', [    3.960,   1.407,   9.962 ])
    ),
    'ASN' :
    (
    ('N', [    5.477,   0.732,  11.502 ]),
    ('H', [    5.990,  -0.105,  11.820 ]),
    ('CA', [    6.087,   2.013,  11.884 ]),
    ('HA', [    6.256,   2.572,  10.960 ]),
    ('CB', [    5.203,   2.866,  12.843 ]),
    ('HB1', [    5.030,   2.304,  13.761 ]),
    ('HB2', [    5.759,   3.755,  13.148 ]),
    ('CG', [    3.861,   3.345,  12.278 ]),
    ('OD1', [    2.781,   2.900,  12.665 ]),
    ('ND2', [    3.885,   4.288,  11.344 ]),
    ('HD21', [    4.769,   4.622,  10.995 ]),
    ('HD22', [    3.029,   4.509,  10.878 ]),
    ('C', [    7.478,   1.762,  12.495 ]),
    ('O', [    7.772,   0.654,  12.925 ])
    ),
    'CYS' :
    (
    ('N', [    8.330,   2.791,  12.593 ]),
    ('H', [    8.046,   3.726,  12.336 ]),
    ('CA', [    9.646,   2.747,  13.235 ]),
    ('HA', [    9.523,   2.268,  14.209 ]),
    ('CB', [   10.646,   1.925,  12.381 ]),
    ('HB1', [   10.211,   0.975,  12.072 ]),
    ('HB2', [   10.894,   2.470,  11.470 ]),
    ('SG', [   12.188,   1.536,  13.272 ]),
    ('HG', [   12.782,   0.910,  12.254 ]),
    ('C', [   10.094,   4.200,  13.474 ]),
    ('O', [    9.519,   5.126,  12.895 ])
    ),
    'CYN' :
    (
    ('N', [    8.330,   2.791,  12.593 ]),
    ('H', [    8.046,   3.726,  12.336 ]),
    ('CA', [    9.646,   2.747,  13.235 ]),
    ('HA', [    9.523,   2.268,  14.209 ]),
    ('CB', [   10.646,   1.925,  12.381 ]),
    ('HB1', [   10.211,   0.975,  12.072 ]),
    ('HB2', [   10.894,   2.470,  11.470 ]),
    ('SG', [   12.188,   1.536,  13.272 ]),
    ('HG', [   12.782,   0.910,  12.254 ]),
    ('C', [   10.094,   4.200,  13.474 ]),
    ('O', [    9.519,   5.126,  12.895 ])
    ),
    'GLU' :
    (
    ('N', [   11.129,   4.390,  14.297 ]),
    ('H', [   11.607,   3.575,  14.666 ]),
    ('CA', [   11.775,   5.646,  14.655 ]),
    ('HA', [   11.908,   6.217,  13.734 ]),
    ('CB', [   10.882,   6.463,  15.608 ]),
    ('HB1', [   11.322,   7.448,  15.772 ]),
    ('HB2', [    9.922,   6.659,  15.129 ]),
    ('CG', [   10.646,   5.795,  16.981 ]),
    ('HG1', [   10.276,   4.779,  16.850 ]),
    ('HG2', [   11.576,   5.723,  17.541 ]),
    ('CD', [    9.637,   6.537,  17.850 ]),
    ('OE1', [    9.152,   7.598,  17.418 ]),
    ('OE2', [    9.380,   6.011,  18.958 ]),
    ('C', [   13.184,   5.345,  15.197 ]),
    ('O', [   13.544,   4.179,  15.372 ])
    ),
    'GLH' :
    (
    ('N', [   11.129,   4.390,  14.297 ]),
    ('H', [   11.607,   3.575,  14.666 ]),
    ('CA', [   11.775,   5.646,  14.655 ]),
    ('HA', [   11.908,   6.217,  13.734 ]),
    ('CB', [   10.882,   6.463,  15.608 ]),
    ('HB1', [   11.322,   7.448,  15.772 ]),
    ('HB2', [    9.922,   6.659,  15.129 ]),
    ('CG', [   10.646,   5.795,  16.981 ]),
    ('HG1', [   10.276,   4.779,  16.850 ]),
    ('HG2', [   11.576,   5.723,  17.541 ]),
    ('CD', [    9.637,   6.537,  17.850 ]),
    ('OE1', [    9.152,   7.598,  17.418 ]),
    ('OE2', [    9.380,   6.011,  18.958 ]),
    ('HE2', [    8.714,   6.572,  19.447 ]),
    ('C', [   13.184,   5.345,  15.197 ]),
    ('O', [   13.544,   4.179,  15.372 ])
    ),
    'GLN' :
    (
    ('N', [   13.984,   6.387,  15.452 ]),
    ('H', [   13.607,   7.322,  15.344 ]),
    ('CA', [   15.305,   6.354,  16.076 ]),
    ('HA', [   15.208,   5.806,  17.017 ]),
    ('CB', [   16.334,   5.638,  15.161 ]),
    ('HB1', [   15.950,   4.651,  14.905 ]),
    ('HB2', [   16.409,   6.192,  14.224 ]),
    ('CG', [   17.746,   5.445,  15.766 ]),
    ('HG1', [   18.213,   6.403,  15.987 ]),
    ('HG2', [   17.674,   4.905,  16.711 ]),
    ('CD', [   18.732,   4.694,  14.870 ]),
    ('OE1', [   19.880,   4.504,  15.249 ]),
    ('NE2', [   18.329,   4.265,  13.679 ]),
    ('HE21', [   17.378,   4.404,  13.378 ]),
    ('HE22', [   18.999,   3.785,  13.099 ]),
    ('C', [   15.697,   7.807,  16.404 ]),
    ('O', [   15.147,   8.743,  15.823 ])
    ),
    'GLY' :
    (
    ('N', [   16.653,   8.006,  17.314 ]),
    ('H', [   17.133,   7.204,  17.704 ]),
    ('CA', [   17.344,   9.266,  17.550 ]),
    ('HA1', [   17.459,   9.812,  16.610 ]),
    ('HA2', [   16.770,   9.889,  18.235 ]),
    ('C', [   18.739,   8.968,  18.109 ]),
    ('O', [   19.039,   7.810,  18.406 ])
    ),
    'HIS' :
    (
    ('N', [   19.586,   9.994,  18.257 ]),
    ('H', [   19.289,  10.931,  18.012 ]),
    ('CA', [   20.955,   9.940,  18.779 ]),
    ('HA', [   20.921,   9.485,  19.773 ]),
    ('CB', [   21.884,   9.126,  17.850 ]),
    ('HB1', [   21.384,   8.885,  16.913 ]),
    ('HB2', [   22.764,   9.706,  17.565 ]),
    ('CG', [   22.387,   7.852,  18.463 ]),
    ('ND1', [   23.475,   7.824,  19.341 ]),
    ('CD2', [   21.895,   6.573,  18.328 ]),
    ('HD2', [   21.049,   6.203,  17.762 ]),
    ('CE1', [   23.617,   6.546,  19.671 ]),
    ('HE1', [   24.394,   6.184,  20.329 ]),
    ('NE2', [   22.700,   5.760,  19.106 ]),
    ('HE2', [   22.608,   4.761,  19.226 ]),
    ('C', [   21.459,  11.379,  18.941 ]),
    ('O', [   21.037,  12.255,  18.185 ])
    ),
    'HIE' :
    (
    ('N', [   19.586,   9.994,  18.257 ]),
    ('H', [   19.289,  10.931,  18.012 ]),
    ('CA', [   20.955,   9.940,  18.779 ]),
    ('HA', [   20.921,   9.485,  19.773 ]),
    ('CB', [   21.884,   9.126,  17.850 ]),
    ('HB1', [   21.384,   8.885,  16.913 ]),
    ('HB2', [   22.764,   9.706,  17.565 ]),
    ('CG', [   22.387,   7.852,  18.463 ]),
    ('ND1', [   23.475,   7.824,  19.341 ]),
    ('CD2', [   21.895,   6.573,  18.328 ]),
    ('HD2', [   21.049,   6.203,  17.762 ]),
    ('CE1', [   23.617,   6.546,  19.671 ]),
    ('HE1', [   24.394,   6.184,  20.329 ]),
    ('NE2', [   22.700,   5.760,  19.106 ]),
    ('HE2', [   22.608,   4.761,  19.226 ]),
    ('C', [   21.459,  11.379,  18.941 ]),
    ('O', [   21.037,  12.255,  18.185 ])
    ),
    'HID' :
    (
    ('N', [   19.586,   9.994,  18.257 ]),
    ('H', [   19.289,  10.931,  18.012 ]),
    ('CA', [   20.955,   9.940,  18.779 ]),
    ('HA', [   20.921,   9.485,  19.773 ]),
    ('CB', [   21.884,   9.126,  17.850 ]),
    ('HB1', [   21.384,   8.885,  16.913 ]),
    ('HB2', [   22.764,   9.706,  17.565 ]),
    ('CG', [   22.387,   7.852,  18.463 ]),
    ('ND1', [   23.475,   7.824,  19.341 ]),
    ('HD1', [   24.026,   8.598,  19.653 ]),
    ('CD2', [   21.895,   6.573,  18.328 ]),
    ('HD2', [   21.049,   6.203,  17.762 ]),
    ('CE1', [   23.617,   6.546,  19.671 ]),
    ('HE1', [   24.394,   6.184,  20.329 ]),
    ('NE2', [   22.700,   5.760,  19.106 ]),
    ('C', [   21.459,  11.379,  18.941 ]),
    ('O', [   21.037,  12.255,  18.185 ])
    ),
    'HIP' :
    (
    ('N', [   19.586,   9.994,  18.257 ]),
    ('H', [   19.289,  10.931,  18.012 ]),
    ('CA', [   20.955,   9.940,  18.779 ]),
    ('HA', [   20.921,   9.485,  19.773 ]),
    ('CB', [   21.884,   9.126,  17.850 ]),
    ('HB1', [   21.384,   8.885,  16.913 ]),
    ('HB2', [   22.764,   9.706,  17.565 ]),
    ('CG', [   22.387,   7.852,  18.463 ]),
    ('ND1', [   23.475,   7.824,  19.341 ]),
    ('HD1', [   24.026,   8.598,  19.653 ]),
    ('CD2', [   21.895,   6.573,  18.328 ]),
    ('HD2', [   21.049,   6.203,  17.762 ]),
    ('CE1', [   23.617,   6.546,  19.671 ]),
    ('HE1', [   24.394,   6.184,  20.329 ]),
    ('NE2', [   22.700,   5.760,  19.106 ]),
    ('HE2', [   22.608,   4.761,  19.226 ]),
    ('C', [   21.459,  11.379,  18.941 ]),
    ('O', [   21.037,  12.255,  18.185 ])
    ),
    'ILE' :
    (
    ('N', [   22.324,  11.629,  19.931 ]),
    ('H', [   22.723,  10.840,  20.429 ]),
    ('CA', [   22.931,  12.912,  20.308 ]),
    ('HA', [   23.149,  13.461,  19.389 ]),
    ('CB', [   22.007,  13.778,  21.234 ]),
    ('HB', [   21.933,  13.244,  22.183 ]),
    ('CG1', [   20.557,  13.962,  20.720 ]),
    ('HG11', [   20.557,  14.524,  19.785 ]),
    ('HG12', [   20.117,  12.994,  20.496 ]),
    ('CG2', [   22.645,  15.157,  21.532 ]),
    ('HG21', [   21.993,  15.806,  22.110 ]),
    ('HG22', [   23.553,  15.068,  22.126 ]),
    ('HG23', [   22.897,  15.683,  20.610 ]),
    ('CD1', [   19.583,  14.585,  21.728 ]),
    ('HD11', [   18.579,  14.615,  21.307 ]),
    ('HD12', [   19.539,  13.991,  22.641 ]),
    ('HD13', [   19.847,  15.607,  21.996 ]),
    ('C', [   24.267,  12.571,  20.995 ]),
    ('O', [   24.368,  11.498,  21.586 ])
    ),
    'LEU' :
    (
    ('N', [   25.254,  13.482,  20.971 ]),
    ('H', [   25.045,  14.387,  20.571 ]),
    ('CA', [   26.521,  13.447,  21.710 ]),
    ('HA', [   26.339,  12.960,  22.670 ]),
    ('CB', [   27.628,  12.676,  20.943 ]),
    ('HB1', [   27.807,  13.202,  20.004 ]),
    ('HB2', [   28.564,  12.744,  21.498 ]),
    ('CG', [   27.367,  11.184,  20.632 ]),
    ('HG', [   26.451,  11.112,  20.046 ]),
    ('CD1', [   28.486,  10.614,  19.751 ]),
    ('HD11', [   28.261,   9.589,  19.455 ]),
    ('HD12', [   28.612,  11.197,  18.839 ]),
    ('HD13', [   29.440,  10.606,  20.279 ]),
    ('CD2', [   27.207,  10.313,  21.892 ]),
    ('HD21', [   26.860,   9.315,  21.621 ]),
    ('HD22', [   28.143,  10.208,  22.439 ]),
    ('HD23', [   26.473,  10.711,  22.590 ]),
    ('C', [   26.945,  14.893,  22.048 ]),
    ('O', [   26.307,  15.843,  21.593 ])
    ),
    'LYS' :
    (
    ('N', [   28.024,  15.069,  22.828 ]),
    ('H', [   28.522,  14.250,  23.158 ]),
    ('CA', [   28.671,  16.335,  23.185 ]),
    ('HA', [   28.871,  16.877,  22.258 ]),
    ('CB', [   27.753,  17.188,  24.100 ]),
    ('HB1', [   26.775,  17.308,  23.632 ]),
    ('HB2', [   27.576,  16.641,  25.027 ]),
    ('CG', [   28.306,  18.591,  24.416 ]),
    ('HG1', [   29.274,  18.511,  24.906 ]),
    ('HG2', [   28.473,  19.131,  23.481 ]),
    ('CD', [   27.405,  19.412,  25.340 ]),
    ('HD1', [   26.445,  19.579,  24.849 ]),
    ('HD2', [   27.202,  18.830,  26.242 ]),
    ('CE', [   28.079,  20.742,  25.711 ]),
    ('HE1', [   29.006,  20.550,  26.255 ]),
    ('HE2', [   28.355,  21.287,  24.806 ]),
    ('NZ', [   27.201,  21.587,  26.537 ]),
    ('HZ1', [   27.673,  22.454,  26.755 ]),
    ('HZ2', [   26.350,  21.791,  26.032 ]),
    ('HZ3', [   26.972,  21.110,  27.398 ]),
    ('C', [   30.019,  16.018,  23.866 ]),
    ('O', [   30.171,  14.936,  24.428 ])
    ),
    'LYP' :
    (
    ('N', [   28.024,  15.069,  22.828 ]),
    ('H', [   28.522,  14.250,  23.158 ]),
    ('CA', [   28.671,  16.335,  23.185 ]),
    ('HA', [   28.871,  16.877,  22.258 ]),
    ('CB', [   27.753,  17.188,  24.100 ]),
    ('HB1', [   26.775,  17.308,  23.632 ]),
    ('HB2', [   27.576,  16.641,  25.027 ]),
    ('CG', [   28.306,  18.591,  24.416 ]),
    ('HG1', [   29.274,  18.511,  24.906 ]),
    ('HG2', [   28.473,  19.131,  23.481 ]),
    ('CD', [   27.405,  19.412,  25.340 ]),
    ('HD1', [   26.445,  19.579,  24.849 ]),
    ('HD2', [   27.202,  18.830,  26.242 ]),
    ('CE', [   28.079,  20.742,  25.711 ]),
    ('HE1', [   29.006,  20.550,  26.255 ]),
    ('HE2', [   28.355,  21.287,  24.806 ]),
    ('NZ', [   27.201,  21.587,  26.537 ]),
    ('HZ1', [   27.673,  22.454,  26.755 ]),
    ('HZ2', [   26.350,  21.791,  26.032 ]),
    ('HZ3', [   26.972,  21.110,  27.398 ]),
    ('C', [   30.019,  16.018,  23.866 ]),
    ('O', [   30.171,  14.936,  24.428 ])
    ),
    'LYN' :
    (
    ('N', [   28.024,  15.069,  22.828 ]),
    ('H', [   28.522,  14.250,  23.158 ]),
    ('CA', [   28.671,  16.335,  23.185 ]),
    ('HA', [   28.871,  16.877,  22.258 ]),
    ('CB', [   27.753,  17.188,  24.100 ]),
    ('HB1', [   26.775,  17.308,  23.632 ]),
    ('HB2', [   27.576,  16.641,  25.027 ]),
    ('CG', [   28.306,  18.591,  24.416 ]),
    ('HG1', [   29.274,  18.511,  24.906 ]),
    ('HG2', [   28.473,  19.131,  23.481 ]),
    ('CD', [   27.405,  19.412,  25.340 ]),
    ('HD1', [   26.445,  19.579,  24.849 ]),
    ('HD2', [   27.202,  18.830,  26.242 ]),
    ('CE', [   28.079,  20.742,  25.711 ]),
    ('HE1', [   29.006,  20.550,  26.255 ]),
    ('HE2', [   28.355,  21.287,  24.806 ]),
    ('NZ', [   27.201,  21.587,  26.537 ]),
    ('HZ1', [   27.673,  22.454,  26.755 ]),
    ('HZ2', [   26.350,  21.791,  26.032 ]),
    ('C', [   30.019,  16.018,  23.866 ]),
    ('O', [   30.171,  14.936,  24.428 ])
    ),
    'MET' :
    (
    ('N', [   30.974,  16.962,  23.847 ]),
    ('H', [   30.754,  17.857,  23.430 ]),
    ('CA', [   32.271,  16.973,  24.542 ]),
    ('HA', [   32.158,  16.446,  25.492 ]),
    ('CB', [   33.362,  16.275,  23.703 ]),
    ('HB1', [   33.496,  16.807,  22.761 ]),
    ('HB2', [   34.316,  16.340,  24.227 ]),
    ('CG', [   33.068,  14.800,  23.414 ]),
    ('HG1', [   32.817,  14.275,  24.337 ]),
    ('HG2', [   32.198,  14.731,  22.763 ]),
    ('SD', [   34.392,  13.901,  22.581 ]),
    ('CE', [   33.436,  12.421,  22.159 ]),
    ('HE1', [   34.064,  11.698,  21.639 ]),
    ('HE2', [   33.036,  11.955,  23.060 ]),
    ('HE3', [   32.601,  12.685,  21.510 ]),
    ('C', [   32.632,  18.436,  24.873 ]),
    ('O', [   31.926,  19.328,  24.407 ])
    ),
    'PHE' :
    (
    ('N', [   33.691,  18.694,  25.659 ]),
    ('H', [   34.285,  17.921,  25.946 ]),
    ('CA', [   34.271,  20.007,  26.025 ]),
    ('HA', [   34.378,  20.568,  25.094 ]),
    ('CB', [   33.300,  20.802,  26.988 ]),
    ('HB1', [   32.433,  20.179,  27.207 ]),
    ('HB2', [   33.761,  20.929,  27.968 ]),
    ('CG', [   32.749,  22.158,  26.481 ]),
    ('CD1', [   31.655,  22.182,  25.588 ]),
    ('HD1', [   31.146,  21.273,  25.334 ]),
    ('CD2', [   33.283,  23.414,  26.877 ]),
    ('HD2', [   34.010,  23.535,  27.661 ]),
    ('CE1', [   31.217,  23.383,  24.996 ]),
    ('HE1', [   30.401,  23.377,  24.290 ]),
    ('CE2', [   32.901,  24.606,  26.238 ]),
    ('HE2', [   33.402,  25.533,  26.485 ]),
    ('CZ', [   31.874,  24.592,  25.283 ]),
    ('HZ', [   31.585,  25.510,  24.792 ]),
    ('C', [   35.732,  19.747,  26.542 ]),
    ('O', [   36.101,  18.575,  26.655 ])
    ),
    'PRO' :
    (
    ('N', [   36.562,  20.776,  26.860 ]),
    ('CA', [   37.872,  20.686,  27.534 ]),
    ('HA', [   38.508,  19.929,  27.073 ]),
    ('CB', [   38.491,  22.094,  27.361 ]),
    ('HB1', [   39.202,  22.372,  28.141 ]),
    ('HB2', [   39.010,  22.172,  26.405 ]),
    ('CG', [   37.296,  23.024,  27.352 ]),
    ('HG1', [   37.015,  23.269,  28.380 ]),
    ('HG2', [   37.454,  23.966,  26.826 ]),
    ('CD', [   36.198,  22.170,  26.738 ]),
    ('HD1', [   35.386,  22.226,  27.427 ]),
    ('HD2', [   35.911,  22.460,  25.725 ]),
    ('C', [   37.694,  20.372,  29.037 ]),
    ('O', [   36.575,  20.369,  29.556 ])
    ),
    'SER' :
    (
    ('N', [   38.802,  20.160,  29.750 ]),
    ('H', [   39.715,  20.217,  29.317 ]),
    ('CA', [   38.879,  19.902,  31.184 ]),
    ('HA', [   38.288,  20.668,  31.690 ]),
    ('CB', [   38.278,  18.510,  31.500 ]),
    ('HB1', [   38.190,  18.382,  32.579 ]),
    ('HB2', [   37.267,  18.425,  31.100 ]),
    ('OG', [   39.069,  17.452,  30.983 ]),
    ('HG', [   38.532,  16.682,  30.878 ]),
    ('C', [   40.346,  20.078,  31.621 ]),
    ('O', [   41.224,  20.117,  30.758 ])
    ),
    'THR' :
    (
    ('N', [   40.597,  20.176,  32.935 ]),
    ('H', [   39.835,  20.071,  33.591 ]),
    ('CA', [   41.900,  20.250,  33.601 ]),
    ('HA', [   42.556,  19.499,  33.154 ]),
    ('CB', [   42.535,  21.665,  33.464 ]),
    ('HB', [   41.822,  22.407,  33.825 ]),
    ('OG1', [   42.879,  21.978,  32.131 ]),
    ('HG1', [   42.964,  22.913,  32.034 ]),
    ('CG2', [   43.861,  21.820,  34.217 ]),
    ('HG21', [   44.311,  22.791,  34.025 ]),
    ('HG22', [   43.723,  21.738,  35.295 ]),
    ('HG23', [   44.579,  21.054,  33.916 ]),
    ('C', [   41.662,  19.864,  35.077 ]),
    ('O', [   40.564,  20.090,  35.587 ])
    ),
    'TRP' :
    (
    ('N', [   42.684,  19.332,  35.765 ]),
    ('H', [   43.592,  19.274,  35.320 ]),
    ('CA', [   42.718,  18.960,  37.182 ]),
    ('HA', [   42.105,  19.670,  37.743 ]),
    ('CB', [   42.168,  17.537,  37.399 ]),
    ('HB1', [   42.745,  16.836,  36.795 ]),
    ('HB2', [   42.313,  17.232,  38.435 ]),
    ('CG', [   40.717,  17.366,  37.083 ]),
    ('CD1', [   40.214,  16.856,  35.937 ]),
    ('HD1', [   40.819,  16.494,  35.117 ]),
    ('CD2', [   39.571,  17.783,  37.881 ]),
    ('NE1', [   38.837,  16.952,  35.960 ]),
    ('HE1', [   38.259,  16.674,  35.181 ]),
    ('CE2', [   38.384,  17.526,  37.131 ]),
    ('CE3', [   39.415,  18.381,  39.153 ]),
    ('HE3', [   40.291,  18.597,  39.746 ]),
    ('CZ2', [   37.107,  17.853,  37.616 ]),
    ('HZ2', [   36.228,  17.658,  37.022 ]),
    ('CZ3', [   38.138,  18.702,  39.657 ]),
    ('HZ3', [   38.046,  19.156,  40.633 ]),
    ('CH2', [   36.986,  18.441,  38.890 ]),
    ('HH2', [   36.010,  18.694,  39.277 ]),
    ('C', [   44.158,  19.093,  37.693 ]),
    ('O', [   45.085,  19.012,  36.890 ])
    ),
    'TYR' :
    (
    ('N', [   44.347,  19.255,  39.010 ]),
    ('H', [   43.549,  19.215,  39.631 ]),
    ('CA', [   45.629,  19.327,  39.719 ]),
    ('HA', [   46.315,  18.604,  39.272 ]),
    ('CB', [   46.260,  20.736,  39.685 ]),
    ('HB1', [   45.605,  21.437,  40.205 ]),
    ('HB2', [   47.188,  20.730,  40.259 ]),
    ('CG', [   46.587,  21.291,  38.319 ]),
    ('CD1', [   47.705,  20.817,  37.601 ]),
    ('HD1', [   48.357,  20.068,  38.027 ]),
    ('CD2', [   45.738,  22.257,  37.746 ]),
    ('HD2', [   44.869,  22.605,  38.284 ]),
    ('CE1', [   47.954,  21.292,  36.300 ]),
    ('HE1', [   48.793,  20.911,  35.737 ]),
    ('CE2', [   46.002,  22.746,  36.456 ]),
    ('HE2', [   45.335,  23.479,  36.031 ]),
    ('CZ', [   47.099,  22.250,  35.724 ]),
    ('OH', [   47.330,  22.681,  34.453 ]),
    ('HH', [   46.661,  23.258,  34.120 ]),
    ('C', [   45.385,  18.925,  41.173 ]),
    ('O', [   44.263,  19.064,  41.661 ])
    ),
    'VAL' :
    (
    ('N', [   46.436,  18.490,  41.874 ]),
    ('H', [   47.353,  18.513,  41.443 ]),
    ('CA', [   46.478,  18.190,  43.304 ]),
    ('HA', [   45.801,  18.874,  43.819 ]),
    ('CB', [   46.020,  16.730,  43.603 ]),
    ('HB', [   46.615,  16.064,  42.975 ]),
    ('CG1', [   46.258,  16.292,  45.061 ]),
    ('HG11', [   45.897,  15.275,  45.220 ]),
    ('HG12', [   47.315,  16.289,  45.325 ]),
    ('HG13', [   45.731,  16.939,  45.763 ]),
    ('CG2', [   44.539,  16.486,  43.250 ]),
    ('HG21', [   44.194,  15.510,  43.590 ]),
    ('HG22', [   43.895,  17.241,  43.704 ]),
    ('HG23', [   44.371,  16.511,  42.174 ]),
    ('C', [   47.870,  18.573,  43.829 ]),
    ('O', [   48.801,  18.766,  43.043 ])
    )
    }

_bonds = {
	'ALA' : (
		('CA',    'C'),
		('CA',    'HA'),
		('CA',    'CB'),
		('N',     'CA'),
		('C',     'O'),
		('C',     'O1'),
		('C',     'O2'),
		('C',     'OXT'),
		('N',     'H'),
		('N',     '1H'),
		('N',     '2H'),
		('N',     '3H'),
		('CB',  '1HB'),
		('CB',  '2HB'),
		('CB',  '3HB'),
	),
	'CYS' : (
		('CA',    'C'),
		('CA',    'HA'),
		('CA',    'CB'),
		('N',     'CA'),
		('C',     'O'),
		('C',     'O1'),
		('C',     'O2'),
		('C',     'OXT'),
		('N',     'H'),
		('N',     '1H'),
		('N',     '2H'),
		('N',     '3H'),
		('CB',  '1HB'),
		('CB',  '2HB'),
		('CB',  'SG'),
		('SG',  'HG'),
	),
	'GLU' : (
		('CA',    'C'),
		('CA',    'HA'),
		('CA',    'CB'),
		('N',     'CA'),
		('C',     'O'),
		('C',     'O1'),
		('C',     'O2'),
		('C',     'OXT'),
		('N',     'H'),
		('N',     '1H'),
		('N',     '2H'),
		('N',     '3H'),
		('CB',  '1HB'),
		('CB',  '2HB'),
		('CB',  'CG'),
		('CG',  '1HG'),
		('CG',  '2HG'),
		('CG',  'CD'),
		('CD',  'OE1'),
		('CD',  'OE2'),
		('OE2',  'HE2'),
	),
	'ASP' : (
		('CA',    'C'),
		('CA',    'HA'),
		('CA',    'CB'),
		('N',     'CA'),
		('C',     'O'),
		('C',     'O1'),
		('C',     'O2'),
		('C',     'OXT'),
		('N',     'H'),
		('N',     '1H'),
		('N',     '2H'),
		('N',     '3H'),
		('CB',  '1HB'),
		('CB',  '2HB'),
		('CB',  'CG'),
		('CG',  'OD1'),
		('CG',  'OD2'),
		('OD2',  'HD2'),
	),
	'GLY' : (
		('CA',    'C'),
		('CA',    '1HA'),
		('CA',    '2HA'),
		('N',     'CA'),
		('C',     'O'),
		('C',     'O1'),
		('C',     'O2'),
		('C',     'OXT'),
		('N',     'H'),
		('N',     '1H'),
		('N',     '2H'),
		('N',     '3H'),
	),
	'PHE' : (
		('CA',    'C'),
		('CA',    'HA'),
		('CA',    'CB'),
		('N',     'CA'),
		('C',     'O'),
		('C',     'O1'),
		('C',     'O2'),
		('C',     'OXT'),
		('N',     'H'),
		('N',     '1H'),
		('N',     '2H'),
		('N',     '3H'),
		('CB',  '1HB'),
		('CB',  '2HB'),
		('CB',  'CG'),
		('CG',  'CD1'),
		('CG',  'CD2'),
		('CD1',  'HD1'),
		('CD1',  'CE1'),
		('CD2',  'HD2'),
		('CD2',  'CE2'),
		('CE1',  'HE1'),
		('CE2',  'HE2'),
		('CE1',  'CZ'),
		('CE2',  'CZ'),
		('CZ',  'HZ'),
	),
	'ILE' : (
		('CA',    'C'),
		('CA',    'HA'),
		('CA',    'CB'),
		('N',     'CA'),
		('C',     'O'),
		('C',     'O1'),
		('C',     'O2'),
		('C',     'OXT'),
		('N',     'H'),
		('N',     '1H'),
		('N',     '2H'),
		('N',     '3H'),
		('CB',  'HB'),
		('CB',  'CG1'),
		('CB',  'CG2'),
		('CG1',  '1HG1'),
		('CG1',  '2HG1'),
		('CG1',  'CD1'),
		('CG2',  '1HG2'),
		('CG2',  '2HG2'),
		('CG2',  '3HG2'),
		('CD1',  '1HD1'),
		('CD1',  '2HD1'),
		('CD1',  '3HD1'),
	),
	'HIS' : (
		('CA',    'C'),
		('CA',    'HA'),
		('CA',    'CB'),
		('N',     'CA'),
		('C',     'O'),
		('C',     'O1'),
		('C',     'O2'),
		('C',     'OXT'),
		('N',     'H'),
		('N',     '1H'),
		('N',     '2H'),
		('N',     '3H'),
		('CB',  '1HB'),
		('CB',  '2HB'),
		('CB',  'CG'),
		('CG',  'ND1'),
		('CG',  'CD2'),
		('ND1',  'CE1'),
		('ND1',  'HD1'),
		('CD2',  'HD2'),
		('CD2',  'NE2'),
		('CE1',  'HE1'),
		('CE1',  'NE2'),
		('NE2',  'HE2'),
	),
	'LYS' : (
		('CA',    'C'),
		('CA',    'HA'),
		('CA',    'CB'),
		('N',     'CA'),
		('C',     'O'),
		('C',     'O1'),
		('C',     'O2'),
		('C',     'OXT'),
		('N',     'H'),
		('N',     '1H'),
		('N',     '2H'),
		('N',     '3H'),
		('CB',  '1HB'),
		('CB',  '2HB'),
		('CB',  'CG'),
		('CG',  '1HG'),
		('CG',  '2HG'),
		('CG',  'CD'),
		('CD',  '1HD'),
		('CD',  '2HD'),
		('CD',  'CE'),
		('CE',  '1HE'),
		('CE',  '2HE'),
		('CE',  'NZ'),
		('NZ',  '1HZ'),
		('NZ',  '2HZ'),
		('NZ',  '3HZ'),
	),
	'MET' : (
		('CA',    'C'),
		('CA',    'HA'),
		('CA',    'CB'),
		('N',     'CA'),
		('C',     'O'),
		('C',     'O1'),
		('C',     'O2'),
		('C',     'OXT'),
		('N',     'H'),
		('N',     '1H'),
		('N',     '2H'),
		('N',     '3H'),
		('CB',  '1HB'),
		('CB',  '2HB'),
		('CB',  'CG'),
		('CG',  '1HG'),
		('CG',  '2HG'),
		('CG',  'SD'),
		('SD',  'CE'),
		('CE',  '1HE'),
		('CE',  '2HE'),
		('CE',  '3HE'),
	),
	'LEU' : (
		('CA',    'C'),
		('CA',    'HA'),
		('CA',    'CB'),
		('N',     'CA'),
		('C',     'O'),
		('C',     'O1'),
		('C',     'O2'),
		('C',     'OXT'),
		('N',     'H'),
		('N',     '1H'),
		('N',     '2H'),
		('N',     '3H'),
		('CB',  '1HB'),
		('CB',  '2HB'),
		('CB',  'CG'),
		('CG',  'HG'),
		('CG',  'CD1'),
		('CG',  'CD2'),
		('CD1',  '1HD1'),
		('CD1',  '2HD1'),
		('CD1',  '3HD1'),
		('CD2',  '1HD2'),
		('CD2',  '2HD2'),
		('CD2',  '3HD2'),
	),
	'ASN' : (
		('CA',    'C'),
		('CA',    'HA'),
		('CA',    'CB'),
		('N',     'CA'),
		('C',     'O'),
		('C',     'O1'),
		('C',     'O2'),
		('C',     'OXT'),
		('N',     'H'),
		('N',     '1H'),
		('N',     '2H'),
		('N',     '3H'),
		('CB',  '1HB'),
		('CB',  '2HB'),
		('CB',  'CG'),
		('CG',  'OD1'),
		('CG',  'ND2'),
		('ND2',  '1HD2'),
		('ND2',  '2HD2'),
	),
	'GLN' : (
		('CA',    'C'),
		('CA',    'HA'),
		('CA',    'CB'),
		('N',     'CA'),
		('C',     'O'),
		('C',     'O1'),
		('C',     'O2'),
		('C',     'OXT'),
		('N',     'H'),
		('N',     '1H'),
		('N',     '2H'),
		('N',     '3H'),
		('CB',  '1HB'),
		('CB',  '2HB'),
		('CB',  'CG'),
		('CG',  '1HG'),
		('CG',  '2HG'),
		('CG',  'CD'),
		('CD',  'OE1'),
		('CD',  'NE2'),
		('NE2',  '1HE2'),
		('NE2',  '2HE2'),
	),
	'PRO' : (
		('CA',    'C'),
		('CA',    'HA'),
		('CA',    'CB'),
		('N',     'CA'),
		('C',     'O'),
		('C',     'O1'),
		('C',     'O2'),
		('C',     'OXT'),
		('N',     'CD'),
		('N',     '1H'),
		('N',     '2H'),
		('CB',  '1HB'),
		('CB',  '2HB'),
		('CB',  'CG'),
		('CG',  '1HG'),
		('CG',  '2HG'),
		('CG',  'CD'),
		('CD',  '1HD'),
		('CD',  '2HD'),
	),
	'SER' : (
		('CA',    'C'),
		('CA',    'HA'),
		('CA',    'CB'),
		('N',     'CA'),
		('C',     'O'),
		('C',     'O1'),
		('C',     'O2'),
		('C',     'OXT'),
		('N',     'H'),
		('N',     '1H'),
		('N',     '2H'),
		('N',     '3H'),
		('CB',  '1HB'),
		('CB',  '2HB'),
		('CB',  'OG'),
		('OG',  'HG'),
	),
	'ARG' : (
		('CA',    'C'),
		('CA',    'HA'),
		('CA',    'CB'),
		('N',     'CA'),
		('C',     'O'),
		('C',     'O1'),
		('C',     'O2'),
		('C',     'OXT'),
		('N',     'H'),
		('N',     '1H'),
		('N',     '2H'),
		('N',     '3H'),
		('CB',  '1HB'),
		('CB',  '2HB'),
		('CB',  'CG'),
		('CG',  '1HG'),
		('CG',  '2HG'),
		('CG',  'CD'),
		('CD',  '1HD'),
		('CD',  '2HD'),
		('CD',  'NE'),
		('NE',  'HE'),
		('NE',  'CZ'),
		('CZ',  'NH1'),
		('CZ',  'NH2'),
		('NH1',  '1HH1'),
		('NH1',  '2HH1'),
		('NH2',  '1HH2'),
		('NH2',  '2HH2'),
	),
	'THR' : (
		('CA',    'C'),
		('CA',    'HA'),
		('CA',    'CB'),
		('N',     'CA'),
		('C',     'O'),
		('C',     'O1'),
		('C',     'O2'),
		('C',     'OXT'),
		('N',     'H'),
		('N',     '1H'),
		('N',     '2H'),
		('N',     '3H'),
		('CB',  'HB'),
		('CB',  'OG1'),
		('CB',  'CG2'),
		('OG1',  'HG1'),
		('CG2',  '1HG2'),
		('CG2',  '2HG2'),
		('CG2',  '3HG2'),
	),
	'TRP' : (
		('CA',    'C'),
		('CA',    'HA'),
		('CA',    'CB'),
		('N',     'CA'),
		('C',     'O'),
		('C',     'O1'),
		('C',     'O2'),
		('C',     'OXT'),
		('N',     'H'),
		('N',     '1H'),
		('N',     '2H'),
		('N',     '3H'),
		('CB',  '1HB'),
		('CB',  '2HB'),
		('CB',  'CG'),
		('CG',  'CD1'),
		('CG',  'CD2'),
		('CD1',  'HD1'),
		('CD1',  'NE1'),
		('CD2',  'CE2'),
                ('CD2',  'CE3'),
		('NE1',  'HE1'),
		('CE2',  'CZ2'),
                ('CE2',  'NE1'),
		('CE3',  'HE3'),
		('CE3',  'CZ3'),
		('CZ2',  'HZ2'),
		('CZ2',  'CH2'),
		('CZ3',  'HZ3'),
		('CZ3',  'CH2'),
		('CH2',  'HH2'),
	),
	'VAL' : (
		('CA',    'C'),
		('CA',    'HA'),
		('CA',    'CB'),
		('N',     'CA'),
		('C',     'O'),
		('C',     'O1'),
		('C',     'O2'),
		('C',     'OXT'),
		('N',     'H'),
		('N',     '1H'),
		('N',     '2H'),
		('N',     '3H'),
		('CB',  'HB'),
		('CB',  'CG1'),
		('CB',  'CG2'),
		('CG1',  '1HG1'),
		('CG1',  '2HG1'),
		('CG1',  '3HG1'),
		('CG2',  '1HG2'),
		('CG2',  '2HG2'),
		('CG2',  '3HG2'),
	),
	'TYR' : (
		('CA',    'C'),
		('CA',    'HA'),
		('CA',    'CB'),
		('N',     'CA'),
		('C',     'O'),
		('C',     'O1'),
		('C',     'O2'),
		('C',     'OXT'),
		('N',     'H'),
		('N',     '1H'),
		('N',     '2H'),
		('N',     '3H'),
		('CB',  '1HB'),
		('CB',  '2HB'),
		('CB',  'CG'),
		('CG',  'CD1'),
		('CG',  'CD2'),
		('CD1',  'HD1'),
		('CD1',  'CE1'),
		('CD2',  'HD2'),
		('CD2',  'CE2'),
		('CE1',  'HE1'),
		('CE2',  'HE2'),
		('CE1',  'CZ'),
		('CE2',  'CZ'),
		('CZ',  'OH'),
		('OH',  'HH'),
	),
}

_aliases = {
	'ALA': {
		'HB1':'1HB',
		'HB2':'2HB',
		'HB3':'3HB',
	},
	'NALA': {
                 'H1':'1H',
                 'H2':'2H',
                 'H3':'3H',
                 'HB1':'1HB',
                 'HB2':'2HB',
                 'HB3':'3HB',
	},
	'CALA': {
		'HB1':'1HB',
		'HB2':'2HB',
		'HB3':'3HB',
	},
	'ARG': {
		'HB1':'1HB',
		'HB2':'2HB',
		'HG1':'1HG',
		'HG2':'2HG',
		'HD1':'1HD',
		'HD2':'2HD',
		'HH11':'1HH1',
		'HH12':'2HH1',
		'HH21':'1HH2',
		'HH22':'2HH2',
	},
	'NARG': {
                 'H1':'1H',
                 'H2':'2H',
                 'H3':'3H',
		'HB1':'1HB',
		'HB2':'2HB',
		'HG1':'1HG',
		'HG2':'2HG',
		'HD1':'1HD',
		'HD2':'2HD',
		'HH11':'1HH1',
		'HH12':'2HH1',
		'HH21':'1HH2',
		'HH22':'2HH2',
	},
	'CARG': {
		'HB1':'1HB',
		'HB2':'2HB',
		'HG1':'1HG',
		'HG2':'2HG',
		'HD1':'1HD',
		'HD2':'2HD',
		'HH11':'1HH1',
		'HH12':'2HH1',
		'HH21':'1HH2',
		'HH22':'2HH2',
	},
	'ASP': {
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'ASPP': {
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'NASP': {
                 'H1':'1H',
                 'H2':'2H',
                 'H3':'3H',
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'CASP': {
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'ASH': {
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'NASH': {
                 'H1':'1H',
                 'H2':'2H',
                 'H3':'3H',
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'CASH': {
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'ASN': {
		'HB1':'1HB',
		'HB2':'2HB',
                'HD21':'1HD2',
                'HD22':'2HD2',
	},
	'NASN': {
                 'H1':'1H',
                 'H2':'2H',
                 'H3':'3H',
		'HB1':'1HB',
		'HB2':'2HB',
                'HD21':'1HD2',
                'HD22':'2HD2',
	},
	'CASN': {
		'HB1':'1HB',
		'HB2':'2HB',
                'HD21':'1HD2',
                'HD22':'2HD2',
	},
	'CYS': {
		'HB1':'1HB',
		'HB2':'2HB',
		'HG1':'1HG',
		#MS: HG1 i only in charmm...
	},
	'CYN': {
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'CYM': {
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'CYS2': {
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'CCYX': {
		'HB1':'1HB',
		'HB2':'2HB',
	},
        'CYSH': {
                'HB1':'1HB',
                'HB2':'2HB',
        },
	'GLU': {
		'HB1':'1HB',
		'HB2':'2HB',
		'HG1':'1HG',
		'HG2':'2HG',
	},
        'GLUH': {
                'HB1':'1HB',
                'HB2':'2HB',
                'HG1':'1HG',
                'HG2':'2HG',
                'HE2':'2HE',
        },
	'GLUP': {
		'HB1':'1HB',
		'HB2':'2HB',
		'HG1':'1HG',
		'HG2':'2HG',
	},
	'NGLU': {
                 'H1':'1H',
                 'H2':'2H',
                 'H3':'3H',
		'HB1':'1HB',
		'HB2':'2HB',
		'HG1':'1HG',
		'HG2':'2HG',
	},
	'CGLU': {
		'HB1':'1HB',
		'HB2':'2HB',
		'HG1':'1HG',
		'HG2':'2HG',
	},
	'GLH': {
		'HB1':'1HB',
		'HB2':'2HB',
		'HG1':'1HG',
		'HG2':'2HG',
	},
	'NGLH': {
                 'H1':'1H',
                 'H2':'2H',
                 'H3':'3H',
		'HB1':'1HB',
		'HB2':'2HB',
		'HG1':'1HG',
		'HG2':'2HG',
	},
	'CGLH': {
		'HB1':'1HB',
		'HB2':'2HB',
		'HG1':'1HG',
		'HG2':'2HG',
	},
	'GLN': {
		'HB1':'1HB',
		'HB2':'2HB',
		'HG1':'1HG',
		'HG2':'2HG',
                'HE21':'1HE2',
                'HE22':'2HE2',
	},
	'NGLN': {
                 'H1':'1H',
                 'H2':'2H',
                 'H3':'3H',
		'HB1':'1HB',
		'HB2':'2HB',
		'HG1':'1HG',
		'HG2':'2HG',
                'HE21':'1HE2',
                'HE22':'2HE2',
	},
	'CGLN': {
		'HB1':'1HB',
		'HB2':'2HB',
		'HG1':'1HG',
		'HG2':'2HG',
                'HE21':'1HE2',
                'HE22':'2HE2',
	},
	'GLY': {
		'HA1':'1HA',
		'HA2':'2HA',
	},
	'NGLY': {
                 'H1':'1H',
                 'H2':'2H',
                 'H3':'3H',
		'HA1':'1HA',
		'HA2':'2HA',
	},
	'CGLY': {
		'HA1':'1HA',
		'HA2':'2HA',
	},
	'HIS': {
		'HB1':'1HB',
		'HB2':'2HB',
	},
        'HIS1': {
                'HB1':'1HB',
                'HB2':'2HB',
        },
	'HSE': {
		'HB1':'1HB',
		'HB2':'2HB',
	},
        'HISE': {
                'HB1':'1HB',
                'HB2':'2HB',
        },
        'HISH': {
                'HB1':'1HB',
                'HB2':'2HB',
        },
	'HID': {
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'HSD': {
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'NHID': {
                 'H1':'1H',
                 'H2':'2H',
                 'H3':'3H',
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'CHID': {
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'HIE': {
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'NHIE': {
                 'H1':'1H',
                 'H2':'2H',
                 'H3':'3H',
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'CHIE': {
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'NHIP': {
                 'H1':'1H',
                 'H2':'2H',
                 'H3':'3H',
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'CHIP': {
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'HIP': {
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'HSP': {
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'ILE': {
                 'CD':'CD1',
		'HD1':'1HD1',
		'HD2':'2HD1',
		'HD3':'3HD1',
                'HG11':'1HG1',
                'HG12':'2HG1',
                'HG21':'1HG2',
                'HG22':'2HG2',
                'HG23':'3HG2',
                'HD11':'1HD1',
                'HD12':'2HD1',
                'HD13':'3HD1',
	},
	'NILE': {
                 'H1':'1H',
                 'H2':'2H',
                 'H3':'3H',
                 'CD':'CD1',
		'HD1':'1HD1',
		'HD2':'2HD1',
		'HD3':'3HD1',
                'HG11':'1HG1',
                'HG12':'2HG1',
                'HG21':'1HG2',
                'HG22':'2HG2',
                'HG23':'3HG2',
                'HD11':'1HD1',
                'HD12':'2HD1',
                'HD13':'3HD1',
	},
	'CILE': {
                 'CD':'CD1',
                'HG11':'1HG1',
                'HG12':'2HG1',
                'HG21':'1HG2',
                'HG22':'2HG2',
                'HG23':'3HG2',
                'HD11':'1HD1',
                'HD12':'2HD1',
                'HD13':'3HD1',
		'HD1':'1HD1',
		'HD2':'2HD1',
		'HD3':'3HD1',
	},
	'LEU': {
		'HB1':'1HB',
		'HB2':'2HB',
                'HD11':'1HD1',
                'HD12':'2HD1',
                'HD13':'3HD1',
                'HD21':'1HD2',
                'HD22':'2HD2',
                'HD23':'3HD2',
	},
	'NLEU': {
                 'H1':'1H',
                 'H2':'2H',
                 'H3':'3H',
		'HB1':'1HB',
		'HB2':'2HB',
                'HD11':'1HD1',
                'HD12':'2HD1',
                'HD13':'3HD1',
                'HD21':'1HD2',
                'HD22':'2HD2',
                'HD23':'3HD2',
	},
	'CLEU': {
		'HB1':'1HB',
		'HB2':'2HB',
                'HD11':'1HD1',
                'HD12':'2HD1',
                'HD13':'3HD1',
                'HD21':'1HD2',
                'HD22':'2HD2',
                'HD23':'3HD2',
	},
	'LYS': {
		'HB1':'1HB',
		'HB2':'2HB',
		'HG1':'1HG',
		'HG2':'2HG',
		'HD1':'1HD',
		'HD2':'2HD',
		'HE1':'1HE',
		'HE2':'2HE',
		'HZ1':'1HZ',
		'HZ2':'2HZ',
		'HZ3':'3HZ',
	},
        'LYSH': {
                'HB1':'1HB',
                'HB2':'2HB',
                'HG1':'1HG',
                'HG2':'2HG',
                'HD1':'1HD',
                'HD2':'2HD',
                'HE1':'1HE',
                'HE2':'2HE',
                'HZ1':'1HZ',
                'HZ2':'2HZ',
                'HZ3':'3HZ',
        },
        'LYN': {
                'HB1':'1HB',
                'HB2':'2HB',
                'HG1':'1HG',
                'HG2':'2HG',
                'HD1':'1HD',
                'HD2':'2HD',
                'HE1':'1HE',
                'HE2':'2HE',
                'HZ1':'1HZ',
                'HZ2':'2HZ',
        },
        'LSN': {
                'HB1':'1HB',
                'HB2':'2HB',
                'HG1':'1HG',
                'HG2':'2HG',
                'HD1':'1HD',
                'HD2':'2HD',
                'HE1':'1HE',
                'HE2':'2HE',
                'HZ1':'1HZ',
                'HZ2':'2HZ',
        },
	'NLYS': {
                 'H1':'1H',
                 'H2':'2H',
                 'H3':'3H',
		'HB1':'1HB',
		'HB2':'2HB',
		'HG1':'1HG',
		'HG2':'2HG',
		'HD1':'1HD',
		'HD2':'2HD',
		'HE1':'1HE',
		'HE2':'2HE',
		'HZ1':'1HZ',
		'HZ2':'2HZ',
		'HZ3':'3HZ',
	},
	'CLYS': {
		'HB1':'1HB',
		'HB2':'2HB',
		'HG1':'1HG',
		'HG2':'2HG',
		'HD1':'1HD',
		'HD2':'2HD',
		'HE1':'1HE',
		'HE2':'2HE',
		'HZ1':'1HZ',
		'HZ2':'2HZ',
		'HZ3':'3HZ',
	},
	'LYP': {
		'HB1':'1HB',
		'HB2':'2HB',
		'HG1':'1HG',
		'HG2':'2HG',
		'HD1':'1HD',
		'HD2':'2HD',
		'HE1':'1HE',
		'HE2':'2HE',
		'HZ1':'1HZ',
		'HZ2':'2HZ',
		'HZ3':'3HZ',
	},
	'NLYP': {
                 'H1':'1H',
                 'H2':'2H',
                 'H3':'3H',
		'HB1':'1HB',
		'HB2':'2HB',
		'HG1':'1HG',
		'HG2':'2HG',
		'HD1':'1HD',
		'HD2':'2HD',
		'HE1':'1HE',
		'HE2':'2HE',
		'HZ1':'1HZ',
		'HZ2':'2HZ',
		'HZ3':'3HZ',
	},
	'CLYP': {
		'HB1':'1HB',
		'HB2':'2HB',
		'HG1':'1HG',
		'HG2':'2HG',
		'HD1':'1HD',
		'HD2':'2HD',
		'HE1':'1HE',
		'HE2':'2HE',
		'HZ1':'1HZ',
		'HZ2':'2HZ',
		'HZ3':'3HZ',
	},
	'MET': {
		'HB1':'1HB',
		'HB2':'2HB',
		'HG1':'1HG',
		'HG2':'2HG',
		'HE1':'1HE',
		'HE2':'2HE',
		'HE3':'3HE',
	},
	'NMET': {
                 'H1':'1H',
                 'H2':'2H',
                 'H3':'3H',
		'HB1':'1HB',
		'HB2':'2HB',
		'HG1':'1HG',
		'HG2':'2HG',
		'HE1':'1HE',
		'HE2':'2HE',
		'HE3':'3HE',
	},
	'CMET': {
		'HB1':'1HB',
		'HB2':'2HB',
		'HG1':'1HG',
		'HG2':'2HG',
		'HE1':'1HE',
		'HE2':'2HE',
		'HE3':'3HE',
	},
	'PHE': {
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'NPHE': {
                 'H1':'1H',
                 'H2':'2H',
                 'H3':'3H',
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'CPHE': {
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'PRO': {
		'HB1':'1HB',
		'HB2':'2HB',
		'HG1':'1HG',
		'HG2':'2HG',
		'HD1':'1HD',
		'HD2':'2HD',
	},
	'NPRO': {
                 'H1':'1H',
                 'H2':'2H',
                 'H3':'3H',
		'HB1':'1HB',
		'HB2':'2HB',
		'HG1':'1HG',
		'HG2':'2HG',
		'HD1':'1HD',
		'HD2':'2HD',
	},
	'CPRO': {
		'HB1':'1HB',
		'HB2':'2HB',
		'HG1':'1HG',
		'HG2':'2HG',
		'HD1':'1HD',
		'HD2':'2HD',
	},
	'SER': {
		'HB1':'1HB',
		'HB2':'2HB',
		'HG1':'1HG',
	},
	'NSER': {
                 'H1':'1H',
                 'H2':'2H',
                 'H3':'3H',
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'CSER': {
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'THR': {
                'HG21':'1HG2',
                'HG22':'2HG2',
                'HG23':'3HG2',
	},
	'NTHR': {
                 'H1':'1H',
                 'H2':'2H',
                 'H3':'3H',
                'HG21':'1HG2',
                'HG22':'2HG2',
                'HG23':'3HG2',
	},
	'CTHR': {
                'HG21':'1HG2',
                'HG22':'2HG2',
                'HG23':'3HG2',
	},
	'TRP': {
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'CTRP': {
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'NTRP': {
                 'H1':'1H',
                 'H2':'2H',
                 'H3':'3H',
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'TYR': {
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'NTYR': {
                 'H1':'1H',
                 'H2':'2H',
                 'H3':'3H',
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'CTYR': {
		'HB1':'1HB',
		'HB2':'2HB',
	},
	'VAL': {
                'HG11':'1HG1',
                'HG12':'2HG1',
                'HG13':'3HG1',
                'HG21':'1HG2',
                'HG22':'2HG2',
                'HG23':'3HG2',
	},
	'NVAL': {
                 'H1':'1H',
                 'H2':'2H',
                 'H3':'3H',
                'HG11':'1HG1',
                'HG12':'2HG1',
                'HG13':'3HG1',
                'HG21':'1HG2',
                'HG22':'2HG2',
                'HG23':'3HG2',
	},
	'CVAL': {
                'HG11':'1HG1',
                'HG12':'2HG1',
                'HG13':'3HG1',
                'HG21':'1HG2',
                'HG22':'2HG2',
                'HG23':'3HG2',
	},
	'DA': {
                '1H5\'':'H5\'1',
                '2H5\'':'H5\'2',
                '1H2\'':'H2\'1',
                '2H2\'':'H2\'2',
	},
	'DA5': {
                '1H5\'':'H5\'1',
                '2H5\'':'H5\'2',
                '1H2\'':'H2\'1',
                '2H2\'':'H2\'2',
	},
	'DA3': {
                '1H5\'':'H5\'1',
                '2H5\'':'H5\'2',
                '1H2\'':'H2\'1',
                '2H2\'':'H2\'2',
	},
	'DAN': {
                '1H5\'':'H5\'1',
                '2H5\'':'H5\'2',
                '1H2\'':'H2\'1',
                '2H2\'':'H2\'2',
	},
	'DT': {
                '1H5\'':'H5\'1',
                '2H5\'':'H5\'2',
                '1H2\'':'H2\'1',
                '2H2\'':'H2\'2',
	},
	'DT5': {
                '1H5\'':'H5\'1',
                '2H5\'':'H5\'2',
                '1H2\'':'H2\'1',
                '2H2\'':'H2\'2',
	},
	'DT3': {
                '1H5\'':'H5\'1',
                '2H5\'':'H5\'2',
                '1H2\'':'H2\'1',
                '2H2\'':'H2\'2',
	},
	'DTN': {
                '1H5\'':'H5\'1',
                '2H5\'':'H5\'2',
                '1H2\'':'H2\'1',
                '2H2\'':'H2\'2',
	},
	'DG': {
                '1H5\'':'H5\'1',
                '2H5\'':'H5\'2',
                '1H2\'':'H2\'1',
                '2H2\'':'H2\'2',
	},
	'DG5': {
                '1H5\'':'H5\'1',
                '2H5\'':'H5\'2',
                '1H2\'':'H2\'1',
                '2H2\'':'H2\'2',
	},
	'DG3': {
                '1H5\'':'H5\'1',
                '2H5\'':'H5\'2',
                '1H2\'':'H2\'1',
                '2H2\'':'H2\'2',
	},
	'DGN': {
                '1H5\'':'H5\'1',
                '2H5\'':'H5\'2',
                '1H2\'':'H2\'1',
                '2H2\'':'H2\'2',
	},
	'DC': {
                '1H5\'':'H5\'1',
                '2H5\'':'H5\'2',
                '1H2\'':'H2\'1',
                '2H2\'':'H2\'2',
	},
	'DC5': {
                '1H5\'':'H5\'1',
                '2H5\'':'H5\'2',
                '1H2\'':'H2\'1',
                '2H2\'':'H2\'2',
	},
	'DC3': {
                '1H5\'':'H5\'1',
                '2H5\'':'H5\'2',
                '1H2\'':'H2\'1',
                '2H2\'':'H2\'2',
	},
	'DCN': {
                '1H5\'':'H5\'1',
                '2H5\'':'H5\'2',
                '1H2\'':'H2\'1',
                '2H2\'':'H2\'2',
	},
}

_atom_types = {
	'GENERIC': {
		"C" : {
			'type':'CGEN',
			'hyb':''
		},
		"CL" : {
			'type':'CL',
			'hyb':''
		},
		"I" : {
			'type':'IGEN',
			'hyb':''
		},
		"NA" : {
			'type':'NA',
			'hyb':''
		},
		"K" : {
			'type':'K',
			'hyb':''
		},
		"F" : {
			'type':'FGEN',
			'hyb':''
		},
		"ZN" : {
			'type':'ZN',
			'hyb':''
		},
		"O" : {
			'type':'OGEN',
			'hyb':''
		},
		"N" : {
			'type':'NGEN',
			'hyb':''
		},
		"P" : {
			'type':'PGEN',
			'hyb':''
		},
		"S" : {
			'type':'SGEN',
			'hyb':''
		},
		"FE" : {
			'type':'FEGEN',
			'hyb':''
		},
		"BR" : {
			'type':'BR',
			'hyb':''
		},
		"H" : {
			'type':'HGEN',
			'hyb':''
		},
		"CA" : {
			'type':'CA',
			'hyb':''
		},
	},
	'DEFAULT': {
		"C" : {
			'type':'C',
			'hyb':'sp2'
		},
		"OT" : {
			'type':'OH1',
			'hyb':''
		},
		"HO" : {
			'type':'H',
			'hyb':''
		},
		"CB" : {
			'type':'CH2E',
			'hyb':'sp3'
		},
		"OXT" : {
			'type':'OC',
			'hyb':''
		},
		"O1" : {
			'type':'OC',
			'hyb':''
		},
		"O2" : {
			'type':'OC',
			'hyb':''
		},
		"OT1" : {
			'type':'OC',
			'hyb':''
		},
		"OT2" : {
			'type':'OC',
			'hyb':''
		},
		"OC1" : {
			'type':'OC',
			'hyb':''
		},
		"OC2" : {
			'type':'OC',
			'hyb':''
                        },
		"O\'" : {
			'type':'OC',
			'hyb':''
		},
		"O\'\'" : {
			'type':'OC',
			'hyb':''
		},
		"H" : {
			'type':'H',
			'hyb':''
		},
		"1H" : {
			'type':'HC',
			'hyb':''
		},
		"2H" : {
			'type':'HC',
			'hyb':''
		},
		"3H" : {
			'type':'HC',
			'hyb':''
		},
		"CA" : {
			'type':'CA',
			'hyb':'sp3'
		},
		"HA" : {
			'type':'HA',
			'hyb':''
		},
		"O" : {
			'type':'O',
			'hyb':''
		},
		"N" : {
			'type':'NH1',
			'hyb':'sp2'
		},
	},
	'ALA': {
		"1HB" : {
			'type':'H0',
			'hyb':''
		},
		"CB" : {
			'type':'CH3E',
			'hyb':'sp3'
		},
		"3HB" : {
			'type':'H0',
			'hyb':''
		},
		"2HB" : {
			'type':'H0',
			'hyb':''
		},
	},
	'ARG': {
		"2HH2" : {
			'type':'HC',
			'hyb':''
		},
		"2HD" : {
			'type':'HDR',
			'hyb':''
		},
		"2HH1" : {
			'type':'HC',
			'hyb':''
		},
		"CG" : {
			'type':'CH2E',
			'hyb':'sp3'
		},
		"NE" : {
			'type':'NH1',
			'hyb':'sp2'
		},
		"CD" : {
			'type':'CDR',
			'hyb':'sp3'
		},
		"1HB" : {
			'type':'H0',
			'hyb':''
		},
		"CZ" : {
			'type':'C',
			'hyb':'sp2'
		},
		"2HG" : {
			'type':'H0',
			'hyb':''
		},
		"NH1" : {
			'type':'NC2',
			'hyb':'sp2'
		},
		"1HG" : {
			'type':'H0',
			'hyb':''
		},
		"1HD" : {
			'type':'HDR',
			'hyb':''
		},
		"NH2" : {
			'type':'NC2',
			'hyb':'sp2'
		},
		"2HB" : {
			'type':'H0',
			'hyb':''
		},
		"1HH2" : {
			'type':'HC',
			'hyb':''
		},
		"1HH1" : {
			'type':'HC',
			'hyb':''
		},
		"HE" : {
			'type':'H',
			'hyb':''
		},
	},
	'ASN': {
		"1HB" : {
			'type':'H0',
			'hyb':''
		},
		"ND2" : {
			'type':'NH2',
			'hyb':'sp2'
		},
		"1HD2" : {
			'type':'H',
			'hyb':''
		},
		"OD1" : {
			'type':'O',
			'hyb':'sp2'
		},
		"2HB" : {
			'type':'H0',
			'hyb':''
		},
		"CG" : {
			'type':'C',
			'hyb':'sp2'
		},
		"2HD2" : {
			'type':'H',
			'hyb':''
		},
	},
	'ASP': {
		"1HB" : {
			'type':'H0',
			'hyb':''
		},
		"HD2" : {
			'type':'H',
			'hyb':''
		},
		"OD1" : {
			'type':'OC',
			'hyb':'sp2'
		},
		"OD2" : {
			'type':'OC',
			'hyb':'sp2'
		},
		"2HB" : {
			'type':'H0',
			'hyb':''
		},
		"CG" : {
			'type':'C',
			'hyb':'sp2'
		},
	},
	'CYS': {
		"1HB" : {
			'type':'H0',
			'hyb':''
		},
		"2HB" : {
			'type':'H0',
			'hyb':''
		},
		"SG" : {
			'type':'S',
			'hyb':'sp3'
		},
		"HG" : {
			'type':'H0',
			'hyb':''
		},
	},
	'GLN': {
		"1HB" : {
			'type':'H0',
			'hyb':''
		},
		"2HB" : {
			'type':'H0',
			'hyb':''
		},
		"1HG" : {
			'type':'H0',
			'hyb':''
		},
		"NE2" : {
			'type':'NH2',
			'hyb':'sp2'
		},
		"OE1" : {
			'type':'O',
			'hyb':'sp2'
		},
		"2HE2" : {
			'type':'H',
			'hyb':''
		},
		"CG" : {
			'type':'CH2E',
			'hyb':'sp3'
		},
		"1HE2" : {
			'type':'H',
			'hyb':''
		},
		"2HG" : {
			'type':'H0',
			'hyb':''
		},
		"CD" : {
			'type':'C',
			'hyb':'sp2'
		},
	},
	'GLU': {
		"1HB" : {
			'type':'H0',
			'hyb':''
		},
		"1HG" : {
			'type':'H0',
			'hyb':''
		},
		"HE2" : {
			'type':'H',
			'hyb':''
		},
		"OE2" : {
			'type':'OC',
			'hyb':'sp2'
		},
		"OE1" : {
			'type':'OC',
			'hyb':'sp2'
		},
		"2HB" : {
			'type':'H0',
			'hyb':''
		},
		"CG" : {
			'type':'CH2E',
			'hyb':'sp3'
		},
		"2HG" : {
			'type':'H0',
			'hyb':''
		},
		"CD" : {
			'type':'C',
			'hyb':'sp2'
		},
	},
	'GLY': {
		"2HA" : {
			'type':'HA',
			'hyb':''
		},
		"CA" : {
			'type':'CH2G',
			'hyb':'sp3'
		},
		"1HA" : {
			'type':'HA',
			'hyb':''
		},
	},
	'HIS': {
		"HE2" : {
			'type':'H',
			'hyb':''
		},
		"HD2" : {
			'type':'HAR',
			'hyb':''
		},
		"HD1" : {
			'type':'H',
			'hyb':''
		},
		"CG" : {
			'type':'C5',
			'hyb':'sp2'
		},
		"HE1" : {
			'type':'HAR',
			'hyb':''
		},
		"1HB" : {
			'type':'H0',
			'hyb':''
		},
		"CD2" : {
			'type':'CR1H',
			'hyb':'sp2'
		},
		"ND1" : {
			'type':'NH1',
			'hyb':'sp2'
		},
		"2HB" : {
			'type':'H0',
			'hyb':''
		},
		"NE2" : {
			'type':'NHS',
			'hyb':'sp2'
		},
		"CE1" : {
			'type':'CRHH',
			'hyb':'sp2'
		},
	},
	'ILE': {
		"2HG2" : {
			'type':'H0',
			'hyb':''
		},
		"2HG1" : {
			'type':'H0',
			'hyb':''
		},
		"CB" : {
			'type':'CH1E',
			'hyb':'sp3'
		},
		"3HG2" : {
			'type':'H0',
			'hyb':''
		},
		"2HD1" : {
			'type':'H0',
			'hyb':''
		},
		"3HD1" : {
			'type':'H0',
			'hyb':''
		},
		"CG1" : {
			'type':'CH2E',
			'hyb':'sp3'
		},
		"1HG2" : {
			'type':'H0',
			'hyb':''
		},
		"CD1" : {
			'type':'CH3E',
			'hyb':'sp3'
		},
		"1HG1" : {
			'type':'H0',
			'hyb':''
		},
		"HB" : {
			'type':'H0',
			'hyb':''
		},
		"CG2" : {
			'type':'CH3E',
			'hyb':'sp3'
		},
		"1HD1" : {
			'type':'H0',
			'hyb':''
		},
	},
	'LEU': {
		"1HB" : {
			'type':'H0',
			'hyb':''
		},
		"2HB" : {
			'type':'H0',
			'hyb':''
		},
		"1HD2" : {
			'type':'H0',
			'hyb':''
		},
		"1HD1" : {
			'type':'H0',
			'hyb':''
		},
		"CG" : {
			'type':'CH1E',
			'hyb':'sp3'
		},
		"2HD2" : {
			'type':'H0',
			'hyb':''
		},
		"3HD1" : {
			'type':'H0',
			'hyb':''
		},
		"3HD2" : {
			'type':'H0',
			'hyb':''
		},
		"CD1" : {
			'type':'CH3E',
			'hyb':'sp3'
		},
		"CD2" : {
			'type':'CH3E',
			'hyb':'sp3'
		},
		"2HD1" : {
			'type':'H0',
			'hyb':''
		},
		"HG" : {
			'type':'H0',
			'hyb':''
		},
	},
	'LYS': {
		"2HZ" : {
			'type':'HC',
			'hyb':''
		},
		"1HZ" : {
			'type':'HC',
			'hyb':''
		},
		"3HZ" : {
			'type':'HC',
			'hyb':''
		},
		"CG" : {
			'type':'CH2E',
			'hyb':'sp3'
		},
		"CE" : {
			'type':'CH2E',
			'hyb':'sp3'
		},
		"CD" : {
			'type':'CH2E',
			'hyb':'sp3'
		},
		"1HB" : {
			'type':'H0',
			'hyb':''
		},
		"NZ" : {
			'type':'NH3',
			'hyb':'sp3'
		},
		"1HG" : {
			'type':'H0',
			'hyb':''
		},
		"1HD" : {
			'type':'H0',
			'hyb':''
		},
		"1HE" : {
			'type':'H0',
			'hyb':''
		},
		"2HB" : {
			'type':'H0',
			'hyb':''
		},
		"2HE" : {
			'type':'H0',
			'hyb':''
		},
		"2HD" : {
			'type':'H0',
			'hyb':''
		},
		"2HG" : {
			'type':'H0',
			'hyb':''
		},
	},
	'LYSH': {
		"2HZ" : {
			'type':'HC',
			'hyb':''
		},
		"1HZ" : {
			'type':'HC',
			'hyb':''
		},
		"3HZ" : {
			'type':'HC',
			'hyb':''
		},
		"CG" : {
			'type':'CH2E',
			'hyb':'sp3'
		},
		"CE" : {
			'type':'CH2E',
			'hyb':'sp3'
		},
		"CD" : {
			'type':'CH2E',
			'hyb':'sp3'
		},
		"1HB" : {
			'type':'H0',
			'hyb':''
		},
		"NZ" : {
			'type':'NH3',
			'hyb':'sp3'
		},
		"1HG" : {
			'type':'H0',
			'hyb':''
		},
		"1HD" : {
			'type':'H0',
			'hyb':''
		},
		"1HE" : {
			'type':'H0',
			'hyb':''
		},
		"2HB" : {
			'type':'H0',
			'hyb':''
		},
		"2HE" : {
			'type':'H0',
			'hyb':''
		},
		"2HD" : {
			'type':'H0',
			'hyb':''
		},
		"2HG" : {
			'type':'H0',
			'hyb':''
		},
	},
	'MET': {
		"1HB" : {
			'type':'H0',
			'hyb':''
		},
		"SD" : {
			'type':'SM',
			'hyb':'sp3'
		},
		"2HG" : {
			'type':'H0',
			'hyb':''
		},
		"1HG" : {
			'type':'H0',
			'hyb':''
		},
		"1HE" : {
			'type':'H0',
			'hyb':''
		},
		"2HE" : {
			'type':'H0',
			'hyb':''
		},
		"2HB" : {
			'type':'H0',
			'hyb':''
		},
		"CG" : {
			'type':'CH2E',
			'hyb':'sp3'
		},
		"CE" : {
			'type':'CH3E',
			'hyb':'sp3'
		},
		"3HE" : {
			'type':'H0',
			'hyb':''
		},
	},
	'PHE': {
		"HZ" : {
			'type':'HAR',
			'hyb':''
		},
		"HD2" : {
			'type':'HAR',
			'hyb':''
		},
		"HE2" : {
			'type':'HAR',
			'hyb':''
		},
		"CZ" : {
			'type':'CR1E',
			'hyb':'sp2'
		},
		"CE1" : {
			'type':'CR1E',
			'hyb':'sp2'
		},
		"CG" : {
			'type':'CF',
			'hyb':'sp2'
		},
		"1HB" : {
			'type':'H0',
			'hyb':''
		},
		"HD1" : {
			'type':'HAR',
			'hyb':''
		},
		"CE2" : {
			'type':'CR1E',
			'hyb':'sp2'
		},
		"CD1" : {
			'type':'CR1E',
			'hyb':'sp2'
		},
		"CD2" : {
			'type':'CR1E',
			'hyb':'sp2'
		},
		"2HB" : {
			'type':'H0',
			'hyb':''
		},
		"HE1" : {
			'type':'HAR',
			'hyb':''
		},
	},
	'PRO': {
		"1HB" : {
			'type':'H0',
			'hyb':''
		},
		"1HG" : {
			'type':'H0',
			'hyb':''
		},
		"1HD" : {
			'type':'H0',
			'hyb':''
		},
		"CB" : {
			'type':'CH2P',
			'hyb':'sp3'
		},
		"2HB" : {
			'type':'H0',
			'hyb':''
		},
		"CG" : {
			'type':'CH2P',
			'hyb':'sp3'
		},
		"2HD" : {
			'type':'H0',
			'hyb':''
		},
		"2HG" : {
			'type':'H0',
			'hyb':''
		},
		"CD" : {
			'type':'CH2P',
			'hyb':'sp3'
		},
	},
	'SER': {
		"OG" : {
			'type':'OH1',
			'hyb':'sp3'
		},
		"1HB" : {
			'type':'H0',
			'hyb':''
		},
		"2HB" : {
			'type':'H0',
			'hyb':''
		},
		"HG" : {
			'type':'H',
			'hyb':''
		},
	},
	'THR': {
		"2HG2" : {
			'type':'H0',
			'hyb':''
		},
		"1HG2" : {
			'type':'H0',
			'hyb':''
		},
		"HB" : {
			'type':'H0',
			'hyb':''
		},
		"CB" : {
			'type':'CH1E',
			'hyb':'sp3'
		},
		"3HG2" : {
			'type':'H0',
			'hyb':''
		},
		"CG2" : {
			'type':'CH3E',
			'hyb':'sp3'
		},
		"OG1" : {
			'type':'OH1',
			'hyb':'sp3'
		},
		"HG1" : {
			'type':'H',
			'hyb':''
		},
	},
	'TRP': {
		"HH2" : {
			'type':'HAR',
			'hyb':''
		},
		"CD1" : {
			'type':'CW',
			'hyb':'sp2'
		},
		"HE1" : {
			'type':'H',
			'hyb':''
		},
		"HD1" : {
			'type':'HAR',
			'hyb':''
		},
		"HE3" : {
			'type':'HAR',
			'hyb':''
		},
		"CZ2" : {
			'type':'CR1W',
			'hyb':'sp2'
		},
		"CZ3" : {
			'type':'CR1E',
			'hyb':'sp2'
		},
		"CG" : {
			'type':'C5W',
			'hyb':'sp2'
		},
		"1HB" : {
			'type':'H0',
			'hyb':''
		},
		"CH2" : {
			'type':'CR1W',
			'hyb':'sp2'
		},
		"CE3" : {
			'type':'CR1E',
			'hyb':'sp2'
		},
		"CE2" : {
			'type':'CW',
			'hyb':'sp2'
		},
		"CD2" : {
			'type':'CR1E',
			'hyb':'sp2'
		},
		"HZ2" : {
			'type':'HAR',
			'hyb':''
		},
		"2HB" : {
			'type':'H0',
			'hyb':''
		},
		"HZ3" : {
			'type':'HAR',
			'hyb':''
		},
		"NE1" : {
			'type':'NH1',
			'hyb':'sp2'
		},
	},
	'TYR': {
		"HD2" : {
			'type':'HAR',
			'hyb':''
		},
		"HE2" : {
			'type':'HAR',
			'hyb':''
		},
		"OH" : {
			'type':'OH1',
			'hyb':'sp3'
		},
		"CE1" : {
			'type':'CR1E',
			'hyb':'sp2'
		},
		"CE2" : {
			'type':'CR1E',
			'hyb':'sp2'
		},
		"CG" : {
			'type':'CY',
			'hyb':'sp2'
		},
		"1HB" : {
			'type':'H0',
			'hyb':''
		},
		"CZ" : {
			'type':'CY2',
			'hyb':'sp2'
		},
		"HH" : {
			'type':'H',
			'hyb':''
		},
		"CD1" : {
			'type':'CR1E',
			'hyb':'sp2'
		},
		"CD2" : {
			'type':'CR1E',
			'hyb':'sp2'
		},
		"HD1" : {
			'type':'HAR',
			'hyb':''
		},
		"2HB" : {
			'type':'H0',
			'hyb':''
		},
		"HE1" : {
			'type':'HAR',
			'hyb':''
		},
	},
	'VAL': {
		"CG1" : {
			'type':'CH3E',
			'hyb':'sp3'
		},
		"2HG2" : {
			'type':'H0',
			'hyb':''
		},
		"2HG1" : {
			'type':'H0',
			'hyb':''
		},
		"CG2" : {
			'type':'CH3E',
			'hyb':'sp3'
		},
		"1HG2" : {
			'type':'H0',
			'hyb':''
		},
		"3HG1" : {
			'type':'H0',
			'hyb':''
		},
		"HB" : {
			'type':'H0',
			'hyb':''
		},
		"CB" : {
			'type':'CH1E',
			'hyb':'sp3'
		},
		"3HG2" : {
			'type':'H0',
			'hyb':''
		},
		"1HG1" : {
			'type':'H0',
			'hyb':''
		},
	},
	'COO-': {
		"OXT" : {
			'type':'OC',
			'hyb':''
		},
	},
	'COOH': {
		"OT" : {
			'type':'O',
			'hyb':''
		},
		"HO" : {
			'type':'H',
			'hyb':''
		},
		"O" : {
			'type':'OH1',
			'hyb':''
		},
	},
	'NH3+': {
		"1H" : {
			'type':'HC',
			'hyb':''
		},
		"3H" : {
			'type':'HC',
			'hyb':''
		},
		"2H" : {
			'type':'HC',
			'hyb':''
		},
	},
	'NH2': {
		"1H" : {
			'type':'H',
			'hyb':''
		},
		"2H" : {
			'type':'H',
			'hyb':''
		},
		"N" : {
			'type':'NHB',
			'hyb':''
		},
	},
	'PRO-NH2+': {
		"1H" : {
			'type':'HC',
			'hyb':''
		},
		"2H" : {
			'type':'HC',
			'hyb':''
		},
	},
	'PRO-NH': {
		"H" : {
			'type':'H',
			'hyb':''
		},
		"N" : {
			'type':'NHB',
			'hyb':''
		},
	},
	'HISB': {
		"ND1" : {
			'type':'NHS',
			'hyb':''
		},
		"NE2" : {
			'type':'NH1',
			'hyb':''
		},
	},
	'HISH': {
		"NE2" : {
			'type':'NH1',
			'hyb':''
		},
		"HE2" : {
			'type':'H',
			'hyb':''
		},
	},
	'LYSN': {
		"NZ" : {
			'type':'NH2',
			'hyb':''
		},
		"HZ1" : {
			'type':'H',
			'hyb':''
		},
		"HZ2" : {
			'type':'H',
			'hyb':''
		},
	},
	'ASPH': {
		"OD1" : {
			'type':'O',
			'hyb':''
		},
		"OD2" : {
			'type':'OH1',
			'hyb':''
		},
		"HD2" : {
			'type':'H',
			'hyb':''
		},
	},
	'GLUH': {
		"OE2" : {
			'type':'OH1',
			'hyb':''
		},
		"OE1" : {
			'type':'O',
			'hyb':''
		},
		"HE2" : {
			'type':'H',
			'hyb':''
		},
	},
	'T': {
		"H72" : {
			'type':'H0',
			'hyb':''
		},
		"OP1" : {
			'type':'OP',
			'hyb':''
		},
		"OP2" : {
			'type':'OP',
			'hyb':''
		},
		"'H2'" : {
			'type':'H0',
			'hyb':''
		},
		"C3'" : {
			'type':'C3S',
			'hyb':''
		},
		"C1'" : {
			'type':'C1S',
			'hyb':''
		},
		"C5'" : {
			'type':'C5S',
			'hyb':''
		},
		"P" : {
			'type':'PN',
			'hyb':''
		},
		"C4'" : {
			'type':'C4S',
			'hyb':''
		},
		"C2'" : {
			'type':'C2S',
			'hyb':''
		},
		"N1" : {
			'type':'N1T',
			'hyb':''
		},
		"H2'" : {
			'type':'H0',
			'hyb':''
		},
		"N3" : {
			'type':'N3T',
			'hyb':''
		},
		"O2" : {
			'type':'O2T',
			'hyb':''
		},
		"H71" : {
			'type':'H0',
			'hyb':''
		},
		"H3'" : {
			'type':'H0',
			'hyb':''
		},
		"H4'" : {
			'type':'H0',
			'hyb':''
		},
		"H73" : {
			'type':'H0',
			'hyb':''
		},
		"O4" : {
			'type':'O4T',
			'hyb':''
		},
		"'H5'" : {
			'type':'H0',
			'hyb':''
		},
		"H3" : {
			'type':'H',
			'hyb':''
		},
		"O5'" : {
			'type':'O5S',
			'hyb':''
		},
		"O4'" : {
			'type':'O4S',
			'hyb':''
		},
		"H5'" : {
			'type':'H0',
			'hyb':''
		},
		"H6" : {
			'type':'H0',
			'hyb':''
		},
		"O3'" : {
			'type':'O3S',
			'hyb':''
		},
		"C2" : {
			'type':'C2T',
			'hyb':''
		},
		"H1'" : {
			'type':'H0',
			'hyb':''
		},
		"C7" : {
			'type':'C7T',
			'hyb':''
		},
		"C6" : {
			'type':'C6T',
			'hyb':''
		},
		"C5" : {
			'type':'C5T',
			'hyb':''
		},
		"C4" : {
			'type':'C4T',
			'hyb':''
		},
	},
	'A': {
		"OP1" : {
			'type':'OP',
			'hyb':''
		},
		"OP2" : {
			'type':'OP',
			'hyb':''
		},
		"H62" : {
			'type':'H',
			'hyb':''
		},
		"H3'" : {
			'type':'H0',
			'hyb':''
		},
		"C3'" : {
			'type':'C3S',
			'hyb':''
		},
		"C1'" : {
			'type':'C1S',
			'hyb':''
		},
		"C5'" : {
			'type':'C5S',
			'hyb':''
		},
		"N9" : {
			'type':'N9A',
			'hyb':''
		},
		"C4'" : {
			'type':'C4S',
			'hyb':''
		},
		"C2'" : {
			'type':'C2S',
			'hyb':''
		},
		"N1" : {
			'type':'N1A',
			'hyb':''
		},
		"H2'" : {
			'type':'H0',
			'hyb':''
		},
		"N3" : {
			'type':'N3A',
			'hyb':''
		},
		"N6" : {
			'type':'N6A',
			'hyb':''
		},
		"N7" : {
			'type':'N7A',
			'hyb':''
		},
		"H4'" : {
			'type':'H0',
			'hyb':''
		},
		"'H2'" : {
			'type':'H0',
			'hyb':''
		},
		"H8" : {
			'type':'H0',
			'hyb':''
		},
		"H5'" : {
			'type':'H0',
			'hyb':''
		},
		"'H5'" : {
			'type':'H0',
			'hyb':''
		},
		"P" : {
			'type':'PN',
			'hyb':''
		},
		"H2" : {
			'type':'H0',
			'hyb':''
		},
		"O5'" : {
			'type':'O5S',
			'hyb':''
		},
		"O4'" : {
			'type':'O4S',
			'hyb':''
		},
		"C8" : {
			'type':'C8A',
			'hyb':''
		},
		"O3'" : {
			'type':'O3S',
			'hyb':''
		},
		"H61" : {
			'type':'H',
			'hyb':''
		},
		"C2" : {
			'type':'C2A',
			'hyb':''
		},
		"H1'" : {
			'type':'H0',
			'hyb':''
		},
		"C6" : {
			'type':'C6A',
			'hyb':''
		},
		"C5" : {
			'type':'C5A',
			'hyb':''
		},
		"C4" : {
			'type':'C4A',
			'hyb':''
		},
	},
	'G': {
		"OP1" : {
			'type':'OP',
			'hyb':''
		},
		"OP2" : {
			'type':'OP',
			'hyb':''
		},
		"H3'" : {
			'type':'H0',
			'hyb':''
		},
		"C3'" : {
			'type':'C3S',
			'hyb':''
		},
		"C1'" : {
			'type':'C1S',
			'hyb':''
		},
		"H21" : {
			'type':'H',
			'hyb':''
		},
		"C5'" : {
			'type':'C5S',
			'hyb':''
		},
		"N9" : {
			'type':'N9G',
			'hyb':''
		},
		"C4'" : {
			'type':'C4S',
			'hyb':''
		},
		"C2'" : {
			'type':'C2S',
			'hyb':''
		},
		"O6" : {
			'type':'O6G',
			'hyb':''
		},
		"N1" : {
			'type':'N1G',
			'hyb':''
		},
		"N2" : {
			'type':'N2G',
			'hyb':''
		},
		"N3" : {
			'type':'N3G',
			'hyb':''
		},
		"H2'" : {
			'type':'H0',
			'hyb':''
		},
		"N7" : {
			'type':'N7G',
			'hyb':''
		},
		"H4'" : {
			'type':'H0',
			'hyb':''
		},
		"'H2'" : {
			'type':'H0',
			'hyb':''
		},
		"H8" : {
			'type':'H0',
			'hyb':''
		},
		"H5'" : {
			'type':'H0',
			'hyb':''
		},
		"'H5'" : {
			'type':'H0',
			'hyb':''
		},
		"P" : {
			'type':'PN',
			'hyb':''
		},
		"H1" : {
			'type':'H',
			'hyb':''
		},
		"O5'" : {
			'type':'O5S',
			'hyb':''
		},
		"O4'" : {
			'type':'O4S',
			'hyb':''
		},
		"H22" : {
			'type':'H',
			'hyb':''
		},
		"C8" : {
			'type':'C8G',
			'hyb':''
		},
		"O3'" : {
			'type':'O3S',
			'hyb':''
		},
		"C2" : {
			'type':'C2G',
			'hyb':''
		},
		"H1'" : {
			'type':'H0',
			'hyb':''
		},
		"C6" : {
			'type':'C6G',
			'hyb':''
		},
		"C5" : {
			'type':'C5G',
			'hyb':''
		},
		"C4" : {
			'type':'C4G',
			'hyb':''
		},
	},
	'C': {
		"OP1" : {
			'type':'OP',
			'hyb':''
		},
		"OP2" : {
			'type':'OP',
			'hyb':''
		},
		"'H2'" : {
			'type':'H0',
			'hyb':''
		},
		"C3'" : {
			'type':'C3S',
			'hyb':''
		},
		"C1'" : {
			'type':'C1S',
			'hyb':''
		},
		"C5'" : {
			'type':'C5S',
			'hyb':''
		},
		"P" : {
			'type':'PN',
			'hyb':''
		},
		"C4'" : {
			'type':'C4S',
			'hyb':''
		},
		"C2'" : {
			'type':'C2S',
			'hyb':''
		},
		"N1" : {
			'type':'N1C',
			'hyb':''
		},
		"H2'" : {
			'type':'H0',
			'hyb':''
		},
		"N3" : {
			'type':'N3C',
			'hyb':''
		},
		"N4" : {
			'type':'N4C',
			'hyb':''
		},
		"O2" : {
			'type':'O2C',
			'hyb':''
		},
		"H3'" : {
			'type':'H0',
			'hyb':''
		},
		"H4'" : {
			'type':'H0',
			'hyb':''
		},
		"H42" : {
			'type':'H',
			'hyb':''
		},
		"H5'" : {
			'type':'H0',
			'hyb':''
		},
		"'H5'" : {
			'type':'H0',
			'hyb':''
		},
		"O5'" : {
			'type':'O5S',
			'hyb':''
		},
		"H5" : {
			'type':'H0',
			'hyb':''
		},
		"O4'" : {
			'type':'O4S',
			'hyb':''
		},
		"H6" : {
			'type':'H0',
			'hyb':''
		},
		"O3'" : {
			'type':'O3S',
			'hyb':''
		},
		"H41" : {
			'type':'H',
			'hyb':''
		},
		"C2" : {
			'type':'C2C',
			'hyb':''
		},
		"H1'" : {
			'type':'H0',
			'hyb':''
		},
		"C6" : {
			'type':'C6C',
			'hyb':''
		},
		"C5" : {
			'type':'C5C',
			'hyb':''
		},
		"C4" : {
			'type':'C4C',
			'hyb':''
		},
	},
	'HOH': {
		"H2" : {
			'type':'HWAT',
			'hyb':''
		},
		"H1" : {
			'type':'HWAT',
			'hyb':''
		},
		"O" : {
			'type':'OWAT',
			'hyb':''
		},
	},
	'SOL': {
		"MW" : {
			'type':'MWAT',
			'hyb':''
		},
		"OW4" : {
			'type':'OWAT',
			'hyb':''
		},
		"OW" : {
			'type':'OWAT',
			'hyb':''
		},
		"MW1" : {
			'type':'MWAT',
			'hyb':''
		},
		"HW2" : {
			'type':'HWAT',
			'hyb':''
		},
		"HW3" : {
			'type':'HWAT',
			'hyb':''
		},
		"HW1" : {
			'type':'HWAT',
			'hyb':''
		},
	},
	'NA': {
		"NA" : {
			'type':'NA',
			'hyb':''
		},
	},
	'MG': {
		"MG" : {
			'type':'MG',
			'hyb':''
		},
	},
	'CA': {
		"CA" : {
			'type':'CAL',
			'hyb':''
		},
	},
	'K': {
		"K" : {
			'type':'K',
			'hyb':''
		},
	},
	'ZN': {
		"ZN" : {
			'type':'ZN',
			'hyb':''
		},
	},
	'FE': {
		"FE" : {
			'type':'FE',
			'hyb':''
		},
	},
	'MN': {
		"MN" : {
			'type':'MN',
			'hyb':''
		},
	},
	'SO4': {
		"S" : {
			'type':'SUL',
			'hyb':''
		},
		"O4" : {
			'type':'OSUL',
			'hyb':''
		},
		"O3" : {
			'type':'OSUL',
			'hyb':''
		},
		"O2" : {
			'type':'OSUL',
			'hyb':''
		},
		"O1" : {
			'type':'OSUL',
			'hyb':''
		},
	},
	'CL-': {
		"CL" : {
			'type':'CL',
			'hyb':''
		},
	},
}

_aa_dihedrals = {
        'PRO' : (
                ['N', 'CA', 'CB', 'HB1', 0, -1] ,
                ['N', 'CA', 'CB', 'HB2', 0, -1] ,
                ['N', 'CA', 'C', 'O', 1, 0] ,
                ['CD', 'N', 'CA', 'HA', 0, -1] ,
                ['CD', 'N', 'CA', 'CB', 0, -1] ,
                ['C', 'CA', 'N', 'CD', 0, -1] ,
                ['HA', 'CA', 'CB', 'HB1', 0, -1] ,
                ['HA', 'CA', 'CB', 'HB2', 0, -1] ,
                ['HA', 'CA', 'C', 'O', 0, -1] ,
                ['CB', 'CA', 'C', 'O', 0, -1] ,
                ['C', 'CA', 'CB', 'HB1', 0, -1] ,
                ['C', 'CA', 'CB', 'HB2', 0, -1] ,
                ['CA', 'N', 'CD', 'HD1', 0, -1] ,
                ['CA', 'N', 'CD', 'HD2', 0, -1] ,
                ['CA', 'N', 'CD', 'CG', 0, -1] ,
                ['N', 'CD', 'CG', 'HG1', 0, -1] ,
                ['N', 'CD', 'CG', 'HG2', 0, -1] ,
                ['N', 'CD', 'CG', 'CB', 0, -1] ,
                ['HD1', 'CD', 'CG', 'HG1', 0, -1] ,
                ['HD1', 'CD', 'CG', 'HG2', 0, -1] ,
                ['HD1', 'CD', 'CG', 'CB', 0, -1] ,
                ['HD2', 'CD', 'CG', 'HG1', 0, -1] ,
                ['HD2', 'CD', 'CG', 'HG2', 0, -1] ,
                ['HD2', 'CD', 'CG', 'CB', 0, -1] ,
                ['CD', 'CG', 'CB', 'HB1', 0, -1] ,
                ['CD', 'CG', 'CB', 'HB2', 0, -1] ,
                ['CD', 'CG', 'CB', 'CA', 0, -1] ,
                ['HG1', 'CG', 'CB', 'HB1', 0, -1] ,
                ['HG1', 'CG', 'CB', 'HB2', 0, -1] ,
                ['HG1', 'CG', 'CB', 'CA', 0, -1] ,
                ['HG2', 'CG', 'CB', 'HB1', 0, -1] ,
                ['HG2', 'CG', 'CB', 'HB2', 0, -1] ,
                ['HG2', 'CG', 'CB', 'CA', 0, -1] ,
                ['CG', 'CB', 'CA', 'C', 0, -1] ,
                ['CG', 'CB', 'CA', 'HA', 0, -1] ,
                ['CG', 'CB', 'CA', 'N', 0, -1] ,
             ),
	'ALA' : (
		['N', 'CA', 'CB', 'HB1', 1, 1] ,
		['N', 'CA', 'CB', 'HB2', 1, -1] ,
		['N', 'CA', 'CB', 'HB3', 1, -1] ,
		['N', 'CA', 'C', 'O', 1, 0] ,
		['H', 'N', 'CA', 'HA', 1, -1] ,
		['H', 'N', 'CA', 'CB', 1, -1] ,
		['C', 'CA', 'N', 'H', 1, -1] ,
		['HA', 'CA', 'CB', 'HB1', 1, -1] ,
		['HA', 'CA', 'CB', 'HB2', 1, -1] ,
		['HA', 'CA', 'CB', 'HB3', 1, -1] ,
		['HA', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CA', 'C', 'O', 1, -1] ,
		['C', 'CA', 'CB', 'HB1', 1, -1] ,
		['C', 'CA', 'CB', 'HB2', 1, -1] ,
		['C', 'CA', 'CB', 'HB3', 1, -1] ,
	     ),
	'CYS' : (
		['N', 'CA', 'CB', 'HB1', 1, -1] ,
		['N', 'CA', 'CB', 'HB2', 1, -1] ,
		['N', 'CA', 'CB', 'SG', 1, 1] ,
		['N', 'CA', 'C', 'O', 1, 0] ,
		['H', 'N', 'CA', 'HA', 1, -1] ,
		['H', 'N', 'CA', 'CB', 1, -1] ,
		['C', 'CA', 'N', 'H', 1, -1] ,
		['CA', 'CB', 'SG', 'HG', 1, 2] ,
		['HA', 'CA', 'CB', 'HB1', 1, -1] ,
		['HA', 'CA', 'CB', 'HB2', 1, -1] ,
		['HA', 'CA', 'CB', 'SG', 1, -1] ,
		['HA', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CA', 'C', 'O', 1, -1] ,
		['C', 'CA', 'CB', 'HB1', 1, -1] ,
		['HB1', 'CB', 'SG', 'HG', 1, -1] ,
		['C', 'CA', 'CB', 'HB2', 1, -1] ,
		['HB2', 'CB', 'SG', 'HG', 1, -1] ,
		['C', 'CA', 'CB', 'SG', 1, -1] ,
	     ),
	'CYM' : (
		['N', 'CA', 'CB', 'HB1', 1, -1] ,
		['N', 'CA', 'CB', 'HB2', 1, -1] ,
		['N', 'CA', 'CB', 'SG', 1, 1] ,
		['N', 'CA', 'C', 'O', 1, 0] ,
		['H', 'N', 'CA', 'HA', 1, -1] ,
		['H', 'N', 'CA', 'CB', 1, -1] ,
		['C', 'CA', 'N', 'H', 1, -1] ,
		['HA', 'CA', 'CB', 'HB1', 1, -1] ,
		['HA', 'CA', 'CB', 'HB2', 1, -1] ,
		['HA', 'CA', 'CB', 'SG', 1, -1] ,
		['HA', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CA', 'C', 'O', 1, -1] ,
		['C', 'CA', 'CB', 'HB1', 1, -1] ,
		['C', 'CA', 'CB', 'HB2', 1, -1] ,
		['C', 'CA', 'CB', 'SG', 1, -1] ,
	     ),
	'ASP' : (
		['N', 'CA', 'CB', 'HB1', 1, -1] ,
		['N', 'CA', 'CB', 'HB2', 1, -1] ,
		['N', 'CA', 'CB', 'CG', 1, 1] ,
		['N', 'CA', 'C', 'O', 1, 0] ,
		['H', 'N', 'CA', 'HA', 1, -1] ,
		['H', 'N', 'CA', 'CB', 1, -1] ,
		['C', 'CA', 'N', 'H', 1, -1] ,
		['CA', 'CB', 'CG', 'OD1', 1, 2] ,
		['CA', 'CB', 'CG', 'OD2', 1, -1] ,
		['HA', 'CA', 'CB', 'HB1', 1, -1] ,
		['HA', 'CA', 'CB', 'HB2', 1, -1] ,
		['HA', 'CA', 'CB', 'CG', 1, -1] ,
		['HA', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CA', 'C', 'O', 1, -1] ,
		['C', 'CA', 'CB', 'HB1', 1, -1] ,
		['HB1', 'CB', 'CG', 'OD1', 1, -1] ,
		['HB1', 'CB', 'CG', 'OD2', 1, -1] ,
		['C', 'CA', 'CB', 'HB2', 1, -1] ,
		['HB2', 'CB', 'CG', 'OD1', 1, -1] ,
		['HB2', 'CB', 'CG', 'OD2', 1, -1] ,
		['C', 'CA', 'CB', 'CG', 1, -1] ,
	     ),
	'GLU' : (
		['N', 'CA', 'CB', 'HB1', 1, -1] ,
		['N', 'CA', 'CB', 'HB2', 1, -1] ,
		['N', 'CA', 'CB', 'CG', 1, 1] ,
		['N', 'CA', 'C', 'O', 1, 0] ,
		['H', 'N', 'CA', 'HA', 1, -1] ,
		['H', 'N', 'CA', 'CB', 1, -1] ,
		['C', 'CA', 'N', 'H', 1, -1] ,
		['CA', 'CB', 'CG', 'HG1', 1, -1] ,
		['CA', 'CB', 'CG', 'HG2', 1, -1] ,
		['CA', 'CB', 'CG', 'CD', 1, 2] ,
		['HA', 'CA', 'CB', 'HB1', 1, -1] ,
		['HA', 'CA', 'CB', 'HB2', 1, -1] ,
		['HA', 'CA', 'CB', 'CG', 1, -1] ,
		['HA', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CG', 'CD', 'OE1', 1, 3] ,
		['CB', 'CG', 'CD', 'OE2', 1, -1] ,
		['C', 'CA', 'CB', 'HB1', 1, -1] ,
		['HB1', 'CB', 'CG', 'HG1', 1, -1] ,
		['HB1', 'CB', 'CG', 'HG2', 1, -1] ,
		['HB1', 'CB', 'CG', 'CD', 1, -1] ,
		['C', 'CA', 'CB', 'HB2', 1, -1] ,
		['HB2', 'CB', 'CG', 'HG1', 1, -1] ,
		['HB2', 'CB', 'CG', 'HG2', 1, -1] ,
		['HB2', 'CB', 'CG', 'CD', 1, -1] ,
		['C', 'CA', 'CB', 'CG', 1, -1] ,
		['HG1', 'CG', 'CD', 'OE1', 1, -1] ,
		['HG1', 'CG', 'CD', 'OE2', 1, -1] ,
		['HG2', 'CG', 'CD', 'OE1', 1, -1] ,
		['HG2', 'CG', 'CD', 'OE2', 1, -1] ,
	     ),
	'PHE' : (
		['N', 'CA', 'CB', 'HB1', 1, -1] ,
		['N', 'CA', 'CB', 'HB2', 1, -1] ,
		['N', 'CA', 'CB', 'CG', 1, 1] ,
		['N', 'CA', 'C', 'O', 1, 0] ,
		['H', 'N', 'CA', 'HA', 1, -1] ,
		['H', 'N', 'CA', 'CB', 1, -1] ,
		['C', 'CA', 'N', 'H', 1, -1] ,
		['CA', 'CB', 'CG', 'CD1', 1, 2] ,
		['CA', 'CB', 'CG', 'CD2', 1, -1] ,
		['HA', 'CA', 'CB', 'HB1', 1, -1] ,
		['HA', 'CA', 'CB', 'HB2', 1, -1] ,
		['HA', 'CA', 'CB', 'CG', 1, -1] ,
		['HA', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CG', 'CD1', 'HD1', 0, -1] ,
		['CB', 'CG', 'CD1', 'CE1', 0, 3] ,
		['CB', 'CG', 'CD2', 'HD2', 0, -1] ,
		['CB', 'CG', 'CD2', 'CE2', 0, -1] ,
		['C', 'CA', 'CB', 'HB1', 1, -1] ,
		['HB1', 'CB', 'CG', 'CD1', 1, -1] ,
		['HB1', 'CB', 'CG', 'CD2', 1, -1] ,
		['C', 'CA', 'CB', 'HB2', 1, -1] ,
		['HB2', 'CB', 'CG', 'CD1', 1, -1] ,
		['HB2', 'CB', 'CG', 'CD2', 1, -1] ,
		['C', 'CA', 'CB', 'CG', 1, -1] ,
		['CG', 'CD1', 'CE1', 'HE1', 0, -1] ,
		['CG', 'CD1', 'CE1', 'CZ', 0, 4] ,
		['CG', 'CD2', 'CE2', 'HE2', 0, -1] ,
		['CG', 'CD2', 'CE2', 'CZ', 0, -1] ,
		['CD1', 'CG', 'CD2', 'HD2', 0, -1] ,
		['CD1', 'CG', 'CD2', 'CE2', 0, -1] ,
		['CD1', 'CE1', 'CZ', 'CE2', 0, 5] ,
		['CD1', 'CE1', 'CZ', 'HZ', 0, -1] ,
		['CD2', 'CG', 'CD1', 'HD1', 0, -1] ,
		['HD1', 'CD1', 'CE1', 'HE1', 0, -1] ,
		['HD1', 'CD1', 'CE1', 'CZ', 0, -1] ,
		['CD2', 'CG', 'CD1', 'CE1', 0, -1] ,
		['CD2', 'CE2', 'CZ', 'CE1', 0, -1] ,
		['CD2', 'CE2', 'CZ', 'HZ', 0, -1] ,
		['HD2', 'CD2', 'CE2', 'HE2', 0, -1] ,
		['HD2', 'CD2', 'CE2', 'CZ', 0, -1] ,
		['CE1', 'CZ', 'CE2', 'HE2', 0, -1] ,
		['CE2', 'CZ', 'CE1', 'HE1', 0, -1] ,
		['HE1', 'CE1', 'CZ', 'HZ', 0, -1] ,
		['HE2', 'CE2', 'CZ', 'HZ', 0, -1] ,
	     ),
	'GLY' : (
		['N', 'CA', 'C', 'O', 1, 0] ,
		['H', 'N', 'CA', 'HA1', 1, -1] ,
		['H', 'N', 'CA', 'HA2', 1, -1] ,
		['C', 'CA', 'N', 'H', 1, -1] ,
		['HA1', 'CA', 'C', 'O', 1, -1] ,
		['HA2', 'CA', 'C', 'O', 1, -1] ,
	     ),
	'HIS' : (
		['N', 'CA', 'CB', 'HB1', 1, -1] ,
		['N', 'CA', 'CB', 'HB2', 1, -1] ,
		['N', 'CA', 'CB', 'CG', 1, 1] ,
		['N', 'CA', 'C', 'O', 1, 0] ,
		['H', 'N', 'CA', 'HA', 1, -1] ,
		['H', 'N', 'CA', 'CB', 1, -1] ,
		['C', 'CA', 'N', 'H', 1, -1] ,
		['CA', 'CB', 'CG', 'ND1', 1, 2] ,
		['CA', 'CB', 'CG', 'CD2', 1, -1] ,
		['HA', 'CA', 'CB', 'HB1', 1, -1] ,
		['HA', 'CA', 'CB', 'HB2', 1, -1] ,
		['HA', 'CA', 'CB', 'CG', 1, -1] ,
		['HA', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CG', 'ND1', 'CE1', 0, 3] ,
		['CB', 'CG', 'CD2', 'HD2', 0, -1] ,
		['CB', 'CG', 'CD2', 'NE2', 0, -1] ,
		['C', 'CA', 'CB', 'HB1', 1, -1] ,
		['HB1', 'CB', 'CG', 'ND1', 1, -1] ,
		['HB1', 'CB', 'CG', 'CD2', 1, -1] ,
		['C', 'CA', 'CB', 'HB2', 1, -1] ,
		['HB2', 'CB', 'CG', 'ND1', 1, -1] ,
		['HB2', 'CB', 'CG', 'CD2', 1, -1] ,
		['C', 'CA', 'CB', 'CG', 1, -1] ,
		['CG', 'ND1', 'CE1', 'HE1', 0, 4] ,
		['CG', 'CD2', 'NE2', 'HE2', 0, -1] ,
		['ND1', 'CG', 'CD2', 'HD2', 0, -1] ,
		['ND1', 'CG', 'CD2', 'NE2', 0, -1] ,
		['CD2', 'CG', 'ND1', 'CE1', 0, -1] ,
		['HD2', 'CD2', 'NE2', 'HE2', 0, -1] ,
	     ),
	'ILE' : (
		['N', 'CA', 'CB', 'HB', 1, -1] ,
		['N', 'CA', 'CB', 'CG1', 1, 1] ,
		['N', 'CA', 'CB', 'CG2', 1, -1] ,
		['N', 'CA', 'C', 'O', 1, 0] ,
		['H', 'N', 'CA', 'HA', 1, -1] ,
		['H', 'N', 'CA', 'CB', 1, -1] ,
		['C', 'CA', 'N', 'H', 1, -1] ,
		['CA', 'CB', 'CG1', 'HG11', 1, -1] ,
		['CA', 'CB', 'CG1', 'HG12', 1, -1] ,
		['CA', 'CB', 'CG1', 'CD1', 1, 2] ,
		['CA', 'CB', 'CG2', 'HG21', 1, -1] ,
		['CA', 'CB', 'CG2', 'HG22', 1, -1] ,
		['CA', 'CB', 'CG2', 'HG23', 1, -1] ,
		['HA', 'CA', 'CB', 'HB', 1, -1] ,
		['HA', 'CA', 'CB', 'CG1', 1, -1] ,
		['HA', 'CA', 'CB', 'CG2', 1, -1] ,
		['HA', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CG1', 'CD1', 'HD11', 1, 3] ,
		['CB', 'CG1', 'CD1', 'HD12', 1, -1] ,
		['CB', 'CG1', 'CD1', 'HD13', 1, -1] ,
		['C', 'CA', 'CB', 'HB', 1, -1] ,
		['HB', 'CB', 'CG1', 'HG11', 1, -1] ,
		['HB', 'CB', 'CG1', 'HG12', 1, -1] ,
		['HB', 'CB', 'CG1', 'CD1', 1, -1] ,
		['HB', 'CB', 'CG2', 'HG21', 1, -1] ,
		['HB', 'CB', 'CG2', 'HG22', 1, -1] ,
		['HB', 'CB', 'CG2', 'HG23', 1, -1] ,
		['C', 'CA', 'CB', 'CG1', 1, -1] ,
		['CG1', 'CB', 'CG2', 'HG21', 1, -1] ,
		['CG1', 'CB', 'CG2', 'HG22', 1, -1] ,
		['CG1', 'CB', 'CG2', 'HG23', 1, -1] ,
		['CG2', 'CB', 'CG1', 'HG11', 1, -1] ,
		['HG11', 'CG1', 'CD1', 'HD11', 1, -1] ,
		['HG11', 'CG1', 'CD1', 'HD12', 1, -1] ,
		['HG11', 'CG1', 'CD1', 'HD13', 1, -1] ,
		['CG2', 'CB', 'CG1', 'HG12', 1, -1] ,
		['HG12', 'CG1', 'CD1', 'HD11', 1, -1] ,
		['HG12', 'CG1', 'CD1', 'HD12', 1, -1] ,
		['HG12', 'CG1', 'CD1', 'HD13', 1, -1] ,
		['C', 'CA', 'CB', 'CG2', 1, -1] ,
		['CG2', 'CB', 'CG1', 'CD1', 1, -1] ,
	     ),
	'LYS' : (
		['N', 'CA', 'CB', 'HB1', 1, -1] ,
		['N', 'CA', 'CB', 'HB2', 1, -1] ,
		['N', 'CA', 'CB', 'CG', 1, 1] ,
		['N', 'CA', 'C', 'O', 1, 0] ,
		['H', 'N', 'CA', 'HA', 1, -1] ,
		['H', 'N', 'CA', 'CB', 1, -1] ,
		['C', 'CA', 'N', 'H', 1, -1] ,
		['CA', 'CB', 'CG', 'HG1', 1, -1] ,
		['CA', 'CB', 'CG', 'HG2', 1, -1] ,
		['CA', 'CB', 'CG', 'CD', 1, 2] ,
		['HA', 'CA', 'CB', 'HB1', 1, -1] ,
		['HA', 'CA', 'CB', 'HB2', 1, -1] ,
		['HA', 'CA', 'CB', 'CG', 1, -1] ,
		['HA', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CG', 'CD', 'HD1', 1, -1] ,
		['CB', 'CG', 'CD', 'HD2', 1, -1] ,
		['CB', 'CG', 'CD', 'CE', 1, 3] ,
		['C', 'CA', 'CB', 'HB1', 1, -1] ,
		['HB1', 'CB', 'CG', 'HG1', 1, -1] ,
		['HB1', 'CB', 'CG', 'HG2', 1, -1] ,
		['HB1', 'CB', 'CG', 'CD', 1, -1] ,
		['C', 'CA', 'CB', 'HB2', 1, -1] ,
		['HB2', 'CB', 'CG', 'HG1', 1, -1] ,
		['HB2', 'CB', 'CG', 'HG2', 1, -1] ,
		['HB2', 'CB', 'CG', 'CD', 1, -1] ,
		['C', 'CA', 'CB', 'CG', 1, -1] ,
		['CG', 'CD', 'CE', 'HE1', 1, -1] ,
		['CG', 'CD', 'CE', 'HE2', 1, -1] ,
		['CG', 'CD', 'CE', 'NZ', 1, 4] ,
		['HG1', 'CG', 'CD', 'HD1', 1, -1] ,
		['HG1', 'CG', 'CD', 'HD2', 1, -1] ,
		['HG1', 'CG', 'CD', 'CE', 1, -1] ,
		['HG2', 'CG', 'CD', 'HD1', 1, -1] ,
		['HG2', 'CG', 'CD', 'HD2', 1, -1] ,
		['HG2', 'CG', 'CD', 'CE', 1, -1] ,
		['CD', 'CE', 'NZ', 'HZ1', 1, 5] ,
		['CD', 'CE', 'NZ', 'HZ2', 1, -1] ,
		['CD', 'CE', 'NZ', 'HZ3', 1, -1] ,
		['HD1', 'CD', 'CE', 'HE1', 1, -1] ,
		['HD1', 'CD', 'CE', 'HE2', 1, -1] ,
		['HD1', 'CD', 'CE', 'NZ', 1, -1] ,
		['HD2', 'CD', 'CE', 'HE1', 1, -1] ,
		['HD2', 'CD', 'CE', 'HE2', 1, -1] ,
		['HD2', 'CD', 'CE', 'NZ', 1, -1] ,
		['HE1', 'CE', 'NZ', 'HZ1', 1, -1] ,
		['HE1', 'CE', 'NZ', 'HZ2', 1, -1] ,
		['HE1', 'CE', 'NZ', 'HZ3', 1, -1] ,
		['HE2', 'CE', 'NZ', 'HZ1', 1, -1] ,
		['HE2', 'CE', 'NZ', 'HZ2', 1, -1] ,
		['HE2', 'CE', 'NZ', 'HZ3', 1, -1] ,
	     ),
	'LEU' : (
		['N', 'CA', 'CB', 'HB1', 1, -1] ,
		['N', 'CA', 'CB', 'HB2', 1, -1] ,
		['N', 'CA', 'CB', 'CG', 1, 1] ,
		['N', 'CA', 'C', 'O', 1, 0] ,
		['H', 'N', 'CA', 'HA', 1, -1] ,
		['H', 'N', 'CA', 'CB', 1, -1] ,
		['C', 'CA', 'N', 'H', 1, -1] ,
		['CA', 'CB', 'CG', 'HG', 1, -1] ,
		['CA', 'CB', 'CG', 'CD1', 1, 2] ,
		['CA', 'CB', 'CG', 'CD2', 1, -1] ,
		['HA', 'CA', 'CB', 'HB1', 1, -1] ,
		['HA', 'CA', 'CB', 'HB2', 1, -1] ,
		['HA', 'CA', 'CB', 'CG', 1, -1] ,
		['HA', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CG', 'CD1', 'HD11', 1, 3] ,
		['CB', 'CG', 'CD1', 'HD12', 1, -1] ,
		['CB', 'CG', 'CD1', 'HD13', 1, -1] ,
		['CB', 'CG', 'CD2', 'HD21', 1, -1] ,
		['CB', 'CG', 'CD2', 'HD22', 1, -1] ,
		['CB', 'CG', 'CD2', 'HD23', 1, -1] ,
		['C', 'CA', 'CB', 'HB1', 1, -1] ,
		['HB1', 'CB', 'CG', 'HG', 1, -1] ,
		['HB1', 'CB', 'CG', 'CD1', 1, -1] ,
		['HB1', 'CB', 'CG', 'CD2', 1, -1] ,
		['C', 'CA', 'CB', 'HB2', 1, -1] ,
		['HB2', 'CB', 'CG', 'HG', 1, -1] ,
		['HB2', 'CB', 'CG', 'CD1', 1, -1] ,
		['HB2', 'CB', 'CG', 'CD2', 1, -1] ,
		['C', 'CA', 'CB', 'CG', 1, -1] ,
		['HG', 'CG', 'CD1', 'HD11', 1, -1] ,
		['HG', 'CG', 'CD1', 'HD12', 1, -1] ,
		['HG', 'CG', 'CD1', 'HD13', 1, -1] ,
		['HG', 'CG', 'CD2', 'HD21', 1, -1] ,
		['HG', 'CG', 'CD2', 'HD22', 1, -1] ,
		['HG', 'CG', 'CD2', 'HD23', 1, -1] ,
		['CD1', 'CG', 'CD2', 'HD21', 1, -1] ,
		['CD1', 'CG', 'CD2', 'HD22', 1, -1] ,
		['CD1', 'CG', 'CD2', 'HD23', 1, -1] ,
		['CD2', 'CG', 'CD1', 'HD11', 1, -1] ,
		['CD2', 'CG', 'CD1', 'HD12', 1, -1] ,
		['CD2', 'CG', 'CD1', 'HD13', 1, -1] ,
	     ),
	'MET' : (
		['N', 'CA', 'CB', 'HB1', 1, -1] ,
		['N', 'CA', 'CB', 'HB2', 1, -1] ,
		['N', 'CA', 'CB', 'CG', 1, 1] ,
		['N', 'CA', 'C', 'O', 1, 0] ,
		['H', 'N', 'CA', 'HA', 1, -1] ,
		['H', 'N', 'CA', 'CB', 1, -1] ,
		['C', 'CA', 'N', 'H', 1, -1] ,
		['CA', 'CB', 'CG', 'HG1', 1, -1] ,
		['CA', 'CB', 'CG', 'HG2', 1, -1] ,
		['CA', 'CB', 'CG', 'SD', 1, 2] ,
		['HA', 'CA', 'CB', 'HB1', 1, -1] ,
		['HA', 'CA', 'CB', 'HB2', 1, -1] ,
		['HA', 'CA', 'CB', 'CG', 1, -1] ,
		['HA', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CG', 'SD', 'CE', 1, 3] ,
		['C', 'CA', 'CB', 'HB1', 1, -1] ,
		['HB1', 'CB', 'CG', 'HG1', 1, -1] ,
		['HB1', 'CB', 'CG', 'HG2', 1, -1] ,
		['HB1', 'CB', 'CG', 'SD', 1, -1] ,
		['C', 'CA', 'CB', 'HB2', 1, -1] ,
		['HB2', 'CB', 'CG', 'HG1', 1, -1] ,
		['HB2', 'CB', 'CG', 'HG2', 1, -1] ,
		['HB2', 'CB', 'CG', 'SD', 1, -1] ,
		['C', 'CA', 'CB', 'CG', 1, -1] ,
		['CG', 'SD', 'CE', 'HE1', 1, 4] ,
		['CG', 'SD', 'CE', 'HE2', 1, -1] ,
		['CG', 'SD', 'CE', 'HE3', 1, -1] ,
		['HG1', 'CG', 'SD', 'CE', 1, -1] ,
		['HG2', 'CG', 'SD', 'CE', 1, -1] ,
	     ),
	'ASN' : (
		['N', 'CA', 'CB', 'HB1', 1, -1] ,
		['N', 'CA', 'CB', 'HB2', 1, -1] ,
		['N', 'CA', 'CB', 'CG', 1, 1] ,
		['N', 'CA', 'C', 'O', 1, 0] ,
		['H', 'N', 'CA', 'HA', 1, -1] ,
		['H', 'N', 'CA', 'CB', 1, -1] ,
		['C', 'CA', 'N', 'H', 1, -1] ,
		['CA', 'CB', 'CG', 'OD1', 1, 2] ,
		['CA', 'CB', 'CG', 'ND2', 1, -1] ,
		['HA', 'CA', 'CB', 'HB1', 1, -1] ,
		['HA', 'CA', 'CB', 'HB2', 1, -1] ,
		['HA', 'CA', 'CB', 'CG', 1, -1] ,
		['HA', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CG', 'ND2', 'HD21', 0, -1] ,
		['CB', 'CG', 'ND2', 'HD22', 0, -1] ,
		['C', 'CA', 'CB', 'HB1', 1, -1] ,
		['HB1', 'CB', 'CG', 'OD1', 1, -1] ,
		['HB1', 'CB', 'CG', 'ND2', 1, -1] ,
		['C', 'CA', 'CB', 'HB2', 1, -1] ,
		['HB2', 'CB', 'CG', 'OD1', 1, -1] ,
		['HB2', 'CB', 'CG', 'ND2', 1, -1] ,
		['C', 'CA', 'CB', 'CG', 1, -1] ,
		['OD1', 'CG', 'ND2', 'HD21', 0, -1] ,
		['OD1', 'CG', 'ND2', 'HD22', 0, -1] ,
	     ),
	'GLN' : (
		['N', 'CA', 'CB', 'HB1', 1, -1] ,
		['N', 'CA', 'CB', 'HB2', 1, -1] ,
		['N', 'CA', 'CB', 'CG', 1, 1] ,
		['N', 'CA', 'C', 'O', 1, 0] ,
		['H', 'N', 'CA', 'HA', 1, -1] ,
		['H', 'N', 'CA', 'CB', 1, -1] ,
		['C', 'CA', 'N', 'H', 1, -1] ,
		['CA', 'CB', 'CG', 'HG1', 1, -1] ,
		['CA', 'CB', 'CG', 'HG2', 1, -1] ,
		['CA', 'CB', 'CG', 'CD', 1, 2] ,
		['HA', 'CA', 'CB', 'HB1', 1, -1] ,
		['HA', 'CA', 'CB', 'HB2', 1, -1] ,
		['HA', 'CA', 'CB', 'CG', 1, -1] ,
		['HA', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CG', 'CD', 'OE1', 1, 3] ,
		['CB', 'CG', 'CD', 'NE2', 1, -1] ,
		['C', 'CA', 'CB', 'HB1', 1, -1] ,
		['HB1', 'CB', 'CG', 'HG1', 1, -1] ,
		['HB1', 'CB', 'CG', 'HG2', 1, -1] ,
		['HB1', 'CB', 'CG', 'CD', 1, -1] ,
		['C', 'CA', 'CB', 'HB2', 1, -1] ,
		['HB2', 'CB', 'CG', 'HG1', 1, -1] ,
		['HB2', 'CB', 'CG', 'HG2', 1, -1] ,
		['HB2', 'CB', 'CG', 'CD', 1, -1] ,
		['C', 'CA', 'CB', 'CG', 1, -1] ,
		['CG', 'CD', 'NE2', 'HE21', 0, -1] ,
		['CG', 'CD', 'NE2', 'HE22', 0, -1] ,
		['HG1', 'CG', 'CD', 'OE1', 1, -1] ,
		['HG1', 'CG', 'CD', 'NE2', 1, -1] ,
		['HG2', 'CG', 'CD', 'OE1', 1, -1] ,
		['HG2', 'CG', 'CD', 'NE2', 1, -1] ,
		['OE1', 'CD', 'NE2', 'HE21', 0, -1] ,
		['OE1', 'CD', 'NE2', 'HE22', 0, -1] ,
	     ),
	'ARG' : (
		['N', 'CA', 'CB', 'HB1', 1, -1] ,
		['N', 'CA', 'CB', 'HB2', 1, -1] ,
		['N', 'CA', 'CB', 'CG', 1, 1] ,
		['N', 'CA', 'C', 'O', 1, 0] ,
		['H', 'N', 'CA', 'HA', 1, -1] ,
		['H', 'N', 'CA', 'CB', 1, -1] ,
		['C', 'CA', 'N', 'H', 1, -1] ,
		['CA', 'CB', 'CG', 'HG1', 1, -1] ,
		['CA', 'CB', 'CG', 'HG2', 1, -1] ,
		['CA', 'CB', 'CG', 'CD', 1, 2] ,
		['HA', 'CA', 'CB', 'HB1', 1, -1] ,
		['HA', 'CA', 'CB', 'HB2', 1, -1] ,
		['HA', 'CA', 'CB', 'CG', 1, -1] ,
		['HA', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CG', 'CD', 'HD1', 1, -1] ,
		['CB', 'CG', 'CD', 'HD2', 1, -1] ,
		['CB', 'CG', 'CD', 'NE', 1, 3] ,
		['C', 'CA', 'CB', 'HB1', 1, -1] ,
		['HB1', 'CB', 'CG', 'HG1', 1, -1] ,
		['HB1', 'CB', 'CG', 'HG2', 1, -1] ,
		['HB1', 'CB', 'CG', 'CD', 1, -1] ,
		['C', 'CA', 'CB', 'HB2', 1, -1] ,
		['HB2', 'CB', 'CG', 'HG1', 1, -1] ,
		['HB2', 'CB', 'CG', 'HG2', 1, -1] ,
		['HB2', 'CB', 'CG', 'CD', 1, -1] ,
		['C', 'CA', 'CB', 'CG', 1, -1] ,
		['CG', 'CD', 'NE', 'HE', 1, -1] ,
		['CG', 'CD', 'NE', 'CZ', 1, 4] ,
		['HG1', 'CG', 'CD', 'HD1', 1, -1] ,
		['HG1', 'CG', 'CD', 'HD2', 1, -1] ,
		['HG1', 'CG', 'CD', 'NE', 1, -1] ,
		['HG2', 'CG', 'CD', 'HD1', 1, -1] ,
		['HG2', 'CG', 'CD', 'HD2', 1, -1] ,
		['HG2', 'CG', 'CD', 'NE', 1, -1] ,
		['CD', 'NE', 'CZ', 'NH1', 0, 5] ,
		['CD', 'NE', 'CZ', 'NH2', 0, -1] ,
		['HD1', 'CD', 'NE', 'HE', 1, -1] ,
		['HD1', 'CD', 'NE', 'CZ', 1, -1] ,
		['HD2', 'CD', 'NE', 'HE', 1, -1] ,
		['HD2', 'CD', 'NE', 'CZ', 1, -1] ,
		['NE', 'CZ', 'NH1', 'HH11', 0, 6] ,
		['NE', 'CZ', 'NH1', 'HH12', 0, -1] ,
		['NE', 'CZ', 'NH2', 'HH21', 0, -1] ,
		['NE', 'CZ', 'NH2', 'HH22', 0, -1] ,
		['HE', 'NE', 'CZ', 'NH1', 0, -1] ,
		['HE', 'NE', 'CZ', 'NH2', 0, -1] ,
		['NH1', 'CZ', 'NH2', 'HH21', 0, -1] ,
		['NH1', 'CZ', 'NH2', 'HH22', 0, -1] ,
		['NH2', 'CZ', 'NH1', 'HH11', 0, -1] ,
		['NH2', 'CZ', 'NH1', 'HH12', 0, -1] ,
	     ),
	'SER' : (
		['N', 'CA', 'CB', 'HB1', 1, -1] ,
		['N', 'CA', 'CB', 'HB2', 1, -1] ,
		['N', 'CA', 'CB', 'OG', 1, 1] ,
		['N', 'CA', 'C', 'O', 1, 0] ,
		['H', 'N', 'CA', 'HA', 1, -1] ,
		['H', 'N', 'CA', 'CB', 1, -1] ,
		['C', 'CA', 'N', 'H', 1, -1] ,
		['CA', 'CB', 'OG', 'HG', 1, 2] ,
		['HA', 'CA', 'CB', 'HB1', 1, -1] ,
		['HA', 'CA', 'CB', 'HB2', 1, -1] ,
		['HA', 'CA', 'CB', 'OG', 1, -1] ,
		['HA', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CA', 'C', 'O', 1, -1] ,
		['C', 'CA', 'CB', 'HB1', 1, -1] ,
		['HB1', 'CB', 'OG', 'HG', 1, -1] ,
		['C', 'CA', 'CB', 'HB2', 1, -1] ,
		['HB2', 'CB', 'OG', 'HG', 1, -1] ,
		['C', 'CA', 'CB', 'OG', 1, -1] ,
	     ),
	'SEP' : (
		['N', 'CA', 'CB', 'HB1', 1, -1] ,
		['N', 'CA', 'CB', 'HB2', 1, -1] ,
		['N', 'CA', 'CB', 'OG', 1, 1] ,
		['N', 'CA', 'C', 'O', 1, 0] ,
		['H', 'N', 'CA', 'HA', 1, -1] ,
		['H', 'N', 'CA', 'CB', 1, -1] ,
		['C', 'CA', 'N', 'H', 1, -1] ,
		['CA', 'CB', 'OG', 'P', 1, 2] ,
		['HA', 'CA', 'CB', 'HB1', 1, -1] ,
		['HA', 'CA', 'CB', 'HB2', 1, -1] ,
		['HA', 'CA', 'CB', 'OG', 1, -1] ,
		['HA', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CA', 'C', 'O', 1, -1] ,
		['C', 'CA', 'CB', 'HB1', 1, -1] ,
		['HB1', 'CB', 'OG', 'P', 1, -1] ,
		['C', 'CA', 'CB', 'HB2', 1, -1] ,
		['HB2', 'CB', 'OG', 'P', 1, -1] ,
		['C', 'CA', 'CB', 'OG', 1, -1] ,
		['CB', 'OG', 'P', 'O1P', 1, 3] ,
		['CB', 'OG', 'P', 'O2P', 1, -1] ,
		['CB', 'OG', 'P', 'O3P', 1, -1] ,

	     ),
	'THR' : (
		['N', 'CA', 'CB', 'HB', 1, -1] ,
		['N', 'CA', 'CB', 'OG1', 1, 1] ,
		['N', 'CA', 'CB', 'CG2', 1, -1] ,
		['N', 'CA', 'C', 'O', 1, 0] ,
		['H', 'N', 'CA', 'HA', 1, -1] ,
		['H', 'N', 'CA', 'CB', 1, -1] ,
		['C', 'CA', 'N', 'H', 1, -1] ,
		['CA', 'CB', 'OG1', 'HG1', 1, 2] ,
		['CA', 'CB', 'CG2', 'HG21', 1, -1] ,
		['CA', 'CB', 'CG2', 'HG22', 1, -1] ,
		['CA', 'CB', 'CG2', 'HG23', 1, -1] ,
		['HA', 'CA', 'CB', 'HB', 1, -1] ,
		['HA', 'CA', 'CB', 'OG1', 1, -1] ,
		['HA', 'CA', 'CB', 'CG2', 1, -1] ,
		['HA', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CA', 'C', 'O', 1, -1] ,
		['C', 'CA', 'CB', 'HB', 1, -1] ,
		['HB', 'CB', 'OG1', 'HG1', 1, -1] ,
		['HB', 'CB', 'CG2', 'HG21', 1, -1] ,
		['HB', 'CB', 'CG2', 'HG22', 1, -1] ,
		['HB', 'CB', 'CG2', 'HG23', 1, -1] ,
		['C', 'CA', 'CB', 'OG1', 1, -1] ,
		['OG1', 'CB', 'CG2', 'HG21', 1, -1] ,
		['OG1', 'CB', 'CG2', 'HG22', 1, -1] ,
		['OG1', 'CB', 'CG2', 'HG23', 1, -1] ,
		['CG2', 'CB', 'OG1', 'HG1', 1, -1] ,
		['C', 'CA', 'CB', 'CG2', 1, -1] ,
	     ),
	'VAL' : (
		['N', 'CA', 'CB', 'HB', 1, -1] ,
		['N', 'CA', 'CB', 'CG1', 1, -1] ,
		['N', 'CA', 'CB', 'CG2', 1, 1] ,
		['N', 'CA', 'C', 'O', 1, 0] ,
		['H', 'N', 'CA', 'HA', 1, -1] ,
		['H', 'N', 'CA', 'CB', 1, -1] ,
		['C', 'CA', 'N', 'H', 1, -1] ,
		['CA', 'CB', 'CG1', 'HG11', 1, -1] ,
		['CA', 'CB', 'CG1', 'HG12', 1, -1] ,
		['CA', 'CB', 'CG1', 'HG13', 1, -1] ,
		['CA', 'CB', 'CG2', 'HG21', 1, 2] ,
		['CA', 'CB', 'CG2', 'HG22', 1, -1] ,
		['CA', 'CB', 'CG2', 'HG23', 1, -1] ,
		['HA', 'CA', 'CB', 'HB', 1, -1] ,
		['HA', 'CA', 'CB', 'CG1', 1, -1] ,
		['HA', 'CA', 'CB', 'CG2', 1, -1] ,
		['HA', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CA', 'C', 'O', 1, -1] ,
		['C', 'CA', 'CB', 'HB', 1, -1] ,
		['HB', 'CB', 'CG1', 'HG11', 1, -1] ,
		['HB', 'CB', 'CG1', 'HG12', 1, -1] ,
		['HB', 'CB', 'CG1', 'HG13', 1, -1] ,
		['HB', 'CB', 'CG2', 'HG21', 1, -1] ,
		['HB', 'CB', 'CG2', 'HG22', 1, -1] ,
		['HB', 'CB', 'CG2', 'HG23', 1, -1] ,
		['C', 'CA', 'CB', 'CG1', 1, -1] ,
		['CG1', 'CB', 'CG2', 'HG21', 1, -1] ,
		['CG1', 'CB', 'CG2', 'HG22', 1, -1] ,
		['CG1', 'CB', 'CG2', 'HG23', 1, -1] ,
		['CG2', 'CB', 'CG1', 'HG11', 1, -1] ,
		['CG2', 'CB', 'CG1', 'HG12', 1, -1] ,
		['CG2', 'CB', 'CG1', 'HG13', 1, -1] ,
		['C', 'CA', 'CB', 'CG2', 1, -1] ,
	     ),
	'TRP' : (
		['N', 'CA', 'CB', 'HB1', 1, -1] ,
		['N', 'CA', 'CB', 'HB2', 1, -1] ,
		['N', 'CA', 'CB', 'CG', 1, 1] ,
		['N', 'CA', 'C', 'O', 1, 0] ,
		['H', 'N', 'CA', 'HA', 1, -1] ,
		['H', 'N', 'CA', 'CB', 1, -1] ,
		['C', 'CA', 'N', 'H', 1, -1] ,
		['CA', 'CB', 'CG', 'CD1', 1, -1] ,
		['CA', 'CB', 'CG', 'CD2', 1, 2] ,
		['HA', 'CA', 'CB', 'HB1', 1, -1] ,
		['HA', 'CA', 'CB', 'HB2', 1, -1] ,
		['HA', 'CA', 'CB', 'CG', 1, -1] ,
		['HA', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CG', 'CD1', 'HD1', 0, -1] ,
		['CB', 'CG', 'CD1', 'NE1', 0, 3] ,
		['CB', 'CG', 'CD2', 'CE2', 0, -1] ,
		['C', 'CA', 'CB', 'HB1', 1, -1] ,
		['HB1', 'CB', 'CG', 'CD1', 1, -1] ,
		['HB1', 'CB', 'CG', 'CD2', 1, -1] ,
		['C', 'CA', 'CB', 'HB2', 1, -1] ,
		['HB2', 'CB', 'CG', 'CD1', 1, -1] ,
		['HB2', 'CB', 'CG', 'CD2', 1, -1] ,
		['C', 'CA', 'CB', 'CG', 1, -1] ,
		['CG', 'CD1', 'NE1', 'HE1', 0, 4] ,
		['CG', 'CD2', 'CE2', 'CZ2', 0, 4] ,
		['CD1', 'CG', 'CD2', 'CE2', 0, -1] ,
		['CD2', 'CG', 'CD1', 'HD1', 0, -1] ,
		['HD1', 'CD1', 'NE1', 'HE1', 0, -1] ,
		['CD2', 'CG', 'CD1', 'NE1', 0, -1] ,
		['CD2', 'CE2', 'CZ2', 'HZ2', 0, -1] ,
		['CD2', 'CE2', 'CZ2', 'CH2', 0, 5] ,
		['CE2', 'CZ2', 'CH2', 'HH2', 0, 6] ,
		['HE3', 'CE3', 'CZ3', 'HZ3', 0, -1] ,
		['HZ2', 'CZ2', 'CH2', 'HH2', 0, -1] ,
	     ),
	'TYR' : (
		['N', 'CA', 'CB', 'HB1', 1, -1] ,
		['N', 'CA', 'CB', 'HB2', 1, -1] ,
		['N', 'CA', 'CB', 'CG', 1, 1] ,
		['N', 'CA', 'C', 'O', 1, 0] ,
		['H', 'N', 'CA', 'HA', 1, -1] ,
		['H', 'N', 'CA', 'CB', 1, -1] ,
		['C', 'CA', 'N', 'H', 1, -1] ,
		['CA', 'CB', 'CG', 'CD1', 1, 2] ,
		['CA', 'CB', 'CG', 'CD2', 1, -1] ,
		['HA', 'CA', 'CB', 'HB1', 1, -1] ,
		['HA', 'CA', 'CB', 'HB2', 1, -1] ,
		['HA', 'CA', 'CB', 'CG', 1, -1] ,
		['HA', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CA', 'C', 'O', 1, -1] ,
		['CB', 'CG', 'CD1', 'HD1', 0, -1] ,
		['CB', 'CG', 'CD1', 'CE1', 0, 3] ,
		['CB', 'CG', 'CD2', 'HD2', 0, -1] ,
		['CB', 'CG', 'CD2', 'CE2', 0, -1] ,
		['C', 'CA', 'CB', 'HB1', 1, -1] ,
		['HB1', 'CB', 'CG', 'CD1', 1, -1] ,
		['HB1', 'CB', 'CG', 'CD2', 1, -1] ,
		['C', 'CA', 'CB', 'HB2', 1, -1] ,
		['HB2', 'CB', 'CG', 'CD1', 1, -1] ,
		['HB2', 'CB', 'CG', 'CD2', 1, -1] ,
		['C', 'CA', 'CB', 'CG', 1, -1] ,
		['CG', 'CD1', 'CE1', 'HE1', 0, -1] ,
		['CG', 'CD1', 'CE1', 'CZ', 0, 4] ,
		['CG', 'CD2', 'CE2', 'HE2', 0, -1] ,
		['CG', 'CD2', 'CE2', 'CZ', 0, -1] ,
		['CD1', 'CG', 'CD2', 'HD2', 0, -1] ,
		['CD1', 'CG', 'CD2', 'CE2', 0, -1] ,
		['CD1', 'CE1', 'CZ', 'CE2', 0, 5] ,
		['CD1', 'CE1', 'CZ', 'OH', 0, -1] ,
		['CD2', 'CG', 'CD1', 'HD1', 0, -1] ,
		['HD1', 'CD1', 'CE1', 'HE1', 0, -1] ,
		['HD1', 'CD1', 'CE1', 'CZ', 0, -1] ,
		['CD2', 'CG', 'CD1', 'CE1', 0, -1] ,
		['CD2', 'CE2', 'CZ', 'CE1', 0, -1] ,
		['CD2', 'CE2', 'CZ', 'OH', 0, -1] ,
		['HD2', 'CD2', 'CE2', 'HE2', 0, -1] ,
		['HD2', 'CD2', 'CE2', 'CZ', 0, -1] ,
		['CE1', 'CZ', 'CE2', 'HE2', 0, -1] ,
		['CE1', 'CZ', 'OH', 'HH', 1, 6] ,
		['CE2', 'CZ', 'CE1', 'HE1', 0, -1] ,
		['HE1', 'CE1', 'CZ', 'OH', 0, -1] ,
		['CE2', 'CZ', 'OH', 'HH', 1, -1] ,
		['HE2', 'CE2', 'CZ', 'OH', 0, -1] ,
	     ),
	}



_mol2_types = {
    'ALA':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'GLU':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.3',0],
    'CD':['C.2',0],
    'OE1':['O.co2',-0.5],
    'OE2':['O.co2',-0.5],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'GLUP':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.3',0],
    'CD':['C.2',0],
    'OE1':['O.co2',-0.5],
    'OE2':['O.co2',-0.5],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'GLH':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.3',0],
    'CD':['C.2',0],
    'OE1':['O.co2',-0.5], # wrong, but doesn't matter
    'OE2':['O.co2',-0.5],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'GLUH':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.3',0],
    'CD':['C.2',0],
    'OE1':['O.co2',-0.5], # wrong, but doesn't matter
    'OE2':['O.co2',-0.5],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'TYR':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.ar',0],
    'CD1':['C.ar',0],
    'CE1':['C.ar',0],
    'CZ':['C.ar',0],
    'OH':['O.3',0],
    'CE2':['C.ar',0],
    'CD2':['C.ar',0],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'HIS':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.2',0],
    'ND1':['N.2',0],
    'CE1':['C.2',0],
    'NE2':['N.pl3',0],
    'CD2':['C.2',0],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'HSE':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.2',0],
    'ND1':['N.2',0],
    'CE1':['C.2',0],
    'NE2':['N.pl3',0],
    'CD2':['C.2',0],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'HID':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.2',0],
    'ND1':['N.2',0],
    'CE1':['C.2',0],
    'NE2':['N.pl3',0],
    'CD2':['C.2',0],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'HSD':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.2',0],
    'ND1':['N.2',0],
    'CE1':['C.2',0],
    'NE2':['N.pl3',0],
    'CD2':['C.2',0],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'HIE':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.2',0],
    'ND1':['N.2',0],
    'CE1':['C.2',0],
    'NE2':['N.pl3',0],
    'CD2':['C.2',0],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'HIP':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.2',0],
    'ND1':['N.2',0],
    'CE1':['C.2',0],
    'NE2':['N.pl3',0],
    'CD2':['C.2',0],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'HSP':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.2',0],
    'ND1':['N.2',0],
    'CE1':['C.2',0],
    'NE2':['N.pl3',0],
    'CD2':['C.2',0],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'HISH':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.2',0],
    'ND1':['N.2',0],
    'CE1':['C.2',0],
    'NE2':['N.pl3',0],
    'CD2':['C.2',0],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'HIS1':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.2',0],
    'ND1':['N.2',0],
    'CE1':['C.2',0],
    'NE2':['N.pl3',0],
    'CD2':['C.2',0],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'HISE':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.2',0],
    'ND1':['N.2',0],
    'CE1':['C.2',0],
    'NE2':['N.pl3',0],
    'CD2':['C.2',0],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'VAL':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG1':['C.3',0],
    'CG2':['C.3',0],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'LEU':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.3',0],
    'CD1':['C.3',0],
    'CD2':['C.3',0],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'LYS':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.3',0],
    'CD':['C.3',0],
    'CE':['C.3',0],
    'NZ':['N.4',1],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'LYP':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.3',0],
    'CD':['C.3',0],
    'CE':['C.3',0],
    'NZ':['N.4',1],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'LYN':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.3',0],
    'CD':['C.3',0],
    'CE':['C.3',0],
    'NZ':['N.4',1],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'LSN':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.3',0],
    'CD':['C.3',0],
    'CE':['C.3',0],
    'NZ':['N.4',1],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'LYSH':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.3',0],
    'CD':['C.3',0],
    'CE':['C.3',0],
    'NZ':['N.4',1],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'THR':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG2':['C.3',0],
    'OG1':['O.3',0],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'SER':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'OG':['O.3',0],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'ASN':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.2',0],
    'OD1':['O.2',0],
    'ND2':['N.am',0],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'PRO':
    {
    'N':['N.am',0],
    'CD':['C.3',0],
    'CG':['C.3',0],
    'CB':['C.3',0],
    'CA':['C.3',0],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'PHE':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.ar',0],
    'CD1':['C.ar',0],
    'CE1':['C.ar',0],
    'CZ':['C.ar',0],
    'CE2':['C.ar',0],
    'CD2':['C.ar',0],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'TRP':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.ar',0],
    'CD1':['C.ar',0],
    'NE1':['N.pl3',0],
    'CE2':['C.ar',0],
    'CZ2':['C.ar',0],
    'CH2':['C.ar',0],
    'CZ3':['C.ar',0],
    'CE3':['C.ar',0],
    'CD2':['C.ar',0],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'ASP':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.2',0],
    'OD1':['O.co2',-0.5],
    'OD2':['O.co2',-0.5],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'ASH':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.2',0],
    'OD1':['O.co2',-0.5],
    'OD2':['O.co2',-0.5],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'ASPP':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.2',0],
    'OD1':['O.co2',-0.5],
    'OD2':['O.co2',-0.5],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'ASPH':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.2',0],
    'OD1':['O.co2',-0.5],
    'OD2':['O.co2',-0.5],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'MET':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.3',0],
    'SD':['S.3',0],
    'CE':['C.3',0],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'GLY':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'ILE':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG2':['C.3',0],
    'CG1':['C.3',0],
    'CD1':['C.3',0],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'GLN':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.3',0],
    'CD':['C.2',0],
    'OE1':['O.2',0],
    'NE2':['N.am',0],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'ARG':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'CG':['C.3',0],
    'CD':['C.3',0],
    'NE':['N.pl3',0.333],
    'CZ':['C.cat',0],
    'NH1':['N.pl3',0.333],
    'NH2':['N.pl3',0.333],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'CYS':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'SG':['S.3',-1],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'CYM':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'SG':['S.3',-1],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'CYN':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'SG':['S.3',-1],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    'CYSH':
    {
    'N':['N.am',0],
    'CA':['C.3',0],
    'CB':['C.3',0],
    'SG':['S.3',-1],
    'C':['C.2',0],
    'O':['O.2',0],
    },
    }

_mol2_bondtypes = [
    ('N.4', 'C.3', '1') ,
    ('N.4', 'H', '1') ,
    ('C.3', 'C.2', '1') ,
    ('C.3', 'C.3', '1') ,
    ('C.3', 'H', '1') ,
    ('C.3', 'N.pl3', '1') ,
    ('N.pl3', 'C.cat', 'ar') ,
    ('N.pl3', 'H', '1') ,
    ('C.cat', 'N.pl3', 'ar') ,
    ('C.2', 'N.am', 'am') ,
    ('C.2', 'O.2', '2') ,
    ('N.am', 'C.3', '1') ,
    ('N.am', 'H', '1') ,
    ('C.2', 'O.co2', 'ar') ,
    ('C.3', 'O.3', '1') ,
    ('O.3', 'H', '1') ,
    ('C.3', 'C.ar', '1') ,
    ('C.ar', 'C.ar', '1') ,
    ('C.ar', 'C.ar', '2') ,
    ('C.ar', 'N.pl3', '1') ,
    ('C.ar', 'H', '1') ,
    ('N.pl3', 'C.ar', '1') ,
    ('C.ar', 'C.ar', 'ar') ,
    ('C.3', 'N.4', '1') ,
    ('C.2', 'N.2', '1') ,
    ('C.2', 'C.2', '2') ,
    ('N.2', 'C.2', '2') ,
    ('C.2', 'N.pl3', '1') ,
    ('C.2', 'H', '1') ,
    ('N.pl3', 'C.2', '1') ,
    ('C.3', 'S.3', '1') ,
    ('S.3', 'C.3', '1') ,
    ('C.ar', 'O.3', '1') ,
    ]

# ==============================================================================
#                           Standard MDP Files
# ==============================================================================
mdps = {}
mdps['enmin'] = ''';====================================================
; Energy minimization
;====================================================

; RUN CONTROL & MINIMIZATION
;----------------------------------------------------
define                 = -DFLEXIBLE
integrator             = steep
nsteps                 = {nsteps}
emtol                  = 10
emstep                 = 0.01
nstcomm                = 100

; OUTPUT CONTROL
;----------------------------------------------------
nstxout                = 0          ; don't save coordinates to .trr
nstvout                = 0          ; don't save velocities to .trr
nstfout                = 0          ; don't save forces to .trr

nstxout-compressed     = 1000       ; xtc trajectory output every 1000 steps
compressed-x-precision = 1000
nstlog                 = 1000       ; update log file every 1000 steps
nstenergy              = 1000       ; save energies every 1000 steps
nstcalcenergy          = 100

; NEIGHBOR SEARCHING
;----------------------------------------------------
cutoff-scheme          = Verlet
ns-type                = grid
nstlist                = 1
rlist                  = {cutoff}

; BONDS
;----------------------------------------------------
constraints            = none

; ELECTROSTATICS
;----------------------------------------------------
coulombtype            = PME
coulomb-modifier       = Potential-shift-Verlet
rcoulomb               = {cutoff}
pme-order              = 4
fourierspacing         = 0.12
ewald-rtol             = 1e-5

; VDW
;----------------------------------------------------
vdw-type                = Cut-off
vdw-modifier            = Potential-shift-Verlet
verlet-buffer-tolerance = 0.005
rvdw                    = {cutoff}
DispCorr                = EnerPres

; TEMPERATURE & PRESSURE COUPL
;----------------------------------------------------
Tcoupl              = no
Pcoupl              = no
gen_vel             = no
'''

mdps['npt-restr'] = ''';====================================================
; Posres equilibration
;====================================================

; RUN CONTROL
;----------------------------------------------------
define       = -DPOSRES      ; use position restraints
integrator   = md            ; leap-frog integrator
nsteps       = {nsteps}      ; number of integration steps
dt           = 0.002         ; 2 fs
comm-mode    = Linear        ; remove center of mass translation
nstcomm      = 100           ; frequency for center of mass motion removal

; OUTPUT CONTROL
;----------------------------------------------------
nstxout          = 0         ; do not save coordinates to .trr
nstvout          = 0         ; do not save velocities to .trr
nstfout          = 0         ; do not save forces to .trr
nstxtcout        = 50000     ; xtc compressed trajectory output every 100 ps
xtc-precision    = 1000      ; precision with which to write to the compressed trajectory file
nstlog           = 10000     ; update log file every 20 ps
nstenergy        = 10000     ; save energies every 20 ps
nstcalcenergy    = 100       ; calculate energies every 100 steps (default=100)

; BONDS
;----------------------------------------------------
constraint_algorithm   = lincs      ; holonomic constraints
constraints            = all-bonds  ; constrain all bonds (you need this with the pmx setup)
lincs-iter             = 2          ; accuracy of LINCS (1 is default)
lincs-order            = 6          ; also related to accuracy (4 is default)
lincs-warnangle        = 30         ; maximum angle that a bond can rotate before LINCS will complain (30 is default)
continuation           = no         ; formerly known as 'unconstrained-start' - useful for exact continuations and reruns

; NEIGHBOR SEARCHING
;----------------------------------------------------
cutoff-scheme         = Verlet ; group or Verlet
ns-type               = grid   ; search neighboring grid cells
nstlist               = 10     ; 20 fs (default is 10)
rlist                 = {cutoff}    ; short-range neighborlist cutoff (in nm)
pbc                   = xyz    ; 3D PBC

; ELECTROSTATICS & EWALD
;----------------------------------------------------
coulombtype      = PME                       ; Particle Mesh Ewald for long-range electrostatics
coulomb-modifier = Potential-shift-Verlet
rcoulomb         = {cutoff}                       ; short-range electrostatic cutoff (in nm)
ewald-geometry   = 3d                        ; Ewald sum is performed in all three dimensions
pme-order        = 4                         ; interpolation order for PME (default is 4)
fourierspacing   = 0.12                      ; grid spacing for FFT
ewald-rtol       = 1e-5                      ; relative strength of the Ewald-shifted direct potential at rcoulomb
optimize-fft     = no                        ; don't calculate the optimal FFT plan for the grid at startup.

; VAN DER WAALS
;----------------------------------------------------
vdw-type          = Cut-off      ; potential switched off at rvdw-switch to reach zero at rvdw
vdw-modifier      = Potential-shift-Verlet
rvdw              = {cutoff}          ; van der Waals cutoff (in nm)
DispCorr          = EnerPres     ; apply analytical long range dispersion corrections for Energy and Pressure

; TEMPERATURE COUPLING (Langevin)
;----------------------------------------------------
tcoupl            = v-rescale
tc-grps           = System
tau-t             = 0.1
ref-t             = {T:.2f}
gen-vel           = yes           ; Velocity generation (if gen-vel is 'yes', continuation should be 'no')
gen-temp          = {T:.2f}
gen-seed          = -1

; PRESSURE COUPLING
;----------------------------------------------------
pcoupl           = Berendsen
pcoupltype       = isotropic            ; uniform scaling of box vectors
tau_p            = 0.5                  ; time constant (ps)
ref_p            = 1.0                  ; reference pressure (bar)
compressibility  = 4.5e-05              ; isothermal compressibility of water (bar^-1)
refcoord-scaling = com
'''

mdps['npt'] = ''';====================================================
; Equilibrium Simulations
;====================================================

; RUN CONTROL
;----------------------------------------------------
integrator   = md            ; stochastic leap-frog integrator
nsteps       = {nsteps}      ; number of integration steps
dt           = 0.002         ; 2 fs
comm-mode    = Linear        ; remove center of mass translation
nstcomm      = 100           ; frequency for center of mass motion removal

; OUTPUT CONTROL
;----------------------------------------------------
nstxout          = 0         ; do not save coordinates to .trr
nstvout          = 0         ; do not save velocities to .trr
nstfout          = 0         ; do not save forces to .trr
nstxtcout        = 50000     ; xtc compressed trajectory output every 100 ps
xtc-precision    = 1000      ; precision with which to write to the compressed trajectory file
nstlog           = 50000     ; update log file every 100 ps
nstenergy        = 50000     ; save energies every 100 ps
nstcalcenergy    = 100       ; calculate energies every 100 steps (default=100)

; BONDS
;----------------------------------------------------
constraint_algorithm   = lincs      ; holonomic constraints
constraints            = all-bonds  ; constrain all bonds (you need this with the pmx setup)
lincs-iter             = 2          ; accuracy of LINCS (1 is default)
lincs-order            = 6          ; also related to accuracy (4 is default)
lincs-warnangle        = 30         ; maximum angle that a bond can rotate before LINCS will complain (30 is default)
continuation           = yes        ; formerly known as 'unconstrained-start' - useful for exact continuations and reruns

; NEIGHBOR SEARCHING
;----------------------------------------------------
cutoff-scheme         = Verlet ; group or Verlet
ns-type               = grid   ; search neighboring grid cells
nstlist               = 10     ; 20 fs (default is 10)
rlist                 = {cutoff}    ; short-range neighborlist cutoff (in nm)
pbc                   = xyz    ; 3D PBC

; ELECTROSTATICS & EWALD
;----------------------------------------------------
coulombtype      = PME                       ; Particle Mesh Ewald for long-range electrostatics
coulomb-modifier = Potential-shift-Verlet
rcoulomb         = {cutoff}                       ; short-range electrostatic cutoff (in nm)
ewald-geometry   = 3d                        ; Ewald sum is performed in all three dimensions
pme-order        = 4                         ; interpolation order for PME (default is 4)
fourierspacing   = 0.12                      ; grid spacing for FFT
ewald-rtol       = 1e-5                      ; relative strength of the Ewald-shifted direct potential at rcoulomb
optimize-fft     = no                        ; don't calculate the optimal FFT plan for the grid at startup.

; VAN DER WAALS
;----------------------------------------------------
vdw-type          = Cut-off      ; potential switched off at rvdw-switch to reach zero at rvdw
vdw-modifier      = Potential-shift-Verlet
rvdw              = {cutoff}          ; van der Waals cutoff (in nm)
DispCorr          = EnerPres     ; apply analytical long range dispersion corrections for Energy and Pressure

; TEMPERATURE COUPLING (Langevin)
;----------------------------------------------------
tcoupl            = v-rescale
tc-grps           = System
tau-t             = 0.1
ref-t             = {T:.2f}
gen-vel           = no           ; Velocity generation (if gen-vel is 'yes', continuation should be 'no')
gen-temp          = {T:.2f}
gen-seed          = -1

; PRESSURE COUPLING
;----------------------------------------------------
pcoupl           = Parrinello-Rahman
pcoupltype       = isotropic            ; uniform scaling of box vectors
tau_p            = 2.0                  ; time constant (ps)
ref_p            = 1.0                  ; reference pressure (bar)
compressibility  = 4.5e-05              ; isothermal compressibility of water (bar^-1)
refcoord-scaling = no
'''
