"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getTsconfigCompilerOptions = exports.extractDependencies = exports.tryGetModuleVersionFromPkg = exports.tryGetModuleVersionFromRequire = exports.exec = exports.findUpMultiple = exports.findUp = exports.callsites = void 0;
const child_process_1 = require("child_process");
const fs = require("fs");
const path = require("path");
/**
 * Get callsites from the V8 stack trace API
 *
 * https://github.com/sindresorhus/callsites
 */
function callsites() {
    var _a;
    const _prepareStackTrace = Error.prepareStackTrace;
    Error.prepareStackTrace = (_, stack) => stack;
    const stack = (_a = new Error().stack) === null || _a === void 0 ? void 0 : _a.slice(1);
    Error.prepareStackTrace = _prepareStackTrace;
    return stack;
}
exports.callsites = callsites;
/**
 * Find a file by walking up parent directories
 */
function findUp(name, directory = process.cwd()) {
    return findUpMultiple([name], directory)[0];
}
exports.findUp = findUp;
/**
 * Find the lowest of multiple files by walking up parent directories. If
 * multiple files exist at the same level, they will all be returned.
 */
function findUpMultiple(names, directory = process.cwd()) {
    const absoluteDirectory = path.resolve(directory);
    const files = [];
    for (const name of names) {
        const file = path.join(directory, name);
        if (fs.existsSync(file)) {
            files.push(file);
        }
    }
    if (files.length > 0) {
        return files;
    }
    const { root } = path.parse(absoluteDirectory);
    if (absoluteDirectory === root) {
        return [];
    }
    return findUpMultiple(names, path.dirname(absoluteDirectory));
}
exports.findUpMultiple = findUpMultiple;
/**
 * Spawn sync with error handling
 */
function exec(cmd, args, options) {
    var _a, _b;
    const proc = child_process_1.spawnSync(cmd, args, options);
    if (proc.error) {
        throw proc.error;
    }
    if (proc.status !== 0) {
        if (proc.stdout || proc.stderr) {
            throw new Error(`[Status ${proc.status}] stdout: ${(_a = proc.stdout) === null || _a === void 0 ? void 0 : _a.toString().trim()}\n\n\nstderr: ${(_b = proc.stderr) === null || _b === void 0 ? void 0 : _b.toString().trim()}`);
        }
        throw new Error(`${cmd} ${args.join(' ')} ${(options === null || options === void 0 ? void 0 : options.cwd) ? `run in directory ${options.cwd}` : ''} exited with status ${proc.status}`);
    }
    return proc;
}
exports.exec = exec;
/**
 * Returns a module version by requiring its package.json file
 */
function tryGetModuleVersionFromRequire(mod) {
    try {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        return require(`${mod}/package.json`).version;
    }
    catch (err) {
        return undefined;
    }
}
exports.tryGetModuleVersionFromRequire = tryGetModuleVersionFromRequire;
/**
 * Gets module version from package.json content
 */
function tryGetModuleVersionFromPkg(mod, pkgJson, pkgPath) {
    var _a, _b, _c;
    const dependencies = {
        ...(_a = pkgJson.dependencies) !== null && _a !== void 0 ? _a : {},
        ...(_b = pkgJson.devDependencies) !== null && _b !== void 0 ? _b : {},
        ...(_c = pkgJson.peerDependencies) !== null && _c !== void 0 ? _c : {},
    };
    if (!dependencies[mod]) {
        return undefined;
    }
    // If it's a "file:" version, make it absolute
    const fileMatch = dependencies[mod].match(/file:(.+)/);
    if (fileMatch && !path.isAbsolute(fileMatch[1])) {
        const absoluteFilePath = path.join(path.dirname(pkgPath), fileMatch[1]);
        return `file:${absoluteFilePath}`;
    }
    ;
    return dependencies[mod];
}
exports.tryGetModuleVersionFromPkg = tryGetModuleVersionFromPkg;
/**
 * Extract versions for a list of modules.
 *
 * First lookup the version in the package.json and then fallback to requiring
 * the module's package.json. The fallback is needed for transitive dependencies.
 */
function extractDependencies(pkgPath, modules) {
    var _a;
    const dependencies = {};
    // Use require for cache
    const pkgJson = require(pkgPath); // eslint-disable-line @typescript-eslint/no-require-imports
    for (const mod of modules) {
        const version = (_a = tryGetModuleVersionFromPkg(mod, pkgJson, pkgPath)) !== null && _a !== void 0 ? _a : tryGetModuleVersionFromRequire(mod);
        if (!version) {
            throw new Error(`Cannot extract version for module '${mod}'. Check that it's referenced in your package.json or installed.`);
        }
        dependencies[mod] = version;
    }
    return dependencies;
}
exports.extractDependencies = extractDependencies;
function getTsconfigCompilerOptions(tsconfigPath) {
    const compilerOptions = extractTsConfig(tsconfigPath);
    const excludedCompilerOptions = [
        'composite',
        'tsBuildInfoFile',
    ];
    const options = {
        ...compilerOptions,
        // Overrides
        incremental: false,
        // Intentionally Setting rootDir and outDir, so that the compiled js file always end up next to .ts file.
        rootDir: './',
        outDir: './',
    };
    let compilerOptionsString = '';
    Object.keys(options).forEach((key) => {
        if (excludedCompilerOptions.includes(key)) {
            return;
        }
        const value = options[key];
        const option = '--' + key;
        const type = typeof value;
        if (type === 'boolean') {
            if (value) {
                compilerOptionsString += option + ' ';
            }
        }
        else if (type === 'string') {
            compilerOptionsString += option + ' ' + value + ' ';
        }
        else if (type === 'object') {
            if (Array.isArray(value)) {
                compilerOptionsString += option + ' ' + value.join(',') + ' ';
            }
        }
        else {
            throw new Error(`Missing support for compilerOption: [${key}]: { ${type}, ${value}} \n`);
        }
    });
    return compilerOptionsString.trim();
}
exports.getTsconfigCompilerOptions = getTsconfigCompilerOptions;
function extractTsConfig(tsconfigPath, previousCompilerOptions) {
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const { extends: extendedConfig, compilerOptions } = require(tsconfigPath);
    const updatedCompilerOptions = {
        ...(previousCompilerOptions !== null && previousCompilerOptions !== void 0 ? previousCompilerOptions : {}),
        ...compilerOptions,
    };
    if (extendedConfig) {
        return extractTsConfig(path.resolve(tsconfigPath.replace(/[^\/]+$/, ''), extendedConfig), updatedCompilerOptions);
    }
    return updatedCompilerOptions;
}
//# sourceMappingURL=data:application/json;base64,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