from typing import Union


def check_type(value):
    if isinstance(value, Channel):
        return value
    if value.__class__.__module__.startswith("napari"):
        value = value.name
    if not isinstance(value, (str, int)):
        raise TypeError(f"Channel need to be int or str, provided {type(value)}")
    return Channel(value)


class Channel:
    """
    This class is introduced to distinguish numerical algorithm parameter from choose chanel.
    In autogenerated interface field with this type limits input values to number of current image channels
    """

    def __init__(self, value: Union[str, int]):
        if isinstance(value, Channel):
            value = value.value
        assert isinstance(value, (str, int)), f"wrong type {value} {type(value)}"  # nosec
        self._value: Union[str, int] = value

    @property
    def value(self) -> Union[str, int]:
        """Value stored in this class"""
        return self._value

    def __str__(self):
        return str(self._value + 1) if isinstance(self._value, int) else self._value

    def __repr__(self):
        return f"<{self.__class__.__module__}.{self.__class__.__name__}(value={repr(self._value)})>"

    def __eq__(self, other):
        return isinstance(other, Channel) and self._value == other.value

    def __hash__(self):
        return hash(self._value)

    def as_dict(self):
        return {"value": self._value}

    @classmethod
    def __get_validators__(cls):
        yield check_type

    @classmethod
    def __modify_schema__(cls, field_schema):
        field_schema["title"] = "Channel"
        field_schema["type"] = "object"
        field_schema["properties"] = {"value": {"title": "value", "anyOf": [{"type": "string"}, {"type": "integer"}]}}
