# Copyright (c) 2023, Djaodjin Inc.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# 1. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
# THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
# PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
# CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
# EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
# PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
# OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
# WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
# OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
# ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

"""Views about contact activities"""

from django.views.generic import DetailView, TemplateView

from .. import settings
from ..compat import reverse
from ..helpers import update_context_urls
from ..mixins import ContactMixin
from ..models import Contact


class ContactDetailView(ContactMixin, DetailView):

    template_name = 'contacts/contact.html'
    model = Contact

    def get_object(self):
        return self.contact

    def get_context_data(self, **kwargs):
        context = super(ContactDetailView, self).get_context_data(**kwargs)
        context.update({'contact': self.contact})
        update_context_urls(context, {
            'api_activities': reverse('api_activities', args=(self.contact,)),
            'api_contact': reverse('api_contact', args=(self.contact,)),
            'api_contacts': reverse('api_contacts')
        })
        return context


class ContactListMixin(settings.EXTRA_MIXIN):
    """
    Mixin such that EXTRA_MIXIN can defaults to `object`.
    """


class ContactListView(ContactListMixin, TemplateView):

    template_name = 'contacts/index.html'

    def get_context_data(self, **kwargs):
        context = super(ContactListView, self).get_context_data(**kwargs)
        self.update_context_urls(context, {
            'api_contacts': reverse('api_contacts'),
            'contacts': reverse('signup_contacts')
        })
        return context
